﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using Renderer2D.Core;
using System.Threading.Tasks;
using System.Threading;
using System.IO;
using System.Reflection;
using EffectCombiner.Core;

namespace WinFormsTestApp
{
    public class CompositionManager
    {
        private readonly object syncRoot = new object();
        private Renderer[] renderers;
        public Renderer[] Renderers
        {
            get
            {
                lock (syncRoot)
                    return renderers;
            }
            private set
            {
                lock (syncRoot)
                    renderers = value;
            }
        }

        private readonly Composer<Renderer> composer = new Composer<Renderer>();

        public Task ComposeAsync()
        {
            var assemblyPath = Path.GetDirectoryName(Assembly.GetEntryAssembly().Location);
            var renderersPath = string.Format("{0}\\Renderers", assemblyPath);

            composer.AddPathLookup(renderersPath, true, "*.dll");

            Globals.ReportManager.Clear();
            Globals.ReportManager.AddInfoReport(
                new InfoReport("Composing renderers", ReportCategory.RendererComposition));

            return composer.ComposeAsync(new CompositionExceptionReporter())
                .ContinueWith(t => Renderers = t.Result);
        }

        private class CompositionExceptionReporter : IExceptionReporter
        {
            private readonly SynchronizationContext synchronizationContext;

            public CompositionExceptionReporter()
            {
                synchronizationContext = SynchronizationContext.Current;
            }

            public void Report(Exception error)
            {
                synchronizationContext.Post(() =>
                    Globals.ReportManager.AddErrorReport(
                    new ErrorReport(error.Message, ReportCategory.RendererComposition)));
            }
        }
    }
}
