﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Text;

namespace Renderer2D.GDI
{
    public abstract class GDIBrushContainer : Core.IBrush
    {
        public Brush NativeBrush { get; protected set; }

        public void Dispose() { }

        public virtual void SetBoundingBox(RectangleF box) { }
    }

    public class GDISolidBrushContainer : GDIBrushContainer, Core.ISolidColorBrush
    {
        public GDISolidBrushContainer(Core.IColor color)
        {
            if (color == null)
                throw new ArgumentNullException("color");

            Color = color;
        }

        private Core.IColor color;
        public Core.IColor Color
        {
            get { return color; }
            set
            {
                color = value;
                NativeBrush = new SolidBrush(color.ToColor());
            }
        }
    }

    public class GDILinearGradientBrushContainer : GDIBrushContainer, Core.ILinearGradientBrush
    {
        private Core.IPoint startPoint;
        private Core.IPoint endPoint;
        private Core.IGradientStop[] gradientStops;

        public GDILinearGradientBrushContainer(Core.IPoint startPoint,
                                               Core.IPoint endPoint,
                                               params Core.IGradientStop[] gradientStops)
        {
            if (startPoint == null)
                throw new ArgumentNullException("startPoint");
            if (endPoint == null)
                throw new ArgumentNullException("endPoint");
            if (gradientStops.Length < 2)
                throw new ArgumentException("There must be at least two gradient stops");

            StartPoint = startPoint;
            EndPoint = endPoint;
            GradientStops = gradientStops;

            SetBoundingBox(new RectangleF(0.0f, 0.0f, 1.0f, 1.0f));
        }

        public Core.IPoint StartPoint
        {
            get { return startPoint; }
            set { startPoint = value; }
        }

        public Core.IPoint EndPoint
        {
            get { return endPoint; }
            set { endPoint = value; }
        }

        public Core.IGradientStop[] GradientStops
        {
            get { return gradientStops; }
            set
            {
                if (value == null)
                    throw new ArgumentNullException("gradientStops");
                if (value.Length < 2)
                    throw new ArgumentException("There must be at least two gradient stops");

                gradientStops = value;
//                DefineBlend((LinearGradientBrush)NativeBrush);
            }
        }

        public override void SetBoundingBox(RectangleF box)
        {
            // Beware:
            // the strict inequality does matter since we want start=end
            // in the case startPoint=endPoint (component wise)
            //var startX = startPoint.X < endPoint.X ? box.X : box.X + box.Width;
            //var startY = startPoint.Y < endPoint.Y ? box.Y : box.Y + box.Height;
            //var endX = startPoint.X > endPoint.X ? box.X : box.X + box.Width;
            //var endY = startPoint.Y > endPoint.Y ? box.Y : box.Y + box.Height;

            double translate, scale;
            PointF realStart, realEnd;
            GetGradientEndingPoints(box, startPoint.ToPointF(), endPoint.ToPointF(),
                out translate, out scale, out realStart, out realEnd);

            // http://stackoverflow.com/a/3322503/253883
            // "
            // Constructing a brush costs almost nothing compared to the
            // work done to actually draw something with the brush.
            //
            // I wrote a very interactive retail graphical application
            // and found that creating new brushes didn't significantly
            // impact performance when measured with dotTrace
            // "
            var brush = new LinearGradientBrush(realStart, realEnd,
                gradientStops.First().Color.ToColor(),
                gradientStops.Last().Color.ToColor());

            DefineBlend(brush, translate, scale);
            NativeBrush = brush;
        }

        private static void GetGradientEndingPoints(RectangleF box,
            PointF start, PointF end,
            out double translate, out double scale,
            out PointF realStart, out PointF realEnd)
        {
            if (start.X == end.X && start.Y == end.Y)
                throw new ArgumentException("start == end");

            var p0 = new Core.Vector2D(box.X + start.X * box.Width, box.Y + start.Y * box.Height);
            var p1 = new Core.Vector2D(box.X + end.X * box.Width, box.Y + end.Y * box.Height);
            var u = p1 - p0;
            var l = u.Norm();
            u *= 1.0 / l;

            var dHa = Core.Vector2D.Dot(new Core.Vector2D(box.X, box.Y) - p0, u);
            var dHb = Core.Vector2D.Dot(new Core.Vector2D(box.X, box.Y + box.Height) - p0, u);
            var dHc = Core.Vector2D.Dot(new Core.Vector2D(box.X + box.Width, box.Y + box.Height) - p0, u);
            var dHd = Core.Vector2D.Dot(new Core.Vector2D(box.X + box.Width, box.Y) - p0, u);

            var d0 = new double[] { dHa, dHb, dHc, dHd }.Min();
            var d1 = new double[] { dHa, dHb, dHc, dHd }.Max();

            var h0 = p0 + u * d0;
            var h1 = p0 + u * d1;

            realStart = new PointF((float)h0.X, (float)h0.Y);
            realEnd = new PointF((float)h1.X, (float)h1.Y);

//            translate = -d0;
//            scale = l / (d1 - d0);
            translate = -d0 / (d1 - d0);
            scale = l / (d1 - d0);
        }

        private void DefineBlend(LinearGradientBrush brush, double translate, double scale)
        {
            if (brush == null)
                return;

            var fitGradientStops = GetFitGradientStops(gradientStops, translate, scale);
            GenerateColorBlends(brush, fitGradientStops);
        }

        private static List<Core.IGradientStop> GetFitGradientStops(Core.IGradientStop[] gradientStops, double translate, double scale)
        {
            var gradientStopsList = GetSortedGradientList(gradientStops);

            var first = GetColorForOffset(0.0, gradientStopsList);
            var last = GetColorForOffset(1.0, gradientStopsList);

            for (int i = 0; i < gradientStopsList.Count; ++i)
            {
                var offset = translate + scale * gradientStopsList[i].Offset;
                gradientStopsList[i].Offset = offset;
            }

            while (gradientStopsList.Count > 0 && gradientStopsList[0].Offset <= 0.0)
                gradientStopsList.RemoveAt(0);
            while (gradientStopsList.Count > 0 && gradientStopsList.Last().Offset >= 1.0)
                gradientStopsList.RemoveAt(gradientStopsList.Count - 1);

            gradientStopsList.Insert(0, new Core.GradientStop(first, 0.0));
            gradientStopsList.Add(new Core.GradientStop(last, 1.0));

            return gradientStopsList;
        }

        private static List<Core.IGradientStop> GetSortedGradientList(Core.IGradientStop[] gradientStops)
        {
            List<Core.IGradientStop> gradientStopsList = gradientStops
                .Select(x => new Core.GradientStop(x.Color, x.Offset) as Core.IGradientStop)
                .ToList();
            gradientStopsList.Sort(new GradientStopComparer());
            return gradientStopsList;
        }

        private class GradientStopComparer : IComparer<Core.IGradientStop>
        {
            public int Compare(Core.IGradientStop x, Core.IGradientStop y)
            {
                return Comparer<double>.Default.Compare(x.Offset, y.Offset);
            }
        }

        private static Core.IColor GetColorForOffset(double d, List<Core.IGradientStop> gradientStopsList)
        {
            var lastStop = gradientStopsList.First();
            foreach (var stop in gradientStopsList)
            {
                if (stop.Offset > d)
                {
                    var coeff = (stop.Offset != lastStop.Offset ? (d - lastStop.Offset) / (stop.Offset - lastStop.Offset) : 0.0);
                    return Interpolate(lastStop.Color, stop.Color, coeff);
                }
                else
                {
                    lastStop = stop;
                }
            }
            return lastStop.Color;
        }

        private static Core.IColor Interpolate(Core.IColor c1, Core.IColor c2, double coeff)
        {
            var r = (byte)(c1.R + (c2.R - c1.R) * coeff);
            var g = (byte)(c1.G + (c2.G - c1.G) * coeff);
            var b = (byte)(c1.B + (c2.B - c1.B) * coeff);
            var a = (byte)(c1.A + (c2.A - c1.A) * coeff);
            return new Core.Color(r, g, b, a);
        }

        private void GenerateColorBlends(LinearGradientBrush brush, List<Core.IGradientStop> fitGradientStops)
        {
            int numberOfStops = fitGradientStops.Count;

            var colorBlend = new ColorBlend();
            colorBlend.Positions = new float[numberOfStops];
            colorBlend.Colors = new Color[numberOfStops];

            for (int i = 0; i < numberOfStops; ++i)
            {
                colorBlend.Positions[i] = (float)fitGradientStops[i].Offset;
                colorBlend.Colors[i] = fitGradientStops[i].Color.ToColor();
            }

            brush.InterpolationColors = colorBlend;
        }
    }

    public class GDIRadialGradientBrushContainer : GDIBrushContainer, Core.IRadialGradientBrush
    {
        public GDIRadialGradientBrushContainer(Core.IPoint center,
                                               Core.IPoint gradientOriginOffset,
                                               double radiusX,
                                               double radiusY,
                                               params Core.IGradientStop[] gradientStops)
        {
            throw new NotImplementedException();
        }

        public Core.IPoint Center
        {
            get
            {
                throw new NotImplementedException();
            }
            set
            {
                throw new NotImplementedException();
            }
        }

        public Core.IPoint GradientOriginOffset
        {
            get
            {
                throw new NotImplementedException();
            }
            set
            {
                throw new NotImplementedException();
            }
        }

        public double RadiusX
        {
            get
            {
                throw new NotImplementedException();
            }
            set
            {
                throw new NotImplementedException();
            }
        }

        public double RadiusY
        {
            get
            {
                throw new NotImplementedException();
            }
            set
            {
                throw new NotImplementedException();
            }
        }

        public Core.IGradientStop[] GradientStops
        {
            get
            {
                throw new NotImplementedException();
            }
            set
            {
                throw new NotImplementedException();
            }
        }
    }
}
