﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;

namespace Renderer2D.Core
{
    public interface IMatrix
    {
        double M11 { get; set; }
        double M12 { get; set; }
        double M21 { get; set; }
        double M22 { get; set; }
        double M31 { get; set; }
        double M32 { get; set; }
    }

    public static class MatrixExtensions
    {
        public static Matrix ToConcreteMatrix(this IMatrix matrix)
        {
            return matrix as Matrix ?? new Matrix(matrix);
        }

        public static IMatrix Multiply(this IMatrix left, IMatrix right)
        {
            return Matrix.Multiply(left, right);
        }

        public static IPoint TransformPoint(this IMatrix matrix, double x, double y)
        {
            return Matrix.TransformPoint(matrix, x, y);
        }

        public static IPoint TransformPoint(this IMatrix matrix, IPoint point)
        {
            return Matrix.TransformPoint(matrix, point);
        }

        public static double Determinant(this IMatrix matrix)
        {
            return Matrix.Determinant(matrix);
        }

        public static bool IsInvertible(this IMatrix matrix)
        {
            return Matrix.IsInvertible(matrix);
        }

        public static IMatrix Inverse(this IMatrix matrix)
        {
            return Matrix.Inverse(matrix);
        }
    }

    public class Matrix : IMatrix
    {
        public double M11 { get; set; }
        public double M12 { get; set; }
        public double M21 { get; set; }
        public double M22 { get; set; }
        public double M31 { get; set; }
        public double M32 { get; set; }

        public Matrix()
        {
        }

        public Matrix(IMatrix matrix)
        {
            if (matrix == null)
                throw new ArgumentNullException("matrix");

            M11 = matrix.M11;
            M12 = matrix.M12;
            M21 = matrix.M21;
            M22 = matrix.M22;
            M31 = matrix.M31;
            M32 = matrix.M32;
        }

        public Matrix(
            double m11, double m12,
            double m21, double m22,
            double m31, double m32)
        {
            M11 = m11;
            M12 = m12;
            M21 = m21;
            M22 = m22;
            M31 = m31;
            M32 = m32;
        }

        public static readonly IMatrix Identity = new Matrix(1.0, 0.0, 0.0, 1.0, 0.0, 0.0);

        public static IMatrix Multiply(IMatrix left, IMatrix right)
        {
            return new Matrix
            {
                M11 = (left.M11 * right.M11) + (left.M12 * right.M21),
                M12 = (left.M11 * right.M12) + (left.M12 * right.M22),
                M21 = (left.M21 * right.M11) + (left.M22 * right.M21),
                M22 = (left.M21 * right.M12) + (left.M22 * right.M22),
                M31 = (left.M31 * right.M11) + (left.M32 * right.M21) + right.M31,
                M32 = (left.M31 * right.M12) + (left.M32 * right.M22) + right.M32,
            };
        }

        public static IMatrix Rotation(double angle, double centerX, double centerY)
        {
            var t1 = Translation(-centerX, -centerY);
            var r = Rotation(angle);
            var t2 = Translation(centerX, centerY);

            return t1.Multiply(r).Multiply(t2);
        }

        public static IMatrix Rotation(double angle)
        {
            var radians = (Math.PI * angle) / 180.0;
            var cos = Math.Cos(radians);
            var sin = Math.Sin(radians);

            return new Matrix
            {
                M11 = cos,
                M12 = sin,
                M21 = -sin,
                M22 = cos,
                M31 = 0.0,
                M32 = 0.0,
            };
        }

        public static IMatrix Scale(double x, double y)
        {
            return new Matrix
            {
                M11 = x,
                M12 = 0.0,
                M21 = 0.0,
                M22 = y,
                M31 = 0.0,
                M32 = 0.0,
            };
        }

        public static IMatrix Scale(double x, double y, double centerX, double centerY)
        {
            return new Matrix
            {
                M11 = x,
                M12 = 0.0,
                M21 = 0.0,
                M22 = y,
                M31 = centerX - (x * centerX),
                M32 = centerY - (y * centerY),
            };
        }

        public static IMatrix Translation(double x, double y)
        {
            return new Matrix
            {
                M11 = 1.0,
                M12 = 0.0,
                M21 = 0.0,
                M22 = 1.0,
                M31 = x,
                M32 = y,
            };
        }

        public static IPoint TransformPoint(IMatrix matrix, double x, double y)
        {
            return new Point
            {
                X = (x * matrix.M11) + (y * matrix.M21) + matrix.M31,
                Y = (x * matrix.M12) + (y * matrix.M22) + matrix.M32,
            };
        }

        public static IPoint TransformPoint(IMatrix matrix, IPoint point)
        {
            return TransformPoint(matrix, point.X, point.Y);
        }

        public static double Determinant(IMatrix matrix)
        {
            return matrix.M11 * matrix.M22 - matrix.M12 * matrix.M21;
        }

        public static bool IsInvertible(IMatrix matrix)
        {
            return Math.Abs(Determinant(matrix)) >= 1e-3;
        }

        public static IMatrix Inverse(IMatrix matrix)
        {
            var determinant = Determinant(matrix);

            if (Math.Abs(determinant) < 1e-3)
                throw new InvalidOperationException(Messages.EXCEPTION_MATRIX_NOT_INVERSIBLE);

            return new Matrix
            {
                M11 = matrix.M22 / determinant,
                M12 = -matrix.M12 / determinant,
                M21 = -matrix.M21 / determinant,
                M22 = matrix.M11 / determinant,
                M31 = (matrix.M21 * matrix.M32 - matrix.M31 * matrix.M22) / determinant,
                M32 = -(matrix.M11 * matrix.M32 - matrix.M31 * matrix.M12) / determinant,
            };
        }
    }
}
