﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace OperationManager.Core
{
    /// <summary>
    /// 履歴の現在位置を示すGUIDの発行イベント用引数です。
    /// </summary>
    public class CurrentGuidAcquiredEventArgs : EventArgs
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="identifier">履歴のGUID</param>
        public CurrentGuidAcquiredEventArgs(Guid identifier)
        {
            if (identifier == null)
            {
                throw new ArgumentNullException("identifier");
            }

            this.Identifier = identifier;
        }

        /// <summary>
        /// 現在位置の履歴を示すGUIDを取得します。
        /// </summary>
        public Guid Identifier { get; private set; }
    }

    /// <summary>
    /// 履歴単位に関するイベント引数です。
    /// </summary>
    /// <typeparam name="T">履歴単位の派生型</typeparam>
    public class OperationEventArgs<T> : EventArgs where T : Operation
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public OperationEventArgs(T operation)
        {
            if (operation == null)
            {
                throw new ArgumentNullException("operation");
            }

            this.Operation = operation;
        }

        /// <summary>
        /// イベント対象の履歴単位を取得します。
        /// </summary>
        public T Operation { get; private set; }
    }

    /// <summary>
    /// 複数の履歴単位に関するイベント引数です。
    /// </summary>
    /// <typeparam name="T">履歴単位の派生型</typeparam>
    public class OperationsEventArgs<T> : EventArgs where T : Operation
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="operations">イベント対象の履歴単位</param>
        public OperationsEventArgs(T operations)
            : this(new[] { operations })
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="operations">イベント対象の履歴単位配列</param>
        public OperationsEventArgs(T[] operations)
        {
            if (operations == null)
            {
                throw new ArgumentNullException("operations");
            }

            this.Operations = operations;
        }

        /// <summary>
        /// イベント対象の履歴単位配列を取得します。
        /// </summary>
        public T[] Operations { get; private set; }
    }

    /// <summary>
    /// 履歴を破棄する要因を表します。
    /// </summary>
    public enum OperationDiscardType
    {
        /// <summary>
        /// 履歴が追加されて最大保持数を超過した時に、最も古い履歴が破棄されることを表します。
        /// </summary>
        Overflowed,

        /// <summary>
        /// アンドゥ後に新たな履歴が追加された時に、最新からアンドゥしたところまでの履歴が破棄されることを表します。
        /// </summary>
        Disbranched,
    }

    /// <summary>
    /// 履歴を破棄する際のイベント引数です。
    /// </summary>
    /// <typeparam name="T">履歴単位の派生型</typeparam>
    public class DiscardedOperationsEventArgs<T> : OperationsEventArgs<T> where T : Operation
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="type">履歴を破棄する要因</param>
        /// <param name="operation">破棄する履歴</param>
        public DiscardedOperationsEventArgs(OperationDiscardType type, T operation)
            : base(operation)
        {
            this.Type = type;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="type">履歴を破棄する要因</param>
        /// <param name="operations">破棄する履歴の配列</param>
        public DiscardedOperationsEventArgs(OperationDiscardType type, T[] operations)
            : base(operations)
        {
            this.Type = type;
        }

        /// <summary>
        /// 履歴を破棄する要因を取得します。
        /// </summary>
        public OperationDiscardType Type { get; private set; }
    }
}
