﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;

namespace EnvironmentReporting
{
    public class ContextInformationPackage
    {
        public ContextInformationPackage(params Tuple<string, ContextInformation>[] contextInformations)
        {
            Date = DateTime.Now;

            ContextInformations = new ContextInformationContainer();
            foreach (var contextInformation in contextInformations)
            {
                ContextInformations.Add(contextInformation.Item1, contextInformation.Item2);
            }

            ContextInformationsString = GetStringFrom(ContextInformations);
            Streams = ExtractStreamsFrom(ContextInformations);
        }

        public DateTime Date { get; private set; }
        public string ContextInformationsString { get; private set; }
        public ContextInformationContainer ContextInformations { get; private set; }
        public IEnumerable<StreamContextInformation> Streams { get; private set; }

        private string GetTextVersion(ContextInformation contextInformations, int indentLevel)
        {
            string log = string.Empty;

            if (contextInformations is DecimalContextInformation)
            {
                log = string.Format("{0}", (contextInformations as DecimalContextInformation).Value);
            }
            else if (contextInformations is StringContextInformation)
            {
                log = string.Format("{0}", (contextInformations as StringContextInformation).Value);
            }
            else if (contextInformations is BinaryDataContextInformation)
            {
                throw new NotImplementedException();
            }
            else if (contextInformations is ContextInformationContainer)
            {
                log = string.Format("{{1}", Environment.NewLine);

                var container = (ContextInformationContainer)contextInformations;
                foreach (var key in container.Keys)
                {
                    log += string.Format("{0}{1} : {2}{3}",
                        new string(' ', 2 * indentLevel),
                        key,
                        GetTextVersion(container[key], indentLevel + 1),
                        Environment.NewLine);
                }
                log += string.Format("{0}}{1}",
                        new string(' ', 2 * indentLevel),
                        Environment.NewLine);
            }
            else
            {
                throw new NotImplementedException();
            }

            return log;
        }

        private string GetStringFrom(ContextInformation contextInformations)
        {
            return GetTextVersion(contextInformations, 0);
        }

        private IEnumerable<StreamContextInformation> ExtractStreamsFrom(ContextInformation contextInformation)
        {
            if (contextInformation is ContextInformationContainer)
            {
                foreach (var child in ((ContextInformationContainer)contextInformation).Values)
                {
                    var subSequence = ExtractStreamsFrom(child);
                    foreach (var item in subSequence)
                    {
                        yield return item;
                    }
                }
            }
            else if (contextInformation is StreamContextInformation)
            {
                yield return (StreamContextInformation)contextInformation;
            }
        }
    }
}
