﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition.Hosting;
using System.ComponentModel.Composition.Primitives;
using System.Linq;
using System.Threading.Tasks;

namespace EffectCombiner.Core
{
    public class Composer<T>
    {
        private readonly object syncRoot = new object();
        private readonly List<PathLookupInfo> paths = new List<PathLookupInfo>();

        public void AddPathLookup(string path)
        {
            lock (syncRoot)
                paths.Add(new PathLookupInfo(path));
        }

        public void AddPathLookup(string path, bool includeSubDirectories, params string[] searchPatterns)
        {
            lock (syncRoot)
                paths.Add(new PathLookupInfo(path, includeSubDirectories, searchPatterns));
        }

        public void AddPathLookup(PathLookupInfo path)
        {
            lock (syncRoot)
                paths.Add(path);
        }

        public T[] Compose()
        {
            return Compose(null);
        }

        public Task<T[]> ComposeAsync()
        {
            return ComposeAsync(null);
        }

        public Task<T[]> ComposeAsync(IExceptionReporter errorReporter)
        {
            return Task.Factory.StartNew(() => Compose(errorReporter));
        }

        public T[] Compose(IExceptionReporter reporter)
        {
            var catalogs = new List<ComposablePartCatalog>();

            catalogs.AddRange(GetAssemblies(reporter));

            var aggregateCatalog = new AggregateCatalog(catalogs);
            var compositionContainer = new CompositionContainer(aggregateCatalog);

            try
            {
                return compositionContainer.GetExportedValues<T>().ToArray();
            }
            catch (Exception ex)
            {
                if (reporter != null)
                {
                    reporter.Report(ex);
                    return new T[0];
                }

                throw;
            }
        }

        private IEnumerable<string> RetrieveFileList()
        {
            PathLookupInfo[] coldPaths;

            lock (syncRoot)
                coldPaths = paths.ToArray();

            return CoreUtility.RetrieveFiles(coldPaths).ToArray();
        }

        private IEnumerable<AssemblyCatalog> GetAssemblies(IExceptionReporter reporter)
        {
            var assemblyCatalogs = RetrieveFileList()
                .Select(f => CreateAssemblyCatalog(f, reporter))
                .Where(ac => ac != null);

            foreach (var assemblyCatalog in assemblyCatalogs)
            {
                try
                {
                    var dummy = assemblyCatalog.Parts.Any();
                }
                catch (Exception ex)
                {
                    if (reporter != null)
                        reporter.Report(ex);
                    continue;
                }

                yield return assemblyCatalog;
            }
        }

        private static AssemblyCatalog CreateAssemblyCatalog(string file, IExceptionReporter reporter)
        {
            try
            {
                return new AssemblyCatalog(file);
            }
            catch (Exception ex)
            {
                if (reporter != null)
                    reporter.Report(ex);
                return null;
            }
        }
    }
}
