﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using EffectDefinitions;

namespace Blocks.Core
{
    public enum PlugHitTestResult
    {
        None,
        Input,
        Output,
    }

    /// <summary>
    /// Visual representation of a block: it belongs to a block manager and has a block definition.
    /// </summary>
    public abstract class BlockElementBase
    {
        private static uint currentUniqueIdentifier = 1;

        protected BlockElementBase(BlockDefinition blockDefinition)
        {
            if (blockDefinition == null)
                throw new ArgumentNullException("blockDefinition");

            BlockDefinition = blockDefinition;
            InstanceIdentifier = currentUniqueIdentifier++;
        }

        public event EventHandler<BlockDefinitionChangedEventArgs> BlockDefinitionChanged;

        protected virtual void OnBlockDefinitionChanged(BlockDefinition oldDefinition, BlockDefinition newDefinition)
        {
            var handler = BlockDefinitionChanged;
            if (handler != null)
                handler(this, new BlockDefinitionChangedEventArgs(oldDefinition, newDefinition));
        }

        public void UpdateBlockDefinition(BlockDefinition blockDefinition)
        {
            if (blockDefinition == null)
                throw new ArgumentNullException("blockDefinition");

            var oldDefinition = BlockDefinition;
            var newDefinition = blockDefinition;
            if (oldDefinition != newDefinition)
            {
                BlockDefinition = blockDefinition;
                OnBlockDefinitionChanged(oldDefinition, newDefinition);
            }
        }

        public uint InstanceIdentifier { get; private set; }
        /// <summary>
        /// Warning: Call this method only if your really known and understand why calling it!
        /// Set a new instance identifier for the block element.
        /// </summary>
        /// <param name="instanceIdentifier">New instance identifier.</param>
        public void SetNewInstanceIdentifier(uint instanceIdentifier)
        {
            if (instanceIdentifier == 0)
                throw new ArgumentException("instanceIdentifier"); // TODO: provide a real message
            InstanceIdentifier = instanceIdentifier;
        }

        public BlockManagerBase BlockManager { get; internal set; }
        public BlockDefinition BlockDefinition { get; private set; }

        #region Visibility

        private bool isVisible = true;
        public virtual bool IsVisible
        {
            get { return isVisible; }
            set
            {
                if (isVisible != value)
                {
                    isVisible = value;
                    if (BlockManager != null)
                        BlockManager.NotifyVisibilityChanged(this);
                    OnVisibilityChanged(EventArgs.Empty);
                }
            }
        }

        public event EventHandler VisibilityChanged;

        protected virtual void OnVisibilityChanged(EventArgs e)
        {
            var handler = VisibilityChanged;
            if (handler != null)
                handler(this, e);
        }

        #endregion // Visibility

        #region Selection

        private bool isSelected;
        public virtual bool IsSelected
        {
            get { return isSelected; }
            set
            {
                if (isSelected != value)
                {
                    isSelected = value;
                    if (BlockManager != null)
                        BlockManager.NotifySelectionChanged(this);
                    OnSelectionChanged(EventArgs.Empty);
                }
            }
        }

        private bool isInvertingSelection;
        public virtual bool IsInvertingSelection
        {
            get { return isInvertingSelection; }
            set
            {
                if (isInvertingSelection != value)
                {
                    isInvertingSelection = value;
                    if (BlockManager != null)
                        BlockManager.NotifySelectionChanged(this);
                    OnSelectionChanged(EventArgs.Empty);
                }
            }
        }

        public event EventHandler SelectionChanged;

        protected virtual void OnSelectionChanged(EventArgs e)
        {
            var handler = SelectionChanged;
            if (handler != null)
                handler(this, e);
        }

        #endregion // Selection

        #region Dragging

        private bool isDragging;
        public virtual bool IsDragging
        {
            get { return isDragging; }
            set
            {
                if (isDragging != value)
                {
                    isDragging = value;
                    if (BlockManager != null)
                        BlockManager.NotifyDraggingChanged(this);
                    OnDraggedSelectionChanged(EventArgs.Empty);
                }
            }
        }

        public event EventHandler DraggedSelectionChanged;

        protected virtual void OnDraggedSelectionChanged(EventArgs e)
        {
            var handler = DraggedSelectionChanged;
            if (handler != null)
                handler(this, e);
        }

        #endregion // Dragging

        #region Position

        public double Left { get; private set; }
        public double Top { get; private set; }

        public void SetPosition(double left, double top)
        {
            var changed = false;

            var oldLeft = Left;
            var oldTop = Top;

            if (Math.Abs(Left - left) > double.Epsilon)
            {
                Left = left;
                changed = true;
            }

            if (Math.Abs(Top - top) > double.Epsilon)
            {
                Top = top;
                changed = true;
            }

            if (changed)
                OnPositionChanged(new PositionChangedEventArgs(oldLeft, oldTop, Left, Top));
        }

        public event EventHandler<PositionChangedEventArgs> PositionChanged;

        protected virtual void OnPositionChanged(PositionChangedEventArgs e)
        {
            var handler = PositionChanged;
            if (handler != null)
                handler(this, e);
        }

        #endregion // Position

        #region Size

        public double Width { get; private set; }
        public double Height { get; private set; }

        protected internal bool Resize(double width, double height)
        {
            var changed = false;

            var oldWidth = Width;
            var oldHeight = Height;

            if (Math.Abs(Width - width) > double.Epsilon)
            {
                Width = width;
                changed = true;
            }

            if (Math.Abs(Height - height) > double.Epsilon)
            {
                Height = height;
                changed = true;
            }

            if (changed)
                OnSizeChanged(new SizeChangedEventArgs(oldWidth, oldHeight, Width, Height));

            return changed;
        }

        public event EventHandler<SizeChangedEventArgs> SizeChanged;

        protected virtual void OnSizeChanged(SizeChangedEventArgs e)
        {
            var handler = SizeChanged;
            if (handler != null)
                handler(this, e);
        }

        #endregion // Size

        #region ZOrder

        private int zOrder;
        public int ZOrder
        {
            get { return zOrder; }
            internal set
            {
                if (zOrder != value)
                {
                    var oldZOrder = zOrder;
                    zOrder = value;
                    OnZOrderChanged(new ZOrderChangedEventArgs(oldZOrder, zOrder));
                }
            }
        }

        public event EventHandler<ZOrderChangedEventArgs> ZOrderChanged;

        protected virtual void OnZOrderChanged(ZOrderChangedEventArgs e)
        {
            var handler = ZOrderChanged;
            if (handler != null)
                handler(this, e);
        }

        public void BringToFront()
        {
            if (BlockManager != null)
                BlockManager.BringToFront(this);
        }

        public void SendToBack()
        {
            if (BlockManager != null)
                BlockManager.SendToBack(this);
        }

        #endregion // ZOrder

        #region Added

        public event EventHandler AddedToBlockManager;

        protected internal virtual void OnAddedToBlockManager()
        {
            var handler = AddedToBlockManager;
            if (handler != null)
                handler(this, EventArgs.Empty);
        }

        #endregion // Added

        #region Removed

        public event EventHandler RemovedFromBlockManager;

        protected internal virtual void OnRemovedFromBlockManager()
        {
            var handler = RemovedFromBlockManager;
            if (handler != null)
                handler(this, EventArgs.Empty);
        }

        #endregion // Removed

        private bool IsInside(double left, double right, double top, double bottom)
        {
            return
                left <= Left && right >= Left + Width &&
                top <= Top && bottom >= Top + Height;
        }

        private bool IsOverlapping(double left, double right, double top, double bottom)
        {
            return
                (left <= Left && right >= Left + Width || left >= Left && left <= Left + Width || right >= Left && right <= Left + Width) &&
                (top <= Top && bottom >= Top + Height || top >= Top && top <= Top + Height || bottom >= Top && bottom <= Top + Height);
        }

        public bool IsWithinSelection(bool completelyIncludeToSelect, double left, double right, double top, double bottom)
        {
            return (completelyIncludeToSelect ? IsInside(left, right, top, bottom) : IsOverlapping(left, right, top, bottom));
        }

        public override string ToString()
        {
            if (BlockDefinition == null)
                return "(no block definition)";

            return BlockDefinition.ToString();
        }
    }
}
