﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Text;
using System.Collections.Generic;
using System.Linq;
using EffectCombiner.Primitives.Generation;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using EffectCombiner.Primitives;

namespace EffectCombinerUnitTests
{
    [TestClass]
    public class GlslTypingUtilityTest
    {
        private struct TypeDescription
        {
            public int Dx;
            public int Dy;
            public string Name;
            public ShaderType Type;
            public PrimitiveShaderType BaseType;

            public TypeDescription(int dx, int dy, string name, ShaderType type, PrimitiveShaderType baseType)
            {
                Dx = dx;
                Dy = dy;
                Name = name;
                Type = type;
                BaseType = baseType;
            }
        }

        private static TypeDescription[] typesInfo = new TypeDescription[] {
                new TypeDescription(1, 1, "bool",  ShaderType.Boolean, PrimitiveShaderType.Boolean),
                new TypeDescription(1, 1, "int",   ShaderType.Integer, PrimitiveShaderType.Integer),
                new TypeDescription(1, 1, "uint",  ShaderType.UnsignedInteger, PrimitiveShaderType.UnsignedInteger),
                new TypeDescription(1, 1, "float", ShaderType.Float, PrimitiveShaderType.Float),
                new TypeDescription(1, 1, "double", ShaderType.Double, PrimitiveShaderType.Double),

                new TypeDescription(1, 2, "bvec2", ShaderType.BooleanVector2, PrimitiveShaderType.Boolean),
                new TypeDescription(1, 2, "ivec2", ShaderType.IntegerVector2, PrimitiveShaderType.Integer),
                new TypeDescription(1, 2, "uvec2", ShaderType.UnsignedIntegerVector2, PrimitiveShaderType.UnsignedInteger),
                new TypeDescription(1, 2, "vec2",  ShaderType.FloatVector2, PrimitiveShaderType.Float),
                new TypeDescription(1, 2, "dvec2", ShaderType.DoubleVector2, PrimitiveShaderType.Double),

                new TypeDescription(1, 3, "bvec3", ShaderType.BooleanVector3, PrimitiveShaderType.Boolean),
                new TypeDescription(1, 3, "ivec3", ShaderType.IntegerVector3, PrimitiveShaderType.Integer),
                new TypeDescription(1, 3, "uvec3", ShaderType.UnsignedIntegerVector3, PrimitiveShaderType.UnsignedInteger),
                new TypeDescription(1, 3, "vec3",  ShaderType.FloatVector3, PrimitiveShaderType.Float),
                new TypeDescription(1, 3, "dvec3", ShaderType.DoubleVector3, PrimitiveShaderType.Double),

                new TypeDescription(1, 4, "bvec4", ShaderType.BooleanVector4, PrimitiveShaderType.Boolean),
                new TypeDescription(1, 4, "ivec4", ShaderType.IntegerVector4, PrimitiveShaderType.Integer),
                new TypeDescription(1, 4, "uvec4", ShaderType.UnsignedIntegerVector4, PrimitiveShaderType.UnsignedInteger),
                new TypeDescription(1, 4, "vec4",  ShaderType.FloatVector4, PrimitiveShaderType.Float),
                new TypeDescription(1, 4, "dvec4", ShaderType.DoubleVector4, PrimitiveShaderType.Double),

                new TypeDescription(2, 2, "mat2x2", ShaderType.Matrix2By2, PrimitiveShaderType.Float),
                new TypeDescription(2, 3, "mat2x3", ShaderType.Matrix2By3, PrimitiveShaderType.Float),
                new TypeDescription(2, 4, "mat2x4", ShaderType.Matrix2By4, PrimitiveShaderType.Float),

                new TypeDescription(3, 2, "mat3x2", ShaderType.Matrix3By2, PrimitiveShaderType.Float),
                new TypeDescription(3, 3, "mat3x3", ShaderType.Matrix3By3, PrimitiveShaderType.Float),
                new TypeDescription(3, 4, "mat3x4", ShaderType.Matrix3By4, PrimitiveShaderType.Float),

                new TypeDescription(4, 2, "mat4x2", ShaderType.Matrix4By2, PrimitiveShaderType.Float),
                new TypeDescription(4, 3, "mat4x3", ShaderType.Matrix4By3, PrimitiveShaderType.Float),
                new TypeDescription(4, 4, "mat4x4", ShaderType.Matrix4By4, PrimitiveShaderType.Float)
            };

        [TestMethod]
        public void GetDimensionXTest()
        {
            foreach (var typeInfo in typesInfo)
            {
                var type = typeInfo.Type;
                var expected = typeInfo.Dx;
                var dx = GlslTypingUtility.GetDimensionX(type);
                if (dx != expected)
                    throw new Exception(string.Format("Dimension X for {0} should be {1} but returned {2}", type, expected, dx));
            }
        }

        [TestMethod]
        public void GetDimensionYTest()
        {
            foreach (var typeInfo in typesInfo)
            {
                var type = typeInfo.Type;
                var expected = typeInfo.Dy;
                var dy = GlslTypingUtility.GetDimensionY(type);
                if (dy != expected)
                    throw new Exception(string.Format("Dimension Y for {0} should be {1} but returned {2}", type, expected, dy));
            }
        }

        [TestMethod]
        public void GetPrimitiveShaderTypeTest()
        {
            foreach (var typeInfo in typesInfo)
            {
                var type = typeInfo.Type;
                var expected = typeInfo.BaseType;
                var baseType = GlslTypingUtility.GetPrimitiveShaderType(type);
                if (baseType != expected)
                    throw new Exception(string.Format("Primitive shader type for {0} should be {1} but returned {2}", type, expected, baseType));
            }
        }

        [TestMethod]
        public void GetGlslTypeNameFromShaderTypeTest()
        {
            foreach (var typeInfo in typesInfo)
            {
                var type = typeInfo.Type;
                var expected = typeInfo.Name;
                var name = GlslTypingUtility.GetGlslTypeNameFromShaderType(type);
                if (name != expected)
                    throw new Exception(string.Format("GLSL name for {0} should be \"{1}\" but returned \"{2}\"", type, expected, name));
            }
        }

        [TestMethod]
        public void GetShaderTypeFromGlslTypeNameTest()
        {
            foreach (var typeInfo in typesInfo)
            {
                var name = typeInfo.Name;
                var expected = typeInfo.Type;
                var type = GlslTypingUtility.GetShaderTypeFromGlslTypeName(name);
                if (type != expected)
                    throw new Exception(string.Format("Shader type for \"{0}\" should be {1} but returned {2}", name, expected, type));
            }
            var duplicates = new[] {
                new { Name = "mat2", Type = ShaderType.Matrix2By2 },
                new { Name = "mat3", Type = ShaderType.Matrix3By3 },
                new { Name = "mat4", Type = ShaderType.Matrix4By4 } };
            foreach (var typeInfo in duplicates)
            {
                var name = typeInfo.Name;
                var expected = typeInfo.Type;
                var type = GlslTypingUtility.GetShaderTypeFromGlslTypeName(name);
                if (type != expected)
                    throw new Exception(string.Format("Shader type for \"{0}\" should be {1} but returned {2}", name, expected, type));
            }
        }
    }
}
