﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Reflection;
using Renderer2D.Core;
using Renderer2D.Core.WinForms;
using System.Xml.Linq;
using System.Windows.Forms;
using Color = Renderer2D.Core.Color;
using Font = Renderer2D.Core.Font;
using Point = Renderer2D.Core.Point;

namespace EffectCombiner.Primitives
{
    public class TagColor
    {
        public System.Drawing.Color ListViewColor;
        public System.Drawing.Color BlockGradationColor;
    }

    public class VisualResourceSet : IDisposable
    {
        public double TitleTextHeight { get; private set; }
        public double SubTitleTextHeight { get; private set; }
        public double PlugTextHeight { get; private set; }
        public double PlugRadius { get; private set; }

        public double RegularBlockRadiusX { get; private set; }
        public double RegularBlockRadiusY { get; private set; }

        public double RegularSelectedBlockBorderStroke { get; private set; }
        public double RegularUnselectedBlockBorderStroke { get; private set; }

        public double RegularBlockShadowOffset { get; private set; }
        public double RegularDraggedBlockLiftOffset { get; private set; }

        public double RegularBlockTitleTextHorizontalPadding { get; private set; }
        public double RegularBlockTitleTextVerticalPadding { get; private set; }
        public double RegularBlockPreviewImageHorizontalPadding { get; private set; }
        public double RegularBlockPreviewImageVerticalPadding { get; private set; }
        public double RegularBlockDecorationHorizontalPadding { get; private set; }
        public double RegularBlockDecorationVerticalPadding { get; private set; }
        public double RegularBlockPlugsMinHorizontalSpacing { get; private set; }

        public double ConnectionLineStroke { get; private set; }

        public double DragWindowWidth { get; set; }
        public double DragWindowHeight { get; set; }

        public ITextFormat RegularBlockTitleTextFormat { get; private set; }
        public ITextFormat RegularBlockInputPlugTextFormat { get; private set; }
        public ITextFormat RegularBlockOutputPlugTextFormat { get; private set; }
        public ITextFormat RegularBlockShortcutInputPlugTextFormat { get; private set; }
        public ITextFormat RegularBlockShortcutOutputPlugTextFormat { get; private set; }

        public IBrush GridLineBrush { get; private set; }

        public IBrush RectangularSelectionBrush { get; private set; }
        public IBrush RectangularSelectionBorderBrush { get; private set; }

        public IBrush CircularReferenceBrush { get; private set; }
        public IBrush ValidConnectionBrush { get; private set; }
        public IBrush DraggedConnectionBrush { get; private set; }

        public IBrush ConnectionLineBrush { get; private set; }
        public IBrush DisconnectingLineBrush { get; private set; }

        public IBrush PluggedRegularBlockTitleTextBrush { get; private set; }
        public IBrush UnpluggedRegularBlockTitleTextBrush { get; private set; }
        public IBrush RegularBlockInputTextBrush { get; private set; }
        public IBrush RegularBlockOutputTextBrush { get; private set; }
        public IBrush RegularBlockShortcutInputTextBrush { get; private set; }
        public IBrush RegularBlockShortcutOutputTextBrush { get; private set; }

        public IBrush RegularBlockShadowBrush { get; private set; }

        public IBrush RegularUnselectedBlockBorderBrush { get; private set; }
        public IBrush RegularSelectedBlockBorderBrush { get; private set; }
        public IBrush RegularConnectedBlockBorderBrush { get; private set; }
        public IBrush CommentBlockBorderBrush { get; private set; }

        public IBrush RegularBlockBackgroundBrush { get; private set; }
        public IBrush RegularUnpluggedBlockBackgroundBrush { get; private set; }
        public IBrush RegularPluggedBlockBackgroundBrush { get; private set; }

        public IBrush ConstantBlockBackgroundBrush { get; private set; }
        public IBrush ConstantUnpluggedBlockBackgroundBrush { get; private set; }
        public IBrush ConstantPluggedBlockBackgroundBrush { get; private set; }

        public IBrush AutogenBlockBackgroundBrush { get; private set; }
        public IBrush AutogenUnpluggedBlockBackgroundBrush { get; private set; }
        public IBrush AutogenPluggedBlockBackgroundBrush { get; private set; }

        public IBrush GhostBlockBackgroundBrush { get; private set; }

        public IBrush InvalidBlockBackgroundBrush { get; private set; }

        public IBrush PlugBackgroundBrush { get; private set; }
        public IBrush PluggedPlugForegroundBrush { get; private set; }
        public IBrush UnpluggedPlugForegroundBrush { get; private set; }
        public IBrush InvalidBlockPluggedPlugForegroundBrush { get; private set; }
        public IBrush InvalidBlockUnpluggedPlugForegroundBrush { get; private set; }
        public IBrush UnusablePlugForegroundBrush { get; private set; }
        public IBrush CompatiblePlugForegroundBrush { get; private set; }
        public IBrush PreferedPlugForegroundBrush { get; private set; }
        public IBrush InvalidPlugForegroundBrush { get; private set; }

        public IBitmap InputNodeBitmap { get; private set; }
        public IBitmap OutputNodeBitmap { get; private set; }
        public IBitmap PreviewImageBitmap { get; private set; }
        public IBitmap PreviewErrorImageBitmap { get; private set; }

        public static System.Drawing.Color ShaderOKBackgroundColor = System.Drawing.Color.FromArgb(220, 255, 220);
        public static System.Drawing.Color ShaderNotOKBackgroundColor = System.Drawing.Color.FromArgb(255, 220, 220);

        public static System.Drawing.Color ShaderOKQuickInfoColor = System.Drawing.Color.LimeGreen;
        public static System.Drawing.Color ShaderNotOKQuickInfoColor = System.Drawing.Color.Red;
        public static System.Drawing.Color CompilerErrorQuickInfoColor = System.Drawing.Color.Orange;

        /// <summary>
        /// ListViewの各タグに対応するColor
        /// </summary>
        public static Dictionary<string, System.Drawing.Color> TagColors = new Dictionary<string, System.Drawing.Color>()
            {
                { "Input",   System.Drawing.Color.FromArgb(255,221,91,88) },
                { "Output",  System.Drawing.Color.FromArgb(255,65,65,65) },
                { "Texture", System.Drawing.Color.FromArgb(255,219,156,65) },
                { "Divide",  System.Drawing.Color.FromArgb(255,150,74,190) },
                { "Combine", System.Drawing.Color.FromArgb(255,71,172,77) },
                { "Convert", System.Drawing.Color.FromArgb(255,95,142,220) },
                { "Math",    System.Drawing.Color.FromArgb(255,95,142,220) },
                { "Utility", System.Drawing.Color.FromArgb(255,127,127,127) },
                { "Comment", System.Drawing.Color.FromArgb(200,127,127,127) },
            };

        /// <summary>
        /// デフォルトのタグのColor
        /// </summary>
        public static readonly System.Drawing.Color DefaultTagColor = System.Drawing.Color.FromArgb(150, 150, 150);

        /// <summary>
        /// ListViewをマウスオーバーした時のBlockのColor
        /// タグごとに分類している
        /// </summary>
        public static readonly Dictionary<string, Color[]> BlockBackgroundColors = new Dictionary<string, Color[]>()
            {
                {"Input",   new Color[] {new Color(242,  94, 102), new Color(182, 71, 77)}},
                {"Output",  new Color[] {new Color(235, 163,  54), new Color(176, 122, 41)}},
                {"Texture", new Color[] {new Color(232, 212,  35), new Color(174, 159, 26)}},
                {"Divide",  new Color[] {new Color(156, 212,  55), new Color(117, 159, 41)}},
                {"Combine", new Color[] {new Color( 70, 212, 162), new Color(53, 159, 122)}},
                {"Convert", new Color[] {new Color( 82, 181, 235), new Color(62, 136, 176)}},
                {"Math",    new Color[] {new Color(136, 120, 255), new Color(102, 90, 191)}},
                {"Utility", new Color[] {new Color(139, 139, 139), new Color(104, 104, 104)}},
                {"Comment", new Color[] {new Color( 64, 128, 64, 128), new Color( 64, 128, 64, 128)}},
            };

        /// <summary>
        /// 結線が不十分な場合のデフォルトのBlockのColor
        /// </summary>
        public static Color[] DefaultBlockBackgroundColor = new Color[] { new Color(150, 150, 150), new Color(120, 120, 120) };

        /// <summary>
        /// 結線が不十分な場合のBlockのColor
        /// タグごとに分類している
        /// </summary>
        public static Dictionary<string, System.Drawing.Color> UnpluggedBlockBackgroundColors = new Dictionary<string, System.Drawing.Color>()
            {
                { "Input",   System.Drawing.Color.FromArgb(255,196,80,80) },
                { "Output",  System.Drawing.Color.FromArgb(255,65,65,65) },
                { "Texture", System.Drawing.Color.FromArgb(255,219,156,65) },
                { "Divide",  System.Drawing.Color.FromArgb(255,150,74,190) },
                { "Combine", System.Drawing.Color.FromArgb(255,71,172,77) },
                { "Convert", System.Drawing.Color.FromArgb(255,95,142,220) },
                { "Math",    System.Drawing.Color.FromArgb(255,95,142,220) },
                { "Utility", System.Drawing.Color.FromArgb(255,127,127,127) },
                { "Comment", System.Drawing.Color.FromArgb(180,60,60,60) },
            };

        /// <summary>
        /// 結線されている場合のBlockのColor
        /// タグごとに分類している
        /// </summary>
        public static Dictionary<string, System.Drawing.Color> PluggedBlockBackgroundColors = new Dictionary<string, System.Drawing.Color>()
            {
                { "Input",   System.Drawing.Color.FromArgb(255,196,80,80) },
                { "Output",  System.Drawing.Color.FromArgb(255,65,65,65) },
                { "Texture", System.Drawing.Color.FromArgb(255,219,156,65) },
                { "Divide",  System.Drawing.Color.FromArgb(255,150,74,190) },
                { "Combine", System.Drawing.Color.FromArgb(255,71,172,77) },
                { "Convert", System.Drawing.Color.FromArgb(255,95,142,220) },
                { "Math",    System.Drawing.Color.FromArgb(255,95,142,220) },
                { "Utility", System.Drawing.Color.FromArgb(255,127,127,127) },
                { "Comment", System.Drawing.Color.FromArgb(180,60,60,60) },
            };

        /// <summary>
        /// 結線が不十分な場合のデフォルトのBlockのColor
        /// </summary>
        public static readonly System.Drawing.Color DefaultUnpluggedBlockBackgroundColor = System.Drawing.Color.FromArgb(164, 164, 164);

        /// <summary>
        /// 結線がされているデフォルトのBlockのColor
        /// </summary>
        public static readonly System.Drawing.Color DefaultPluggedBlockBackgroundColor = System.Drawing.Color.FromArgb(190, 190, 190);

        /// <summary>
        /// ListViewの各タグに対応するListViewItemのBackColor
        /// </summary>
        public static Dictionary<string, Brush> ListViewTagBrushes = new Dictionary<string, Brush>();

        /// <summary>
        /// ListViewのデフォルトのタグのBackColor
        /// </summary>
        public static SolidBrush ListViewItemDefaultBrush = new SolidBrush(DefaultTagColor);

        /// <summary>
        /// ListViewでのマウスオーバー時に表示されるBlockのBrush
        /// タグごとに分類している
        /// </summary>
        public Dictionary<string, IBrush> BlockBackgroundBrushes = new Dictionary<string, IBrush>();

        /// <summary>
        /// 結線が不十分な場合のBlockのBrush
        /// タグごとに分類している
        /// </summary>
        public Dictionary<string, IBrush> UnpluggedBlockBackgroundBrushes = new Dictionary<string, IBrush>();

        /// <summary>
        /// 結線されたBlockのBrush
        /// タグごとに分類している
        /// </summary>
        public Dictionary<string, IBrush> PluggedBlockBackgroundBrushes = new Dictionary<string, IBrush>();

        /// <summary>
        /// デフォルトのBlockのBrush
        /// </summary>
        public IBrush DefaultBlockBackgroundBrush { get; private set; }

        /// <summary>
        /// 結線が不十分な場合のBlockのBrush
        /// </summary>
        public IBrush DefaultPluggedBlockBackgroundBrush { get; private set; }

        /// <summary>
        /// 結線されたBlockのBrush
        /// </summary>
        public IBrush DefaultUnpluggedBlockBackgroundBrush { get; private set; }

        /// <summary>
        /// デフォルトのBrushとColorをセットする.
        /// </summary>
        /// <param name="renderer">使用するレンダラ</param>
        public void SetupDefault(Renderer renderer)
        {
            if (renderer == null)
                throw new ArgumentNullException("renderer");

            DragWindowWidth = SystemInformation.DragSize.Width;
            DragWindowHeight = SystemInformation.DragSize.Height;

            TitleTextHeight = 32.0;
            SubTitleTextHeight = 32.0;
            PlugTextHeight = 28.0;
            PlugRadius = 10.0;

            RegularBlockRadiusX = 6.0;
            RegularBlockRadiusY = 6.0;

            RegularSelectedBlockBorderStroke = 5.0;
            RegularUnselectedBlockBorderStroke = 2.0;

            RegularBlockShadowOffset = 5.0;
            RegularDraggedBlockLiftOffset = 3.0;

            RegularBlockTitleTextHorizontalPadding = 4.0;
            RegularBlockTitleTextVerticalPadding = 4.0;
            RegularBlockPreviewImageHorizontalPadding = 4.0;
            RegularBlockPreviewImageVerticalPadding = 4.0;
            RegularBlockDecorationHorizontalPadding = 8.0;
            RegularBlockDecorationVerticalPadding = 8.0;
            RegularBlockPlugsMinHorizontalSpacing = 8.0;

            ConnectionLineStroke = 4.0;

            RegularBlockTitleTextFormat = new TextFormat(new Font("メイリオ", 18.0));
            RegularBlockInputPlugTextFormat = new TextFormat(new Font("メイリオ", 14.0));
            RegularBlockOutputPlugTextFormat = RegularBlockInputPlugTextFormat;

            RegularBlockShortcutInputPlugTextFormat = new TextFormat(new Font("メイリオ", 11.0));
            RegularBlockShortcutOutputPlugTextFormat = RegularBlockShortcutInputPlugTextFormat;

            PluggedRegularBlockTitleTextBrush = renderer.CreateSolidColorBrush(System.Drawing.Color.FromArgb(255,255,253).ToRendererColor());
            UnpluggedRegularBlockTitleTextBrush = renderer.CreateSolidColorBrush(System.Drawing.Color.FromArgb(208,208,206).ToRendererColor());
            RegularBlockInputTextBrush = renderer.CreateSolidColorBrush(System.Drawing.Color.Black.ToRendererColor());
            RegularBlockOutputTextBrush = renderer.CreateSolidColorBrush(System.Drawing.Color.Black.ToRendererColor());
            RegularBlockShortcutInputTextBrush = renderer.CreateSolidColorBrush(System.Drawing.Color.Gray.ToRendererColor());
            RegularBlockShortcutOutputTextBrush = renderer.CreateSolidColorBrush(System.Drawing.Color.Gray.ToRendererColor());

            GridLineBrush = renderer.CreateSolidColorBrush(System.Drawing.Color.DarkGray.ToRendererColor());

            RectangularSelectionBrush = renderer.CreateSolidColorBrush(new Color(253, 154, 15, 32));
            RectangularSelectionBorderBrush = renderer.CreateSolidColorBrush(new Color(253, 154, 15, 255));

            CircularReferenceBrush = renderer.CreateSolidColorBrush(System.Drawing.Color.Red.ToRendererColor());
            ValidConnectionBrush = renderer.CreateSolidColorBrush(new Color(22, 205, 12, 255));
            DraggedConnectionBrush = renderer.CreateSolidColorBrush(new Color(123, 186, 243, 255));
            ConnectionLineBrush = renderer.CreateSolidColorBrush(System.Drawing.Color.White.ToRendererColor());
            DisconnectingLineBrush = renderer.CreateSolidColorBrush(System.Drawing.Color.DarkGray.ToRendererColor());

            PlugBackgroundBrush = renderer.CreateSolidColorBrush(System.Drawing.Color.WhiteSmoke.ToRendererColor());
            PluggedPlugForegroundBrush = renderer.CreateSolidColorBrush(new Color(123, 186, 243, 255));
            UnpluggedPlugForegroundBrush = renderer.CreateSolidColorBrush(System.Drawing.Color.DimGray.ToRendererColor());
            InvalidBlockPluggedPlugForegroundBrush = renderer.CreateSolidColorBrush(new Color(254, 119, 122, 255));
            InvalidBlockUnpluggedPlugForegroundBrush = renderer.CreateSolidColorBrush(new Color(202, 49, 53, 255));
            UnusablePlugForegroundBrush = renderer.CreateSolidColorBrush(System.Drawing.Color.LightGray.ToRendererColor());
            CompatiblePlugForegroundBrush = renderer.CreateSolidColorBrush(new Color(255, 199, 119, 255));
            PreferedPlugForegroundBrush = renderer.CreateSolidColorBrush(new Color(123, 248, 116, 255));
            InvalidPlugForegroundBrush = renderer.CreateSolidColorBrush(System.Drawing.Color.DarkGray.ToRendererColor());

            RegularBlockShadowBrush = renderer.CreateSolidColorBrush(System.Drawing.Color.FromArgb(50, 0, 0, 0).ToRendererColor());

            RegularUnselectedBlockBorderBrush = renderer.CreateSolidColorBrush(System.Drawing.Color.Black.ToRendererColor());
            RegularSelectedBlockBorderBrush = renderer.CreateSolidColorBrush(new Color(253, 154, 15, 255));
            RegularConnectedBlockBorderBrush = renderer.CreateSolidColorBrush(System.Drawing.Color.White.ToRendererColor());
            CommentBlockBorderBrush = renderer.CreateSolidColorBrush(System.Drawing.Color.FromArgb(66, 66, 66).ToRendererColor());

            InvalidBlockBackgroundBrush = renderer.CreateLinearGradientBrush(new Point(0.0, 0.0), new Point(0.0, 1.0),
                new GradientStop(new Color(202, 49, 53, 255), 0.0),
                new GradientStop(System.Drawing.Color.DarkGray.ToRendererColor(), 1.0));

            RegularBlockBackgroundBrush = renderer.CreateLinearGradientBrush(new Point(0.0, 0.0), new Point(0.0, 1.0),
                new GradientStop(new Color(123, 186, 243, 255), 0.0),
                new GradientStop(new Color(21, 96, 164, 255), 1.0));
            RegularUnpluggedBlockBackgroundBrush = renderer.CreateLinearGradientBrush(new Point(0.0, 0.0), new Point(0.0, 1.0),
                new GradientStop(new Color(196, 221, 243, 255), 0.0),
                new GradientStop(new Color(123, 186, 243, 255), 1.0));
            RegularPluggedBlockBackgroundBrush = renderer.CreateLinearGradientBrush(new Point(0.0, 0.0), new Point(0.0, 1.0),
                new GradientStop(new Color(123, 186, 243, 255), 0.0),
                new GradientStop(new Color(21, 96, 164, 255), 1.0));

            ConstantBlockBackgroundBrush = renderer.CreateLinearGradientBrush(new Point(0.0, 0.0), new Point(0.0, 1.0),
                new GradientStop(new Color(243, 243, 123, 255), 0.0),
                new GradientStop(new Color(168, 168, 64, 255), 1.0));
            ConstantUnpluggedBlockBackgroundBrush = renderer.CreateLinearGradientBrush(new Point(0.0, 0.0), new Point(0.0, 1.0),
                new GradientStop(new Color(250, 250, 150, 255), 0.0),
                new GradientStop(new Color(243, 243, 123, 255), 1.0));
            ConstantPluggedBlockBackgroundBrush = renderer.CreateLinearGradientBrush(new Point(0.0, 0.0), new Point(0.0, 1.0),
                new GradientStop(new Color(243, 243, 123, 255), 0.0),
                new GradientStop(new Color(168, 168, 64, 255), 1.0));

            AutogenBlockBackgroundBrush = renderer.CreateLinearGradientBrush(new Point(0.0, 0.0), new Point(0.0, 1.0),
                new GradientStop(new Color(123, 243, 186, 255), 0.0),
                new GradientStop(new Color(21, 164, 96, 255), 1.0));
            AutogenUnpluggedBlockBackgroundBrush = renderer.CreateLinearGradientBrush(new Point(0.0, 0.0), new Point(0.0, 1.0),
                new GradientStop(new Color(150, 250, 230, 255), 0.0),
                new GradientStop(new Color(123, 243, 186, 255), 1.0));
            AutogenPluggedBlockBackgroundBrush = renderer.CreateLinearGradientBrush(new Point(0.0, 0.0), new Point(0.0, 1.0),
                new GradientStop(new Color(123, 243, 186, 255), 0.0),
                new GradientStop(new Color(21, 164, 96, 255), 1.0));

            GhostBlockBackgroundBrush = renderer.CreateLinearGradientBrush(new Point(0.0, 0.0), new Point(0.0, 1.0),
                new GradientStop(new Color(125, 96, 164, 255), 0.0),
                new GradientStop(new Color(100, 50, 140, 255), 1.0));
        }

        /// <summary>
        /// BlockのBrushをセットする.
        /// </summary>
        /// <param name="renderer">使用するレンダラ</param>
        public void SetUpBlockBackgroundBrushes(Renderer renderer)
        {
            // タグカラーをもとに、Brushのインタンスを作る
            IPoint startPoint = new Point(0.0, 0.0);
            IPoint endPoint = new Point(0.0, 1.0);

            ListViewTagBrushes.Clear();
            foreach (var tagColor in TagColors)
            {
                ListViewTagBrushes.Add(tagColor.Key, new SolidBrush(tagColor.Value));
            }

            // 結線が不十分な場合のBlockのBrush
            this.UnpluggedBlockBackgroundBrushes.Clear();
            foreach (var blockColor in UnpluggedBlockBackgroundColors)
            {
                this.UnpluggedBlockBackgroundBrushes.Add(blockColor.Key,
                    renderer.CreateSolidColorBrush(new Color(blockColor.Value.R, blockColor.Value.G, blockColor.Value.B, blockColor.Value.A)));
            }

            // 結線されたBlockのBrush
            this.PluggedBlockBackgroundBrushes.Clear();
            foreach (var blockColor in PluggedBlockBackgroundColors)
            {
                this.PluggedBlockBackgroundBrushes.Add(blockColor.Key,
                    renderer.CreateSolidColorBrush(new Color(blockColor.Value.R, blockColor.Value.G, blockColor.Value.B, blockColor.Value.A)));
            }

            // ListViewのマウスオーバー時に表示されるBlockのBrush
            this.BlockBackgroundBrushes.Clear();
            foreach (var blockColor in TagColors)
            {
                this.BlockBackgroundBrushes.Add(blockColor.Key,
                    renderer.CreateSolidColorBrush(new Color(blockColor.Value.R, blockColor.Value.G, blockColor.Value.B)));
            }

            // デフォルトのBlockのBrush
            DefaultBlockBackgroundBrush = renderer.CreateSolidColorBrush(DefaultBlockBackgroundColor[0]);

            // 結線がされていないBlockのBrush
            DefaultUnpluggedBlockBackgroundBrush = renderer.CreateSolidColorBrush(new Color(DefaultUnpluggedBlockBackgroundColor.R,
                                                                                            DefaultUnpluggedBlockBackgroundColor.G,
                                                                                            DefaultUnpluggedBlockBackgroundColor.B));

            // 決戦されたBlockのBrush
            DefaultPluggedBlockBackgroundBrush = renderer.CreateSolidColorBrush(new Color(DefaultPluggedBlockBackgroundColor.R,
                                                                                          DefaultPluggedBlockBackgroundColor.G,
                                                                                          DefaultPluggedBlockBackgroundColor.B));
        }

        public void SetInputNodeBitmap(Renderer renderer, System.Drawing.Bitmap bitmap)
        {
            InputNodeBitmap = renderer.CreateBitmap(bitmap.ToRendererBitmapDefinition());
        }

        public void SetOutputNodeBitmap(Renderer renderer, System.Drawing.Bitmap bitmap)
        {
            OutputNodeBitmap = renderer.CreateBitmap(bitmap.ToRendererBitmapDefinition());
        }

        public void SetDefaultPreviewBitmap(Renderer renderer,System.Drawing.Bitmap bitmap)
        {
            PreviewImageBitmap = renderer.CreateBitmap(bitmap.ToRendererBitmapDefinition());
        }

        public void SetDefaultPreviewErrorBitmap(Renderer renderer, System.Drawing.Bitmap bitmap)
        {
            PreviewErrorImageBitmap = renderer.CreateBitmap(bitmap.ToRendererBitmapDefinition());
        }

        public void Dispose()
        {
            var disposableType = typeof(IDisposable);

            const BindingFlags flags =
                BindingFlags.Public |
                BindingFlags.NonPublic |
                BindingFlags.Instance;

            var properties = typeof(VisualResourceSet).GetProperties(flags)
                .Where(p => disposableType.IsAssignableFrom(p.PropertyType));

            foreach (var property in properties)
            {
                var disposable = (IDisposable)property.GetValue(this, null);
                if (disposable == null)
                    continue;

                disposable.Dispose();
                if (disposable.GetType().IsClass)
                    property.SetValue(this, null, null); // set to null
            }
        }
    }
}
