﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Xml.Linq;
using EffectCombiner.Core;
using System.Xml;
using Renderer2D.Core;

namespace EffectCombiner.Primitives
{
    public interface IXmlElementConsumer
    {
        Type InstanceType { get; }
        object ProduceInstance(XElement element, Renderer renderer);
    }

    public interface IXmlElementProducer
    {
        Type InstanceType { get; }
        XElement ProduceXmlElement(object instance, string id);
    }

    // -------------------------------

    public class TextFormatXmlElementProducer : IXmlElementProducer
    {
        public Type InstanceType
        {
            get { return typeof(ITextFormat); }
        }

        public XElement ProduceXmlElement(object instance, string id)
        {
            var textFormat = (ITextFormat)instance;

            return new XElement(id,
                new XAttribute("Name", textFormat.Font.FamilyName),
                new XAttribute("Size", textFormat.Font.Size));
        }
    }

    public class SolidColorBrushXmlElementProducer : IXmlElementProducer
    {
        public Type InstanceType
        {
            get { return typeof(ISolidColorBrush); }
        }

        public XElement ProduceXmlElement(object instance, string id)
        {
            var brush = (ISolidColorBrush)instance;

            return new XElement(id,
                new XAttribute("Color", VisualResourceParser.ColorToString(brush.Color)));
        }
    }

    public class LinearGradientBrushXmlElementProducer : IXmlElementProducer
    {
        public Type InstanceType
        {
            get { return typeof(ILinearGradientBrush); }
        }

        public XElement ProduceXmlElement(object instance, string id)
        {
            var brush = (ILinearGradientBrush)instance;

            return new XElement(id,
                new XAttribute("StartPoint", VisualResourceParser.PointToString(brush.StartPoint)),
                new XAttribute("EndPoint", VisualResourceParser.PointToString(brush.EndPoint)),
                    new XElement("GradientStops", brush.GradientStops.Select(gs =>
                        new XElement("GradientStop",
                            new XAttribute("Color", VisualResourceParser.ColorToString(gs.Color)),
                            new XAttribute("Offset", gs.Offset)))));
        }
    }

    public class RadialGradientBrushXmlElementProducer : IXmlElementProducer
    {
        public Type InstanceType
        {
            get { return typeof(IRadialGradientBrush); }
        }

        public XElement ProduceXmlElement(object instance, string id)
        {
            var brush = (IRadialGradientBrush)instance;

            return new XElement(id,
                new XAttribute("Center", VisualResourceParser.PointToString(brush.Center)),
                new XAttribute("GradientOriginOffset", VisualResourceParser.PointToString(brush.GradientOriginOffset)),
                new XAttribute("RadiusX", brush.RadiusX),
                new XAttribute("RadiusY", brush.RadiusY),
                    new XElement("GradientStops", brush.GradientStops.Select(gs =>
                        new XElement("GradientStop",
                            new XAttribute("Color", VisualResourceParser.ColorToString(gs.Color)),
                            new XAttribute("Offset", gs.Offset)))));
        }
    }

    public class ColorXmlElementProducer : IXmlElementProducer
    {
        public Type InstanceType
        {
            get { return typeof(IColor); }
        }

        public XElement ProduceXmlElement(object instance, string id)
        {
            var color = (IColor)instance;

            return new XElement(id,
                new XAttribute("Color", VisualResourceParser.ColorToString(color)));
        }
    }

    public class NumberXmlElementProducer : IXmlElementProducer
    {
        public Type InstanceType
        {
            get { return typeof(double); }
        }

        public XElement ProduceXmlElement(object instance, string id)
        {
            var value = (double)instance;

            return new XElement(id,
                new XAttribute("Value", value));
        }
    }

    public class BitmapXmlElementProducer : IXmlElementProducer
    {
        public Type InstanceType
        {
            get { return typeof(IBitmap); }
        }

        public XElement ProduceXmlElement(object instance, string id)
        {
            return null;
        }
    }

     // -------------------------------

    public abstract class XmlElementConsumerBase : IXmlElementConsumer
    {
        public IEventReporter EventReporter { get; internal set; }

        protected void AttributeMissing(XElement element, string attributeName)
        {
            var lineInfo = (IXmlLineInfo)element;
            // blabla
        }

        public abstract Type InstanceType { get; }
        public abstract object ProduceInstance(XElement element, Renderer renderer);
    }

    public class NumberXmlElementConsumer : XmlElementConsumerBase
    {
        public override Type InstanceType
        {
            get { return typeof(double); }
        }

        public override object ProduceInstance(XElement element, Renderer renderer)
        {
            var isValid = true;

            var attrValue = element.Attribute("Value");
            if (attrValue == null)
            {
                AttributeMissing(element, "Value");
                isValid = false;
            }

            if (isValid == false)
                return null;

            return (double)attrValue;
        }
    }

    public class TextFormatXmlElementConsumer : XmlElementConsumerBase
    {
        public override Type InstanceType
        {
            get { return typeof(ITextFormat); }
        }

        public XElement ProduceXmlElement(object instance, string id)
        {
            var textFormat = (ITextFormat)instance;

            return new XElement(id,
                new XAttribute("Name", textFormat.Font.FamilyName),
                new XAttribute("Size", textFormat.Font.Size));
        }

        public override object ProduceInstance(XElement element, Renderer renderer)
        {
            var isValid = true;

            var attrName = element.Attribute("Name");
            if (attrName == null)
            {
                AttributeMissing(element, "Name");
                isValid = false;
            }

            var attrSize = element.Attribute("Size");
            if (attrSize == null)
            {
                AttributeMissing(element, "Size");
                isValid = false;
            }

            if (isValid == false)
                return null;

            return new TextFormat(new Font((string)attrName, (double)attrSize));
        }
    }

    public class ColorXmlElementConsumer : XmlElementConsumerBase
    {
        public override Type InstanceType
        {
            get { return typeof(IColor); }
        }

        public override object ProduceInstance(XElement element, Renderer renderer)
        {
            var isValid = true;

            var attrColor = element.Attribute("Color");
            if (attrColor == null)
            {
                AttributeMissing(element, "Color");
                isValid = false;
            }

            if (isValid == false)
                return null;

            return VisualResourceParser.StringToColor((string)attrColor);
        }
    }

    public class BrushXmlElementConsumer : XmlElementConsumerBase
    {
        public override Type InstanceType
        {
            get { return typeof(IBrush); }
        }

        public override object ProduceInstance(XElement element, Renderer renderer)
        {
            var brush = GetRadialGradientBrush(element, renderer);
            if (brush == null)
            {
                brush = GetLinearGradientBrush(element, renderer);
                if (brush == null)
                    brush = GetSolidColorBrush(element, renderer);
            }

            return brush;
        }

        private IBrush GetSolidColorBrush(XElement element, Renderer renderer)
        {
            var attrColor = element.Attribute("Color");
            if (attrColor == null)
                return null;

            return renderer.CreateSolidColorBrush(VisualResourceParser.StringToColor((string)attrColor));
        }

        private IBrush GetLinearGradientBrush(XElement element, Renderer renderer)
        {
            var attrStartPoint = element.Attribute("StartPoint");
            if (attrStartPoint == null)
                return null;

            var attrEndPoint = element.Attribute("EndPoint");
            if (attrEndPoint == null)
                return null;

            var elemGradientStops = element.Element("GradientStops");
            if (elemGradientStops == null)
                return null;

            var gradientStops = new List<GradientStop>();
            foreach (var gs in elemGradientStops.Elements("GradientStop"))
            {
                var gradientStop = GetGradientStop(gs);
                if (gradientStop == null)
                    return null;

                gradientStops.Add(gradientStop);
            }

            if (gradientStops.Count <= 1)
                return null;

            return renderer.CreateLinearGradientBrush(
                VisualResourceParser.StringToPoint((string)attrStartPoint),
                VisualResourceParser.StringToPoint((string)attrEndPoint),
                gradientStops.Cast<IGradientStop>().ToArray());
        }

        private IBrush GetRadialGradientBrush(XElement element, Renderer renderer)
        {
            var attrCenter = element.Attribute("Center");
            if (attrCenter == null)
                return null;

            var attrGradientOriginOffset = element.Attribute("GradientOriginOffset");
            if (attrGradientOriginOffset == null)
                return null;

            var attrRadiusX = element.Attribute("RadiusX");
            if (attrRadiusX == null)
                return null;

            var attrRadiusY = element.Attribute("RadiusY");
            if (attrRadiusY == null)
                return null;

            var elemGradientStops = element.Element("GradientStops");
            if (elemGradientStops == null)
                return null;

            var gradientStops = new List<GradientStop>();
            foreach (var gs in elemGradientStops.Elements("GradientStop"))
            {
                var gradientStop = GetGradientStop(gs);
                if (gradientStop == null)
                    return null;

                gradientStops.Add(gradientStop);
            }

            if (gradientStops.Count <= 1)
                return null;

            return renderer.CreateRadialGradientBrush(
                VisualResourceParser.StringToPoint((string)attrCenter),
                VisualResourceParser.StringToPoint((string)attrGradientOriginOffset),
                (double)attrRadiusX,
                (double)attrRadiusY,
                gradientStops.Cast<IGradientStop>().ToArray());
        }

        private GradientStop GetGradientStop(XElement gradientStop)
        {
            var isValid = true;

            var attrColor = gradientStop.Attribute("Color");
            if (attrColor == null)
            {
                AttributeMissing(gradientStop, "Color");
                isValid = false;
            }

            var attrOffset = gradientStop.Attribute("Offset");
            if (attrOffset == null)
            {
                AttributeMissing(gradientStop, "Offset");
                isValid = false;
            }

            if (isValid == false)
                return null;

            return new GradientStop(VisualResourceParser.StringToColor((string)attrColor), (double)attrOffset);
        }
    }

    // -====================================================================-

    public class VisualResourceParser
    {
        public static string ColorToString(IColor color)
        {
            var strComponents = new[] { color.R, color.G, color.B, color.A }
                .Select(XmlConvert.ToString);

            return string.Join(";", strComponents);
        }

        public static IColor StringToColor(string color)
        {
            var elements = color.Split(';');

            return new Color(
                XmlConvert.ToByte(elements[0]),
                XmlConvert.ToByte(elements[1]),
                XmlConvert.ToByte(elements[2]),
                XmlConvert.ToByte(elements[3]));
        }

        public static string PointToString(IPoint point)
        {
            return string.Format("{0};{1}", XmlConvert.ToString(point.X), XmlConvert.ToString(point.Y));
        }

        public static IPoint StringToPoint(string point)
        {
            var elements = point.Split(';');
            return new Point(XmlConvert.ToDouble(elements[0]), XmlConvert.ToDouble(elements[1]));
        }

        private readonly IXmlElementConsumer[] consumers = new IXmlElementConsumer[]
        {
            new NumberXmlElementConsumer(),
            new TextFormatXmlElementConsumer(),
            new ColorXmlElementConsumer(),
            new BrushXmlElementConsumer(),
            new TextFormatXmlElementConsumer(),
        };

        private readonly IXmlElementProducer[] producers = new IXmlElementProducer[]
            {
                new BitmapXmlElementProducer(),
                new TextFormatXmlElementProducer(),
                new SolidColorBrushXmlElementProducer(),
                new LinearGradientBrushXmlElementProducer(),
                new RadialGradientBrushXmlElementProducer(),
                new ColorXmlElementProducer(),
                new NumberXmlElementProducer(),
            };

        public void GenerateFile(string filename, VisualResourceSet resources)
        {
            var root = new XElement("VisualResourceSet");

            foreach (var propertyInfo in GetAllProperties())
            {
                var instance = propertyInfo.GetValue(resources, null);
                if (instance == null)
                    continue;

                var instanceType = instance.GetType();

                var found = false;
                foreach (var producer in producers)
                {
                    if (producer.InstanceType.IsAssignableFrom(instanceType))
                    {
                        found = true;

                        var element = producer.ProduceXmlElement(instance, propertyInfo.Name);
                        if (element == null)
                            continue; // skipped type

                        root.Add(element);

                        break;
                    }
                }

                if (found == false)
                {
                    // unknown type
                }
            }

            root.Save(filename);
        }

        public void LoadFile(string filename, Renderer renderer, VisualResourceSet resources)
        {
            var root = XElement.Load(filename);
            var properties = GetAllProperties().ToArray();

            foreach (var element in root.Elements())
            {
                var property = properties.FirstOrDefault(pi => pi.Name == element.Name);
                if (property == null)
                {
                    // TODO: unknown property
                    continue;
                }

                var consumer = consumers.FirstOrDefault(c => c.InstanceType.IsAssignableFrom(property.PropertyType));
                if (consumer == null)
                {
                    // TODO: unknown type
                    continue;
                }

                var instance = consumer.ProduceInstance(element, renderer);
                if (instance == null)
                {
                    // TODO: could not read value(s)
                    continue;
                }

                property.SetValue(resources, instance, null);
            }

            Globals.Options.PostProcessValues();
        }

        private static IEnumerable<PropertyInfo> GetAllProperties()
        {
            const BindingFlags flags =
                BindingFlags.Public |
                BindingFlags.NonPublic |
                BindingFlags.Instance;

            return typeof(VisualResourceSet).GetProperties(flags);
        }
    }
}
