﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml.Serialization;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;

using EffectCombiner.Core;

namespace EffectCombiner.Primitives
{
    public static class OptionsManager
    {
        private static Options instance;
        private static string optionsFilename;

        public static void Save()
        {
            var ns = new XmlSerializerNamespaces();
            ns.Add("", "");

            using (var fs = File.Open(optionsFilename, FileMode.Create, FileAccess.Write))
            {
                var serializer = new XmlSerializer(typeof(Options), "");
                serializer.Serialize(fs, instance, ns);
            }
        }

        /// <summary>
        /// オプションを複製します。
        /// </summary>
        /// <returns>複製したオプションを返します。</returns>
        public static Options CloneOptions()
        {
            if (instance == null)
                return null;

            var ns = new XmlSerializerNamespaces();
            ns.Add("", "");

            Options options;

            using (Stream stream = new MemoryStream())
            {
                var serializer = new XmlSerializer(typeof(Options), "");
                serializer.Serialize(stream, instance, ns);
                stream.Position = 0;
                options = (Options)serializer.Deserialize(stream);
            }

            if (instance.ProjectSettings != null)
            {
                ProjectSettings projectSettings = null;

                using (Stream stream = new MemoryStream())
                {
                    var serializer = new XmlSerializer(typeof(ProjectSettings), "");
                    serializer.Serialize(stream, instance.ProjectSettings, ns);
                    stream.Position = 0;
                    projectSettings = (ProjectSettings)serializer.Deserialize(stream);
                }

                if (projectSettings != null)
                {
                    options.ProjectSettings = projectSettings;
                }
            }

            return options;
        }

        public static bool Initialize(string projectFilePath)
        {
            var assemblyFilename = System.Reflection.Assembly.GetEntryAssembly().Location;
            var path = Path.GetDirectoryName(assemblyFilename);

            optionsFilename = string.Format("{0}\\options.xml", path);

            bool success = true;
            if (File.Exists(optionsFilename) == false)
            {
                instance = new Options();
            }
            else
            {
                try
                {
                    using (var fs = File.OpenRead(optionsFilename))
                    {
                        var serializer = new XmlSerializer(typeof(Options));
                        instance = serializer.Deserialize(fs) as Options;
                    }
                }
                catch
                {
                    instance = new Options();
                    success = false;
                }
            }

            // メイン関数名が未設定のときはデフォルトの "main" にする
            if (string.IsNullOrWhiteSpace(instance.ShaderGeneration.MainFunctionName))
            {
                instance.ShaderGeneration.MainFunctionName = "main";
            }

            // プロジェクト設定ファイルを読み込む
            if (!String.IsNullOrEmpty(projectFilePath))
            {
                instance.EnvironmentSettings.ProjectSettingFilePath = projectFilePath;
            }

            if (!String.IsNullOrEmpty(instance.EnvironmentSettings.ProjectSettingFilePath))
            {
                ProjectSettings projectSettings = LoadProjectSetting(instance.EnvironmentSettings.ProjectSettingFilePath);

                if (projectSettings != null)
                {
                    instance.ProjectSettings = projectSettings;
                }
            }

            Globals.SetOptions(instance);

            return success;
        }

        /// <summary>
        /// プロジェクト設定ファイルを読み込みます。
        /// </summary>
        /// <param name="filePath">プロジェクト設定ファイルのパス</param>
        /// <returns>プロジェクト設定を返します。</returns>
        public static ProjectSettings LoadProjectSetting(string filePath)
        {
            string basePath = AppDomain.CurrentDomain.BaseDirectory;
            string fullPath = PathUtility.ToAbsolutePath(filePath, basePath);

            if (string.IsNullOrEmpty(fullPath))
            {
                return null;
            }

            ProjectSettings projectSettings = null;

            try
            {
                using (var fs = File.OpenRead(fullPath))
                {
                    var serializer = new XmlSerializer(typeof(ProjectSettings));
                    projectSettings = serializer.Deserialize(fs) as ProjectSettings;
                }
            }
            catch
            {
                return null;
            }

            var ns = new XmlSerializerNamespaces();
            ns.Add("", "");

            using (var fs = File.Open(optionsFilename, FileMode.Create, FileAccess.Write))
            {
                var serializer = new XmlSerializer(typeof(Options), "");
                serializer.Serialize(fs, instance, ns);
            }

            return projectSettings;
        }

        /// <summary>
        /// プロジェクト設定ファイルを保存します。
        /// </summary>
        /// <param name="filePath">ファイルパス</param>
        /// <param name="projectSettings">プロジェクト設定</param>
        public static void SaveProjectSetting(string filePath, ProjectSettings projectSettings)
        {
            string basePath = AppDomain.CurrentDomain.BaseDirectory;
            string fullPath = PathUtility.ToAbsolutePath(filePath, basePath);

            try
            {
                if (File.Exists(fullPath))
                {
                    File.Copy(fullPath, string.Format("{0}.backup", fullPath));
                }
            }
            catch
            {
                // 失敗しても無視する
            }

            var ns = new XmlSerializerNamespaces();
            ns.Add("", "");

            using (var fs = File.Open(fullPath, FileMode.Create, FileAccess.Write))
            {
                var serializer = new XmlSerializer(typeof(ProjectSettings), "");
                serializer.Serialize(fs, projectSettings, ns);
            }
        }
    }

    public static class OptionsExtensions
    {
        public static void RecordFormWindowState(WindowSettings windowSettings, System.Windows.Forms.Form form)
        {
            bool fullscreen = (form.WindowState == System.Windows.Forms.FormWindowState.Maximized);
            if (fullscreen == false)
            {
                windowSettings.Left = form.Left;
                windowSettings.Top = form.Top;
                windowSettings.Width = form.Width;
                windowSettings.Height = form.Height;
            }
            windowSettings.FullScreen = fullscreen;
        }

        public static void ApplyRecordedFormWindowState(WindowSettings windowSettings, System.Windows.Forms.Form form)
        {
            if (windowSettings.Left > 0)
                form.Left = windowSettings.Left;

            if (windowSettings.Top > 0)
                form.Top = windowSettings.Top;

            if (windowSettings.Width > 0)
                form.Width = windowSettings.Width;

            if (windowSettings.Height > 0)
                form.Height = windowSettings.Height;

            form.WindowState = windowSettings.FullScreen
                ? System.Windows.Forms.FormWindowState.Maximized
                : System.Windows.Forms.FormWindowState.Normal;
        }
    }
}
