﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using Blocks.Core;
using EffectCombiner.Primitives.Generation;
using EffectCombiner.Primitives.Generation.Semantic;
using Renderer2D.Core;
using EffectDefinitions;
using EffectCombiner.Primitives.Blocks;
using System.Collections.Generic;

namespace EffectCombiner.Primitives.Operations
{
    public class AddBlockOperation : OperationBase
    {
        /// <summary>
        /// 処理対象のブロックエレメントの情報です。
        /// </summary>
        private readonly BlockElementInfo[] blockElementInfos;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="blockManager">ブロックマネージャ</param>
        /// <param name="blockElement">ブロックエレメント</param>
        public AddBlockOperation(BlockManagerBase blockManager, BlockElementBase blockElement)
            : this(blockManager, new BlockElementBase[] { blockElement })
        {
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="blockManager">ブロックマネージャ</param>
        /// <param name="blockElements">ブロクケレメントリスト</param>
        public AddBlockOperation(BlockManagerBase blockManager, IEnumerable<BlockElementBase> blockElements)
            : base(OperationType.AddBlock, blockManager)
        {
            if (blockElements == null)
                throw new ArgumentNullException("blockElement");

            this.blockElementInfos = new BlockElementInfo[blockElements.Count()];

            int blockElementIndex = 0;

            foreach (var blockElement in blockElements)
            {
                BlockElementInfo info = new BlockElementInfo();

                info.InstanceIdentifier = blockElement.InstanceIdentifier;
                info.BlockDefinition = blockElement.BlockDefinition;
                info.Position = new Point(blockElement.Left, blockElement.Top);

                var constElement = blockElement as ConstantBlockElement;
                if (constElement != null) info.ConstValue = constElement.Values;

                var commentElement = blockElement as CommentBlockElement;
                if (commentElement != null)
                {
                    info.CommentText = commentElement.Text;
                    info.Size = new Size(commentElement.Width, commentElement.Height);
                }
                else
                {
                    info.Size = null;
                }

                this.blockElementInfos[blockElementIndex] = info;
                ++blockElementIndex;
            }
        }

        /// <summary>
        /// 編集操作をアンドゥします。
        /// </summary>
        public override void Rollback()
        {
            this.BlockManager.ClearSelections();

            foreach (var info in this.blockElementInfos)
            {
                var blockElement = FindBlock(info.InstanceIdentifier);
                this.BlockManager.RemoveBlock(blockElement);
            }
        }

        /// <summary>
        /// 編集操作を実行します。
        /// </summary>
        public override void Execute()
        {
            this.BlockManager.ClearSelections();

            foreach (var info in this.blockElementInfos)
            {
                var newBlockElement = this.BlockManager.CreateBlockElement(info.BlockDefinition);
                newBlockElement.SetNewInstanceIdentifier(info.InstanceIdentifier);

                var manager = this.BlockManager as CompositionBlockManager;

                // セマンティックを展開する
                if (manager != null)
                {
                    var expandedBlocks = SemanticHelper.ExpandSemantics(BlockManager, Generation.Globals.BlockDefinitions.Values, (EffectBlockElementBase)newBlockElement);

                    foreach (var eb in expandedBlocks)
                    {
                        this.BlockManager.AddBlock(eb);
                    }
                }

                newBlockElement.SetPosition(info.Position.X, info.Position.Y);

                var element = newBlockElement as ConstantBlockElement;
                if (element != null) element.Values = info.ConstValue;

                var commentElement = newBlockElement as CommentBlockElement;
                if (commentElement != null) commentElement.Text = info.CommentText;

                this.BlockManager.AddBlock(newBlockElement);
                newBlockElement.IsSelected = true;
            }
        }

        /// <summary>
        /// 処理対象のブロックエレメントの情報です。
        /// </summary>
        private class BlockElementInfo
        {
            /// <summary>
            /// 識別子を取得または設定します。
            /// </summary>
            public uint InstanceIdentifier { get; set; }

            /// <summary>
            /// 位置を取得または設定します。
            /// </summary>
            public IPoint Position { get; set; }

            /// <summary>
            /// サイズを取得または設定します。
            /// </summary>
            public ISize Size { get; set; }

            /// <summary>
            /// ブロック定義データを取得または設定します。
            /// </summary>
            public BlockDefinition BlockDefinition;

            /// <summary>
            /// constant 値を取得または設定します。
            /// </summary>
            public string[,] ConstValue { get; set; }

            /// <summary>
            /// コメントを取得または設定します。
            /// </summary>
            public string CommentText { get; set; }
        }
    }
}
