﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Globalization;

namespace EffectCombiner.Primitives
{
    public enum PrimitiveShaderType
    {
        Boolean,
        UnsignedInteger,
        Integer,
        Float,
        Double,
    }

    public enum ShaderType
    {
        Boolean,
        Integer,
        UnsignedInteger,
        Float,
        Double,

        BooleanVector2,
        BooleanVector3,
        BooleanVector4,

        IntegerVector2,
        IntegerVector3,
        IntegerVector4,

        UnsignedIntegerVector2,
        UnsignedIntegerVector3,
        UnsignedIntegerVector4,

        FloatVector2,
        FloatVector3,
        FloatVector4,

        DoubleVector2,
        DoubleVector3,
        DoubleVector4,

        Matrix2By2,
        Matrix3By2,
        Matrix4By2,

        Matrix2By3,
        Matrix3By3,
        Matrix4By3,

        Matrix2By4,
        Matrix3By4,
        Matrix4By4,
    }

    public static class GlslTypingUtility
    {
        public static string FormatNumericValue(PrimitiveShaderType shaderType, int baseValue)
        {
            switch (shaderType)
            {
                case PrimitiveShaderType.Float:
                    return string.Format("{0}.0f", baseValue);
                case PrimitiveShaderType.Double:
                    return string.Format("{0}.0", baseValue);
            }
            return baseValue.ToString(CultureInfo.InvariantCulture);
        }

        public static string GetValueExpression(ShaderType shaderType, object[,] values)
        {
            var dimensionX = GetDimensionX(shaderType);
            var dimensionY = GetDimensionY(shaderType);
            var typeName = GetGlslTypeNameFromShaderType(shaderType);

            if (dimensionX == 1 && dimensionY == 1)
                return values[0, 0].ToString();

            var list = new List<string>();
            for (var y = 0; y < dimensionY; y++)
            {
                for (var x = 0; x < dimensionX; x++)
                    list.Add(values[x, y].ToString());
            }

            return string.Format("{0}({1})", typeName, string.Join(", ", list));
        }

        public static string GetValueExpression(ShaderType shaderType, string[] values)
        {
            if (values == null)
                return "?";

            var dimensionX = GetDimensionX(shaderType);
            var dimensionY = GetDimensionY(shaderType);
            var typeName = GetGlslTypeNameFromShaderType(shaderType);

            if (dimensionX == 1 && dimensionY == 1)
                return values[0];

            return string.Format("{0}({1})", typeName, string.Join(", ", values));
        }

        public static int GetDimensionX(ShaderType shaderType)
        {
            switch (shaderType)
            {
                case ShaderType.Boolean:
                case ShaderType.Integer:
                case ShaderType.UnsignedInteger:
                case ShaderType.Float:
                case ShaderType.Double:
                    return 1;

                case ShaderType.BooleanVector2:
                case ShaderType.IntegerVector2:
                case ShaderType.UnsignedIntegerVector2:
                case ShaderType.FloatVector2:
                case ShaderType.DoubleVector2:
                    return 2;

                case ShaderType.BooleanVector3:
                case ShaderType.IntegerVector3:
                case ShaderType.UnsignedIntegerVector3:
                case ShaderType.FloatVector3:
                case ShaderType.DoubleVector3:
                    return 3;

                case ShaderType.BooleanVector4:
                case ShaderType.IntegerVector4:
                case ShaderType.UnsignedIntegerVector4:
                case ShaderType.FloatVector4:
                case ShaderType.DoubleVector4:
                    return 4;

                case ShaderType.Matrix2By2:
                case ShaderType.Matrix2By3:
                case ShaderType.Matrix2By4:
                    return 2;

                case ShaderType.Matrix3By2:
                case ShaderType.Matrix3By3:
                case ShaderType.Matrix3By4:
                    return 3;

                case ShaderType.Matrix4By2:
                case ShaderType.Matrix4By3:
                case ShaderType.Matrix4By4:
                    return 4;
            }

            throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "shaderType"), "shaderType");
        }

        public static int GetDimensionY(ShaderType shaderType)
        {
            switch (shaderType)
            {
                case ShaderType.Boolean:
                case ShaderType.Integer:
                case ShaderType.UnsignedInteger:
                case ShaderType.Float:
                case ShaderType.Double:
                    return 1;

                case ShaderType.BooleanVector2:
                case ShaderType.IntegerVector2:
                case ShaderType.UnsignedIntegerVector2:
                case ShaderType.FloatVector2:
                case ShaderType.DoubleVector2:
                    return 1;

                case ShaderType.BooleanVector3:
                case ShaderType.IntegerVector3:
                case ShaderType.UnsignedIntegerVector3:
                case ShaderType.FloatVector3:
                case ShaderType.DoubleVector3:
                    return 1;

                case ShaderType.BooleanVector4:
                case ShaderType.IntegerVector4:
                case ShaderType.UnsignedIntegerVector4:
                case ShaderType.FloatVector4:
                case ShaderType.DoubleVector4:
                    return 1;

                case ShaderType.Matrix2By2:
                case ShaderType.Matrix3By2:
                case ShaderType.Matrix4By2:
                    return 2;

                case ShaderType.Matrix2By3:
                case ShaderType.Matrix3By3:
                case ShaderType.Matrix4By3:
                    return 3;

                case ShaderType.Matrix2By4:
                case ShaderType.Matrix3By4:
                case ShaderType.Matrix4By4:
                    return 4;
            }

            throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "shaderType"), "shaderType");
        }

        public static PrimitiveShaderType GetPrimitiveShaderType(ShaderType shaderType)
        {
            switch (shaderType)
            {
                case ShaderType.Boolean: return PrimitiveShaderType.Boolean;
                case ShaderType.Integer: return PrimitiveShaderType.Integer;
                case ShaderType.UnsignedInteger: return PrimitiveShaderType.UnsignedInteger;
                case ShaderType.Float: return PrimitiveShaderType.Float;
                case ShaderType.Double: return PrimitiveShaderType.Double;

                case ShaderType.BooleanVector2:
                case ShaderType.BooleanVector3:
                case ShaderType.BooleanVector4:
                    return PrimitiveShaderType.Boolean;

                case ShaderType.IntegerVector2:
                case ShaderType.IntegerVector3:
                case ShaderType.IntegerVector4:
                    return PrimitiveShaderType.Integer;

                case ShaderType.UnsignedIntegerVector2:
                case ShaderType.UnsignedIntegerVector3:
                case ShaderType.UnsignedIntegerVector4:
                    return PrimitiveShaderType.UnsignedInteger;

                case ShaderType.FloatVector2:
                case ShaderType.FloatVector3:
                case ShaderType.FloatVector4:
                    return PrimitiveShaderType.Float;

                case ShaderType.DoubleVector2:
                case ShaderType.DoubleVector3:
                case ShaderType.DoubleVector4:
                    return PrimitiveShaderType.Double;

                case ShaderType.Matrix2By2:
                case ShaderType.Matrix3By2:
                case ShaderType.Matrix4By2:
                case ShaderType.Matrix2By3:
                case ShaderType.Matrix3By3:
                case ShaderType.Matrix4By3:
                case ShaderType.Matrix2By4:
                case ShaderType.Matrix3By4:
                case ShaderType.Matrix4By4:
                    return PrimitiveShaderType.Float;
            }

            throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "shaderType"), "shaderType");
        }

        public static string GetGlslTypeNameFromShaderType(ShaderType shaderType)
        {
            switch (shaderType)
            {
                case ShaderType.Boolean: return "bool";
                case ShaderType.Integer: return "int";
                case ShaderType.UnsignedInteger: return "uint";
                case ShaderType.Float: return "float";
                case ShaderType.Double: return "double";

                case ShaderType.BooleanVector2: return "bvec2";
                case ShaderType.BooleanVector3: return "bvec3";
                case ShaderType.BooleanVector4: return "bvec4";

                case ShaderType.IntegerVector2: return "ivec2";
                case ShaderType.IntegerVector3: return "ivec3";
                case ShaderType.IntegerVector4: return "ivec4";

                case ShaderType.UnsignedIntegerVector2: return "uvec2";
                case ShaderType.UnsignedIntegerVector3: return "uvec3";
                case ShaderType.UnsignedIntegerVector4: return "uvec4";

                case ShaderType.FloatVector2: return "vec2";
                case ShaderType.FloatVector3: return "vec3";
                case ShaderType.FloatVector4: return "vec4";

                case ShaderType.DoubleVector2: return "dvec2";
                case ShaderType.DoubleVector3: return "dvec3";
                case ShaderType.DoubleVector4: return "dvec4";

                case ShaderType.Matrix2By2: return "mat2";
                case ShaderType.Matrix3By2: return "mat3x2";
                case ShaderType.Matrix4By2: return "mat4x2";
                case ShaderType.Matrix2By3: return "mat2x3";
                case ShaderType.Matrix3By3: return "mat3";
                case ShaderType.Matrix4By3: return "mat4x3";
                case ShaderType.Matrix2By4: return "mat2x4";
                case ShaderType.Matrix3By4: return "mat3x4";
                case ShaderType.Matrix4By4: return "mat4";
            }

            throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "shaderType"), "shaderType");
        }

        public static ShaderType GetShaderTypeFromGlslTypeName(string glslShaderTypeName)
        {
            switch (glslShaderTypeName)
            {
                case "bool": return ShaderType.Boolean;
                case "int": return ShaderType.Integer;
                case "uint": return ShaderType.UnsignedInteger;
                case "float": return ShaderType.Float;
                case "double": return ShaderType.Double;

                case "bvec2": return ShaderType.BooleanVector2;
                case "bvec3": return ShaderType.BooleanVector3;
                case "bvec4": return ShaderType.BooleanVector4;

                case "ivec2": return ShaderType.IntegerVector2;
                case "ivec3": return ShaderType.IntegerVector3;
                case "ivec4": return ShaderType.IntegerVector4;

                case "uvec2": return ShaderType.UnsignedIntegerVector2;
                case "uvec3": return ShaderType.UnsignedIntegerVector3;
                case "uvec4": return ShaderType.UnsignedIntegerVector4;

                case "vec2": return ShaderType.FloatVector2;
                case "vec3": return ShaderType.FloatVector3;
                case "vec4": return ShaderType.FloatVector4;

                case "dvec2": return ShaderType.DoubleVector2;
                case "dvec3": return ShaderType.DoubleVector3;
                case "dvec4": return ShaderType.DoubleVector4;

                case "mat2":
                case "mat2x2": return ShaderType.Matrix2By2;
                case "mat3x2": return ShaderType.Matrix3By2;
                case "mat4x2": return ShaderType.Matrix4By2;
                case "mat2x3": return ShaderType.Matrix2By3;
                case "mat3":
                case "mat3x3": return ShaderType.Matrix3By3;
                case "mat4x3": return ShaderType.Matrix4By3;
                case "mat2x4": return ShaderType.Matrix2By4;
                case "mat3x4": return ShaderType.Matrix3By4;
                case "mat4":
                case "mat4x4": return ShaderType.Matrix4By4;
            }

            throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "glslShaderTypeName"), "glslShaderTypeName");
        }
    }
}
