﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Blocks.Core;
using EffectCombiner.Primitives.Generation;
using Workflow.Core;

namespace EffectCombiner.Primitives.Blocks
{
    /// <summary>
    /// プラグの接続を行うユーティリティです。
    /// </summary>
    public static class ConnectionUtility
    {
        /// <summary>
        /// 指定のブロックエレメントの入力接続状態を復元します。
        /// </summary>
        /// <param name="blockElement">ブロックエレメント</param>
        /// <param name="inputConnections">復元する入力接続情報</param>
        /// <param name="findBlock">ブロックエレメントの検索関数</param>
        public static void RestoreInputConnections(EffectBlockElementBase blockElement, IEnumerable<InputPlugInfo> inputConnections, Func<uint, EffectBlockElementBase> findBlock)
        {
            // 指定のブロックエレメントについて、入力接続状態を復元する
            foreach (var inputConnection in inputConnections)
            {
                // 指定のブロックエレメントの入力プラグ数をチェック
                if (inputConnection.Index >= blockElement.WorkflowItem.InputPlugs.Length)
                {
                    break;
                }

                var remoteBlockElement = findBlock(inputConnection.Remote.InstanceIdentifier);

                // 出力元のブロックエレメントの出力プラグ数をチェック
                if (remoteBlockElement.WorkflowItem.OutputPlugs.Length <= inputConnection.Remote.Index)
                {
                    continue;
                }

                // プラグ間の適性をチェック
                var connectability = ConnectionUtility.DetermineConnectability(
                    remoteBlockElement, inputConnection.Remote.Index,
                    blockElement, inputConnection.Index);

                if (connectability == Connectability.Invalid)
                {
                    continue;
                }

                var outputPlug = remoteBlockElement.WorkflowItem.OutputPlugs[inputConnection.Remote.Index];
                var inputPlug = blockElement.WorkflowItem.InputPlugs[inputConnection.Index];

                // 接続！
                ConnectionManager.Connect(outputPlug, inputPlug);
            }
        }

        /// <summary>
        /// 指定のブロックエレメントの出力接続状態を復元します。
        /// </summary>
        /// <param name="blockElement">ブロックエレメント</param>
        /// <param name="outputConnections">復元する出力接続情報</param>
        /// <param name="findBlock">ブロックエレメントの検索関数</param>
        public static void RestoreOutputConnections(EffectBlockElementBase blockElement, IEnumerable<OutputPlugInfo> outputConnections, Func<uint, EffectBlockElementBase> findBlock)
        {
            // 指定のブロックエレメントについて、出力接続状態を復元する
            foreach (var outputConnection in outputConnections)
            {
                // 指定のブロックエレメントの出力プラグ数をチェック
                if (outputConnection.Index >= blockElement.WorkflowItem.OutputPlugs.Length)
                {
                    break;
                }

                foreach (var remote in outputConnection.Remotes)
                {
                    var remoteBlockElement = findBlock(remote.InstanceIdentifier);

                    // 出力先のブロックエレメントの入力プラグ数をチェック
                    if (remoteBlockElement.WorkflowItem.InputPlugs.Length <= remote.Index)
                    {
                        continue;
                    }

                    // プラグ間の適性をチェック
                    var connectability = ConnectionUtility.DetermineConnectability(
                        blockElement, outputConnection.Index,
                        remoteBlockElement, remote.Index);

                    if (connectability == Connectability.Invalid)
                    {
                        continue;
                    }

                    var outputPlug = blockElement.WorkflowItem.OutputPlugs[outputConnection.Index];
                    var inputPlug = remoteBlockElement.WorkflowItem.InputPlugs[remote.Index];

                    // 接続！
                    ConnectionManager.Connect(outputPlug, inputPlug);
                }
            }
        }

        /// <summary>
        /// プラグ間の適正をチェックします。
        /// </summary>
        /// <param name="outputBlock">出力ブロックエレメント</param>
        /// <param name="outputPlugIndex">出力プラグ番号</param>
        /// <param name="inputBlock">入力ブロックエレメント</param>
        /// <param name="inputPlugIndex">入力プラグ番号</param>
        /// <returns>プラグ間の適正を返します。</returns>
        public static Connectability DetermineConnectability(
            EffectBlockElementBase outputBlock, int outputPlugIndex,
            EffectBlockElementBase inputBlock, int inputPlugIndex)
        {
            var outputDef = outputBlock.BlockDefinition;
            var inputDef = inputBlock.BlockDefinition;

            ShaderTyping.ShaderTypeDefinition[] possibleOutputTypes;

            // 出力プラグの対応タイプを取得
            {
                var polymorphicOutput = outputBlock as PolymorphicBlockElement;

                if (polymorphicOutput != null)
                {
                    possibleOutputTypes = polymorphicOutput.GetPossibleTypes(outputBlock.WorkflowItem, false, outputPlugIndex);
                }
                else
                {
                    possibleOutputTypes = new ShaderTyping.ShaderTypeDefinition[] { outputDef.OutputPlugs[outputPlugIndex].Type };
                }
            }

            ShaderTyping.ShaderTypeDefinition[] possibleInputTypes;

            // 入力プラグの対応タイプを取得
            {
                var polymorphicInput = inputBlock as PolymorphicBlockElement;

                if (polymorphicInput != null)
                {
                    possibleInputTypes = polymorphicInput.GetPossibleTypes(inputBlock.WorkflowItem, true, inputPlugIndex);
                }
                else
                {
                    possibleInputTypes = new[] { inputDef.InputPlugs[inputPlugIndex].Type };
                }
            }

            // プラグ間の接続が未対応のときは Invalid
            if (Generation.AutoGen.OperatorDefinitionSetUtility.TypesMatch(possibleOutputTypes, possibleInputTypes) == false)
            {
                return Connectability.Invalid;
            }

            var outPlugName = outputDef.OutputPlugs[outputPlugIndex].Name;
            var inPlugName = inputDef.InputPlugs[inputPlugIndex].Name;

            // プラグ名が違うときは Good!
            if (outPlugName != inPlugName)
            {
                return Connectability.Good;
            }

            // プラグ名も同じなら Best!!
            return Connectability.Best;
        }
    }
}
