﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Linq;
using EffectCombiner.Primitives.Generation;
using EffectCombiner.Primitives.Generation.Semantic;
using EffectDefinitions;
using Renderer2D.Core;

namespace EffectCombiner.Primitives.Blocks.BlockRenderers
{
    public class RegularBlockRenderer : BlockRendererBase
    {
        /// <summary>
        /// ブロック領域のサイズ取得を行います。
        /// </summary>
        public override ISize Measure(Renderer renderer, BlockRenderInfo renderInfo, BlockDefinition definition, IRectangle specifiedRect)
        {
            if (renderer == null)
                throw new ArgumentNullException("renderer");
            if (definition == null)
                throw new ArgumentNullException("definition");

            string text = definition.Name;

            // コメントノードだったらテキストを差し替える
            if (definition is CommentBlockDefinition)
            {
                var commentDef = definition as CommentBlockDefinition;
                Debug.Assert(commentDef != null);
                text = commentDef.Text;
            }

            // コメントブロック以外はサイズ固定のため specifiedRect を無視する
            if ((definition is CommentBlockDefinition) == false)
            {
                specifiedRect = null;
            }

            // タイトルサイズを計算
            var titleTextSize = renderer.MeasureText(text, Globals.VisualResources.RegularBlockTitleTextFormat);
            var titleWidth = titleTextSize.Width;
            var titleHeight = Globals.VisualResources.RegularBlockTitleTextVerticalPadding +
                              Math.Max(Globals.VisualResources.TitleTextHeight, titleTextSize.Height) +
                              Math.Max(Globals.VisualResources.RegularBlockTitleTextVerticalPadding, Globals.VisualResources.PlugRadius);

            if (string.IsNullOrEmpty(definition.SubName) == false)
            {
                var subTitleTextSize = renderer.MeasureText(definition.SubName, Globals.VisualResources.RegularBlockTitleTextFormat);
                titleWidth = Math.Max(titleWidth, subTitleTextSize.Width);
                titleHeight += Math.Max(Globals.VisualResources.SubTitleTextHeight, subTitleTextSize.Height);
            }

            var decorationWidth = 0.0;
            if (Globals.VisualResources.InputNodeBitmap != null && Globals.VisualResources.OutputNodeBitmap != null)
            {
                decorationWidth = Math.Max(Globals.VisualResources.InputNodeBitmap.Size.Width,
                                           Globals.VisualResources.OutputNodeBitmap.Size.Width);
            }

            // タイトル領域の幅を計算
            var headerWidth = Globals.VisualResources.RegularBlockTitleTextHorizontalPadding +
                titleWidth +
                Math.Max(Globals.VisualResources.RegularBlockTitleTextHorizontalPadding,
                         Globals.VisualResources.RegularBlockDecorationHorizontalPadding) +
                decorationWidth +
                Globals.VisualResources.RegularBlockDecorationHorizontalPadding;

            var plugs = this.GetPlugAreaSize(renderer, definition);

            if (specifiedRect != null)
            {
                return new Size(
                    Math.Max(specifiedRect.Width, Math.Max(headerWidth, plugs.Width)),
                    Math.Max(specifiedRect.Height, titleHeight + plugs.Height));
            }

            return new Size(Math.Max(headerWidth, plugs.Width), titleHeight + plugs.Height);
        }

        /// <summary>
        /// Plug領域のサイズを取得します。
        /// </summary>
        /// <param name="renderer">レンダーターゲット</param>
        /// <param name="definition">領域を測定するブロック</param>
        /// <returns>Plug領域のサイズです。</returns>
        private Size GetPlugAreaSize(Renderer renderer, BlockDefinition definition)
        {
            if (renderer == null)
                throw new ArgumentNullException("renderer");
            if (definition == null)
                throw new ArgumentNullException("definition");

            // 入力プラグのテキストの文字列長
            var inputAreaSize = GetInputPlugAreaSize(renderer, definition);

            // 出力プラグのテキストの文字列長
            var outputAreaSize = GetOutputPlugAreaSize(renderer, definition);

            // 入出力のテキスト、行数からContentサイズの取得
            var plugsWidth = inputAreaSize.Width + outputAreaSize.Width + Globals.VisualResources.RegularBlockPlugsMinHorizontalSpacing + 4.0 * Globals.VisualResources.PlugRadius;
            var plugsHeight = Math.Max(inputAreaSize.Height, outputAreaSize.Height);

            var previewWidth = 0.0;
            var previewHeight = 0.0;

            if (definition.Preview.IsEnabled)
            {
                previewWidth = definition.Preview.Width + Globals.VisualResources.RegularBlockPreviewImageHorizontalPadding * 2;
                previewHeight = definition.Preview.Height + Globals.VisualResources.RegularBlockPreviewImageVerticalPadding * 2;
                if (definition.Preview.Location == PreviewDefinition.LocationType.Auto)
                {
                    double plugpreviewHeight = 0;
                    if (outputAreaSize.Height != 0 && inputAreaSize.Height != 0)
                    {
                        plugpreviewHeight = Math.Min(inputAreaSize.Height, outputAreaSize.Height) + previewHeight;
                    }
                    else
                    {
                        plugpreviewHeight = Math.Max(inputAreaSize.Height, outputAreaSize.Height) + previewHeight;
                    }
                    plugsHeight = Math.Max(plugsHeight, plugpreviewHeight);
                }
            }
            return new Size(plugsWidth, plugsHeight);
        }

        /// <summary>
        /// 入力側のPlug領域のサイズを取得します。
        /// </summary>
        /// <param name="renderer">レンダーターゲット</param>
        /// <param name="definition">領域を測定するブロック</param>
        /// <returns>Plug領域のサイズです。</returns>
        public Size GetInputPlugAreaSize(Renderer renderer, BlockDefinition definition)
        {
            var inputMaxWidth = 0.0;
            var inputHeight = 0.0;

            foreach (var inputPlug in definition.InputPlugs)
            {
                var size = inputPlug is SubPlug
                    ? renderer.MeasureText(inputPlug.Name, Globals.VisualResources.RegularBlockShortcutInputPlugTextFormat)
                    : renderer.MeasureText(inputPlug.Name, Globals.VisualResources.RegularBlockInputPlugTextFormat);
                inputMaxWidth = Math.Max(inputMaxWidth, size.Width);
                inputHeight += size.Height;
            }

            return new Size(inputMaxWidth, inputHeight);
        }

        /// <summary>
        /// 出力側のPlug領域のサイズを取得します。
        /// </summary>
        /// <param name="renderer">レンダーターゲット</param>
        /// <param name="definition">領域を測定するブロック</param>
        /// <returns>Plug領域のサイズです。</returns>
        public Size GetOutputPlugAreaSize(Renderer renderer, BlockDefinition definition)
        {
            var outputMaxWidth = 0.0;
            var outputHeight = 0.0;

            foreach (var outputPlug in definition.OutputPlugs)
            {
                var text = outputPlug.Name;
                if (definition is ConstantBlockDefinition)
                {
                    var constantDefinition = (ConstantBlockDefinition)definition;
                    text = constantDefinition.ParameterText;
                }

                if (definition is UserDefinition)
                {
                    var userDefinition = (UserDefinition)definition;
                    if (userDefinition.CanEditValue)
                    {
                        text += userDefinition.ParameterText;
                    }
                }

                var size = outputPlug is SubPlug
                   ? renderer.MeasureText(text, Globals.VisualResources.RegularBlockShortcutOutputPlugTextFormat)
                   : renderer.MeasureText(text, Globals.VisualResources.RegularBlockOutputPlugTextFormat);

                outputMaxWidth = Math.Max(outputMaxWidth, size.Width);
                outputHeight += size.Height;
            }
            return new Size(outputMaxWidth, outputHeight);
        }

        #region [rendering]
        public override void Render(Renderer renderer, BlockRenderInfo renderInfo, BlockDefinition definition, IRectangle rect)
        {
            if (renderer == null)
                throw new ArgumentNullException("renderer");
            if (definition == null)
                throw new ArgumentNullException("definition");
            if (rect == null)
                throw new ArgumentNullException("rect");

            // コメントノードの場合はIsUsableを更新しない
            var isComment = definition is CommentBlockDefinition;

            if (renderInfo.IsUsable && !isComment)
            {
                renderInfo.IsUsable = renderInfo.InputsConnectability.Any(c => c != Connectability.Invalid) ||
                    renderInfo.OutputsConnectability.Any(c => c != Connectability.Invalid);
            }

            // ドラッグ中は少し浮かす
            var liftOffset = (renderInfo.IsDragging ? Globals.VisualResources.RegularDraggedBlockLiftOffset : 0);
            var blockRect = new Rectangle(rect.X - liftOffset + Globals.VisualResources.PlugRadius,
                rect.Y - liftOffset,
                rect.Width - (Globals.VisualResources.PlugRadius * 2.0),
                rect.Height);

            if (!isComment)
            {
                RenderShadow(renderer, renderInfo, blockRect);
            }

            // 接続できないノードは影のみになるが、コメントノードは描画を続行する
            if (renderInfo.IsUsable == false && !isComment)
                return;

            RenderBackground(renderer, renderInfo, definition, blockRect);
            RenderBorder(renderer, renderInfo, definition, blockRect);

                // プレビューイメージを描画する
                if (definition.Preview.IsEnabled)
                {
                    this.RenderPreview(renderer, renderInfo, definition, blockRect);
                }

                string titleText = definition.Name;

            // コメントノードはノード名の部分にコメントを表示するため、ノード名を差し替える
            if (isComment)
            {
                var commentDef = definition as CommentBlockDefinition;
                Debug.Assert(commentDef != null);
                titleText = commentDef.Text;
            }

            IBrush titleTextBrush;

            // 入出力の定義がない場合(マウスオーバーでリストビュー上に表示されるときなど)
            if (renderInfo.PluggedInputs == null || renderInfo.PluggedOutputs == null)
            {
                titleTextBrush = Globals.VisualResources.PluggedRegularBlockTitleTextBrush;
            }
            // 入出力が十分に繋がれていた場合
            else if ((renderInfo.PluggedInputs.Length == 0 || renderInfo.PluggedInputs.All(x => x == true)) &&
                (renderInfo.PluggedOutputs.Length == 0 || renderInfo.PluggedOutputs.Any(x => x == true)))
            {
                titleTextBrush = Globals.VisualResources.PluggedRegularBlockTitleTextBrush;
            }
            // 入出力が十分に繋がれていない場合
            else
            {
                titleTextBrush = Globals.VisualResources.UnpluggedRegularBlockTitleTextBrush;
            }

            // ノード名の影を描画する
            if (Globals.Options != null &&
                Globals.Options.EnvironmentSettings != null &&
                Globals.Options.EnvironmentSettings.DrawTextShadow)
            {
                renderer.DrawText(titleText,
                    Globals.VisualResources.RegularBlockTitleTextFormat,
                    new Point(blockRect.X + Globals.VisualResources.RegularBlockTitleTextHorizontalPadding + 1.0,
                        blockRect.Y + Globals.VisualResources.RegularBlockTitleTextVerticalPadding + 1.0),
                    new Size(blockRect.Width, blockRect.Height),
                    Globals.VisualResources.RegularBlockShadowBrush);
            }

            // ノード名を描画する
            renderer.DrawText(titleText,
                Globals.VisualResources.RegularBlockTitleTextFormat,
                new Point(blockRect.X + Globals.VisualResources.RegularBlockTitleTextHorizontalPadding,
                    blockRect.Y + Globals.VisualResources.RegularBlockTitleTextVerticalPadding),
                new Size(blockRect.Width, blockRect.Height),
                titleTextBrush);

            var textSize = renderer.MeasureText(titleText, Globals.VisualResources.RegularBlockTitleTextFormat);

            // サブノード名を描画する
            if (string.IsNullOrEmpty(definition.SubName) == false)
            {
                renderer.DrawText(definition.SubName,
                    Globals.VisualResources.RegularBlockTitleTextFormat,
                    new Point(blockRect.X + Globals.VisualResources.RegularBlockTitleTextHorizontalPadding,
                        blockRect.Y + Globals.VisualResources.RegularBlockTitleTextVerticalPadding + textSize.Height),
                    new Size(blockRect.Width, blockRect.Height),
                    titleTextBrush);
            }

            // 接続プラグの描画
            if (!isComment)
            {
                var i = 0;
                foreach (var inputPlug in definition.InputPlugs)
                {
                    var pos = renderInfo.InputPlugPositions[i];
                    RenderInput(renderer, renderInfo, definition, inputPlug, rect.X + pos.X - liftOffset, rect.Y + pos.Y - liftOffset, i++);
                }

                i = 0;
                foreach (var outputPlug in definition.OutputPlugs)
                {
                    var pos = renderInfo.OutputPlugPositions[i];
                    RenderOutput(renderer, renderInfo, definition, outputPlug, rect.X + pos.X - liftOffset, rect.Y + pos.Y - liftOffset, i++);
                }
            }

            DrawDecorations(renderer, renderInfo, blockRect);
        }

        /// <summary>
        /// プレビューイメージを描画します。
        /// </summary>
        /// <param name="renderer">レンダーターゲット</param>
        /// <param name="renderInfo">描画情報</param>
        /// <param name="blockDefinition">ブロック定義データ</param>
        /// <param name="rect">描画領域</param>
        protected void RenderPreview(Renderer renderer, BlockRenderInfo renderInfo, BlockDefinition blockDefinition, IRectangle rect)
        {
            if (blockDefinition.Preview.Location == PreviewDefinition.LocationType.Auto)
            {
                var inputSize = this.GetInputPlugAreaSize(renderer, blockDefinition);
                var outputSize = this.GetOutputPlugAreaSize(renderer, blockDefinition);
                var point = new Point();
                if (inputSize.Height > outputSize.Height)
                {
                    point.X = rect.X + rect.Width - blockDefinition.Preview.Width - Globals.VisualResources.PlugRadius * 2;
                    point.Y = rect.Y + rect.Height - Globals.VisualResources.RegularBlockDecorationVerticalPadding - blockDefinition.Preview.Height;
                }
                else
                {
                    point.X = rect.X + Globals.VisualResources.PlugRadius * 2;
                    point.Y = rect.Y + rect.Height - Globals.VisualResources.RegularBlockDecorationVerticalPadding - blockDefinition.Preview.Height;
                }

                IBitmap previewImage = renderInfo.PreviewImage;

                if (previewImage == null)
                {
                    previewImage = Globals.VisualResources.PreviewImageBitmap;
                }

                renderer.DrawBitmap(previewImage,
                     new Rectangle(
                         point.X,
                         point.Y,
                         blockDefinition.Preview.Width,
                         blockDefinition.Preview.Height));
            }
        }

        protected void DrawDecorations(Renderer renderer, BlockRenderInfo renderInfo, IRectangle blockRect)
        {
            if (renderInfo.IsGraphInput == false && renderInfo.IsGraphOutput == false)
            {
                return;
            }

            IBitmap bmp = null;

            if (renderInfo.IsGraphInput)
            {
                bmp = Globals.VisualResources.InputNodeBitmap;
            }
            else if (renderInfo.IsGraphOutput)
            {
                bmp = Globals.VisualResources.OutputNodeBitmap;
            }

            if (bmp != null)
            {
                renderer.DrawBitmap(bmp,
                    new Rectangle(
                        blockRect.X + blockRect.Width - bmp.Size.Width - Globals.VisualResources.RegularBlockDecorationHorizontalPadding,
                        blockRect.Y + Globals.VisualResources.RegularBlockDecorationVerticalPadding,
                        bmp.Size.Width,
                        bmp.Size.Height));
            }
        }

        protected void RenderInput(Renderer renderer, BlockRenderInfo renderInfo, BlockDefinition blockDefinition, Plug plug, double centerX, double centerY, int index)
        {
            var largeEllipse = new Ellipse(
                centerX,
                centerY,
                Globals.VisualResources.PlugRadius,
                Globals.VisualResources.PlugRadius);

            var smallEllipse = new Ellipse(
                centerX,
                centerY,
                Globals.VisualResources.PlugRadius * 0.7,
                Globals.VisualResources.PlugRadius * 0.7);

            IBrush foregroundBrush = GetPlugBrush(blockDefinition, renderInfo.InputsConnectability, renderInfo.PluggedInputs, index);

            renderer.FillEllipse(largeEllipse, Globals.VisualResources.PlugBackgroundBrush);
            renderer.FillEllipse(smallEllipse, foregroundBrush);
            renderer.DrawEllipse(largeEllipse, Globals.VisualResources.RegularUnselectedBlockBorderBrush, Globals.VisualResources.RegularUnselectedBlockBorderStroke);

            centerX += Globals.VisualResources.PlugRadius;

            var textBrush = Globals.VisualResources.RegularBlockInputTextBrush;
            var textFormat = Globals.VisualResources.RegularBlockInputPlugTextFormat;

            if (plug is SubPlug)
            {
                textBrush = Globals.VisualResources.RegularBlockShortcutInputTextBrush;
                textFormat = Globals.VisualResources.RegularBlockShortcutInputPlugTextFormat;
            }

            var textSize = renderer.MeasureText(plug.Name, textFormat);

            if (Globals.Options != null &&
                Globals.Options.EnvironmentSettings != null &&
                Globals.Options.EnvironmentSettings.DrawTextShadow)
            {
                renderer.DrawText(plug.Name,
                    textFormat,
                    new Point(centerX + 1.0, (centerY - textSize.Height / 2.0) + 1.0),
                    Globals.VisualResources.RegularBlockShadowBrush);
            }

            renderer.DrawText(plug.Name,
                textFormat,
                new Point(centerX, centerY - textSize.Height / 2.0),
                textBrush);
        }

        protected void RenderOutput(Renderer renderer, BlockRenderInfo renderInfo, BlockDefinition blockDefinition, Plug plug, double centerX, double centerY, int index)
        {
            var largeEllipse = new Ellipse(
                centerX,
                centerY,
                Globals.VisualResources.PlugRadius,
                Globals.VisualResources.PlugRadius);

            var smallEllipse = new Ellipse(
                centerX,
                centerY,
                Globals.VisualResources.PlugRadius * 0.7,
                Globals.VisualResources.PlugRadius * 0.7);

            IBrush foregroundBrush = GetPlugBrush(blockDefinition, renderInfo.OutputsConnectability, renderInfo.PluggedOutputs, index);

            renderer.FillEllipse(largeEllipse, Globals.VisualResources.PlugBackgroundBrush);
            renderer.FillEllipse(smallEllipse, foregroundBrush);
            renderer.DrawEllipse(largeEllipse, Globals.VisualResources.RegularUnselectedBlockBorderBrush, Globals.VisualResources.RegularUnselectedBlockBorderStroke);

            centerX -= Globals.VisualResources.PlugRadius;

            var textBrush = Globals.VisualResources.RegularBlockOutputTextBrush;
            var textFormat = Globals.VisualResources.RegularBlockOutputPlugTextFormat;

            if (plug is SubPlug)
            {
                textBrush = Globals.VisualResources.RegularBlockShortcutOutputTextBrush;
                textFormat = Globals.VisualResources.RegularBlockShortcutOutputPlugTextFormat;
            }

            var text = plug.Name;
            if(blockDefinition is ConstantBlockDefinition)
            {
                var constantDefinition = (ConstantBlockDefinition)blockDefinition;
                text = constantDefinition.ParameterText;
            }

            if (blockDefinition is UserDefinition)
            {
                var userDefinition = (UserDefinition)blockDefinition;

                if (userDefinition.CanEditValue && index == 0)
                {
                    text = userDefinition.ParameterText + text;
                }
            }

            var textSize = renderer.MeasureText(text, textFormat);
            textSize.Height /= text.Count(p => p == '\n') + 1;
            if (Globals.Options != null &&
                Globals.Options.EnvironmentSettings != null &&
                Globals.Options.EnvironmentSettings.DrawTextShadow)
            {
                renderer.DrawText(text,
                    textFormat,
                    new Point(centerX - textSize.Width + 1.0, (centerY - textSize.Height / 2.0) + 1.0),
                    Globals.VisualResources.RegularBlockShadowBrush);
            }

            renderer.DrawText(text,
                textFormat,
                new Point(centerX - textSize.Width, centerY - textSize.Height / 2.0),
                textBrush);
        }

        protected void RenderBackground(Renderer renderer, BlockRenderInfo renderInfo, BlockDefinition blockDefinition, IRectangle rect)
        {
            IBrush textBackgroundBrush;
            IBrush backgroundBrush;

            if (blockDefinition.IsBound)
            {
                // 入出力の定義がない場合(マウスオーバーでリストビュー上に表示されるときなど)
                if (renderInfo.PluggedInputs == null || renderInfo.PluggedOutputs == null)
                {
                    if (Globals.VisualResources.BlockBackgroundBrushes.TryGetValue(blockDefinition.Tags.FirstOrDefault(),
                                                     out textBackgroundBrush) == false)
                    {
                        textBackgroundBrush = Globals.VisualResources.DefaultBlockBackgroundBrush;
                    }

                    backgroundBrush = Globals.VisualResources.DefaultPluggedBlockBackgroundBrush;
                }
                // 入出力が十分に繋がれていた場合
                else if ((renderInfo.PluggedInputs.Length == 0 || renderInfo.PluggedInputs.All(x => x == true)) &&
                    (renderInfo.PluggedOutputs.Length == 0 || renderInfo.PluggedOutputs.Any(x => x == true)))
                {
                    if (blockDefinition is GhostBlockDefinition)
                    {
                        textBackgroundBrush = Globals.VisualResources.GhostBlockBackgroundBrush;
                    }
                    else if (Globals.VisualResources.PluggedBlockBackgroundBrushes.TryGetValue(blockDefinition.Tags.FirstOrDefault(), out textBackgroundBrush) == false)
                    {
                        textBackgroundBrush = Globals.VisualResources.DefaultPluggedBlockBackgroundBrush;
                    }

                    backgroundBrush = Globals.VisualResources.DefaultPluggedBlockBackgroundBrush;
                }
                // 入出力が十分に繋がれていない場合
                else
                {
                    if (Globals.VisualResources.UnpluggedBlockBackgroundBrushes.TryGetValue(blockDefinition.Tags.FirstOrDefault(), out textBackgroundBrush) == false)
                    {
                        textBackgroundBrush = Globals.VisualResources.DefaultUnpluggedBlockBackgroundBrush;
                    }

                    backgroundBrush = Globals.VisualResources.DefaultUnpluggedBlockBackgroundBrush;
                }
            }
            else
            {
                textBackgroundBrush = Globals.VisualResources.InvalidBlockBackgroundBrush;
                backgroundBrush = Globals.VisualResources.DefaultUnpluggedBlockBackgroundBrush;
            }

            double subTitleHeight = (string.IsNullOrEmpty(blockDefinition.SubName) == false ? 2.0 * Globals.VisualResources.RegularBlockTitleTextFormat.Font.Size : 0.0);

            RoundedRectangle upperRect = new RoundedRectangle(rect.X, rect.Y,
                rect.Width,
                Globals.VisualResources.RegularBlockTitleTextHorizontalPadding + 2.0 * Globals.VisualResources.RegularBlockTitleTextFormat.Font.Size + subTitleHeight,
                Globals.VisualResources.RegularBlockRadiusX,
                Globals.VisualResources.RegularBlockRadiusY,
                CornerStyle.RoundedUpper);
            RoundedRectangle bottomRect = new RoundedRectangle(
                rect.X, rect.Y + upperRect.Height,
                rect.Width, rect.Height - upperRect.Height,
                Globals.VisualResources.RegularBlockRadiusX,
                Globals.VisualResources.RegularBlockRadiusY,
                CornerStyle.RoundedBottom);

            // コメントブロックはひとつのブラシで塗りつぶす
            if (blockDefinition is CommentBlockDefinition)
            {
                renderer.FillRoundedRectangle(rect,
                    Globals.VisualResources.RegularBlockRadiusX,
                    Globals.VisualResources.RegularBlockRadiusY,
                    CornerStyle.RoundedUpper,
                    textBackgroundBrush);
            }
            else
            {
                renderer.FillRoundedRectangle(upperRect,
                    Globals.VisualResources.RegularBlockRadiusX,
                    Globals.VisualResources.RegularBlockRadiusY,
                    CornerStyle.RoundedUpper,
                    textBackgroundBrush);

                renderer.FillRoundedRectangle(bottomRect,
                    Globals.VisualResources.RegularBlockRadiusX,
                    Globals.VisualResources.RegularBlockRadiusY,
                    CornerStyle.RoundedBottom,
                    backgroundBrush);
            }
        }

        protected void RenderBorder(Renderer renderer, BlockRenderInfo renderInfo, BlockDefinition blockDefinition, IRectangle rect)
        {
            IBrush borderBrush;
            double borderStroke;

            if (renderInfo.IsSelected ^ renderInfo.IsInvertingSelection)
            {
                if (renderInfo.IsPartOfCircularReference)
                {
                    borderBrush = Globals.VisualResources.CircularReferenceBrush;
                }
                else
                {
                    borderBrush = Globals.VisualResources.RegularSelectedBlockBorderBrush;
                }

                borderStroke = Globals.VisualResources.RegularSelectedBlockBorderStroke;
            }
            else
            {
                if (renderInfo.IsPartOfCircularReference)
                {
                    borderBrush = Globals.VisualResources.CircularReferenceBrush;
                }
                else
                {
                    // コメントブロックであったとき
                    if (blockDefinition is CommentBlockDefinition)
                    {
                        borderBrush = Globals.VisualResources.CommentBlockBorderBrush;
                    }
                    // 入出力の定義がない場合(マウスオーバーでリストビュー上に表示されるときなど)
                    else if (renderInfo.PluggedInputs == null || renderInfo.PluggedOutputs == null)
                    {
                        borderBrush = Globals.VisualResources.RegularUnselectedBlockBorderBrush;
                    }
                    // 入出力が十分に繋がれていた場合
                    else if ((renderInfo.PluggedInputs.Length == 0 || renderInfo.PluggedInputs.All(x => x == true)) &&
                             (renderInfo.PluggedOutputs.Length == 0 || renderInfo.PluggedOutputs.Any(x => x == true)))
                    {
                        borderBrush = Globals.VisualResources.RegularConnectedBlockBorderBrush;
                    }
                    // 入出力が十分に繋がれていない場合
                    else
                    {
                        borderBrush = Globals.VisualResources.RegularUnselectedBlockBorderBrush;
                    }
                }

                borderStroke = Globals.VisualResources.RegularUnselectedBlockBorderStroke;
            }

            renderer.DrawRoundedRectangle(
                new RoundedRectangle(
                    rect,
                    Globals.VisualResources.RegularBlockRadiusX,
                    Globals.VisualResources.RegularBlockRadiusY),
                borderBrush,
                borderStroke);
        }

        protected void RenderShadow(Renderer renderer, BlockRenderInfo renderInfo, IRectangle rect)
        {
            var shadowOffset = renderInfo.IsUsable ? Globals.VisualResources.RegularBlockShadowOffset : 0.0;

            if (renderInfo.IsDragging)
            {
                shadowOffset += 2.0 * Globals.VisualResources.RegularDraggedBlockLiftOffset;
            }

            Rectangle shadowRect = new Rectangle(rect.X + shadowOffset,
                                                 rect.Y + shadowOffset,
                                                 rect.Width,
                                                 rect.Height);
            renderer.FillRoundedRectangle(
                new RoundedRectangle(
                    shadowRect,
                    Globals.VisualResources.RegularBlockRadiusX,
                    Globals.VisualResources.RegularBlockRadiusY),
                Globals.VisualResources.RegularBlockShadowBrush);
        }
        #endregion
        #region [brush getter]
        private IBrush GetPlugBrush(BlockDefinition blockDefinition, Connectability[] connectability, bool[] plugged, int index)
        {
            var foregroundBrush = Globals.VisualResources.UnpluggedPlugForegroundBrush;

            if (connectability[index] != Connectability.None)
            {
                foregroundBrush = GetConnectabilityBrush(connectability[index]);
            }
            else if (plugged != null && plugged[index])
            {
                foregroundBrush = (blockDefinition.IsBound ?
                    Globals.VisualResources.PluggedPlugForegroundBrush :
                    Globals.VisualResources.InvalidBlockPluggedPlugForegroundBrush);
            }

            return foregroundBrush;
        }

        private IBrush GetConnectabilityBrush(Connectability connectability)
        {
            switch (connectability)
            {
                case Connectability.Good:
                    return Globals.VisualResources.CompatiblePlugForegroundBrush;
                case Connectability.Best:
                    return Globals.VisualResources.PreferedPlugForegroundBrush;
                case Connectability.Invalid:
                    return Globals.VisualResources.InvalidPlugForegroundBrush;
                case Connectability.None:
                case Connectability.Unmatching:
                    return Globals.VisualResources.UnpluggedPlugForegroundBrush;
            }

            return null;
        }
        #endregion
    }
}
