﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
#define GHOST_GET_HIDDEN

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using EffectDefinitions;
using Blocks.Core;
using Workflow.Core;
using ShaderGenerator.GLSL;
using System.Drawing;

namespace EffectCombiner.Primitives.Generation.Semantic
{
    public class GhostBlockElement : RegularEffectBlockElement
    {
        public EffectBlockElementBase GhostedBlockElement { get; private set; }

        public GhostBlockElement(BlockDefinition blockDefinition)
            : base(blockDefinition)
        {
        }

        internal void SetGhostedBlockElement(EffectBlockElementBase ghostedBlockElement)
        {
            if (GhostedBlockElement != null)
                throw new InvalidOperationException(Messages.EXCEPTION_LOCKED);

            if (ghostedBlockElement == null)
                throw new ArgumentNullException("ghostedBlockElement");

            GhostedBlockElement = ghostedBlockElement;

            GhostedBlockElement.SelectionChanged += GhostedBlockElement_SelectionChanged;
            GhostedBlockElement.VisibilityChanged += GhostedBlockElement_VisibilityChanged;
            GhostedBlockElement.PositionChanged += GhostedBlockElement_PositionChanged;
            GhostedBlockElement.RemovedFromBlockManager += GhostedBlockElementRemovedFromBlockManager;

            foreach (var op in WorkflowItem.OutputPlugs.Where((p, i) => BlockDefinition.OutputPlugs[i].Semantic != null))
                op.InputPlugRemoved += InputPlugRemoved;

#if GHOST_GET_HIDDEN
            IsVisible = GhostedBlockElement.IsSelected && GhostedBlockElement.IsVisible;
#endif // GHOST_GET_HIDDEN
        }

        private void GhostedBlockElementRemovedFromBlockManager(object sender, EventArgs e)
        {
            BlockManager.RemoveBlock(this);
        }

        private void InputPlugRemoved(object sender, InputPlugEventArgs<PlugValue> e)
        {
            if (IsOrphan)
                IsVisible = false;
        }

        private bool IsOrphan
        {
            get
            {
                return WorkflowItem.OutputPlugs.All(p => p.RemoteInputPlugs.Length == 0);
            }
        }

        protected override void OnRemovedFromBlockManager()
        {
            base.OnRemovedFromBlockManager();

            GhostedBlockElement.SelectionChanged -= GhostedBlockElement_SelectionChanged;
            GhostedBlockElement.VisibilityChanged -= GhostedBlockElement_VisibilityChanged;
            GhostedBlockElement.PositionChanged -= GhostedBlockElement_PositionChanged;
            GhostedBlockElement.RemovedFromBlockManager -= GhostedBlockElementRemovedFromBlockManager;

            foreach (var op in WorkflowItem.OutputPlugs.Where((p, i) => BlockDefinition.OutputPlugs[i].Semantic != null))
                op.InputPlugRemoved -= InputPlugRemoved;
        }

#if GHOST_GET_HIDDEN
        public override bool IsSelected
        {
            get
            {
                // make it unselectable
                return false;
            }
            set
            {
            }
        }
#endif // GHOST_GET_HIDDEN

        private void GhostedBlockElement_SelectionChanged(object sender, EventArgs e)
        {
            if (IsOrphan)
                return;

#if GHOST_GET_HIDDEN
            IsVisible = GhostedBlockElement.IsSelected;
#endif // GHOST_GET_HIDDEN

            var selected = GhostedBlockElement.IsSelected;

            //if (GhostedBlockElement.IsInvertingSelection)
            //    selected = !selected;

            IsSelected = selected;
        }

        private void GhostedBlockElement_VisibilityChanged(object sender, EventArgs e)
        {
            if (IsOrphan)
                return;

            IsVisible = GhostedBlockElement.IsVisible;
        }

        private void GhostedBlockElement_PositionChanged(object sender, PositionChangedEventArgs e)
        {
#if GHOST_GET_HIDDEN
            UpdatePosition();
#endif // GHOST_GET_HIDDEN
        }

        protected override void OnSizeChanged(SizeChangedEventArgs e)
        {
            base.OnSizeChanged(e);
#if GHOST_GET_HIDDEN
            UpdatePosition();
#endif // GHOST_GET_HIDDEN
        }

        private void UpdatePosition()
        {
            MeasureChildren(GhostedBlockElement);
            ArrangeChildren(GhostedBlockElement);
        }

        private const double BlockMargin = 8.0;
        private const double BlockSpacing = 96.0;

        public SizeF DesiredSpace { get; private set; }

        private static void MeasureChildren(EffectBlockElementBase block)
        {
            var selfSize = new SizeF(
                (float)(block.Width + BlockMargin * 2.0),
                (float)(block.Height + BlockMargin * 2.0));

            var inputBlocks = block.GetInputBlocks().OfType<GhostBlockElement>().ToArray();
            var ghostBlock = block as GhostBlockElement;

            if (inputBlocks.Length == 0)
            {
                if (ghostBlock != null)
                    ghostBlock.DesiredSpace = selfSize;
                return;
            }

            foreach (var child in inputBlocks)
                MeasureChildren(child);

            if (ghostBlock == null)
                return;

            var width = inputBlocks.Max(b => b.DesiredSpace.Width);
            width += (float)BlockSpacing + selfSize.Width;

            var height = inputBlocks.Sum(b => b.DesiredSpace.Height);
            height = Math.Max(height, selfSize.Height);

            ghostBlock.DesiredSpace = new SizeF(width, height);
        }

        private static void ArrangeChildren(EffectBlockElementBase block)
        {
            var inputBlocks = block.GetInputBlocks().OfType<GhostBlockElement>().ToArray();

            if (inputBlocks.Length == 0)
                return;

            var totalHeight = inputBlocks.Sum(b => b.DesiredSpace.Height);
            var childMaxWidth = inputBlocks.Max(b => b.Width) + BlockMargin * 2.0;

            var top = block.Top - (totalHeight - (block.Height + (BlockMargin * 2.0))) / 2.0;

            foreach (var ghost in inputBlocks)
            {
                var blockLeft = block.Left - BlockMargin * 2.0 - BlockSpacing - (childMaxWidth - (ghost.Width + BlockMargin * 2.0)) / 2.0 - ghost.Width;
                var blockTop = top + (ghost.DesiredSpace.Height - (ghost.Height + BlockMargin * 2.0)) / 2.0;

                ghost.SetPosition(blockLeft, blockTop);

                top += ghost.DesiredSpace.Height;

                ArrangeChildren(ghost);
            }
        }
    }
}
