﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using EffectCombiner.Primitives.Generation.Extensions;
using EffectCombiner.Primitives.Generation.Usage;
using EffectDefinitions;
using Workflow.Core;

namespace EffectCombiner.Primitives.Generation
{
    public class EffectBlockDefinition : BlockDefinition
    {
        private readonly BlockDefinition originalBlockDefinition;

        public EffectBlockDefinition(BlockDefinition blockDefinition)
            : base(
            blockDefinition.Source,
            blockDefinition.Name,
            blockDefinition.Guid,
            blockDefinition.Group,
            ProcessInputPlugs(blockDefinition.InputPlugs, blockDefinition),
            ProcessOutputPlugs(blockDefinition.OutputPlugs, blockDefinition),
            blockDefinition.Uniform,
            blockDefinition.Preview,
            blockDefinition.FunctionBindings)
        {
            originalBlockDefinition = blockDefinition;
        }

        private static ICollection<Plug> ProcessInputPlugs(ICollection<Plug> originalPlugs, BlockDefinition originalBlockDefinition)
        {
            if (originalBlockDefinition.CurrentFunctionDefinition == null)
                return originalPlugs;

            var workingList = new List<Plug>();

            var inputParameters = originalBlockDefinition.CurrentFunctionDefinition.Parameters
                .Where(p => p.IsAnInput)
                .Cast<ParameterDefinition>()
                .ToArray();

            var i = -1;
            foreach (var plug in originalPlugs)
            {
                i++;
                workingList.Add(plug);

                var param = inputParameters.FirstOrDefault(pd => pd.Name == plug.Target);
                if (param == null)
                    continue;

                if (string.IsNullOrWhiteSpace(param.Usage))
                    continue;

                var usageDescriptor = UsageManager.GetUsageByName(param.Usage);
                if (usageDescriptor == null)
                    continue;

                if (param.Type.Equals(usageDescriptor.Type) == false)
                    continue;

                var expandedPlugs = usageDescriptor.ExpandInput(plug);
                if (expandedPlugs == null || expandedPlugs.Length == 0)
                    continue;

                workingList.AddRange(expandedPlugs);
            }

            return workingList;
        }

        private static ICollection<Plug> ProcessOutputPlugs(ICollection<Plug> originalPlugs, BlockDefinition originalBlockDefinition)
        {
            var funcDef = originalBlockDefinition.CurrentFunctionDefinition;
            if (funcDef == null)
                return originalPlugs;

            var workingList = new List<Plug>();

            var outputParameters = funcDef.Parameters
                .Where(p => p.IsAnOutput)
                .Cast<ParameterDefinition>()
                .ToList();

            if (funcDef.ReturnType.IsVoid == false)
                outputParameters.Insert(0, funcDef.GetReturnAsParameter());

            foreach (var plug in originalPlugs)
            {
                workingList.Add(plug);

                var param = outputParameters.FirstOrDefault(pd => pd.Name == plug.Target);
                if (param == null)
                    continue;

                if (string.IsNullOrWhiteSpace(param.Usage))
                    continue;

                var usageDescriptor = UsageManager.GetUsageByName(param.Usage);
                if (usageDescriptor == null)
                    continue;

                if (param.Type.Equals(usageDescriptor.Type) == false)
                    continue;

                var expandedPlugs = usageDescriptor.ExpandOutput(plug);
                if (expandedPlugs == null || expandedPlugs.Length == 0)
                    continue;

                workingList.AddRange(expandedPlugs);
            }

            return workingList;
        }
    }

    public class SubPlug : Plug
    {
        public Plug RelatedPlug { get; private set; }
        public string TargetBase { get; private set; }
        public string TargetExtension { get; private set; }
        public SubPlug[] PairedSubPlugs { get; private set; }
        public int[] FieldIndices { get; private set; }

        public SubPlug(
            Plug relatedPlug,
            string name,
            ShaderTypeDefinition type,
            string targetExtension,
            bool isInput,
            int[] fieldIndices)
            : base(
            name,
            type,
            string.Format("{0}.{1}", relatedPlug.Target, targetExtension),
            isInput,
            null,
            SliceDefaultValues(relatedPlug, fieldIndices))
        {
            if (relatedPlug == null)
                throw new ArgumentNullException("relatedPlug");
            if (fieldIndices == null)
                throw new ArgumentNullException("fieldIndices");
            if (fieldIndices.Length == 0)
                throw new ArgumentException(string.Format(Messages.EXCEPTION_ARRAY_MUST_NOT_BE_EMPTY, "fieldIndices"), "fieldIndices");
            if (string.IsNullOrWhiteSpace(targetExtension))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "targetExtension"), "targetExtension");

            RelatedPlug = relatedPlug;
            PairedSubPlugs = new SubPlug[0];
            FieldIndices = fieldIndices;
            TargetExtension = targetExtension;
        }

        private static string[] SliceDefaultValues(Plug originalPlug, IEnumerable<int> fieldIndices)
        {
            if (originalPlug.DefaultValues == null)
                return null;
            return fieldIndices.Select(index => originalPlug.DefaultValues[index]).ToArray();
        }

        internal void SetPairedSubPlugs(params SubPlug[] pairedSubPlugs)
        {
            if (pairedSubPlugs == null)
                throw new ArgumentNullException("pairedSubPlugs");
            if (pairedSubPlugs.Length == 0)
                throw new ArgumentException(string.Format(Messages.EXCEPTION_ARRAY_MUST_NOT_BE_EMPTY, "pairedSubPlugs"), "pairedSubPlugs");

            PairedSubPlugs = pairedSubPlugs;
        }
    }

    public struct InputPlugGroup<T>
    {
        public InputPlug<T> Plug;
        public InputPlug<T>[] SubPlugs;
    }

    public struct OutputPlugGroup<T>
    {
        public OutputPlug<T> Plug;
        public OutputPlug<T>[] SubPlugs;
    }

    public static class WorkflowItemExtensions
    {
        public static InputPlugGroup<T>[] GetGroupedEffectInputPlugs<T>(this InputPlug<T>[] plugs, Plug[] mapping)
        {
            var groups = new List<InputPlugGroup<T>>();

            for (var i = 0; i < plugs.Length; i++)
            {
                var plug = plugs[i];
                var mappingPlug = mapping[i];

                if (mappingPlug is SubPlug)
                    continue;

                var subList = new List<InputPlug<T>>();

                for (i++; i < plugs.Length; i++)
                {
                    var subPlug = plugs[i];
                    var mappingSubPlug = mapping[i] as SubPlug;
                    if (mappingSubPlug == null)
                    {
                        i--;
                        break;
                    }
                    subList.Add(subPlug);
                }

                groups.Add(new InputPlugGroup<T>
                {
                    Plug = plug,
                    SubPlugs = subList.ToArray(),
                });
            }

            return groups.ToArray();
        }

        public static OutputPlugGroup<T>[] GetGroupedEffectOutputPlugs<T>(this OutputPlug<T>[] plugs, Plug[] mapping)
        {
            var groups = new List<OutputPlugGroup<T>>();

            for (var i = 0; i < plugs.Length; i++)
            {
                var plug = plugs[i];
                var mappingPlug = mapping[i];

                if (mappingPlug is SubPlug)
                    continue;

                var subList = new List<OutputPlug<T>>();

                for (i++; i < plugs.Length; i++)
                {
                    var subPlug = plugs[i];
                    var mappingSubPlug = mapping[i] as SubPlug;
                    if (mappingSubPlug == null)
                    {
                        i--;
                        break;
                    }
                    subList.Add(subPlug);
                }

                groups.Add(new OutputPlugGroup<T>
                {
                    Plug = plug,
                    SubPlugs = subList.ToArray(),
                });
            }

            return groups.ToArray();
        }
    }
}
