﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using EffectCombiner.Data.Processors.Version2;
using EffectCombiner.Primitives.Generation;
using EffectCombiner.Primitives.Generation.Semantic;
using EffectDefinitions;
using Workflow.Core;

namespace EffectCombiner.Generator
{
    public static class WorkflowLoader
    {
        private static readonly BlockElementGenerator blockGenerator = new BlockElementGenerator();

        public static EffectWorkflowItem[] GetWorkflow(
            object data,
            IDictionary<string, BlockDefinition> blockDefinitions)
        {
            var project = (ProjectV2)data;

            var blocks = project.Blocks.Block
                .Select(b => new
                    {
                        MetadataId = Convert.ToUInt32(b.Identifier, 16),
                        BlockElement = CreateBlockElement(b, blockDefinitions[b.BlockGuid]),
                    })
                .ToDictionary(b => b.MetadataId);

            if (project.Connections != null && project.Connections.Connection != null)
            {
                foreach (var connectionInfo in project.Connections.Connection)
                {
                    var sourceBlockInfo = connectionInfo.Source;
                    var targetBlockInfo = connectionInfo.Target;

                    var sourceBlock = blocks[Convert.ToUInt32(sourceBlockInfo.BlockIdentifier, 16)];
                    var targetBlock = blocks[Convert.ToUInt32(targetBlockInfo.BlockIdentifier, 16)];

                    ConnectionManager.Connect(
                        sourceBlock.BlockElement.WorkflowItem.OutputPlugs[connectionInfo.Source.PlugIndex],
                        targetBlock.BlockElement.WorkflowItem.InputPlugs[connectionInfo.Target.PlugIndex]);
                }
            }

            var workflowItems = new List<EffectWorkflowItem>();

            foreach (var blockElement in blocks.Values.Select(b => b.BlockElement))
            {
                workflowItems.Add(blockElement.WorkflowItem);

                workflowItems.AddRange(SemanticHelper.ExpandSemantics(blockGenerator, blockDefinitions.Values, blockElement)
                    .Cast<EffectBlockElementBase>()
                    .Select(b => b.WorkflowItem));
            }

            return workflowItems.ToArray();
        }

        private static EffectBlockElementBase CreateBlockElement(BlockV2 blockData, BlockDefinition blockDefinition)
        {
            var newBlockElement = (EffectBlockElementBase)blockGenerator.CreateBlockElement(blockDefinition);

            if (newBlockElement is ConstantBlockElement && blockData.ConstInfo != null)
                ((ConstantBlockElement)newBlockElement).SetupConstantBlockElement(blockData.ConstInfo.Type, blockData.ConstInfo.Values);

            if (newBlockElement is CommentBlockElement && blockData.ConstInfo != null)
                ((CommentBlockElement)newBlockElement).SetupCommentBlockElement(blockData.ConstInfo.Values);

            return newBlockElement;
        }
    }
}
