﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;
using System.Windows.Forms;
using EffectCombiner.Primitives;

namespace EffectCombiner.Editor
{
    public partial class SourceCodeViewForm : Form
    {
        public SourceCodeViewForm()
        {
            InitializeComponent();

            btnShowErrorLocation.Click += btnShowErrorLocation_Click;
            txtSourceCode.KeyDown += txtSourceCode_KeyDown;
            txtSourceCode.MouseDown += txtSourceCode_MouseDown;
            txtSourceCode.MouseMove += txtSourceCode_MouseMove;

            var localizationSubscription = Globals.Localization.RegisterLocalization(() =>
                {
                    Text = Localization.Controls.FORM_SOURCE_CODE_VIEW_TITLE;
                    btnShowErrorLocation.Text = Localization.Controls.FORM_SOURCE_CODE_VIEW_SHOW_LOCATION_BUTTON;
                });

            FormClosed += (ss, ee) => localizationSubscription.Dispose();
        }

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            SetCaretPositionInfo();
            BeginInvoke((Action)ShowDefinitionLocation);
        }

        private int line;
        private int column;
        private int length;

        public void SetInfo(string filename, string content, int line, int column, int length)
        {
            this.line = line - 1;
            this.column = column - 1;
            this.length = length;

            txtSourceCode.Text = content;

            ShowDefinitionLocation();

            if (string.IsNullOrWhiteSpace(filename))
                txtFilename.Visible = false;
            else
            {
                txtFilename.Text = filename;
                txtFilename.SelectionStart = filename.Length;
                txtFilename.Visible = true;
            }
        }

        private void txtSourceCode_KeyDown(object sender, KeyEventArgs e)
        {
            BeginInvoke((Action)SetCaretPositionInfo);
        }

        private void txtSourceCode_MouseDown(object sender, MouseEventArgs e)
        {
            SetCaretPositionInfo();
        }

        private void txtSourceCode_MouseMove(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
                SetCaretPositionInfo();
        }

        private void btnShowErrorLocation_Click(object sender, EventArgs e)
        {
            ShowDefinitionLocation();
        }

        private void SetCaretPositionInfo()
        {
            if (txtSourceCode.SelectionLength == 0)
            {
                var position = txtSourceCode.SelectionStart + txtSourceCode.SelectionLength;

                var currentLine = txtSourceCode.GetLineFromCharIndex(position);
                var temp = txtSourceCode.GetFirstCharIndexOfCurrentLine();
                var currentColumn = position - temp;

                lblInfo.Text = string.Format(Localization.Controls.LINE_POSITION, currentLine + 1, currentColumn + 1);
            }
            else
            {
                var startPosition = txtSourceCode.SelectionStart;
                var endPosition = txtSourceCode.SelectionStart + txtSourceCode.SelectionLength;

                var startLine = txtSourceCode.GetLineFromCharIndex(startPosition);
                var endLine = txtSourceCode.GetLineFromCharIndex(endPosition);

                var temp1 = txtSourceCode.GetFirstCharIndexFromLine(startLine);
                var temp2 = txtSourceCode.GetFirstCharIndexFromLine(endLine);

                var startColumn = startPosition - temp1;
                var endColumn = endPosition - temp2;

                lblInfo.Text = string.Format(Localization.Controls.LINE_POSITION_SELECT,
                    startLine + 1, startColumn + 1,
                    endLine + 1, endColumn + 1,
                    txtSourceCode.SelectionLength);
            }
        }

        private void ShowDefinitionLocation()
        {
            txtSourceCode.SelectionStart = txtSourceCode.GetFirstCharIndexFromLine(line) + column;
            txtSourceCode.SelectionLength = length;
            txtSourceCode.Focus();
            txtSourceCode.ScrollToCaret();
            SetCaretPositionInfo();
        }
    }
}
