﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using System.Windows.Forms;
using EffectCombiner.Core;
using EffectCombiner.Primitives;

namespace EffectCombiner.Editor
{
    public partial class PathInfoForm : Form
    {
        public PathLookupInfo PathInfo { get; private set; }

        private readonly PathLookupInfo[] pathInfos;

        public PathInfoForm()
        {
            InitializeComponent();

            var localizationSubscription = Globals.Localization.RegisterLocalization(() =>
                {
                    Text = Localization.Controls.FORM_PATH_INFO_TITLE;
                    lblPath.Text = Localization.Controls.FORM_PATH_INFO_PATH_LABEL;
                    chkRecursive.Text = Localization.Controls.FORM_PATH_INFO_RECURSIVE_CHECKBOX;
                    lblPatterns.Text = Localization.Controls.FORM_PATH_INFO_SEARCH_PATTERNS_LABEL;
                    lblPatternExplanation.Text = Localization.Controls.FORM_PATH_INFO_SEARCH_PATTERNS_EXPLANATION;
                    btnOK.Text = Localization.Controls.BUTTON_OK;
                    btnCancel.Text = Localization.Controls.BUTTON_CANCEL;
                });

            FormClosed += (ss, ee) => localizationSubscription.Dispose();
        }

        public PathInfoForm(params PathLookupInfo[] pathInfos)
            : this()
        {
            if (pathInfos == null)
                throw new ArgumentNullException("pathInfos");

            this.pathInfos = pathInfos;
            SetupUserInterface();
        }

        private void SetupUserInterface()
        {
            if (pathInfos.Length == 1)
            {
                txtPath.Text = pathInfos[0].Path;
                chkRecursive.Checked = pathInfos[0].IncludeSubDirectories;
                txtPatterns.Text = string.Join(", ", pathInfos[0].SearchPatterns);
            }
            else if (pathInfos.Length > 1)
            {
                string path;
                bool? rec;
                string[] exts;

                FindCommonPathInfo(out path, out rec, out exts);

                txtPath.Text = path;
                chkRecursive.CheckState = rec.HasValue ?
                    (rec.Value ? CheckState.Checked : CheckState.Unchecked) :
                    CheckState.Indeterminate;
                txtPatterns.Text = string.Join(", ", exts);
            }
        }

        private void FindCommonPathInfo(out string path, out bool? rec, out string[] ext)
        {
            var first = pathInfos.First();

            path = string.Empty;
            rec = null;
            ext = new string[0];

            if (pathInfos.All(p => string.Equals(p.Path, first.Path,
                StringComparison.InvariantCultureIgnoreCase)))
                path = first.Path;
            if (pathInfos.All(p => p.IncludeSubDirectories == first.IncludeSubDirectories))
                rec = first.IncludeSubDirectories;

            var firstSortedExt = first.SearchPatterns
                .Select(x => x.ToLower())
                .OrderBy(x => x)
                .ToArray();
            if (pathInfos.All(p =>
            {
                var tmp = p.SearchPatterns
                    .Select(x => x.ToLower())
                    .OrderBy(x => x)
                    .ToArray();
                return tmp.SequenceEqual(firstSortedExt);
            }))
                ext = first.SearchPatterns;
        }

        private void btnBrowsePath_Click(object sender, EventArgs e)
        {
            var dlg = new FolderBrowserDialog
            {
                Description = Localization.Messages.DLG_DEFINITION_FOLDER_DESCRIPTION,
                ShowNewFolderButton = true,
                SelectedPath = string.IsNullOrEmpty(txtPath.Text) ? Environment.CurrentDirectory : txtPath.Text,
            };

            if (dlg.ShowDialog() == DialogResult.OK)
                txtPath.Text = dlg.SelectedPath;
        }

        private void chkRecursive_CheckedChanged(object sender, EventArgs e)
        {
            btnOK.Enabled = chkRecursive.CheckState != CheckState.Indeterminate;
        }

        private void btnOK_Click(object sender, EventArgs e)
        {
            var path = txtPath.Text.Trim();
            if (string.IsNullOrWhiteSpace(path))
            {
                MessageBox.Show(Localization.Messages.MSG_PATH_IS_MANDATORY, Localization.Messages.ERROR,
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            var patterns = txtPatterns.Text.Trim();
            if (string.IsNullOrWhiteSpace(patterns))
            {
                MessageBox.Show(Localization.Messages.MSG_SEARCH_PATTERN_IS_MANDATORY, Localization.Messages.ERROR,
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            PathInfo = new PathLookupInfo(
                path,
                chkRecursive.Checked,
                CoreUtility.Split(patterns, ',', ';', ' '));

            DialogResult = DialogResult.OK;
            Close();
        }

        private void btnCancel_Click(object sender, EventArgs e)
        {
            DialogResult = DialogResult.Cancel;
            Close();
        }
    }

    public class PathInfoListViewItem : ListViewItem, IDisposable
    {
        private PathLookupInfo pathInfo;
        public PathLookupInfo PathInfo
        {
            get { return pathInfo; }
            set { pathInfo = value; UpdateData(); }
        }

        public PathInfoListViewItem(bool isActive, PathLookupInfo pathInfo)
        {
            Checked = isActive;

            SubItems.Add("");
            SubItems.Add("");

            PathInfo = pathInfo;

            Globals.Localization.CultureChanged += LocalizationCultureChanged;
        }

        private void LocalizationCultureChanged(object sender, EventArgs e)
        {
            SubItems[1].Text = pathInfo.IncludeSubDirectories ? Localization.Messages.TEXT_YES : "";
        }

        private void UpdateData()
        {
            SubItems[0].Text = pathInfo.Path;
            SubItems[1].Text = pathInfo.IncludeSubDirectories ? Localization.Messages.TEXT_YES : "";
            SubItems[2].Text = string.Join(", ", pathInfo.SearchPatterns);
        }

        public void Dispose()
        {
            Globals.Localization.CultureChanged -= LocalizationCultureChanged;
        }
    }
}
