﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectCombiner.Core;
using EffectCombiner.Primitives;

namespace EffectCombiner.Editor.OptionPanes
{
    /// <summary>
    /// 定義パスリストのUIです。
    /// </summary>
    public partial class DefinitionPathList : UserControl
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public DefinitionPathList()
        {
            this.InitializeComponent();

            var localizationSubscription = Globals.Localization.RegisterLocalization(() =>
                {
                    colDirectory.Text = Localization.Controls.OPTION_PANE_DEFINITION_PATH_COLUMN_DIRECTORY;
                    colRecursive.Text = Localization.Controls.OPTION_PANE_DEFINITION_PATH_COLUMN_RECURSIVE;
                    colExtensions.Text = Localization.Controls.OPTION_PANE_DEFINITION_PATH_COLUMN_EXTENSIONS;
                    ctxDuplicate.Text = Localization.Controls.CONTEXT_MENU_DUPLICATE;
                    ctxRemove.Text = Localization.Controls.CONTEXT_MENU_REMOVE;
                });
        }

        /// <summary>
        /// リストビューを取得します。
        /// </summary>
        public ListView ListView { get { return this.listView; } }

        /// <summary>
        /// パスの追加または編集を行います。
        /// </summary>
        /// <param name="addMode">追加を行うときtrue</param>
        public void AddOrEditDefinitionPath(bool addMode)
        {
            PathInfoForm form;

            if (addMode)
            {
                form = new PathInfoForm();
            }
            else
            {
                var pathInfos = this.listView.SelectedItems
                    .Cast<PathInfoListViewItem>()
                    .Select(item => item.PathInfo)
                    .ToArray();

                form = new PathInfoForm(pathInfos);
            }

            if (form.ShowDialog(this) == DialogResult.OK)
            {
                var p = form.PathInfo;

                if (addMode)
                {
                    this.listView.Items.Add(new PathInfoListViewItem(true, p));
                }
                else
                {
                    foreach (var item in this.listView.SelectedItems.Cast<PathInfoListViewItem>())
                    {
                        item.PathInfo = new PathLookupInfo(p.Path, p.IncludeSubDirectories, p.SearchPatterns);
                    }
                }
            }
        }

        /// <summary>
        /// 定義パスリストを設定します。
        /// </summary>
        /// <param name="paths">パスリスト</param>
        public void SetDefinitionPaths(DefinitionPathInfo[] paths)
        {
            ListViewItem[] items = new ListViewItem[paths.Length];

            for (int i = 0; i < paths.Length; ++i)
            {
                PathLookupInfo lookupInfo = new PathLookupInfo(
                    paths[i].Path.Trim(),
                    paths[i].IsRecursive,
                    CoreUtility.Split(paths[i].Extensions, ',', ';', ' '));

                items[i] = new PathInfoListViewItem(paths[i].IsActive, lookupInfo);
            }

            this.listView.Items.AddRange(items);
        }

        /// <summary>
        /// 定義パスリストを取得します。
        /// </summary>
        /// <returns>定義パスリストを返します。</returns>
        public DefinitionPathInfo[] GetDefinitionPaths()
        {
            DefinitionPathInfo[] paths = new DefinitionPathInfo[this.listView.Items.Count];

            for (int i = 0; i < this.listView.Items.Count; ++i)
            {
                var item = (PathInfoListViewItem)this.listView.Items[i];

                paths[i] = new DefinitionPathInfo
                {
                    IsActive = item.Checked,
                    Path = item.PathInfo.Path,
                    IsRecursive = item.PathInfo.IncludeSubDirectories,
                    Extensions = string.Join(", ", item.PathInfo.SearchPatterns)
                };
            }

            return paths;
        }

        /// <summary>
        /// 選択中のパスを削除します。
        /// </summary>
        private void RemoveSelectedItems()
        {
            foreach (var item in this.listView.SelectedItems.Cast<ListViewItem>().ToArray())
            {
                this.listView.Items.Remove(item);
            }
        }

        /// <summary>
        /// コンテキストメニューの追加編集ボタンを押したときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void ctxAddEdit_Click(object sender, EventArgs e)
        {
            this.AddOrEditDefinitionPath(this.listView.SelectedItems.Count == 0);
        }

        /// <summary>
        /// コンテキストメニューの複製を押したときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void ctxDuplicate_Click(object sender, EventArgs e)
        {
            if (this.listView.SelectedItems.Count != 1)
                return;

            var item = this.listView.SelectedItems[0] as PathInfoListViewItem;
            if (item == null)
                return;

            var pathInfo = item.PathInfo;
            var newPathInfo = new PathLookupInfo(pathInfo.Path, pathInfo.IncludeSubDirectories, pathInfo.SearchPatterns);

            this.listView.Items.Add(new PathInfoListViewItem(false, newPathInfo));
        }

        /// <summary>
        /// コンテキストメニューの削除ボタンを押したときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void ctxRemove_Click(object sender, EventArgs e)
        {
            this.RemoveSelectedItems();
        }

        /// <summary>
        /// リストビューの選択アイテムを変更したときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void listView_SelectedIndexChanged(object sender, EventArgs e)
        {
            // 選択アイテムに合わせてコンテキストメニューの表示内容を更新する
            if (this.listView.SelectedItems.Count == 0)
            {
                this.ctxAddEdit.Text = Localization.Controls.CONTEXT_MENU_ADD;
                this.ctxDuplicate.Enabled = false;
                this.ctxRemove.Enabled = false;
            }
            else
            {
                this.ctxAddEdit.Text = Localization.Controls.CONTEXT_MENU_EDIT;
                this.ctxDuplicate.Enabled = this.listView.SelectedItems.Count == 1;
                this.ctxRemove.Enabled = true;
            }
        }

        /// <summary>
        /// リストビューのアイテムをダブルクリックしたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void listView_MouseDoubleClick(object sender, MouseEventArgs e)
        {
            // パスの編集処理を行う
            this.AddOrEditDefinitionPath(false);
        }

        /// <summary>
        /// リストビューのアイテムについてキー入力を行ったときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void listView_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.Modifiers == Keys.None)
            {
                if (e.KeyCode == Keys.Add)
                {
                    // テンキーの"+"でパスの追加処理を行う
                    this.AddOrEditDefinitionPath(true);
                }
                else if (e.KeyCode == Keys.Delete)
                {
                    // Deleteキーでパスの削除処理を行う
                    this.RemoveSelectedItems();
                }
            }
            else if ((e.Modifiers & Keys.Control) != 0)
            {
                if (e.KeyCode == Keys.C)
                {
                    // Ctrl+Cでアイテムをコピーする
                    var sb = new StringBuilder();

                    foreach (var item in this.listView.SelectedItems.OfType<PathInfoListViewItem>())
                    {
                        sb.AppendLine(item.PathInfo.ToClipboardString());
                    }

                    if (sb.Length > 0)
                    {
                        Clipboard.SetText(sb.ToString());
                    }
                }
                else if (e.KeyCode == Keys.V)
                {
                    // Ctrl+Vでアイテムをペーストする
                    var clipboard = Clipboard.GetText().Split(new[] { '\n' }, StringSplitOptions.RemoveEmptyEntries);
                    var list = new List<PathLookupInfo>();

                    foreach (var line in clipboard.Select(s => s.Trim()))
                    {
                        PathLookupInfo p;

                        if (PathLookupInfo.FromClipboardString(line, out p) == false)
                        {
                            return;
                        }

                        list.Add(p);
                    }

                    list.ForEach(p => this.listView.Items.Add(new PathInfoListViewItem(false, p)));
                }
            }
        }
    }
}
