﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Windows.Forms;
using EffectCombiner.Core;
using EffectCombiner.Core.Extensions;
using EffectCombiner.Primitives;
using EffectCombiner.Primitives.Operations;
using EffectCombiner.Editor.Controls;
using EffectCombiner.Primitives.Blocks;
using OperationManager.Core;
using EffectCombiner.Core.IO;
using EffectDefinitions;

namespace EffectCombiner.Editor
{
    public partial class MainForm : Form
    {
        private RendererManager rendererManager;
        private GenerationManager generationManager;
        private WorkspaceManager workspaceManager;
        private ProjectManager projectManager;
        private DefinitionsManager definitionsManager;

        private readonly CommunicationManager communicationManager = new CommunicationManager();
        private readonly IDisposable localizationSubscription;

        public MainForm()
        {
            InitializeComponent();

            localizationSubscription = Globals.Localization.RegisterLocalization(() =>
                {
                    mnuFile.Text = Localization.Controls.FORM_MAIN_MENU_FILE;
                    mnuFileNew.Text = Localization.Controls.FORM_MAIN_MENU_FILE_NEW;
                    mnuFileLoad.Text = Localization.Controls.FORM_MAIN_MENU_FILE_OPEN;
                    mnuFileSave.Text = Localization.Controls.FORM_MAIN_MENU_FILE_SAVE;
                    mnuFileSaveAs.Text = Localization.Controls.FORM_MAIN_MENU_FILE_SAVE_AS;
                    mnuFileRecent.Text = Localization.Controls.FORM_MAIN_MENU_FILE_RECENT_PROJECTS;
                    mnuFileReloadDefs.Text = Localization.Controls.FORM_MAIN_MENU_FILE_RELOAD_DEFINITIONS;
                    mnuFileClose.Text = Localization.Controls.FORM_MAIN_MENU_FILE_EXIT;

                    mnuEdit.Text = Localization.Controls.FORM_MAIN_MENU_EDIT;
                    mnuEditUndo.Text = Localization.Controls.FORM_MAIN_MENU_EDIT_UNDO;
                    mnuEditRedo.Text = Localization.Controls.FORM_MAIN_MENU_EDIT_REDO;

                    mnuView.Text = Localization.Controls.FORM_MAIN_MENU_VIEW;
                    mnuViewEventsList.Text = Localization.Controls.FORM_MAIN_MENU_VIEW_EVENT_REPORTS;
                    mnuViewLastProducedShaderCodes.Text = Localization.Controls.FORM_MAIN_MENU_VIEW_LAST_PRODUCED_SHADER;
                    mnuViewOperationManager.Text = Localization.Controls.FORM_MAIN_MENU_VIEW_ACTION_STACK;

                    mnuShader.Text = Localization.Controls.FORM_MAIN_MENU_SHADER;
                    mnuShaderBuild.Text = Localization.Controls.FORM_MAIN_MENU_SHADER_BUILD;
                    mnuShaderUpdateEffectMaker.Text = Localization.Controls.FORM_MAIN_MENU_SHADER_UPDATE_EFFECT_MAKER;
                    mnuShaderGenerate.Text = Localization.Controls.FORM_MAIN_MENU_SHADER_GENERATE;
                    mnuShaderSaveMathFunctions.Text = Localization.Controls.FORM_MAIN_MENU_SHADER_SAVE_MATH_FUNCTIONS;

                    mnuOptions.Text = Localization.Controls.FORM_MAIN_MENU_OPTIONS;

                    mnuHelp.Text = Localization.Controls.FORM_MAIN_MENU_HELP;
                    mnuHelpAbout.Text = Localization.Controls.FORM_MAIN_MENU_HELP_ABOUT;
                });

            this.elhViewer.Child = new HwndPresenterControl();
            this.elhViewer.Size = new System.Drawing.Size(256 + 0, 256 + 16);

            FormClosed += (ss, ee) => localizationSubscription.Dispose();
        }

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);
            SetTitle();

            Globals.SetVisualResourceSet(new VisualResourceSet());

            Reporting.Initialize();

            if (OptionsManager.Initialize(Program.argProjectConfigPath) == false)
            {
                Reporting.Report(new EventReport(() => Localization.Messages.OPTIONS_FAILED_TO_LOAD,
                    ReportLevel.Error, ReportCategory.Options, null, null, null));
            }

            if (string.IsNullOrWhiteSpace(Globals.Options.LanguageSettings.CurrentLanguage) == false)
                Globals.Localization.ChangeLanguage(Globals.Options.LanguageSettings.CurrentLanguage.Trim());

            OptionsExtensions.ApplyRecordedFormWindowState(Globals.Options.EnvironmentSettings.MainWindow, this);

            Globals.SetConstantValuePanel(constantValuePanel1);
            Globals.SetEffectBlockElementPanel(effectBlockValuePanel1);
            Globals.SetCommentTextPanel(commentTextPanel1);

            rendererManager = new RendererManager(cboRender, pnlRenderSurfaceContainer);
            workspaceManager = new WorkspaceManager(rendererManager);
            Globals.SetWorkspaceManager(workspaceManager);

            projectManager = new ProjectManager(workspaceManager, new WorkflowDataEventReporter());
            projectManager.FilenameChanged += ProjectManagerFilenameChanged;
            projectManager.IsModifiedChanged += ProjectManagerIsModifiedChanged;
            projectManager.Opened += ProjectManagerOpened;
            projectManager.Closed += ProjectManagerClosed;

            ExtendedOperationManager.Initialize(projectManager);
            QuickInfoManager.Initialize(lblQuickInfo);

            InitializeCompiler();
            mnuShaderBuild.Enabled = ShaderGenerator.CafeCompiler.Compiler.IsAvailable;

            Globals.MainOperationManager.CanUndoChanged += (ss, ee) => mnuEditUndo.Enabled = Globals.MainOperationManager.CanUndo;
            Globals.MainOperationManager.CanRedoChanged += (ss, ee) => mnuEditRedo.Enabled = Globals.MainOperationManager.CanRedo;

            workspaceManager.BlockManager.SelectionChanged += (ss, ee) =>
            {
                var anySelected = workspaceManager.BlockManager.SelectedItems.Any();
                mnuEditCut.Enabled = anySelected;
                mnuEditCopy.Enabled = anySelected;
                mnuEditDelete.Enabled = anySelected;
            };

            Globals.MainOperationManager.Undone += MainOperationManagerSomethingDone;
            Globals.MainOperationManager.Redone += MainOperationManagerSomethingDone;

            mnuFileNew.Click += MenuFileNewClick;
            mnuFileLoad.Click += MenuFileLoadClick;
            mnuFileSave.Click += MenuFileSaveClick;
            mnuFileSave.Click += SendShaderToEffectMakerWhenSave;
            mnuFileSaveAs.Click += MenuFileSaveAsClick;
            mnuFileSaveAs.Click += SendShaderToEffectMakerWhenSave;
            mnuFileRecent.DropDownOpened += MenuFileRecentDropDownOpened;

            mnuFileClose.Click += MenuFileCloseClick;
            mnuFileReloadDefs.Click += MenuFileReloadDefsClick;

            mnuEditUndo.Click += MenuEditUndoClick;
            mnuEditRedo.Click += MenuEditRedoClick;
            mnuEditCut.Click += MenuEditCutClick;
            mnuEditCopy.Click += MenuEditCopyClick;
            mnuEditPaste.Click += MenuEditPasteClick;
            mnuEditDelete.Click += MenuEditDeleteClick;

            mnuViewEventsList.Click += MenuViewEventsListClick;
            mnuViewLastProducedShaderCodes.Click += MenuViewLastProducedShaderCodesClick;
            mnuViewOperationManager.Click += MenuViewHistoryClick;

            mnuShaderBuild.Click += MenuShaderBuildClick;
            mnuShaderUpdateEffectMaker.Click += MenuShaderUpdateEffectMakerClick;
            mnuShaderGenerate.Click += MenuShaderGenerateClick;
            mnuShaderSaveMathFunctions.Click += MenuShaderSaveMathFunctionsClick;

            mnuOptions.Click += MenuOptionsClick;

            mnuHelpAbout.Click += MenuHelpAboutClick;
            mnuHelpOpenExeFolder.Click += MenuHelpOpenExeFolderClick;
            mnuOpenCombinerNodeEditor.Click += MenuOpenCombinerNodeEditorClick;

            lblQuickInfo.ForeColorChanged += (ss, ee) => pnlRenderSurfaceContainer.BackColor = lblQuickInfo.ForeColor;

            rendererManager.CurrentRendererChanged += CurrentRendererChanged;
            rendererManager.RenderSurfaceCreated += RenderSurfaceCreated;

            generationManager = new GenerationManager(workspaceManager.BlockManager);

            definitionsManager = new DefinitionsManager(workspaceManager.BlockManager, lstBlockCollection);
            definitionsManager.RequestRender += (ss, ee) =>
                {
                    if (rendererManager.RenderSurface != null)
                        rendererManager.RenderSurface.Invalidate();
                };

            definitionsManager.DefinitionsContainer.BindingsUpdated += this.DefinitionsContainer_BindingsUpdated;

            workspaceManager.PreviewController.DefinitionContainer = definitionsManager.DefinitionsContainer;
            this.effectBlockValuePanel1.DefinitionContainer = definitionsManager.DefinitionsContainer;

            HistoryForm.PrepareForm();

            // postpone initializations for main form to appear first

            this.BeginInvoke(() => rendererManager.LoadRenderersAsync());
            this.BeginInvoke(() => definitionsManager.LoadDefinitionsAsync(this));
        }

        private void MainOperationManagerSomethingDone(object sender, OperationEventArgs<OperationBase> e)
        {
            workspaceManager.InvalidateRender();
        }

        private void MenuEditUndoClick(object sender, EventArgs e)
        {
            workspaceManager.BlockManager.BeginShaderCodeGenerationLock();
            try
            {
                Globals.MainOperationManager.Undo();
            }
            finally
            {
                workspaceManager.BlockManager.EndShaderCodeGenerationLock();
            }
        }

        private void MenuEditRedoClick(object sender, EventArgs e)
        {
            workspaceManager.BlockManager.BeginShaderCodeGenerationLock();
            try
            {
                Globals.MainOperationManager.Redo();
            }
            finally
            {
                workspaceManager.BlockManager.EndShaderCodeGenerationLock();
            }
        }

        private void MenuEditCutClick(object sender, EventArgs e)
        {
            workspaceManager.BlockManager.CopyBlocks();
            if (workspaceManager.BlockManager.RemoveSelectedBlocks())
            {
                workspaceManager.BlockManager.InvalidateRender();
                mnuEditPaste.Enabled = workspaceManager.BlockManager.CopiedElements.Any();
            }
        }

        private void MenuEditCopyClick(object sender, EventArgs e)
        {
            workspaceManager.BlockManager.CopyBlocks();
            mnuEditPaste.Enabled = workspaceManager.BlockManager.CopiedElements.Any();
        }

        private void MenuEditPasteClick(object sender, EventArgs e)
        {
            var result = workspaceManager.BlockManager.PasteBlocks();
            if (result.Any())
            {
                workspaceManager.BlockManager.InvalidateRender();
            }
        }

        private void MenuEditDeleteClick(object sender, EventArgs e)
        {
            if (workspaceManager.BlockManager.RemoveSelectedBlocks())
                workspaceManager.BlockManager.InvalidateRender();
        }

        private RenderSurface previousRenderSurface;
        private void RenderSurfaceCreated(object sender, EventArgs e)
        {
            if (previousRenderSurface != null)
            {
                previousRenderSurface.DragEnter -= RenderSurfaceDragEnter;
                previousRenderSurface.DragDrop -= RenderSurfaceDragDrop;
            }

            if (rendererManager.RenderSurface != null)
            {
                rendererManager.RenderSurface.DragEnter += RenderSurfaceDragEnter;
                rendererManager.RenderSurface.DragDrop += RenderSurfaceDragDrop;
            }

            previousRenderSurface = rendererManager.RenderSurface;

            workspaceManager.BlockManager.SelectionChanged += blockManager_SelectionChanged;
        }

        private void RenderSurfaceDragEnter(object sender, DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
                e.Effect = DragDropEffects.Copy;
        }

        private void RenderSurfaceDragDrop(object sender, DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                var files = (string[])e.Data.GetData(DataFormats.FileDrop);
                var extension = Path.GetExtension(files[0]);

                if (extension == FileExtensions.ExtEcmbFile)
                {
                    projectManager.Open(files[0]);
                }
            }
        }

        private void CurrentRendererChanged(object sender, EventArgs e)
        {
            lstBlockCollection.SetRenderer(rendererManager.CurrentRenderer);
        }

        private void MenuOptionsClick(object sender, EventArgs e)
        {
            var oldOptions = OptionsManager.CloneOptions();

            var options = new OptionsForm();
            if (options.ShowDialog() == DialogResult.OK)
            {
                ApplyChangesFromOptions(oldOptions, Globals.Options);
                workspaceManager.InvalidateRender();
            }
        }

        private void ApplyChangesFromOptions(Options oldOptions, Options newOptions)
        {
            if (newOptions.DefinitionPaths.ReloadDefinitionsOnPathChange)
            {
                var oldPaths = oldOptions.GetPathLookupInfo();
                var newPaths = newOptions.GetPathLookupInfo();

                if (newPaths.SequenceEqual(oldPaths) == false)
                {
                    definitionsManager.ReloadDefinitions(this);
                }
            }
        }

        private void InitializeCompiler()
        {
            // CAFE_ROOTが定義されていないと警告が出る&いちいちShader Build Failedが出るので無効化
#if false
            var cafeRootPath = Environment.GetEnvironmentVariable("CAFE_ROOT");
            if (cafeRootPath == null)
            {
                Reporting.Report(new EventReport(() => Localization.Messages.CANNOT_FIND_CAFE_ROOT_ENV_VAR,
                    ReportLevel.Warning, ReportCategory.Application, null, null, null));
            }
            else
                ShaderGenerator.CafeCompiler.Compiler.TryInitialize(cafeRootPath);
#endif
        }

        private void MenuFileNewClick(object sender, EventArgs e)
        {
            if (projectManager == null)
                return;

            projectManager.Close();
        }

        private void MenuFileLoadClick(object sender, EventArgs e)
        {
            if (projectManager == null)
                return;

            projectManager.Open();
        }

        private void MenuFileSaveClick(object sender, EventArgs e)
        {
            if (projectManager == null)
                return;

            projectManager.Save();
        }

        private void MenuFileSaveAsClick(object sender, EventArgs e)
        {
            if (projectManager == null)
                return;

            projectManager.SaveAs();
        }

        private void MenuFileRecentDropDownOpened(object sender, EventArgs e)
        {
            if (projectManager.MostRecentlyUsed.Any())
            {
                var dropDownList = projectManager.MostRecentlyUsed
                    .Select(CreateRecentMenuItem)
                    .ToArray();

                mnuFileRecent.DropDownItems.Clear();
                mnuFileRecent.DropDownItems.AddRange(dropDownList);
            }
            else
            {
                var dropDownList = new ToolStripItem[] { mnuEmptyHistory };
                mnuFileRecent.DropDownItems.Clear();
                mnuFileRecent.DropDownItems.AddRange(dropDownList);
            }
        }

        private ToolStripMenuItem CreateRecentMenuItem(string filename, int index)
        {
            var isEnabled = false;
            string toolTip = null;

            try
            {
                if (File.Exists(filename))
                    isEnabled = true;
                else
                    toolTip = Localization.Messages.FILE_NOT_FOUND;
            }
            catch (Exception ex)
            {
                toolTip = ex.Message;
            }

            return new ToolStripMenuItem(
                string.Format("{0} {1}", index + 1, Core.CoreUtility.ShortenPath(filename)),
                null,
                delegate { projectManager.Open(filename); })
                {
                    Enabled = isEnabled,
                    ToolTipText = toolTip,
                };
        }

        private void MenuFileReloadDefsClick(object sender, EventArgs e)
        {
            definitionsManager.ReloadDefinitions(this);
        }

        private void MenuFileCloseClick(object sender, EventArgs e)
        {
            Close();
        }

        private void MenuViewEventsListClick(object sender, EventArgs e)
        {
            Reporting.ShowReportForm();
        }

        private void MenuViewLastProducedShaderCodesClick(object sender, EventArgs e)
        {
            if (workspaceManager.BlockManager.ReviewLastProducedShaders() == false)
                Globals.QuickInfo.SetQuickInfo(Localization.Controls.QUICK_INFO_NO_SHADER_PRODUCED, false);
        }

        private void MenuViewHistoryClick(object sender, EventArgs e)
        {
            HistoryForm.ShowForm();
        }

        /// <summary>
        /// EffectCombinerからEffectMakerへ、コンバイナシェーダのキャッシュを送る.
        /// 送信に失敗したときに、イベントログを表示するかどうか選択できる.
        /// </summary>
        /// <param name="showForm">送信失敗時にイベントログを表示するか</param>
        private void SendShaderToEffectMaker(bool showForm)
        {
            if (projectManager.Filename == null)
            {
                if (projectManager.IsModified == false)
                    return;

                projectManager.Save();
                if (projectManager.Filename == null)
                {
                    Globals.QuickInfo.SetQuickInfo(Localization.Controls.QUICK_INFO_EFFECT_MAKER_UPDATE_ABORTED,
                        false, System.Drawing.Color.Orange);
                    return;
                }
            }

            string shaderMainName = "main";  // EffectMaker へ送るシェーダコードはメイン関数名を "main" に固定する
            var task = workspaceManager.BlockManager.ProduceShaderCodesAsync(false, false, shaderMainName);

            if (task == null)
            {
                Globals.QuickInfo.SetQuickInfo(Localization.Controls.QUICK_INFO_ALREADY_GENERATING, true, System.Drawing.Color.Orange);
                return;
            }

            task.ContinueWith(t =>
            {
                if (t.IsFaulted)
                {
                    if (t.Exception != null)
                    {
                        Reporting.Report(new EventReport(
                            () => t.Exception.Message,
                            ReportLevel.Error,
                            ReportCategory.ShaderGeneration,
                            null,
                            null,
                            t.Exception));
                    }
                    else
                    {
                        var dummy = t.Result;
                    }
                }
                else if (t.IsCompleted)
                {
                    if (t.Result.Length == 0 || string.IsNullOrWhiteSpace(t.Result[0]))
                        Globals.QuickInfo.SetQuickInfo(Localization.Controls.QUICK_INFO_NO_SHADER_PRODUCED, false, System.Drawing.Color.Orange);
                    else
                        communicationManager.SendShaderCodeAsync(projectManager.Filename, t.Result[0], showForm);
                }
            });
        }

        /// <summary>
        /// ユーザがF5を押した時に、EffectCombinerからEffectMakerへ、コンバイナシェーダのキャッシュを送る.
        /// </summary>
        /// <param name="sender">特に使用しない</param>
        /// <param name="e">イベントの情報</param>
        private void MenuShaderUpdateEffectMakerClick(object sender, EventArgs e)
        {
            // 送信に失敗した場合は、イベントログを表示する.
            this.SendShaderToEffectMaker(true);
        }

        /// <summary>
        /// ユーザがF7を押したときに、EffectCombinerのグラフ情報から、シェーダコードを作成し、最終出力プレビューを更新します。
        /// </summary>
        /// <param name="sender">特に使用しない</param>
        /// <param name="e">イベントの情報</param>
        private void MenuShaderGenerateClick(object sender, EventArgs e)
        {
            string shaderMainName = Globals.Options.ShaderGeneration.MainFunctionName;
            string[] shaderCode = this.workspaceManager.BlockManager.ProduceShaderCodesSync(true, true, shaderMainName);

            if (shaderCode.Length < 1)
            {
                MessageBox.Show(Localization.Messages.CANNOT_GENERATE_SHADER_CODE, Localization.Messages.CREATE_SHADER_FILE_ERROR, MessageBoxButtons.OK, MessageBoxIcon.Error);

                return;
            }

            string outputShaderFileName = Globals.Options.ShaderGeneration.OutputShaderFilename;

            // シェーダコードの出力先がオプションで設定されていない場合、初回に保存場所を確認する
            if (outputShaderFileName.IsNullOrEmpty())
            {
                SaveFileDialog saveFileDialog = new SaveFileDialog();
                saveFileDialog.Filter = string.Format("{0}|*.*", Localization.Messages.DLG_FILTER_ALL_FILES);

                if (saveFileDialog.ShowDialog() != DialogResult.OK)
                {
                    return;
                }
                else
                {
                    outputShaderFileName = saveFileDialog.FileName;
                }
            }

            try
            {
                GenerationManager.WriteGeneratedShaderToFile(shaderCode[shaderCode.Length - 1], outputShaderFileName);
            }
            catch
            {
                var sb = new StringBuilder();
                sb.AppendLine(Localization.Messages.FAILED_WRITE_FILE);
                sb.Append("Path : " + outputShaderFileName);

                MessageBox.Show(sb.ToString(), Localization.Messages.CREATE_SHADER_FILE_ERROR, MessageBoxButtons.OK, MessageBoxIcon.Error);

                return;
            }

            // 最終出力プレビューを更新
            workspaceManager.PreviewController.UpdateOutputPreview(workspaceManager.BlockManager);
        }

        /// <summary>
        /// ユーザがファイルを保存したときに、EffectCombinerからEffectMakerへコンバイナシェーダのキャッシュを送る.
        /// </summary>
        /// <param name="sender">特に使用しない</param>
        /// <param name="e">イベントの情報</param>
        private void SendShaderToEffectMakerWhenSave(object sender, EventArgs e)
        {
            // 送信に失敗した場合でも、イベントログは表示しない.
            this.SendShaderToEffectMaker(false);
        }

        private void MenuShaderBuildClick(object sender, EventArgs e)
        {
            if (workspaceManager != null && workspaceManager.BlockManager != null)
                workspaceManager.BlockManager.RunShaderGenerationProcessAsync(true);
        }

        /// <summary>
        /// "Save Math Functions" メニューを選択したときの処理を行います。
        /// g3d のプロジェクトにて Math 組み込みノードの実装が必要となる場面があったため
        /// 暫定機能として実装しています。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void MenuShaderSaveMathFunctionsClick(object sender, EventArgs e)
        {
            SaveFileDialog saveFileDialog = new SaveFileDialog
            {
                Filter = string.Format(
                    "{0}|*.glsl;*.vsh;*.gsh;*.fsh|{1}|*.*",
                    Localization.Messages.DLG_FILTER_SHADER_FILES,
                    Localization.Messages.DLG_FILTER_ALL_FILES),
                Title = Localization.Messages.DLG_SAVE_MATH_FUNCTION_TITLE,
            };

            DialogResult dialogResult = saveFileDialog.ShowDialog();

            if (dialogResult != DialogResult.OK)
            {
                return;
            }

            // 組み込み関数をファイルに出力する
            var shaderCodes = this.definitionsManager.GetOperatorBlockFunctions();
            File.WriteAllText(saveFileDialog.FileName, string.Join(string.Empty, shaderCodes));
        }

        private void MenuHelpAboutClick(object sender, EventArgs e)
        {
            var version = Assembly.GetEntryAssembly().GetName().Version;

            var sb = new StringBuilder();

            var versionText = string.Format("{0}.{1}.{2}.{3}",
                version.Major,
                version.Minor,
                version.Build,
                version.Revision);

            sb.AppendLine("Effect Combiner Editor");
            sb.Append(string.Format(Localization.Messages.VERSION_N, versionText));
            if (RepositoryRevision != null)
                sb.Append(string.Format(" [{0}]", RepositoryRevision));
            sb.AppendLine();

            sb.AppendLine();
            sb.AppendLine(string.Format(Localization.Messages.PROJECT_VERSION_N, ProjectManager.CurrentVersion));

            MessageBox.Show(sb.ToString(), Localization.Messages.ABOUT_EFFECT_COMBINER_EDITOR,
                MessageBoxButtons.OK, MessageBoxIcon.Information);
        }

        /// <summary>
        /// "Open Exe Folder" メニューを選択したときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void MenuHelpOpenExeFolderClick(object sender, EventArgs e)
        {
            string commandline = "/e, /select,\"" + IOConstants.ExecutableFilePath + "\"";
            Process.Start("EXPLORER.EXE", commandline);
        }

        private void MenuOpenCombinerNodeEditorClick(object sender, EventArgs e)
        {
            var combinerNodeEditor = IOConstants.ExecutableFolderPath + @"\CombinerNodeEditor\CombinerNodeEditor.exe";

            if (File.Exists(combinerNodeEditor))
            {
                Process.Start(combinerNodeEditor);
            }
        }

        protected override void OnFormClosing(FormClosingEventArgs e)
        {
            base.OnFormClosing(e);

            bool cancel;
            projectManager.ApplicationClose(out cancel);

            if (cancel == false)
            {
                OptionsExtensions.RecordFormWindowState(Globals.Options.EnvironmentSettings.MainWindow, this);
                OptionsManager.Save();
                localizationSubscription.Dispose();
                communicationManager.CloseConnection();

                // Kill CombinerViewer
                Process[] ps = Process.GetProcessesByName(PreviewController.CombinerViewerProcessName);
                foreach (Process p in ps)
                {
                    p.Kill();
                }
            }

            e.Cancel = cancel;
        }

        private void ProjectManagerFilenameChanged(object sender, EventArgs e)
        {
            SetTitle();
        }

        private void ProjectManagerIsModifiedChanged(object sender, EventArgs e)
        {
            SetTitle();
        }

        private void ProjectManagerOpened(object sender, EventArgs e)
        {
            workspaceManager.InvalidateRender();
        }

        private void ProjectManagerClosed(object sender, EventArgs e)
        {
            Globals.MainOperationManager.Clear();
            workspaceManager.InvalidateRender();
        }

        private void SetTitle()
        {
            var version = Assembly.GetEntryAssembly().GetName().Version;

            var title = string.Format("Effect Combiner Editor v{0}.{1}", version.Major, version.Minor);

            if (projectManager != null)
            {
                if (projectManager.Filename != null)
                {
                    var file = System.IO.Path.GetFileName(projectManager.Filename);
                    if (projectManager.IsModified)
                        title += " - * " + file;
                    else
                        title += " - " + file;
                }
                else if (projectManager.IsModified)
                    title += " - *";
            }
            this.Text = title;
        }

        private void blockManager_SelectionChanged(object sender, EventArgs e)
        {
            var selectedBlocks = workspaceManager.BlockManager.SelectedItems
                .Select(b => b.BlockDefinition)
                .Distinct(b => b.Guid.GetHashCode(), (b1, b2) => b1.Guid == b2.Guid);

            lstBlockCollection.SelectBlocks(selectedBlocks.ToArray());
        }

        /// <summary>
        /// ブロック定義と関数定義のバインディングが更新されたときの処理を行います。
        /// このイベントは、ブロック定義の読み込みが完了したときに呼ばれます。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void DefinitionsContainer_BindingsUpdated(object sender, BlockBindingEventArgs e)
        {
            // uniform 変数リストを更新
            {
                Primitives.Generation.Globals.UniformManager.Clear();

                var uniformBlockDefinitions = this.definitionsManager.DefinitionsContainer.Blocks
                    .Where(p => p is BlockDefinitionWithSource)
                    .Where(p => p.Uniform != null);

                foreach (var blockDefinition in uniformBlockDefinitions)
                {
                    var uniformDefinition = blockDefinition.Uniform;

                    Primitives.Generation.Globals.UniformManager.AddUniformParameter(uniformDefinition.Name, uniformDefinition.Type);
                }

                Primitives.Generation.Globals.UniformManager.InitializeValues();
            }
        }

        protected override void WndProc(ref Message m)
        {
            workspaceManager?.PreviewController?.ProcessWindowMessage(ref m);

            base.WndProc(ref m);
        }

        private void InvalidateElementHostViewer()
        {
            // Viewer 埋め込み用の ElementHost を右上に設置します。
            int top = this.pnlRoot.Top + this.pnlRoot.Panel2.Top + 10;
            this.elhViewer.Top = top;

            int left = this.Width - elhViewer.Width - 40;

            this.elhViewer.Left = left;
            this.Invalidate();
        }

        private void MainForm_SizeChanged(object sender, EventArgs e)
        {
            this.InvalidateElementHostViewer();
        }
    }
}
