﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectCombiner.Primitives;
using OperationManager.Core;
using EffectCombiner.Primitives.Operations;
using EffectCombiner.Core;

namespace EffectCombiner.Editor
{
    public partial class HistoryForm : Form
    {
        public HistoryForm()
        {
            InitializeComponent();

            lstOperations.Items.AddRange(Globals.MainOperationManager.Operations
                .Select(i => new HistoryListViewItem(i))
                .Cast<ListViewItem>()
                .ToArray());

            lstOperations.SelectedIndexChanged += lstOperations_SelectedIndexChanged;

            Globals.MainOperationManager.OperationsAdded += OnOperationsAdded;
            Globals.MainOperationManager.OperationsDiscarded += OnOperationsDiscarded;
            Globals.MainOperationManager.OperationsCleared += OnOperationsCleared;

            Globals.MainOperationManager.Redone += OnRedone;
            Globals.MainOperationManager.Undone += OnUndone;

            Globals.MainOperationManager.CurrentGuidAcquired += OnCurrentGuidAcquired;

            lstOperations.MouseDoubleClick += lstOperations_MouseDoubleClick;

            var localizationSubscription = Globals.Localization.RegisterLocalization(() =>
            {
                Text = Localization.Controls.FORM_ACTION_STACK_TITLE;
            });

            FormClosed += (ss, ee) => localizationSubscription.Dispose();
        }

        private void lstOperations_MouseDoubleClick(object sender, MouseEventArgs e)
        {
            if (e.Button != MouseButtons.Left)
            {
                return;
            }

            var item = (HistoryListViewItem)lstOperations.GetItemAt(e.X, e.Y);
            var operationManager = Globals.MainOperationManager;

            var undo = item.Index < operationManager.Index;

            while (true)
            {
                if (operationManager.Index == 0)
                {
                    break;
                }

                if (operationManager.Operations.ElementAt(operationManager.Index - 1).Identifier == item.Operation.Identifier)
                {
                    break;
                }

                if (undo)
                {
                    operationManager.Undo();
                }
                else
                {
                    operationManager.Redo();
                }
            }

            SetIndex();
        }

        private void OnOperationsCleared(object sender, EventArgs e)
        {
            foreach (var item in lstOperations.Items.Cast<HistoryListViewItem>())
            {
                item.Dispose();
            }

            lstOperations.Items.Clear();
        }

        private void OnCurrentGuidAcquired(object sender, CurrentGuidAcquiredEventArgs e)
        {
            var identifier = e.Identifier;
            var operationManager = Globals.MainOperationManager;

            foreach (var operation in operationManager.Operations)
            {
                if (operation.Identifier == identifier)
                {
                    foreach (HistoryListViewItem lvItem in lstOperations.Items)
                    {
                        if (lvItem.Operation == operation)
                        {
                            lvItem.SetMarked();
                        }
                    }
                }
            }
        }

        private void SetIndex()
        {
            lstOperations.SelectedIndexChanged -= lstOperations_SelectedIndexChanged;

            try
            {
                int index = Globals.MainOperationManager.Index - 1;

                for (var i = 0; i < lstOperations.Items.Count; i++)
                {
                    lstOperations.Items[i].ForeColor = i <= index ? SystemColors.WindowText : Color.LightGray;
                    lstOperations.Items[i].Selected = false;
                }

                if (0 <= index && index < lstOperations.Items.Count)
                {
                    lstOperations.Items[index].Selected = true;
                }
            }
            finally
            {
                lstOperations.SelectedIndexChanged += lstOperations_SelectedIndexChanged;
            }
        }

        private void lstOperations_SelectedIndexChanged(object sender, EventArgs e)
        {
            SetIndex();
        }

        private void OnRedone(object sender, OperationEventArgs<OperationBase> e)
        {
            SetIndex();
        }

        private void OnUndone(object sender, OperationEventArgs<OperationBase> e)
        {
            SetIndex();
        }

        private static HistoryForm form;

        public static void PrepareForm()
        {
            form = new HistoryForm();
        }

        public static void ShowForm()
        {
            form.Show();
            form.BringToFront();
        }

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);
            OptionsExtensions.ApplyRecordedFormWindowState(Globals.Options.EnvironmentSettings.OperationManagerWindow, this);
        }

        protected override void OnFormClosing(FormClosingEventArgs e)
        {
            base.OnFormClosing(e);

            OptionsExtensions.RecordFormWindowState(Globals.Options.EnvironmentSettings.OperationManagerWindow, this);

            if (e.CloseReason != CloseReason.UserClosing)
            {
                return;
            }

            Hide();
            e.Cancel = true;
        }

        private void OnOperationsAdded(object sender, OperationsEventArgs<OperationBase> e)
        {
            using (new DrawingSuspender(lstOperations))
            {
                lstOperations.Items.AddRange(e.Operations
                    .Select(i => new HistoryListViewItem(i))
                    .Cast<ListViewItem>()
                    .ToArray());

                SetIndex();
            }
        }

        private void OnOperationsDiscarded(object sender, DiscardedOperationsEventArgs<OperationBase> e)
        {
            var operations = e.Operations.SelectMany(i1 => lstOperations.Items
                .Cast<HistoryListViewItem>()
                .Where(i2 => i2.Operation == i1));

            using (new DrawingSuspender(lstOperations))
            {
                foreach (var unit in operations)
                {
                    lstOperations.Items.Remove(unit);
                    unit.Dispose();
                }
            }

            SetIndex();
        }
    }

    public class HistoryListViewItem : ListViewItem, IDisposable
    {
        public OperationBase Operation { get; private set; }

        public HistoryListViewItem(OperationBase operation)
        {
            if (operation == null)
            {
                throw new ArgumentNullException("operation");
            }

            Operation = operation;

            SubItems[0].Text = operation.DisplayName;
            SubItems.Add(string.Empty);

            Globals.Localization.CultureChanged += LocalizationCultureChanged;
        }

        public void Dispose()
        {
            Globals.Localization.CultureChanged -= LocalizationCultureChanged;
        }

        private void LocalizationCultureChanged(object sender, EventArgs e)
        {
            if (isMarked)
            {
                SubItems[1].Text = Localization.Messages.TEXT_MARKED;
            }
        }

        private bool isMarked;

        public void SetMarked()
        {
            SubItems[1].Text = Localization.Messages.TEXT_MARKED;
            isMarked = true;
        }
    }
}
