﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectCombiner.Core;
using EffectCombiner.Core.Extensions;
using EffectCombiner.Editor.Controls;
using EffectCombiner.Primitives;
using EffectCombiner.Primitives.Blocks;
using EffectCombiner.Primitives.Generation;
using EffectCombiner.Primitives.Generation.AutoGen;
using EffectCombiner.Primitives.Generation.Semantic;
using EffectDefinitions;
using ShaderGenerator.Core;
using ShaderGenerator.GLSL;
using Workflow.Core;
using Globals = EffectCombiner.Primitives.Globals;

namespace EffectCombiner.Editor
{
    public class DefinitionsManager
    {
        public EffectDefinitionsContainer DefinitionsContainer { get; private set; }

        private readonly CompositionBlockManager blockManager;
        private readonly BlockCollectionControl blockCollection;

        public DefinitionsManager(CompositionBlockManager blockManager, BlockCollectionControl blockCollection)
        {
            this.blockManager = blockManager;
            this.blockCollection = blockCollection;

            DefinitionsContainer = new CachedEffectDefinitionsContainer();

            DefinitionsContainer.BlockDefinitionsAdded += BlockDefinitionsAdded;
            DefinitionsContainer.BlockDefinitionsRemoved += BlockDefinitionsRemoved;
            DefinitionsContainer.BlockDefinitionsUpdated += BlockDefinitionsUpdated;
            DefinitionsContainer.FunctionDefinitionsUpdated += FunctionDefinitionsUpdated;
            DefinitionsContainer.BindingsUpdated += BindingsUpdated;
            DefinitionsContainer.SourcesResolved += DefinitionsContainerSourcesResolved;
            DefinitionsContainer.FileLoadingFinished += DefinitionsContainerFileLoadingFinished;
            DefinitionsContainer.NewErrorsReported += EffectDefinitionsContainerNewErrors;

            // ビルドインのノード(Math,Constant,Comment)をここで生成してリストに追加する
            GenerateOperatorBlocks();
            GenerateConstantBlock();
            GenerateCommentBlock();

            SemanticHelper.SemanticCorrespondanceNotFound += SemanticHelper_SemanticCorrespondanceNotFound;
            SemanticHelper.SemanticOverDefined += SemanticHelper_SemanticOverDefined;
        }

        private static void SemanticHelper_SemanticOverDefined(object sender, InputPlugEventArgs<PlugValue> e)
        {
            var ip = (EffectInputPlug)e.InputPlug;
            var plugWithSource = ip.BlockDefinitionPlug as PlugWithSource;

            Reporting.Report(new EventReport(
                () => string.Format(Localization.Messages.SEMANTIC_OVER_DEFINED,
                    ip.BlockDefinitionPlug.Semantic,
                    ip.BlockElement.BlockDefinition.Name,
                    ip.BlockElement.BlockDefinition.Guid),
                ReportLevel.Error,
                ReportCategory.Definition,
                plugWithSource != null ? plugWithSource.SemanticSource.ToString() : null,
                null, null));
        }

        private static void SemanticHelper_SemanticCorrespondanceNotFound(object sender, InputPlugEventArgs<PlugValue> e)
        {
            var ip = (EffectInputPlug)e.InputPlug;
            var plugWithSource = ip.BlockDefinitionPlug as PlugWithSource;

            Reporting.Report(new EventReport(
                () => string.Format(Localization.Messages.SEMANTIC_CORRESPONDANCE_NOT_FOUND,
                    ip.BlockDefinitionPlug.Semantic,
                    ip.BlockElement.BlockDefinition.Name,
                    ip.BlockElement.BlockDefinition.Guid),
                ReportLevel.Error,
                ReportCategory.Definition,
                plugWithSource != null ? plugWithSource.SemanticSource.ToString() : null,
                null, null));
        }

        public event EventHandler RequestRender;

        private readonly ProgressionReporter progressionReporter = new ProgressionReporter(0);

        private bool requestStackClear;

        private void CheckOperationManager()
        {
            if (requestStackClear == false)
                return;

            requestStackClear = false;

            var message = Localization.Messages.BLOCK_DEFINITIONS_DISAPPEARED + "\r\n" +
                Localization.Messages.ACTION_ITEM_STACK_WILL_BE_DISCARDED;
            MessageBox.Show(message, Localization.Messages.DEFINITION_RELOAD, MessageBoxButtons.OK, MessageBoxIcon.Information);

            Globals.MainOperationManager.Clear();
        }

        public void LoadDefinitionsAsync(Form form)
        {
            if (Globals.Options.DefinitionPaths == null || Globals.Options.DefinitionPaths.Paths == null)
                return;

            var cts = new CancellationTokenSource();
            var ready = new ManualResetEvent(false);

            progressionReporter.UpdateMaximum(0);
            progressionReporter.ReportProgression(0, Localization.Controls.INITIALIZING);

            var waitForm = new WaitForm(cts, progressionReporter, ready);

            var jobDone = new ManualResetEvent(false);

            Task.Factory.StartNew(() =>
                {
                    try
                    {
                        DefinitionsContainer.ClearErrors();
                        DefinitionsContainer.SetDefinitionsPaths(cts.Token, Globals.Options.GetPathLookupInfo());
                        DefinitionsContainer.UpdateBindings();
                    }
                    catch (Exception ex)
                    {
                        Reporting.Report(new EventReport(() => ex.Message,
                            ReportLevel.Error,
                            ReportCategory.Definition,
                            null, null, ex));
                    }
                    finally
                    {
                        WaitHandle.SignalAndWait(jobDone, ready);
                        waitForm.SafeClose();
                    }
                });

            if (jobDone.WaitOne(0))
                ready.Set();
            else
                waitForm.ShowDialog(form);

            CheckOperationManager();
        }

        private int currentFile;

        private void DefinitionsContainerSourcesResolved(object sender, SourcesResolvedEventArgs e)
        {
            currentFile = 0;
            progressionReporter.UpdateMaximum(e.Sources.Length);
        }

        private void DefinitionsContainerFileLoadingFinished(object sender, FileIOEventArgs e)
        {
            progressionReporter.ReportProgression(++currentFile, e.Path);
        }

        private void BlockDefinitionsAdded(object sender, BlockDefinitionEventArgs e)
        {
            Primitives.Generation.Globals.SetBlockDefinitions(DefinitionsContainer.Blocks);

            blockCollection.Invoke(() => blockCollection.AddBlocks(e.Blocks.ToArray()));
/*
            var blocksUpdates = e.Blocks
                .SelectMany(x => blockManager.BlockElements
                    .Where(y => y.BlockDefinition.Guid == x.Guid)
                    .Select(y => Tuple.Create(y, x)))
                    .ToArray();

            foreach (var blockUpdate in blocksUpdates)
            {
                var workflowItem = ((EffectBlockElementBase)blockUpdate.Item1).WorkflowItem;
                var oldDefinition = blockUpdate.Item1.BlockDefinition;
                var newDefinition = blockUpdate.Item2;

                for (int i = 0; i < workflowItem.InputPlugs.Length; ++i)
                {
                    if (i >= newDefinition.InputPlugs.Length ||
                        oldDefinition.InputPlugs[i].Type != newDefinition.InputPlugs[i].Type)
                    {
                        Workflow.Core.ConnectionManager.Disconnect(workflowItem.InputPlugs[i]);
                    }
                }

                for (int i = 0; i < workflowItem.OutputPlugs.Length; ++i)
                {
                    if (i >= newDefinition.OutputPlugs.Length ||
                        oldDefinition.OutputPlugs[i].Type != newDefinition.OutputPlugs[i].Type)
                    {
                        Workflow.Core.ConnectionManager.Disconnect(workflowItem.OutputPlugs[i]);
                    }
                }

                blockUpdate.Item1.UpdateBlockDefinition(blockUpdate.Item2);
            }

            var handler = RequestRender;
            if (handler != null)
                handler(this, EventArgs.Empty);
*/
        }

        private void BlockDefinitionsRemoved(object sender, BlockDefinitionEventArgs e)
        {
            Primitives.Generation.Globals.SetBlockDefinitions(DefinitionsContainer.Blocks);

            blockCollection.Invoke(() => blockCollection.RemoveBlocks(e.Blocks.ToArray()));

            var blocks = blockManager.BlockElements
                .Where(b => e.Blocks.Any(bd => bd.Guid == b.BlockDefinition.Guid))
                .ToArray();

            if (blocks.Length == 0)
                return;

            requestStackClear = true;

            foreach (var block in blocks)
                block.UpdateBlockDefinition(new FakeBlockDefinition(block.BlockDefinition));

            var handler = RequestRender;
            if (handler != null)
                handler(this, EventArgs.Empty);
        }

        private void BlockDefinitionsUpdated(object sender, BlockDefinitionEventArgs e)
        {
            Primitives.Generation.Globals.SetBlockDefinitions(DefinitionsContainer.Blocks);

            blockCollection.Invoke(() => blockCollection.UpdateBlocks(e.Blocks.ToArray()));

            var blocksUpdates = e.Blocks
                .SelectMany(x => blockManager.BlockElements
                    .Where(y => y.BlockDefinition.Guid == x.Guid)
                    .Select(y => Tuple.Create(y, x)))
                .ToArray();

            if (blocksUpdates.Length == 0)
                return;

            foreach (var blockUpdate in blocksUpdates)
            {
                var oldDef = blockUpdate.Item1.BlockDefinition;
                var newDef = blockUpdate.Item2;

                var needInputsDisconnection = newDef.AreInputPlugsSameAs(oldDef) == false;
                var needOutputsDisconnection = newDef.AreOutputPlugsSameAs(oldDef) == false;

                requestStackClear |= (needInputsDisconnection || needOutputsDisconnection);

                blockUpdate.Item1.UpdateBlockDefinition(new EffectBlockDefinition(blockUpdate.Item2));
            }

            var handler = RequestRender;
            if (handler != null)
                handler(this, EventArgs.Empty);
        }

        private void FunctionDefinitionsUpdated(object sender, FunctionDefinitionEventArgs e)
        {
            foreach (var fd in e.Functions)
            {
                var blockDefs = DefinitionsContainer.BlocksByFunctionId(fd.Id);
                foreach (var blockDef in blockDefs)
                {
                    var localBlockDef = blockDef;
                    var blockElements = Globals.WorkspaceManager.BlockManager.BlockElements
                        .Where(b => b.BlockDefinition.Guid == localBlockDef.Guid)
                        .ToArray();

                    foreach (var blockElement in blockElements)
                    {
                        blockElement.UpdateBlockDefinition(new EffectBlockDefinition(localBlockDef));
                    }
                }
            }
        }

        private void BindingsUpdated(object sender, BlockBindingEventArgs e)
        {
            Primitives.Generation.Globals.SetBlockDefinitions(DefinitionsContainer.Blocks);

            var affectedBlocks = e.Blocks;
            blockCollection.UpdateTags(affectedBlocks);

            blockCollection.Invoke(() => blockCollection.UpdateBlocks(e.Blocks.ToArray()));
        }

        private void EffectDefinitionsContainerNewErrors(object sender, EventArgs e)
        {
            blockCollection.Invoke(() =>
                {
                    foreach (var error in DefinitionsContainer.Errors)
                        Reporting.Report(error.ToEventReport());
                });
        }

        public void ReloadDefinitions(Form form)
        {
            DefinitionsContainer.ClearErrors();
            LoadDefinitionsAsync(form);
        }

        ///
        /// 基本算術演算ノードを生成
        ///
        public void GenerateOperatorBlocks()
        {
            var reporter = new ReportingExceptionReporter(ReportCategory.Definition);
            var operatorDefinitionSets = OperatorDefinitionSetUtility.InstanceAllDefinitionSets(reporter);

            foreach (var defSet in operatorDefinitionSets)
            {
                DefinitionsContainer.AddFunctionDefinitions(defSet.BlockDefinition.FunctionDefinitions);
                DefinitionsContainer.AddBlockDefinitions(defSet.BlockDefinition);

                defSet.BlockDefinition.Preview.IsEnabled = true;

                DefinitionsContainer.AddTaggings(defSet.BlockDefinition.FunctionDefinitions.SelectMany(f => new[]
                {
                        new Tagging(null, "Math", f.Id),
                }).ToArray());

                var contentRepository = Globals.ContentRepository as CustomContentRepository;
                if (contentRepository != null)
                {
                    for (var i = 0; i < defSet.BlockDefinition.FunctionDefinitions.Length; i++)
                    {
                        var funcDef = defSet.BlockDefinition.FunctionDefinitions[i];
                        if (defSet.ShaderCodes != null)
                            contentRepository.SetCustomContent(funcDef.GetMangling(), defSet.ShaderCodes[i]);
                    }
                }
            }
        }

        /// <summary>
        /// 基本算術演算ノードの関数文字列を取得します。
        /// </summary>
        /// <returns>算術演算ノードの関数の文字列を返します。</returns>
        public string[] GetOperatorBlockFunctions()
        {
            var reporter = new ReportingExceptionReporter(ReportCategory.Definition);
            var operatorDefinitionsSets = OperatorDefinitionSetUtility.InstanceAllDefinitionSets(reporter);

            IEnumerable<string> shaderCodes = Enumerable.Empty<string>();

            foreach (var defSet in operatorDefinitionsSets)
            {
                // glsl 組み込み関数は、シェーダコードが定義されていない
                if (defSet.ShaderCodes != null)
                {
                    shaderCodes = shaderCodes.Concat(defSet.ShaderCodes);
                }
            }

            return shaderCodes.ToArray();
        }

        /// <summary>
        /// 定数ノードを生成
        /// </summary>
        private void GenerateConstantBlock()
        {
            var def = new ConstantBlockDefinition(new ShaderTypeDefinition("float"));
            DefinitionsContainer.AddFunctionDefinitions(def.FunctionDefinition);
            DefinitionsContainer.AddBlockDefinitions(def);
        }

        /// <summary>
        /// コメントノードを生成
        /// </summary>
        private void GenerateCommentBlock()
        {
            var def = new CommentBlockDefinition();
            DefinitionsContainer.AddFunctionDefinitions(def.FunctionDefinition);
            DefinitionsContainer.AddBlockDefinitions(def);
        }
    }
}
