﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using System.Windows.Forms;
using EffectCombiner.Core;
using EffectCombiner.Primitives.Constant;
using EffectCombiner.Primitives.Generation;

namespace EffectCombiner.Editor.Controls
{
    /// <summary>
    /// uniform 値を変更するコントロールです。
    /// </summary>
    public partial class UniformValueControl : UserControl
    {
        /// <summary>
        /// コントロールのマージンです。
        /// </summary>
        private const int ControlsMargin = 1;

        /// <summary>
        /// コントロールです。
        /// </summary>
        private Control[,] controls;

        /// <summary>
        /// リサイズをスキップするかどうか。
        /// </summary>
        private bool skipResizeEvent;

        /// <summary>
        /// タブキー押下中かどうか
        /// </summary>
        private bool tabKeyDown;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UniformValueControl()
        {
            this.InitializeComponent();
        }

        /// <summary>
        /// データ変更イベントです。
        /// </summary>
        public event EventHandler DataChanged;

        public event EventHandler DataValidated;

        /// <summary>
        /// 値の配列のX要素です。
        /// </summary>
        public int DimensionX { get; private set; }

        /// <summary>
        /// 値の配列のY要素です。
        /// </summary>
        public int DimensionY { get; private set; }

        /// <summary>
        /// データの型です。
        /// </summary>
        public PrimitiveShaderType DataType { get; private set; }

        /// <summary>
        /// 値です。
        /// </summary>
        public string[,] Values { get; private set; }

        /// <summary>
        /// 前の値です。
        /// </summary>
        public string[,] OldValues { get; private set; }

        /// <summary>
        /// 値を設定します。
        /// </summary>
        /// <param name="dataType">データの型</param>
        /// <param name="dimensionX">値の配列のX要素</param>
        /// <param name="dimensionY">値の配列のY要素</param>
        /// <param name="values">値</param>
        public void SetInfo(PrimitiveShaderType dataType, int dimensionX, int dimensionY, string[,] values)
        {
            this.DataType = dataType;
            this.DimensionX = dimensionX;
            this.DimensionY = dimensionY;
            this.Values = values;
            this.OldValues = CoreUtility.DuplicateArray(this.Values);

            // キー入力で値変更後、タブキーでコントロール切り替えたときに、UI再生成を行うとクラッシュするのでUI再生成を防止。
            if (!this.tabKeyDown)
            {
                this.RecreateControlArray();
            }
        }

        /// <summary>
        /// リサイズイベントです。
        /// </summary>
        /// <param name="e">イベント</param>
        protected override void OnResize(EventArgs e)
        {
            base.OnResize(e);

            if (this.skipResizeEvent == false)
            {
                this.ArrangeLayout();
            }
        }

        /// <summary>
        /// データが変更されたときに呼ばれます。
        /// </summary>
        protected virtual void OnDataChanged()
        {
            this.DataChanged?.Invoke(this, EventArgs.Empty);
        }

        protected virtual void OnDataValidated()
        {
            this.DataValidated?.Invoke(this, EventArgs.Empty);
        }

        /// <summary>
        /// タブキー押下時の処理を行います。
        /// </summary>
        /// <param name="forward">コントロールを循環して選択するかどうか</param>
        /// <returns>コントロールが選択されたかどうか</returns>
        protected override bool ProcessTabKey(bool forward)
        {
            this.tabKeyDown = true;
            return base.ProcessTabKey(forward);
        }

        /// <summary>
        /// フォーカスを失った時の処理を行います。
        /// </summary>
        /// <param name="sender">送信元</param>
        /// <param name="e">イベント情報</param>
        private void ValueControl_LostFocus(object sender, EventArgs e)
        {
            // タブキー押上の取得は、keyup イベント時には、既にフォーカスが変わっているので、
            // 末端の ( vec4 の場合 w 要素) NumericUpDown 選択中の keyup イベントが取得できないため、 LostFocus で対応。
            this.tabKeyDown = false;

            base.OnLostFocus(e);
        }

        /// <summary>
        /// コントロールの値が変更されたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント情報</param>
        private void OnControlValueChanged(object sender, EventArgs e)
        {
            this.UpdateValuesWithUserInterface();
            this.OnDataChanged();
        }

        /// <summary>
        /// UIからの値の変更です。
        /// </summary>
        private void UpdateValuesWithUserInterface()
        {
            if (this.Values == null || this.controls == null)
            {
                return;
            }

            var typeDescriptor = DataTypeDescriptors.Descriptors.Single(d => d.DataType == this.DataType);

            for (var y = 0; y < this.DimensionY; y++)
            {
                for (var x = 0; x < this.DimensionX; x++)
                {
                    var value = typeDescriptor.GetValue(this.controls[x, y]);

                    if (this.Values != null)
                    {
                        this.Values.SetValue(value, x, y);
                    }
                }
            }
        }

        /// <summary>
        /// コントロールを生成します。
        /// </summary>
        private void RecreateControlArray()
        {
            this.SuspendLayout();

            try
            {
                var typeDescriptor = DataTypeDescriptors.Descriptors.Single(d => d.DataType == this.DataType);

                if (this.controls != null)
                {
                    foreach (var ctrl in this.controls)
                    {
                        var eventSubscription = ctrl.Tag as IDisposable;
                        if (eventSubscription != null)
                        {
                            eventSubscription.Dispose();
                        }
                    }
                }

                this.Controls.Clear();
                this.controls = new Control[this.DimensionX, this.DimensionY];

                for (var y = 0; y < this.DimensionY; y++)
                {
                    for (var x = 0; x < this.DimensionX; x++)
                    {
                        var ctrl = typeDescriptor.ProduceControl();

                        if (this.Values != null)
                        {
                            var value = this.Values[x, y];
                            if (value != null)
                            {
                                typeDescriptor.SetValue(ctrl, value);
                            }
                        }

                        var eventSubscription = typeDescriptor.AttachHandlers(ctrl, this.OnControlValueChanged, this.OnControlValueValidated);
                        ctrl.Tag = eventSubscription;
                        ctrl.LostFocus += this.ValueControl_LostFocus;

                        this.controls[x, y] = ctrl;
                        this.Controls.Add(ctrl);
                    }
                }
            }
            finally
            {
                this.ResumeLayout();
            }

            this.ArrangeLayout();
        }

        /// <summary>
        /// コントロールの値が検証されたときの処理を行います。
        /// </summary>
        /// <param name="sender">イベントの発生元</param>
        /// <param name="e">イベント</param>
        private void OnControlValueValidated(object sender, EventArgs e)
        {
            this.UpdateValuesWithUserInterface();

            if (CoreUtility.ArrayEquals(this.OldValues, this.Values))
            {
                return;
            }

            this.Values = CoreUtility.DuplicateArray(this.Values);
            this.OnDataValidated();
            this.OldValues = CoreUtility.DuplicateArray(this.Values);
        }

        /// <summary>
        /// UIを配置します。
        /// </summary>
        private void ArrangeLayout()
        {
            if (this.controls == null)
            {
                return;
            }

            //using (new DrawingSuspender(this))
            {
                var usableWidth = this.Width - ControlsMargin * (this.DimensionX + 1);
                var controlWidth = usableWidth / this.DimensionX;

                var top = ControlsMargin;
                var left = ControlsMargin;

                for (var y = 0; y < this.DimensionY; y++)
                {
                    var maxControlHeight = 0;

                    for (var x = 0; x < this.DimensionX; x++)
                    {
                        var ctrl = this.controls[x, y];

                        ctrl.Left = left;
                        ctrl.Top = top;

                        var minControlWidth = ctrl.MinimumSize.Width;
                        var maxControlWidth = ctrl.MaximumSize.Width > 0 ? ctrl.MaximumSize.Width : controlWidth;

                        var ctrlWidth = Math.Max(minControlWidth, Math.Min(controlWidth, maxControlWidth));
                        ctrl.Width = ctrlWidth;

                        left += ctrlWidth + ControlsMargin;
                        maxControlHeight = Math.Max(maxControlHeight, ctrl.Height);
                    }

                    top += maxControlHeight + ControlsMargin;
                    left = ControlsMargin;
                }

                this.skipResizeEvent = true;
                try
                {
                    this.Height = top;
                }
                finally
                {
                    this.skipResizeEvent = false;
                }
            }

            this.Update();
        }
    }
}
