﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace EffectCombiner.Editor.Controls
{
    public partial class TagContainerControl : UserControl
    {
        public TagContainerControl()
        {
            InitializeComponent();
            locked = true;

            btnScrollLeft.Left = 0;
            btnScrollLeft.Top = 0;
            btnScrollLeft.Height = Height;

            btnScrollRight.Left = Width - btnScrollRight.Width;
            btnScrollRight.Top = 0;
            btnScrollRight.Height = Height;

            pnlInnerContainer.Left = 0;
            pnlInnerContainer.Height = pnlOutterContainer.Height;

            pnlOutterContainer.Resize += OutterContainerResize;

            btnScrollLeft.MouseClick += ScrollLeftMouseClick;
            btnScrollRight.MouseClick += ScrollRightMouseClick;

            Relayout();
            SetScrollButtonsVisibility();
        }

        private int scrollDelta;
        public int ScrollDelta
        {
            get { return scrollDelta; }
            set { scrollDelta = Math.Max(1, value); }
        }

        private void ScrollLeftMouseClick(object sender, MouseEventArgs e)
        {
            pnlInnerContainer.Left = ComputeInnerContainerPosition(pnlInnerContainer.Left + ScrollDelta);
            SetScrollButtonsVisibility();
        }

        private void ScrollRightMouseClick(object sender, MouseEventArgs e)
        {
            pnlInnerContainer.Left = ComputeInnerContainerPosition(pnlInnerContainer.Left - ScrollDelta);
            SetScrollButtonsVisibility();
        }

        private void OutterContainerResize(object sender, EventArgs e)
        {
            pnlInnerContainer.Left = ComputeInnerContainerPosition(pnlInnerContainer.Left);
            SetScrollButtonsVisibility();
        }

        private int horizontalItemSpacing = 1;
        public int HorizontalItemSpacing
        {
            get { return horizontalItemSpacing; }
            set
            {
                if (horizontalItemSpacing == value)
                    return;

                horizontalItemSpacing = value;
                Relayout();
            }
        }

        private void Relayout()
        {
            var left = HorizontalItemSpacing;
            foreach (Control control in pnlInnerContainer.Controls)
            {
                control.Left = left;
                left += control.Width + HorizontalItemSpacing;

                if (control.Height > pnlInnerContainer.Height)
                    control.Top = 0;
                else
                    control.Top = (pnlInnerContainer.Height - control.Height) / 2;
            }
            pnlInnerContainer.Width = left;
        }

        private void SetScrollButtonsVisibility()
        {
            if (pnlInnerContainer.Width <= Width)
            {
                btnScrollLeft.Visible = false;
                btnScrollRight.Visible = false;
                pnlOutterContainer.Left = 0;
                pnlOutterContainer.Width = Width;
            }
            else
            {
                btnScrollLeft.Visible = true;
                btnScrollRight.Visible = true;
                pnlOutterContainer.Left = btnScrollLeft.Width;
                pnlOutterContainer.Width = Width - btnScrollLeft.Width - btnScrollRight.Width;
            }
        }

        private int ComputeInnerContainerPosition(int desiredPosition)
        {
            const int maxLeft = 0;
            var minLeft = Math.Min(0, pnlOutterContainer.Width - pnlInnerContainer.Width);

            return Math.Max(minLeft, Math.Min(desiredPosition, maxLeft));
        }

        public Control[] GetTags()
        {
            return pnlInnerContainer.Controls
                .Cast<Control>()
                .ToArray();
        }

        public void AddControl(Control control)
        {
            pnlInnerContainer.Controls.Add(control);
            Relayout();
            pnlInnerContainer.Left = ComputeInnerContainerPosition(pnlInnerContainer.Left);
            SetScrollButtonsVisibility();
        }

        public void RemoveControl(Control control)
        {
            pnlInnerContainer.Controls.Remove(control);
            Relayout();
            pnlInnerContainer.Left = ComputeInnerContainerPosition(pnlInnerContainer.Left);
            SetScrollButtonsVisibility();
        }

        public void ClearControls()
        {
            pnlInnerContainer.Controls.Clear();
            Relayout();
            pnlInnerContainer.Left = ComputeInnerContainerPosition(pnlInnerContainer.Left);
            SetScrollButtonsVisibility();
        }



        private readonly bool locked;

        protected override void OnControlAdded(ControlEventArgs e)
        {
            base.OnControlAdded(e);
            if (locked)
                throw new InvalidOperationException(Localization.Messages.EXCEPTION_LOCKED);
        }

        protected override void OnControlRemoved(ControlEventArgs e)
        {
            base.OnControlRemoved(e);
            if (locked)
                throw new InvalidOperationException(Localization.Messages.EXCEPTION_LOCKED);
        }
    }
}
