﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using Blocks.Core;
using EffectCombiner.Primitives.Blocks;
using EffectCombiner.Primitives.Generation;
using EffectCombiner.Primitives.Operations;
using OperationManager.Core;
using Workflow.Core;
using Globals = EffectCombiner.Primitives.Globals;

namespace EffectCombiner.Editor.Controls
{
    /// <summary>
    /// Constantの値を入力するパネル
    /// </summary>
    public partial class ConstantValuePanel : UserControl, IBlockElementEditPanel
    {
        /// <summary>
        /// 編集中のブロックエレメントです。
        /// </summary>
        private ConstantBlockElement blockElement;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ConstantValuePanel()
        {
            InitializeComponent();

            SetupSizersLayout();

            foreach (ShaderType shaderType in Enum.GetValues(typeof(ShaderType)))
                cboShaderType.Items.Add(GetShaderTypeText(shaderType));

            cboShaderType.SelectedIndexChanged += cboShaderType_SelectedIndexChanged;

            primitiveValueControl.DataChanged += primitiveValueControl_DataChanged;

            var localizationSubscription = Globals.Localization.RegisterLocalization(() =>
                {
                    lblShaderType.Text = Localization.Controls.CONST_VALUE_PANEL_GLSL_TYPE;
                    cboShaderType.Left = lblShaderType.Right + 6;
                });

            Disposed += (ss, ee) => localizationSubscription.Dispose();
        }

        /// <summary>
        /// ユーザがConstantのパネルに値を打ち込むと呼ばれるメソッド
        /// </summary>
        /// <param name="sender">送信元</param>
        /// <param name="e">イベントデータ</param>
        private void primitiveValueControl_DataChanged(object sender, EventArgs e)
        {
            if (this.blockElement != null)
            {
                // ユーザが値を変更する度に、履歴を積む
                var newValue = primitiveValueControl.Values;
                var oldValue = primitiveValueControl.OldValues;

                var operation = new ConstValueChangeOperation(Globals.WorkspaceManager.BlockManager, this.blockElement, oldValue, newValue);
                Globals.MainOperationManager.Add(operation, true);
            }
        }

        private void SetupSizersLayout()
        {
            const int sizerThickness = 4;

            pnlAngleSizer.Left = 0;
            pnlAngleSizer.Top = 0;
            pnlAngleSizer.Width = sizerThickness;
            pnlAngleSizer.Height = sizerThickness;
            pnlAngleSizer.Cursor = Cursors.SizeNWSE;
            pnlAngleSizer.MouseDown += pnlAnySizer_MouseDown;
            pnlAngleSizer.MouseMove += pnlAngleSizer_MouseMove;
            pnlAngleSizer.MouseUp += pnlAnySizer_MouseUp;

            pnlHorizontalSizer.Left = 0;
            pnlHorizontalSizer.Top = sizerThickness;
            pnlHorizontalSizer.Width = sizerThickness;
            pnlHorizontalSizer.Height = Height - sizerThickness;
            pnlHorizontalSizer.Cursor = Cursors.SizeWE;
            pnlHorizontalSizer.MouseDown += pnlAnySizer_MouseDown;
            pnlHorizontalSizer.MouseMove += pnlHorizontalSizer_MouseMove;
            pnlHorizontalSizer.MouseUp += pnlAnySizer_MouseUp;

            pnlVerticalSizer.Left = sizerThickness;
            pnlVerticalSizer.Top = 0;
            pnlVerticalSizer.Width = Width - sizerThickness;
            pnlVerticalSizer.Height = sizerThickness;
            pnlVerticalSizer.Cursor = Cursors.SizeNS;
            pnlVerticalSizer.MouseDown += pnlAnySizer_MouseDown;
            pnlVerticalSizer.MouseMove += pnlVerticalSizer_MouseMove;
            pnlVerticalSizer.MouseUp += pnlAnySizer_MouseUp;
        }

        private bool leftMouseDown;
        private Point screenMouseDownPosition;

        void pnlAnySizer_MouseDown(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                leftMouseDown = true;
                screenMouseDownPosition = MousePosition;
            }
        }

        void pnlAnySizer_MouseUp(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
                leftMouseDown = false;
        }

        void pnlAngleSizer_MouseMove(object sender, MouseEventArgs e)
        {
            if (leftMouseDown)
            {
                var dx = MousePosition.X - screenMouseDownPosition.X;
                var dy = MousePosition.Y - screenMouseDownPosition.Y;

                MouseMoveX(dx);
                MouseMoveY(dy);

                screenMouseDownPosition = MousePosition;
            }
        }

        void pnlHorizontalSizer_MouseMove(object sender, MouseEventArgs e)
        {
            if (leftMouseDown)
            {
                var dx = MousePosition.X - screenMouseDownPosition.X;

                MouseMoveX(dx);

                screenMouseDownPosition = MousePosition;
            }
        }

        void pnlVerticalSizer_MouseMove(object sender, MouseEventArgs e)
        {
            if (leftMouseDown)
            {
                var dy = MousePosition.Y - screenMouseDownPosition.Y;

                MouseMoveY(dy);

                screenMouseDownPosition = MousePosition;
            }
        }

        private void MouseMoveX(int deltaX)
        {
            this.Width -= deltaX;
            this.Left += deltaX;
        }

        private void MouseMoveY(int deltaY)
        {
            this.Height -= deltaY;
            this.Top += deltaY;
        }

        /// <summary>
        /// 指定されたブロックエレメントがこのパネルで編集可能かどうか取得します。
        /// </summary>
        /// <param name="blockElement">ブロックエレメント</param>
        /// <returns>編集可能なときはtrue、それ以外はfalseを返します。</returns>
        public bool CanEditBlockElement(BlockElementBase blockElement)
        {
            return (blockElement is ConstantBlockElement);
        }

        /// <summary>
        /// 編集するブロックエレメントを設定します。
        /// </summary>
        /// <param name="blockElement">編集するブロックエレメント</param>
        public void SetBlockElement(BlockElementBase blockElement)
        {
            // 前のブロックエレメントに登録していたイベントを解除する
            if (this.blockElement != null)
            {
                this.blockElement.ValuesChanged -= constantBlockElement_ValuesChanged;
            }

            this.blockElement = blockElement as ConstantBlockElement;

            // ブロックエレメントが null ならパネルを非表示にする
            if (this.blockElement == null)
            {
                this.Visible = false;
                return;
            }

            this.blockElement.ValuesChanged += constantBlockElement_ValuesChanged;

            var glslTypeName = this.blockElement.BlockDefinition.OutputPlugs[0].Type.TypeString;

            lockTypeSet = true;
            try
            {
                cboShaderType.SelectedIndex = (int)GlslTypingUtility.GetShaderTypeFromGlslTypeName(glslTypeName);
                UpdateData();
            }
            finally
            {
                lockTypeSet = false;
            }

            this.Visible = true;
        }

        /// <summary>
        /// 編集中のブロックエレメントを取得します。
        /// </summary>
        /// <returns>編集中のブロックエレメントを返します。</returns>
        public BlockElementBase GetBlockElement()
        {
            return this.blockElement;
        }

        private void constantBlockElement_ValuesChanged(object sender, EventArgs e)
        {
            var typeName = blockElement.BlockDefinition.OutputPlugs[0].Type.TypeString;
            var shaderType = GlslTypingUtility.GetShaderTypeFromGlslTypeName(typeName);

            cboShaderType.SelectedIndexChanged -= cboShaderType_SelectedIndexChanged;
            cboShaderType.SelectedIndex = (int)shaderType;
            cboShaderType.SelectedIndexChanged += cboShaderType_SelectedIndexChanged;

            var dimmensionX = GlslTypingUtility.GetDimensionX(shaderType);
            var dimmensionY = GlslTypingUtility.GetDimensionY(shaderType);
            var dataType = GlslTypingUtility.GetPrimitiveShaderType(shaderType);

            primitiveValueControl.SetInfo(dataType, dimmensionX, dimmensionY, blockElement.Values);
        }

        private bool lockTypeSet;

        private void cboShaderType_SelectedIndexChanged(object sender, EventArgs e)
        {
            var guiShaderType = (ShaderType)cboShaderType.SelectedIndex;

            var typeName = blockElement.BlockDefinition.OutputPlugs[0].Type.TypeString;
            var blockShaderType = GlslTypingUtility.GetShaderTypeFromGlslTypeName(typeName);

            if (guiShaderType == blockShaderType)
            {
                return; // WinForms bug, index didn't change but SelectedIndexChanged event is raised -_-;
            }

            this.UpdateData();

            Globals.WorkspaceManager.InvalidateRender();
        }

        private void UpdateData()
        {
            var shaderType = (ShaderType)cboShaderType.SelectedIndex;

            var dimmensionX = GlslTypingUtility.GetDimensionX(shaderType);
            var dimmensionY = GlslTypingUtility.GetDimensionY(shaderType);
            var dataType = GlslTypingUtility.GetPrimitiveShaderType(shaderType);
            var glslShaderTypeName = GlslTypingUtility.GetGlslTypeNameFromShaderType(shaderType);

            if (lockTypeSet == false)
            {
                var blockManager = Globals.WorkspaceManager.BlockManager;

                var oldBlockDef = (ConstantBlockDefinition)blockElement.BlockDefinition;
                var newBlockDef = new ConstantBlockDefinition(new EffectDefinitions.ShaderTypeDefinition(glslShaderTypeName));
                blockElement.UpdateBlockDefinition(newBlockDef);

                var operations = new List<IOperation>
                {
                    new ConstTypeChangeOperation(blockManager, blockElement, newBlockDef, oldBlockDef),
                };

                if (blockElement.WorkflowItem.OutputPlugs[0].RemoteInputPlugs.Any())
                {
                    var outputPlug = (EffectOutputPlug)blockElement.WorkflowItem.OutputPlugs[0];

                    operations.AddRange(outputPlug.RemoteInputPlugs
                        .Cast<EffectInputPlug>()
                        .Select(inputPlug => new ConnectionOperation(blockManager, false, inputPlug, outputPlug)));

                    ConnectionManager.Disconnect(outputPlug);

                    Globals.WorkspaceManager.InvalidateRender();
                }

                Globals.MainOperationManager.Add(new AggregateOperation(OperationType.ConstantTypeChange, operations));
            }

            blockElement.UpdateData();
            primitiveValueControl.SetInfo(dataType, dimmensionX, dimmensionY, blockElement.Values);
        }

        public static string GetShaderTypeText(ShaderType shaderType)
        {
            switch (shaderType)
            {
                case ShaderType.Boolean: return "Boolean";
                case ShaderType.Integer: return "Integer";
                case ShaderType.UnsignedInteger: return "Unsigned Integer";
                case ShaderType.Float: return "Float";
                case ShaderType.Double: return "Double";

                case ShaderType.BooleanVector2: return "Boolean Vector 2";
                case ShaderType.BooleanVector3: return "Boolean Vector 3";
                case ShaderType.BooleanVector4: return "Boolean Vector 4";

                case ShaderType.IntegerVector2: return "Integer Vector 2";
                case ShaderType.IntegerVector3: return "Integer Vector 3";
                case ShaderType.IntegerVector4: return "Integer Vector 4";

                case ShaderType.UnsignedIntegerVector2: return "Unsigned Integer Vector 2";
                case ShaderType.UnsignedIntegerVector3: return "Unsigned Integer Vector 3";
                case ShaderType.UnsignedIntegerVector4: return "Unsigned Integer Vector 4";

                case ShaderType.FloatVector2: return "Float Vector 2";
                case ShaderType.FloatVector3: return "Float Vector 3";
                case ShaderType.FloatVector4: return "Float Vector 4";

                case ShaderType.DoubleVector2: return "Double Vector 2";
                case ShaderType.DoubleVector3: return "Double Vector 3";
                case ShaderType.DoubleVector4: return "Double Vector 4";

                case ShaderType.Matrix2By2: return "Matrix 2 x 2";
                case ShaderType.Matrix3By2: return "Matrix 3 x 2";
                case ShaderType.Matrix4By2: return "Matrix 4 x 2";
                case ShaderType.Matrix2By3: return "Matrix 2 x 3";
                case ShaderType.Matrix3By3: return "Matrix 3 x 3";
                case ShaderType.Matrix4By3: return "Matrix 4 x 3";
                case ShaderType.Matrix2By4: return "Matrix 2 x 4";
                case ShaderType.Matrix3By4: return "Matrix 3 x 4";
                case ShaderType.Matrix4By4: return "Matrix 4 x 4";
            }

            throw new ArgumentException(string.Format(Localization.Messages.EXCEPTION_INVALID_ARGUMENT, "shaderType"), "shaderType");
        }
    }
}
