﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectCombiner.Core;
using EffectCombiner.Core.Extensions;
using EffectCombiner.Primitives;
using EffectCombiner.Primitives.Generation;
using EffectCombiner.Primitives.Generation.AutoGen;
using EffectDefinitions;
using Globals = EffectCombiner.Primitives.Globals;

namespace EffectCombiner.Editor.Controls
{
    /// <summary>
    /// MainFormのListView, TagViewを管理するコントロール
    /// </summary>
    public partial class BlockCollectionControl : UserControl
    {
        private readonly List<BlockListViewItem> availableBlocks = new List<BlockListViewItem>();

        /// <summary>
        /// アイテムソーター
        /// </summary>
        private BlockListViewItemComparer itemSorter = null;

        public BlockCollectionControl()
        {
            InitializeComponent();

            filterTimer = new System.Threading.Timer(_ => BeginInvoke((Action)ApplyFilter), null, -1, -1);

            txtFilterBlockCollection.TextChanged += txtFilterBlockCollection_TextChanged;
            lstBlockCollection.MouseUp += lstBlockCollection_MouseUp;
            trvBlockCollectionTagView.MouseUp += lstBlockCollection_MouseUp;

            var localizationSubscription = Globals.Localization.RegisterLocalization(() =>
                {
                    tabBlocksListView.Text = Localization.Controls.BLOCK_COLLECTION_TAB_LIST_VIEW;
                    tabBlocksTagsView.Text = Localization.Controls.BLOCK_COLLECTION_TAB_TAGS_VIEW;
                    txtFilterBlockCollection.Watermark = Localization.Controls.BLOCK_COLLECTION_SEARCH_WATERMARK;
                    colBlockName.Text = Localization.Controls.BLOCK_COLLECTION_COLUMN_NAME;
                    colInputOutput.Text = Localization.Controls.BLOCK_COLLECTION_COLUMN_INPUT_OUTPUT;
                    colBlockTags.Text = Localization.Controls.BLOCK_COLLECTION_COLUMN_TAGS;
                    colBlockDescription.Text = Localization.Controls.BLOCK_COLLECTION_COLUMN_DESCRIPTION;
                });

            Disposed += (ss, ee) => localizationSubscription.Dispose();

            lstBlockCollection.MouseDown += lstBlockCollection_MouseDown;
            lstBlockCollection.MouseMove += lstBlockCollection_MouseMove;
            lstBlockCollection.MouseEnter += lstBlockCollection_MouseEnter;
            lstBlockCollection.MouseLeave += lstBlockCollection_MouseLeave;

            trvBlockCollectionTagView.MouseDown += lstBlockCollection_MouseDown;
            trvBlockCollectionTagView.MouseMove += lstBlockCollection_MouseMove;
            trvBlockCollectionTagView.MouseEnter += lstBlockCollection_MouseEnter;
            trvBlockCollectionTagView.MouseLeave += lstBlockCollection_MouseLeave;

            // SubItemだけ自前で描画する.
            // 自前で描画する.
            this.lstBlockCollection.OwnerDraw = true;

            // SubItem以外はデフォルトを使用する.
            this.lstBlockCollection.DrawColumnHeader += (s, e) => e.DrawDefault = true;
            this.lstBlockCollection.DrawItem += (s, e) => e.DrawDefault = true;

            // SubItemは自前で描画する.
            this.lstBlockCollection.DrawSubItem += (s, e) =>
            {
                e.DrawDefault = false;

                // Get the bound for the sub item
                Rectangle rect = e.Bounds;

                // Draw background color
                if (e.Item.Selected)
                {
                    if (this.lstBlockCollection.Focused == true)
                    {
                        e.Graphics.FillRectangle(SystemBrushes.Highlight, rect);
                    }
                    else
                    {
                        e.Graphics.FillRectangle(
                            new SolidBrush(Color.FromArgb(255, 240, 240, 240)), rect);
                    }
                }
                else
                {
                    // 一番最初のカラムだった場合(Tagだった場合)は、背景色をタグごとに変更する.
                    if (e.ColumnIndex == 0)
                    {
                        Brush brush = Brushes.White;
                        string tagName = e.SubItem.Text;
                        if (!string.IsNullOrEmpty(tagName) && VisualResourceSet.ListViewTagBrushes.TryGetValue(tagName, out brush))
                        {
                            e.Graphics.FillRectangle(brush, rect);
                        }
                        else
                        {
                            e.Graphics.FillRectangle(VisualResourceSet.ListViewItemDefaultBrush, rect);
                        }
                    }
                }

                // Setup string format
                var format = new StringFormat(StringFormatFlags.NoWrap)
                {
                    Alignment = StringAlignment.Near,
                    LineAlignment = StringAlignment.Near
                };

                // Render the text
                Brush textBrush = new SolidBrush(e.SubItem.ForeColor);
                if (e.Item.Selected)
                {
                    if (this.lstBlockCollection.Focused == true)
                    {
                        textBrush = SystemBrushes.HighlightText;
                    }
                }
                else
                {
                    // 一番最初のカラムだった場合(Tagだった場合)は、背景色をタグごとに変更する.
                    if (e.ColumnIndex == 0)
                    {
                        textBrush = new SolidBrush(Color.White);
                    }
                }

                e.Graphics.DrawString(e.SubItem.Text, e.SubItem.Font, textBrush, rect, format);
            };

            // ソートの仕込み
            this.itemSorter = new BlockListViewItemComparer
            {
                // 全カラムで文字列でソートするように設定
                ColumnModes = new BlockListViewItemComparer.ComparerMode[]
                {
                    BlockListViewItemComparer.ComparerMode.String,
                    BlockListViewItemComparer.ComparerMode.String,
                    BlockListViewItemComparer.ComparerMode.String,
                    BlockListViewItemComparer.ComparerMode.String,
                    BlockListViewItemComparer.ComparerMode.String,
                }
            };

            // リストビューのソーターを設定する
            this.lstBlockCollection.ListViewItemSorter = this.itemSorter;
        }

        private Renderer2D.Core.Renderer renderer;

        public void SetRenderer(Renderer2D.Core.Renderer renderer)
        {
            if (renderer == null)
                throw new ArgumentNullException("renderer");

            this.renderer = renderer;
        }

        public void ClearBlocks()
        {
            foreach (var item in availableBlocks)
                item.Dispose();

            availableBlocks.Clear();
            lstBlockCollection.Items.Clear();
        }

        public void AddBlocks(BlockDefinition[] blocks)
        {
            var index = availableBlocks.Count;

            var newItems = blocks
                .Select((b, i) => new BlockListViewItem(index + i, b))
                .ToArray();

            availableBlocks.AddRange(newItems);

            using (new DrawingSuspender(lstBlockCollection))
            {
                lstBlockCollection.Items.AddRange(newItems);
                //lstBlockCollection.AutoResizeColumns(ColumnHeaderAutoResizeStyle.ColumnContent);
                foreach (ColumnHeader col in lstBlockCollection.Columns)
                    col.Width += 8;
            }
        }

        public void RemoveBlocks(BlockDefinition[] blocks)
        {
            var toRemove = availableBlocks
                .Where(item => blocks.Any(bd => bd.Guid == item.BlockDefinition.Guid))
                .ToArray();

            foreach (var item in toRemove)
            {
                availableBlocks.Remove(item);
                item.Dispose();
            }

            using (new DrawingSuspender(lstBlockCollection))
            {
                lstBlockCollection.Items.Clear();
                lstBlockCollection.Items.AddRange(availableBlocks.Cast<ListViewItem>().ToArray());
                lstBlockCollection.AutoResizeColumns(ColumnHeaderAutoResizeStyle.ColumnContent);
                foreach (ColumnHeader col in lstBlockCollection.Columns)
                    col.Width += 8;
            }
        }

        public void UpdateBlocks(BlockDefinition[] blocks)
        {
            var toRemove = availableBlocks
                .Where(item => blocks.Any(bd => bd.Guid == item.BlockDefinition.Guid))
                .ToArray();

            foreach (var item in toRemove)
            {
                availableBlocks.Remove(item);
                item.Dispose();
            }

            var index = availableBlocks.Count;
            var newItems = blocks
                .Select((b, i) => new BlockListViewItem(index + i, b))
                .ToArray();
            availableBlocks.AddRange(newItems);

            using (new DrawingSuspender(lstBlockCollection))
            {
                lstBlockCollection.Items.Clear();
                lstBlockCollection.Items.AddRange(availableBlocks.Cast<ListViewItem>().ToArray());
                lstBlockCollection.AutoResizeColumns(ColumnHeaderAutoResizeStyle.ColumnContent);
                foreach (ColumnHeader col in lstBlockCollection.Columns)
                    col.Width += 8;
            }
        }

        public void SelectBlocks(BlockDefinition[] blocks)
        {
            foreach (var item in lstBlockCollection.Items.Cast<BlockListViewItem>())
                item.Selected = blocks.Any(bd => bd.Guid == item.BlockDefinition.Guid);
        }

        private void lstBlockCollection_MouseUp(object sender, MouseEventArgs e)
        {
            Control ctrl = null;
            BlockDefinition blockDef = null;

            var lst = sender as ListView;
            if (lst != null)
            {
                if (lst.SelectedItems.Count != 1)
                    return;

                var item = lst.SelectedItems[0] as BlockListViewItem;
                if (item == null)
                    return;

                ctrl = lst;
                blockDef = item.BlockDefinition;
            }

            var trv = sender as TreeView;
            if (trv != null)
            {
                var item = trv.SelectedNode as BlockTreeNode;
                if (item == null)
                    return;

                ctrl = trv;
                blockDef = item.BlockDefinition;
            }

            // 処理対象とするノードの種類だったらここで引っ掛けてリターン
            if (blockDef is PolymorphicBlockDefinition ||
                blockDef is ConstantBlockDefinition ||
                blockDef is CommentBlockDefinition)
                return;

            if (ctrl == null || blockDef == null)
                return;

            if (e.Button == MouseButtons.Right)
            {
                var contextMenu = CreateContextMenu(blockDef);
                contextMenu.Show(ctrl, e.Location);
            }
        }

        private static ContextMenu CreateContextMenu(BlockDefinition blockDef)
        {
            var contextMenu = new ContextMenu();

            var showFileContextMenu = new MenuItem(Localization.Messages.OPEN_FILE_LOCATION);
            {
                showFileContextMenu.MenuItems.Add(
                    new MenuItem(
                        string.Format("{0}  {1}", blockDef.Name, Localization.Messages.INFO_DETAIL_BLOCK),
                        (ss, ee) => Core.CoreUtility.ShowFileInExplorer(blockDef.Source.Uri.ToString())));

                showFileContextMenu.MenuItems.AddRange((
                    from binding in blockDef.FunctionBindings
                    where binding.Definition != null &&
                          binding.Definition.Source != null
                    let uri = binding.Definition.Source.Uri
                    select new MenuItem(
                        string.Format("{0}  {1}", binding.Definition.Name, Localization.Messages.INFO_DETAIL_FUNCTION),
                        (ss, ee) => Core.CoreUtility.ShowFileInExplorer(uri.ToString())))
                    .ToArray());

                contextMenu.MenuItems.Add(showFileContextMenu);
            }

            var showSourceContextMenu = new MenuItem(Localization.Messages.SHOW_SOURCE_CODE_LOCATION);
            {
                EventHandler click = (ss, ee) =>
                {
                    var defloc = (DefinitionLocation)((MenuItem)ss).Tag;
                    var f = new SourceCodeViewForm();
                    f.SetInfo(defloc.Uri.ToString(),
                        Globals.ContentRepository.GetContent(defloc.Uri, Encoding.UTF8),
                        defloc.LineNumber, defloc.LinePosition, defloc.Length);
                    f.Show();
                };

                showSourceContextMenu.MenuItems.Add(
                    new MenuItem(string.Format("{0}  {1}", blockDef.Name, Localization.Messages.INFO_DETAIL_BLOCK),
                        click) { Tag = blockDef.Source });

                showSourceContextMenu.MenuItems.AddRange((
                    from binding in blockDef.FunctionBindings
                    where binding.Definition != null
                    let src = binding.Definition.Source
                    select new MenuItem(
                        string.Format("{0}  {1}", binding.Definition.Name, Localization.Messages.INFO_DETAIL_FUNCTION),
                        click) { Tag = src })
                    .ToArray());
            }

            contextMenu.MenuItems.Add(showSourceContextMenu);

            return contextMenu;
        }

        private Point mouseDownPosition;

        private void lstBlockCollection_MouseDown(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
                mouseDownPosition = e.Location;

            var trv = sender as TreeView;
            if (trv != null)
            {
                var tn = trv.GetNodeAt(e.Location);
                if (tn != null)
                    trv.SelectedNode = tn;
            }
        }

        private void lstBlockCollection_MouseMove(object sender, MouseEventArgs e)
        {
            var ctrl = (Control)sender;

            if (e.Button == MouseButtons.Left)
            {
                var dx = Math.Abs(e.X - mouseDownPosition.X);
                var dy = Math.Abs(e.Y - mouseDownPosition.Y);

                if (dx >= SystemInformation.DragSize.Width ||
                    dy >= SystemInformation.DragSize.Height)
                {
                    BlockDefinition[] blocks = null;

                    var lst = sender as ListView;
                    if (lst != null)
                    {
                        blocks = lst.SelectedItems
                            .Cast<BlockListViewItem>()
                            .Select(lvi => lvi.BlockDefinition)
                            .ToArray();
                    }
                    var trv = sender as TreeView;
                    if (trv != null)
                    {
                        if (trv.SelectedNode is BlockTreeNode)
                            blocks = new[] { ((BlockTreeNode)trv.SelectedNode).BlockDefinition };
                    }

                    if (blocks != null && blocks.Length > 0)
                        DoDragDrop(blocks, DragDropEffects.Copy);
                }
            }
            else if (e.Button == MouseButtons.None && blockPreviewForm != null)
            {
                //using (new DrawingSuspender(lst))
                {
                    BlockDefinition item = null;

                    var lst = sender as ListView;
                    if (lst != null)
                    {
                        var lvi = lst.GetItemAt(e.X, e.Y) as BlockListViewItem;
                        if (lvi != null)
                            item = lvi.BlockDefinition;
                    }

                    var trv = sender as TreeView;
                    if (trv != null)
                    {
                        var tn = trv.GetNodeAt(e.X, e.Y) as BlockTreeNode;
                        if (tn != null)
                            item = tn.BlockDefinition;
                    }

                    if (item != null)
                    {
                        var screenLocation = ctrl.PointToScreen(e.Location);
                        blockPreviewForm.SetBlockDefinition(item);
                        SetPreviewPosition(screenLocation, ctrl);
                        blockPreviewForm.Show();
                    }
                    else
                        blockPreviewForm.Hide();

                    blockPreviewForm.Update();
                }
            }
        }

        private void SetPreviewPosition(Point screenLocation, Control ctrl)
        {
            var screen = Screen.FromControl(ctrl);

            var left = screenLocation.X + 64;
            var top = screenLocation.Y + 64;

            if (left < screen.WorkingArea.Left)
                left = screen.WorkingArea.Left;
            else if (left > screen.WorkingArea.Right - blockPreviewForm.Width)
                left = screen.WorkingArea.Right - blockPreviewForm.Width;

            if (top < screen.WorkingArea.Top)
                top = screen.WorkingArea.Top;
            else if (top > screen.WorkingArea.Bottom - blockPreviewForm.Height)
                top = screen.WorkingArea.Bottom - blockPreviewForm.Height;

            blockPreviewForm.Left = left;
            blockPreviewForm.Top = top;
        }

        private BlockPreviewForm blockPreviewForm;

        private void lstBlockCollection_MouseEnter(object sender, EventArgs e)
        {
            var ctrl = (Control)sender;

            using (new DrawingSuspender(ctrl))
            {
                if (blockPreviewForm != null)
                {
                    blockPreviewForm.Close();
                    blockPreviewForm.Dispose();
                    blockPreviewForm = null;
                }

                var screenLocation = MousePosition;

                blockPreviewForm = new BlockPreviewForm();
                blockPreviewForm.SetRenderer(renderer);
                SetPreviewPosition(screenLocation, ctrl);

                if (blockPreviewForm.IsSurfaceReady == false)
                    return;

                blockPreviewForm.Show(this);
            }
        }

        private void lstBlockCollection_MouseLeave(object sender, EventArgs e)
        {
            var ctrl = (Control)sender;

            using (new DrawingSuspender(ctrl))
            {
                if (blockPreviewForm != null)
                {
                    blockPreviewForm.Close();
                    blockPreviewForm.Dispose();
                    blockPreviewForm = null;
                }
            }
        }

        private readonly System.Threading.Timer filterTimer;

        private void txtFilterBlockCollection_TextChanged(object sender, EventArgs e)
        {
            filterTimer.Change(500, -1); // reset timer
        }

        private void ApplyFilter()
        {
            var filterText = txtFilterBlockCollection.Text;

            Func<BlockDefinition, bool> toggleButtonsFilter = block =>
                {
                    var io = GetInOutStatus(block);
                    return (io == null && chkShowMiddles.Checked) ||
                        (io == true && chkShowInputs.Checked) ||
                        (io == false && chkShowOutputs.Checked);
                };

            Func<BlockDefinition, bool> searchFilter = block =>
                IsMatchingName(block, filterText) ||
                IsMatchingTags(block, filterText) ||
                IsMatchingDescription(block, filterText);

            var query = from block in availableBlocks
                        where toggleButtonsFilter(block.BlockDefinition)
                        where searchFilter(block.BlockDefinition)
                        select block;

            using (new DrawingSuspender(lstBlockCollection))
            {
                lstBlockCollection.Items.Clear();
                lstBlockCollection.Items.AddRange(query.ToArray());
            }
        }

        private static bool IsMatchingName(BlockDefinition block, string filterText)
        {
            if (block == null || block.Name == null)
                return false;
            return block.Name.IndexOf(filterText, StringComparison.InvariantCultureIgnoreCase) >= 0;
        }

        private static bool IsMatchingTags(BlockDefinition block, string filterText)
        {
            if (block == null || block.Tags == null)
                return false;
            return block.Tags.Any(t => t.IndexOf(filterText, StringComparison.InvariantCultureIgnoreCase) >= 0);
        }

        private static bool IsMatchingDescription(BlockDefinition block, string filterText)
        {
            if (block == null || block.Description == null)
                return false;
            return block.Description.IndexOf(filterText, StringComparison.InvariantCultureIgnoreCase) >= 0;
        }

        private static bool? GetInOutStatus(BlockDefinition definition)
        {
            var inCount = definition.InputPlugs.Length;
            var outCount = definition.OutputPlugs.Length;

            if (outCount == 0 && inCount > 0)
                return false;
            if (inCount == 0 && outCount > 0)
                return true;

            return null;
        }

        /// <summary>
        /// カラムがクリックされた時に実行されるメソッド
        /// </summary>
        /// <param name="sender">イベントのソース</param>
        /// <param name="e">イベントデータ</param>
        private void lstBlockCollection_ColumnClick(object sender, ColumnClickEventArgs e)
        {
            // カラムをクリックしたときに、ソートが実行されるようにする
            this.itemSorter.Column = e.Column;
            this.lstBlockCollection.Sort();
        }

        private void chkShowInputs_CheckedChanged(object sender, EventArgs e)
        {
            ApplyFilter();
        }

        private void chkShowMiddles_CheckedChanged(object sender, EventArgs e)
        {
            ApplyFilter();
        }

        private void chkShowOutputs_CheckedChanged(object sender, EventArgs e)
        {
            ApplyFilter();
        }

        public void UpdateTags(IEnumerable<BlockDefinition> affectedBlocks)
        {
            lstBlockCollection.Invoke(() =>
                {
                    var blocks = affectedBlocks.ToArray();

                    using (new DrawingSuspender(trvBlockCollectionTagView))
                    {
                        trvBlockCollectionTagView.Nodes.Clear();
                        trvBlockCollectionTagView.Nodes.AddRange(CreateTreeNodesFromBlockDefinitions(Primitives.Generation.Globals.BlockDefinitions.Values).ToArray());
                    }

                    var itemsToUpdate = lstBlockCollection.Items
                        .Cast<BlockListViewItem>()
                        .Where(x => blocks.Any(a => a.Guid == x.BlockDefinition.Guid));

                    foreach (var item in itemsToUpdate)
                        item.UpdateTags();
                });
        }

        private static IEnumerable<TreeNode> CreateTreeNodesFromBlockDefinitions(IEnumerable<BlockDefinition> blockDefinitions)
        {
            var rootNodes = new Dictionary<string, List<BlockTreeNode>>();

            var noneRootNodeList = new List<BlockTreeNode>();

            foreach (var blockDef in blockDefinitions)
            {
                if (blockDef.Tags.Any() == false)
                    noneRootNodeList.Add(new BlockTreeNode(blockDef));
                else
                {
                    foreach (var tag in blockDef.Tags)
                    {
                        List<BlockTreeNode> list;

                        if (rootNodes.TryGetValue(tag, out list) == false)
                        {
                            list = new List<BlockTreeNode>();
                            rootNodes.Add(tag, list);
                        }

                        list.Add(new BlockTreeNode(blockDef));
                    }
                }
            }

            if (noneRootNodeList.Count > 0)
            {
                var tn = new TreeNode("(None)");
                tn.Nodes.AddRange(noneRootNodeList.OrderBy(btn => btn.Text).Cast<TreeNode>().ToArray());
                yield return tn;
            }

            foreach (var elem in rootNodes.OrderBy(kv => kv.Key))
            {
                var tn = new TreeNode(elem.Key);
                tn.Nodes.AddRange(elem.Value.OrderBy(btn => btn.Text).Cast<TreeNode>().ToArray());
                yield return tn;
            }
        }
    }
}
