﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Reflection;
using System.Xml;
using System.Threading.Tasks;
using System.IO;
using System.Xml.Serialization;

namespace EffectCombiner.Data
{
    public class WorkflowDataHelper
    {
        private static readonly IWorkflowProcessor[] processors;
        private static readonly IWorkflowConverter[] converters;

        #region Initialization

        static WorkflowDataHelper()
        {
            processors = GetInstances<IWorkflowProcessor>();
            converters = GetInstances<IWorkflowConverter>();
        }

        private static T[] GetInstances<T>() where T : class
        {
            var instanceType = typeof(T);

            var asm = Assembly.GetExecutingAssembly();

            var query = from type in asm.GetTypes()
                        where type.IsInterface == false
                        where type.ContainsGenericParameters == false
                        where instanceType.IsAssignableFrom(type)
                        let instance = (T)CreateInstance(type)
                        where instance != null
                        select instance;

            return query.ToArray();
        }

        private static object CreateInstance(Type type)
        {
            try
            {
                return Activator.CreateInstance(type);
            }
            catch
            {
                return null;
            }
        }

        #endregion // Initialization

        public static bool FindVersion(Stream input, out ushort version)
        {
            using (var reader = XmlReader.Create(input, GetXmlReaderSettings()))
            {
                return FindVersion(reader, out version);
            }
        }

        public static bool FindVersion(string filename, out ushort version)
        {
            using (var reader = XmlReader.Create(filename, GetXmlReaderSettings()))
            {
                return FindVersion(reader, out version);
            }
        }

        private static XmlReaderSettings GetXmlReaderSettings()
        {
            return new XmlReaderSettings
            {
                IgnoreComments = true,
                IgnoreProcessingInstructions = true,
                IgnoreWhitespace = true,
            };
        }

        private static bool FindVersion(XmlReader reader, out ushort version)
        {
            version = 0;

            while (reader.Read())
            {
                if (reader.NodeType != XmlNodeType.Element || reader.LocalName != "project")
                    continue;

                var strVersion = reader.GetAttribute("version");
                if (string.IsNullOrWhiteSpace(strVersion) == false)
                {
                    try
                    {
                        version = XmlConvert.ToUInt16(strVersion);
                        return true;
                    }
                    catch { }
                }

                break;
            }

            return false;
        }

        public static object Deserialize<T>(Stream input, IWorkflowDataEventReporter reporter)
        {
            try
            {
                var serializer = new XmlSerializer(typeof(T));
                return serializer.Deserialize(input);
            }
            catch (Exception ex)
            {
                if (reporter != null)
                {
                    reporter.Report(new WorkflowDataEventReport
                    {
                        Message = () => ex.Message,
                        IsError = true,
                        Exception = ex,
                    });
                }
                return null;
            }
        }

        public static void Serialize<T>(Stream output, object data, IWorkflowDataEventReporter reporter)
        {
            var ns = new XmlSerializerNamespaces();
            ns.Add("", "");

            try
            {
                var serializer = new XmlSerializer(typeof(T));
                serializer.Serialize(output, data, ns);
            }
            catch (Exception ex)
            {
                if (reporter != null)
                {
                    reporter.Report(new WorkflowDataEventReport
                    {
                        Message = () => ex.Message,
                        IsError = true,
                        Exception = ex,
                    });
                }
            }
        }

        public static IWorkflowProcessor GetProcessor(ushort version)
        {
            return processors.FirstOrDefault(p => p.Version == version);
        }

        public static IWorkflowConverter GetUpgradeConverter(ushort sourceVersion, ushort targetVersion)
        {
            if (sourceVersion <= 0)
                throw new ArgumentException(string.Format(Messages.EXCEPTION_GREATER_EQUAL_ONE, "sourceVersion"), "sourceVersion");
            if (targetVersion <= 0)
                throw new ArgumentException(string.Format(Messages.EXCEPTION_GREATER_EQUAL_ONE, "targetVersion"), "targetVersion");
            if (targetVersion < sourceVersion)
                throw new ArgumentException(string.Format(Messages.EXCEPTION_GREATER_EQUAL_TO, "targetVersion", "sourceVersion."));
            if (converters == null || converters.Length == 0)
                throw new InvalidOperationException(Messages.EXCEPTION_NO_CONVERTER_AVAILABLE);

            for (var v = sourceVersion; v < targetVersion; v++)
            {
                var converter = converters
                    .FirstOrDefault(c => c.SourceVersion == v && c.TargetVersion == v + 1);
                if (converter != null)
                    continue;

                var message = string.Format(Messages.EXCEPTION_CONVERTER_MISSING, v, v + 1);
                throw new InvalidOperationException(message);
            }

            return new AggregateWorkflowConverter(sourceVersion, targetVersion, true, converters);
        }

        public static IWorkflowConverter GetDowngradeConverter(ushort sourceVersion, ushort targetVersion)
        {
            if (sourceVersion <= 0)
                throw new ArgumentException(string.Format(Messages.EXCEPTION_GREATER_EQUAL_ONE, "sourceVersion"), "sourceVersion");
            if (targetVersion <= 0)
                throw new ArgumentException(string.Format(Messages.EXCEPTION_GREATER_EQUAL_ONE, "targetVersion"), "targetVersion");
            if (targetVersion > sourceVersion)
                throw new ArgumentException(string.Format(Messages.EXCEPTION_SMALLER_EQUAL_TO, "targetVersion", "sourceVersion."));
            if (converters == null || converters.Length == 0)
                throw new InvalidOperationException(Messages.EXCEPTION_NO_CONVERTER_AVAILABLE);

            for (var v = sourceVersion; v > targetVersion; v--)
            {
                var converter = converters
                    .FirstOrDefault(c => c.SourceVersion == v && c.TargetVersion == v - 1);
                if (converter != null)
                    continue;

                var message = string.Format(Messages.EXCEPTION_CONVERTER_MISSING, v, v + 1);
                throw new InvalidOperationException(message);
            }

            return new AggregateWorkflowConverter(sourceVersion, targetVersion, false, converters);
        }

        private class AggregateWorkflowConverter : IWorkflowConverter
        {
            private readonly bool localIsUpgrading;
            private readonly IWorkflowConverter[] localConverters;

            public AggregateWorkflowConverter(
                ushort sourceVersion,
                ushort targetVersion,
                bool isUpgrading,
                IWorkflowConverter[] converters)
            {
                SourceVersion = sourceVersion;
                TargetVersion = targetVersion;
                localIsUpgrading = isUpgrading;
                localConverters = converters;
            }

            public ushort SourceVersion { get; private set; }
            public ushort TargetVersion { get; private set; }

            public object Convert(object sourceWorkflowData)
            {
                var source = sourceWorkflowData;

                if (localIsUpgrading)
                {
                    for (var v = SourceVersion; v < TargetVersion; v++)
                    {
                        var converter = localConverters
                            .FirstOrDefault(c => c.SourceVersion == v && c.TargetVersion == v + 1);
                        if (converter == null)
                            return null; // this case cannot happen, but need to satisfy static code analyzer
                        source = converter.Convert(source);
                    }
                }
                else
                {
                    for (var v = SourceVersion; v > TargetVersion; v--)
                    {
                        var converter = localConverters
                            .FirstOrDefault(c => c.SourceVersion == v && c.TargetVersion == v - 1);
                        if (converter == null)
                            return null; // this case cannot happen, but need to satisfy static code analyzer
                        source = converter.Convert(source);
                    }
                }

                return source;
            }
        }

        /*
        private class WorkflowProcessorEqualityComparer : IEqualityComparer<IWorkflowProcessor>
        {
            public bool Equals(IWorkflowProcessor x, IWorkflowProcessor y)
            {
                if (x == null || y == null)
                    return false;
                return x.Version == y.Version;
            }

            public int GetHashCode(IWorkflowProcessor obj)
            {
                if (obj == null)
                    return 0;
                return obj.Version.GetHashCode();
            }
        }
        */
    }
}
