﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectDefinitions;
using EffectCombiner.Primitives;
using Renderer2D.GDI;
using EffectCombiner.Core;

namespace EffectCombiner.BlockAssistant
{
    public partial class MainForm : Form
    {
        private readonly LocalizationManager localizationManager = new LocalizationManager();

        public MainForm()
        {
            InitializeComponent();

            localizationManager.RegisterLocalization(() =>
                {
                    lblShaderCode.Text = Localization.Controls.FORM_MAIN_SHADER_FUNCTION_PROTOTYPE;
                    lblDefinition.Text = Localization.Controls.FORM_MAIN_SHADER_DEFINITION;
                    chkGenerateFullDefinition.Text = Localization.Controls.FORM_MAIN_GENERATE_FULL_DEFINITION;
                    chkGenerateFullDefinition.Left = lblDefinition.Right + 6;
                });
        }

        private readonly PrototypeParser prototypeParser = new PrototypeParser();

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            txtCode.TextChanged += CodeTextChanged;

            prototypeParser.Report += PrototypeParserReport;
            prototypeParser.ParsingDone += PrototypeParserParsingDone;

            renderSurface.SetRenderer(new GDIRenderer());
            renderSurface.SizeChanged += RenderSurfaceSizeChanged;
            renderSurface.Parent.SizeChanged += RenderSurfaceParentSizeChanged;

            chkGenerateFullDefinition.CheckedChanged += CheckBoxGenerateFullDefinitionCheckedChanged;

            ResourcesManager.InitializeResources(renderSurface.Renderer);
        }

        private void CheckBoxGenerateFullDefinitionCheckedChanged(object sender, EventArgs e)
        {
            var blockDef = renderSurface.BlockDefinition;

            if (blockDef == null)
                return;

            var full = chkGenerateFullDefinition.Checked;

            renderSurface.BlockDefinition = blockDef;
            txtDefinition.Text = GenerateBlockXml(blockDef, full);
        }

        private void RenderSurfaceParentSizeChanged(object sender, EventArgs e)
        {
            UpdateBlockPreviewPosition();
        }

        private void RenderSurfaceSizeChanged(object sender, EventArgs e)
        {
            UpdateBlockPreviewPosition();
        }

        private void UpdateBlockPreviewPosition()
        {
            var parent = renderSurface.Parent;
            renderSurface.Left = (parent.Width - renderSurface.Width) / 2;
            renderSurface.Top = (parent.Height - renderSurface.Height) / 2;
        }

        private void CodeTextChanged(object sender, EventArgs e)
        {
            foreach (var item in lstParseReports.Items.OfType<IDisposable>())
                item.Dispose();
            lstParseReports.Items.Clear();

            prototypeParser.Parse(txtCode.Text);
        }

        private void PrototypeParserReport(object sender, ReportEventArgs e)
        {
            lstParseReports.Items.Add(new ParseReportListViewItem(e.Report));
        }

        private void PrototypeParserParsingDone(object sender, ParsingEventArgs e)
        {
            if (e.Info != null &&
                e.Info.Definition != null &&
                (e.Info.Definition.Parameters.Length > 0 ||
                 e.Info.Definition.ReturnType.IsVoid == false))
            {
                lstParseReports.Items.Add(
                    new ParseReportListViewItem(
                        new Report(
                            () => Localization.Messages.MSG_SUCCESS,
                            ReportLevel.Information,
                            null)));

                var full = chkGenerateFullDefinition.Checked;

                var block = GetBlockDefinitionFromFunctionDefinition(e.Info.Definition);
                renderSurface.BlockDefinition = block;

                txtDefinition.Text = GenerateBlockXml(block, full);
            }
            else
            {
                renderSurface.BlockDefinition = null;
                txtDefinition.Text = "";
            }

            renderSurface.Invalidate();
        }

        private string GenerateBlockXml(BlockDefinition block, bool full)
        {
            var sb = new StringBuilder();
            sb.AppendLine("/**");
            sb.AppendLine(XmlGeneration.FromFunctionDefinition(block.CurrentFunctionDefinition, full));
            sb.AppendLine();
            sb.AppendLine(XmlGeneration.FromBlockDefinition(block, full));
            sb.AppendLine("*/");

            return sb.ToString();
        }

        private static BlockDefinition GetBlockDefinitionFromFunctionDefinition(FunctionDefinition functionDefinition)
        {
            var inPlugs = new List<Plug>();
            var outPlugs = new List<Plug>();

            foreach (var p in functionDefinition.Parameters.Where(p => p.IsAnInput))
            {
                var name = p.Name;
                var target = name;
                if (p.Direction == ShaderTyping.ParameterDirection.Reference)
                    name += " (ref)";
                inPlugs.Add(new Plug(name, new ShaderTypeDefinition(p.Type.ToString()), target, true, null, null));
            }

            if (functionDefinition.ReturnType.IsVoid == false)
                outPlugs.Add(new Plug("return", new ShaderTypeDefinition(functionDefinition.ReturnType.ToString()), "return", true, null, null));

            foreach (var p in functionDefinition.Parameters.Where(p => p.IsAnOutput))
            {
                var name = p.Name;
                var target = name;
                if (p.Direction == ShaderTyping.ParameterDirection.Reference)
                    name = "(ref) " + name;
                outPlugs.Add(new Plug(name, new ShaderTypeDefinition(p.Type.ToString()), target, false, null, null));
            }

            var block = new BlockDefinition(null,
                MakeBetterName(functionDefinition.Name),
                "{" + Guid.NewGuid().ToString().ToUpper() + "}",
                inPlugs,
                outPlugs,
                new FunctionBinding(
                    null,
                    null,
                    functionDefinition.Id,
                    null,
                    functionDefinition.Name));

            block.BindToFunctionDefinition(functionDefinition);

            return block;
        }

        private static string MakeBetterName(string name)
        {
            if (string.IsNullOrWhiteSpace(name))
                return string.Empty;

            name = name.Trim(' ', '\t', '\r', '\n', '_');

            var list = new List<string>();
            var sb = new StringBuilder();

            foreach (var c in name)
            {
                if (char.IsUpper(c))
                {
                    if (sb.Length > 0)
                        list.Add(sb.ToString());
                    sb.Clear();
                }
                else if (c == '_')
                {
                    if (sb.Length > 0)
                        list.Add(sb.ToString());
                    sb.Clear();
                    continue;
                }

                sb.Append(c);
            }

            if (sb.Length > 0)
                list.Add(sb.ToString());

            for (var i = 0; i < list.Count; i++)
                list[i] = string.Format("{0}{1}", char.ToUpper(list[i][0]), new string(list[i].Skip(1).ToArray()));

            return string.Join(" ", list);
        }
    }
}
