﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// variables constants
// PluginMain InitGlobals ReadScriptParams WriteScriptParams
// DoPrepare
// DoStart OutputSliceInfo
// DoContinue
// DoFinish

//=============================================================================
// include
//=============================================================================
#include "NintendoSliceInfo.h"
#include "NintendoSliceInfoScripting.h"
#include "../NintendoFtx/NpsVersion.h"
#include "Resource.h"

using namespace nn::gfx::tool::nps;

//-----------------------------------------------------------------------------
//! @brief グローバルデータを初期化します。
//!        プラグインが最初に使用されたときに 1 度だけ呼ばれます。
//!
//! @param[in,out] globalPtr グローバルデータのポインタです。
//-----------------------------------------------------------------------------
static void InitGlobals(Ptr globalPtr)
{
    //RNoteTrace("init globals");
    GPtr globals = reinterpret_cast<GPtr>(globalPtr);
    globals->Init();
}

//-----------------------------------------------------------------------------
//! @brief スクリプトパラメータをリードします。
//!
//! @param[in,out] globals グローバルデータです。
//! @param[out] outputPath 出力ファイルパスを格納します。
//! @param[out] isImageOnly 画像のあるスライスのみ取得するなら true を格納します。
//!
//! @return エラーを返します。
//-----------------------------------------------------------------------------
static OSErr ReadScriptParams(
    GPtr globals,
    std::string& outputPath,
    bool& isImageOnly
)
{
    OSErr error = noErr;

    DescriptorKeyIDArray keyArray =
    {
        keyIn,
        nnSliceInfoKeyImageOnly,
        NULLID
    };

    if (DescriptorAvailable(NULL))
    {
        PIReadDescriptor token = OpenReader(keyArray);
        if (token)
        {
            DescriptorKeyID key = NULLID;
            DescriptorTypeID type = NULLID;
            int32 flags = 0;
            while (PIGetKey(token, &key, &type, &flags))
            {
                switch (key)
                {
                    case keyIn:
                    {
                        Handle hFileRef = NULL;
                        error = PIGetAlias(token, &hFileRef);
                        //RNoteTrace("file ref: %d: %p", error, hFileRef);
                        if (error == noErr && hFileRef != NULL)
                        {
                            char strBuf[512];
                            RHandle2CString(gStuff->handleProcs, hFileRef, strBuf, sizeof(strBuf));
                            PIDisposeHandle(hFileRef);
                            //RNoteTrace("in: [%s]", strBuf);
                            outputPath = strBuf;
                        }
                        break;
                    }

                    case nnSliceInfoKeyImageOnly:
                    {
                        Boolean boolVal;
                        error = PIGetBool(token, &boolVal);
                        isImageOnly = (boolVal == TRUE);
                        break;
                    }
                }
            }
            error = CloseReader(&token);
            //RNoteTrace("close reader: %d", error);
            if (error == errMissingParameter) // keyArray のキーが一部なかった場合はエラーにしません。
            {
                error = noErr;
            }
        }
        gQueryForParameters = PlayDialog();
    }

    return error;
}

//-----------------------------------------------------------------------------
//! @brief スクリプトパラメータをライトします。
//!
//! @param[in,out] globals グローバルデータです。
//!
//! @return エラーを返します。
//-----------------------------------------------------------------------------
static OSErr WriteScriptParams(GPtr globals)
{
    OSErr error = noErr;

    if (DescriptorAvailable(NULL))
    {
        PIWriteDescriptor token = OpenWriter();
        if (token)
        {
            error = CloseWriter(&token);
        }
    }

    return error;
}

//-----------------------------------------------------------------------------
//! @brief 出力処理の準備をします。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoPrepare(GPtr globals)
{
    gStuff->maxData = 0;
}

//-----------------------------------------------------------------------------
//! @brief スライス情報をファイルに出力します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] outputPath 出力ファイルパスです。
//! @param[in] isImageOnly 画像のあるスライスのみ取得するなら true を指定します。
//-----------------------------------------------------------------------------
static void OutputSliceInfo(
    GPtr globals,
    const std::string& outputPath,
    const bool isImageOnly
)
{
    //-----------------------------------------------------------------------------
    // スライス情報を取得します。
    const std::string docPath = RGetDocumentFilePath(globals);
    const std::string docName = RGetNoExtensionFilePath(RGetFileNameFromFilePath(docPath));
    const std::string cutPrefix = (!docName.empty()) ? docName + "_" : "";
    //RNoteTrace("out slice: [%s] %d [%s]", outputPath.c_str(), (isImageOnly ? 1 : 0), docName.c_str());
    RSliceInfoArray infos;
    RGetSliceInfos(infos, isImageOnly, cutPrefix);

    //-----------------------------------------------------------------------------
    // ファイルに出力します。
    std::ofstream ofs(outputPath);
    if (!ofs)
    {
        RShowError(globals, "Cannot open the file: %s", outputPath.c_str());
        gResult = userCanceledErr;
        return;
    }
    std::ostream& os = ofs;

    const char sep = '?';
    for (size_t iInfo = 0; iInfo < infos.size(); ++iInfo)
    {
        RSliceInfo& info = infos[iInfo];
        os << info.m_Index << sep
           << (info.m_IsUser ? 'U' : 'A') << sep
           << (info.m_IsImage ? '1' : '0') << sep
           << (info.m_IsDivided ? '1' : '0') << sep
           << info.m_X << sep
           << info.m_Y << sep
           << info.m_W << sep
           << info.m_H << sep
           << info.m_IndexName << sep
           << info.m_ZeroBasedName << sep
           << info.m_Name << endl;
    }
}

//-----------------------------------------------------------------------------
//! @brief 出力処理を開始します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoStart(GPtr globals)
{
    //-----------------------------------------------------------------------------
    // test
    //RNoteTrace("slice info start");
    //RNoteTrace("filename: [%s]", RGetStringFromStr255(gStuff->filename).c_str());
        // ↑ドキュメントのファイル名（フォルダのパスなし。拡張子あり）

    //-----------------------------------------------------------------------------
    // スクリプトパラメータをリードします。
    std::string outputPath;
    bool isImageOnly = true;
    gResult = ReadScriptParams(globals, outputPath, isImageOnly);
    if (gResult != noErr)
    {
        return;
    }
    if (outputPath.empty())
    {
        RShowError(globals, "Output path is not specified");
        gResult = userCanceledErr;
        return;
    }

    //-----------------------------------------------------------------------------
    // スライス情報をファイルに出力します。
    OutputSliceInfo(globals, outputPath, isImageOnly);

    //-----------------------------------------------------------------------------
    // 出力処理を完了に移行します。
    PISetRect(&gStuff->theRect, 0, 0, 0, 0); // no continue
    memset(&gStuff->theRect32, 0x00, sizeof(gStuff->theRect32)); // no continue
}

//-----------------------------------------------------------------------------
//! @brief 出力処理を継続します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoContinue(GPtr globals)
{
    R_UNUSED_VARIABLE(globals);
}

//-----------------------------------------------------------------------------
//! @brief 出力処理を完了します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoFinish(GPtr globals)
{
    gResult = WriteScriptParams(globals);
}

//-----------------------------------------------------------------------------
//! @brief アバウトダイアログを表示します。
//!
//! @param[in] about アバウトレコードです。
//-----------------------------------------------------------------------------
static void DoAboutDialog(AboutRecordPtr about)
{
    HINSTANCE instance = GetDLLInstance();
    PlatformData* platform = reinterpret_cast<PlatformData*>(about->platformData);
    RAboutParam aboutParam =
    {
        NPS_ABOUT_TITLE, NPS_ABOUT_MESSAGE,
        IDC_ABOUT_TEXT, IDC_HELP_INDEX, NPS_HELP_INDEX_URL
    };
    DialogBoxParam(instance,
        MAKEINTRESOURCE(IDD_ABOUT), reinterpret_cast<HWND>(platform->hwnd),
        reinterpret_cast<DLGPROC>(RAboutDialogProc), reinterpret_cast<LPARAM>(&aboutParam));
}

//-----------------------------------------------------------------------------
//! @brief プラグインのメイン関数です。
//-----------------------------------------------------------------------------
DLLExport MACPASCAL void PluginMain(
    const int16 selector,
    ExportRecord* exportParamBlock,
    intptr_t* data,
    int16* result
)
{
    //-----------------------------------------------------------------------------
    // do each proc
    if (selector == exportSelectorAbout)
    {
        //-----------------------------------------------------------------------------
        // do about
        AboutRecordPtr aboutRecord = reinterpret_cast<AboutRecordPtr>(exportParamBlock);
        sSPBasic = aboutRecord->sSPBasic;
        DoAboutDialog(aboutRecord);
    }
    else
    {
        //-----------------------------------------------------------------------------
        // グローバルデータを確保します。
        sSPBasic = exportParamBlock->sSPBasic;
        Ptr globalPtr = NULL;
        globalPtr = AllocateGlobals(result,
            exportParamBlock, exportParamBlock->handleProcs,
            sizeof(Globals), data, InitGlobals);
        if (globalPtr == NULL)
        {
            *result = memFullErr;
            return;
        }
        GPtr globals = reinterpret_cast<GPtr>(globalPtr);

        //-----------------------------------------------------------------------------
        // 32bit の座標を使用します。
        gStuff->PluginUsing32BitCoordinates = 1;

        //-----------------------------------------------------------------------------
        // dispatch selector
        try
        {
            switch (selector)
            {
                case exportSelectorPrepare:
                    DoPrepare(globals);
                    break;
                case exportSelectorStart:
                    DoStart(globals);
                    break;
                case exportSelectorContinue:
                    DoContinue(globals);
                    break;
                case exportSelectorFinish:
                    DoFinish(globals);
                    break;
            }
        }
        catch (int16 inError)
        {
            *result = inError;
        }
        catch (...)
        {
            *result = -1; // program error
        }

        //-----------------------------------------------------------------------------
        // グローバルデータを解放します。
        if (selector == exportSelectorFinish || *result != noErr)
        {
            globals->Free();
        }

        //-----------------------------------------------------------------------------
        // unlock data
        if (reinterpret_cast<Handle>(*data) != NULL)
        {
            PIUnlockHandle(reinterpret_cast<Handle>(*data));
        }
    }

    //-----------------------------------------------------------------------------
    // release suites
    if (selector == exportSelectorAbout  ||
        selector == exportSelectorFinish ||
        *result != noErr)
    {
        PIUSuitesRelease();
    }
}

