﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// PluginMain InitGlobals ValidateParameters InitializeParameters
// DoParameters DoPrepare DoStart DoContinue DoFinish
// DoFilter DoConvertFromHeightMap DoNormalize DoScaleSlope
// CombineNormalMap CombineLayers
// ReadPixelsFromHost WritePixelsToHost
// GetImageInfo

//=============================================================================
// include
//=============================================================================
#include "NintendoNormalMapFilter.h"
#include "NintendoNormalMapFilterScripting.h"
#include "NintendoNormalMapFilterUi.h"

using namespace nn::gfx::tool::nps;

//=============================================================================
// constants
//=============================================================================
const std::string RParameters::ActiveLayerName = "<Active>";

//=============================================================================
//! @brief サンプリング要素の構造体です。
//=============================================================================
struct SamplingElement
{
    int x; //!< 水平方向のオフセットです。
    int y; //!< 垂直方向のオフセットです。
    float w; //!< ウェイトです。
};

//-----------------------------------------------------------------------------
//! @brief パラメーターを初期化します。
//!        フィルタープラグインでは Photoshop 起動時に一度だけ呼ばれます。
//!        パラメータークラスのメンバーを変更した際は Photoshop を起動しなおす必要があります。
//-----------------------------------------------------------------------------
void RParameters::Initialize() // InitializeParameters
{
    // common
    m_Operation     = Operation_ConvertFromHeightMap;
    m_PositiveZ     = true;

    // convert from height map
    m_HeightScale   = 100;
    m_FilterType    = FilterType_3x3;
    m_EdgeWrap      = true;
    m_MultiplyAlpha = false;

    // scale slope
    m_SlopeScale    = 120;

    // ファイル合成
    memset(m_OtherPath, 0x00, sizeof(m_OtherPath));
    m_CombineScale  = 100;

    // レイヤー合成
    ClearCombineLayerParams();
}

//-----------------------------------------------------------------------------
//! @brief パラメーターを有効にします。
//!        確保されていなければ確保して初期化します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void ValidateParameters(GPtr globals)
{
    if (gStuff->parameters == nullptr)
    {
        gStuff->parameters = PINewHandle(sizeof(RParameters));
        if (gStuff->parameters == nullptr)
        {
            gResult = memFullErr;
            return;
        }
        gParams->Initialize();
    }
}

//-----------------------------------------------------------------------------
//! @brief グローバルデータを初期化します。
//!        プラグインが最初に使用されたときに 1 度だけ呼ばれます。
//!
//! @param[in,out] globalPtr グローバルデータのポインタです。
//-----------------------------------------------------------------------------
static void InitGlobals(Ptr globalPtr)
{
    //RMsgBox("init globals");

    //-----------------------------------------------------------------------------
    // copy for use macro
    //GPtr globals = (GPtr)globalPtr;

    //-----------------------------------------------------------------------------
    // call constructor (name must be "globals" for use macro (g????))
    GPtr globals = new (globalPtr) Globals;

    //-----------------------------------------------------------------------------
    // validate parameters
    ValidateParameters(globals);

    //-----------------------------------------------------------------------------
    // init element
    gQueryForParameters = FALSE; // DoParameters が呼ばれるまでは FALSE

    globals->m_pLayerDataArray = nullptr;
    globals->m_pLayerNameArray = nullptr;
    globals->m_ActiveLayerIdx = -1;

    globals->m_pSrcPixels = nullptr;
    globals->m_pDstPixels = nullptr;
    globals->m_pSrcAlphas = nullptr;
    globals->m_pGrayMatPixels = nullptr;

    globals->m_DisplaysPreview = true;
    globals->m_LightingPreview = false;
    globals->m_RotatesLightingPreview = false;
}

//-----------------------------------------------------------------------------
//! @brief パラメーターの処理をします。
//!        フィルターの再実行を行った場合は呼ばれません。
//!        アクションから実行する際にダイアログアイコンが OFF なら呼ばれません。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoParameters(GPtr globals)
{
    //RMsgBox("do parameters");

    //-----------------------------------------------------------------------------
    // validate parameters
    ValidateParameters(globals);

    //-----------------------------------------------------------------------------
    // set query for parameters
    gQueryForParameters = true; // show option dialog
}

//-----------------------------------------------------------------------------
//! @brief フィルター処理の準備をします。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoPrepare(GPtr globals)
{
    //RMsgBox("do prepare: %d %d", gStuff->bufferSpace, gStuff->maxSpace);
    // デフォルトで bufferSpace は 0、maxSpace は環境設定の最大使用メモリの値

    //-----------------------------------------------------------------------------
    // validate parameters
    ValidateParameters(globals);
}

//-----------------------------------------------------------------------------
//! @brief ピクセルデータを Photoshop からリードします。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void ReadPixelsFromHost(GPtr globals)
{
    //-----------------------------------------------------------------------------
    // ピクセルデータを確保します。
    globals->m_ImageBounds = gDocInfo->targetCompositeChannels->bounds;
    globals->m_ImageW = globals->m_ImageBounds.right  - globals->m_ImageBounds.left;
    globals->m_ImageH = globals->m_ImageBounds.bottom - globals->m_ImageBounds.top;
    const size_t pixelBytes = R_RGBA_BYTES;
    const size_t pixelCount = static_cast<size_t>(globals->m_ImageW) * globals->m_ImageH;
    const size_t pixelDataSize = pixelBytes * pixelCount;
    globals->m_pSrcPixels = new uint8_t[pixelDataSize];
    globals->m_pDstPixels = new uint8_t[pixelDataSize];

    //-----------------------------------------------------------------------------
    // ピクセルデータをリードします。
    memset(globals->m_pSrcPixels, 0x00, pixelDataSize);
    RReadMultiChannelsToPixels(globals->m_pSrcPixels, globals->m_ImageBounds, pixelBytes,
        gDocInfo->targetCompositeChannels, gDocInfo->targetTransparency, gDocInfo->targetLayerMask);

    //-----------------------------------------------------------------------------
    // アルファチャンネルが存在すれば一番上のものをリードします。
    if (gDocInfo->alphaChannels != nullptr)
    {
        const size_t alphaDataSize = sizeof(uint8_t) * pixelCount;
        globals->m_pSrcAlphas = new uint8_t[alphaDataSize];
        memset(globals->m_pSrcAlphas, 0x00, alphaDataSize);
        RReadChannelToPixels(globals->m_pSrcAlphas, globals->m_ImageBounds, sizeof(uint8_t),
            gDocInfo->alphaChannels, false);
    }
}

//-----------------------------------------------------------------------------
//! @brief ピクセルデータを Photoshop にライトします。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void WritePixelsToHost(GPtr globals)
{
    //-----------------------------------------------------------------------------
    // チャンネルバッファーを確保します。
    const size_t pixelCount = static_cast<size_t>(globals->m_ImageW) * globals->m_ImageH;
    uint8_t* chanBuf = new uint8_t[pixelCount];
    PixelMemoryDesc memDesc;
    memDesc.data = chanBuf;
    memDesc.bitOffset = 0;

    //-----------------------------------------------------------------------------
    // カラーチャンネルをライトします。
    int rgbIdx = 0;
    const ReadChannelDesc* pChan = gDocInfo->targetCompositeChannels;
    while (pChan != nullptr && rgbIdx < R_RGB_COUNT)
    {
        if (RGetChannelWriteFlag(pChan))
        {
            RGetOneChannelFromPixels(chanBuf, globals->m_pDstPixels + rgbIdx,
                globals->m_ImageW, globals->m_ImageH, globals->m_ImageW);

            VRect bounds = globals->m_ImageBounds;
            memDesc.depth   = pChan->depth;
            memDesc.colBits = pChan->depth;
            memDesc.rowBits = memDesc.colBits * globals->m_ImageW;
            if (sPSChannelProcs->WritePixelsToBaseLevel(pChan->writePort, &bounds, &memDesc))
            {
                gResult = errPlugInHostInsufficient;
                break;
            }
            ++rgbIdx;
        }
        pChan = pChan->next;
    }

    //-----------------------------------------------------------------------------
    // ピクセルの不透明度チャンネルをライトします。
    const bool writesTransparency =
        gParams->m_Operation == RParameters::Operation_ConvertFromHeightMap ||
        gParams->m_Operation == RParameters::Operation_CombineLayer;
    pChan = gDocInfo->targetTransparency;
    if (gResult == noErr && writesTransparency && pChan != nullptr)
    {
        if (RGetChannelWriteFlag(pChan))
        {
            RGetOneChannelFromPixels(chanBuf, globals->m_pDstPixels + 3,
                globals->m_ImageW, globals->m_ImageH, globals->m_ImageW);

            VRect bounds = globals->m_ImageBounds;
            memDesc.depth   = pChan->depth;
            memDesc.colBits = pChan->depth;
            memDesc.rowBits = memDesc.colBits * globals->m_ImageW;
            if (sPSChannelProcs->WritePixelsToBaseLevel(pChan->writePort, &bounds, &memDesc))
            {
                gResult = errPlugInHostInsufficient;
            }
        }
    }

    delete[] chanBuf;
}

//-----------------------------------------------------------------------------
//! @brief ピクセルの RGB 成分が +Z 方向の法線に一致するなら true を返します。
//!
//! @param[in] pRgb RGB の順に格納されたピクセルデータです。
//!
//! @return ピクセルの RGB 成分が +Z 方向の法線に一致するなら true を返します。
//-----------------------------------------------------------------------------
static bool IsAxisZNormalColor(const uint8_t* pRgb)
{
    return (pRgb[0] == 0x80 && pRgb[1] == 0x80 && pRgb[2] == 0xff);
}

//-----------------------------------------------------------------------------
//! @brief 法線マップを正規化します。
//!
//! @param[out] pDstPixels 変換先のピクセルデータを格納します。
//!             pSrcData と同じでも構いません。
//! @param[in] pSrcPixels 変換元のピクセルデータです。
//! @param[in] imageW 処理する画像の幅です。
//! @param[in] imageH 処理する画像の高さです。
//! @param[in] fullW 画像全体の幅です。
//! @param[in] isPositiveZ Z 成分が 0 以上になるように調整するなら true を指定します。
//-----------------------------------------------------------------------------
static void DoNormalize(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const int imageW,
    const int imageH,
    const int fullW,
    const bool isPositiveZ
)
{
    const uint8_t* pSrc = pSrcPixels;
    uint8_t* pDst = pDstPixels;
    for (int iy = 0; iy < imageH; ++iy)
    {
        for (int ix = 0; ix < imageW; ++ix)
        {
            float n[3];
            RGetNormalFromColor(n, pSrc);
            RNormalizeNormal(n, isPositiveZ);
            RGetColorFromNormal(pDst, n);
            pDst[3] = pSrc[3];
            pSrc += R_RGBA_BYTES;
            pDst += R_RGBA_BYTES;
        }
        pSrc += (fullW - imageW) * R_RGBA_BYTES;
        pDst += (fullW - imageW) * R_RGBA_BYTES;
    }
}

//-----------------------------------------------------------------------------
//! @brief ベクトルの長さを返します。
//!
//! @param[in] v ベクトルです。
//!
//! @return ベクトルの長さを返します。
//-----------------------------------------------------------------------------
static float GetVectorLength(const float* v)
{
    return sqrtf(v[0] * v[0] + v[1] * v[1] + v[2] * v[2]);
}

//-----------------------------------------------------------------------------
//! @brief ベクトルの内積を返します。
//!
//! @param[in] va ベクトルです。
//! @param[in] vb ベクトルです。
//!
//! @return ベクトルの内積を返します。
//-----------------------------------------------------------------------------
static float GetDotProduct(const float* va, const float* vb)
{
    return va[0] * vb[0] + va[1] * vb[1] + va[2] * vb[2];
}

//-----------------------------------------------------------------------------
//! @brief ベクトルの外積を返します。
//!
//! @param[out] dst 外積したベクトル（va x vb）を格納します。
//!             va または vb と同じでも構いません。
//! @param[in] va ベクトルです。
//! @param[in] vb ベクトルです。
//-----------------------------------------------------------------------------
static void GetCrossProduct(float* dst, const float* va, const float* vb)
{
    const float x = va[1] * vb[2] - va[2] * vb[1];
    const float y = va[2] * vb[0] - va[0] * vb[2];
    const float z = va[0] * vb[1] - va[1] * vb[0];
    dst[0] = x;
    dst[1] = y;
    dst[2] = z;
}

//-----------------------------------------------------------------------------
//! @brief あるベクトルを別のベクトルへ回転するクォータニオンを取得します。
//!
//! @param[out] q クォータニオンを格納します。
//! @param[in] fromV 起点ベクトルです。正規化されている必要があります。
//! @param[in] toV 終点ベクトルです。正規化されている必要があります。
//! @param[in] angleFactor 回転角の係数です。1.0 でちょうど終点ベクトルまで回転します。
//-----------------------------------------------------------------------------
static void GetVectorRotationQuat(
    float* q,
    const float* fromV,
    const float* toV,
    const float angleFactor)
{
    const float EPSILON = 1.0e-6f;
    const float PI = 3.14159265f;
    const float ANGLE_MAX = PI;

    // 以下、JGeometry の TQuat4::setRotate の移植です。
    float axis[3];
    GetCrossProduct(axis, fromV, toV);
    const float sinT = GetVectorLength(axis);
    if (sinT < EPSILON)
    {
        q[0] = 0.0f;
        q[1] = 0.0f;
        q[2] = 0.0f;
        q[3] = 1.0f;
    }
    else
    {
        const float cosT = GetDotProduct(fromV, toV);
        const float angle = RMin(atan2f(sinT, cosT) * angleFactor, ANGLE_MAX) * 0.5f;
        const float s = sinf(angle) / sinT;
        q[0] = s * axis[0];
        q[1] = s * axis[1];
        q[2] = s * axis[2];
        q[3] = cosf(angle);
    }

    // 以下、NW4F の QUATMakeVectorRotation の移植です（math_Quaternion.ipp）。
    //const float dPlus1 = GetDotProduct(fromV, toV) + 1.0f;
    //if (dPlus1 <= EPSILON)
    //{
    //  q[0] = 1.0f; // (0, 0, 0, 1) でないが正しいのか？
    //  q[1] = 0.0f;
    //  q[2] = 0.0f;
    //  q[3] = 0.0f;
    //}
    //else
    //{
    //  float c[3];
    //  GetCrossProduct(c, fromV, toV);
    //  float s = sqrtf(dPlus1 * 2.0f);
    //  float oos = 1.0f / s;
    //  q[0] = c[0] * oos;
    //  q[1] = c[1] * oos;
    //  q[2] = c[2] * oos;
    //  q[3] = s * 0.5f;
    //}
}

//-----------------------------------------------------------------------------
//! @brief クォータニオンから回転行列を取得します。
//!
//! @param[out] m 3 x 4 行列を格納します。
//! @param[in] q クォータニオンです。
//-----------------------------------------------------------------------------
//static void GetMatrixFromQuat(float m[3][4], const float* q)
//{
//  float s, xs, ys, zs;
//  float wx, wy, wz, xx, xy, xz, yy, yz, zz;
//
//  float len = (q[0] * q[0]) + (q[1] * q[1]) + (q[2] * q[2]) + (q[3] * q[3]);
//  if (len == 0.0f)
//  {
//      len = 1.0f;
//  }
//  s = 2.0f / len;
//
//  xs = q[0] *  s;  ys = q[1] *  s;  zs = q[2] *  s;
//  wx = q[3] * xs;  wy = q[3] * ys;  wz = q[3] * zs;
//  xx = q[0] * xs;  xy = q[0] * ys;  xz = q[0] * zs;
//  yy = q[1] * ys;  yz = q[1] * zs;  zz = q[2] * zs;
//
//  m[0][0] = 1.0f - (yy + zz);
//  m[0][1] = xy   - wz;
//  m[0][2] = xz   + wy;
//  m[0][3] = 0.0f;
//
//  m[1][0] = xy   + wz;
//  m[1][1] = 1.0f - (xx + zz);
//  m[1][2] = yz   - wx;
//  m[1][3] = 0.0f;
//
//  m[2][0] = xz   - wy;
//  m[2][1] = yz   + wx;
//  m[2][2] = 1.0f - (xx + yy);
//  m[2][3] = 0.0f;
//}

//-----------------------------------------------------------------------------
//! @brief 行列でベクトルを変換します。
//!
//! @param[out] dst 変換したベクトルを格納します。src と同じでも構いません。
//! @param[in] src 元のベクトルです。
//! @param[in] m 3 x 4 行列です。
//-----------------------------------------------------------------------------
//static void TransformVectorByMatrix(float* dst, const float* src, float m[3][4])
//{
//  const float x = m[0][0] * src[0] + m[0][1] * src[1] + m[0][2] * src[2];
//  const float y = m[1][0] * src[0] + m[1][1] * src[1] + m[1][2] * src[2];
//  const float z = m[2][0] * src[0] + m[2][1] * src[1] + m[2][2] * src[2];
//  dst[0] = x;
//  dst[1] = y;
//  dst[2] = z;
//}

//-----------------------------------------------------------------------------
//! @brief クォータニオンでベクトルを変換します。
//!
//! @param[out] dst 変換したベクトルを格納します。src と同じでも構いません。
//! @param[in] src 元のベクトルです。
//! @param[in] q クォータニオンです。
//-----------------------------------------------------------------------------
static void TransformVectorByQuat(float* dst, const float* src, const float* q)
{
    // 以下、JGeometry の TQuat4::transform の移植です。
    float vq[4];
    vq[0] =  q[1] * src[2] - q[2] * src[1] + q[3] * src[0];
    vq[1] = -q[0] * src[2] + q[2] * src[0] + q[3] * src[1];
    vq[2] =  q[0] * src[1] - q[1] * src[0] + q[3] * src[2];
    vq[3] = -q[0] * src[0] - q[1] * src[1] - q[2] * src[2];
    dst[0] =  vq[0] *  q[3] + vq[1] * -q[2] - vq[2] * -q[1] + vq[3] * -q[0];
    dst[1] = -vq[0] * -q[2] + vq[1] *  q[3] + vq[2] * -q[0] + vq[3] * -q[1];
    dst[2] =  vq[0] * -q[1] - vq[1] * -q[0] + vq[2] *  q[3] + vq[3] * -q[2];

    //float m[3][4];
    //GetMatrixFromQuat(m, q);
    //TransformVectorByMatrix(dst, src, m);
}

//-----------------------------------------------------------------------------
//! @brief 接空間法線マップの傾きをスケールします。
//!
//! @param[out] pDstPixels 変換先のピクセルデータを格納します。
//!             pSrcPixels と同じでも構いません。
//! @param[in] pSrcPixels 変換元のピクセルデータです。
//! @param[in] imageW 処理する画像の幅です。
//! @param[in] imageH 処理する画像の高さです。
//! @param[in] fullW 画像全体の幅です。
//! @param[in] slopeScale 傾きのスケール値です。100 で等倍です。
//! @param[in] isPositiveZ Z 成分が 0 以上になるように調整するなら true を指定します。
//-----------------------------------------------------------------------------
static void DoScaleSlope(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const int imageW,
    const int imageH,
    const int fullW,
    const int slopeScale,
    const bool isPositiveZ
)
{
    static const float zAxis[] = { 0.0f, 0.0f, 1.0f };
    const float floatScale = static_cast<float>(slopeScale) / 100.0f;

    const uint8_t* pSrc = pSrcPixels;
    uint8_t* pDst = pDstPixels;
    for (int iy = 0; iy < imageH; ++iy)
    {
        for (int ix = 0; ix < imageW; ++ix)
        {
            //-----------------------------------------------------------------------------
            // get normal
            float n[3];
            RGetNormalFromColor(n, pSrc);
            RNormalizeNormal(n, false);

            //-----------------------------------------------------------------------------
            // get quarternion (Z axis to normal rotation)
            float q[4];
            GetVectorRotationQuat(q, zAxis, n, floatScale);

            //-----------------------------------------------------------------------------
            // rotate Z axis by quarternion
            TransformVectorByQuat(n, zAxis, q);
            RNormalizeNormal(n, isPositiveZ);

            //-----------------------------------------------------------------------------
            // set normal
            RGetColorFromNormal(pDst, n);
            pDst[3] = pSrc[3];
            pSrc += R_RGBA_BYTES;
            pDst += R_RGBA_BYTES;
        }
        pSrc += (fullW - imageW) * R_RGBA_BYTES;
        pDst += (fullW - imageW) * R_RGBA_BYTES;
    }
}

//-----------------------------------------------------------------------------
//! @brief 他の接空間法線マップと合成します。
//!
//! @param[out] pDstPixels 変換先のピクセルデータを格納します。
//!             pSrcPixels と同じでも構いません。
//! @param[in] pSrcPixels 変換元のピクセルデータです。
//! @param[in] imageW 処理する画像の幅です。
//! @param[in] imageH 処理する画像の高さです。
//! @param[in] fullW 画像全体の幅です。
//! @param[in] pOtherPixels 他の法線マップのピクセルデータです。
//! @param[in] otherW 他の法線マップの幅です。
//! @param[in] otherH 他の法線マップの高さです。
//! @param[in] combineScale 合成のスケール値です。1.0 で他の法線マップをそのまま合成します。
//! @param[in] isPositiveZ Z 成分が 0 以上になるように調整するなら true を指定します。
//-----------------------------------------------------------------------------
static void CombineNormalMap(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const int imageW,
    const int imageH,
    const int fullW,
    const uint8_t* pOtherPixels,
    const int otherW,
    const int otherH,
    const float combineScale,
    const bool isPositiveZ
)
{
    static const float AxisZ[] = { 0.0f, 0.0f, 1.0f };
    const size_t otherLineBytes = R_RGBA_BYTES * otherW;

    const uint8_t* pSrc = pSrcPixels;
    uint8_t* pDst = pDstPixels;
    for (int iy = 0; iy < imageH; ++iy)
    {
        const int oy = iy % otherH;
        const uint8_t* pOtherLine = pOtherPixels + otherLineBytes * oy;
        for (int ix = 0; ix < imageW; ++ix)
        {
            const int ox = ix % otherW;
            const uint8_t* pOther = pOtherLine + R_RGBA_BYTES * ox;
            const uint8_t otherA = pOther[3];
            if (otherA == 0x00 || IsAxisZNormalColor(pOther))
            {
                //-----------------------------------------------------------------------------
                // 他の法線マップのピクセルが透明なら合成しません。
                // 他の法線マップの RGB 成分が +Z 方向の法線に一致する場合も合成しません（高速化のため）。
                if (pDst != pSrc)
                {
                    memcpy(pDst, pSrc, R_RGBA_BYTES);
                }
                if (isPositiveZ)
                {
                    float n[3];
                    RGetNormalFromColor(n, pDst);
                    RNormalizeNormal(n, true);
                    RGetColorFromNormal(pDst, n);
                }
            }
            else
            {
                //-----------------------------------------------------------------------------
                // 他の法線マップの RGB 成分を法線に変換します。
                float otherN[3];
                RGetNormalFromColor(otherN, pOther);
                RNormalizeNormal(otherN, false);

                //-----------------------------------------------------------------------------
                // +Z 方向から他の法線マップの方向へ回転するクォータニオンを取得します。
                const float pixelScale = (otherA == 0xff) ?
                    combineScale : combineScale * otherA / 0xff;
                float q[4];
                GetVectorRotationQuat(q, AxisZ, otherN, pixelScale);

                //-----------------------------------------------------------------------------
                // 変換元の RGB 成分を法線に変換します。
                float n[3];
                RGetNormalFromColor(n, pSrc);
                RNormalizeNormal(n, false);

                //-----------------------------------------------------------------------------
                // 変換元の法線をクォータニオンで回転します。
                TransformVectorByQuat(n, n, q);
                RNormalizeNormal(n, isPositiveZ);

                //-----------------------------------------------------------------------------
                // 回転した法線を変換先の RGB 成分に設定します。
                RGetColorFromNormal(pDst, n);
                pDst[3] = pSrc[3];
                #if 0
                // view other test
                pDst[0] = pOther[0];
                pDst[1] = pOther[1];
                pDst[2] = pOther[2];
                #endif
            }
            pSrc += R_RGBA_BYTES;
            pDst += R_RGBA_BYTES;
        }
        pSrc += (fullW - imageW) * R_RGBA_BYTES;
        pDst += (fullW - imageW) * R_RGBA_BYTES;
    }
}

//-----------------------------------------------------------------------------
//! @brief レイヤー名でレイヤーデータ配列を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//!
//! @param[in] layerDatas レイヤーデータ配列です。
//! @param[in] name レイヤー名です。
//! @param[in] isVisibleOnly 可視レイヤーのみ検索するなら true を指定します。
//! @param[in] isPriorityOnly 同名レイヤーの中で最優先のものだけ検索するなら true、
//!                           優先度に関係なく先頭のものを検索するなら false を指定します。
//!
//! @return 配列内のインデックス（見つからなければ -1）を返します。
//-----------------------------------------------------------------------------
static int FindLayerData(
    const LayerDataArray& layerDatas,
    const std::string& name,
    const bool isVisibleOnly,
    const bool isPriorityOnly
)
{
    for (size_t layerIdx = 0; layerIdx < layerDatas.size(); ++layerIdx)
    {
        const LayerData& layerData = layerDatas[layerIdx];
        if ((!isVisibleOnly  || layerData.m_IsVisible  ) &&
            (!isPriorityOnly || layerData.m_HasPriority) &&
            layerData.m_Name == name)
        {
            return static_cast<int>(layerIdx);
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief 複数レイヤーの法線マップを合成します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void CombineLayers(GPtr globals)
{
    uint8_t* pDstPixels       = globals->m_pDstPixels;
    const uint8_t* pSrcPixels = globals->m_pSrcPixels;
    const int imageW = globals->m_ImageW;
    const int imageH = globals->m_ImageH;
    const size_t pixelBytes = R_RGBA_BYTES;
    const size_t pixelCount = static_cast<size_t>(imageW) * imageH;
    const size_t pixelDataSize = pixelBytes * pixelCount;

    //-----------------------------------------------------------------------------
    // 合成するピクセルデータ群を取得します。
    std::vector<const uint8_t*> combinePixelPtrs;
    RFloatArray combineScales;
    LayerDataArray& layerDatas = *globals->m_pLayerDataArray;
    for (int layerIdx = 0; layerIdx < RParameters::CombineLayerCountMax; ++layerIdx)
    {
        const uint8_t* pCombinePixels = (layerIdx == 0) ? pSrcPixels : nullptr;
        const std::string layerName = gParams->m_CombineLayerNames[layerIdx];
        float combineScale = static_cast<float>(gParams->m_CombineLayerScales[layerIdx]) / 100.0f;
        if (layerName == RParameters::ActiveLayerName)
        {
            // アクティブレイヤー指定なら変換元のピクセルデータを使用します。
            pCombinePixels = pSrcPixels;
        }
        else if (!layerName.empty())
        {
            const int layerDataIdx = FindLayerData(layerDatas, layerName, false, true);
            if (layerDataIdx != -1)
            {
                LayerData& layerData = layerDatas[layerDataIdx];
                auto& layerPixels = layerData.m_Pixels;
                if (layerPixels.empty())
                {
                    // レイヤーのピクセルデータが未取得なら Photoshop から取得します。
                    //RNoteTrace("get layer pixel: %s", layerData.m_Name.c_str());
                    const ReadLayerDesc* pLayerDesc = layerData.m_pLayerDesc;
                    layerPixels.resize(pixelDataSize);
                    memset(layerPixels.data(), 0x00, pixelDataSize);
                    RReadMultiChannelsToPixels(layerPixels.data(), globals->m_ImageBounds, pixelBytes,
                        pLayerDesc->compositeChannelsList, pLayerDesc->transparency, pLayerDesc->layerMask);
                }
                pCombinePixels = layerPixels.data();
                //combineScale *= static_cast<float>(layerData.m_Opacity) / 0xff;
                    // ↑レイヤーの不透明度をスケールに乗算するならコメント化を解除します。
            }
        }

        if (pCombinePixels != nullptr)
        {
            //RNoteTrace("combine layers: %s (%.2f)", layerName.c_str(), combineScale * 100.0f);
            combinePixelPtrs.push_back(pCombinePixels);
            combineScales.push_back(combineScale);
        }
    }

    //-----------------------------------------------------------------------------
    // ピクセルデータ群を合成します。
    for (size_t combineIdx = 0; combineIdx < combinePixelPtrs.size(); ++combineIdx)
    {
        const uint8_t* pCombinePixels = combinePixelPtrs[combineIdx];
        if (combineIdx == 0)
        {
            memcpy(pDstPixels, pCombinePixels, pixelDataSize);
        }
        else
        {
            const bool isTop = (combineIdx == combinePixelPtrs.size() - 1);
            CombineNormalMap(pDstPixels, pDstPixels, imageW, imageH, imageW,
                pCombinePixels, imageW, imageH,
                combineScales[combineIdx], (isTop && gParams->m_PositiveZ));
                // 最後の合成の際だけ positiveZ を適用します。
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief サンプリングウェイト配列からサンプリング要素配列を設定します。
//!
//! @param[out] elems サンプリング要素配列です。
//! @param[in] weights サンプリングウェイト配列です。
//! @param[in] samplingCount サンプリング数です。
//-----------------------------------------------------------------------------
static void SetSamplingElements(SamplingElement* elems, const float* weights, const int samplingCount)
{
    const int halfCount = samplingCount / 2;
    int iElem = 0;
    for (int y = 0; y < samplingCount; ++y)
    {
        for (int x = 0; x < samplingCount; ++x)
        {
            elems[iElem].x = x - halfCount;
            elems[iElem].y = halfCount - y;
            elems[iElem].w = weights[iElem];
            ++iElem;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief サンプリングウェイト配列を U 方向から V 方向に変換します。
//!
//! @param[out] dstWeights V 方向のサンプリングウェイト配列です。
//! @param[in] srcWeights U 方向のサンプリングウェイト配列です。
//! @param[in] samplingCount サンプリング数です。
//-----------------------------------------------------------------------------
static void ConvertSamplingWeightsUtoV(float* dstWeights, const float* srcWeights, const int samplingCount)
{
    int iSrc = 0;
    for (int y = 0; y < samplingCount; ++y)
    {
        for (int x = 0; x < samplingCount; ++x)
        {
            const int dy = samplingCount - 1 - x;
            const int dx = y;
            dstWeights[dx + dy * samplingCount] = srcWeights[iSrc++];
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ラップした座標を計算します。
//!
//! @param[in] pos 座標です。
//! @param[in] length 長さです。
//! @param[in] edgeWrap 画像が繰り返しているとして端のピクセルを処理するなら true です。
//!
//! @return ラップした座標を返します。
//-----------------------------------------------------------------------------
static inline int CalcWrapPos(int pos, int length, bool edgeWrap)
{
    if (edgeWrap)
    {
        while (pos < 0)
        {
            pos += length;
        }
        while (pos >= length)
        {
            pos -= length;
        }
    }
    else
    {
        if (pos < 0)
        {
            pos = 0;
        }
        else if (pos > length - 1)
        {
            pos = length - 1;
        }
    }
    return pos;
}

//-----------------------------------------------------------------------------
//! @brief 高さマップを法線マップに変換します。
//!
//! @param[out] pDstPixels 変換先のピクセルデータを格納します。
//!             pSrcPixels とは異なるバッファを指定します。
//! @param[in] pSrcPixels 変換元のピクセルデータです。
//! @param[in] imageW 処理する画像の幅です。
//! @param[in] imageH 処理する画像の高さです。
//! @param[in] fullW 画像全体の幅です。
//! @param[in] heightScale 凹凸の高さを調整するスケール値です。100 で等倍です。
//! @param[in] filterType サンプリングする範囲です。
//! @param[in] edgeWrap 画像が繰り返しているとして端のピクセルを処理するなら true です。
//! @param[in] pAlphas アルファチャンネルのピクセルデータです。
//!                    アルファを高さマップに乗算しない場合は nullptr を指定します。
//-----------------------------------------------------------------------------
static void DoConvertFromHeightMap(
    uint8_t* pDstPixels,
    const uint8_t* pSrcPixels,
    const int imageW,
    const int imageH,
    const int fullW,
    const int heightScale,
    const RParameters::FilterType filterType,
    const bool edgeWrap,
    const uint8_t* pAlphas
)
{
    //-----------------------------------------------------------------------------
    // 入力画像の RGB 値をグレースケールに変換し、float 型で保持します。
    const uint8_t* pSrc = pSrcPixels;
    const uint8_t* pA = pAlphas;
    float* heights = new float[imageW * imageH]; // グレースケール保持用
    for (int y = 0; y < imageH; ++y)
    {
        for (int x = 0; x < imageW; ++x)
        {
            // RGB 値を輝度（0.299 * R + 0.587 * G + 0.114 * B）に変換します。
            const float val =
                static_cast<float>(pSrc[0]) * 0.299f +
                static_cast<float>(pSrc[1]) * 0.587f +
                static_cast<float>(pSrc[2]) * 0.114f;
            if (pA != nullptr)
            {
                heights[x + y * imageW] = val * (1.0f / 255.0f) * (static_cast<float>(*pA) * (1.0f / 255.0f));
                ++pA;
            }
            else
            {
                heights[x + y * imageW] = val * (1.0f / 255.0f);
            }
            pSrc += R_RGBA_BYTES;
        }
        pSrc += (fullW - imageW) * R_RGBA_BYTES;
        if (pA != nullptr)
        {
            pA += fullW - imageW;
        }
    }

    //-----------------------------------------------------------------------------
    // サンプリング数に対応したウェイトを計算します。
    // weightDus が X 方向、weightDvs が Y 方向の、位置と重みを保持します。
    int numElements = 0;
    SamplingElement* weightDus = nullptr;
    SamplingElement* weightDvs = nullptr;
    switch (filterType)
    {
        case RParameters::FilterType_4:
        {
            numElements = 2;
            weightDus = new SamplingElement[numElements * sizeof(SamplingElement)];
            weightDvs = new SamplingElement[numElements * sizeof(SamplingElement)];

            // -0.5 * +0.5
            weightDus[0].x = -1; weightDus[0].y = 0; weightDus[0].w = -0.5f;
            weightDus[1].x =  1; weightDus[1].y = 0; weightDus[1].w =  0.5f;

            // -0.5
            //   *
            // +0.5
            weightDvs[0].x = 0; weightDvs[0].y =  1; weightDvs[0].w =  0.5f;
            weightDvs[1].x = 0; weightDvs[1].y = -1; weightDvs[1].w = -0.5f;
            break;
        }

        case RParameters::FilterType_3x3:
        {
            numElements = 6;
            weightDus = new SamplingElement[numElements * sizeof(SamplingElement)];
            weightDvs = new SamplingElement[numElements * sizeof(SamplingElement)];
            const float weight = 1.0f / 6.0f;

            // -w   +w
            // -w * +w
            // -w   +w
            weightDus[0].x = -1; weightDus[0].y =  1; weightDus[0].w = -weight;
            weightDus[1].x = -1; weightDus[1].y =  0; weightDus[1].w = -weight;
            weightDus[2].x = -1; weightDus[2].y = -1; weightDus[2].w = -weight;
            weightDus[3].x =  1; weightDus[3].y =  1; weightDus[3].w =  weight;
            weightDus[4].x =  1; weightDus[4].y =  0; weightDus[4].w =  weight;
            weightDus[5].x =  1; weightDus[5].y = -1; weightDus[5].w =  weight;

            // -w -w -w
            //     *
            // +w +w +w
            weightDvs[0].x = -1; weightDvs[0].y =  1; weightDvs[0].w =  weight;
            weightDvs[1].x =  0; weightDvs[1].y =  1; weightDvs[1].w =  weight;
            weightDvs[2].x =  1; weightDvs[2].y =  1; weightDvs[2].w =  weight;
            weightDvs[3].x = -1; weightDvs[3].y = -1; weightDvs[3].w = -weight;
            weightDvs[4].x =  0; weightDvs[4].y = -1; weightDvs[4].w = -weight;
            weightDvs[5].x =  1; weightDvs[5].y = -1; weightDvs[5].w = -weight;
            break;
        }

        case RParameters::FilterType_5x5:
        {
            numElements = 20;
            weightDus = new SamplingElement[numElements * sizeof(SamplingElement)];
            weightDvs = new SamplingElement[numElements * sizeof(SamplingElement)];
            const float weight22 = 1.0f / 16.0f;
            const float weight12 = 1.0f / 10.0f;
            const float weight02 = 1.0f /  8.0f;
            const float weight11 = 1.0f /  2.8f;

            weightDus[0 ].x = -2; weightDus[0 ].y =  2; weightDus[0 ].w = -weight22;
            weightDus[1 ].x = -1; weightDus[1 ].y =  2; weightDus[1 ].w = -weight12;
            weightDus[2 ].x =  1; weightDus[2 ].y =  2; weightDus[2 ].w =  weight12;
            weightDus[3 ].x =  2; weightDus[3 ].y =  2; weightDus[3 ].w =  weight22;
            weightDus[4 ].x = -2; weightDus[4 ].y =  1; weightDus[4 ].w = -weight12;
            weightDus[5 ].x = -1; weightDus[5 ].y =  1; weightDus[5 ].w = -weight11;
            weightDus[6 ].x =  1; weightDus[6 ].y =  1; weightDus[6 ].w =  weight11;
            weightDus[7 ].x =  2; weightDus[7 ].y =  1; weightDus[7 ].w =  weight12;
            weightDus[8 ].x = -2; weightDus[8 ].y =  0; weightDus[8 ].w = -weight02;
            weightDus[9 ].x = -1; weightDus[9 ].y =  0; weightDus[9 ].w = -0.5f;
            weightDus[10].x =  1; weightDus[10].y =  0; weightDus[10].w =  0.5f;
            weightDus[11].x =  2; weightDus[11].y =  0; weightDus[11].w =  weight02;
            weightDus[12].x = -2; weightDus[12].y = -1; weightDus[12].w = -weight12;
            weightDus[13].x = -1; weightDus[13].y = -1; weightDus[13].w = -weight11;
            weightDus[14].x =  1; weightDus[14].y = -1; weightDus[14].w =  weight11;
            weightDus[15].x =  2; weightDus[15].y = -1; weightDus[15].w =  weight12;
            weightDus[16].x = -2; weightDus[16].y = -2; weightDus[16].w = -weight22;
            weightDus[17].x = -1; weightDus[17].y = -2; weightDus[17].w = -weight12;
            weightDus[18].x =  1; weightDus[18].y = -2; weightDus[18].w =  weight12;
            weightDus[19].x =  2; weightDus[19].y = -2; weightDus[19].w =  weight22;

            weightDvs[0 ].x = -2; weightDvs[0 ].y =  2; weightDvs[0 ].w =  weight22;
            weightDvs[1 ].x = -1; weightDvs[1 ].y =  2; weightDvs[1 ].w =  weight12;
            weightDvs[2 ].x =  0; weightDvs[2 ].y =  2; weightDvs[2 ].w =  0.25f;
            weightDvs[3 ].x =  1; weightDvs[3 ].y =  2; weightDvs[3 ].w =  weight12;
            weightDvs[4 ].x =  2; weightDvs[4 ].y =  2; weightDvs[4 ].w =  weight22;
            weightDvs[5 ].x = -2; weightDvs[5 ].y =  1; weightDvs[5 ].w =  weight12;
            weightDvs[6 ].x = -1; weightDvs[6 ].y =  1; weightDvs[6 ].w =  weight11;
            weightDvs[7 ].x =  0; weightDvs[7 ].y =  1; weightDvs[7 ].w =  0.5f;
            weightDvs[8 ].x =  1; weightDvs[8 ].y =  1; weightDvs[8 ].w =  weight11;
            weightDvs[9 ].x =  2; weightDvs[9 ].y =  1; weightDvs[9 ].w =  weight22;
            weightDvs[10].x = -2; weightDvs[10].y = -1; weightDvs[10].w = -weight22;
            weightDvs[11].x = -1; weightDvs[11].y = -1; weightDvs[11].w = -weight11;
            weightDvs[12].x =  0; weightDvs[12].y = -1; weightDvs[12].w = -0.5f;
            weightDvs[13].x =  1; weightDvs[13].y = -1; weightDvs[13].w = -weight11;
            weightDvs[14].x =  2; weightDvs[14].y = -1; weightDvs[14].w = -weight12;
            weightDvs[15].x = -2; weightDvs[15].y = -2; weightDvs[15].w = -weight22;
            weightDvs[16].x = -1; weightDvs[16].y = -2; weightDvs[16].w = -weight12;
            weightDvs[17].x =  0; weightDvs[17].y = -2; weightDvs[17].w = -0.25f;
            weightDvs[18].x =  1; weightDvs[18].y = -2; weightDvs[18].w = -weight12;
            weightDvs[19].x =  2; weightDvs[19].y = -2; weightDvs[19].w = -weight22;

            float uSum = 0, vSum = 0;
            for (int n = 0; n < numElements; ++n)
            {
                uSum += fabsf(weightDus[n].w);
                vSum += fabsf(weightDvs[n].w);
            }
            for (int n = 0; n < numElements; ++n)
            {
                weightDus[n].w /= uSum;
                weightDvs[n].w /= vSum;
            }
            break;
        }

        case RParameters::FilterType_7x7:
        case RParameters::FilterType_9x9:
        case RParameters::FilterType_13x13:
        case RParameters::FilterType_17x17:
        case RParameters::FilterType_21x21:
        case RParameters::FilterType_25x25:
        {
            int samplingCount = 0;
            switch (filterType)
            {
                default:
                case RParameters::FilterType_7x7  : samplingCount =  7; break;
                case RParameters::FilterType_9x9  : samplingCount =  9; break;
                case RParameters::FilterType_13x13: samplingCount = 13; break;
                case RParameters::FilterType_17x17: samplingCount = 17; break;
                case RParameters::FilterType_21x21: samplingCount = 21; break;
                case RParameters::FilterType_25x25: samplingCount = 25; break;
            }

            numElements = samplingCount * samplingCount;
            weightDus = new SamplingElement[numElements * sizeof(SamplingElement)];
            weightDvs = new SamplingElement[numElements * sizeof(SamplingElement)];

            // 7x7 の場合、U 方向のサンプリングウェイトは次のようになります。
            // -1, -2, -3, 0, 3, 2, 1,
            // -2, -3, -4, 0, 4, 3, 2,
            // -3, -4, -5, 0, 5, 4, 3,
            // -4, -5, -6, 0, 6, 5, 4,
            // -3, -4, -5, 0, 5, 4, 3,
            // -2, -3, -4, 0, 4, 3, 2,
            // -1, -2, -3, 0, 3, 2, 1
            float* samplingWeightsU = new float[numElements];
            const int halfCount = samplingCount / 2;
            int iWeight = 0;
            for (int y = 0; y < samplingCount; ++y)
            {
                const int ofs = (y <= halfCount) ? y : samplingCount - 1 - y;
                for (int x = 0; x < samplingCount; ++x)
                {
                    const int w = (x == halfCount) ? 0 :
                        ((x < halfCount) ? - (x + 1) - ofs : samplingCount - x + ofs);
                    samplingWeightsU[iWeight++] = static_cast<float>(w);
                }
            }
            float* samplingWeightsV = new float[numElements];
            ConvertSamplingWeightsUtoV(samplingWeightsV, samplingWeightsU, samplingCount);
            SetSamplingElements(weightDus, samplingWeightsU, samplingCount);
            SetSamplingElements(weightDvs, samplingWeightsV, samplingCount);
            delete[] samplingWeightsU;
            delete[] samplingWeightsV;

            float uSum = 0;
            float vSum = 0;
            for (int n = 0; n < numElements; ++n)
            {
                uSum += fabsf(weightDus[n].w);
                vSum += fabsf(weightDvs[n].w);
            }
            for (int n = 0; n < numElements; ++n)
            {
                weightDus[n].w /= uSum;
                weightDvs[n].w /= vSum;
            }
            break;
        }

        default:
            break;
    }

    //-----------------------------------------------------------------------------
    // 各画素の周辺をサンプリングして法線を算出します。
    const float floatScale = static_cast<float>(heightScale) / 100.0f;
    pSrc = pSrcPixels;
    uint8_t* pDst = pDstPixels;
    for (int y = 0; y < imageH; ++y)
    {
        for (int x = 0; x < imageW; ++x)
        {
            // サンプル数分、重みづけしつつ、加算します。
            float du = 0;
            float dv = 0;

            for (int i = 0; i < numElements; ++i)
            {
                const int tmpX = CalcWrapPos(x + weightDus[i].x, imageW, edgeWrap);
                const int tmpY = CalcWrapPos(y + weightDus[i].y, imageH, edgeWrap);
                du += heights[tmpX + tmpY * imageW] * weightDus[i].w;
            }

            for (int i = 0; i < numElements; ++i)
            {
                const int tmpX = CalcWrapPos(x + weightDvs[i].x, imageW, edgeWrap);
                const int tmpY = CalcWrapPos(y + weightDvs[i].y, imageH, edgeWrap);
                dv += heights[tmpX + tmpY * imageW] * weightDvs[i].w;
            }

            // スケール分を掛けます。
            float n[3];
            n[0] = -du * floatScale;
            n[1] = -dv * floatScale;
            n[2] = 1.0f;

            RNormalizeNormal(n, false);

            // TBN が右手座標系なので Y 軸方向を反転します。
            n[1] = -n[1];

            RGetColorFromNormal(pDst, n);

            // 法線が +Z 方向でなければ、ピクセルの不透明度を 0xff に変更します（後で合成する場合のため）。
            pDst[3] = (!IsAxisZNormalColor(pDst)) ? 0xff : pSrc[3];
            //pDst[3] = 0xff;;
            //pDst[3] = pSrc[3];

            pDst += R_RGBA_BYTES;
            pSrc += R_RGBA_BYTES;
        }
        pDst += (fullW - imageW) * R_RGBA_BYTES;
        pSrc += (fullW - imageW) * R_RGBA_BYTES;
    }

    //-----------------------------------------------------------------------------
    // 作成したバッファを解放します。
    delete[] heights;
    delete[] weightDus;
    delete[] weightDvs;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief グレーの背景（RGB = 128）とブレンドしたピクセルデータを作成します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void CreateGrayMatPixels(GPtr globals)
{
    const float GrayValue = 128.0f;

    const size_t pixelBytes = R_RGBA_BYTES;
    const size_t pixelCount = static_cast<size_t>(globals->m_ImageW) * globals->m_ImageH;
    const size_t pixelDataSize = pixelBytes * pixelCount;
    globals->m_pGrayMatPixels = new uint8_t[pixelDataSize];
    memcpy(globals->m_pGrayMatPixels, globals->m_pSrcPixels, pixelDataSize);

    uint8_t* pDst = globals->m_pGrayMatPixels;
    for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
    {
        if (pDst[3] != 0xff)
        {
            const float a = static_cast<float>(pDst[3]) / 0xff;
            const float ia = 1.0f - a;
            const float r = a * pDst[0] + ia * GrayValue;
            const float g = a * pDst[1] + ia * GrayValue;
            const float b = a * pDst[2] + ia * GrayValue;
            pDst[0] = static_cast<uint8_t>(RClampValue(0x00, 0xff, RRound(r)));
            pDst[1] = static_cast<uint8_t>(RClampValue(0x00, 0xff, RRound(g)));
            pDst[2] = static_cast<uint8_t>(RClampValue(0x00, 0xff, RRound(b)));
        }
        pDst += pixelBytes;
    }
}

//-----------------------------------------------------------------------------
//! @brief フィルター処理を実行します。
//!
//! @param[in,out] globals グローバルデータです。
//! @param[in] otherImg 合成する法線マップの画像データです。
//-----------------------------------------------------------------------------
void DoFilter(GPtr globals, const RImage& otherImg)
{
    uint8_t* pDstPixels       = globals->m_pDstPixels;
    const uint8_t* pSrcPixels = globals->m_pSrcPixels;
    const int imageW = globals->m_ImageW;
    const int imageH = globals->m_ImageH;
    const int fullW = imageW;

    //RTimeMeasure tm1;
    if (gParams->m_Operation == RParameters::Operation_ConvertFromHeightMap)
    {
        // 不透明度チャンネルがあれば、グレーの背景とブレンドしたピクセルデータを作成して、
        // フィルターの入力データとします。
        if (gDocInfo->targetTransparency != nullptr)
        {
            if (globals->m_pGrayMatPixels == nullptr)
            {
                CreateGrayMatPixels(globals);
            }
            pSrcPixels = globals->m_pGrayMatPixels;
        }

        // 高さマップを法線マップに変換します。
        const uint8_t* pAlpha = (gParams->m_MultiplyAlpha) ? globals->m_pSrcAlphas : nullptr;
        DoConvertFromHeightMap(pDstPixels, pSrcPixels, imageW, imageH, fullW,
            gParams->m_HeightScale, gParams->m_FilterType,
            gParams->m_EdgeWrap, pAlpha);
    }
    else if (gParams->m_Operation == RParameters::Operation_Normalize)
    {
        DoNormalize(pDstPixels, pSrcPixels, imageW, imageH, fullW, gParams->m_PositiveZ);
    }
    else if (gParams->m_Operation == RParameters::Operation_ScaleSclope)
    {
        DoScaleSlope(pDstPixels, pSrcPixels, imageW, imageH, fullW,
            gParams->m_SlopeScale, gParams->m_PositiveZ);
    }
    else if (gParams->m_Operation == RParameters::Operation_CombineFile)
    {
        if (otherImg.GetImagePtr() != nullptr)
        {
            const float combineScale = static_cast<float>(gParams->m_CombineScale) / 100.0f;
            CombineNormalMap(pDstPixels, pSrcPixels, imageW, imageH, fullW,
                reinterpret_cast<const uint8_t*>(otherImg.GetImagePtr()),
                otherImg.GetImageW(), otherImg.GetImageH(),
                combineScale, gParams->m_PositiveZ);
        }
    }
    else if (gParams->m_Operation == RParameters::Operation_CombineLayer)
    {
        CombineLayers(globals);
    }
    //RNoteTrace("do filter: %.2f ms", tm1.GetMilliSec());
}

//-----------------------------------------------------------------------------
//! @brief 画像の情報を取得します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void GetImageInfo(GPtr globals)
{
    //-----------------------------------------------------------------------------
    // 全レイヤーのリードレイヤー記述子を取得します。
    // ただし、レイヤー（背景）が 1 つだけなら取得できません。
    std::vector<const ReadLayerDesc*> layerDescPtrs;
    const ReadLayerDesc* pLayerDescTop = gDocInfo->layersDescriptor; // レイヤーが 1 つだけなら nullptr
    for (auto pLayerDesc = pLayerDescTop; pLayerDesc != nullptr; pLayerDesc = pLayerDesc->next)
    {
        layerDescPtrs.push_back(pLayerDesc); // 下位レイヤーが先
    }
    std::reverse(layerDescPtrs.begin(), layerDescPtrs.end()); // 上位レイヤーが先になるように並び替え
    const bool backGroundExists = (pLayerDescTop != nullptr && pLayerDescTop->transparency == nullptr);

    //-----------------------------------------------------------------------------
    // リードレイヤー記述子から情報を取得します。
    LayerDataArray& layerDatas = *globals->m_pLayerDataArray;
    layerDatas.clear();

    OSErr error;
    const DescriptorKeyID layerSectionKeyId = RGetTypeId("layerSection");
    std::stack<bool> isVisibleStack; // 親レイヤーグループの可視性スタックです。
    std::stack<int> opacityStack; // 親レイヤーグループの不透明度スタックです。
    for (auto pLayerDesc : layerDescPtrs)
    {
        //-----------------------------------------------------------------------------
        // レイヤーグループかどうか判定します。
        const int layerRefIdx = (backGroundExists) ? pLayerDesc->sheetID : 1 + pLayerDesc->sheetID;
            // 【注意】アクションリファレンス用インデックスは、背景が存在しない場合は 1 + sheetID
        const DescriptorEnumID layerSection = RGetEnumeratedProperty(&error,
            layerRefIdx, classLayer, layerSectionKeyId);
        const std::string layerSectionStr = RGetStringId(layerSection);
        const bool isGroupStart = (layerSectionStr == "layerSectionStart");
        const bool isGroupEnd   = (layerSectionStr == "layerSectionEnd");

        //-----------------------------------------------------------------------------
        // 親レイヤーグループの可視性を反映したグローバルな可視性を取得します。
        const bool isParentVisible = (isVisibleStack.empty() || isVisibleStack.top());
        const bool isGlobalVisible = (isParentVisible && pLayerDesc->isVisible == TRUE);
        if (isGroupStart)
        {
            isVisibleStack.push(isGlobalVisible);
        }
        else if (isGroupEnd && !isVisibleStack.empty())
        {
            isVisibleStack.pop();
        }

        //-----------------------------------------------------------------------------
        // 親レイヤーグループの不透明度を乗算したグローバルな不透明度を取得します。
        const int parentOpacity = (opacityStack.empty()) ? 0xff : opacityStack.top();
        const int globalOpacity = (parentOpacity == 0xff) ? pLayerDesc->opacity :
            pLayerDesc->opacity * parentOpacity / 0xff;
        if (isGroupStart)
        {
            opacityStack.push(globalOpacity);
        }
        else if (isGroupEnd && !opacityStack.empty())
        {
            opacityStack.pop();
        }
        //RNoteTrace("layer %d: v%d/%d (%02x %02x): %s: %s", pLayerDesc->sheetID, pLayerDesc->isVisible, isGlobalVisible,
        //    pLayerDesc->opacity, globalOpacity, pLayerDesc->name, layerSectionStr.c_str());

        //-----------------------------------------------------------------------------
        // ピクセルベースのレイヤーならレイヤーデータ配列に追加します。
        if (pLayerDesc->isPixelBased)
        {
            layerDatas.push_back(LayerData(pLayerDesc, isGlobalVisible, globalOpacity));
        }
    }

    //-----------------------------------------------------------------------------
    // 同名レイヤーが複数存在する場合、最優先のもの以外は優先フラグを false にします。
    // 可視で上位にあるレイヤーが優先です。
    RStringArray& layerNames = *globals->m_pLayerNameArray;
    layerNames.clear();
    for (size_t layerIdx = 0; layerIdx < layerDatas.size(); ++layerIdx)
    {
        auto& layerData = layerDatas[layerIdx];
        const int sameNameIdx        = FindLayerData(layerDatas, layerData.m_Name, false, false);
        const int visibleSameNameIdx = FindLayerData(layerDatas, layerData.m_Name, true , false);
        if (!layerData.m_IsVisible)
        {
            // 不可視の場合、同名で可視のレイヤーが存在するか、
            // 上位に同名のレイヤーが存在すれば低優先とします。
            if (visibleSameNameIdx != -1                     ||
                (0 <= sameNameIdx && sameNameIdx < layerIdx))
            {
                layerData.m_HasPriority = false;
            }
        }
        else
        {
            // 可視の場合、上位に同名で可視のレイヤーが存在すれば低優先とします。
            if (0 <= visibleSameNameIdx && visibleSameNameIdx < layerIdx)
            {
                layerData.m_HasPriority = false;
            }
        }

        // 優先フラグが true かつ名前が <Active> でないレイヤーを合成可能とみなして、
        // レイヤー名配列に追加します。
        if (layerData.m_HasPriority &&
            layerData.m_Name != RParameters::ActiveLayerName)
        {
            layerNames.push_back(layerData.m_Name);
        }
        //RNoteTrace("layer: %s: v%d p%d", layerData.m_Name.c_str(), layerData.m_IsVisible, layerData.m_HasPriority);
    }
    //for (const auto& n : layerNames) RNoteTrace("layer name: %s", n.c_str());

    //-----------------------------------------------------------------------------
    // 合成可能なレイヤー名配列中のアクティブレイヤーのインデックスを取得します。
    const std::string activeLayerName = RGetStringProperty(&error, -1, classLayer, keyName);
    globals->m_ActiveLayerIdx = RFindValueInArray(layerNames, activeLayerName);
}

//-----------------------------------------------------------------------------
//! @brief フィルター処理を開始します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoStart(GPtr globals)
{
    //-----------------------------------------------------------------------------
    // validate parameters
    ValidateParameters(globals);

    //-----------------------------------------------------------------------------
    // ワーク用変数を初期化します。
    LayerDataArray layerDatas;
    globals->m_pLayerDataArray = &layerDatas;
    RStringArray layerNames;
    globals->m_pLayerNameArray = &layerNames;
    globals->m_ActiveLayerIdx = -1;

    globals->m_pSrcPixels = nullptr;
    globals->m_pDstPixels = nullptr;
    globals->m_pSrcAlphas = nullptr;
    globals->m_pGrayMatPixels = nullptr;

    //-----------------------------------------------------------------------------
    // エラー発生時に break するための for 文です。
    for (;;)
    {
        //-----------------------------------------------------------------------------
        // 画像の情報を取得します。
        GetImageInfo(globals);

        //-----------------------------------------------------------------------------
        // イメージモードと選択されたチャンネルをチェックします。
        if (gDocInfo->targetCompositeChannels->channelType == ctBlack)
        {
            RShowError(globals, "Please change to RGB color mode");
            gResult = userCanceledErr;
            break;
        }

        if (RGetSelectedChannelCount(gDocInfo->targetCompositeChannels) != R_RGB_COUNT)
        {
            RShowError(globals, "Please select RGB channels");
            gResult = userCanceledErr;
            break;
        }

        //-----------------------------------------------------------------------------
        // スクリプトパラメーターをリードします。
        gResult = ReadScriptParameters(globals);
        if (gResult != noErr)
        {
            break;
        }

        //-----------------------------------------------------------------------------
        // ピクセルデータを Photoshop からリードします。
        ReadPixelsFromHost(globals);
        if (gResult != noErr)
        {
            break;
        }

        //-----------------------------------------------------------------------------
        // プレビュー関連のパラメーターを初期化します。
        globals->m_IsPreviewFiltered = false;
        // オプションダイアログ初期化時にエディットテキストのコールバックで更新されるように
        // UI では設定できない値をプレビュー用パラメーターに設定します。
        memset(&globals->m_PreviewParams, 0x00, sizeof(globals->m_PreviewParams));
        globals->m_PreviewParams.m_HeightScale = -1;

        //-----------------------------------------------------------------------------
        // 必要ならオプションダイアログを表示します。
        if (gQueryForParameters)
        {
            if (!DoOptionDialog(globals))
            {
                gResult = userCanceledErr;
                break;
            }
            gQueryForParameters = FALSE;
        }

        //-----------------------------------------------------------------------------
        // プレビューで計算済みでなければフィルター処理します。
        if (!globals->m_IsPreviewFiltered)
        {
            // 合成する他の法線マップをリードします。
            //RNoteTrace("not previewed");
            RImage otherImg;
            if (gParams->m_Operation == RParameters::Operation_CombineFile)
            {
                if (!otherImg.ReadFile(gParams->m_OtherPath))
                {
                    RShowError(globals, otherImg.GetErrorString().c_str());
                    gResult = userCanceledErr;
                    break;
                }
            }

            // フィルター処理します。
            DoFilter(globals, otherImg);
        }

        //-----------------------------------------------------------------------------
        // ピクセルデータを Photoshop にライトします。
        if (gResult == noErr)
        {
            WritePixelsToHost(globals);
        }
        break;
    }

    //-----------------------------------------------------------------------------
    // ワーク用変数をクリアし、メモリーを解放します。
    globals->m_pLayerDataArray = nullptr;
    globals->m_pLayerNameArray = nullptr;
    //RNoteTrace("free: alpha %p, gray mat %p", globals->m_pSrcAlphas, globals->m_pGrayMatPixels);
    RFreeAndClearArray(globals->m_pSrcPixels);
    RFreeAndClearArray(globals->m_pDstPixels);
    RFreeAndClearArray(globals->m_pSrcAlphas);
    RFreeAndClearArray(globals->m_pGrayMatPixels);

    //-----------------------------------------------------------------------------
    // フィルター処理を継続しないように設定します。
    PISetRect(&gStuff->outRect, 0, 0, 0, 0); // no continue
    memset(&gBigDoc->outRect32, 0x00, sizeof(gBigDoc->outRect32)); // no continue
}

//-----------------------------------------------------------------------------
//! @brief フィルター処理を継続します。
//!        gStuff->inRect、gStuff->outRect、gStuff->maskRect
//!        のいずれかひとつが空でなければ呼ばれます。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoContinue(GPtr globals)
{
    //RNoteTrace("do continue");
    R_UNUSED_VARIABLE(globals);
}

//-----------------------------------------------------------------------------
//! @brief フィルター処理を終了します。
//!
//! @param[in,out] globals グローバルデータです。
//-----------------------------------------------------------------------------
static void DoFinish(GPtr globals)
{
    //RNoteTrace("do finish");

    //-----------------------------------------------------------------------------
    // write script parameter
    gResult = WriteScriptParameters(globals);
}

//-----------------------------------------------------------------------------
//! @brief プラグインのメイン関数です。
//-----------------------------------------------------------------------------
DLLExport MACPASCAL void PluginMain(
    const int16 selector,
    FilterRecordPtr filterParamBlock,
    intptr_t* data,
    int16* result
)
{
    //-----------------------------------------------------------------------------
    // do each proc
    if (selector == filterSelectorAbout)
    {
        //-----------------------------------------------------------------------------
        // do about
        AboutRecordPtr aboutRecord = reinterpret_cast<AboutRecordPtr>(filterParamBlock);
        sSPBasic = aboutRecord->sSPBasic;
        DoAboutDialog(aboutRecord);
    }
    else
    {
        //-----------------------------------------------------------------------------
        // alloc global
        FilterRecord* filterRecord = reinterpret_cast<FilterRecord*>(filterParamBlock);
        sSPBasic = filterRecord->sSPBasic;

        Ptr globalPtr = nullptr; // Pointer for global structure
        GPtr globals = nullptr; // actual globals

        // alloc & init globals
        globalPtr = AllocateGlobals(result,
            filterParamBlock, filterParamBlock->handleProcs,
            sizeof(Globals), data, InitGlobals);
        if (globalPtr == nullptr)
        {
            *result = memFullErr;
            return;
        }

        globals = reinterpret_cast<GPtr>(globalPtr);

        //-----------------------------------------------------------------------------
        // 32bit の座標を使用します。
        gBigDoc->PluginUsing32BitCoordinates = 1;

        //-----------------------------------------------------------------------------
        // dispatch selector
        try
        {
            switch (selector)
            {
                case filterSelectorParameters:
                    DoParameters(globals);
                    break;
                case filterSelectorPrepare:
                    DoPrepare(globals);
                    break;
                case filterSelectorStart:
                    DoStart(globals);
                    break;
                case filterSelectorContinue:
                    DoContinue(globals);
                    break;
                case filterSelectorFinish:
                    DoFinish(globals);
                    break;
                default:
                    break;
            }
        }
        catch (int16 inError)
        {
            *result = inError;
        }
        catch (...)
        {
            *result = -1; // program error
        }

        //-----------------------------------------------------------------------------
        // unlock data
        if (reinterpret_cast<Handle>(*data) != nullptr)
        {
            PIUnlockHandle(reinterpret_cast<Handle>(*data));
        }
    }

    //-----------------------------------------------------------------------------
    // release suites
    if (selector == filterSelectorAbout  ||
        selector == filterSelectorFinish ||
        *result != noErr)
    {
        PIUSuitesRelease();
    }
}

