﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// constants
// RXMLElement
// RBinDataStream
// RUserData
// RUnicodeCArray
// RFileBuf

// RProgress RAboutParam
// RTexCvtr RTexEncoder RImage RDParse

//=============================================================================
// include
//=============================================================================
#pragma warning(push)
    #pragma warning(disable: 4121) // メンバのアライメントは過剰にパッキングされています。
    #pragma warning(disable: 4512) // このクラスに対する代入演算子を生成できません。
    #pragma warning(disable: 4800) // ブール値を 'true' または 'false' に強制的に設定します
    #include "PIUtilities.h" // -> FileUtilities.h -> PIFormat.h
    #include "PIChannelPortsSuite.h"
    #include "PIChannelPortOperations.h"
    #include "DialogUtilities.h"
    #include "PropertyUtils.h"
#pragma warning(pop)

#include <algorithm>
#include <cmath>
#include <cstdint>
#include <ctime>
#include <iomanip>
#include <iostream>
#include <fstream>
#include <sstream>
#include <stack>
#include <string>
#include <vector>
#include <map>
#include <shlwapi.h>

// TextureConverter headers
#include <Encoder/Encoder.h>

// 3dTextureConverter headers
#include <Sources/3dTextureConverter.h>
using namespace nn::gfx::tool::texcvtr;

#ifndef NN_IMPLICIT
    #define NN_IMPLICIT
#endif

//=============================================================================
// グローバルを関数の引数で使うために宣言します。
//=============================================================================
class Globals;
typedef Globals* GPtr;

//=============================================================================
// nps ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace nps {

//=============================================================================
// constants
//=============================================================================

//-----------------------------------------------------------------------------
// color
const int R_RGB_COUNT  = 3; //!< RGB の成分数です。
const int R_RGBA_COUNT = 4; //!< RGBA の成分数です。

const int R_RGB_BYTES  = 3; //!< RGB 各 8 ビット整数のピクセルのバイト数です。
const int R_RGBA_BYTES = 4; //!< RGBA 各 8 ビット整数のピクセルのバイト数です。

const int R_RGB_FLOAT_BYTES  = 3 * sizeof(float); //!< RGB 各 32 ビット浮動小数点数のピクセルのバイト数です。
const int R_RGBA_FLOAT_BYTES = 4 * sizeof(float); //!< RGBA 各 32 ビット浮動小数点数のピクセルのバイト数です。

const int R_OPA_ALPHA_BORDER = 0x80; //!< 8bit のアルファを 2 値化する際にこの値以上なら 1 とします。

//-----------------------------------------------------------------------------
// math
const double R_SAME_TOLERANCE   = 1.0e-6;  //!< double 型の値を比較する際の許容値です。
const float  R_SAME_TOLERANCE_F = 1.0e-6f; //!< float 型の値を比較する際の許容値です。

const double R_M_PI = 3.14159265358979323846;
const double R_M_RAD_TO_DEG = 180.0 / R_M_PI;
const double R_M_DEG_TO_RAD = R_M_PI / 180.0;

const int R_XYZ_COUNT  = 3; //!< XYZ の成分数です。
const int R_XYZW_COUNT = 4; //!< XYZW の成分数です。

//-----------------------------------------------------------------------------
// output
#define R_ENDL "\r\n"

const int R_FLOAT_OUT_PRECISION = 6; //!< std::ostream::precision で指定する精度です。

const int R_BINARY_ALIGNMENT = 32; //!< バイナリー形式の中間ファイルのアライメントです。

//-----------------------------------------------------------------------------
// テクスチャー

//! @brief テクスチャーの次元を表す列挙型です。
enum FtxDimension
{
    FtxDimension_1d, //!< 一次元です。
    FtxDimension_2d, //!< 二次元です。
    FtxDimension_3d, //!< 三次元です。
    FtxDimension_CubeMap, //!< キューブマップイメージです。
    FtxDimension_1dArray, //!< 一次元配列です。
    FtxDimension_2dArray, //!< 二次元配列です。
    FtxDimension_2dMultisample, //<! マルチサンプルイメージです。
    FtxDimension_2dMultisampleArray, //!< マルチサンプルイメージ配列です。
    FtxDimension_CubeMapArray, //!< キューブマップイメージ配列です。

    FtxDimension_End
};

//! @brief テクスチャーフォーマットを表す列挙型です。
enum FtxFormat
{
    FtxFormat_Invalid,

    FtxFormat_Unorm_8,
    FtxFormat_Snorm_8,
    FtxFormat_Uint_8,
    FtxFormat_Sint_8,
    FtxFormat_Unorm_16,
    FtxFormat_Snorm_16,
    FtxFormat_Uint_16,
    FtxFormat_Sint_16,
    FtxFormat_Uint_32,
    FtxFormat_Sint_32,

    FtxFormat_Unorm_4_4,
    FtxFormat_Unorm_8_8,
    FtxFormat_Snorm_8_8,
    FtxFormat_Uint_8_8,
    FtxFormat_Sint_8_8,
    FtxFormat_Unorm_16_16,
    FtxFormat_Snorm_16_16,
    FtxFormat_Uint_16_16,
    FtxFormat_Sint_16_16,
    FtxFormat_Uint_32_32,
    FtxFormat_Sint_32_32,

    FtxFormat_Unorm_5_6_5,

    FtxFormat_Unorm_5_5_5_1,
    FtxFormat_Unorm_4_4_4_4,

    FtxFormat_Unorm_8_8_8_8,
    FtxFormat_Snorm_8_8_8_8,
    FtxFormat_Srgb_8_8_8_8,
    FtxFormat_Uint_8_8_8_8,
    FtxFormat_Sint_8_8_8_8,
    FtxFormat_Unorm_10_10_10_2 ,
    FtxFormat_Uint_10_10_10_2,
    FtxFormat_Unorm_16_16_16_16,
    FtxFormat_Snorm_16_16_16_16,
    FtxFormat_Uint_16_16_16_16,
    FtxFormat_Sint_16_16_16_16,
    FtxFormat_Uint_32_32_32_32,
    FtxFormat_Sint_32_32_32_32,

    FtxFormat_Float_16,
    FtxFormat_Float_32,
    FtxFormat_Float_16_16,
    FtxFormat_Float_32_32,
    FtxFormat_Float_11_11_10,
    FtxFormat_Float_16_16_16_16,
    FtxFormat_Float_32_32_32_32,

    FtxFormat_Unorm_Bc1,
    FtxFormat_Srgb_Bc1,
    FtxFormat_Unorm_Bc2,
    FtxFormat_Srgb_Bc2,
    FtxFormat_Unorm_Bc3,
    FtxFormat_Srgb_Bc3,
    FtxFormat_Unorm_Bc4,
    FtxFormat_Snorm_Bc4,
    FtxFormat_Unorm_Bc5,
    FtxFormat_Snorm_Bc5,

    FtxFormat_Ufloat_Bc6,
    FtxFormat_Float_Bc6,
    FtxFormat_Unorm_Bc7,
    FtxFormat_Srgb_Bc7,

    FtxFormat_Unorm_Etc1,
    FtxFormat_Unorm_Etc2,
    FtxFormat_Srgb_Etc2,
    FtxFormat_Unorm_Etc2_Mask,
    FtxFormat_Srgb_Etc2_Mask,
    FtxFormat_Unorm_Etc2_Alpha,
    FtxFormat_Srgb_Etc2_Alpha,

    FtxFormat_Unorm_Eac_11,
    FtxFormat_Snorm_Eac_11,
    FtxFormat_Unorm_Eac_11_11,
    FtxFormat_Snorm_Eac_11_11,

    FtxFormat_Unorm_Pvrtc1_2Bpp,
    FtxFormat_Srgb_Pvrtc1_2Bpp,
    FtxFormat_Unorm_Pvrtc1_4Bpp,
    FtxFormat_Srgb_Pvrtc1_4Bpp,
    FtxFormat_Unorm_Pvrtc1_Alpha_2Bpp,
    FtxFormat_Srgb_Pvrtc1_Alpha_2Bpp,
    FtxFormat_Unorm_Pvrtc1_Alpha_4Bpp,
    FtxFormat_Srgb_Pvrtc1_Alpha_4Bpp,
    FtxFormat_Unorm_Pvrtc2_Alpha_2Bpp,
    FtxFormat_Srgb_Pvrtc2_Alpha_2Bpp,
    FtxFormat_Unorm_Pvrtc2_Alpha_4Bpp,
    FtxFormat_Srgb_Pvrtc2_Alpha_4Bpp,

    FtxFormat_Unorm_Astc_4x4,
    FtxFormat_Srgb_Astc_4x4,
    FtxFormat_Unorm_Astc_5x4,
    FtxFormat_Srgb_Astc_5x4,
    FtxFormat_Unorm_Astc_5x5,
    FtxFormat_Srgb_Astc_5x5,
    FtxFormat_Unorm_Astc_6x5,
    FtxFormat_Srgb_Astc_6x5,
    FtxFormat_Unorm_Astc_6x6,
    FtxFormat_Srgb_Astc_6x6,
    FtxFormat_Unorm_Astc_8x5,
    FtxFormat_Srgb_Astc_8x5,
    FtxFormat_Unorm_Astc_8x6,
    FtxFormat_Srgb_Astc_8x6,
    FtxFormat_Unorm_Astc_8x8,
    FtxFormat_Srgb_Astc_8x8,
    FtxFormat_Unorm_Astc_10x5,
    FtxFormat_Srgb_Astc_10x5,
    FtxFormat_Unorm_Astc_10x6,
    FtxFormat_Srgb_Astc_10x6,
    FtxFormat_Unorm_Astc_10x8,
    FtxFormat_Srgb_Astc_10x8,
    FtxFormat_Unorm_Astc_10x10,
    FtxFormat_Srgb_Astc_10x10,
    FtxFormat_Unorm_Astc_12x10,
    FtxFormat_Srgb_Astc_12x10,
    FtxFormat_Unorm_Astc_12x12,
    FtxFormat_Srgb_Astc_12x12,

    FtxFormat_Uint_32_32_32,
    FtxFormat_Sint_32_32_32,
    FtxFormat_Float_32_32_32,

    FtxFormat_End
};

//! @brief テクスチャーの成分選択の成分を表す列挙型です。
enum FtxComponent
{
    FtxComponent_Red,   //!< R 成分です。
    FtxComponent_Green, //!< G 成分です。
    FtxComponent_Blue,  //!< B 成分です。
    FtxComponent_Alpha, //!< A 成分です。
    FtxComponent_Zero,  //!< 固定値 0 です。
    FtxComponent_One,   //!< 固定値 1 です。
    FtxComponent_Count  //!< 成分数です。
};

//! @brief テクスチャーの成分選択を表す型定義です。
typedef uint32_t FtxCompSel;

const int FtxCompSelShiftR = 24; //!< 成分選択値の R 成分のシフト数です。
const int FtxCompSelShiftG = 16; //!< 成分選択値の G 成分のシフト数です。
const int FtxCompSelShiftB =  8; //!< 成分選択値の B 成分のシフト数です。
const int FtxCompSelShiftA =  0; //!< 成分選択値の A 成分のシフト数です。
#define FTX_GET_COMP_SEL(r, g, b, a) (\
    (((r) & 0xff) << FtxCompSelShiftR) |\
    (((g) & 0xff) << FtxCompSelShiftG) |\
    (((b) & 0xff) << FtxCompSelShiftB) |\
    (((a) & 0xff) << FtxCompSelShiftA))

const FtxCompSel FtxCompSelNone = FTX_GET_COMP_SEL(FtxComponent_Zero, FtxComponent_Zero , FtxComponent_Zero, FtxComponent_Zero );
const FtxCompSel FtxCompSelRgba = FTX_GET_COMP_SEL(FtxComponent_Red , FtxComponent_Green, FtxComponent_Blue, FtxComponent_Alpha);
const FtxCompSel FtxCompSelRrr1 = FTX_GET_COMP_SEL(FtxComponent_Red , FtxComponent_Red  , FtxComponent_Red , FtxComponent_One  );
const FtxCompSel FtxCompSelRrrg = FTX_GET_COMP_SEL(FtxComponent_Red , FtxComponent_Red  , FtxComponent_Red , FtxComponent_Green);
const FtxCompSel FtxCompSelRa01 = FTX_GET_COMP_SEL(FtxComponent_Red , FtxComponent_Alpha, FtxComponent_Zero, FtxComponent_One  );

const int AstcBlockBytes = 16; //!< ASTC フォーマットの 1 ブロックのバイト数です。

//-----------------------------------------------------------------------------
// ヒント情報
extern const std::string HintValueDefault; //!< デフォルトのヒント情報です。
extern const std::string HintValueNormal; //!< 法線用のヒント情報です。

//-----------------------------------------------------------------------------
// Photoshop
const int R_PS_PALETTE_MAX = 256;
const double R_PS_BASE_RESOLUTION = 72.0;

const int PhotoshopVersionMajorCs6      = 13; //!< Photoshop CS6 のメジャーバージョンナンバーです。
const int PhotoshopVersionMajorCc       = 14; //!< Photoshop CC のメジャーバージョンナンバーです。
const int PhotoshopVersionMajorCc2014   = 15; //!< Photoshop CC 2014 のメジャーバージョンナンバーです。
const int PhotoshopVersionMajorCc2015   = 16; //!< Photoshop CC 2015 のメジャーバージョンナンバーです。
const int PhotoshopVersionMajorCc2015_5 = 17; //!< Photoshop CC 2015.5 のメジャーバージョンナンバーです。

//-----------------------------------------------------------------------------
// add info
const uint32_t AI_TAG_SIZE = 8;
const uint32_t AI_SIZE_SIZE = 4;
const uint32_t AI_HEADER_SIZE = AI_TAG_SIZE + AI_SIZE_SIZE;

//-----------------------------------------------------------------------------
// リニア変換
class RLinearFlag
{
public:
    //! @brief リニア変換フラグを表す列挙型です。
    enum LinearFlag
    {
        LINEAR_NONE    = 0,        //!< sRGB からリニアに変換しません。
        LINEAR_R       = (1 << 3), //!< R 成分を sRGB からリニアに変換します。
        LINEAR_G       = (1 << 2), //!< G 成分を sRGB からリニアに変換します。
        LINEAR_B       = (1 << 1), //!< B 成分を sRGB からリニアに変換します。
        LINEAR_A       = (1 << 0), //!< A 成分を sRGB からリニアに変換します。
        LINEAR_RGB     = LINEAR_R | LINEAR_G | LINEAR_B, //!< RGB 成分を sRGB からリニアに変換します。
        LINEAR_RGBA    = LINEAR_RGB | LINEAR_A,          //!< RGBA 成分を sRGB からリニアに変換します。
        LINEAR_INVALID = -2,       //!< 無効なリニア変換です。
        LINEAR_DEFAULT = -1        //!< デフォルトのリニア変換（指定なし）です。
    };
};

//-----------------------------------------------------------------------------
// ミップマップ生成フィルタ
class RMipGenFilter
{
public:
    //! @brief ミップマップ生成フィルタを表す列挙型です。
    enum MipGenFilter
    {
        MIP_GEN_POINT,  //!< ポイントサンプルです。
        MIP_GEN_LINEAR, //!< リニア補間です。
        MIP_GEN_CUBIC,  //!< キュービック補間です。
        MIP_GEN_FILTER_COUNT, //!< ミップマップ生成フィルタの総数です。
        MIP_GEN_INVALID = -2, //!< 無効なミップマップ生成フィルタです。
        MIP_GEN_DEFAULT = -1  //!< デフォルトのミップマップ生成フィルタ（指定なし）です。
    };
};

//-----------------------------------------------------------------------------
// リソースデータ
#ifdef __PIWin__
const uint32_t RD_TYPE_NN = 'Pfwn'; // resource type (nwfP) (Win)
#else
const uint32_t RD_TYPE_NN = 'nwfP'; // resource type (nwfP) (Mac)
#endif

extern const char RD_TAG_RD_VER[]; //!< リソースデータのバージョンタグです。
extern const char RD_TAG_DCC_PRESET[]; //!< リソースデータのプリセット名（コンフィグ名）タグです。
extern const char RD_TAG_HINT[]; //!< リソースデータのヒント情報タグです。
extern const char RD_TAG_LINEAR[]; //!< リソースデータのリニア変換フラグタグです。
extern const char RD_TAG_DIMENSION[]; //!< リソースデータの次元タグです。
extern const char RD_TAG_FORMAT[]; //!< リソースデータのフォーマットタグです。
extern const char RD_TAG_WEIGHT_CMPR[]; //!< リソースデータの BC 圧縮誤差の重み付けタグです。
extern const char RD_TAG_INI_SWIZZLE[]; //!< リソースデータの初期スウィズル値タグです。
extern const char RD_TAG_MIP_LEVEL[]; //!< リソースデータのミップマップのレベル数タグです。
extern const char RD_TAG_MIP_GEN_FILTER[]; //!< リソースデータのミップマップ生成フィルタタグです。
extern const char RD_TAG_COMP_SEL[]; //!< リソースデータの成分選択タグです。
extern const char RD_TAG_ORG_PATHS[]; //!< リソースデータのオリジナルパス配列タグです。
extern const char RD_TAG_CREATE_INFO[]; //!< リソースデータの <create> 要素タグです。
extern const char RD_TAG_USER_DATA[]; //!< リソースデータのユーザーデータ群タグです。
extern const char RD_TAG_COMMENT_LABEL[]; //!< リソースデータの編集用コメントラベルタグです。
extern const char RD_TAG_COMMENT_COLOR[]; //!< リソースデータの編集用コメントカラー文字列タグです。
extern const char RD_TAG_COMMENT_TEXT[]; //!< リソースデータの編集用コメント文タグです。
extern const char RD_TAG_TOOL_DATA[]; //!< リソースデータの <tool_data> 要素タグです。
extern const char RD_TAG_USER_TOOL_DATA[]; //!< リソースデータの <user_tool_data> 要素タグです。

const int RD_VER_CURRENT = 10; //!< 現在のリソースデータのバージョンです。

//-----------------------------------------------------------------------------
// コンフィグファイル
extern const char* R_STANDARD_CONFIG_NAME; //!< 標準のコンフィグ名（内部処理用）です。

//-----------------------------------------------------------------------------
// help
#define NPS_HELP_INDEX_URL "PhotoshopPlugin.html"

//=============================================================================
//! @brief 未使用変数の警告を抑止するためのマクロです。
//=============================================================================
#define R_UNUSED_VARIABLE(Variable) (void)(&Variable);

//=============================================================================
//! @brief コンテナの定義です。
//=============================================================================

//! @brief char 型配列の定義です。
typedef std::vector<char> RCharArray;

//! @brief unsigned char 型配列の定義です。
typedef std::vector<unsigned char> RUCharArray;

//! @brief short 型配列の定義です。
typedef std::vector<short> RShortArray;

//! @brief unsigned short 型配列の定義です。
typedef std::vector<unsigned short> RUShortArray;

//! @brief int 型配列の定義です。
typedef std::vector<int> RIntArray;

//! @brief unsigned int 型配列の定義です。
typedef std::vector<unsigned int> RUIntArray;

//! @brief float 型配列の定義です。
typedef std::vector<float> RFloatArray;

//! @brief double 型配列の定義です。
typedef std::vector<double> RDoubleArray;

//! @brief 文字列型配列の定義です。
typedef std::vector<std::string> RStringArray;

//! @brief ユニコード文字列型配列の定義です。
typedef std::vector<std::wstring> RWStringArray;

//! @brief bool 型配列の定義です。
typedef std::vector<bool> RBoolArray;

//! @brief 文字列から文字列へのマップの定義です。
typedef std::map<std::string, std::string> RStringMap;

//=============================================================================
// デバッグ関連の関数です。
//=============================================================================

//! @brief printf の書式でメッセージボックスを表示します（デバッグ用）。
//!
//! @param[in] format 書式です。
//!
void RMsgBox(const char* format, ...);

//! @brief printf の書式でデバッグウィンドウに出力します（デバッグ用）。
//!
//! @param[in] format 書式です。
//!
void RTrace(const char* format, ...);

//! @brief デバッグウィンドウにメモリーダンプを出力します（デバッグ用）。
//!
//! @param[in] mem メモリーのポインタです。
//! @param[in] size ダンプ出力するバイト数です。
//! @param[in] title 出力するタイトル文字列です。NULL ならタイトルを出力しません。
//!
void RDumpMemory(const void* mem, const int size, const char* title = NULL);

//! @brief メッセージボックスにメモリーダンプを出力します（デバッグ用）。
//!
//! @param[in] mem メモリーのポインタです。
//! @param[in] size ダンプ出力するバイト数です。
//! @param[in] title 出力するタイトル文字列です。NULL ならタイトルを出力しません。
//!
void RDumpMemoryBox(const void* mem, const int size, const char* title = NULL);

#ifdef __PIWin__
//! @brief 開いているメモ帳のウィンドウにメッセージを出力します（デバッグ用）。
//!
//! @param[in] format 書式です。
//!
void RNoteTrace(const char* format, ...);

//! @brief 開いているメモ帳のウィンドウにメモリーダンプを出力します（デバッグ用）。
//!
//! @param[in] mem メモリーのポインタです。
//! @param[in] size ダンプ出力するバイト数です。
//! @param[in] title 出力するタイトル文字列です。NULL ならタイトルを出力しません。
//!
void RNoteDumpMemory(const void* mem, const int size, const char* title = NULL);
#endif

//! @brief メモリーの状態をデバッグウィンドウに出力します（デバッグ用）。
//!
//! @param[in] title 出力するタイトル文字列です。
//!
void RCheckMemoryStatus(const char* title);

//=============================================================================
// エラー・警告関連の関数です。
//=============================================================================

//! @brief プラグインのエラーを表示します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] format 書式です。
//!
void RShowError(GPtr globals, const char* format, ...);

//! @brief プラグインの警告を表示します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] format 書式です。
//!
void RShowWarning(GPtr globals, const char* format, ...);

//=============================================================================
// 一般的な数学関連の関数です。
//=============================================================================

//! @brief 絶対値を返します。
template <typename T>
inline T RAbs(const T x)
{
    return (x >= (T)0) ? x : -x;
}

//! @brief 四捨五入した値を返します。
template <typename T>
inline int RRound(const T x)
{
    return (x >= (T)0) ? static_cast<int>(x + (T)0.5) : static_cast<int>(x - (T)0.5);
}

//! @brief 小さいほうの値を返します。
template <typename T>
inline T RMin(const T a, const T b)
{
    return (a < b) ? a : b;
}

//! @brief 大きいほうの値を返します。
template <typename T>
inline T RMax(const T a, const T b)
{
    return (a > b) ? a : b;
}

//! @brief 指定した範囲にクランプした値を返します。
template <typename T>
inline T RClampValue(const T valMin, const T valMax, const T val)
{
    if (val < valMin)
    {
        return valMin;
    }
    else if (val > valMax)
    {
        return valMax;
    }
    else
    {
        return val;
    }
}

//! @brief アライメントした値を返します。
template <typename T>
inline T RAlignValue(T val, const int align)
{
    return (T)((val + align - 1) & (~(align - 1)));
}

//! @brief アライメントしたポインタを返します。
template <typename T>
inline T* RAlignPointer(T* val, const int align)
{
    #ifdef _WIN64
    return (T*)(((unsigned __int64)val + align - 1) & (~(align - 1)));
    #else
    return (T*)(((unsigned __int32)val + align - 1) & (~(align - 1)));
    #endif
}

//! @brief 2 つの値を交換します。
template <typename T>
inline void RSwapValue(T& r1, T& r2)
{
    T tmpVal = r1;
    r1 = r2;
    r2 = tmpVal;
}

//! @brief 2 のべき乗の数なら true を返します。
//!
//! @param[in] number 判定する数です。
//! @param[in] includeOne 1 も 2 のべき乗の数（2^0）とみなすなら true を指定します。
//!
//! @return 2 のべき乗の数なら true を返します。
//!
bool RIsPower2Number(const int number, const bool includeOne = true);

//! @brief 指定した数以上で最小の 2 のべき乗の数を返します。
//!
//! @param[in] number 数です。
//! @param[in] includeOne 1 も 2 のべき乗の数（2^0）とみなすなら true を指定します。
//!
//! @return 指定した数以上で最小の 2 のべき乗の数を返します。
//!
int RGetPower2Number(const int number, const bool includeOne = true);

//=============================================================================
// 文字列関連の関数です。
//=============================================================================

//! @brief 文字が半角スペースかタブなら true を返します。
inline bool RIsSpace(const char c)
{
    return (c == ' ' || c == '\t');
}

bool RIsShiftJisFirstByte(const unsigned char val);
std::string RGetUtf8FromShiftJis(const std::string& src);
std::string RGetShiftJisFromUtf8(const std::string& src);
std::wstring RGetUnicodeFromShiftJis(const std::string& src);
std::string RGetShiftJisFromUnicode(const std::wstring& src);
bool RIsAsciiString(const std::string& str);

std::string RJoinStrings(const RStringArray& srcs, const std::string& sep);
int RSeparateString(RStringArray& dsts, const std::string& src, const std::string& sep);

//! @brief ユニコード C 配列を RStringArray に変換します。
//!
//! @param[out] dsts 変換先の文字列配列です。
//! @param[in] srcs ユニコード C 配列です。最後に NULL を格納します。
//!
//! @return 文字列数を返します。
//!
int RGetStringArray(RStringArray& dsts, const wchar_t* srcs[]);

std::string RGetLowerCaseString(const std::string& src);
std::string RGetUpperCaseString(const std::string& src);
bool RIsSameStringNoCase(const std::string& str1, const std::string& str2);
int RFindNoCaseStringInArray(const RStringArray& array, const std::string& value);
std::string RTrimLeftString(const std::string& src, const char* targets = " \t\r\n");
std::string RTrimRightString(const std::string& src, const char* targets = " \t\r\n");
std::string RTrimString(const std::string& src, const char* targets = " \t\r\n");
std::string RCutEndlString(const std::string& src);
std::string RDequoteString(const std::string& src, const char quote = '\"');

//! @brief 文字列を指定した区切り文字で分割します。
//!        ",abc,def,,ghi," -> { "", "abc", "def", "", "ghi", "" }
//!
//! @param[out] words 単語配列を格納します。
//! @param[in] input 文字列です。
//! @param[in] delim 区切り文字です。
//!
//! @return 単語数を返します。
//!
int RSplitString(RStringArray& words, const std::string& input, const char delim);

//! @brief 文字列をトークンに分解します。
//!        "abc\n def\n \n ghi" -> { "abc", "def", "ghi" }
//!
//! @param[out] tokens トークン配列を格納します。
//! @param[in] input 文字列です。
//! @param[in] delims 区切り文字群です。
//!
//! @return トークン数を返します。
//!
int RTokenizeString(RStringArray& tokens, const std::string& input, const char* delims = " \t\r\n");

//! @brief int 型の数値を文字列に変換します。
//!
//! @param[in] num 数値です。
//! @param[in] format フォーマット文字列です。NULL なら "%d" を使用します。
//!
//! @return 文字列を返します。
//!
std::string RGetNumberString(const int num, const char* format = NULL);

//! @brief uint32_t 型の数値を文字列に変換します。
//!
//! @param[in] num 数値です。
//! @param[in] format フォーマット文字列です。NULL なら "%u" を使用します。
//!
//! @return 文字列を返します。
//!
std::string RGetNumberString(const uint32_t num, const char* format = NULL);

//! @brief ポインタを文字列に変換します。
//!
//! @param[in] ptr ポインタです。
//! @param[in] format フォーマット文字列です。NULL なら "%p" を使用します。
//!
//! @return 文字列を返します。
//!
std::string RGetPointerString(const void* ptr, const char* format = NULL);

//! @brief Shift-JIS 文字列を XML 用にエンコードします。
std::string REncodeXmlString(const std::string& src);

//! @brief XML 用にエンコードされた Shift-JIS 文字列をデコードします。
std::string RDecodeXmlString(const std::string& src);

//! @brief C/C++ ソースのコメントをカットします。
//!
//! @param[in] src C/C++ ソース文字列です。
//!
//! @return コメントをカットした文字列を返します。
//!
std::string RCutCSourceComment(const std::string& src);

//! @brief パスカル文字列を std::string に変換して返します。
inline std::string RGetStringFromStr255(const Str255& src)
{
    return std::string(reinterpret_cast<const char*>(&src) + 1, src[0]);
}

//! @brief SDK の HostHandle2CString は末尾に余計な文字が残るのでこちらを使用します。
//!        outString は char[outMaxSize] の大きさで確保しておきます。
//!
//! @param[in] procs ハンドルプロシジャ群です。
//! @param[in] inDataHandle 文字列データのハンドルです。
//! @param[out] outString C 文字列を格納します。
//! @param[in] outMaxSize C 文字列の '\0' を含む最大長です。
//!
void RHandle2CString(
    HandleProcs* procs,
    Handle inDataHandle,
    char* outString,
    const size_t outMaxSize
);

//! @brief ポイントの表示用文字列を取得します。(h,v) の形式です。
//!
//! @param[in] point ポイントです。
//!
//! @return ポイントの表示用文字列を返します。
//!
std::string RGetPointString(const VPoint& point);

//! @brief 矩形の表示用文字列を取得します。(left,top)-(right,bottom) の形式です。
//!
//! @param[in] rect 矩形です。
//!
//! @return 矩形の表示用文字列を返します。
//!
std::string RGetRectString(const VRect& rect);

//! @brief 32bit ID を 4 文字の文字列に変換して返します。
//!
//! @param[in] id 32bit ID です。
//!
//! @return ID を表す文字列を返します。
//!         アスキー文字列で表現できない場合、0x + 16 進数文字列を返します。
//!
std::string RGetStringFrom32BitId(const uint32_t id);

//! @brief StringID から TypeID を取得します。
//!
//! @param[in] stringId StringID です。
//!
//! @return TypeID を返します。変換できない場合は 0 を返します。
//!
DescriptorTypeID RGetTypeId(const std::string& stringId);

//! @brief TypeID から StringID を取得します。
//!        TypeID の値が衝突しているものは正確に変換できません。
//!
//! @param[in] typeId TypeID です。
//!
//! @return StringID を返します。変換できない場合は TypeID を 4 文字の文字列形式で返します。
//!
std::string RGetStringId(const uint32_t typeId);

//=============================================================================
//! @brief ユニコード C 配列のクラスです。
//=============================================================================
class RUnicodeCArray
{
protected:
    RWStringArray m_WStrings; //!< ユニコード文字列配列です。
    const wchar_t** m_CArray; //!< ユニコード C 配列です。

public:
    //! コンストラクタです（RStringArray から生成）。
    explicit RUnicodeCArray(const RStringArray& srcs)
    {
        //std::cerr << "RUnicodeCArray(): " << srcs.size() << std::endl;
        const int strCount = static_cast<int>(srcs.size());
        for (int iStr = 0; iStr < strCount; ++iStr)
        {
            m_WStrings.push_back(RGetUnicodeFromShiftJis(srcs[iStr]));
        }
        CreateCArray();
    }

    //! コンストラクタです（RWStringArray から生成）。
    explicit RUnicodeCArray(const RWStringArray& srcs)
    {
        //std::cerr << "RUnicodeCArray(): " << srcs.size() << std::endl;
        const int strCount = static_cast<int>(srcs.size());
        for (int iStr = 0; iStr < strCount; ++iStr)
        {
            m_WStrings.push_back(srcs[iStr]);
        }
        CreateCArray();
    }

    //! ユニコード C 配列を作成します。
    void CreateCArray()
    {
        const int strCount = static_cast<int>(m_WStrings.size());
        m_CArray = new const wchar_t*[strCount + 1];
        for (int iStr = 0; iStr < strCount; ++iStr)
        {
            m_CArray[iStr] = m_WStrings[iStr].c_str();
        }
        m_CArray[strCount] = NULL; // 最終要素を NULL にします。
    }

    //! デストラクタです。
    virtual ~RUnicodeCArray()
    {
        //std::cerr << "~RUnicodeCArray(): " << m_WStrings.size() << std::endl;
        delete[] m_CArray;
    }

    //! ユニコード C 配列を返します。
    const wchar_t** GetCArray() const { return m_CArray; }
};

//=============================================================================
// ファイルパス関連の関数です。
//=============================================================================
std::string RGetWindowsFilePath(const std::string& path);
std::string RGetUnixFilePath(const std::string& path);
std::string RGetFileNameFromFilePath(const std::string& path);
std::string RGetFolderFromFilePath(const std::string& path);
std::string RGetExtensionFromFilePath(const std::string& path, const bool lower = true);
std::string RGetNoExtensionFilePath(const std::string& path);
bool RIsFullFilePath(const std::string& path);
std::string RGetFullFilePath(const std::string& path, const bool isUnix);

//=============================================================================
// ファイル関連の関数です。
//=============================================================================
bool RFileExists(const std::string& path);
bool RFolderExists(const std::string& path);

//=============================================================================
// ファイル出力関連の関数です。
//=============================================================================
void RInitOutStreamFormat(std::ostream& os);
void ROutUtf8Bom(std::ostream& os);
void ROutUtf16Bom(std::ostream& os);
void ROutUtf8FromShiftJis(std::ostream& os, const std::string& sjisStr);
const char* RTab(const int count = 1);
const char* RBoolStr(const bool value);

//=============================================================================
// 配列関連の関数です。
//=============================================================================

//! @brief std::vector 配列から値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
template <typename T>
int RFindValueInArray(const std::vector<T>& array, const T& value)
{
    const int size = static_cast<int>(array.size());
    for (int index = 0; index < size; ++index)
    {
        if (array[index] == value)
        {
            return index;
        }
    }
    return -1;
}

//=============================================================================
// メモリー関連の関数です。
//=============================================================================

//! @brief オブジェクトへのポインタが NULL でなければ delete して NULL を代入します。
//!
//! @param[in,out] ptr オブジェクトへのポインタです。
//!
template <typename T>
inline void RFreeAndClearObject(T*& ptr)
{
    if (ptr != NULL)
    {
        delete ptr;
        ptr = NULL;
    }
}

//! @brief 配列へのポインタが NULL でなければ delete[] して NULL を代入します。
//!
//! @param[in,out] ptr 配列へのポインタです。
//!
template <typename T>
inline void RFreeAndClearArray(T*& ptr)
{
    if (ptr != NULL)
    {
        delete[] ptr;
        ptr = NULL;
    }
}

//! @brief リトルエンディアンで値をメモリーに格納します。
template <typename T>
void RSetMemLittle(void* mem, T value)
{
    #ifdef __PIMac__
    // swap
    uint8_t* top = reinterpret_cast<uint8_t*>(&value);
    uint8_t* p1 = top;
    uint8_t* p2 = p1 + sizeof(T) - 1;
    int byteSizeHalf = sizeof(T) / 2;
    for (int ibyte = 0; ibyte < byteSizeHalf; ++ibyte, ++p1, --p2)
    {
        RSwapValue(*p1, *p2);
    }
    #endif
    memcpy(mem, &value, sizeof(T));
}

//! @brief ビッグエンディアンで値をメモリーに格納します。
template <typename T>
void RSetMemBig(void* mem, T value)
{
    #ifdef __PIWin__
    // swap
    uint8_t* top = reinterpret_cast<uint8_t*>(&value);
    uint8_t* p1 = top;
    uint8_t* p2 = p1 + sizeof(T) - 1;
    int byteSizeHalf = sizeof(T) / 2;
    for (int ibyte = 0; ibyte < byteSizeHalf; ++ibyte, ++p1, --p2)
    {
        RSwapValue(*p1, *p2);
    }
    #endif
    memcpy(mem, &value, sizeof(T));
}

//! @brief メモリーからリトルエンディアンの値を取得します。
template <typename T>
void RGetMemLittle(T& value, const void* mem)
{
    memcpy(&value, mem, sizeof(T));
    #ifdef __PIMac__
    // swap
    uint8_t* p1 = reinterpret_cast<uint8_t*>(&value);
    uint8_t* p2 = p1 + sizeof(T) - 1;
    int byteSizeHalf = sizeof(T) / 2;
    for (int ibyte = 0; ibyte < byteSizeHalf; ++ibyte, ++p1, --p2)
    {
        RSwapValue(*p1, *p2);
    }
    #endif
}

//! @brief メモリーからビッグエンディアンの値を取得します。
template <typename T>
void RGetMemBig(T& value, const void* mem)
{
    memcpy(&value, mem, sizeof(T));
    #ifdef __PIWin__
    // swap
    uint8_t* p1 = reinterpret_cast<uint8_t*>(&value);
    uint8_t* p2 = p1 + sizeof(T) - 1;
    int byteSizeHalf = sizeof(T) / 2;
    for (int ibyte = 0; ibyte < byteSizeHalf; ++ibyte, ++p1, --p2)
    {
        RSwapValue(*p1, *p2);
    }
    #endif
}

//! @brief メモリーからリトルエンディアンの short 値を取得します。
inline short RGetMemLittleShort(const void* pMemory)
{
    short value;
    RGetMemLittle(value, pMemory);
    return value;
}

//! @brief メモリーからリトルエンディアンの unsigned short 値を取得します。
inline unsigned short RGetMemLittleUShort(const void* pMemory)
{
    unsigned short value;
    RGetMemLittle(value, pMemory);
    return value;
}

//! @brief メモリーからリトルエンディアンの int 値を取得します。
inline int RGetMemLittleInt(const void* pMemory)
{
    int value;
    RGetMemLittle(value, pMemory);
    return value;
}

//! @brief メモリーからリトルエンディアンの unsigned int 値を取得します。
inline unsigned int RGetMemLittleUInt(const void* pMemory)
{
    unsigned int value;
    RGetMemLittle(value, pMemory);
    return value;
}

//! @brief メモリーからビッグエンディアンの short 値を取得します。
inline short RGetMemBigShort(const void* pMemory)
{
    short value;
    RGetMemBig(value, pMemory);
    return value;
}

//! @brief メモリーからビッグエンディアンの unsigned short 値を取得します。
inline unsigned short RGetMemBigUShort(const void* pMemory)
{
    unsigned short value;
    RGetMemBig(value, pMemory);
    return value;
}

//! @brief メモリーからビッグエンディアンの int 値を取得します。
inline int RGetMemBigInt(const void* pMemory)
{
    int value;
    RGetMemBig(value, pMemory);
    return value;
}

//! @brief メモリーからビッグエンディアンの unsigned int 値を取得します。
inline unsigned int RGetMemBigUInt(const void* pMemory)
{
    unsigned int value;
    RGetMemBig(value, pMemory);
    return value;
}

//=============================================================================
// 一般的な Windows 関連の関数です。
//=============================================================================

//! @brief 環境変数の値を取得します。
//!
//! @param[in] name 環境変数の名前です。
//!
//! @return 環境変数の値を返します。環境変数が定義されていない場合は空文字を返します。
//!
std::string RGetEnvVar(const char* name);

//! @brief 環境変数が定義されていて値が 0 でなければ true を返します。
//!
//! @param[in] name 環境変数の名前です。
//!
//! @return 環境変数が定義されていて値が 0 でなければ true を返します。
//!
bool RIsEnvVarNotZero(const char* name);

//! @brief 指定されたモジュールを含む実行ファイルのフルパスを返します。
//!
//! @param[in] hModule モジュールのハンドルです
//!                    NULL なら現在のプロセスを作成するために使われたファイルのパスを取得します。
//!
//! @return 実行ファイルのフルパスを返します。
//!
std::string RGetModuleFileName(HMODULE hModule);

//=============================================================================
// Windows のプロセス関連の関数です。
//=============================================================================

#ifdef __PIWin__
//! @brief Windows のプロセスを実行します。
//!
//! @param[out] pStatus エラーなら 0、成功なら 0 以外を格納します。
//!                     終了コードでエラーを判別できない場合に使用します。
//!                     NULL なら何も格納しません。
//! @param[in] cmd コマンド文字列です。
//! @param[in] waitsExit コマンドの終了を待つなら true を指定します。
//! @param[in] showWindow ウィンドウの表示指定です。
//!                       SW_SHOW, SW_HIDE, SW_SHOWNORMAL, SW_MAXIMIZE, SW_MINIMIZE
//!
//! @return コマンドの終了コードを返します。
//!
int RExecProcess(
    int* pStatus,
    const char* cmd,
    const bool waitsExit,
    int showWindow = SW_HIDE
);
#endif

//=============================================================================
// Windows のレジストリ関連の関数です。
//=============================================================================

#ifdef __PIWin__
//! @brief Windows の DWORD 型のレジストリ値を取得します。
//!
//! @param[in] hParentKey 開いている親キーのハンドルです。
//! @param[in] lpSubKey 開くべきサブキーの名前です。
//! @param[in] lpValueName 取得するレジストリエントリ名です。
//! @param[in] defaultValue 取得できなかった場合に返す値です。
//!
DWORD RGetWinRegDword(
    HKEY hParentKey,
    const char* lpSubKey,
    const char* lpValueName,
    const DWORD defaultValue
);

//! @brief Windows の文字列型のレジストリ値を取得します。
//!
//! @param[in] hParentKey 開いている親キーのハンドルです。
//! @param[in] lpSubKey 開くべきサブキーの名前です。
//! @param[in] lpValueName 取得するレジストリエントリ名です。
//! @param[in] defaultValue 取得できなかった場合に返す値です。
//!
std::string RGetWinRegString(
    HKEY hParentKey,
    const char* lpSubKey,
    const char* lpValueName,
    const char* defaultValue = ""
);
#endif

//=============================================================================
//! @brief XML 要素のクラスです。
//=============================================================================
class RXMLElement
{
public:
    typedef std::map<std::string, std::string> AttrMap;
    typedef void (*ErrorFunc)(const char* message, void* pUserData);

    std::string name;
    AttrMap attributes;
    std::string text;
    std::vector<RXMLElement> nodes;

    ErrorFunc m_ErrorFunc; //!< エラー表示関数です。
    void* m_pErrorUserData; //!< エラー表示関数に渡すユーザーデータのポインタです。

private:
    //! @brief 属性を解析してノードに記録します。
    //!
    //! @param[in] src 要素の文字列です。
    //! @param[in] end 要素の終端の文字です。
    //!
    //! @return 要素の終端までの文字数を返します。
    //!
    int LoadAttributes(const char* src, const char end);

    //! @brief エラーを表示します。
    //!
    //! @param[in] message エラーメッセージです。
    //!
    void DisplayError(const std::string& message) const;

public:
    //! コンストラクタです。
    RXMLElement(ErrorFunc errorFunc = NULL, void* pErrorUserData = NULL)
    : m_ErrorFunc(errorFunc),
      m_pErrorUserData(pErrorUserData)
    {
    }

    //! コピーコンストラクタです。
    RXMLElement(const RXMLElement &r)
    {
        operator=(r);
    }

    //! 再初期化します。
    void Clear()
    {
        name = "";
        text = "";
        attributes.clear();
        nodes.clear();
    }

    //! 代入演算子です。
    RXMLElement& operator=(const RXMLElement& r);

    //! @brief 下層ノードを新規作成します。
    //!
    //! @return 新規作成した要素のポインタを返します。
    //!
    RXMLElement* CreateElement()
    {
        nodes.push_back(RXMLElement(m_ErrorFunc, m_pErrorUserData));
        return &nodes[nodes.size() - 1];
    }

    //! @brief 指定のパスに該当する最初の要素のポインタを取得します。
    //!
    //! @param[in] path パスです。
    //! @param[in] showError 要素が見つからなかった場合に標準エラーに表示するなら true を指定します。
    //!
    //! @return 要素のポインタを返します。見つからなかった場合は NULL を返します。
    //!
    const RXMLElement* FindElement(const std::string& path, const bool showError = true) const
    {
        std::vector<const RXMLElement*> list;
        FindElements(list, path, showError, 1);
        return (list.size() > 0) ? list[0] : NULL;
    }

    //! @brief 指定のパスに該当する要素を指定個数分検索します。
    //!
    //! @param[out] list 要素のポインタ配列です。
    //! @param[in] path パスです。
    //! @param[in] maxCount 取得する最大要素数です。
    //! @param[in] showError 要素が見つからなかった場合に標準エラーに表示するなら true を指定します。
    //!
    void FindElements(
        std::vector<const RXMLElement*>& list,
        const std::string& path,
        const bool showError = true,
        const int maxCount = 0x7fffffff
    ) const;

    //! @brief 属性の値を取得します。
    //!
    //! @param[in] tag 属性名です。
    //! @param[in] alt 属性が見つからなかった場合に返す値です。
    //! @param[in] showError 要素が見つからなかった場合に標準エラーに表示するなら true を指定します。
    //!
    //! @return 属性の値を返します。
    //!
    std::string GetAttribute(
        const std::string& tag,
        const char* alt = "",
        const bool showError = true
    ) const
    {
        AttrMap::const_iterator it = attributes.find(tag);
        if (showError && it == attributes.end())
        {
            DisplayError("Attribute is not found: " + tag + " in <" + name + ">");
        }
        return (it != attributes.end()) ? it->second : alt;
    }

    //! @brief XML 文書を解析してツリーを構築します（ASCII テキスト限定）。
    //!
    //! @param[in,out] decl 宣言要素です。宣言要素の属性を取得しないなら NULL を指定します。
    //! @param[in] xmlDoc XML 文書です。
    //!
    void LoadDocument(RXMLElement* decl, const std::string& xmlDoc);

    //! @brief XML 文書を取得します。
    //!        要素、属性、平データ、下層ノードの内容を再帰的に文字列に出力します。
    //!
    //! @param[in] tabCount 要素のインデントに必要なタブの数です。
    //! @param[in] lf 改行コードです。
    //!
    //! @return XML 文書の文字列を返します。
    //!
    std::string GetDocument(const int tabCount = 0, const char* lf = "\n") const;

    //! @brief エラー表示関数を設定します。
    //!
    //! @param[in] func エラー表示関数です。
    //! @param[in] pUserData エラー表示関数に渡すユーザーデータのポインタです。
    //!
    void SetErrorFunc(ErrorFunc func, void* pUserData)
    {
        m_ErrorFunc = func;
        m_pErrorUserData = pUserData;
    }
};

//=============================================================================
//! @brief 時間計測のクラスです。
//=============================================================================
class RTimeMeasure
{
protected:
    std::clock_t m_StartClock; //!< 計測開始時のクロックです。

public:
    //! @brief コンストラクタです。
    RTimeMeasure()
    {
        Reset();
    }

    //! 計測開始時間を現在のクロックに設定します。
    void Reset()
    {
        m_StartClock = std::clock();
    }

    //! 計測開始時からの秒数を取得します。
    float GetSec() const
    {
        return static_cast<float>(std::clock() - m_StartClock) / CLOCKS_PER_SEC;
    }

    //! 計測開始時からのミリ秒数を取得します。
    float GetMilliSec() const
    {
        return GetSec() * 1000.0f;
    }
};

//=============================================================================
// Nintendo Photoshop プラグイン関連の関数です。
//=============================================================================

//! @brief Nintendo Photoshop プラグインのルートフォルダーのパスを返します。
//!
std::string RGetNintendoPhotoshopRootPath();

//! @brief Nintendo Photoshop プラグインが使用するグラフィックスツールフォルダーのパスを返します。
//!
std::string RGetNintendoPhotoshopGraphicsToolsPath();

//! @brief Nintendo Photoshop プラグインが使用する 3D ツールフォルダーのパスを返します。
//!
std::string RGetNintendoPhotoshop3dToolsPath();

//! @brief Nintendo Photoshop プラグインのコンフィグフォルダーのパスを返します。
//!
//! @param[in] getsCofigEnv 専用の環境変数による指定を取得するなら true です。
//!
//! @return コンフィグフォルダーのパスを返します。
//!
std::string RGetConfigFolderPath(const bool getsCofigEnv);

//! @brief コンフィグ名からコンフィグファイルのパスを取得します。
//!
//! @param[in] configName コンフィグ名です。
//! @param[in] configFolderPath コンフィグフォルダーのパスです。
//!
//! @return コンフィグファイルのパスを返します。
//!
std::string RGetConfigPathFromConfigName(
    const std::string& configName,
    const std::string& configFolderPath
);

//! @brief コンフィグ名一覧を取得します。
//!
//! @param[out] allConfigNames コンフィグ名一覧を格納します。
//!
void RGetAllConfigNames(RStringArray& allConfigNames);

//! @brief コンフィグファイルをリードして、内容からコメントを除いた文字列を取得します。
//!
//! @param[in] configPath コンフィグファイルのパスです。
//!
//! @return コンフィグファイルの内容からコメントを除いた文字列を返します。
//!         コンフィグファイルが存在しなければ空文字を返します。
//!
std::string RReadConfigFile(const std::string& configPath);

//! @brief コンフィグファイルから変数名を検索します。
//!
//! @param[in] configStr コンフィグファイルをリードした文字列です。
//! @param[in] name 変数名です。
//!
//! @return 変数名の先頭位置（見つからなければ std::string::npos）を返します。
//!
size_t RFindConfigVariable(const std::string& configStr, const std::string& name);

//! @brief コンフィグファイルからブーリアン型変数の値を取得します。
//!
//! @param[in] configStr コンフィグファイルをリードした文字列です。
//! @param[in] name 変数名です。
//! @param[in] altValue 変数が定義されていない場合や定義が不正な場合に返す値です。
//!
//! @return ブーリアン型変数の値を返します。
//!
bool RGetConfigBooleanValue(
    const std::string& configStr,
    const std::string& name,
    const bool altValue
);

//! @brief コンフィグファイルから文字列型変数の値を取得します。
//!
//! @param[in] configStr コンフィグファイルをリードした文字列です。
//! @param[in] name 変数名です。
//! @param[in] altValue 変数が定義されていない場合や定義が不正な場合に返す値です。
//!
//! @return 文字列型変数の値を返します。
//!
std::string RGetConfigStringValue(
    const std::string& configStr,
    const std::string& name,
    const std::string& altValue
);

//! @brief コンフィグファイルのオブジェクト定義から値の配列を取得します。
//!
//! @param[out] pValues 値の配列へのポインタです。
//!                     オブジェクトの場合 { key0, value0, key1, value1, ... } の形式で格納します。
//! @param[in] configStr コンフィグファイルをリードした文字列です。
//! @param[in] varIdx 変数名の先頭位置です。
//! @param[in] isArray 配列なら true、オブジェクトなら false を指定します。
//!
//! @return 値の数を返します。
//!
int RGetConfigObjectValues(
    RStringArray* pValues,
    const std::string& configStr,
    const size_t varIdx,
    const bool isArray
);

//! @brief コンフィグファイルから重み付け圧縮有効フラグを取得します。
//!
//! @param[in] configStr コンフィグファイルをリードした文字列です。
//!
//! @return 重み付け圧縮有効フラグを返します。
//!
bool RGetEnablesWeightedCompress(const std::string& configStr);

//=============================================================================
// 中間ファイル関連の関数です。
//=============================================================================

//! @brief 中間ファイルバージョンを整数（major * 100 + minor）で返します。
//!        1.0.2 -> 100
//!        1.3.4 -> 103
//!
//! @param[in] value version 属性の値です。
//!
//! @return 中間ファイルバージョンを整数で返します。
//!
int RGet3dIntermediateFileVersionAsInt(const std::string& value);

//! @brief 中間ファイルのプリセット名に使用できる文字列なら true を返します。
//!
bool RIsValidPresetNameString(const std::string& name);

//! @brief 中間ファイルの先頭からバイナリーデータまでのオフセットを取得します。
//!
//! @param[in] fileBuf 中間ファイルをリードしたメモリーのポインタです。
//! @param[in] fileSize 中間ファイルのサイズ（バイト数）です。
//!
//! @return バイナリーデータまでのオフセットを返します。
//!         バイナリーデータがない場合は fileSize を返します。
//!
size_t RGetBinaryOffset(const void* fileBuf, const size_t fileSize);

//=============================================================================
// @brief バイナリーデータ列のクラスです。
//=============================================================================
class RBinDataStream
{
public:
    //! 値の型を表す列挙型です。
    enum Type { FLOAT, INT, BYTE, STRING, WSTRING };

    char m_Identifier[8]; //!< 識別子です。
    int m_Type; //!< 値の型です。
    int m_Count; //!< 値の個数です。
    int m_Column; //!< アスキー出力時の列数です。
    int m_Size; //!< データサイズ（バイト数）です。文字列の場合は終端文字を含んだバイト数です。
    int m_Padding[2]; //!< 実データの先頭アドレスを 32 バイトアライメントにするためのパディングです。
    int m_Data[1]; //!< 実データです。実際は可変長です。

public:
    //! @brief バイナリーデータ列へのポインタを取得します（static 関数）。
    //!
    //! @param[in] binTop 中間ファイルのバイナリー部分の先頭アドレスです。
    //! @param[in] streamIdx バイナリーデータ列のインデックスです。
    //!
    //! @return バイナリーデータ列へのポインタを返します。
    //!         インデックスが不正な場合は nullptr を返します。
    //!
    static const RBinDataStream* Get(const void* binTop, const int streamIdx);
};

//=============================================================================
// @brief ユーザーデータのクラスです。
//=============================================================================
class RUserData
{
public:
    //! 値の型を表す列挙型です。
    enum Type { FLOAT, INT, STRING, WSTRING, STREAM };

    std::string m_Name; //!< 名前です。
    Type m_Type; //!< 値の型です。
    int m_NumberValueCount; //!< 浮動小数点数値または整数値の値の数です。
    std::string m_NumberValuesText; //!< 浮動小数点数値または整数値の値の配列の中間ファイル上の文字列です。
    RStringArray m_StringValues; //!< 文字列値（Shift-JIS）の配列です。
    RUCharArray m_StreamValues; //!< ストリーム型値の配列です。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] userDataElem <user_data> 要素です。
    //! @param[in] streamsElem <stream_array> 要素です（バイナリー形式なら nullptr を指定します）。
    //! @param[in] binTop 中間ファイルのバイナリー部分の先頭アドレスです（アスキー形式なら nullptr を指定します）。
    //!
    RUserData(
        const RXMLElement* userDataElem,
        const RXMLElement* streamsElem,
        const void* binTop
    );

    //! @brief コンストラクタです。
    //!
    //! @param[in] name 名前です。
    //! @param[in] type 値の型です。
    //!
    RUserData(const std::string& name, const Type type)
    : m_Name(name),
      m_Type(type),
      m_NumberValueCount(0)
    {
    }
};

//! @brief ユーザーデータ配列の定義です。
typedef std::vector<RUserData> RUserDataArray;

//=============================================================================
//! @brief ファイルバッファのクラスです。
//=============================================================================
class RFileBuf
{
protected:
    std::string m_FilePath; //!< ファイルのパスです。
    uint8_t* m_FileBuf; //!< ファイルバッファです。
    size_t m_FileSize; //!< ファイルサイズです。

public:
    //! @biref コンストラクタです。
    //!        ファイルをリードします。
    //!
    //! @param[in] path ファイルのパスです。
    //!
    explicit RFileBuf(const std::string& path)
    : m_FileBuf(NULL)
    {
        Read(path);
    }

    //! @biref コンストラクタです。
    //!        ファイルをリードします。
    //!
    //! @param[in] path ファイルのパスです。
    //!
    explicit RFileBuf(const std::wstring& path)
    : m_FileBuf(NULL)
    {
        Read(RGetShiftJisFromUnicode(path));
    }

    //! デストラクタです。
    virtual ~RFileBuf()
    {
        //std::cerr << "~RFileBuf(): " << m_FilePath << std::endl;
        RFreeAndClearArray(m_FileBuf);
    }

    //! メモリーを解放します。
    void Clear()
    {
        RFreeAndClearArray(m_FileBuf);
        m_FilePath.clear();
        m_FileSize = 0;
    }

    //! @biref ファイルをリードします。
    //!
    //! @param[in] path ファイルのパスです。
    //!
    //! @return リード成功なら true を返します。
    //!
    bool Read(const std::string& path);

    //! リード成功なら true を返します。
    NN_IMPLICIT operator bool() const
    {
        return (m_FileBuf != NULL);
    }

    //! ファイルのパスを返します。
    std::string GetPath() const { return m_FilePath; };

    //! ファイルバッファを返します。
    const uint8_t* GetBuf() const { return m_FileBuf; };

    //! ファイルバッファを返します（非 const 版）。
    uint8_t* GetBuf() { return m_FileBuf; };

    //! ファイルサイズを返します。
    size_t GetSize() const { return m_FileSize; };

    //! 文字コードが UTF-8 なら true を返します。
    bool IsUtf8() const
    {
        return (m_FileSize >= 3 &&
            m_FileBuf[0] == 0xef &&
            m_FileBuf[1] == 0xbb &&
            m_FileBuf[2] == 0xbf);
    }

    //! @brief ファイルバッファの要素の参照を返す添え字演算子です。
    //!
    //! @param[in] i 要素のインデックスです。
    //!
    //! @return 要素の参照を返します。
    //!
    const uint8_t& operator[](size_t i) const { return m_FileBuf[i]; }

    //! @brief ファイルバッファの要素の参照を返す添え字演算子です（非 const 版）。
    //!
    //! @param[in] i 要素のインデックスです。
    //!
    //! @return 要素の参照を返します。
    //!
    uint8_t& operator[](size_t i) { return m_FileBuf[i]; }

private:
    RFileBuf(const RFileBuf& other);
    RFileBuf& operator=(const RFileBuf& other);
};

//=============================================================================
// テクスチャー関連の関数です。
//=============================================================================

//! @brief 中間ファイルの文字列からリニア変換フラグを取得します。
//!
//! @param[in] str リニア変換フラグを表す文字列です。
//!
//! @return リニア変換フラグを返します。
//!
int RGetLinearFlagFromString(const std::string& str);

//! @brief オプション文字列からリニア変換フラグを取得します。
//!
//! @param[in] str リニア変換フラグを表すオプション文字列です。
//!
//! @return リニア変換フラグを返します。
//!
int RGetLinearFlagFromOptString(const std::string& str);

//! @brief リニア変換フラグからオプション文字列を取得します。
//!
//! @param[in] flags リニア変換フラグです。
//!
//! @return オプション文字列を返します。
//!
std::string RGetOptStringFromLinearFlag(const int linearFlag);

//! @brief テクスチャーの次元が配列なら true を返します。
//!
//! @param[in] dimension テクスチャーの次元です。
//!
//! @return テクスチャーの次元が配列なら true を返します。
//!
inline bool RIsArrayDimension(const FtxDimension dimension)
{
    return (
        dimension == FtxDimension_1dArray            ||
        dimension == FtxDimension_2dArray            ||
        dimension == FtxDimension_2dMultisampleArray ||
        dimension == FtxDimension_CubeMapArray);
}

//! @brief テクスチャーの次元がキューブマップまたはキューブマップ配列なら true を返します。
//!
//! @param[in] dimension テクスチャーの次元です。
//!
//! @return テクスチャーの次元がキューブマップまたはキューブマップ配列なら true を返します。
//!
inline bool RIsCubMapDimension(const FtxDimension dimension)
{
    return (
        dimension == FtxDimension_CubeMap      ||
        dimension == FtxDimension_CubeMapArray);
}

//! @brief 中間ファイルの文字列からテクスチャーの次元を取得します。
//!
//! @param[in] str テクスチャーの次元を表す文字列です。
//!
//! @return テクスチャーの次元を返します。
//!
FtxDimension RGetTextureDimensionFromString(const std::string& str);

//! @brief テクスチャーの次元から中間ファイルの文字列を取得します。
//!
//! @param[in] dimension テクスチャーの次元です。
//!
//! @return テクスチャーの次元を表す文字列を返します。
//!
std::string RGetTextureDimensionString(const FtxDimension dimension);

//! @brief 中間ファイルの文字列からテクスチャーフォーマットを取得します。
//!
//! @param[in] str テクスチャーフォーマットを表す文字列です。
//!
//! @return テクスチャーフォーマットを返します。
//!
FtxFormat RGetTextureFormatFromString(const std::string& str);

//! @brief テクスチャーフォーマットから中間ファイルの文字列を取得します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットを表す文字列を返します。
//!
std::string RGetTextureFormatString(const FtxFormat format);

//! @brief テクスチャーフォーマットが BC1/BC2/BC3 フォーマットなら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが BC1/BC2/BC3 フォーマットなら true を返します。
//!
bool RIsBc123TextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが BC6/BC7 フォーマットなら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが BC6/BC7 フォーマットなら true を返します。
//!
bool RIsBc67TextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが BC フォーマットなら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが BC フォーマットなら true を返します。
//!
bool RIsBcTextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが 1 成分あたり 16bit の unorm / snorm なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが 1 成分あたり 16bit の unorm / snorm なら true を返します。
//!
bool RIsUnormSnorm16TextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが 1 成分あたり 16bit の uint / sint なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが 1 成分あたり 16bit の uint / sint なら true を返します。
//!
bool RIsUintSint16TextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが 1 成分あたり 32bit の uint / sint なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが 1 成分あたり 32bit の uint / sint なら true を返します。
//!
bool RIsUintSint32TextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが浮動小数点数型なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが浮動小数点数型なら true を返します。
//!
bool RIsFloatTextureFormat(const FtxFormat format);

//! @brief 実数型として扱うテクスチャーフォーマットなら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return 実数型として扱うテクスチャーフォーマットなら true を返します。
//!
bool RIsRealNumberTextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが整数型なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが整数型なら true を返します。
//!
bool RIsIntTextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが符号ありなら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが符号ありなら true を返します。
//!
bool RIsSignedTextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが sRGB フェッチなら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが sRGB フェッチなら true を返します。
//!
bool RIsSrgbFetchTextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが ETC1 なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが ETC1 なら true を返します。
//!
bool RIsEtc1TextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが ETC2 なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが ETC2 なら true を返します。
//!
bool RIsEtc2TextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが ETC1 または ETC2 なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが ETC1 または ETC2 なら true を返します。
//!
bool RIsEtcTextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが EAC なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが EAC なら true を返します。
//!
bool RIsEacTextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが PVRTC1 なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが PVRTC1 なら true を返します。
//!
bool RIsPvrtc1TextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが PVRTC2 なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが PVRTC2 なら true を返します。
//!
bool RIsPvrtc2TextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが PVRTC1 または PVRTC2 なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが PVRTC1 または PVRTC2 なら true を返します。
//!
bool RIsPvrtcTextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが ASTC なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが ASTC なら true を返します。
//!
bool RIsAstcTextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットが圧縮形式なら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return テクスチャーフォーマットが圧縮形式なら true を返します。
//!
bool RIsCompressedTextureFormat(const FtxFormat format);

//! @brief テクスチャーフォーマットから成分数を取得します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return 成分数を返します。
//!
int RGetComponentCount(const FtxFormat format);

//! @brief テクスチャーフォーマットの 1 ピクセルのビット数を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return 1 ピクセルのビット数を返します。
//!         ASTC フォーマットの場合は最大ビット数（8）を返します。
//!
int RGetBitsPerPixel(const FtxFormat format);

//! @brief テクスチャーフォーマットの 1 ピクセルのビット数を表す文字列を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return 1 ピクセルのビット数を表す文字列を返します。
//!
std::string RGetBitsPerPixelString(const FtxFormat format);

//! @brief 指定した型のデータから変換可能なテクスチャーフォーマットなら true を返します。
//!
//! @param[in] format テクスチャーフォーマットです。
//! @param[in] isFromRealNumber データが実数型なら true、整数型なら false を指定します。
//!
//! @return 指定した型のデータから変換可能なテクスチャーフォーマットなら true を返します。
//!
bool RIsConvertibleTextureFormat(const FtxFormat format, const bool isFromRealNumber);

//! @brief 中間ファイルの文字列からミップマップ生成フィルタを取得します。
//!
//! @param[in] str ミップマップ生成フィルタを表す文字列です。
//! @param[in] linearIfInvalid 文字列が不正な場合にリニアを返すなら true です。
//!
//! @return ミップマップ生成フィルタを返します。
//!
RMipGenFilter::MipGenFilter RGetMipGenFilterFromString(const std::string& str, const bool linearIfInvalid);

//! @brief ミップマップ生成フィルタから中間ファイルの文字列を取得します。
//!
//! @param[in] mipGenFilter ミップマップ生成フィルタです。
//!
//! @return ミップマップ生成フィルタを表す文字列を返します。
//!
std::string RGetMipGenFilterString(const RMipGenFilter::MipGenFilter mipGenFilter);

//! @brief 中間ファイルの文字列から成分選択を取得します。
//!
//! @param[in] str 成分選択を表す文字列です。
//!
//! @return 成分選択を返します。
//!
FtxCompSel RGetCompSelFromString(const std::string& str);

//! @brief 成分選択から中間ファイルの文字列を取得します。
//!
//! @param[in] compSel 成分選択です。
//!
//! @return 成分選択を表す文字列を返します。
//!
std::string RGetCompSelString(const FtxCompSel compSel);

//! @brief オプション文字列から成分選択を取得します。
//!
//! @param[in] str 成分選択を表すオプション文字列です。
//!
//! @return 成分選択を返します。
//!
FtxCompSel RGetCompSelFromOptionString(const std::string& str);

//! @brief 成分選択からオプション文字列を取得します。
//!
//! @param[in] compSel 成分選択です。
//!
//! @return 成分選択を表すオプション文字列を返します。
//!
std::string RGetCompSelOptionString(const FtxCompSel compSel);

//! @brief テクスチャーフォーマットに対応したデフォルトの成分選択を取得します。
//!
//! @param[in] format テクスチャーフォーマットです。
//! @param[in] hint ヒント情報です。
//!
//! @return 成分選択を返します。
//!
FtxCompSel RGetDefaultCompSel(const FtxFormat format, const std::string& hint);

//! @brief テクスチャーフォーマットに対する最小の幅と高さを取得します。
//!
//! @param[out] minW 最小の幅を格納します。
//! @param[out] minW 最小の高さを格納します。
//! @param[in] format テクスチャーフォーマットです。
//!
void RGetMinimumWH(int& minW, int& minH, const FtxFormat format);

//! @brief テクスチャーフォーマットに対応したデフォルトのリニア変換フラグを取得します。
//!
//! @param[in] format テクスチャーフォーマットです。
//!
//! @return リニア変換フラグを返します。
//!
int RGetDefaultLinearFlag(const FtxFormat format);

//! @brief カラーから法線を取得します。
//!
//! @param[out] n 法線です（長さ 3 の配列を指定します）。
//! @param[in] c カラーです（長さ 3 の配列を指定します）。
//!
inline void RGetNormalFromColor(float* n, const uint8_t* c)
{
    n[0] = (c[0] == 0) ? -1.0f : (c[0] - 128) / 127.0f;
    n[1] = (c[1] == 0) ? -1.0f : (c[1] - 128) / 127.0f;
    n[2] = (c[2] == 0) ? -1.0f : (c[2] - 128) / 127.0f;
}

//! @brief 法線からカラーを取得します。
//!
//! @param[out] c カラーです（長さ 3 の配列を指定します）。
//! @param[in] n 法線です（長さ 3 の配列を指定します）。
//!
inline void RGetColorFromNormal(uint8_t* c, const float* n)
{
    for (int iXyz = 0; iXyz < 3; ++iXyz)
    {
        const float nn = n[iXyz];
        if (nn <= -1.0f)
        {
            c[iXyz] = 0x01;
        }
        else if (nn >= 1.0f)
        {
            c[iXyz] = 0xff;
        }
        else
        {
            c[iXyz] = static_cast<uint8_t>(RRound(nn * 127.0f) + 128);
        }
    }
}

//! @brief 法線を正規化します。
//!
//! @param[in,out] n 法線です（長さ 3 の配列を指定します）。
//! @param[in] isPositiveZ Z 成分が 0 以上になるように調整するなら true を指定します。
//!
inline void RNormalizeNormal(float* n, const bool isPositiveZ)
{
    if (isPositiveZ && n[2] < 0.0f)
    {
        n[2] = 0.0f;
    }

    const float len = sqrtf(n[0] * n[0] + n[1] * n[1] + n[2] * n[2]);
    if (len != 0.0f)
    {
        const float invLen = 1.0f / len;
        n[0] *= invLen;
        n[1] *= invLen;
        n[2] *= invLen;
    }
}

//=============================================================================
//! @brief プログレスのクラスです。
//=============================================================================
class RProgress
{
public:
    int m_Count; //!< 現在の進行カウントです。
    int m_Total; //!< 処理全体のカウント数です
    int m_BaseCount; //!< 現在の処理を開始する前の m_Count を保存します。
    int m_CurStep; //!< 現在の処理が終了した際の m_Count の増分です。
    ProgressProc m_Proc; //!< プログレス関数の関数ポインタです。
    TestAbortProc m_Abort; //!< 中断判定関数の関数ポインタです。

public:
    //! @brief コンストラクタです。
    //!
    //! @param[in] total 処理全体のカウント数です。
    //! @param[in] proc プログレス関数の関数ポインタです。
    //! @param[in] abort 中断判定関数の関数ポインタです。
    //!
    RProgress(const int total = 1, const ProgressProc proc = NULL, const TestAbortProc abort = NULL)
    : m_Proc(proc),
      m_Abort(abort)
    {
        Init(total);
    }

    //! デストラクタです。
    virtual ~RProgress()
    {
    }

    //! @brief 初期化します。現在の進行カウントを 0 にします。
    //!
    //! @param[in] total 処理全体のカウント数です。
    //!
    virtual void Init(const int total)
    {
        m_Count = m_BaseCount = 0;
        m_Total = total;
        m_CurStep = 1;
    }

    //! @brief 更新します。
    //!
    //! @param[in] step 現在の進行カウントに加算する値です。
    //!
    //! @return 処理を続行するなら true、中断するなら false を返します。
    //!
    virtual bool Update(const int step = 1)
    {
        m_Count += step;
        if (m_Proc != NULL)
        {
            (*m_Proc)(m_Count, m_Total);
        }
        if (m_Abort != NULL && (*m_Abort)())
        {
            return false;
        }
        return true;
    }
};

//=============================================================================
//! @brief アバウトダイアログのパラメータのクラスです。
//=============================================================================
class RAboutParam
{
public:
    char* m_TitleText;
    char* m_MessageText;
    short m_AboutTextId;
    short m_HelpIndexId;
    char* m_HelpIndexUrl;
};

//=============================================================================
//! @brief 3D テクスチャーコンバーターのクラスです。
//=============================================================================
class RTexCvtr
{
public:
    GetCvtrVersionFunc GetCvtrVersion;
    SetOptionsFunc SetOptions;
    ClearFunc Clear;
    ReadBitmapDataFunc ReadBitmapData;
    ConvertToDataFunc ConvertToData;
    GetErrorShiftJisStringFunc GetErrorShiftJisString;
    SetCreateInfoFunc SetCreateInfo;
    SetUserDataFunc SetUserData;
    SetCommentLabelFunc SetCommentLabel;
    SetCommentColorFunc SetCommentColor;
    SetToolDataFunc SetToolData;
    SetUserToolDataFunc SetUserToolData;

protected:
    std::string m_DllPath; //!< 3D テクスチャーコンバーターの DLL ファイルのパスです。
    HINSTANCE m_hDll; //!< 3D テクスチャーコンバーターの DLL のインスタンスハンドルです。

public:
    //! @biref コンストラクターです。
    RTexCvtr()
    : m_hDll(nullptr)
    {
    }

    //! @biref デストラクターです。
    virtual ~RTexCvtr()
    {
        Finalize();
    }

    //! @brief 初期化します（DLL をロードします）。
    //!
    //! @param[in] globals グローバルデータです。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool Initialize(GPtr globals);

    //! @biref 終了します（DLL を解放します）。
    void Finalize();

    //! @biref nvtt の DLL が Photoshop 終了までアンロードされないようにします。
    void PinNvttDll() const;
};

//=============================================================================
//! @brief テクスチャーエンコーダーのクラスです。
//=============================================================================
class RTexEncoder
{
public:
    nn::gfx::tool::texenc::GetDataSizeFunction GetDataSize; //!< フォーマットに応じたデータサイズ（バイト数）を取得する関数です。
    nn::gfx::tool::texenc::ConvertFormatFunction ConvertFormat; //!< 画像のフォーマットを変換する関数です。

public:
    //! @biref コンストラクターです。
    RTexEncoder()
    : m_hDll(nullptr)
    {
    }

    //! @biref デストラクターです。
    virtual ~RTexEncoder()
    {
        Finalize();
    }

    //! @brief 初期化します（DLL をロードします）。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool Initialize();

    //! @biref 終了します（DLL を解放します）。
    void Finalize();


    //! @biref エンコーダ用のイメージの次元を取得します（static 関数）。
    //!
    //! @param[in] dimension 次元です。
    //!
    //! @return エンコーダ用のイメージの次元を返します。
    //!
    static nn::gfx::tool::texenc::ImageDimension GetImageDimension(const FtxDimension dimension);

protected:
    std::string m_DllPath; //!< テクスチャーエンコーダーの DLL ファイルのパスです。
    HINSTANCE m_hDll; //!< テクスチャーエンコーダーの DLL のインスタンスハンドルです。
};

//=============================================================================
//! @brief 画像データのクラスです。
//=============================================================================
class RImage
{
public:
    static const int WidthHeightMin = 1; //!< 画像の幅と高さの最小値です。
    static const int MipCountMax = 15; //!< ミップマップのレベル数の最大値です。

    static const int CUBE_HC_COUNT_H = 4; //!< 水平十字キューブマップの水平方向のフェース数です。
    static const int CUBE_HC_COUNT_V = 3; //!< 水平十字キューブマップの垂直方向のフェース数です。
    static const int CUBE_VC_COUNT_H = 3; //!< 垂直十字キューブマップの水平方向のフェース数です。
    static const int CUBE_VC_COUNT_V = 4; //!< 垂直十字キューブマップの垂直方向のフェース数です。

    //! @brief 透明モードを表す列挙型です。
    enum TransparencyMode
    {
        OPA,    //!< 不透明です（アルファがすべて 255）。
        MASK,   //!< 抜きありです（アルファが 0 または 255）。
        XLU,    //!< 半透明です（1 以上 254 以下のアルファが存在）。
    };

    //! @brief キューブマップのフェースを表す列挙型です。
    enum CubeFace
    {
        CUBE_FACE_PX,   //!< +X 面です。
        CUBE_FACE_NX,   //!< -X 面です。
        CUBE_FACE_PY,   //!< +Y 面です。
        CUBE_FACE_NY,   //!< -Y 面です。
        CUBE_FACE_PZ,   //!< +Z 面です。
        CUBE_FACE_NZ,   //!< -Z 面です。
        CUBE_FACE_COUNT //!< フェースの最大数です。
    };

    //! @brief PSD ファイルのイメージモードを表す列挙型です（5 と 6 は使用されていないので注意）。
    enum PsdImageMode
    {
        PSD_IM_BITMAP    = 0,
        PSD_IM_GRAYSCALE = 1,
        PSD_IM_INDEXED   = 2,
        PSD_IM_RGB       = 3,
        PSD_IM_CMYK      = 4,
        PSD_IM_MULTI     = 7,
        PSD_IM_DUO       = 8,
        PSD_IM_LAB       = 9
    };

    static const int PSD_COLOR_TABLE_MAX = 256; //!< PSD ファイルのカラーテーブルの最大カラー数です。

protected:
    std::string m_FilePath; //!< ファイルパスです。
    int m_ImageW; //!< 幅です。
    int m_ImageH; //!< 高さです。
    int m_ImageD; //!< 奥行きです。
    bool m_HasAlpha; //!< アルファ成分を持つなら true です。
    void* m_pImageData; //!< イメージデータへのポインタです。RGBA の順に左上から右下へ格納されます。
    std::string m_ErrorString; //!< エラーの文字列です。

public:
    //! コンストラクタです。
    RImage()
    : m_ImageW(0),
      m_ImageH(0),
      m_ImageD(1),
      m_HasAlpha(false),
      m_pImageData(NULL)
    {
    }

    //! メモリーを解放します。
    void FreeMemory()
    {
        RFreeAndClearArray(m_pImageData);
    }

    //! デストラクタです。
    virtual ~RImage() { FreeMemory(); }

    //! @brief ファイルから画像データをリードします。
    //!        現在 TGA ファイルに対応しています。
    //!        NPS_READ_PSD_ENABLE が定義されていれば PSD ファイルもリード可能です。
    //!
    //! @param[in] filePath ファイルのパスです。
    //!
    //! @return 処理成功なら true を返します。
    //!
    bool ReadFile(const std::string& filePath);

    //-----------------------------------------------------------------------------
    // アクセサ

    //! 画像ファイルのパスを取得します。
    std::string GetFilePath() const { return m_FilePath; }

    //! 画像ファイルのパスを設定します。
    void SetFilePath(const std::string& path) { m_FilePath = path; }

    //! 幅を返します。
    int GetImageW() const { return m_ImageW; }

    //! 高さを返します。
    int GetImageH() const { return m_ImageH; }

    //! 奥行きを返します。
    int GetImageD() const { return m_ImageD; }

    //! アルファ成分が存在するなら true を返します。
    bool HasAlpha() const { return m_HasAlpha; }

    //! イメージデータへのポインタを返します。
    const void* GetImagePtr() const { return m_pImageData; }

    //! イメージデータへのポインタを返します。
    void* GetImagePtr() { return m_pImageData; }

    //! エラーの文字列を返します。
    std::string GetErrorString() const { return m_ErrorString; }

    //-----------------------------------------------------------------------------
    // static メンバ関数

    //! @brief 水平十字キューブマップの幅と高さなら true を返します。
    //!
    //! @param[in] imageW 画像の幅です。
    //! @param[in] imageH 画像の高さです。
    //!
    //! @return 水平十字キューブマップの幅と高さなら true を返します。
    //!
    static bool IsCubeHCWH(const int imageW, const int imageH);

    //! @brief 垂直十字キューブマップの幅と高さなら true を返します。
    //!
    //! @param[in] imageW 画像の幅です。
    //! @param[in] imageH 画像の高さです。
    //!
    //! @return 垂直十字キューブマップの幅と高さなら true を返します。
    //!
    static bool IsCubeVCWH(const int imageW, const int imageH);

    //! @brief 水平十字キューブマップのフェースの位置を取得します。
    //!
    //! @param[in] faceIdx フェースのインデックスです。
    //!
    //! @return X 位置（0 ～ 3）と Y 位置（0 ～ 2）を格納した長さ 2 の配列へのポインタを返します。
    //!
    static const int* GetCubeHCFacePos(const int faceIdx);

    //! @brief 垂直十字キューブマップのフェースの位置を取得します。
    //!
    //! @param[in] faceIdx フェースのインデックスです。
    //!
    //! @return X 位置（0 ～ 2）と Y 位置（0 ～ 3）を格納した長さ 2 の配列へのポインタを返します。
    //!
    static const int* GetCubeVCFacePos(const int faceIdx);

    //! @brief 幅と高さからミップマップの最大レベル数を返します。
    //!
    //! @param[in] imageW 画像の幅です。
    //! @param[in] imageH 画像の高さです。
    //!
    //! @return ミップマップの最大レベル数を返します。
    //!
    static int GetMaxMipLevel(const int imageW, const int imageH)
    {
        int mipMax = 1;
        int maxWH = (imageW > imageH) ? imageW : imageH;
        while (maxWH >= 2)
        {
            ++mipMax;
            maxWH >>= 1;
        }
        return mipMax;
    }

protected:
    bool DecodePsd(const uint8_t* fileBuf, const size_t fileSize);
    bool DecodeTga(const uint8_t* fileBuf, const size_t fileSize);

private:
    RImage(const RImage& other);
    RImage& operator=(const RImage& other);
};

//=============================================================================
// ピクセルデータ関連の関数です。
//=============================================================================

void RFlipHVBitmap(uint8_t* dstBuf, const int dstW, const int dstH);

//! @brief ビットマップのすべてのピクセルで R = G = B なら true を返します。
//!
//! @param[in] pData ビットマップデータです。
//! @param[in] width ビットマップの幅です。
//! @param[in] height ビットマップの高さです。
//! @param[in] isFloat データ型が浮動小数点数なら true、整数なら false です。
//!
//! @return すべてのピクセルで R = G = B なら true を返します。
//!
bool RIsAllGrayBitmap(
    const void* pData,
    const int width,
    const int height,
    const bool isFloat
);

//! @brief ビットマップの透明モードを返します。
//!
//! @param[in] pData ビットマップデータです。
//! @param[in] width ビットマップの幅です。
//! @param[in] height ビットマップの高さです。
//! @param[in] isFloat データ型が浮動小数点数なら true、整数なら false です。
//!
//! @return ビットマップの透明モードを返します。
//!
RImage::TransparencyMode RGetBitmapTransparencyMode(
    const void* pData,
    const int width,
    const int height,
    const bool isFloat
);

//! @brief ピクセルデータの全成分を指定した値でフィルします。
//!
//! @param[out] pPixels ピクセルデータです。
//! @param[in] pixelDataSize ピクセルデータのサイズ（バイト数）です。
//! @param[in] value フィルする整数型の値です。
//! @param[in] floatValue フィルする浮動小数点数型の値です。
//! @param[in] isFloat データ型が浮動小数点数なら true、整数なら false です。
//!
void FillPixelsValue(
    uint8_t* pPixels,
    const size_t pixelDataSize,
    const uint8_t value,
    const float floatValue,
    const bool isFloat
);

//! @brief ピクセルデータの RGBA 成分をスワップします。
//!
//! @param[in,out] pPixels ピクセルデータです。
//! @param[in] pixelDataSize ピクセルデータのサイズ（バイト数）です。
//! @param[in] isFloat データ型が浮動小数点数なら true、整数なら false です。
//! @param[in] compSel 成分選択です。
//!
//! @return スワップ後の A 成分存在フラグを返します。
//!
bool SwapPixelsRgbaComponent(
    uint8_t* pPixels,
    const size_t pixelDataSize,
    const bool isFloat,
    const int compSel
);

//=============================================================================
// Photoshop 本体関連の関数です。
//=============================================================================

//! @brief Photoshop のバージョンが指定したバージョン以上かチェックします。
//!
//! @param[in] major メジャーバージョンです。
//! @param[in] minor マイナーバージョンです。
//!
//! @brief 指定したバージョン以上なら true を返します。
//!
bool RIsPhotoshopVersionGreaterEqual(const int major, const int minor);

//=============================================================================
//! @brief リソースデータを解析するクラスです。
//=============================================================================
class RDParse
{
public:
    const char* m_Tag;
    void (*m_Proc)(GPtr global, const uint8_t* dataPtr, const uint32_t dataSize);
};

//=============================================================================
// リソースデータ関連の関数です。
//=============================================================================

int RGetResourceDataVersion(GPtr globals);
int RGetResourceDataMipLevel(GPtr globals);

bool RDoesResourceDataExist(GPtr globals, const char* tagName);
bool RDeleteResourceData(GPtr globals, const char* tagName);
void RDeleteResourceDataAll(GPtr globals);

//! @brief リソースデータブロックを追加します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] tagName タグ名です。
//! @param[in] pData データです。
//! @param[in] dataSize データサイズです。
//!
void RAddRDBlock(
    GPtr globals,
    const char* tagName,
    const void* pData,
    const uint32_t dataSize
);

//! @brief リソースデータブロックを追加します。
//!
//! @param[in] globals グローバルデータです。
//! @param[in] tagName タグ名です。
//! @param[in] dataStr データ文字列です。
//!
void RAddRDBlock(GPtr globals, const char* tagName, const std::string& dataStr);

void RDoRDParseTable(
    GPtr globals,
    const RDParse* parseTable,
    const uint32_t parseSize
);

//=============================================================================
// プロパティ関連の関数です。
//=============================================================================

//! @brief プロパティに対応したアクション記述子を取得します。
//!
//! @param[out] pDesc アクション記述子を格納します。
//! @param[in] index 対象のアクションリファレンス用インデックスです（-1 ならターゲット）。
//! @param[in] desiredClass 対象のクラス ID です。
//! @param[in] desiredKey プロパティのキー ID です。
//!
//! @result エラーを返します。
//!
OSErr RGetActionDescriptorForProperty(
    PIActionDescriptor* pDesc,
    const int index,
    const DescriptorClassID desiredClass,
    const DescriptorKeyID desiredKey
);

//! @brief アクション記述子から整数型プロパティを取得します。
//!
//! @param[out] pError エラーを格納します（nullptr なら格納しません）。
//! @param[in] index 対象のアクションリファレンス用インデックスです（-1 ならターゲット）。
//! @param[in] desiredClass 対象のクラス ID です。
//! @param[in] desiredKey プロパティのキー ID です。
//!
//! @result プロパティの値を返します。
//!
int RGetIntegerProperty(
    OSErr* pError,
    const int index,
    const DescriptorClassID desiredClass,
    const DescriptorKeyID desiredKey
);

//! @brief アクション記述子から実数型プロパティを取得します。
//!
//! @param[out] pError エラーを格納します（nullptr なら格納しません）。
//! @param[in] index 対象のアクションリファレンス用インデックスです（-1 ならターゲット）。
//! @param[in] desiredClass 対象のクラス ID です。
//! @param[in] desiredKey プロパティのキー ID です。
//!
//! @result プロパティの値を返します。
//!
double RGetFloatProperty(
    OSErr* pError,
    const int index,
    const DescriptorClassID desiredClass,
    const DescriptorKeyID desiredKey
);

//! @brief アクション記述子から unit float 型プロパティを取得します。
//!
//! @param[out] pError エラーを格納します（nullptr なら格納しません）。
//! @param[out] pUnitId プロパティの単位 ID を格納します（nullptr なら格納しません）。
//! @param[in] index 対象のアクションリファレンス用インデックスです（-1 ならターゲット）。
//! @param[in] desiredClass 対象のクラス ID です。
//! @param[in] desiredKey プロパティのキー ID です。
//!
//! @result プロパティの値を返します。
//!
double RGetUnitFloatProperty(
    OSErr* pError,
    DescriptorUnitID* pUnitId,
    const int index,
    const DescriptorClassID desiredClass,
    const DescriptorKeyID desiredKey
);

//! @brief アクション記述子から文字列型プロパティを取得します。
//!
//! @param[out] pError エラーを格納します（nullptr なら格納しません）。
//! @param[in] index 対象のアクションリファレンス用インデックスです（-1 ならターゲット）。
//! @param[in] desiredClass 対象のクラス ID です。
//! @param[in] desiredKey プロパティのキー ID です。
//!
//! @result プロパティの値を返します。
//!
std::string RGetStringProperty(
    OSErr* pError,
    const int index,
    const DescriptorClassID desiredClass,
    const DescriptorKeyID desiredKey
);

//! @brief アクション記述子から bool 型プロパティを取得します。
//!
//! @param[out] pError エラーを格納します（nullptr なら格納しません）。
//! @param[in] index 対象のアクションリファレンス用インデックスです（-1 ならターゲット）。
//! @param[in] desiredClass 対象のクラス ID です。
//! @param[in] desiredKey プロパティのキー ID です。
//!
//! @result プロパティの値を返します。
//!
bool RGetBooleanProperty(
    OSErr* pError,
    const int index,
    const DescriptorClassID desiredClass,
    const DescriptorKeyID desiredKey
);

//! @brief アクション記述子から列挙型プロパティを取得します。
//!
//! @param[out] pError エラーを格納します（nullptr なら格納しません）。
//! @param[in] index 対象のアクションリファレンス用インデックスです（-1 ならターゲット）。
//! @param[in] desiredClass 対象のクラス ID です。
//! @param[in] desiredKey プロパティのキー ID です。
//!
//! @result プロパティの値を返します。
//!
DescriptorEnumID RGetEnumeratedProperty(
    OSErr* pError,
    const int index,
    const DescriptorClassID desiredClass,
    const DescriptorKeyID desiredKey
);

//=============================================================================
// ドキュメント関連の関数です。
//=============================================================================

//! @brief ドキュメントのファイルパスを返します。
//!
//! @param[in] globals グローバルデータです。
//!
//! @return ドキュメントのファイルパスを返します。
//!
std::string RGetDocumentFilePath(GPtr globals);

//=============================================================================
// チャンネル関連の関数です。
//=============================================================================

//! @brief 指定したチャンネルがリード可能なら true を返します。
//!
//! @param[in] pChan リードチャンネル記述子へのポインターです。
//!
//! @return 指定したチャンネルがリード可能なら true を返します。
//!
bool RGetChannelReadFlag(const ReadChannelDesc* pChan);

//! @brief 指定したチャンネルがライト可能なら true を返します。
//!
//! @param[in] pChan リードチャンネル記述子へのポインターです。
//!
//! @return 指定したチャンネルがライト可能なら true を返します。
//!
bool RGetChannelWriteFlag(const ReadChannelDesc* pChan);

//! @brief チャンネル数を取得します。
//!
//! @param[in] pChanTop 先頭の成分のリードチャンネル記述子へのポインターです。
//!
//! @return チャンネル数を返します。
//!
int RGetChannelCount(const ReadChannelDesc* pChanTop);

//! @brief 選択されたチャンネル数を取得します。
//!
//! @param[in] pChanTop 先頭の成分のリードチャンネル記述子へのポインターです。
//!
//! @return 選択されたチャンネル数を返します。
//!
int RGetSelectedChannelCount(const ReadChannelDesc* pChanTop);

//! @brief 1 チャンネルのデータをパディングしながらピクセルデータに設定します。
//!        出力先ピクセルデータが大きい場合は端のピクセルの値を延長します。
//!
//! @param[out] dst 出力先ピクセルデータです。
//!                 RGBA の順に入っているバッファーの設定したいチャンネルの
//!                 先頭を指定します。
//!                 R = top + 0
//!                 G = top + 1
//!                 B = top + 2
//!                 A = top + 3
//! @param[in] src 1 チャンネルのデータのポインタです。
//! @param[in] dstW 出力先ピクセルデータの幅です。
//! @param[in] dstH 出力先ピクセルデータの高さです。
//! @param[in] srcW 1 チャンネルのデータの幅です。
//! @param[in] srcH 1 チャンネルのデータの高さです。
//!
void RSetOneChannelToPixelsPadding(
    uint8_t* dst,
    const uint8_t* src,
    const int dstW,
    const int dstH,
    const int srcW,
    const int srcH
);

//! @brief ピクセルデータから 1 チャンネルのデータを取得します。
//!
//! @param[out] dst 出力先の 1 チャンネルのデータのポインタです。
//! @param[in] src ピクセルデータです。
//!                RGBA の順に入っているバッファーの取得したいチャンネルの
//!                先頭を指定します。
//!                R = top + 0
//!                G = top + 1
//!                B = top + 2
//!                A = top + 3
//! @param[in] dstW 1 チャンネルのデータの幅です。
//! @param[in] dstH 1 チャンネルのデータの高さです。
//! @param[in] srcW ピクセルデータの幅です。
//!
void RGetOneChannelFromPixels(
    uint8_t* dst,
    const uint8_t* src,
    const int dstW,
    const int dstH,
    const int srcW
);

//! @brief ピクセルデータの共通領域をコピーします。
//!
//! @param[in,out] pDstPixels コピー先ピクセルデータです。
//! @param[in] dstBounds コピー先ピクセルデータの領域です。
//! @param[in] dstPixelBytes コピー先の 1 ピクセルあたりのバイト数です。
//! @param[in] pSrcPixels コピー元ピクセルデータです。
//! @param[in] srcBounds コピー元ピクセルデータの領域です。
//! @param[in] srcPixelBytes コピー元の 1 ピクセルあたりのバイト数です。
//! @param[in] multiplies 乗算するなら true、コピーするなら false です。
//!
void RCopyPixelsWithBounds(
    uint8_t* pDstPixels,
    const VRect& dstBounds,
    const size_t dstPixelBytes,
    const uint8_t* pSrcPixels,
    const VRect& srcBounds,
    const size_t srcPixelBytes,
    const bool multiplies
);

//! @brief チャンネルをリードしてピクセルデータに反映します。
//!
//! @param[in,out] pPixels ピクセルデータです。
//! @param[in] dstBounds ピクセルデータの領域です。
//! @param[in] dstPixelBytes 1 ピクセルあたりのバイト数です。
//! @param[in] pChan リードチャンネル記述子へのポインターです。
//! @param[in] multiplies 乗算するなら true、コピーするなら false です。
//!
void RReadChannelToPixels(
    uint8_t* pPixels,
    const VRect& dstBounds,
    const size_t dstPixelBytes,
    const ReadChannelDesc* pChan,
    const bool multiplies
);

//! @brief 複数チャンネルをリードしてピクセルデータに反映します。
//!
//! @param[in,out] pPixels ピクセルデータです（RGBA 形式）。
//! @param[in] dstBounds ピクセルデータの領域です。
//! @param[in] dstPixelBytes 1 ピクセルあたりのバイト数です。
//! @param[in] pColorChan 先頭のカラー成分のリードチャンネル記述子へのポインターです。
//! @param[in] pTransparencyChan 不透明度のリードチャンネル記述子へのポインターです。
//!                              不透明度チャンネルがない場合は nullptr を指定します。
//! @param[in] pMaskChan レイヤーマスクのリードチャンネル記述子へのポインターです。
//!                      レイヤーマスクチャンネルがない場合は nullptr を指定します。
//!
void RReadMultiChannelsToPixels(
    uint8_t* pPixels,
    const VRect& dstBounds,
    const size_t dstPixelBytes,
    const ReadChannelDesc* pColorChan,
    const ReadChannelDesc* pTransparencyChan,
    const ReadChannelDesc* pMaskChan
);

//=============================================================================
// Photoshop スクリプト関連の関数です。
//=============================================================================

//! @brief リード記述子から文字列パラメータを取得します。
//!
//! @param[in,out] globals グローバルデータです。
//! @param[out] pString 取得した文字列を格納します。
//! @param[in] desc リード記述子です。
//! @param[in] utf8ToSjis 文字コードを UTF-8 から Shift-JIS に変換するなら true です。
//!
//! @return 取得に成功すれば true を返します。
//!
bool RGetStringParam(
    GPtr globals,
    std::string* pString,
    const PIReadDescriptor& desc,
    const bool utf8ToSjis
);

//! @brief リード記述子から文字列パラメータを取得します。
//!
//! @param[in,out] globals グローバルデータです。
//! @param[out] stringBuf 取得した文字列を格納するバッファです。
//! @param[in] maxStringSize 文字列バッファの最大サイズです。
//! @param[in] desc リード記述子です。
//! @param[in] utf8ToSjis 文字コードを UTF-8 から Shift-JIS に変換するなら true です。
//!
//! @return 取得に成功すれば true を返します。
//!
bool RGetStringParam(
    GPtr globals,
    char* stringBuf,
    const size_t maxStringSize,
    const PIReadDescriptor& desc,
    const bool utf8ToSjis
);

//! @brief ライト記述子に文字列パラメータを設定します。
//!
//! @param[in,out] globals グローバルデータです。
//! @param[in] token ライト記述子です。
//! @param[in] key パラメータのキーです。
//! @param[in] value 値です。
//!
//! @return 設定に成功すれば true を返します。
//!
bool RPutStringParam(
    GPtr globals,
    const PIWriteDescriptor& token,
    const DescriptorKeyID key,
    const std::string& value
);

//=============================================================================
// UI 関連の関数です。
//=============================================================================

#ifdef __PIWin__
BOOL WINAPI RAboutDialogProc(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam);

void RSetPSRectFromWinRect(VRect& dst, const RECT& src);

void RSetWinRectFromPSRect(RECT& dst, const VRect& src);

void RShowPluginHelp(const char* url);

//! @brief POINT 型の座標が RECT 型の矩形に含まれれば true を返します。
inline bool RIsPointInRect(const RECT& rect, const POINT& point)
{
    return (
        point.x >= rect.left  &&
        point.x <  rect.right &&
        point.y >= rect.top   &&
        point.y <  rect.bottom);
}

//! @brief POINTS 型の座標が RECT 型の矩形に含まれれば true を返します。
inline bool RIsPointInRect(const RECT& rect, const POINTS& point)
{
    return (
        point.x >= rect.left  &&
        point.x <  rect.right &&
        point.y >= rect.top   &&
        point.y <  rect.bottom);
}

//! @brief 時計カーソルのクラスです。
class RWaitCursor
{
public:
    //! コンストラクタです。
    RWaitCursor()
    {
        m_hPreviousCursor = SetCursor(LoadCursor(nullptr, IDC_WAIT));
    }
    //! デストラクタです。
    virtual ~RWaitCursor()
    {
        SetCursor(m_hPreviousCursor);
    }
protected:
    HCURSOR m_hPreviousCursor; //!< 以前のカーソルハンドルです。
};
#endif

#if __PIMac__
ControlHandle GetControlHandleMac(const DialogPtr pDlg, const short id);

void ActivateControlMac(const DialogPtr pDlg, const short id, const bool activeFlag);

void GetStringForMacUI(Str255& dst, const char* src);

void SetTextMacUI(DialogPtr pDlg, const short id, const char* text);

void ParamTextForMacUI(
    const char* text0,
    const char* text1 = NULL,
    const char* text2 = NULL,
    const char* text3 = NULL
);

void ShowAboutCommonMac(const char* message, const char* title);
#endif

//! @brief 日本語 UI を表示するなら true を返します。
bool RIsJapaneseUI();

//! @brief ダイアログのコントロールにツールチップを追加します。
//!
//! @param[in] hDlg ダイアログハンドルです。
//! @param[in] id UI コントロールの ID です。
//! @param[in] pText テキストへのポインタです。
//!
void RAddToolTip(HWND hDlg, const int id, const char* pText);

//=============================================================================
// nps ネームスペースを終了します。
//=============================================================================
} // namespace nps
} // namespace tool
} // namespace gfx
} // namespace nn

