﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// RXMLElement LoadDocument
// RBinDataStream
// RUserData
// RFileBuf

// RTexCvtr RTexEncoder
// RShowError RCheckMemoryStatus RShowPluginHelp

// RAddRDBlock
// RImage RImage_ReadFile RImage_DecodePsd

//=============================================================================
// include
//=============================================================================
#ifndef NPS_CLONE_SOURCE
    #include "Globals.h"
#endif
#include <commctrl.h> // for slider & tooltip
#pragma comment(lib, "shlwapi.lib")
#include <PITerminology.h>

//=============================================================================
// グローバルネームスペースの変数です。
//=============================================================================

//-----------------------------------------------------------------------------
// new & delete
SPBasicSuite* sSPBasic = NULL; // extern 宣言は PIUNew.h 内

//=============================================================================
// nps ネームスペースを開始します。
//=============================================================================
namespace nn {
namespace gfx {
namespace tool {
namespace nps {

//=============================================================================
// constants
//=============================================================================

//-----------------------------------------------------------------------------
// ヒント情報
const std::string HintValueDefault = "<default>"; //!< デフォルトのヒント情報です。
const std::string HintValueNormal = "normal"; //!< 法線用のヒント情報です。

//-----------------------------------------------------------------------------
// リニア変換
static const int s_LinearRgbaFlags[R_RGBA_COUNT] =
{
    RLinearFlag::LINEAR_R,
    RLinearFlag::LINEAR_G,
    RLinearFlag::LINEAR_B,
    RLinearFlag::LINEAR_A,
};

//-----------------------------------------------------------------------------
// resouce data tag
const char RD_TAG_RD_VER[]         = "nw4f_rdv";
const char RD_TAG_DCC_PRESET[]     = "nw4f_dpr";
const char RD_TAG_HINT[]           = "nw4f_hin";
const char RD_TAG_LINEAR[]         = "nw4f_lin";
const char RD_TAG_DIMENSION[]      = "nw4f_dim";
const char RD_TAG_FORMAT[]         = "nw4f_fmt";
const char RD_TAG_WEIGHT_CMPR[]    = "nw4f_wcp";
const char RD_TAG_INI_SWIZZLE[]    = "nw4f_isw";
const char RD_TAG_MIP_LEVEL[]      = "nw4f_mpl";
const char RD_TAG_MIP_GEN_FILTER[] = "nw4f_mgf";
const char RD_TAG_COMP_SEL[]       = "nw4f_cps";
const char RD_TAG_ORG_PATHS[]      = "nw4f_ops";
const char RD_TAG_CREATE_INFO[]    = "nw4f_cri";
const char RD_TAG_USER_DATA[]      = "nw4f_udt";
const char RD_TAG_COMMENT_LABEL[]  = "nw4f_cml";
const char RD_TAG_COMMENT_COLOR[]  = "nw4f_cmc";
const char RD_TAG_COMMENT_TEXT[]   = "nw4f_cmt";
const char RD_TAG_TOOL_DATA[]      = "nw4f_tdt";
const char RD_TAG_USER_TOOL_DATA[] = "nw4f_utd";

//-----------------------------------------------------------------------------
// コンフィグファイル
static const char* CONFIG_FILE_PREFIX        = "ExportTexture_"; //!< コンフィグファイル名のプレフィックスです。
static const char* STANDARD_CONFIG_FILE_NAME = "ExportTexture"; //!< 標準のコンフィグファイル名です。
const char* R_STANDARD_CONFIG_NAME = "<Standard>";

//-----------------------------------------------------------------------------
//! @brief 整数値と文字列のポインタの構造体です。
//-----------------------------------------------------------------------------
struct RIntAndCharPtr
{
public:
    int m_Value;
    const char* m_String;
};

//-----------------------------------------------------------------------------
//! @brief 2 のべき乗の数なら true を返します。
//-----------------------------------------------------------------------------
bool RIsPower2Number(const int number, const bool includeOne)
{
    if (includeOne && number == 1)
    {
        return true;
    }

    int cur = 1;
    do
    {
        cur *= 2;
        if (number == cur)
        {
            return true;
        }
    } while (cur < number);

    return false;
}

//-----------------------------------------------------------------------------
//! @brief 指定した数以上で最小の 2 のべき乗の数を返します。
//-----------------------------------------------------------------------------
int RGetPower2Number(const int number, const bool includeOne)
{
    int ret = (includeOne) ? 1 : 2;
    while (ret < number)
    {
        ret <<= 1;
    }
    return ret;
}

//-----------------------------------------------------------------------------
//! @brief 指定した文字が Shift JIS の 2 バイト文字の 1 バイト目なら true を返します。
//-----------------------------------------------------------------------------
bool RIsShiftJisFirstByte(const unsigned char val)
{
    return ((0x81 <= val && val <= 0x9f) ||
            (0xe0 <= val && val <= 0xef));
}

//-----------------------------------------------------------------------------
//! @brief Shift JIS 文字列を UTF-8 文字列に変換して返します。
//-----------------------------------------------------------------------------
std::string RGetUtf8FromShiftJis(const std::string& src)
{
    //-----------------------------------------------------------------------------
    // Shift JIS -> Unicode (UCS-2)
    // 変換後のサイズ（終端の L'\0' を含む）を求めます。
    int wcharSize = MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, NULL, 0);
    if (wcharSize == 0)
    {
        return std::string();
    }
    wchar_t* wcharBuf = new wchar_t[wcharSize];
    MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, wcharBuf, wcharSize);

    //-----------------------------------------------------------------------------
    // Unicode (UCS-2) -> UTF-8
    // 変換後のサイズ（終端の '\0' を含む）を求めます。
    int utf8Size = WideCharToMultiByte(CP_UTF8, 0, wcharBuf, -1, NULL, 0, NULL, NULL);
    char* utf8Buf = new char[utf8Size];
    WideCharToMultiByte(CP_UTF8, 0, wcharBuf, -1, utf8Buf, utf8Size, NULL, NULL);
    std::string dst(utf8Buf, strlen(utf8Buf));
    // 通常 strlen(utf8Buf) = utf8Size - 1

#if 0
    cerr << "[" << src.c_str() << "] [" << dst.c_str() << "] " << src.size() << ", " << wcharSize << ", " << utf8Size << ", " << dst.size() << endl;
    for (int i = 0; i < utf8Size; ++i)
    {
        cerr << hex << static_cast<int>(static_cast<uint8_t>(utf8Buf[i])) << dec << endl;
    }
#endif

    //-----------------------------------------------------------------------------
    // free memory
    delete[] wcharBuf;
    delete[] utf8Buf;

    return dst;
}

//-----------------------------------------------------------------------------
//! @brief UTF-8 文字列を Shift JIS 文字列に変換して返します。
//-----------------------------------------------------------------------------
std::string RGetShiftJisFromUtf8(const std::string& src)
{
    //-----------------------------------------------------------------------------
    // UTF-8 -> Unicode (UCS-2)
    // 変換後のサイズ（終端の L'\0' を含む）を求めます。
    int wcharSize = MultiByteToWideChar(CP_UTF8, 0, src.c_str(), -1, NULL, 0);
    if (wcharSize == 0)
    {
        return std::string();
    }
    wchar_t* wcharBuf = new wchar_t[wcharSize];
    MultiByteToWideChar(CP_UTF8, 0, src.c_str(), -1, wcharBuf, wcharSize);

    //-----------------------------------------------------------------------------
    // Unicode (UCS-2) -> Shift JIS
    // 変換後のサイズ（終端の '\0' を含む）を求めます。
    int sjisSize = WideCharToMultiByte(CP_ACP, 0, wcharBuf, -1, NULL, 0, NULL, NULL);
    char* sjisBuf = new char[sjisSize];
    WideCharToMultiByte(CP_ACP, 0, wcharBuf, -1, sjisBuf, sjisSize, NULL, NULL);
    std::string dst(sjisBuf, strlen(sjisBuf));
    // 通常 strlen(sjisBuf) = sjisSize - 1

    //-----------------------------------------------------------------------------
    // free memory
    delete[] wcharBuf;
    delete[] sjisBuf;

    return dst;
}

//-----------------------------------------------------------------------------
//! @brief Shift JIS 文字列を Unicode (UCS-2) 文字列に変換して返します。
//-----------------------------------------------------------------------------
std::wstring RGetUnicodeFromShiftJis(const std::string& src)
{
    //-----------------------------------------------------------------------------
    // Shift JIS -> Unicode (UCS-2)
    // 変換後のサイズ（終端の L'\0' を含む）を求めます。
    int wcharSize = MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, NULL, 0);
    if (wcharSize == 0)
    {
        return std::wstring();
    }
    wchar_t* wcharBuf = new wchar_t[wcharSize];
    MultiByteToWideChar(CP_ACP, 0, src.c_str(), -1, wcharBuf, wcharSize);
    std::wstring dst(wcharBuf, wcslen(wcharBuf));

    //-----------------------------------------------------------------------------
    // free memory
    delete[] wcharBuf;

    return dst;
}

//-----------------------------------------------------------------------------
//! @brief Unicode (UCS-2) 文字列を Shift JIS 文字列に変換して返します。
//-----------------------------------------------------------------------------
std::string RGetShiftJisFromUnicode(const std::wstring& src)
{
    //-----------------------------------------------------------------------------
    // Unicode (UCS-2) -> Shift JIS
    // 変換後のサイズ（終端の '\0' を含む）を求めます。
    int sjisSize = WideCharToMultiByte(CP_ACP, 0, src.c_str(), -1, NULL, 0, NULL, NULL);
    if (sjisSize == 0)
    {
        return std::string();
    }

    char* sjisBuf = new char[sjisSize];
    WideCharToMultiByte(CP_ACP, 0, src.c_str(), -1, sjisBuf, sjisSize, NULL, NULL);
    std::string dst(sjisBuf, strlen(sjisBuf));

    //-----------------------------------------------------------------------------
    // free memory
    delete[] sjisBuf;

    return dst;
}

//-----------------------------------------------------------------------------
//! @brief ASCII 文字列なら true を返します。
//-----------------------------------------------------------------------------
bool RIsAsciiString(const std::string& str)
{
    for (size_t ic = 0; ic < str.length(); ++ic)
    {
        const char c = str[ic];
        if (c < 0x20)
        {
            return false;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief 文字列配列を結合した 1 つの文字列を返します。
//!
//! @param[in] srcs 文字列配列です。
//! @param[in] sep 区切り文字列です。
//!
//! @return 結合した 1 つの文字列を返します。
//-----------------------------------------------------------------------------
std::string RJoinStrings(const RStringArray& srcs, const std::string& sep)
{
    std::string dst;
    for (size_t iSrc = 0; iSrc < srcs.size(); ++iSrc)
    {
        if (iSrc > 0) dst += sep;
        dst += srcs[iSrc];
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief 文字列を指定した文字で分割した文字列配列を取得します。
//!
//! @param[out] dsts 文字列配列を格納します。
//! @param[in] src 文字列です。
//! @param[in] sep 区切り文字列です。
//!
//! @return 文字列配列のサイズを返します。
//-----------------------------------------------------------------------------
int RSeparateString(RStringArray& dsts, const std::string& src, const std::string& sep)
{
    dsts.clear();
    std::string cur = src;
    while (!cur.empty())
    {
        const size_t iSep = cur.find(sep);
        if (iSep == std::string::npos)
        {
            dsts.push_back(cur);
            break;
        }
        dsts.push_back(cur.substr(0, iSep));
        cur = (iSep + 1 < cur.size()) ? cur.substr(iSep + 1) : std::string();
    }
    return static_cast<int>(dsts.size());
}

//-----------------------------------------------------------------------------
//! @brief ユニコード C 配列を RStringArray に変換します。
//-----------------------------------------------------------------------------
int RGetStringArray(RStringArray& dsts, const wchar_t* srcs[])
{
    dsts.clear();
    int iStr = 0;
    while (srcs[iStr] != NULL)
    {
        dsts.push_back(RGetShiftJisFromUnicode(srcs[iStr]));
        ++iStr;
    }
    return static_cast<int>(dsts.size());
}

//-----------------------------------------------------------------------------
//! @brief すべて小文字にした文字列を返します。
//-----------------------------------------------------------------------------
std::string RGetLowerCaseString(const std::string& src)
{
    std::string dst = src;
    int size = static_cast<int>(dst.size());
    for (int ic = 0; ic < size; ++ic)
    {
        char c = dst[ic];
        if ('A' <= c && c <= 'Z')
        {
            dst[ic] = c - 'A' + 'a';
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief すべて大文字にした文字列を返します。
//-----------------------------------------------------------------------------
std::string RGetUpperCaseString(const std::string& src)
{
    std::string dst = src;
    int size = static_cast<int>(dst.size());
    for (int ic = 0; ic < size; ++ic)
    {
        char c = dst[ic];
        if ('a' <= c && c <= 'z')
        {
            dst[ic] = c - 'a' + 'A';
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief 2 つの文字列が大文字／小文字を無視して同じなら true を返します。
//-----------------------------------------------------------------------------
bool RIsSameStringNoCase(const std::string& str1, const std::string& str2)
{
    return (RGetLowerCaseString(str1) == RGetLowerCaseString(str2));
}

//-----------------------------------------------------------------------------
//! @brief 文字列配列から大文字小文字の区別なしで値を検索してインデックスを返します。
//!        見つからなければ -1 を返します。
//!
//! @param[in] array 文字列配列です。
//! @param[in] value 検索する値です。
//!
//! @return インデックスを返します。見つからなければ -1 を返します。
//-----------------------------------------------------------------------------
int RFindNoCaseStringInArray(const RStringArray& array, const std::string& value)
{
    const std::string valueL = RGetLowerCaseString(value);
    const int count = static_cast<int>(array.size());
    for (int index = 0; index < count; ++index)
    {
        if (RGetLowerCaseString(array[index]) == valueL)
        {
            return index;
        }
    }
    return -1;
}

//-----------------------------------------------------------------------------
//! @brief 文字列の前の不要な文字を削除します。
//-----------------------------------------------------------------------------
std::string RTrimLeftString(const std::string& src, const char* targets)
{
    const size_t iLeft = src.find_first_not_of(targets);
    return (iLeft != std::string::npos) ? src.substr(iLeft) : std::string();
}

//-----------------------------------------------------------------------------
//! @brief 文字列の後の不要な文字を削除します。
//-----------------------------------------------------------------------------
std::string RTrimRightString(const std::string& src, const char* targets)
{
    const size_t iRight = src.find_last_not_of(targets);
    return (iRight != std::string::npos) ? src.substr(0, iRight + 1) : std::string();
}

//-----------------------------------------------------------------------------
//! @brief 文字列の前後の不要な文字を削除します。
//-----------------------------------------------------------------------------
std::string RTrimString(const std::string& src, const char* targets)
{
    const size_t iLeft = src.find_first_not_of(targets);
    if (iLeft != std::string::npos)
    {
        const size_t iRight = src.find_last_not_of(targets);
        if (iRight != std::string::npos)
        {
            return src.substr(iLeft, iRight - iLeft + 1);
        }
    }
    return std::string();
}

//-----------------------------------------------------------------------------
//! @brief 文字列の最後の改行コードを削除します。
//-----------------------------------------------------------------------------
std::string RCutEndlString(const std::string& src)
{
    const char* str = src.c_str();
    const int size = static_cast<int>(strlen(str));
    for (int ic = size - 1; ic >= 0; --ic)
    {
        const char c = str[ic];
        if (c != '\r' && c != '\n')
        {
            return std::string(str, ic + 1);
        }
    }
    return "";
}

//-----------------------------------------------------------------------------
//! @brief 文字列の前後の引用符を削除します。
//-----------------------------------------------------------------------------
std::string RDequoteString(const std::string& src, const char quote)
{
    std::string dst(src);
    const size_t iquotL = dst.find(quote);
    if (iquotL != std::string::npos)
    {
        if (iquotL == dst.size() - 1) // 末尾にのみ引用符が存在する場合、その引用符を削除します。
        {
            dst = dst.substr(0, iquotL);
        }
        else
        {
            dst = dst.substr(iquotL + 1);
            const size_t iquotR = dst.rfind(quote);
            if (iquotR != std::string::npos)
            {
                dst = dst.substr(0, iquotR);
            }
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief 文字列を指定した区切り文字で分割します。
//!        ",abc,def,,ghi," -> { "", "abc", "def", "", "ghi", "" }
//-----------------------------------------------------------------------------
int RSplitString(RStringArray& words, const std::string& input, const char delim)
{
    words.clear();
    if (input.empty())
    {
        words.push_back(input);
    }
    else
    {
        std::string cur = input;
        while (!cur.empty())
        {
            int iDelim = -1;
            for (int ic = 0; ic < static_cast<int>(cur.size()); ++ic)
            {
                if (cur[ic] == delim)
                {
                    iDelim = ic;
                    break;
                }
            }

            if (iDelim == -1)
            {
                words.push_back(cur);
                break;
            }
            else
            {
                words.push_back(cur.substr(0, iDelim));
                cur = cur.substr(iDelim + 1);
                if (cur.empty()) // 区切り文字で終わっている場合
                {
                    words.push_back(cur);
                    break;
                }
            }
        }
    }
    return static_cast<int>(words.size());
}

//-----------------------------------------------------------------------------
//! @brief 文字列をトークンに分解します。
//!        "abc\n def\n \n ghi" -> { "abc", "def", "ghi" }
//-----------------------------------------------------------------------------
int RTokenizeString(RStringArray& tokens, const std::string& input, const char* delims)
{
    tokens.clear();
    size_t iEnd = 0;
    for (;;)
    {
        const size_t iStart = input.find_first_not_of(delims, iEnd);
        if (iStart == std::string::npos)
        {
            break;
        }
        iEnd = input.find_first_of(delims, iStart);
        if (iEnd == std::string::npos)
        {
            tokens.push_back(input.substr(iStart));
            break;
        }
        tokens.push_back(input.substr(iStart, iEnd - iStart));
    }
    return static_cast<int>(tokens.size());
}

//-----------------------------------------------------------------------------
//! @brief int 型の数値を文字列に変換します。
//-----------------------------------------------------------------------------
std::string RGetNumberString(const int num, const char* format)
{
    char buf[64];
    const char* fmt = (format != NULL) ? format : "%d";
    _snprintf_s(buf, sizeof(buf), _TRUNCATE, fmt, num);
    return std::string(buf);
}

//-----------------------------------------------------------------------------
//! @brief uint32_t 型の数値を文字列に変換します。
//-----------------------------------------------------------------------------
std::string RGetNumberString(const uint32_t num, const char* format)
{
    char buf[64];
    const char* fmt = (format != NULL) ? format : "%u";
    _snprintf_s(buf, sizeof(buf), _TRUNCATE, fmt, num);
    return std::string(buf);
}

//-----------------------------------------------------------------------------
//! @brief ポインタを文字列に変換します。
//-----------------------------------------------------------------------------
std::string RGetPointerString(const void* ptr, const char* format)
{
    char buf[64];
    const char* fmt = (format != NULL) ? format : "%p";
    _snprintf_s(buf, sizeof(buf), _TRUNCATE, fmt, ptr);
    return std::string(buf);
}

//-----------------------------------------------------------------------------
//! @brief Shift-JIS 文字列を XML 用にエンコードします。
//-----------------------------------------------------------------------------
static const std::string XML_AMP_STR  = "&amp;";
static const std::string XML_LT_STR   = "&lt;";
static const std::string XML_GT_STR   = "&gt;";
static const std::string XML_QUOT_STR = "&quot;";

std::string REncodeXmlString(const std::string& src)
{
    std::string dst;
    for (size_t ic = 0; ic < src.size(); ++ic)
    {
        const char c = src[ic];
        if (RIsShiftJisFirstByte(static_cast<unsigned char>(c)))
        {
            dst += c;
            if (++ic < src.size())
            {
                dst += src[ic];
            }
        }
        else if (c == '&')
        {
            dst += XML_AMP_STR;
        }
        else if (c == '<')
        {
            dst += XML_LT_STR;
        }
        else if (c == '>')
        {
            dst += XML_GT_STR;
        }
        else if (c == '\"')
        {
            dst += XML_QUOT_STR;
        }
        else
        {
            dst += c;
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief XML 用にエンコードされた Shift-JIS 文字列をデコードします。
//-----------------------------------------------------------------------------
std::string RDecodeXmlString(const std::string& src)
{
    std::string dst;
    for (size_t ic = 0; ic < src.size(); ++ic)
    {
        const char c = src[ic];
        if (c == '&')
        {
            if (src.find(XML_AMP_STR, ic) == ic)
            {
                dst += '&';
                ic += XML_AMP_STR.size() - 1;
            }
            else if (src.find(XML_LT_STR, ic) == ic)
            {
                dst += '<';
                ic += XML_LT_STR.size() - 1;
            }
            else if (src.find(XML_GT_STR, ic) == ic)
            {
                dst += '>';
                ic += XML_GT_STR.size() - 1;
            }
            else if (src.find(XML_QUOT_STR, ic) == ic)
            {
                dst += '\"';
                ic += XML_QUOT_STR.size() - 1;
            }
            else
            {
                dst += c;
            }
        }
        else
        {
            dst += c;
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief C/C++ ソースのコメントをカットします。
//-----------------------------------------------------------------------------
std::string RCutCSourceComment(const std::string& src)
{
    std::string dst;
    const int count = static_cast<int>(src.size());

    bool inQuat = false;   // ダブルクォート内なら true です。
    bool inCCom = false;   // C コメント内なら true です。
    bool inCppCom = false; // C++ コメント内なら true です。
    for (int ic = 0; ic < count; ++ic)
    {
        const char c0 = src[ic];
        const char c1 = (ic + 1 < count) ? src[ic + 1] : '\0';
        if (inQuat)
        {
            dst += c0;
            if (c0 == '\"')
            {
                inQuat = false;
            }
        }
        else if (inCCom)
        {
            if (c0 == '*' || c1 == '/')
            {
                inCCom = false;
                ic += 1;
            }
        }
        else if (inCppCom)
        {
            if (c0 == '\r' || c0 == '\n')
            {
                inCppCom = false;
                dst += c0;
            }
        }
        else
        {
            if (c0 == '/' && c1 == '*')
            {
                inCCom = true;
                ic += 1;
            }
            else if (c0 == '/' && c1 == '/')
            {
                inCppCom = true;
                ic += 1;
            }
            else if (c0 == '\"')
            {
                inQuat = true;
                dst += c0;
            }
            else
            {
                dst += c0;
            }
        }
    }
    return dst;
}

//-----------------------------------------------------------------------------
//! @brief SDK の HostHandle2CString は末尾に余計な文字が残るのでこちらを使用します。
//!        outString は char[outMaxSize] の大きさで確保しておきます。
//-----------------------------------------------------------------------------
void RHandle2CString(
    HandleProcs* procs,
    Handle inDataHandle,
    char* outString,
    const size_t outMaxSize
)
{
    PIResetString((unsigned char *)outString); // macro to clear a string. See header file.

    if (inDataHandle != NULL)
    { // inDataHandle is valid.

        Ptr p = HostLockHandle(procs, inDataHandle, FALSE);

        if (p != NULL)
        { // Got a valid pointer.

            size_t size = HostGetHandleSize(procs, inDataHandle);

            if (size >= outMaxSize - 1)
                size = outMaxSize - 1; // Have to pin to maximum string length.

            // Append handle to string:
            PICopy(outString, p, size);
            outString[size] = '\0'; // changed

        } // p

        HostUnlockHandle(procs, inDataHandle); // Done copying.  Let it move.

    } // inDataHandle

} // end HostHandle2CString

//-----------------------------------------------------------------------------
//! @brief ポイントの表示用文字列を取得します。
//-----------------------------------------------------------------------------
std::string RGetPointString(const VPoint& point)
{
    char strBuf[256];
    _snprintf_s(strBuf, _TRUNCATE, "(%d,%d)", point.h, point.v);
    return std::string(strBuf);
}

//-----------------------------------------------------------------------------
//! @brief 矩形の表示用文字列を取得します。
//-----------------------------------------------------------------------------
std::string RGetRectString(const VRect& rect)
{
    char strBuf[256];
    _snprintf_s(strBuf, _TRUNCATE, "(%d,%d)-(%d,%d)", rect.left, rect.top, rect.right, rect.bottom);
    return std::string(strBuf);
}

//-----------------------------------------------------------------------------
//! @brief 32bit ID を 4 文字の文字列に変換して返します。
//-----------------------------------------------------------------------------
std::string RGetStringFrom32BitId(const uint32_t id)
{
    char strBuf[4];
    uint32_t value = id;
    for (int charIdx = 0; charIdx < sizeof(strBuf); ++charIdx)
    {
        const char c = static_cast<char>(value & 0xff);
        if (0x20 <= c && c < 0x80)
        {
            strBuf[sizeof(strBuf) - 1 - charIdx] = c;
        }
        else
        {
            return RGetNumberString(id, "0x%x");
        }
        value >>= 8;
    }
    return std::string(strBuf, sizeof(strBuf));
}

//-----------------------------------------------------------------------------
//! @brief StringID から TypeID を取得します。
//-----------------------------------------------------------------------------
DescriptorTypeID RGetTypeId(const std::string& stringId)
{
    std::string idStr = stringId;
    DescriptorTypeID typeId;
    const OSErr err = sPSBasicActionControl->StringIDToTypeID(const_cast<char*>(idStr.c_str()), &typeId);
    if (err == noErr)
    {
        return typeId;
    }
    else
    {
        //RNoteTrace("type id error: %d (%s)", err, stringId.c_str());
        return static_cast<DescriptorTypeID>(0);
    }
}

//-----------------------------------------------------------------------------
//! @brief TypeID から StringID を取得します。
//!        TypeID の値が衝突しているものは正確に変換できません。
//-----------------------------------------------------------------------------
std::string RGetStringId(const uint32_t typeId)
{
    char strBuf[256] = { 0 };
    const OSErr err = sPSBasicActionControl->TypeIDToStringID(typeId, strBuf, sizeof(strBuf));
    if (err == noErr)
    {
        // enumGreen / keyGreen / eventGrain / keyGrain → すべて grain
        return std::string(strBuf);
    }
    else
    {
        //RNoteTrace("string id error: %d (%s)", err, RGetStringFrom32BitId(typeId).c_str());
        return RGetStringFrom32BitId(typeId);
    }
}

//-----------------------------------------------------------------------------
//! @brief ファイルパス文字列（Shift JIS）を Windows 形式に変換して返します。
//-----------------------------------------------------------------------------
std::string RGetWindowsFilePath(const std::string& path)
{
    const int size = static_cast<int>(path.size());
    char* buf = new char[size + 1];

    const char* src = path.c_str();
    char* dst = buf;

    while (*src)
    {
        char c = *src++;
        if (c == '/')
        {
            c = '\\';
        }
        *dst++ = c;
    }
    *dst = '\0';

    std::string ret = buf;
    delete[] buf;

    return ret;
}

//-----------------------------------------------------------------------------
//! @brief ファイルパス文字列（Shift JIS）を UNIX 形式に変換して返します。
//-----------------------------------------------------------------------------
std::string RGetUnixFilePath(const std::string& path)
{
    const int size = static_cast<int>(path.size());
    char* buf = new char[size + 1];

    const char* src = path.c_str();
    char* dst = buf;

    while (*src)
    {
        char c = *src++;
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            if (*src)
            {
                *dst++ = c;
                c = *src++;
            }
        }
        else if (c == '\\')
        {
            c = '/';
        }
        *dst++ = c;
    }
    *dst = '\0';

    std::string ret = buf;
    delete[] buf;

    return ret;
}

//-----------------------------------------------------------------------------
//! @brief ファイルパス文字列（Shift JIS）のファイル名部分を返します。
//-----------------------------------------------------------------------------
std::string RGetFileNameFromFilePath(const std::string& path)
{
    int size = static_cast<int>(path.size());
    const char* str = path.c_str();
    int iSlash = -1;
    for (int ic = 0; ic < size; ++ic)
    {
        char c = str[ic];
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            ++ic; // skip next byte
        }
        else if (c == '\\' || c == '/')
        {
            iSlash = ic;
        }
    }
    if (iSlash >= 0)
    {
        return std::string(str + iSlash + 1);
    }
    else
    {
        return path;
    }
}

//-----------------------------------------------------------------------------
//! @brief ファイルパス文字列（Shift JIS）のフォルダー部分を返します。
//!        最後の \ または / は含まれません。
//-----------------------------------------------------------------------------
std::string RGetFolderFromFilePath(const std::string& path)
{
    int size = static_cast<int>(path.size());
    const char* str = path.c_str();
    int iSlash = -1;
    for (int ic = 0; ic < size; ++ic)
    {
        char c = str[ic];
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            ++ic; // skip next byte
        }
        else if (c == '\\' || c == '/')
        {
            iSlash = ic;
        }
    }
    if (iSlash >= 0)
    {
        return std::string(str, iSlash); // without last slash
        //return std::string(str, iSlash + 1); // with last slash
    }
    else
    {
        return std::string(".");
    }
}

//-----------------------------------------------------------------------------
//! @brief ファイルパス文字列（Shift JIS）の拡張子を返します。
//!        . は含まれません。
//-----------------------------------------------------------------------------
std::string RGetExtensionFromFilePath(const std::string& path, const bool lower)
{
    const int size = static_cast<int>(path.size());
    const char* str = path.c_str();
    int iSlash = -1;
    int iDot = -1;
    for (int ic = 0; ic < size; ++ic)
    {
        char c = str[ic];
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            ++ic; // skip next byte
        }
        else if (c == '\\' || c == '/')
        {
            iSlash = ic;
        }
        else if (c == '.')
        {
            iDot = ic;
        }
    }

    if (iDot >= 0 && (iSlash == -1 || iDot > iSlash))
    {
        const std::string ext = std::string(str + iDot + 1);
        return (lower) ? RGetLowerCaseString(ext) : ext;
    }
    else
    {
        return std::string(); // empty
    }
}

//-----------------------------------------------------------------------------
//! @brief ファイルパス文字列（Shift JIS）の拡張子以外の部分を返します。
//-----------------------------------------------------------------------------
std::string RGetNoExtensionFilePath(const std::string& path)
{
    int size = static_cast<int>(path.size());
    const char* str = path.c_str();
    int iSlash = -1;
    int iDot = -1;
    for (int ic = 0; ic < size; ++ic)
    {
        char c = str[ic];
        if (RIsShiftJisFirstByte((unsigned char)c))
        {
            ++ic; // skip next byte
        }
        else if (c == '\\' || c == '/')
        {
            iSlash = ic;
        }
        else if (c == '.')
        {
            iDot = ic;
        }
    }
    if (iDot >= 0 && (iSlash == -1 || iDot > iSlash))
    {
        return std::string(str, iDot);
    }
    else
    {
        return path;
    }
}

//-----------------------------------------------------------------------------
//! @brief ファイルパスがフルパスなら true を返します。
//-----------------------------------------------------------------------------
bool RIsFullFilePath(const std::string& path)
{
    if (path.size() >= 2)
    {
        if (path[1] == ':'                       ||
            (path[0] == '\\' && path[1] == '\\') ||
            (path[0] == '/'  && path[1] == '/' ))
        {
            return true;
        }
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief ファイルパスをフルパスに変換します。
//!
//! @param[in] path 元のファイルパスです。
//! @param[in] isUnix フォルダーの区切りを / にするなら true を指定します。
//!
//! @return フルパスを返します。
//-----------------------------------------------------------------------------
std::string RGetFullFilePath(const std::string& path, const bool isUnix)
{
    // 環境変数を展開します。
    std::string dst(path);
    if (dst.find('%') != std::string::npos)
    {
        char expandPath[MAX_PATH];
        if (::ExpandEnvironmentStringsA(dst.c_str(), expandPath, sizeof(expandPath)) != 0)
        {
            dst = expandPath;
        }
    }

    // フルパスに変換します。
    char fullPath[MAX_PATH];
    if (::PathSearchAndQualifyA(dst.c_str(), fullPath, sizeof(fullPath)))
    {
        dst = fullPath;
    }

    // ドライブ文字を大文字にします。
    if (dst.size() >= 2 && dst[1] == ':')
    {
        const char drive = dst[0];
        if ('a' <= drive && drive <= 'z')
        {
            dst[0] = drive - 'a' + 'A';
        }
    }

    return (isUnix) ? RGetUnixFilePath(dst) : dst;
}

//-----------------------------------------------------------------------------
//! @brief ファイルが存在なら true を返します。
//-----------------------------------------------------------------------------
bool RFileExists(const std::string& path)
{
    WIN32_FIND_DATA ffd;
    HANDLE  hFindFile = FindFirstFileA(path.c_str(), &ffd);
    if (hFindFile == INVALID_HANDLE_VALUE)
    {
        return false;
    }
    FindClose(hFindFile);

    if (!(ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) // check is file
    {
        return true;
    }

    return false;
}

//-----------------------------------------------------------------------------
//! @brief フォルダーが存在なら true を返します。
//!        パスの最後に \ または / が付いていても付いていなくても構いません。
//-----------------------------------------------------------------------------
bool RFolderExists(const std::string& path)
{
    std::string unixPath = RGetUnixFilePath(path);
    int len = static_cast<int>(unixPath.length());
    if (len == 0)
    {
        return false;
    }

    while (len > 0)
    {
        if (unixPath[len - 1] != '/')
        {
            break;
        }
        --len;
    }
    string noSlashPath = string(unixPath.c_str(), len);
    //if (len != strlen(path))
    //  cerr << "no slash: " << path.c_str() << " -> " << noSlashPath.c_str() << endl;

    WIN32_FIND_DATA ffd;
    HANDLE  hFindFile = FindFirstFileA(noSlashPath.c_str(), &ffd);
    if (hFindFile == INVALID_HANDLE_VALUE)
    {
        return false;
    }
    FindClose(hFindFile);

    if (ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) // check is folder
    {
        return true;
    }

    return false;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイル用出力ストリームのフォーマットを初期化します。
//-----------------------------------------------------------------------------
void RInitOutStreamFormat(std::ostream& os)
{
    //-----------------------------------------------------------------------------
    // set format
    os.setf(ios_base::dec, ios_base::basefield);
    os.fill(' ');
    os.setf(ios_base::left, ios_base::adjustfield);
    os.setf(ios_base::uppercase); // 浮動小数の e を大文字に
    #if (_MSC_VER < 1900)
    _set_output_format(_TWO_DIGIT_EXPONENT); // 浮動小数の指数部を 2 桁に
        // VS2015 ではデフォルトで 2 桁です。
    #endif
    os.precision(R_FLOAT_OUT_PRECISION);
    //os.setf(ios_base::showpoint);
    //os.setf(ios_base::fixed, ios_base::floatfield);
    //os.setf(ios_base::scientific, ios_base::floatfield);
    //os.setf(ios_base::fmtflags(0), ios_base::floatfield); // default
}

//-----------------------------------------------------------------------------
//! @brief UTF-8 の BOM を出力します。
//-----------------------------------------------------------------------------
void ROutUtf8Bom(std::ostream& os)
{
    static const unsigned char utf8BOM[] = { 0xef, 0xbb, 0xbf };
    os.write(reinterpret_cast<const char*>(utf8BOM), 3);
}

//-----------------------------------------------------------------------------
//! @brief UTF-16 の BOM を出力します。
//-----------------------------------------------------------------------------
void ROutUtf16Bom(std::ostream& os)
{
    static const unsigned char utf16BOM[] = { 0xff, 0xfe };
    os.write(reinterpret_cast<const char*>(utf16BOM), 2);
}

//-----------------------------------------------------------------------------
//! @brief Shift JIS 文字列を UTF-8 文字列に変換して出力します。
//-----------------------------------------------------------------------------
void ROutUtf8FromShiftJis(std::ostream& os, const std::string& sjisStr)
{
    // UTF-8 の複数バイトコードは 0x80 から 0xfd までの範囲で表現されるので、
    // os はテキストモードでも改行の問題はない
    std::string utf8Str = RGetUtf8FromShiftJis(sjisStr);
    if (utf8Str.size() > 0)
    {
        os.write(utf8Str.c_str(), static_cast<int>(utf8Str.size()));
    }
}

//-----------------------------------------------------------------------------
//! @brief 指定した数のタブ文字を出力します。
//-----------------------------------------------------------------------------
const char* RTab(const int count)
{
    const int TABS_MAX = 16;
    static const char* tabs = "\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t\t"; // TABS_MAX 個

    if (count == 0)
    {
        return "";
    }
    else
    {
        return tabs + (TABS_MAX - count);
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの bool 型用文字列を出力します。
//-----------------------------------------------------------------------------
const char* RBoolStr(const bool value)
{
    return (value) ? "true" : "false";
}

//-----------------------------------------------------------------------------
//! @brief 環境変数の値を取得します。
//-----------------------------------------------------------------------------
std::string RGetEnvVar(const char* name)
{
    std::string ret;
    char* pValue;
    size_t length;
    if (_dupenv_s(&pValue, &length, name) == 0)
    {
        // 環境変数が見つからない場合は、pValue が NULL に
        // length が 0、戻り値が 0（成功）となります。
        if (pValue != NULL)
        {
            ret = pValue;
            free(pValue);
        }
    }
    return ret;
}

//-----------------------------------------------------------------------------
//! @brief 環境変数が定義されていて値が 0 でなければ true を返します。
//-----------------------------------------------------------------------------
bool RIsEnvVarNotZero(const char* name)
{
    const std::string value = RGetEnvVar(name);
    return (!value.empty() && atoi(value.c_str()) != 0);
}

//-----------------------------------------------------------------------------
//! @brief 指定されたモジュールを含む実行ファイルのフルパスを返します。
//-----------------------------------------------------------------------------
std::string RGetModuleFileName(HMODULE hModule)
{
    char processPathBuf[MAX_PATH];
    if (GetModuleFileNameA(hModule, processPathBuf, MAX_PATH))
    {
        return std::string(processPathBuf);
    }
    else
    {
        return std::string();
    }
}

//-----------------------------------------------------------------------------
//! @brief Windows の DWORD 型のレジストリ値を取得します。
//-----------------------------------------------------------------------------
DWORD RGetWinRegDword(
    HKEY hParentKey,
    const char* lpSubKey,
    const char* lpValueName,
    const DWORD defaultValue
)
{
    DWORD value = defaultValue;
    HKEY hKey;
    if (::RegOpenKeyExA(hParentKey, lpSubKey, 0, KEY_READ, &hKey) == ERROR_SUCCESS)
    {
        DWORD dwSize = sizeof(value);
        if (::RegQueryValueExA(hKey, lpValueName, NULL, NULL,
            reinterpret_cast<LPBYTE>(&value), &dwSize) == ERROR_SUCCESS)
        {
            // success
        }
        ::RegCloseKey(hKey);
    }
    return value;
}

//-----------------------------------------------------------------------------
//! @brief Windows の文字列型のレジストリ値を取得します。
//-----------------------------------------------------------------------------
std::string RGetWinRegString(
    HKEY hParentKey,
    const char* lpSubKey,
    const char* lpValueName,
    const char* defaultValue
)
{
    std::string value = defaultValue;
    HKEY hKey;
    if (::RegOpenKeyExA(hParentKey, lpSubKey, 0, KEY_READ, &hKey) == ERROR_SUCCESS)
    {
        DWORD dwSize;
        if (::RegQueryValueExA(hKey, lpValueName, NULL, NULL, NULL, &dwSize) == ERROR_SUCCESS)
        {
            char* strBuf = new char[dwSize + 1];
            if (::RegQueryValueExA(hKey, lpValueName, NULL, NULL,
                reinterpret_cast<LPBYTE>(strBuf), &dwSize) == ERROR_SUCCESS)
            {
                value = strBuf;
            }
            delete[] strBuf;
        }
        ::RegCloseKey(hKey);
    }
    return value;
}

//-----------------------------------------------------------------------------
//! @brief XML 要素のエラーを表示します。
//-----------------------------------------------------------------------------
void RXMLElement::DisplayError(const std::string& message) const
{
    if (m_ErrorFunc != NULL)
    {
        m_ErrorFunc(message.c_str(), m_pErrorUserData);
    }
    else
    {
        cerr << message << endl;
    }
}

//-----------------------------------------------------------------------------
//! @brief XML 要素の属性を解析してノードに記録します。
//-----------------------------------------------------------------------------
int RXMLElement::LoadAttributes(const char* src, const char end)
{
    int pos = 0;
    for (;;)
    {
        pos += static_cast<int>(strspn(&src[pos], " \t\r\n"));
        if (src[pos] == '>' || src[pos] == end)
        {
            break;
        }
        int namelen = static_cast<int>(strcspn(&src[pos], "="));
        std::string attrName(&src[pos], namelen);
        pos += namelen;
        std::string value;
        if (src[pos] == '=')
        {
            ++pos;
            if (src[pos] == '\"' || src[pos] == '\'')
            {
                char end2[2] = { src[pos], };
                ++pos;
                int valuelen = static_cast<int>(strcspn(&src[pos], end2));
                value = std::string(&src[pos], valuelen);
                pos += valuelen;
                pos += (src[pos] != '\0');
            }
            else
            {
                char end2[] = " \t\r\n";
                int valuelen = static_cast<int>(strcspn(&src[pos], end2));
                value = std::string(&src[pos], valuelen);
                pos += valuelen;
            }
        }
        attributes[attrName] = value;
    }
    return pos;
}

//-----------------------------------------------------------------------------
//! @brief XML 要素の代入演算子です。
//-----------------------------------------------------------------------------
RXMLElement& RXMLElement::operator=(const RXMLElement& r)
{
    name = r.name;
    text = r.text;
    attributes.clear();
    for (AttrMap::const_iterator it = r.attributes.begin(); it != r.attributes.end(); ++it)
    {
        attributes[it->first] = it->second;
    }
    nodes.clear();
    nodes.resize(r.nodes.size());
    for (size_t i = 0; i < r.nodes.size(); ++i)
    {
        nodes[i] = r.nodes[i];
    }

    m_ErrorFunc = r.m_ErrorFunc;
    m_pErrorUserData = r.m_pErrorUserData;

    return *this;
}

//-----------------------------------------------------------------------------
//! @brief XML 要素の指定のパスに該当する要素を指定個数分検索します。
//-----------------------------------------------------------------------------
void RXMLElement::FindElements(
    std::vector<const RXMLElement*>& list,
    const std::string& path,
    const bool showError,
    const int maxCount
) const
{
    // パス名を階層に区切ります。
    RStringArray entries;
    std::string subPath(path);
    for (size_t ic = 0; ic < subPath.size(); ++ic)
    {
        if (subPath[ic] == '\\')
        {
            subPath[ic] = '/';
        }
    }
    while (!subPath.empty())
    {
        int span = static_cast<int>(strcspn(subPath.c_str(), "/"));
        entries.push_back(subPath.substr(0, span));
        span += (subPath[span] == '/');
        subPath = subPath.substr(span);
    }
    const int entryCount = static_cast<int>(entries.size());

    // 階層の上から順に探索します（一意性が無いので総当たり）。
    int count = maxCount;
    std::vector<const RXMLElement*> stack(entryCount);
    std::vector<int> index(entryCount);
    int level = 0;
    index[0] = 0;
    stack[0] = this;
    const RXMLElement* node = this;
    int pos = 0;
    for (;;)
    {
        bool found = false;
        if (level >= entryCount)
        {
            // 全階層マッチしたら現在位置を記録します。
            list.push_back(node);
            if (--count <= 0)
            {
                // 指定の個数が見つかればここで終了です。
                break;
            }
        }
        else
        {
            // そうでなければ現在の階層を探索します。
            for (; pos < static_cast<int>(node->nodes.size()); ++pos)
            {
                if (node->nodes[pos].name == entries[level])
                {
                    found = true;
                    break;
                }
            }
        }
        if (found)
        {
            // マッチするノードがあればさらに下層へ。
            // 失敗時のために現在位置を退避します。
            index[level] = pos + 1;
            stack[level] = node;
            ++level;
            node = &node->nodes[pos];
            pos = 0;
        }
        else if (--level >= 0)
        {
            // 該当ノードが無ければスタックを戻ります。
            node = stack[level];
            pos = index[level];
        }
        else // 全要素の探索が終了した。
        {
            break;
        }
    }

    if (showError && list.size() == 0)
    {
        DisplayError("Element is not found: <" + path + ">");
    }
}

//-----------------------------------------------------------------------------
//! @brief XML 文書を解析してツリーを構築します（ASCII テキスト限定）。
//-----------------------------------------------------------------------------
void RXMLElement::LoadDocument(RXMLElement* decl, const std::string& xmlDoc)
{
    RXMLElement dummy;
    decl = (decl != NULL) ? decl : &dummy;
    const char* src = xmlDoc.c_str();
    int len = static_cast<int>(xmlDoc.size());
    if (static_cast<uint8_t>(src[0]) == 0xef)
    {
        // BOM を読み捨てます。
        src += 3;
        len -= 3;
    }
    std::vector<RXMLElement*> stack;
    RXMLElement* node = this;
    stack.push_back(node);
    for (int pos = 0; pos < len; )
    {
        if (src[pos] == '\0')
        {
            break;
        }
        const int span = static_cast<int>(strcspn(&src[pos], "<"));
        node->text += std::string(&src[pos], span);
        pos += span;
        if (src[pos] == '<')
        {
            ++pos;
            if (src[pos] == '?') // 宣言要素。
            {
                ++pos;
                int namelen = static_cast<int>(strcspn(&src[pos], " \t\r\n?"));
                std::string nodeName(&src[pos], namelen);
                pos += namelen;
                pos += decl->LoadAttributes(&src[pos], '?');
                pos += (src[pos] == '?');
            }
            else // 開始・終了タグ。
            {
                bool endtag = (src[pos] == '/');
                pos += endtag;
                int namelen = static_cast<int>(strcspn(&src[pos], " \t\r\n/>"));
                std::string nodeName(&src[pos], namelen);
                pos += namelen;
                if (!endtag) // 開始タグ。
                {
                    node = node->CreateElement();
                    node->name = nodeName;
                    stack.push_back(node);
                    pos += node->LoadAttributes(&src[pos], '/');
                    endtag = (src[pos] == '/');
                    pos += endtag;
                }
                if (endtag) // 終了タグ。
                {
                    for (int i = static_cast<int>(stack.size()) - 1; i >= 0; --i)
                    {
                        if (stack[i]->name == nodeName)
                        {
                            if (i > 0 && stack[i] == node)
                            {
                                node = stack[i - 1];
                            }
                            stack.erase(stack.end() - 1);
                            break;
                        }
                    }
                }
                pos += static_cast<int>(strcspn(&src[pos], ">"));
            }
            pos += (src[pos] == '>');
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief XML 文書を取得します。
//-----------------------------------------------------------------------------
std::string RXMLElement::GetDocument(const int tabCount, const char* lf) const
{
    const int& tc = tabCount;
    std::string doc;
    const std::string txt = RTrimString(text);
    if (name.empty())
    {
        doc += std::string("<?xml version=\"1.0\" encoding=\"utf-8\"?>") + lf;
        for (size_t iChild = 0; iChild < nodes.size(); ++iChild)
        {
            doc += nodes[iChild].GetDocument(tc, lf);
        }
    }
    else
    {
        doc += std::string(RTab(tc)) + "<" + name;
        for (AttrMap::const_iterator it = attributes.begin(); it != attributes.end(); ++it)
        {
            doc += " " + it->first + "=\"" + it->second + "\"";
        }
        if (txt.empty() && nodes.size() == 0)
        {
            doc += std::string(" />") + lf;
        }
        else
        {
            doc += std::string(">") + lf;
            if (!txt.empty())
            {
                doc += RTab(tc + 1) + txt + lf;
            }
            for (size_t iChild = 0; iChild < nodes.size(); ++iChild)
            {
                doc += nodes[iChild].GetDocument(tc + 1, lf);
            }
            doc += std::string(RTab(tc)) + "</" + name + ">" + lf;
        }
    }
    return doc;
}

//-----------------------------------------------------------------------------
//! @brief NintendoSDK のルートフォルダーのパスを返します。
//-----------------------------------------------------------------------------
static std::string GetNintendoSdkRootPath()
{
    std::string rootPath = RGetEnvVar("NINTENDO_SDK_ROOT");
    if (!rootPath.empty() && rootPath[rootPath.size() - 1] != '\\')
    {
        rootPath += '\\';
    }
    return rootPath;
}

//-----------------------------------------------------------------------------
//! @brief 従来の NintendoWare のルートフォルダーのパスを返します。
//-----------------------------------------------------------------------------
static std::string GetOldNintendoWareRootPath()
{
    const std::string root = RGetEnvVar("NW4F_G3D_ROOT");
    return (!root.empty()) ? root : RGetEnvVar("NW4F_ROOT");
}

//-----------------------------------------------------------------------------
//! @brief Nintendo Photoshop プラグインのルートフォルダーのパスを返します。
//-----------------------------------------------------------------------------
std::string RGetNintendoPhotoshopRootPath()
{
    HINSTANCE hDll = GetDLLInstance();
    const std::string dllPath = RGetModuleFileName(hDll);
    return RGetFullFilePath(RGetFolderFromFilePath(dllPath) + "\\..\\", false);
}

//-----------------------------------------------------------------------------
//! @brief Nintendo Photoshop プラグインが使用するグラフィックスツールフォルダーのパスを返します。
//-----------------------------------------------------------------------------
std::string RGetNintendoPhotoshopGraphicsToolsPath()
{
    std::string rootPath = RGetNintendoPhotoshopRootPath();

    std::string gfxToolsPath = RGetFullFilePath(rootPath + "..\\GraphicsTools\\", false);
    if (RFolderExists(gfxToolsPath))
    {
        return gfxToolsPath;
    }

    gfxToolsPath = RGetFullFilePath(rootPath + "..\\..\\G3dTool\\", false) + "win64\\";
    if (RFolderExists(gfxToolsPath))
    {
        return gfxToolsPath;
    }

    gfxToolsPath = GetNintendoSdkRootPath() + "Tools\\Graphics\\GraphicsTools\\";
    return gfxToolsPath;
}

//-----------------------------------------------------------------------------
//! @brief Nintendo Photoshop プラグインが使用する 3D ツールフォルダーのパスを返します。
//-----------------------------------------------------------------------------
std::string RGetNintendoPhotoshop3dToolsPath()
{
    std::string rootPath = RGetNintendoPhotoshopRootPath();

    std::string g3dToolPath = RGetFullFilePath(rootPath + "..\\3dTools\\", false);
    if (RFolderExists(g3dToolPath))
    {
        return g3dToolPath;
    }

    g3dToolPath = RGetFullFilePath(rootPath + "..\\..\\G3dTool\\", false) + "win64\\";
    if (RFolderExists(g3dToolPath))
    {
        return g3dToolPath;
    }

    g3dToolPath = GetNintendoSdkRootPath() + "Tools\\Graphics\\3dTools\\";
    return g3dToolPath;
}

//-----------------------------------------------------------------------------
//! @brief Nintendo Photoshop プラグインのコンフィグフォルダーのパスを返します。
//-----------------------------------------------------------------------------
std::string RGetConfigFolderPath(const bool getsCofigEnv)
{
    std::string path = RGetEnvVar("NINTENDO_PHOTOSHOP_CONFIG");
    if (path.empty())
    {
        path = RGetEnvVar("NW4F_PHOTOSHOP_CONFIG");
    }

    if (!path.empty() && getsCofigEnv)
    {
        path = RGetUnixFilePath(path);
        if (path.find_last_of('/') != path.length() - 1) // 最後にスラッシュがなければ追加します。
        {
            path += "/";
        }
    }
    else
    {
        const std::string rootPath = RGetUnixFilePath(RGetNintendoPhotoshopRootPath());
        path = rootPath + "Config/";
        const std::string scriptsPath = rootPath + "NintendoPhotoshopScripts/";
        // Config というフォルダーは偶然存在する可能性があるので、スクリプトフォルダーの存在もチェックします。
        if (!RFolderExists(path) || !RFolderExists(scriptsPath))
        {
            const std::string oldNwRootPath = GetOldNintendoWareRootPath();
            if (!oldNwRootPath.empty())
            {
                path = RGetUnixFilePath(oldNwRootPath) +
                    "/Tool/DccPlugin/Photoshop/Config/";
            }
        }
    }
    return path;
}

//-----------------------------------------------------------------------------
//! @brief コンフィグ名からコンフィグファイルのパスを取得します。
//!
//! @param[in] configName コンフィグ名です。
//! @param[in] configFolderPath コンフィグフォルダーのパスです。
//!
//! @return コンフィグファイルのパスを返します。
//-----------------------------------------------------------------------------
std::string RGetConfigPathFromConfigName(
    const std::string& configName,
    const std::string& configFolderPath
)
{
    const std::string fileName = (configName.empty() || configName == R_STANDARD_CONFIG_NAME) ?
        STANDARD_CONFIG_FILE_NAME :
        CONFIG_FILE_PREFIX + configName;
    return configFolderPath + fileName + ".jsxinc";
}

//-----------------------------------------------------------------------------
//! @brief コンフィグファイルのパスからコンフィグ名を抽出して返します。
//!        パスが不正なら空文字を返します。
//!
//! @param[in] configPath コンフィグファイルのパスです。
//!
//! @return コンフィグ名を返します。
//-----------------------------------------------------------------------------
static std::string GetConfigNameFromConfigPath(const std::string& configPath)
{
    std::string configName;
    if (RGetExtensionFromFilePath(configPath) == "jsxinc")
    {
        const std::string fileName = RGetNoExtensionFilePath(RGetFileNameFromFilePath(configPath));
        if (fileName == STANDARD_CONFIG_FILE_NAME)
        {
            return R_STANDARD_CONFIG_NAME;
        }
        else if (fileName.find(CONFIG_FILE_PREFIX) == 0)
        {
            configName = fileName.substr(strlen(CONFIG_FILE_PREFIX));
        }
    }
    return configName;
}

//-----------------------------------------------------------------------------
//! @brief コンフィグ名一覧を取得します。
//-----------------------------------------------------------------------------
void RGetAllConfigNames(RStringArray& allConfigNames)
{
    allConfigNames.clear();
    allConfigNames.push_back(std::string(R_STANDARD_CONFIG_NAME));

    const std::string configFolderPath = RGetConfigFolderPath(true);
    WIN32_FIND_DATA ffd;
    HANDLE  hFindFile = FindFirstFile((configFolderPath + "*.jsxinc").c_str(), &ffd);
    if (hFindFile != INVALID_HANDLE_VALUE)
    {
        do
        {
            if (!(ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))
            {
                const std::string configName = GetConfigNameFromConfigPath(ffd.cFileName);
                if (!configName.empty()                  &&
                    configName != R_STANDARD_CONFIG_NAME &&
                    configName != "Sample"               &&
                    RIsValidPresetNameString(configName))
                {
                    allConfigNames.push_back(configName);
                }
            }
        } while (FindNextFile(hFindFile, &ffd));
        FindClose(hFindFile);
    }
}

//-----------------------------------------------------------------------------
//! @brief コンフィグファイルをリードして、内容からコメントを除いた文字列を取得します。
//-----------------------------------------------------------------------------
std::string RReadConfigFile(const std::string& configPath)
{
    std::string configStr;
    if (RFileExists(configPath))
    {
        RFileBuf configFile(configPath);
        const bool isUtf8 = configFile.IsUtf8();
        const uint32_t readOfs = (isUtf8) ? 3 : 0;
        configStr = std::string(reinterpret_cast<const char*>(configFile.GetBuf()) + readOfs,
            configFile.GetSize() - readOfs);
        if (isUtf8)
        {
            configStr = RGetShiftJisFromUtf8(configStr);
        }

        // ExtendScript のコメントをカットします。
        configStr = RCutCSourceComment(configStr);
    }
    return configStr;
}

//-----------------------------------------------------------------------------
//! @brief コンフィグファイルの変数名に使用可能な文字なら true を返します。
//!
//! @param[in] c 文字です。
//!
//! @return コンフィグファイルの変数名に使用可能な文字なら true を返します。
//-----------------------------------------------------------------------------
static bool IsConfigVariableCharacter(const char c)
{
    return (c == '_' ||
        ('0' <= c && c <= '9') ||
        ('A' <= c && c <= 'Z') ||
        ('a' <= c && c <= 'z'));
}

//-----------------------------------------------------------------------------
//! @brief コンフィグファイルの指定した位置の前の単語を取得します。
//!
//! @param[in] configStr コンフィグファイルをリードした文字列です。
//! @param[in] index 文字列中の位置です。
//!
//! @return 指定した位置の前の単語を返します。
//-----------------------------------------------------------------------------
static const std::string g_ConfigDelims = " \t\r\n";

static std::string GetConfigPrevWord(const std::string& configStr, const size_t index)
{
    size_t endIdx = index;
    while (endIdx > 0 && g_ConfigDelims.find(configStr[endIdx - 1]) == std::string::npos)
    {
        --endIdx;
    }
    while (endIdx > 0 && g_ConfigDelims.find(configStr[endIdx - 1]) != std::string::npos)
    {
        --endIdx;
    }
    size_t startIdx = endIdx;
    while (startIdx > 0 && g_ConfigDelims.find(configStr[startIdx - 1]) == std::string::npos)
    {
        --startIdx;
    }
    return (startIdx < endIdx) ? configStr.substr(startIdx, endIdx - startIdx) : std::string();
}

//-----------------------------------------------------------------------------
//! @brief コンフィグファイルから変数名を検索します。
//!
//! @param[in] configStr コンフィグファイルをリードした文字列です。
//! @param[in] name 変数名です。
//!
//! @return 変数名の先頭位置（見つからなければ std::string::npos）を返します。
//-----------------------------------------------------------------------------
size_t RFindConfigVariable(const std::string& configStr, const std::string& name)
{
    size_t varIdx = configStr.rfind(name); // 最後の定義が優先なので末尾から検索します。
    while (varIdx != std::string::npos)
    {
        const size_t nextIdx = varIdx + name.size();
        const char prevC = (varIdx >= 1               ) ? configStr[varIdx - 1] : ' ';
        const char nextC = (nextIdx < configStr.size()) ? configStr[nextIdx   ] : ' ';
        if (!IsConfigVariableCharacter(prevC) &&
            !IsConfigVariableCharacter(nextC))
        {
            // 変数名の直後の文字が = かつ直前の単語が var なら変数とみなします。
            const size_t equalIdx = configStr.find_first_not_of(g_ConfigDelims, nextIdx);
            if (equalIdx != std::string::npos && configStr[equalIdx] == '=')
            {
                const std::string prevWord = GetConfigPrevWord(configStr, varIdx);
                if (prevWord == "var")
                {
                    break;
                }
            }
        }
        varIdx = (varIdx == 0) ? std::string::npos : configStr.rfind(name, varIdx - 1);
    }
    return varIdx;
}

//-----------------------------------------------------------------------------
//! @brief コンフィグファイルからブーリアン型変数の値を取得します。
//-----------------------------------------------------------------------------
bool RGetConfigBooleanValue(
    const std::string& configStr,
    const std::string& name,
    const bool altValue
)
{
    bool value = altValue;
    const size_t varIdx = RFindConfigVariable(configStr, name);
    if (varIdx != std::string::npos)
    {
        const size_t equalIdx = configStr.find("=", varIdx);
        if (equalIdx != std::string::npos)
        {
            const size_t startIdx = equalIdx + 1;
            const size_t endIdx = configStr.find(";", startIdx);
            if (endIdx != std::string::npos)
            {
                const std::string valueStr = RTrimString(
                    configStr.substr(startIdx, endIdx - startIdx));
                value = (valueStr == "true" ) ? true  :
                        (valueStr == "false") ? false :
                        altValue;
            }
        }
    }
    return value;
}

//-----------------------------------------------------------------------------
//! @brief コンフィグファイルから文字列型変数の値を取得します。
//-----------------------------------------------------------------------------
std::string RGetConfigStringValue(
    const std::string& configStr,
    const std::string& name,
    const std::string& altValue
)
{
    std::string value = altValue;
    const size_t varIdx = RFindConfigVariable(configStr, name);
    if (varIdx != std::string::npos)
    {
        const size_t equalIdx = configStr.find("=", varIdx);
        if (equalIdx != std::string::npos)
        {
            const size_t startIdx = equalIdx + 1;
            const size_t endIdx = configStr.find(";", startIdx);
            if (endIdx != std::string::npos)
            {
                value = configStr.substr(startIdx, endIdx - startIdx);
                const size_t sqIdx = value.find('\'');
                const size_t dqIdx = value.find('\"');
                const char quote = (sqIdx != std::string::npos && (dqIdx == std::string::npos || sqIdx < dqIdx)) ?
                    '\'' : '\"';
                value = RDequoteString(value, quote);
            }
        }
    }
    return value;
}

//-----------------------------------------------------------------------------
//! @brief コンフィグファイルのオブジェクト定義から値の配列を取得します。
//!
//! @param[out] pValues 値の配列へのポインタです。
//!                     オブジェクトの場合 { key0, value0, key1, value1, ... } の形式で格納します。
//! @param[in] configStr コンフィグファイルをリードした文字列です。
//! @param[in] varIdx 変数名の先頭位置です。
//! @param[in] isArray 配列なら true、オブジェクトなら false を指定します。
//!
//! @return 値の数を返します。
//-----------------------------------------------------------------------------
int RGetConfigObjectValues(
    RStringArray* pValues,
    const std::string& configStr,
    const size_t varIdx,
    const bool isArray
)
{
    pValues->clear();

    const size_t startIdx = configStr.find((isArray) ? "[" : "{", varIdx);
    if (startIdx == std::string::npos)
    {
        return 0;
    }
    const size_t endIdx = configStr.find((isArray) ? "]" : "}", startIdx + 1);
    if (endIdx == std::string::npos)
    {
        return 0;
    }
    const std::string valuesStr = configStr.substr(startIdx + 1, endIdx - (startIdx + 1));

    size_t curIdx = 0;
    while (curIdx < valuesStr.size())
    {
        const size_t quoteIdxS = valuesStr.find("\"", curIdx);
        if (quoteIdxS == std::string::npos)
        {
            break;
        }
        const size_t quoteIdxE = valuesStr.find("\"", quoteIdxS + 1);
        if (quoteIdxE == std::string::npos)
        {
            break;
        }
        pValues->push_back(valuesStr.substr(quoteIdxS + 1, quoteIdxE - (quoteIdxS + 1)));
        curIdx = quoteIdxE + 1;
    }

    if (!isArray && (pValues->size() % 2) != 0)
    {
        pValues->clear();
    }
    return static_cast<int>(pValues->size());
}

//-----------------------------------------------------------------------------
//! @brief コンフィグファイルから重み付け圧縮有効フラグを取得します。
//-----------------------------------------------------------------------------
bool RGetEnablesWeightedCompress(const std::string& configStr)
{
    return RGetConfigBooleanValue(configStr, "enable_weighted_compress", false);
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルバージョンを整数（major * 100 + minor）で返します。
//-----------------------------------------------------------------------------
int RGet3dIntermediateFileVersionAsInt(const std::string& value)
{
    int ver = 0;
    RStringArray tokens;
    if (RTokenizeString(tokens, value, ".") >= 2)
    {
        ver = atoi(tokens[0].c_str()) * 100 + atoi(tokens[1].c_str());
    }
    return ver;
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルのプリセット名に使用できる文字列なら true を返します。
//-----------------------------------------------------------------------------
bool RIsValidPresetNameString(const std::string& name)
{
    if (!RIsAsciiString(name))
    {
        return false;
    }
    else
    {
        for (size_t ic = 0; ic < name.length(); ++ic)
        {
            const char c = name[ic];
            if (c == ';'  ||
                c == ':'  ||
                c == '/'  ||
                c == '\\' ||
                c == '*'  ||
                c == '?'  ||
                c == '<'  ||
                c == '>'  ||
                c == '|'  ||
                c == '\"')
            {
                return false;
            }
        }
        return true;
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの先頭からバイナリーデータまでのオフセットを取得します。
//-----------------------------------------------------------------------------
size_t RGetBinaryOffset(const void* fileBuf, const size_t fileSize)
{
    size_t binOfs = fileSize;
    const uint8_t* buf = reinterpret_cast<const uint8_t*>(fileBuf);
    for (size_t iBuf = 0; iBuf < fileSize; ++iBuf)
    {
        if (buf[iBuf] == 0x00)
        {
            binOfs = RAlignValue(iBuf + 1, R_BINARY_ALIGNMENT);
            break;
        }
    }
    return binOfs;
}

//-----------------------------------------------------------------------------
//! @brief バイナリーデータ列へのポインタを取得します（static 関数）。
//-----------------------------------------------------------------------------
const RBinDataStream* RBinDataStream::Get(const void* binTop, const int streamIdx)
{
    const RBinDataStream* streamTop = NULL;
    const uint8_t* src = reinterpret_cast<const uint8_t*>(binTop);
    src += 8; // "g3dstrma"
    const int streamCount = *reinterpret_cast<const int*>(src);
    src += sizeof(int);
    if (streamIdx < streamCount)
    {
        src += (4 + 4) * streamIdx; // address, size
        const int streamAddr = *reinterpret_cast<const int*>(src);
        src = reinterpret_cast<const uint8_t*>(binTop) + streamAddr;
        streamTop = reinterpret_cast<const RBinDataStream*>(src);
    }
    return streamTop;
}

//-----------------------------------------------------------------------------
//! @brief ユーザーデータのコンストラクタです。
//-----------------------------------------------------------------------------
RUserData::RUserData(
    const RXMLElement* userDataElem,
    const RXMLElement* streamsElem,
    const void* binTop
)
{
    m_Name = RGetShiftJisFromUtf8(userDataElem->GetAttribute("name"));

    const RXMLElement* valueElem = &(userDataElem->nodes[0]);
    if (valueElem->name == "user_float" || valueElem->name == "user_int")
    {
        //-----------------------------------------------------------------------------
        // 数値型ユーザーデータを解析します。
        m_Type = (valueElem->name == "user_float") ? FLOAT : INT;
        m_NumberValueCount = atoi(valueElem->GetAttribute("count").c_str());
        if (m_NumberValueCount > 0)
        {
            m_NumberValuesText = valueElem->text;
        }
    }
    else if (valueElem->name == "user_string" || valueElem->name == "user_wstring")
    {
        //-----------------------------------------------------------------------------
        // 文字列型ユーザーデータを解析します。
        m_Type = (valueElem->name == "user_string") ? STRING : WSTRING;
        for (size_t iString = 0; iString < valueElem->nodes.size(); ++iString)
        {
            const RXMLElement* stringElem = &(valueElem->nodes[iString]);
            m_StringValues.push_back(
                RDecodeXmlString(RGetShiftJisFromUtf8(stringElem->text)));
        }
    }
    else
    {
        //-----------------------------------------------------------------------------
        // ストリーム型ユーザーデータを解析します。
        m_Type = STREAM;
        const int iStream = atoi(valueElem->GetAttribute("stream_index").c_str());
        if (iStream == -1) // データなし
        {

        }
        else if (streamsElem != NULL)
        {
            const RXMLElement* streamElem = &(streamsElem->nodes[iStream]);
            const int dataCount = atoi(streamElem->GetAttribute("count").c_str());
            m_StreamValues.resize(dataCount);
            int iData = 0;
            std::istringstream iss(streamElem->text.c_str());
            while (iss && iData < dataCount)
            {
                std::string value;
                iss >> value;
                if (!value.empty())
                {
                    m_StreamValues[iData++] = static_cast<uint8_t>(strtol(value.c_str(), NULL, 16));
                }
            }
        }
        else
        {
            const RBinDataStream* binStream = RBinDataStream::Get(binTop, iStream);
            m_StreamValues.resize(binStream->m_Count);
            const unsigned char* pSrc = reinterpret_cast<const unsigned char*>(binStream->m_Data);
            for (int iData = 0; iData < binStream->m_Count; ++iData)
            {
                m_StreamValues[iData] = *pSrc++;
            }
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ファイルバッファにファイルをリードします。
//-----------------------------------------------------------------------------
bool RFileBuf::Read(const std::string& path)
{
    RFreeAndClearArray(m_FileBuf);

    m_FilePath = path;
    m_FileSize = 0;

    std::ifstream ifs(path.c_str(), ios_base::in | ios_base::binary);
    if (ifs)
    {
        //-----------------------------------------------------------------------------
        // read file
        ifs.seekg(0L, ios_base::end);
        m_FileSize = static_cast<size_t>(ifs.tellg());
        ifs.seekg(0L, ios_base::beg);
        if (m_FileSize > 0)
        {
            m_FileBuf = new uint8_t[m_FileSize];
            ifs.read(reinterpret_cast<char*>(m_FileBuf), m_FileSize);
        }
        else
        {
            m_FileBuf = new uint8_t[1];
        }

        //-----------------------------------------------------------------------------
        // close file
        ifs.close();
        return true;
    }
    else
    {
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの文字列からリニア変換フラグを取得します。
//-----------------------------------------------------------------------------
int RGetLinearFlagFromString(const std::string& str)
{
    RStringArray tokens;
    if (RTokenizeString(tokens, str) == R_RGBA_COUNT)
    {
        int linearFlag = RLinearFlag::LINEAR_NONE;
        for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
        {
            if (tokens[iRgba] == "true")
            {
                linearFlag |= s_LinearRgbaFlags[iRgba];
            }
        }
        return linearFlag;
    }
    return RLinearFlag::LINEAR_NONE;
}

//-----------------------------------------------------------------------------
//! @brief オプション文字列からリニア変換フラグを取得します。
//-----------------------------------------------------------------------------
int RGetLinearFlagFromOptString(const std::string& str)
{
    if (str.size() == R_RGBA_COUNT)
    {
        int linearFlag = RLinearFlag::LINEAR_NONE;
        for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
        {
            const char c = str[iRgba];
            if (c == '0')
            {

            }
            else if (c == '1')
            {
                linearFlag |= s_LinearRgbaFlags[iRgba];
            }
            else
            {
                return RLinearFlag::LINEAR_INVALID;
            }
        }
        return linearFlag;
    }
    return RLinearFlag::LINEAR_INVALID;
}

//-----------------------------------------------------------------------------
//! @brief リニア変換フラグからオプション文字列を取得します。
//-----------------------------------------------------------------------------
std::string RGetOptStringFromLinearFlag(const int linearFlag)
{
    if (linearFlag < 0)
    {
        return "0000";
    }
    else
    {
        char strBuf[R_RGBA_COUNT];
        for (int iRgba = 0; iRgba < R_RGBA_COUNT; ++iRgba)
        {
            strBuf[iRgba] = ((linearFlag & s_LinearRgbaFlags[iRgba]) != 0) ? '1' : '0';
        }
        return std::string(strBuf, sizeof(strBuf));
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの文字列からテクスチャーの次元を取得します。
//-----------------------------------------------------------------------------
FtxDimension RGetTextureDimensionFromString(const std::string& str)
{
    return
        (str == "1d"           ) ? FtxDimension_1d                 :
        (str == "2d"           ) ? FtxDimension_2d                 :
        (str == "3d"           ) ? FtxDimension_3d                 :
        (str == "cube"         ) ? FtxDimension_CubeMap            :
        (str == "cube_array"   ) ? FtxDimension_CubeMapArray       :
        (str == "1d_array"     ) ? FtxDimension_1dArray            :
        (str == "2d_array"     ) ? FtxDimension_2dArray            :
        (str == "2d_msaa"      ) ? FtxDimension_2dMultisample      :
        (str == "2d_msaa_array") ? FtxDimension_2dMultisampleArray :
        FtxDimension_2d;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーの次元から中間ファイルの文字列を取得します。
//-----------------------------------------------------------------------------
std::string RGetTextureDimensionString(const FtxDimension dimension)
{
    switch (dimension)
    {
        case FtxDimension_1d:                 return "1d";
        case FtxDimension_2d:                 return "2d";
        case FtxDimension_3d:                 return "3d";
        case FtxDimension_CubeMap:            return "cube";
        case FtxDimension_CubeMapArray:       return "cube_array";
        case FtxDimension_1dArray:            return "1d_array";
        case FtxDimension_2dArray:            return "2d_array";
        case FtxDimension_2dMultisample:      return "2d_msaa";
        case FtxDimension_2dMultisampleArray: return "2d_msaa_array";
        default:                              return "invalid";
    }
}

//-----------------------------------------------------------------------------
//! @brief フォーマットと文字列のペアのデータです。
//-----------------------------------------------------------------------------
namespace {

const RIntAndCharPtr s_FormatStrings[] =
{
    { FtxFormat_Unorm_8      , "unorm_8"       },
    { FtxFormat_Snorm_8      , "snorm_8"       },
    { FtxFormat_Uint_8       , "uint_8"        },
    { FtxFormat_Sint_8       , "sint_8"        },
    { FtxFormat_Unorm_16     , "unorm_16"      },
    { FtxFormat_Snorm_16     , "snorm_16"      },
    { FtxFormat_Uint_16      , "uint_16"       },
    { FtxFormat_Sint_16      , "sint_16"       },
    { FtxFormat_Uint_32      , "uint_32"       },
    { FtxFormat_Sint_32      , "sint_32"       },

    { FtxFormat_Unorm_4_4    , "unorm_4_4"     },
    { FtxFormat_Unorm_8_8    , "unorm_8_8"     },
    { FtxFormat_Snorm_8_8    , "snorm_8_8"     },
    { FtxFormat_Uint_8_8     , "uint_8_8"      },
    { FtxFormat_Sint_8_8     , "sint_8_8"      },
    { FtxFormat_Unorm_16_16  , "unorm_16_16"   },
    { FtxFormat_Snorm_16_16  , "snorm_16_16"   },
    { FtxFormat_Uint_16_16   , "uint_16_16"    },
    { FtxFormat_Sint_16_16   , "sint_16_16"    },
    { FtxFormat_Uint_32_32   , "uint_32_32"    },
    { FtxFormat_Sint_32_32   , "sint_32_32"    },

    { FtxFormat_Unorm_5_6_5  , "unorm_5_6_5"   },

    { FtxFormat_Unorm_5_5_5_1, "unorm_5_5_5_1" },
    { FtxFormat_Unorm_4_4_4_4, "unorm_4_4_4_4" },

    { FtxFormat_Unorm_8_8_8_8    , "unorm_8_8_8_8"     },
    { FtxFormat_Snorm_8_8_8_8    , "snorm_8_8_8_8"     },
    { FtxFormat_Srgb_8_8_8_8     , "srgb_8_8_8_8"      },
    { FtxFormat_Uint_8_8_8_8     , "uint_8_8_8_8"      },
    { FtxFormat_Sint_8_8_8_8     , "sint_8_8_8_8"      },
    { FtxFormat_Unorm_10_10_10_2 , "unorm_10_10_10_2"  },
    { FtxFormat_Uint_10_10_10_2  , "uint_10_10_10_2"   },
    { FtxFormat_Unorm_16_16_16_16, "unorm_16_16_16_16" },
    { FtxFormat_Snorm_16_16_16_16, "snorm_16_16_16_16" },
    { FtxFormat_Uint_16_16_16_16 , "uint_16_16_16_16"  },
    { FtxFormat_Sint_16_16_16_16 , "sint_16_16_16_16"  },
    { FtxFormat_Uint_32_32_32_32 , "uint_32_32_32_32"  },
    { FtxFormat_Sint_32_32_32_32 , "sint_32_32_32_32"  },

    { FtxFormat_Float_16         , "float_16"          },
    { FtxFormat_Float_32         , "float_32"          },
    { FtxFormat_Float_16_16      , "float_16_16"       },
    { FtxFormat_Float_32_32      , "float_32_32"       },
    { FtxFormat_Float_11_11_10   , "float_11_11_10"    },
    { FtxFormat_Float_16_16_16_16, "float_16_16_16_16" },
    { FtxFormat_Float_32_32_32_32, "float_32_32_32_32" },

    { FtxFormat_Unorm_Bc1, "unorm_bc1" },
    { FtxFormat_Srgb_Bc1 , "srgb_bc1"  },
    { FtxFormat_Unorm_Bc2, "unorm_bc2" },
    { FtxFormat_Srgb_Bc2 , "srgb_bc2"  },
    { FtxFormat_Unorm_Bc3, "unorm_bc3" },
    { FtxFormat_Srgb_Bc3 , "srgb_bc3"  },
    { FtxFormat_Unorm_Bc4, "unorm_bc4" },
    { FtxFormat_Snorm_Bc4, "snorm_bc4" },
    { FtxFormat_Unorm_Bc5, "unorm_bc5" },
    { FtxFormat_Snorm_Bc5, "snorm_bc5" },

    { FtxFormat_Ufloat_Bc6, "ufloat_bc6" },
    { FtxFormat_Float_Bc6 , "float_bc6"  },
    { FtxFormat_Unorm_Bc7 , "unorm_bc7"  },
    { FtxFormat_Srgb_Bc7  , "srgb_bc7"   },

    { FtxFormat_Unorm_Etc1      , "unorm_etc1"       },
    { FtxFormat_Unorm_Etc2      , "unorm_etc2"       },
    { FtxFormat_Srgb_Etc2       , "srgb_etc2"        },
    { FtxFormat_Unorm_Etc2_Mask , "unorm_etc2_mask"  },
    { FtxFormat_Srgb_Etc2_Mask  , "srgb_etc2_mask"   },
    { FtxFormat_Unorm_Etc2_Alpha, "unorm_etc2_alpha" },
    { FtxFormat_Srgb_Etc2_Alpha , "srgb_etc2_alpha"  },

    { FtxFormat_Unorm_Eac_11   , "unorm_eac_11"    },
    { FtxFormat_Snorm_Eac_11   , "snorm_eac_11"    },
    { FtxFormat_Unorm_Eac_11_11, "unorm_eac_11_11" },
    { FtxFormat_Snorm_Eac_11_11, "snorm_eac_11_11" },

    { FtxFormat_Unorm_Pvrtc1_2Bpp      , "unorm_pvrtc1_2bpp"       },
    { FtxFormat_Srgb_Pvrtc1_2Bpp       , "srgb_pvrtc1_2bpp"        },
    { FtxFormat_Unorm_Pvrtc1_4Bpp      , "unorm_pvrtc1_4bpp"       },
    { FtxFormat_Srgb_Pvrtc1_4Bpp       , "srgb_pvrtc1_4bpp"        },
    { FtxFormat_Unorm_Pvrtc1_Alpha_2Bpp, "unorm_pvrtc1_alpha_2bpp" },
    { FtxFormat_Srgb_Pvrtc1_Alpha_2Bpp , "srgb_pvrtc1_alpha_2bpp"  },
    { FtxFormat_Unorm_Pvrtc1_Alpha_4Bpp, "unorm_pvrtc1_alpha_4bpp" },
    { FtxFormat_Srgb_Pvrtc1_Alpha_4Bpp , "srgb_pvrtc1_alpha_4bpp"  },
    { FtxFormat_Unorm_Pvrtc2_Alpha_2Bpp, "unorm_pvrtc2_alpha_2bpp" },
    { FtxFormat_Srgb_Pvrtc2_Alpha_2Bpp , "srgb_pvrtc2_alpha_2bpp"  },
    { FtxFormat_Unorm_Pvrtc2_Alpha_4Bpp, "unorm_pvrtc2_alpha_4bpp" },
    { FtxFormat_Srgb_Pvrtc2_Alpha_4Bpp , "srgb_pvrtc2_alpha_4bpp"  },

    { FtxFormat_Unorm_Astc_4x4  , "unorm_astc_4x4"   },
    { FtxFormat_Srgb_Astc_4x4   , "srgb_astc_4x4"    },
    { FtxFormat_Unorm_Astc_5x4  , "unorm_astc_5x4"   },
    { FtxFormat_Srgb_Astc_5x4   , "srgb_astc_5x4"    },
    { FtxFormat_Unorm_Astc_5x5  , "unorm_astc_5x5"   },
    { FtxFormat_Srgb_Astc_5x5   , "srgb_astc_5x5"    },
    { FtxFormat_Unorm_Astc_6x5  , "unorm_astc_6x5"   },
    { FtxFormat_Srgb_Astc_6x5   , "srgb_astc_6x5"    },
    { FtxFormat_Unorm_Astc_6x6  , "unorm_astc_6x6"   },
    { FtxFormat_Srgb_Astc_6x6   , "srgb_astc_6x6"    },
    { FtxFormat_Unorm_Astc_8x5  , "unorm_astc_8x5"   },
    { FtxFormat_Srgb_Astc_8x5   , "srgb_astc_8x5"    },
    { FtxFormat_Unorm_Astc_8x6  , "unorm_astc_8x6"   },
    { FtxFormat_Srgb_Astc_8x6   , "srgb_astc_8x6"    },
    { FtxFormat_Unorm_Astc_8x8  , "unorm_astc_8x8"   },
    { FtxFormat_Srgb_Astc_8x8   , "srgb_astc_8x8"    },
    { FtxFormat_Unorm_Astc_10x5 , "unorm_astc_10x5"  },
    { FtxFormat_Srgb_Astc_10x5  , "srgb_astc_10x5"   },
    { FtxFormat_Unorm_Astc_10x6 , "unorm_astc_10x6"  },
    { FtxFormat_Srgb_Astc_10x6  , "srgb_astc_10x6"   },
    { FtxFormat_Unorm_Astc_10x8 , "unorm_astc_10x8"  },
    { FtxFormat_Srgb_Astc_10x8  , "srgb_astc_10x8"   },
    { FtxFormat_Unorm_Astc_10x10, "unorm_astc_10x10" },
    { FtxFormat_Srgb_Astc_10x10 , "srgb_astc_10x10"  },
    { FtxFormat_Unorm_Astc_12x10, "unorm_astc_12x10" },
    { FtxFormat_Srgb_Astc_12x10 , "srgb_astc_12x10"  },
    { FtxFormat_Unorm_Astc_12x12, "unorm_astc_12x12" },
    { FtxFormat_Srgb_Astc_12x12 , "srgb_astc_12x12"  },

    { FtxFormat_Uint_32_32_32 , "uint_32_32_32"  },
    { FtxFormat_Sint_32_32_32 , "sint_32_32_32"  },
    { FtxFormat_Float_32_32_32, "float_32_32_32" },
};

} // namespace

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの文字列からテクスチャーフォーマットを取得します。
//-----------------------------------------------------------------------------
FtxFormat RGetTextureFormatFromString(const std::string& str)
{
    const int fmtCount = sizeof(s_FormatStrings) / sizeof(RIntAndCharPtr);
    for (int fmtIdx = 0; fmtIdx < fmtCount; ++fmtIdx)
    {
        const RIntAndCharPtr& fmtStr = s_FormatStrings[fmtIdx];
        if (str == fmtStr.m_String)
        {
            return static_cast<FtxFormat>(fmtStr.m_Value);
        }
    }
    return FtxFormat_Invalid;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットから中間ファイルの文字列を取得します。
//-----------------------------------------------------------------------------
std::string RGetTextureFormatString(const FtxFormat format)
{
    const int fmtCount = sizeof(s_FormatStrings) / sizeof(RIntAndCharPtr);
    for (int fmtIdx = 0; fmtIdx < fmtCount; ++fmtIdx)
    {
        const RIntAndCharPtr& fmtStr = s_FormatStrings[fmtIdx];
        if (format == fmtStr.m_Value)
        {
            return fmtStr.m_String;
        }
    }
    return "invalid";
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが BC1/BC2/BC3 フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool RIsBc123TextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Unorm_Bc1:
    case FtxFormat_Srgb_Bc1:
    case FtxFormat_Unorm_Bc2:
    case FtxFormat_Srgb_Bc2:
    case FtxFormat_Unorm_Bc3:
    case FtxFormat_Srgb_Bc3:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが BC6/BC7 フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool RIsBc67TextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Ufloat_Bc6:
    case FtxFormat_Float_Bc6:
    case FtxFormat_Unorm_Bc7:
    case FtxFormat_Srgb_Bc7:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが BC フォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool RIsBcTextureFormat(const FtxFormat format)
{
    if (RIsBc123TextureFormat(format) ||
        RIsBc67TextureFormat(format))
    {
        return true;
    }
    else
    {
        switch (format)
        {
        case FtxFormat_Unorm_Bc4:
        case FtxFormat_Snorm_Bc4:
        case FtxFormat_Unorm_Bc5:
        case FtxFormat_Snorm_Bc5:
            return true;
        default:
            return false;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが 1 成分あたり 16bit の unorm / snorm なら true を返します。
//-----------------------------------------------------------------------------
bool RIsUnormSnorm16TextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Unorm_16:
    case FtxFormat_Snorm_16:
    case FtxFormat_Unorm_16_16:
    case FtxFormat_Snorm_16_16:
    case FtxFormat_Unorm_16_16_16_16:
    case FtxFormat_Snorm_16_16_16_16:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが 1 成分あたり 16bit の uint / sint なら true を返します。
//-----------------------------------------------------------------------------
bool RIsUintSint16TextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Uint_16:
    case FtxFormat_Sint_16:
    case FtxFormat_Uint_16_16:
    case FtxFormat_Sint_16_16:
    case FtxFormat_Uint_16_16_16_16:
    case FtxFormat_Sint_16_16_16_16:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが 1 成分あたり 32bit の uint / sint なら true を返します。
//-----------------------------------------------------------------------------
bool RIsUintSint32TextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Uint_32:
    case FtxFormat_Sint_32:
    case FtxFormat_Uint_32_32:
    case FtxFormat_Sint_32_32:
    case FtxFormat_Uint_32_32_32:
    case FtxFormat_Sint_32_32_32:
    case FtxFormat_Uint_32_32_32_32:
    case FtxFormat_Sint_32_32_32_32:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが浮動小数点数型なら true を返します。
//-----------------------------------------------------------------------------
bool RIsFloatTextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Float_16:
    case FtxFormat_Float_32:
    case FtxFormat_Float_16_16:
    case FtxFormat_Float_32_32:
    case FtxFormat_Float_11_11_10:
    case FtxFormat_Float_32_32_32:
    case FtxFormat_Float_16_16_16_16:
    case FtxFormat_Float_32_32_32_32:
    case FtxFormat_Ufloat_Bc6:
    case FtxFormat_Float_Bc6:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief 実数型として扱うテクスチャーフォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool RIsRealNumberTextureFormat(const FtxFormat format)
{
    return (format == FtxFormat_Unorm_10_10_10_2 ||
            format == FtxFormat_Uint_10_10_10_2  ||
            RIsUnormSnorm16TextureFormat(format) ||
            RIsUintSint16TextureFormat(format)   ||
            RIsUintSint32TextureFormat(format)   ||
            RIsFloatTextureFormat(format));
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが整数型なら true を返します。
//-----------------------------------------------------------------------------
bool RIsIntTextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Uint_8:
    case FtxFormat_Sint_8:
    case FtxFormat_Uint_8_8:
    case FtxFormat_Sint_8_8:
    case FtxFormat_Uint_8_8_8_8:
    case FtxFormat_Sint_8_8_8_8:
    case FtxFormat_Uint_10_10_10_2:
        return true;
    default:
        return (RIsUintSint16TextureFormat(format) ||
                RIsUintSint32TextureFormat(format));
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが符号ありなら true を返します。
//-----------------------------------------------------------------------------
bool RIsSignedTextureFormat(const FtxFormat format)
{
    if (RIsFloatTextureFormat(format) && format != FtxFormat_Ufloat_Bc6)
    {
        return true;
    }

    switch (format)
    {
    case FtxFormat_Snorm_8:
    case FtxFormat_Sint_8:
    case FtxFormat_Snorm_16:
    case FtxFormat_Sint_16:
    case FtxFormat_Sint_32:
    case FtxFormat_Snorm_8_8:
    case FtxFormat_Sint_8_8:
    case FtxFormat_Snorm_16_16:
    case FtxFormat_Sint_16_16:
    case FtxFormat_Sint_32_32:
    case FtxFormat_Sint_32_32_32:
    case FtxFormat_Snorm_8_8_8_8:
    case FtxFormat_Sint_8_8_8_8:
    case FtxFormat_Snorm_16_16_16_16:
    case FtxFormat_Sint_16_16_16_16:
    case FtxFormat_Sint_32_32_32_32:
    case FtxFormat_Snorm_Bc4:
    case FtxFormat_Snorm_Bc5:
    case FtxFormat_Snorm_Eac_11:
    case FtxFormat_Snorm_Eac_11_11:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが sRGB フェッチなら true を返します。
//-----------------------------------------------------------------------------
bool RIsSrgbFetchTextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Srgb_8_8_8_8:
    case FtxFormat_Srgb_Bc1:
    case FtxFormat_Srgb_Bc2:
    case FtxFormat_Srgb_Bc3:
    case FtxFormat_Srgb_Bc7:
    case FtxFormat_Srgb_Etc2:
    case FtxFormat_Srgb_Etc2_Mask:
    case FtxFormat_Srgb_Etc2_Alpha:
    case FtxFormat_Srgb_Pvrtc1_2Bpp:
    case FtxFormat_Srgb_Pvrtc1_4Bpp:
    case FtxFormat_Srgb_Pvrtc1_Alpha_2Bpp:
    case FtxFormat_Srgb_Pvrtc1_Alpha_4Bpp:
    case FtxFormat_Srgb_Pvrtc2_Alpha_2Bpp:
    case FtxFormat_Srgb_Pvrtc2_Alpha_4Bpp:
    case FtxFormat_Srgb_Astc_4x4:
    case FtxFormat_Srgb_Astc_5x4:
    case FtxFormat_Srgb_Astc_5x5:
    case FtxFormat_Srgb_Astc_6x5:
    case FtxFormat_Srgb_Astc_6x6:
    case FtxFormat_Srgb_Astc_8x5:
    case FtxFormat_Srgb_Astc_8x6:
    case FtxFormat_Srgb_Astc_8x8:
    case FtxFormat_Srgb_Astc_10x5:
    case FtxFormat_Srgb_Astc_10x6:
    case FtxFormat_Srgb_Astc_10x8:
    case FtxFormat_Srgb_Astc_10x10:
    case FtxFormat_Srgb_Astc_12x10:
    case FtxFormat_Srgb_Astc_12x12:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが ETC1 なら true を返します。
//-----------------------------------------------------------------------------
bool RIsEtc1TextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Unorm_Etc1:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが ETC2 なら true を返します。
//-----------------------------------------------------------------------------
bool RIsEtc2TextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Unorm_Etc2:
    case FtxFormat_Srgb_Etc2:
    case FtxFormat_Unorm_Etc2_Mask:
    case FtxFormat_Srgb_Etc2_Mask:
    case FtxFormat_Unorm_Etc2_Alpha:
    case FtxFormat_Srgb_Etc2_Alpha:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが ETC1 または ETC2 なら true を返します。
//-----------------------------------------------------------------------------
bool RIsEtcTextureFormat(const FtxFormat format)
{
    return RIsEtc1TextureFormat(format) || RIsEtc2TextureFormat(format);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが EAC なら true を返します。
//-----------------------------------------------------------------------------
bool RIsEacTextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Unorm_Eac_11:
    case FtxFormat_Snorm_Eac_11:
    case FtxFormat_Unorm_Eac_11_11:
    case FtxFormat_Snorm_Eac_11_11:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが PVRTC1 なら true を返します。
//-----------------------------------------------------------------------------
bool RIsPvrtc1TextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Unorm_Pvrtc1_2Bpp:
    case FtxFormat_Srgb_Pvrtc1_2Bpp:
    case FtxFormat_Unorm_Pvrtc1_4Bpp:
    case FtxFormat_Srgb_Pvrtc1_4Bpp:
    case FtxFormat_Unorm_Pvrtc1_Alpha_2Bpp:
    case FtxFormat_Srgb_Pvrtc1_Alpha_2Bpp:
    case FtxFormat_Unorm_Pvrtc1_Alpha_4Bpp:
    case FtxFormat_Srgb_Pvrtc1_Alpha_4Bpp:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが PVRTC2 なら true を返します。
//-----------------------------------------------------------------------------
bool RIsPvrtc2TextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Unorm_Pvrtc2_Alpha_2Bpp:
    case FtxFormat_Srgb_Pvrtc2_Alpha_2Bpp:
    case FtxFormat_Unorm_Pvrtc2_Alpha_4Bpp:
    case FtxFormat_Srgb_Pvrtc2_Alpha_4Bpp:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが PVRTC1 または PVRTC2 なら true を返します。
//-----------------------------------------------------------------------------
bool RIsPvrtcTextureFormat(const FtxFormat format)
{
    return RIsPvrtc1TextureFormat(format) || RIsPvrtc2TextureFormat(format);
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが ASTC なら true を返します。
//-----------------------------------------------------------------------------
bool RIsAstcTextureFormat(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Unorm_Astc_4x4:
    case FtxFormat_Srgb_Astc_4x4:
    case FtxFormat_Unorm_Astc_5x4:
    case FtxFormat_Srgb_Astc_5x4:
    case FtxFormat_Unorm_Astc_5x5:
    case FtxFormat_Srgb_Astc_5x5:
    case FtxFormat_Unorm_Astc_6x5:
    case FtxFormat_Srgb_Astc_6x5:
    case FtxFormat_Unorm_Astc_6x6:
    case FtxFormat_Srgb_Astc_6x6:
    case FtxFormat_Unorm_Astc_8x5:
    case FtxFormat_Srgb_Astc_8x5:
    case FtxFormat_Unorm_Astc_8x6:
    case FtxFormat_Srgb_Astc_8x6:
    case FtxFormat_Unorm_Astc_8x8:
    case FtxFormat_Srgb_Astc_8x8:
    case FtxFormat_Unorm_Astc_10x5:
    case FtxFormat_Srgb_Astc_10x5:
    case FtxFormat_Unorm_Astc_10x6:
    case FtxFormat_Srgb_Astc_10x6:
    case FtxFormat_Unorm_Astc_10x8:
    case FtxFormat_Srgb_Astc_10x8:
    case FtxFormat_Unorm_Astc_10x10:
    case FtxFormat_Srgb_Astc_10x10:
    case FtxFormat_Unorm_Astc_12x10:
    case FtxFormat_Srgb_Astc_12x10:
    case FtxFormat_Unorm_Astc_12x12:
    case FtxFormat_Srgb_Astc_12x12:
        return true;
    default:
        return false;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットが圧縮形式なら true を返します。
//-----------------------------------------------------------------------------
bool RIsCompressedTextureFormat(const FtxFormat format)
{
    return (RIsBcTextureFormat(format)    ||
            RIsEtcTextureFormat(format)   ||
            RIsEacTextureFormat(format)   ||
            RIsPvrtcTextureFormat(format) ||
            RIsAstcTextureFormat(format));
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットから成分数を取得します。
//-----------------------------------------------------------------------------
int RGetComponentCount(const FtxFormat format)
{
    switch (format)
    {
    case FtxFormat_Unorm_8:
    case FtxFormat_Uint_8:
    case FtxFormat_Snorm_8:
    case FtxFormat_Sint_8:
    case FtxFormat_Unorm_16:
    case FtxFormat_Snorm_16:
    case FtxFormat_Uint_16:
    case FtxFormat_Sint_16:
    case FtxFormat_Uint_32:
    case FtxFormat_Sint_32:
    case FtxFormat_Float_16:
    case FtxFormat_Float_32:
    case FtxFormat_Unorm_Bc4:
    case FtxFormat_Snorm_Bc4:
    case FtxFormat_Unorm_Eac_11:
    case FtxFormat_Snorm_Eac_11:
        return 1;

    case FtxFormat_Unorm_4_4:
    case FtxFormat_Unorm_8_8:
    case FtxFormat_Uint_8_8:
    case FtxFormat_Snorm_8_8:
    case FtxFormat_Sint_8_8:
    case FtxFormat_Unorm_16_16:
    case FtxFormat_Snorm_16_16:
    case FtxFormat_Uint_16_16:
    case FtxFormat_Sint_16_16:
    case FtxFormat_Uint_32_32:
    case FtxFormat_Sint_32_32:
    case FtxFormat_Float_16_16:
    case FtxFormat_Float_32_32:
    case FtxFormat_Unorm_Bc5:
    case FtxFormat_Snorm_Bc5:
    case FtxFormat_Unorm_Eac_11_11:
    case FtxFormat_Snorm_Eac_11_11:
        return 2;

    case FtxFormat_Unorm_5_6_5:
    case FtxFormat_Uint_32_32_32:
    case FtxFormat_Sint_32_32_32:
    case FtxFormat_Float_11_11_10:
    case FtxFormat_Float_32_32_32:
    case FtxFormat_Ufloat_Bc6:
    case FtxFormat_Float_Bc6:
    case FtxFormat_Unorm_Etc1:
    case FtxFormat_Unorm_Etc2:
    case FtxFormat_Srgb_Etc2:
        return 3;

    default:
        return 4;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットの 1 ピクセルのビット数を返します。
//!        ASTC フォーマットの場合は最大ビット数（8）を返します。
//-----------------------------------------------------------------------------
int RGetBitsPerPixel(const FtxFormat format)
{
    if (RIsAstcTextureFormat(format))
    {
        return 8;
    }

    switch (format)
    {
    case FtxFormat_Unorm_Pvrtc1_2Bpp:
    case FtxFormat_Srgb_Pvrtc1_2Bpp:
    case FtxFormat_Unorm_Pvrtc1_Alpha_2Bpp:
    case FtxFormat_Srgb_Pvrtc1_Alpha_2Bpp:
    case FtxFormat_Unorm_Pvrtc2_Alpha_2Bpp:
    case FtxFormat_Srgb_Pvrtc2_Alpha_2Bpp:
        return 2;

    case FtxFormat_Unorm_Bc1:
    case FtxFormat_Srgb_Bc1:
    case FtxFormat_Unorm_Bc4:
    case FtxFormat_Snorm_Bc4:
    case FtxFormat_Unorm_Etc1:
    case FtxFormat_Unorm_Etc2:
    case FtxFormat_Srgb_Etc2:
    case FtxFormat_Unorm_Etc2_Mask:
    case FtxFormat_Srgb_Etc2_Mask:
    case FtxFormat_Unorm_Eac_11:
    case FtxFormat_Snorm_Eac_11:
    case FtxFormat_Unorm_Pvrtc1_4Bpp:
    case FtxFormat_Srgb_Pvrtc1_4Bpp:
    case FtxFormat_Unorm_Pvrtc1_Alpha_4Bpp:
    case FtxFormat_Srgb_Pvrtc1_Alpha_4Bpp:
    case FtxFormat_Unorm_Pvrtc2_Alpha_4Bpp:
    case FtxFormat_Srgb_Pvrtc2_Alpha_4Bpp:
        return 4;

    case FtxFormat_Unorm_8:
    case FtxFormat_Uint_8:
    case FtxFormat_Snorm_8:
    case FtxFormat_Sint_8:
    case FtxFormat_Unorm_4_4:
    case FtxFormat_Unorm_Bc2:
    case FtxFormat_Srgb_Bc2:
    case FtxFormat_Unorm_Bc3:
    case FtxFormat_Srgb_Bc3:
    case FtxFormat_Unorm_Bc5:
    case FtxFormat_Snorm_Bc5:
    case FtxFormat_Ufloat_Bc6:
    case FtxFormat_Float_Bc6:
    case FtxFormat_Unorm_Bc7:
    case FtxFormat_Srgb_Bc7:
    case FtxFormat_Unorm_Etc2_Alpha:
    case FtxFormat_Srgb_Etc2_Alpha:
    case FtxFormat_Unorm_Eac_11_11:
    case FtxFormat_Snorm_Eac_11_11:
        return 8;

    case FtxFormat_Unorm_16:
    case FtxFormat_Snorm_16:
    case FtxFormat_Uint_16:
    case FtxFormat_Sint_16:
    case FtxFormat_Float_16:
    case FtxFormat_Unorm_8_8:
    case FtxFormat_Uint_8_8:
    case FtxFormat_Snorm_8_8:
    case FtxFormat_Sint_8_8:
    case FtxFormat_Unorm_5_6_5:
    case FtxFormat_Unorm_5_5_5_1:
    case FtxFormat_Unorm_4_4_4_4:
        return 16;

    case FtxFormat_Unorm_16_16:
    case FtxFormat_Snorm_16_16:
    case FtxFormat_Uint_16_16:
    case FtxFormat_Sint_16_16:
    case FtxFormat_Uint_32:
    case FtxFormat_Sint_32:
    case FtxFormat_Float_32:
    case FtxFormat_Float_16_16:
    case FtxFormat_Float_11_11_10:
    case FtxFormat_Unorm_8_8_8_8:
    case FtxFormat_Uint_8_8_8_8:
    case FtxFormat_Snorm_8_8_8_8:
    case FtxFormat_Sint_8_8_8_8:
    case FtxFormat_Srgb_8_8_8_8:
    case FtxFormat_Unorm_10_10_10_2:
    case FtxFormat_Uint_10_10_10_2:
        return 32;

    case FtxFormat_Unorm_16_16_16_16:
    case FtxFormat_Snorm_16_16_16_16:
    case FtxFormat_Uint_16_16_16_16:
    case FtxFormat_Sint_16_16_16_16:
    case FtxFormat_Uint_32_32:
    case FtxFormat_Sint_32_32:
    case FtxFormat_Float_32_32:
    case FtxFormat_Float_16_16_16_16:
        return 64;

    case FtxFormat_Uint_32_32_32:
    case FtxFormat_Sint_32_32_32:
    case FtxFormat_Float_32_32_32:
        return 96;

    case FtxFormat_Uint_32_32_32_32:
    case FtxFormat_Sint_32_32_32_32:
    case FtxFormat_Float_32_32_32_32:
        return 128;

    default:
        return 32;
    }
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットの 1 ピクセルのビット数を表す文字列を返します。
//-----------------------------------------------------------------------------
std::string RGetBitsPerPixelString(const FtxFormat format)
{
    if (RIsAstcTextureFormat(format))
    {
        int minW;
        int minH;
        RGetMinimumWH(minW, minH, format);
        const int bpp10 = RRound(10.0f * AstcBlockBytes * 8 / minW / minH);
        const int bppH = bpp10 / 10;
        const int bppL = bpp10 % 10;
        return (bppL == 0) ?
            RGetNumberString(bppH) :
            RGetNumberString(bppH) + "." + RGetNumberString(bppL);
    }
    else
    {
        return RGetNumberString(RGetBitsPerPixel(format));
    }
}

//-----------------------------------------------------------------------------
//! @brief 指定した型のデータから変換可能なテクスチャーフォーマットなら true を返します。
//-----------------------------------------------------------------------------
bool RIsConvertibleTextureFormat(const FtxFormat format, const bool isFromRealNumber)
{
    if (!isFromRealNumber)
    {
        return !RIsRealNumberTextureFormat(format);
    }
    else
    {
        return (
            RIsRealNumberTextureFormat(format) ||
            format == FtxFormat_Unorm_8_8_8_8  ||
            format == FtxFormat_Snorm_8_8_8_8  ||
            format == FtxFormat_Srgb_8_8_8_8   ||
            RIsBcTextureFormat(format)         ||
            RIsAstcTextureFormat(format));
    }
}

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの文字列からミップマップ生成フィルタを取得します。
//-----------------------------------------------------------------------------
RMipGenFilter::MipGenFilter RGetMipGenFilterFromString(const std::string& str, const bool linearIfInvalid)
{
    if (str == "point" ) return RMipGenFilter::MIP_GEN_POINT;
    if (str == "linear") return RMipGenFilter::MIP_GEN_LINEAR;
    if (str == "cubic" ) return RMipGenFilter::MIP_GEN_CUBIC;
    return (linearIfInvalid) ? RMipGenFilter::MIP_GEN_LINEAR : RMipGenFilter::MIP_GEN_INVALID;
}

//-----------------------------------------------------------------------------
//! @brief ミップマップ生成フィルタから中間ファイルの文字列を取得します。
//!
//! @param[in] mipGenFilter ミップマップ生成フィルタです。
//!
//! @return ミップマップ生成フィルタを表す文字列を返します。
//-----------------------------------------------------------------------------
std::string RGetMipGenFilterString(const RMipGenFilter::MipGenFilter mipGenFilter)
{
    switch (mipGenFilter)
    {
        case RMipGenFilter::MIP_GEN_POINT:  return "point";
        case RMipGenFilter::MIP_GEN_LINEAR: return "linear";
        case RMipGenFilter::MIP_GEN_CUBIC:  return "cubic";
        default:                            return "invalid";
    }
}

//-----------------------------------------------------------------------------
//! @brief 成分選択の成分をあらわす文字のデータです。
//-----------------------------------------------------------------------------
namespace {

const int k_CompSelMax = 6;
const char* const s_CompSelChars[k_CompSelMax] =
{
    "r", "g", "b", "a", "0", "1"
};

} // namespace

//-----------------------------------------------------------------------------
//! @brief 中間ファイルの文字列から成分選択を取得します。
//-----------------------------------------------------------------------------
FtxCompSel RGetCompSelFromString(const std::string& str)
{
    int comps[R_RGBA_BYTES] =
    {
        FtxComponent_Red,
        FtxComponent_Green,
        FtxComponent_Blue,
        FtxComponent_Alpha,
    };
    int validCount = 0;
    std::istringstream iss(str);
    for (int iRgba = 0; iRgba < R_RGBA_BYTES; ++iRgba)
    {
        if (iss)
        {
            std::string c;
            iss >> c;
            for (int iComp = 0; iComp < k_CompSelMax; ++iComp)
            {
                if (c == s_CompSelChars[iComp])
                {
                    comps[iRgba] = iComp;
                    ++validCount;
                    break;
                }
            }
        }
    }
    return FTX_GET_COMP_SEL(comps[0], comps[1], comps[2], comps[3]);
}

//-----------------------------------------------------------------------------
//! @brief 成分選択から中間ファイルの文字列を取得します。
//-----------------------------------------------------------------------------
std::string RGetCompSelString(const FtxCompSel compSel)
{
    std::string value;
    for (int iRgba = 0; iRgba < R_RGBA_BYTES; ++iRgba)
    {
        const int iComp = (compSel >> (3 - iRgba) * 8) & 0xff;
        if (iRgba > 0) value += " ";
        value += ((iComp < k_CompSelMax) ? s_CompSelChars[iComp] : "0");
    }
    return value;
}

//-----------------------------------------------------------------------------
//! @brief オプション文字列から成分選択を取得します。
//-----------------------------------------------------------------------------
FtxCompSel RGetCompSelFromOptionString(const std::string& str)
{
    if (str.size() == 4)
    {
        const std::string value = std::string() +
            str[0] + " " + str[1] + " " + str[2] + " " + str[3];
        return RGetCompSelFromString(value);
    }
    else
    {
        return FtxCompSelRgba;
    }
}

//-----------------------------------------------------------------------------
//! @brief 成分選択からオプション文字列を取得します。
//-----------------------------------------------------------------------------
std::string RGetCompSelOptionString(const FtxCompSel compSel)
{
    std::string value;
    for (int iRgba = 0; iRgba < R_RGBA_BYTES; ++iRgba)
    {
        const int iComp = (compSel >> (3 - iRgba) * 8) & 0xff;
        value += ((iComp < k_CompSelMax) ? s_CompSelChars[iComp] : "0");
    }
    return value;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットに対応したデフォルトの成分選択を取得します。
//-----------------------------------------------------------------------------
FtxCompSel RGetDefaultCompSel(const FtxFormat format, const std::string& hint)
{
    if (RIsAstcTextureFormat(format) && hint == HintValueNormal)
    {
        // ra01
        return FTX_GET_COMP_SEL(FtxComponent_Red, FtxComponent_Alpha, FtxComponent_Zero, FtxComponent_One);
    }

    switch (format)
    {
    case FtxFormat_Unorm_8:
    case FtxFormat_Uint_8:
    case FtxFormat_Snorm_8:
    case FtxFormat_Sint_8:
    case FtxFormat_Unorm_16:
    case FtxFormat_Snorm_16:
    case FtxFormat_Uint_16:
    case FtxFormat_Sint_16:
    case FtxFormat_Uint_32:
    case FtxFormat_Sint_32:
    case FtxFormat_Float_16:
    case FtxFormat_Float_32:
    case FtxFormat_Unorm_Bc4:
    case FtxFormat_Snorm_Bc4:
    case FtxFormat_Unorm_Eac_11:
    case FtxFormat_Snorm_Eac_11:
        // rrr1
        return FTX_GET_COMP_SEL(FtxComponent_Red, FtxComponent_Red, FtxComponent_Red, FtxComponent_One);

    case FtxFormat_Unorm_4_4:
    case FtxFormat_Unorm_8_8:
    case FtxFormat_Unorm_Bc5:
    case FtxFormat_Unorm_Eac_11_11:
        // rrrg
        return FTX_GET_COMP_SEL(FtxComponent_Red, FtxComponent_Red, FtxComponent_Red, FtxComponent_Green);

    case FtxFormat_Snorm_8_8:
    case FtxFormat_Uint_8_8:
    case FtxFormat_Sint_8_8:
    case FtxFormat_Unorm_16_16:
    case FtxFormat_Snorm_16_16:
    case FtxFormat_Uint_16_16:
    case FtxFormat_Sint_16_16:
    case FtxFormat_Uint_32_32:
    case FtxFormat_Sint_32_32:
    case FtxFormat_Float_16_16:
    case FtxFormat_Float_32_32:
    case FtxFormat_Snorm_Bc5:
    case FtxFormat_Snorm_Eac_11_11:
        // rg01
        return FTX_GET_COMP_SEL(FtxComponent_Red, FtxComponent_Green, FtxComponent_Zero, FtxComponent_One);

    case FtxFormat_Unorm_5_6_5:
    case FtxFormat_Uint_32_32_32:
    case FtxFormat_Sint_32_32_32:
    case FtxFormat_Float_11_11_10:
    case FtxFormat_Float_32_32_32:
    case FtxFormat_Ufloat_Bc6:
    case FtxFormat_Float_Bc6:
    case FtxFormat_Unorm_Etc1:
    case FtxFormat_Unorm_Etc2:
    case FtxFormat_Srgb_Etc2:
        // rgb1
        return FTX_GET_COMP_SEL(FtxComponent_Red, FtxComponent_Green, FtxComponent_Blue, FtxComponent_One);

    default:
        // rgba
        return FtxCompSelRgba;
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットに対する最小の幅と高さを取得します。
//-----------------------------------------------------------------------------
void RGetMinimumWH(int& minW, int& minH, const FtxFormat format)
{
    if (RIsBcTextureFormat(format)                  ||
        RIsEtcTextureFormat(format)                 ||
        RIsEacTextureFormat(format)                 ||
        format == FtxFormat_Unorm_Pvrtc2_Alpha_4Bpp ||
        format == FtxFormat_Srgb_Pvrtc2_Alpha_4Bpp  ||
        format == FtxFormat_Unorm_Astc_4x4          ||
        format == FtxFormat_Srgb_Astc_4x4)
    {
        minW = 4;
        minH = 4;
    }
    else if (
        format == FtxFormat_Unorm_Pvrtc1_2Bpp       ||
        format == FtxFormat_Srgb_Pvrtc1_2Bpp        ||
        format == FtxFormat_Unorm_Pvrtc1_Alpha_2Bpp ||
        format == FtxFormat_Srgb_Pvrtc1_Alpha_2Bpp)
    {
        minW = 16;
        minH = 8;
    }
    else if (
        format == FtxFormat_Unorm_Pvrtc1_4Bpp       ||
        format == FtxFormat_Srgb_Pvrtc1_4Bpp        ||
        format == FtxFormat_Unorm_Pvrtc1_Alpha_4Bpp ||
        format == FtxFormat_Srgb_Pvrtc1_Alpha_4Bpp  ||
        format == FtxFormat_Unorm_Astc_8x8          ||
        format == FtxFormat_Srgb_Astc_8x8)
    {
        minW = 8;
        minH = 8;
    }
    else if (
        format == FtxFormat_Unorm_Pvrtc2_Alpha_2Bpp ||
        format == FtxFormat_Srgb_Pvrtc2_Alpha_2Bpp)
    {
        minW = 8;
        minH = 4;
    }
    else if (
        format == FtxFormat_Unorm_Astc_5x4 ||
        format == FtxFormat_Srgb_Astc_5x4)
    {
        minW = 5;
        minH = 4;
    }
    else if (
        format == FtxFormat_Unorm_Astc_5x5 ||
        format == FtxFormat_Srgb_Astc_5x5)
    {
        minW = 5;
        minH = 5;
    }
    else if (
        format == FtxFormat_Unorm_Astc_6x5 ||
        format == FtxFormat_Srgb_Astc_6x5)
    {
        minW = 6;
        minH = 5;
    }
    else if (
        format == FtxFormat_Unorm_Astc_6x6 ||
        format == FtxFormat_Srgb_Astc_6x6)
    {
        minW = 6;
        minH = 6;
    }
    else if (
        format == FtxFormat_Unorm_Astc_8x5 ||
        format == FtxFormat_Srgb_Astc_8x5)
    {
        minW = 8;
        minH = 5;
    }
    else if (
        format == FtxFormat_Unorm_Astc_8x6 ||
        format == FtxFormat_Srgb_Astc_8x6)
    {
        minW = 8;
        minH = 6;
    }
    else if (
        format == FtxFormat_Unorm_Astc_10x5 ||
        format == FtxFormat_Srgb_Astc_10x5)
    {
        minW = 10;
        minH = 5;
    }
    else if (
        format == FtxFormat_Unorm_Astc_10x6 ||
        format == FtxFormat_Srgb_Astc_10x6)
    {
        minW = 10;
        minH = 6;
    }
    else if (
        format == FtxFormat_Unorm_Astc_10x8 ||
        format == FtxFormat_Srgb_Astc_10x8)
    {
        minW = 10;
        minH = 8;
    }
    else if (
        format == FtxFormat_Unorm_Astc_10x10 ||
        format == FtxFormat_Srgb_Astc_10x10)
    {
        minW = 10;
        minH = 10;
    }
    else if (
        format == FtxFormat_Unorm_Astc_12x10 ||
        format == FtxFormat_Srgb_Astc_12x10)
    {
        minW = 12;
        minH = 10;
    }
    else if (
        format == FtxFormat_Unorm_Astc_12x12 ||
        format == FtxFormat_Srgb_Astc_12x12)
    {
        minW = 12;
        minH = 12;
    }
    else
    {
        minW = 1;
        minH = 1;
    }
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief テクスチャーフォーマットに対応したデフォルトのリニア変換フラグを取得します。
//-----------------------------------------------------------------------------
int RGetDefaultLinearFlag(const FtxFormat format)
{
    return RIsSrgbFetchTextureFormat(format) ?
        RLinearFlag::LINEAR_RGB : RLinearFlag::LINEAR_NONE;
}

//-----------------------------------------------------------------------------
//! @brief 3D テクスチャーコンバーターを初期化します（DLL をロードします）。
//-----------------------------------------------------------------------------
bool RTexCvtr::Initialize(GPtr globals)
{
    if (m_hDll == nullptr)
    {
        //-----------------------------------------------------------------------------
        // DLL をロードします。
        m_DllPath = RGetNintendoPhotoshop3dToolsPath() + "3dTextureConverter.dll";
        m_hDll = LoadLibrary(m_DllPath.c_str());
        if (m_hDll == nullptr)
        {
            RShowError(globals, "Cannot load dll: %s", m_DllPath.c_str());
            return false;
        }
        //RNoteTrace("load dll  : %s", m_DllPath.c_str());

        //-----------------------------------------------------------------------------
        // 関数のアドレスを取得します。
        #define PROC_ADDRESS(handle, name)                                                      \
            name = reinterpret_cast<name##Func>(GetProcAddress(handle, #name));                 \
            if (name == nullptr)                                                                \
            {                                                                                   \
                RShowError(globals, "Cannot find function: %s (%s)", #name, m_DllPath.c_str()); \
                return false;                                                                   \
            }

        PROC_ADDRESS(m_hDll, GetCvtrVersion);
        PROC_ADDRESS(m_hDll, SetOptions);
        PROC_ADDRESS(m_hDll, Clear);
        PROC_ADDRESS(m_hDll, ReadBitmapData);
        PROC_ADDRESS(m_hDll, ConvertToData);
        PROC_ADDRESS(m_hDll, GetErrorShiftJisString);
        PROC_ADDRESS(m_hDll, SetCreateInfo);
        PROC_ADDRESS(m_hDll, SetUserData);
        PROC_ADDRESS(m_hDll, SetCommentLabel);
        PROC_ADDRESS(m_hDll, SetCommentColor);
        PROC_ADDRESS(m_hDll, SetToolData);
        PROC_ADDRESS(m_hDll, SetUserToolData);

        #undef PROC_ADDRESS
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief 3D テクスチャーコンバーターを終了します（DLL を解放します）。
//-----------------------------------------------------------------------------
void RTexCvtr::Finalize()
{
    if (m_hDll != nullptr)
    {
        if (!FreeLibrary(m_hDll))
        {
            RMsgBox("Cannot unload dll: %s", m_DllPath.c_str());
        }
        m_hDll = nullptr;
        //RNoteTrace("unload dll: %s", m_DllPath.c_str());
    }
}

//-----------------------------------------------------------------------------
//! @brief 3D テクスチャーコンバーターの nvtt の DLL が Photoshop 終了までアンロードされないようにします。
//-----------------------------------------------------------------------------
void RTexCvtr::PinNvttDll() const
{
    const std::string nvttDllPath = RGetNintendoPhotoshopGraphicsToolsPath() +
        "TextureConverterNvtt.dll";
    HMODULE hNvttDll;
    if (RFileExists(nvttDllPath) &&
        GetModuleHandleExA(GET_MODULE_HANDLE_EX_FLAG_UNCHANGED_REFCOUNT, nvttDllPath.c_str(), &hNvttDll))
    {
        if (GetModuleHandleExA(GET_MODULE_HANDLE_EX_FLAG_PIN, nvttDllPath.c_str(), &hNvttDll))
        {
            //RNoteTrace("pinned: %s", nvttDllPath.c_str());
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーエンコーダーを初期化します（DLL をロードします）。
//-----------------------------------------------------------------------------
bool RTexEncoder::Initialize()
{
    if (m_hDll == nullptr)
    {
        //-----------------------------------------------------------------------------
        // DLL をロードします。
        m_DllPath = RGetNintendoPhotoshopGraphicsToolsPath() + "TextureConverterEncoder.dll";
        m_hDll = LoadLibraryEx(m_DllPath.c_str(), nullptr, LOAD_WITH_ALTERED_SEARCH_PATH);
        if (m_hDll == nullptr)
        {
            return false;
        }
        //RNoteTrace("load dll  : %s", m_DllPath.c_str());

        //-----------------------------------------------------------------------------
        // 関数のアドレスを取得します。
        #define PROC_ADDRESS(handle, name)                                    \
            *reinterpret_cast<void**>(&name) = GetProcAddress(handle, #name); \
            if (name == nullptr)                                              \
            {                                                                 \
                Finalize();                                                   \
                return false;                                                 \
            }
        PROC_ADDRESS(m_hDll, GetDataSize);
        PROC_ADDRESS(m_hDll, ConvertFormat);
        #undef PROC_ADDRESS
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief テクスチャーエンコーダーを終了します（DLL を解放します）。
//-----------------------------------------------------------------------------
void RTexEncoder::Finalize()
{
    //-----------------------------------------------------------------------------
    // DLL を解放します。
    if (m_hDll != nullptr)
    {
        FreeLibrary(m_hDll);
        m_hDll = nullptr;
        //RNoteTrace("unload dll: %s", m_DllPath.c_str());
    }

    //-----------------------------------------------------------------------------
    // 関数アドレスをクリアします。
    GetDataSize   = nullptr;
    ConvertFormat = nullptr;
}

//-----------------------------------------------------------------------------
//! @brief エンコーダ用のイメージの次元を取得します（static 関数）。
//-----------------------------------------------------------------------------
nn::gfx::tool::texenc::ImageDimension RTexEncoder::GetImageDimension(const FtxDimension dimension)
{
    switch (dimension)
    {
    case FtxDimension_1d:                 return nn::gfx::tool::texenc::ImageDimension_1d;
    case FtxDimension_2d:                 return nn::gfx::tool::texenc::ImageDimension_2d;
    case FtxDimension_3d:                 return nn::gfx::tool::texenc::ImageDimension_3d;
    case FtxDimension_CubeMap:            return nn::gfx::tool::texenc::ImageDimension_CubeMap;
    case FtxDimension_1dArray:            return nn::gfx::tool::texenc::ImageDimension_1dArray;
    case FtxDimension_2dArray:            return nn::gfx::tool::texenc::ImageDimension_2dArray;
    case FtxDimension_2dMultisample:      return nn::gfx::tool::texenc::ImageDimension_2dMultisample;
    case FtxDimension_2dMultisampleArray: return nn::gfx::tool::texenc::ImageDimension_2dMultisampleArray;
    case FtxDimension_CubeMapArray:       return nn::gfx::tool::texenc::ImageDimension_CubeMapArray;
    default:                              return nn::gfx::tool::texenc::ImageDimension_2d;
    }
}

//-----------------------------------------------------------------------------
//! @brief 水平十字キューブマップの幅と高さなら true を返します。
//-----------------------------------------------------------------------------
bool RImage::IsCubeHCWH(const int imageW, const int imageH)
{
    if (imageW % CUBE_HC_COUNT_H == 0 &&
        imageH % CUBE_HC_COUNT_V == 0)
    {
        const int faceW = imageW / CUBE_HC_COUNT_H;
        const int faceH = imageH / CUBE_HC_COUNT_V;
        return (faceW == faceH && faceW >= WidthHeightMin);
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief 垂直十字キューブマップの幅と高さなら true を返します。
//-----------------------------------------------------------------------------
bool RImage::IsCubeVCWH(const int imageW, const int imageH)
{
    if (imageW % CUBE_VC_COUNT_H == 0 &&
        imageH % CUBE_VC_COUNT_V == 0)
    {
        const int faceW = imageW / CUBE_VC_COUNT_H;
        const int faceH = imageH / CUBE_VC_COUNT_V;
        return (faceW == faceH && faceW >= WidthHeightMin);
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief 水平十字キューブマップのフェースの位置を取得します。
//-----------------------------------------------------------------------------
const int* RImage::GetCubeHCFacePos(const int faceIdx)
{
    static const int cubeHCFacePoss[CUBE_FACE_COUNT][2] =
    {
        //      [+Y]
        // [-X] [-Z] [+X] [+Z]
        //      [-Y]

        { 2, 1 }, // +X
        { 0, 1 }, // -X
        { 1, 0 }, // +Y
        { 1, 2 }, // -Y
        { 3, 1 }, // +Z
        { 1, 1 }, // -Z
    };
    return cubeHCFacePoss[faceIdx];
};

//-----------------------------------------------------------------------------
//! @brief 垂直十字キューブマップのフェースの位置を取得します。
//-----------------------------------------------------------------------------
const int* RImage::GetCubeVCFacePos(const int faceIdx)
{
    static const int cubeVCFacePoss[CUBE_FACE_COUNT][2] =
    {
        //      [+Y]
        // [-X] [-Z] [+X]
        //      [-Y]
        //      [+Z]

        { 2, 1 }, // +X
        { 0, 1 }, // -X
        { 1, 0 }, // +Y
        { 1, 2 }, // -Y
        { 1, 3 }, // +Z
        { 1, 1 }, // -Z
    };
    return cubeVCFacePoss[faceIdx];
};

#ifdef NPS_READ_PSD_ENABLE
//-----------------------------------------------------------------------------
//! @brief 圧縮なしの PSD データをデコードします。
//!
//! @param[out] pDst デコードしたデータを格納するバッファのポインタです。
//! @param[in] pSrc 元データのバッファのポインタです。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] chanCount チャンネル数です。
//! @param[in] colorDepth カラーデプス（ビット数）です。
//-----------------------------------------------------------------------------
static void DecodePsdDataPlain(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const int imageW,
    const int imageH,
    const int chanCount,
    const int colorDepth
)
{
    const int dstSize = imageW * imageH * chanCount;
    if (colorDepth == 1)
    {
        const int rowBytes = (imageW + 7) / 8;
        for (int iy = 0; iy < imageH; ++iy)
        {
            int ix = 0;
            for (int iByte = 0; iByte < rowBytes; ++iByte)
            {
                const uint8_t val = *pSrc++;
                for (int iBit = 7; iBit >= 0; --iBit)
                {
                    *pDst++ = (val & (1 << iBit)) ? 0x00 : 0xff;
                    if (++ix >= imageW) break;
                }
                if (ix >= imageW) break;
            }
        }
    }
    else if (colorDepth == 8)
    {
        memcpy(pDst, pSrc, dstSize);
    }
    else if (colorDepth == 16)
    {
        for (int iDst = 0; iDst < dstSize; ++iDst)
        {
            *pDst++ = *pSrc; // 上位 8 ビットを取得します。
            pSrc += 2;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief RLE 圧縮された PSD データをデコードします。
//!
//! @param[out] pDst デコードしたデータを格納するバッファのポインタです。
//! @param[in] pSrc 元データのバッファのポインタです。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] chanCount チャンネル数です。
//! @param[in] colorDepth カラーデプス（ビット数）です。
//-----------------------------------------------------------------------------
static void DecodePsdDataRLE(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const int imageW,
    const int imageH,
    const int chanCount,
    const int colorDepth
)
{
    const uint8_t* pSize = pSrc; // 各チャンネルの各行のデータサイズが uint16_t で格納されています。
    pSrc += chanCount * imageH * sizeof(uint16_t);
    if (colorDepth == 1)
    {
        //-----------------------------------------------------------------------------
        // 1 bit/channel
        const int rowBytes = (imageW + 7) / 8;
        uint8_t* pFullBuf = new uint8_t[rowBytes];
        for (int iy = 0; iy < imageH; ++iy)
        {
            //-----------------------------------------------------------------------------
            // decode
            const int lineBytes = (*pSize << 8) | *(pSize + 1);
            pSize += 2;
            const uint8_t* pLineEnd = pSrc + lineBytes;
            uint8_t* pFull = pFullBuf;
            while (pSrc < pLineEnd)
            {
                int pixCount = *pSrc++;
                if (pixCount & 0x80)
                {
                    pixCount = (256 - pixCount) + 1;
                    const uint8_t val = *pSrc++;
                    for (int iPix = 0; iPix < pixCount; ++iPix)
                    {
                        *pFull++ = val;
                    }
                }
                else
                {
                    ++pixCount;
                    for (int iPix = 0; iPix < pixCount; ++iPix)
                    {
                        *pFull++ = *pSrc++;
                    }
                }
            }

            //-----------------------------------------------------------------------------
            // 1 bit to 8 bit
            pFull = pFullBuf;
            int ix = 0;
            for (int iByte = 0; iByte < rowBytes; ++iByte)
            {
                const uint8_t val = *pFull++;
                for (int iBit = 7; iBit >= 0; --iBit)
                {
                    *pDst++ = (val & (1 << iBit)) ? 0x00 : 0xff;
                    if (++ix >= imageW) break;
                }
                if (ix >= imageW) break;
            }
        }
        delete[] pFullBuf;
    }
    else if (colorDepth == 8)
    {
        //-----------------------------------------------------------------------------
        // 8 bit/channel
        for (int iChan = 0; iChan < chanCount; ++iChan)
        {
            for (int iy = 0; iy < imageH; ++iy)
            {
                const int lineBytes = (*pSize << 8) | *(pSize + 1);
                pSize += 2;
                const uint8_t* pLineEnd = pSrc + lineBytes;
                while (pSrc < pLineEnd)
                {
                    int pixCount = *pSrc++;
                    if (pixCount & 0x80)
                    {
                        pixCount = (256 - pixCount) + 1;
                        const uint8_t val = *pSrc++;
                        for (int iPix = 0; iPix < pixCount; ++iPix)
                        {
                            *pDst++ = val;
                        }
                    }
                    else
                    {
                        ++pixCount;
                        for (int iPix = 0; iPix < pixCount; ++iPix)
                        {
                            *pDst++ = *pSrc++;
                        }
                    }
                }
            }
        }
    }
    else if (colorDepth == 16)
    {
        //-----------------------------------------------------------------------------
        // 16 bit/channel
        // 出力されない？
        for (int iChan = 0; iChan < chanCount; ++iChan)
        {
            for (int iy = 0; iy < imageH; ++iy)
            {
                int lineBytes = (*pSize << 8) | *(pSize + 1);
                pSize += 2;
                const uint8_t* pLineEnd = pSrc + lineBytes;
                while (pSrc < pLineEnd)
                {
                    int pixCount = *pSrc++;
                    if (pixCount & 0x80)
                    {
                        pixCount = (256 - pixCount) + 1;
                        const uint8_t val = *pSrc; // 上位 8 ビットを取得します。
                        pSrc += 2;
                        for (int iPix = 0; iPix < pixCount; ++iPix)
                        {
                            *pDst++ = val;
                        }
                    }
                    else
                    {
                        ++pixCount;
                        for (int iPix = 0; iPix < pixCount; ++iPix)
                        {
                            *pDst++ = *pSrc; // 上位 8 ビットを取得します。
                            pSrc += 2;
                        }
                    }
                }
            }
        }
    }
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief PSD のビットマップをデコードします。
//!
//! @param[out] pDst デコードしたデータを格納するバッファのポインタです。
//! @param[in] pSrc 元データのバッファのポインタです。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] chanCount チャンネル数です。
//! @param[in] colorDepth カラーデプス（ビット数）です。
//-----------------------------------------------------------------------------
static void DecodePsdBitmap(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const uint32_t imageW,
    const uint32_t imageH,
    const uint16_t chanCount,
    const uint16_t colorDepth
)
{
    uint16_t compression;
    RGetMemBig(compression, pSrc);
    pSrc += 2;
    //RNoteTrace("psd cmpr: %d", compression);

    switch (compression)
    {
        case 0:
            DecodePsdDataPlain(pDst, pSrc, imageW, imageH, chanCount, colorDepth); break;
        case 1:
            DecodePsdDataRLE(pDst, pSrc, imageW, imageH, chanCount, colorDepth); break;
        //case 2: // 行ごとに ZIP 圧縮？
        //  DecodePsdDataZip(pDst, pSrc, imageW, imageH, chanCount, colorDepth); break;
        //case 3:
        //  DecodePsdDataPredictZip(pDst, pSrc, imageW, imageH, chanCount, colorDepth); break;
        default:
            break;
    }
}

//-----------------------------------------------------------------------------
//! @brief PSD の CMYK カラーを RGB カラーに変換します。
//!        現状、Photoshop 上で変換した場合と同じ RGB 値になりません。
//!        参考 http://image-d.isp.jp/commentary/color_cformula/CMYK.html
//!
//! @param[out] cr R 成分を格納します。
//! @param[out] cg G 成分を格納します。
//! @param[out] cb B 成分を格納します。
//! @param[in] cc C 成分です。
//! @param[in] cm M 成分です。
//! @param[in] cy Y 成分です。
//! @param[in] ck K 成分です。
//-----------------------------------------------------------------------------
static void ConvertPsdCmykToRgb(
    uint8_t& cr,
    uint8_t& cg,
    uint8_t& cb,
    const uint8_t cc,
    const uint8_t cm,
    const uint8_t cy,
    const uint8_t ck
)
{
    const float fc = 1.0f - cc / 255.0f; // 元の C は 255 で 0 %
    const float fm = 1.0f - cm / 255.0f; // 元の M は 255 で 0 %
    const float fy = 1.0f - cy / 255.0f; // 元の Y は 255 で 0 %
    const float fk = 1.0f - ck / 255.0f; // 元の K は 255 で 0 %
    const float oneMinusK = 1.0f - fk;

    const float fr = 1.0f - RMin(1.0f, fc * oneMinusK + fk);
    const float fg = 1.0f - RMin(1.0f, fm * oneMinusK + fk);
    const float fb = 1.0f - RMin(1.0f, fy * oneMinusK + fk);

    cr = static_cast<uint8_t>(RRound(fr * 255.0f));
    cg = static_cast<uint8_t>(RRound(fg * 255.0f));
    cb = static_cast<uint8_t>(RRound(fb * 255.0f));
}

//-----------------------------------------------------------------------------
//! @brief PSD のビットマップを RGBA8 形式に変換します。
//!
//! @param[out] pDst 変換したデータを格納するバッファのポインタです。
//! @param[in] pSrc 元データのバッファのポインタです。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//! @param[in] imageMode イメージモードです。
//! @param[in] pColorTables カラーテーブルです（長さ 3 のポインタ配列）。
//! @param[in] transparentIndex カラーテーブルの透明色インデックスです。
//! @param[in] alphaChanCount アルファチャンネル数です。
//!                           カラーテーブルの透明色よりアルファチャンネルを優先します。
//-----------------------------------------------------------------------------
static void ConvertPsdBitmapToRgba8(
    uint8_t* pDst,
    const uint8_t* pSrc,
    const int imageW,
    const int imageH,
    const uint16_t imageMode,
    uint8_t* pColorTables[],
    const uint16_t transparentIndex,
    const uint16_t alphaChanCount
)
{
    const int pixelCount = imageW * imageH;
    const uint8_t* pSrc0 = pSrc;
    const uint8_t* pSrc1 = pSrc0 + pixelCount;
    const uint8_t* pSrc2 = pSrc1 + pixelCount;
    const uint8_t* pSrc3 = pSrc2 + pixelCount;
    const uint8_t* pSrcA;
    switch (imageMode)
    {
        case RImage::PSD_IM_RGB : pSrcA = pSrc3             ; break;
        case RImage::PSD_IM_CMYK: pSrcA = pSrc3 + pixelCount; break;
        default                 : pSrcA = pSrc1             ; break;
    }

    for (int iy = 0; iy < imageH; ++iy)
    {
        for (int ix = 0; ix < imageW; ++ix)
        {
            uint8_t cr, cg, cb, ca = 0xff;
            if (imageMode == RImage::PSD_IM_INDEXED)
            {
                const uint8_t iCol = *pSrc0++;
                cr = pColorTables[0][iCol];
                cg = pColorTables[1][iCol];
                cb = pColorTables[2][iCol];
                if (iCol == transparentIndex) ca = 0x00;
            }
            else if (imageMode == RImage::PSD_IM_RGB)
            {
                cr = *pSrc0++;
                cg = *pSrc1++;
                cb = *pSrc2++;
            }
            else if (imageMode == RImage::PSD_IM_CMYK)
            {
                const uint8_t cc = *pSrc0++;
                const uint8_t cm = *pSrc1++;
                const uint8_t cy = *pSrc2++;
                const uint8_t ck = *pSrc3++;
                ConvertPsdCmykToRgb(cr, cg, cb, cc, cm, cy, ck);
                //RNoteTrace("cmyk: %3d %3d %3d %3d -> %3d %3d %3d", cc, cm, cy, ck, cr, cg, cb);
            }
            else // BITMAP, GRAYSCALE
            {
                cr = cg = cb = *pSrc0++;
            }
            if (alphaChanCount > 0) ca = *pSrcA++;

            *pDst++ = cr;
            *pDst++ = cg;
            *pDst++ = cb;
            *pDst++ = ca;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief PSD ファイルのリソースデータ群をデコードします。
//!
//! @param[out] alphaNames アルファチャンネル名配列を格納します。
//! @param[out] colorTableCount カラーテーブルの色数を格納します。
//! @param[out] transparentIndex カラーテーブルの透明色インデックスを格納します。
//! @param[out] hasRealMergedData レイヤー統合画像データの有無を格納します。
//! @param[in] pSrc リソースデータ群の先頭ポインタです。
//! @param[in] resoucesDataSize リソースデータ群のサイズです。
//-----------------------------------------------------------------------------
void DecodePsdResources(
    RStringArray& alphaNames,
    uint16_t& colorTableCount,
    uint16_t& transparentIndex,
    bool& hasRealMergedData,
    const uint8_t* pSrc,
    const uint32_t resoucesDataSize
)
{
    alphaNames.clear();

    const uint8_t* pResEnd = pSrc + resoucesDataSize;
    while (pSrc < pResEnd)
    {
        //-----------------------------------------------------------------------------
        // check signature
        if (strncmp(reinterpret_cast<const char*>(pSrc), "8BIM", 4) != 0)
        {
            break;
        }
        pSrc += 4;

        //-----------------------------------------------------------------------------
        // リソース ID を取得します。
        // 各 ID の説明は CS2 SDK の "Photoshop File Formats.pdf" P.13- にあります。
        uint16_t id;
        RGetMemBig(id, pSrc);
        pSrc += 2;
        // 040c = thumbnail
        // 0415 = unicode alpha names
        // 041a = slices
        // 0421 = version info
        // 0fa0 - = pseudo resource

        //-----------------------------------------------------------------------------
        // リソース名を取得します（通常空文字？）。
        const uint8_t nameSize = *pSrc++;
        std::string resName(reinterpret_cast<const char*>(pSrc), nameSize);
        pSrc += nameSize;
        if ((nameSize & 0x01) == 0) // リソース名の文字数が偶数なら 1 バイトスキップします。
        {
            pSrc += 1;
        }

        //-----------------------------------------------------------------------------
        // リソースサイズを取得します。
        uint32_t dataSize;
        RGetMemBig(dataSize, pSrc);
        pSrc += 4;

        //-----------------------------------------------------------------------------
        // リソースデータをデコードします。
        if (id == 0x03ee) // Names of the alpha channels as a series of Pascal strings
        {
            const uint8_t* pName = pSrc;
            while (pName < pSrc + dataSize)
            {
                const uint8_t alphaNameSize = *pName++;
                const std::string alphaName(reinterpret_cast<const char*>(pName), alphaNameSize);
                //RNoteTrace("alpha name: %s", alphaName.c_str());
                alphaNames.push_back(alphaName);
                pName += alphaNameSize;
            }
        }
        else if (id == 0x0416) // Indexed Color Table Count
        {
            RGetMemBig(colorTableCount, pSrc);
        }
        else if (id == 0x0417) // Transparency Index
        {
            RGetMemBig(transparentIndex, pSrc); // 透明色なしなら 0xffff
        }
        else if (id == 0x0421) // Version Info
        {
            // 4 bytes version (0x00 0x00 0x00 0x01)
            // 1 byte hasRealMergedData (0x00 or 0x01)
            // Unicode string: writer name (4 バイトの文字数の後に文字数 x uint16_t のデータ)
            // Unicode string: reader name (4 バイトの文字数の後に文字数 x uint16_t のデータ)
            // 4 bytes file version (0x00 0x00 0x00 0x01)
            hasRealMergedData = (pSrc[4] != 0);
            //RNoteDumpMemory(pSrc, dataSize, "psd ver info:");
        }
        else if (id >= 0x0fa0)
        {
            //DecodePsdPseudoResource(pSrc, dataSize);
        }
        //RNoteTrace("psd res: 0x%04x: %s: %d", id, resName.c_str(), dataSize);

        pSrc += dataSize;
        if (dataSize & 0x01) // リソースデータのサイズが奇数なら 1 バイトスキップします。
        {
            pSrc += 1;
        }
    }
}

//=============================================================================
//! @brief PSD レイヤー情報のクラスです。
//=============================================================================
//class RPsdLayer
//{
//  public:
//  enum SectionDivider
//  {
//      PSD_SCT_NONE,
//      PSD_SCT_OPEN_FOLDER,
//      PSD_SCT_CLOSE_FOLDER,
//      PSD_SCT_BOUNDING
//  };
//
//  // basic
//  uint32_t m_Top;
//  uint32_t m_Left;
//  uint32_t m_Bottom;
//  uint32_t m_Right;
//  uint32_t m_LayerW;
//  uint32_t m_LayerH;
//
//  uint16_t m_ChanCount; //!< チャンネル数です。
//  uint16_t m_ColorChanCount; //!< グレー / RGB / CMYK チャンネルの数です。
//  RShortArray m_ChanIds;
//  RUIntArray m_ChanDataSizes;
//  uint32_t m_ChanDataSizeTotal;
//
//  std::string m_BlendMode;
//
//  uint8_t m_Opacity;
//  uint8_t m_Clipping;
//  uint8_t m_Flags;
//
//  // mask
//  uint32_t m_MaskTop;
//  uint32_t m_MaskLeft;
//  uint32_t m_MaskBottom;
//  uint32_t m_MaskRight;
//  uint8_t m_MaskDefaultColor;
//  uint8_t m_MaskFlags;
//  uint8_t m_MaskRealFlags;
//  uint8_t m_MaskBackground;
//  uint32_t m_MaskTop2;
//  uint32_t m_MaskLeft2;
//  uint32_t m_MaskBottom2;
//  uint32_t m_MaskRight2;
//
//  // blending range
//
//  // name
//  std::string m_Name;
//
//  // additional info
//  uint32_t m_Section;
//
//  // image data
//  const uint8_t* m_pColorImageData; //!< 最初のカラーチャンネルのイメージデータへのポインタです。
//};
//
//typedef std::vector<RPsdLayer> RPsdLayerArray;

//-----------------------------------------------------------------------------
//! @brief PSD ファイルのレイヤーデータ群をデコードします。
//!
//! @param[out] layers レイヤー情報配列を格納します。
//! @param[in] pSrc レイヤーデータ群の先頭ポインタです。
//! @param[in] layersDataSize レイヤーデータ群のサイズです。
//-----------------------------------------------------------------------------
//static void DecodePsdLayers(
//  RPsdLayerArray& layers,
//  const uint8_t* pSrc,
//  const uint32_t layersDataSize
//)
//{
//  //-----------------------------------------------------------------------------
//  // init output
//  layers.clear();
//
//  //-----------------------------------------------------------------------------
//  // parse layer info
//  if (layersDataSize == 0)
//  {
//      //RNoteTrace("no layers data");
//      return;
//  }
//
//  uint32_t layerInfoSize;
//  RGetMemBig(layerInfoSize, pSrc);
//  pSrc += 4;
//  if (layerInfoSize == 0) // 背景のみの場合 0 になります。
//  {
//      //RNoteTrace("no layer info");
//      return;
//  }
//  const uint8_t* pSrcGlobalLayerMask = pSrc + layerInfoSize;
//
//  s16 layerCount;
//  RGetMemBig(layerCount, pSrc);
//  //RNoteTrace("layer count: %d", layerCount);
//  if (layerCount < 0)
//  {
//      layerCount = -layerCount;
//      // レイヤー数が負の場合、絶対値がレイヤー数で、
//      // 最初のアルファチャンネルがレイヤー統合画像の透明度データとなります。
//      // （どういう場合に負になるのか？）
//  }
//  pSrc += 2;
//
//  //-----------------------------------------------------------------------------
//  // parse each layer record
//  for (int iLayer = 0; iLayer < layerCount; ++iLayer)
//  {
//      RPsdLayer layer;
//
//      //-----------------------------------------------------------------------------
//      // basic
//
//      // rectangle
//      RGetMemBig(layer.m_Top, pSrc);
//      pSrc += 4;
//      RGetMemBig(layer.m_Left, pSrc);
//      pSrc += 4;
//      RGetMemBig(layer.m_Bottom, pSrc);
//      pSrc += 4;
//      RGetMemBig(layer.m_Right, pSrc);
//      pSrc += 4;
//      layer.m_LayerW = layer.m_Right  - layer.m_Left;
//      layer.m_LayerH = layer.m_Bottom - layer.m_Top;
//
//      // channel size
//      RGetMemBig(layer.m_ChanCount, pSrc);
//      pSrc += 2;
//
//      // channel info
//      layer.m_ColorChanCount = 0;
//      layer.m_ChanDataSizeTotal = 0;
//      for (int iChan = 0; iChan < layer.m_ChanCount; ++iChan)
//      {
//          // 0:R, 1:G, 2:B, -1:transparency mask, -2:user supplied layer mask
//          s16 chanId;
//          RGetMemBig(chanId, pSrc);
//          pSrc += 2;
//
//          uint32_t chanDataSize;
//          RGetMemBig(chanDataSize, pSrc);
//          pSrc += 4;
//          //RNoteTrace("layer chan: %d: %d", chanId, chanDataSize);
//
//          layer.m_ChanIds.push_back(chanId);
//          layer.m_ChanDataSizes.push_back(chanDataSize);
//          layer.m_ChanDataSizeTotal += chanDataSize;
//          if (chanId >= 0)
//          {
//              ++layer.m_ColorChanCount;
//          }
//      }
//
//      // blend mode signature (8BIM)
//      pSrc += 4;
//
//      // blend mode
//      const int BLEND_MODE_NAME_SIZE = 4;
//      layer.m_BlendMode = std::string(reinterpret_cast<const char*>(pSrc), BLEND_MODE_NAME_SIZE);
//      pSrc += BLEND_MODE_NAME_SIZE;
//
//      // opacity & clipping & flags
//      layer.m_Opacity = *pSrc++;
//      layer.m_Clipping = *pSrc++;
//      layer.m_Flags = *pSrc++;
//      // bit0: transparency protected
//      // bit1: visiblility (1 if invisible)
//      // bit2: obsolete
//      // bit3: 1 for Photoshop 5.0 and later
//      // bit4: pixel data irrelevant (←関係のない) to appearance of document
//      ++pSrc; // filler
//
//      // extra data size
//      uint32_t extraDataSize;
//      RGetMemBig(extraDataSize, pSrc);
//      pSrc += 4;
//      const uint8_t* pSrcNextLayer = pSrc + extraDataSize;
//
//      //-----------------------------------------------------------------------------
//      // layer mask
//      uint32_t layerMaskSize; // 36, 20, 0
//      RGetMemBig(layerMaskSize, pSrc);
//      pSrc += 4;
//      const uint8_t* pSrcLayerBlending = pSrc + layerMaskSize;
//      if (layerMaskSize > 0)
//      {
//          RGetMemBig(layer.m_MaskTop, pSrc);
//          pSrc += 4;
//          RGetMemBig(layer.m_MaskLeft, pSrc);
//          pSrc += 4;
//          RGetMemBig(layer.m_MaskBottom, pSrc);
//          pSrc += 4;
//          RGetMemBig(layer.m_MaskRight, pSrc);
//          pSrc += 4;
//
//          layer.m_MaskDefaultColor = *pSrc++;
//          layer.m_MaskFlags = *pSrc++;
//
//          if (layerMaskSize >= 36)
//          {
//              layer.m_MaskRealFlags = *pSrc++;
//              layer.m_MaskBackground = *pSrc++;
//              RGetMemBig(layer.m_MaskTop2, pSrc);
//              pSrc += 4;
//              RGetMemBig(layer.m_MaskLeft2, pSrc);
//              pSrc += 4;
//              RGetMemBig(layer.m_MaskBottom2, pSrc);
//              pSrc += 4;
//              RGetMemBig(layer.m_MaskRight2, pSrc);
//              pSrc += 4;
//          }
//      }
//
//      //-----------------------------------------------------------------------------
//      // layer blending range
//      pSrc = pSrcLayerBlending;
//      uint32_t layerBlendingSize;
//      RGetMemBig(layerBlendingSize, pSrc);
//      pSrc += 4;
//      const uint8_t* pSrcLayerName = pSrc + layerBlendingSize;
//      if (layerBlendingSize > 0)
//      {
//          // not implemented
//      }
//
//      //-----------------------------------------------------------------------------
//      // layer name
//      pSrc = pSrcLayerName;
//      const uint8_t nameSize = *pSrc++;
//      layer.m_Name = std::string(reinterpret_cast<const char*>(pSrc), nameSize);
//      pSrc += nameSize;
//      if (((1 + nameSize) & 3) != 0) // 4 バイトアライメントなのでパディングをスキップします。
//      {
//          pSrc += 4 - ((1 + nameSize) & 3);
//      }
//
//      //-----------------------------------------------------------------------------
//      // additional layer info
//      layer.m_Section = RPsdLayer::PSD_SCT_NONE;
//
//      while (pSrc < pSrcNextLayer)
//      {
//          if (strncmp(reinterpret_cast<const char*>(pSrc), "8BIM", 4) != 0)
//          {
//              break;
//          }
//          pSrc += 4;
//
//          std::string keyName(reinterpret_cast<const char*>(pSrc), 4);
//          pSrc += 4;
//
//          uint32_t dataSize;
//          RGetMemBig(dataSize, pSrc);
//          pSrc += 4;
//
//          if (keyName == "lsct")
//          {
//              RGetMemBig(layer.m_Section, pSrc);
//          }
//
//          pSrc += dataSize;
//      }
//
//      //-----------------------------------------------------------------------------
//      // append
//      layers.push_back(layer);
//      //RNoteTrace("layer%d: %s: %d x %d, %d chans, %s, %02x, %d", iLayer, layer.m_Name.c_str(),
//      //layer.m_LayerW, layer.m_LayerH, layer.m_ChanCount, layer.m_BlendMode.c_str(),
//      //layer.m_Flags, layer.m_Section);
//
//      pSrc = pSrcNextLayer;
//  }
//
//  //-----------------------------------------------------------------------------
//  // get channel image pointer
//  for (int iLayer = 0; iLayer < layerCount; ++iLayer)
//  {
//      RPsdLayer& layer = layers[iLayer];
//      layer.m_pColorImageData = NULL;
//      if (layer.m_LayerW > 0 && layer.m_LayerH > 0)
//      {
//          for (int iChan = 0; iChan < layer.m_ChanCount; ++iChan)
//          {
//              if (layer.m_ChanIds[iChan] == 0)
//              {
//                  layer.m_pColorImageData = pSrc;
//                  break;
//              }
//          }
//      }
//      pSrc += layer.m_ChanDataSizeTotal;
//  }
//
//  //-----------------------------------------------------------------------------
//  // global layer mask
//  pSrc = pSrcGlobalLayerMask;
//
//  uint32_t globalLayerMaskSize;
//  RGetMemBig(globalLayerMaskSize, pSrc);
//  pSrc += 4;
//
//  // not implemented
//}
#endif // NPS_READ_PSD_ENABLE

//-----------------------------------------------------------------------------
//! @brief PSD ファイルのデータをデコードします。
//-----------------------------------------------------------------------------
bool RImage::DecodePsd(const uint8_t* fileBuf, const size_t fileSize) // RImage_DecodePsd
{
#ifdef NPS_READ_PSD_ENABLE
    //-----------------------------------------------------------------------------
    // check header
    if (fileSize < 4 || strncmp(reinterpret_cast<const char*>(fileBuf), "8BPS", 4) != 0)
    {
        m_ErrorString = "PSD file is wrong: (invalid header): " + m_FilePath; // RShowError
        return false;
    }

    uint16_t psdVer;
    RGetMemBig(psdVer, fileBuf + 0x04);
    if (psdVer != 0x0001)
    {
        m_ErrorString = "PSD file is wrong: (invalid version): " + m_FilePath; // RShowError
        return false;
    }

    // 0x06 - 0x0b is reserved

    //-----------------------------------------------------------------------------
    // get parameter
    uint16_t chanCount;
    RGetMemBig(chanCount, fileBuf + 0x0c);

    RGetMemBig(m_ImageH, fileBuf + 0x0e);
    RGetMemBig(m_ImageW, fileBuf + 0x12);

    uint16_t colorDepth;
    RGetMemBig(colorDepth, fileBuf + 0x16);
    if (colorDepth != 1 && colorDepth != 8)
    {
        m_ErrorString = "PSD file is wrong: (unsupported color depth): " + m_FilePath; // RShowError
        return false;
    }

    uint16_t imageMode;
    RGetMemBig(imageMode, fileBuf + 0x18);
    if (imageMode != PSD_IM_BITMAP    &&
        imageMode != PSD_IM_GRAYSCALE &&
        imageMode != PSD_IM_INDEXED   &&
        imageMode != PSD_IM_RGB       &&
        imageMode != PSD_IM_CMYK)
    {
        m_ErrorString = "PSD file is wrong: (unsupported image mode): " + m_FilePath; // RShowError
        return false;
    }

    //-----------------------------------------------------------------------------
    // アルファチャンネル数を計算します。
    uint16_t alphaChanCount = 0;
    switch (imageMode)
    {
        case PSD_IM_RGB : alphaChanCount = chanCount - 3; break;
        case PSD_IM_CMYK: alphaChanCount = chanCount - 4; break;
        default         : alphaChanCount = chanCount - 1; break;
    }

    //static const char* const imageModeNames[] =
    //{ "bitmap", "grayscale", "indexed", "rgb", "cmyk", "mode5", "mode6", "multi", "duo", "lab" };
    //RNoteTrace("psd: %d x %d, %d chans, %d bit, %s", m_ImageW, m_ImageH, chanCount, colorDepth, imageModeNames[imageMode]);

    //-----------------------------------------------------------------------------
    // get color table
    uint8_t* pColorTables[3] = { NULL, NULL, NULL };
    const uint8_t* pSrc = fileBuf + 0x1a;
    uint32_t colorTableSize;
    RGetMemBig(colorTableSize, pSrc);
    pSrc += 4;
    if (imageMode == PSD_IM_INDEXED)
    {
        for (int iRgb = 0; iRgb < 3; ++iRgb)
        {
            pColorTables[iRgb] = new uint8_t[PSD_COLOR_TABLE_MAX];
            memcpy(pColorTables[iRgb], pSrc + iRgb * PSD_COLOR_TABLE_MAX, PSD_COLOR_TABLE_MAX);
        }
    }
    pSrc += colorTableSize;

    //-----------------------------------------------------------------------------
    // parse image resources
    uint32_t resoucesDataSize;
    RGetMemBig(resoucesDataSize, pSrc);
    pSrc += 4;

    RStringArray alphaNames;
    uint16_t colorTableCount;
    uint16_t transparentIndex = 0xffff;
    bool hasRealMergedData = true;
    DecodePsdResources(alphaNames, colorTableCount, transparentIndex, hasRealMergedData,
        pSrc, resoucesDataSize);
    pSrc += resoucesDataSize;

    m_HasAlpha = (alphaChanCount > 0 || transparentIndex <= 0xff);

    //-----------------------------------------------------------------------------
    // parse layers (and mask information)
    uint32_t layersDataSize;
    RGetMemBig(layersDataSize, pSrc);
    pSrc += 4;

    //RPsdLayerArray layers;
    //DecodePsdLayers(layers, pSrc, layersDataSize);
    pSrc += layersDataSize;

    //-----------------------------------------------------------------------------
    // ビットマップを取得します。
    bool success = true;
    if (hasRealMergedData)
    {
        //-----------------------------------------------------------------------------
        // 画像を統合した状態のビットマップを取得します。
        //RNoteTrace("psd merged ofs: %x", pSrc - fileBuf);
        uint8_t* pMergedBitmap = new uint8_t[m_ImageW * m_ImageH * chanCount];
        DecodePsdBitmap(pMergedBitmap, pSrc, m_ImageW, m_ImageH, chanCount, colorDepth);

        //-----------------------------------------------------------------------------
        // PSD のビットマップを RGBA8 に変換します。
        const int fullBitmapSize = m_ImageW * m_ImageH * R_RGBA_BYTES;
        m_pImageData = new uint8_t[fullBitmapSize];
        ConvertPsdBitmapToRgba8(reinterpret_cast<uint8_t*>(m_pImageData),
            pMergedBitmap, m_ImageW, m_ImageH, imageMode, pColorTables, transparentIndex, alphaChanCount);
        RFreeAndClearArray(pMergedBitmap);
    }
    else
    {
        success = false;
        m_ErrorString = "PSD file is wrong: (no merged image data): " + m_FilePath; // RShowError
    }

    //-----------------------------------------------------------------------------
    // free memory
    for (int iRgb = 0; iRgb < 3; ++iRgb)
    {
        RFreeAndClearArray(pColorTables[iRgb]);
    }

    return success;
#else
    R_UNUSED_VARIABLE(fileBuf);
    R_UNUSED_VARIABLE(fileSize);
    m_ErrorString = "PSD file is not supported: " + m_FilePath; // RShowError
    return false;
#endif
}

//-----------------------------------------------------------------------------
//! @brief TGA ファイルのビットマップをデコードします。
//!
//! @param[out] pDst ビットマップバッファのポインタです。
//! @param[in] pSrc TGA ファイルをリードしたバッファのポインタです。
//! @param[in] imageW 画像の幅です。
//! @param[in] imageH 画像の高さです。
//-----------------------------------------------------------------------------
static void DecodeTgaBitmap(
    uint8_t* pDst,
    const uint8_t* pSrc,
    int imageW,
    int imageH
)
{
    //-----------------------------------------------------------------------------
    // ヘッダを解析します。
    const int TGA_HEADER_SIZE = 0x12; //!< TGA ファイルのヘッダサイズです。
    const int idSize = static_cast<int>(pSrc[0x00]);
    int iBuf = TGA_HEADER_SIZE + idSize;

    const bool hasPal = (pSrc[0x01] != 0);
    const int imageType = pSrc[0x02];
    const int bitPerPixel = pSrc[0x10];

    // decode image descriptor
    //const int attrBitSize = pSrc[0x11] & 0x0f;
    const bool rightToLeft  = ((pSrc[0x11] & 0x10) != 0);
    const bool upperToLower = ((pSrc[0x11] & 0x20) != 0);
    const int interLeave    =  (pSrc[0x11] & 0xc0) >> 6; // 0=none, 1=two, 2=four

    //-----------------------------------------------------------------------------
    // パレットをデコードします。
    uint8_t* pPalBuf = NULL;
    if (hasPal)
    {
        const int ciMin   = pSrc[0x03] | (pSrc[0x04] << 8);
        const int colSize = pSrc[0x05] | (pSrc[0x06] << 8);
        const int ciMax = ciMin + colSize;
        const int colBitSize = pSrc[0x07];
        pPalBuf = new uint8_t[ciMax * R_RGBA_BYTES];
        int iCol;
        for (iCol = 0; iCol < ciMin; ++iCol)
        {
            // ciMin までを黒で埋めます。
            int ofs = iCol * R_RGBA_BYTES;
            pPalBuf[ofs    ] = pPalBuf[ofs + 1] = pPalBuf[ofs + 2] = 0x00;
            pPalBuf[ofs + 3] = 0xff;
        }
        for ( ; iCol < ciMax; ++iCol)
        {
            uint8_t cr, cg, cb, ca;
            if (colBitSize == 8) // GRAY 8 bit
            {
                cr = cg = cb = ca = pSrc[iBuf++];
            }
            else if (colBitSize == 15 ||    // RGB(A) 15/16 bit
                     colBitSize == 16)
            {
                int val = pSrc[iBuf] | (pSrc[iBuf + 1] << 8);
                iBuf += 2;
                cr = static_cast<uint8_t>((val >> 10) & 0x1f);
                cg = static_cast<uint8_t>((val >>  5) & 0x1f);
                cb = static_cast<uint8_t>((val      ) & 0x1f);
                cr = (cr << 3) | (cr >> 2);
                cg = (cg << 3) | (cg >> 2);
                cb = (cb << 3) | (cb >> 2);
                ca = 0xff;
                //if (attrBitSize == 0)
                //{
                //  ca = 0xff;
                //}
                //else
                //{
                //  ca = (val & 0x8000) ? 0xff : 0x00;
                //}
            }
            else if (colBitSize == 24) // RGB 24 bit
            {
                cb = pSrc[iBuf    ];
                cg = pSrc[iBuf + 1];
                cr = pSrc[iBuf + 2];
                ca = 0xff;
                iBuf += 3;
            }
            else if (colBitSize == 32) // RGBA 32 bit
            {
                cb = pSrc[iBuf    ];
                cg = pSrc[iBuf + 1];
                cr = pSrc[iBuf + 2];
                ca = pSrc[iBuf + 3];
                iBuf += 4;
            }
            else
            {
                cr = cg = cb = ca = 0;
            }
            const int palOfs = iCol * R_RGBA_BYTES;
            pPalBuf[palOfs    ] = cr;
            pPalBuf[palOfs + 1] = cg;
            pPalBuf[palOfs + 2] = cb;
            pPalBuf[palOfs + 3] = ca;
        }
    }

    //-----------------------------------------------------------------------------
    // ビットマップをデコードします。
    const bool rleFlag = (imageType == 9 || imageType == 10 || imageType == 11);
    int rleCount = 0;
    bool sameFlag = false;
    uint8_t cr = 0, cg = 0, cb = 0, ca = 0;
    int iyOfs = 0;
    int iyCur = iyOfs;
    for (int iy = 0; iy < imageH; ++iy)
    {
        const int iyDst = (upperToLower) ? iyCur : imageH - 1 - iyCur;
        int dstOfs = iyDst * imageW * R_RGBA_BYTES;
        if (rightToLeft)
        {
            dstOfs += (imageW - 1) * R_RGBA_BYTES;
        }
        for (int ix = 0; ix < imageW; ++ix)
        {
            bool getColorFlag = true;
            if (rleFlag)
            {
                if (rleCount == 0)
                {
                    int count = pSrc[iBuf++];
                    if (count & 0x80)
                    {
                        rleCount = count - 0x80;
                    }
                    else
                    {
                        rleCount = count;
                    }
                    sameFlag = ((count & 0x80) != 0);
                }
                else
                {
                    --rleCount;
                    getColorFlag = !sameFlag;
                }
            }

            if (getColorFlag)
            {
                if (hasPal)                 // CI 8/16 bit
                {
                    int palOfs;
                    if (bitPerPixel == 8)
                    {
                        palOfs = pSrc[iBuf++] * R_RGBA_BYTES;
                    }
                    else
                    {
                        palOfs = (pSrc[iBuf] | (pSrc[iBuf + 1] << 8)) * R_RGBA_BYTES;
                        iBuf += 2;
                    }
                    cr = pPalBuf[palOfs    ];
                    cg = pPalBuf[palOfs + 1];
                    cb = pPalBuf[palOfs + 2];
                    ca = pPalBuf[palOfs + 3];
                }
                else if (bitPerPixel == 8)  // GRAY 8 bit
                {
                    cr = cg = cb = pSrc[iBuf++];
                    ca = 0xff;
                }
                else if (bitPerPixel == 15 ||
                         bitPerPixel == 16) // RGB(A) 16 bit
                {
                    int val = pSrc[iBuf] | (pSrc[iBuf + 1] << 8);
                    iBuf += 2;
                    cr = static_cast<uint8_t>((val >> 10) & 0x1f);
                    cg = static_cast<uint8_t>((val >>  5) & 0x1f);
                    cb = static_cast<uint8_t>((val      ) & 0x1f);
                    cr = (cr << 3) | (cr >> 2);
                    cg = (cg << 3) | (cg >> 2);
                    cb = (cb << 3) | (cb >> 2);
                    ca = 0xff;
                    //if (attrBitSize == 0)
                    //{
                    //  ca = 0xff;
                    //}
                    //else
                    //{
                    //  ca = (val & 0x8000) ? 0xff : 0x00;
                    //}
                }
                else if (bitPerPixel == 24) // RGB 24 bit
                {
                    cb = pSrc[iBuf    ];
                    cg = pSrc[iBuf + 1];
                    cr = pSrc[iBuf + 2];
                    ca = 0xff;
                    iBuf += 3;
                }
                else if (bitPerPixel == 32) // RGBA 32 bit
                {
                    cb = pSrc[iBuf    ];
                    cg = pSrc[iBuf + 1];
                    cr = pSrc[iBuf + 2];
                    ca = pSrc[iBuf + 3];
                    iBuf += 4;
                }
            }
            pDst[dstOfs    ] = cr;
            pDst[dstOfs + 1] = cg;
            pDst[dstOfs + 2] = cb;
            pDst[dstOfs + 3] = ca;
            if (rightToLeft)
            {
                dstOfs -= 4;
            }
            else
            {
                dstOfs += 4;
            }
        }
        if (interLeave == 2)        // 4
        {
            iyCur += 4;
        }
        else if (interLeave == 1)   // 2
        {
            iyCur += 2;
        }
        else                        // 0
        {
            ++iyCur;
        }
        if (iyCur >= imageH)
        {
            iyCur = ++iyOfs;
        }
    }

    //-----------------------------------------------------------------------------
    // パレットを解放します。
    if (pPalBuf != NULL)
    {
        delete[] pPalBuf;
    }
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief TGA ファイルのデータをデコードします。
//-----------------------------------------------------------------------------
bool RImage::DecodeTga(const uint8_t* fileBuf, const size_t fileSize)
{
    //-----------------------------------------------------------------------------
    // ヘッダをデコードします。
    const int imageType = fileBuf[0x02];
    if (imageType !=  1 &&  // CI
        imageType !=  2 &&  // RGB
        imageType !=  3 &&  // GRAY
        imageType !=  9 &&  // CI (RLE)
        imageType != 10 &&  // RGB (RLE)
        imageType != 11)    // GRAY (RLE)
    {
        m_ErrorString = "TGA file is wrong: (invalid image type): " + m_FilePath; // RShowError
        return false;
    }

    const int bitPerPixel = fileBuf[0x10];
    if (bitPerPixel !=  8 &&
        bitPerPixel != 15 &&
        bitPerPixel != 16 &&
        bitPerPixel != 24 &&
        bitPerPixel != 32)
    {
        m_ErrorString = "TGA file is wrong: (invalid bit per pixel): " + m_FilePath; // RShowError
        return false;
    }

    m_ImageW = fileBuf[0x0c] + (fileBuf[0x0d] << 8);
    m_ImageH = fileBuf[0x0e] + (fileBuf[0x0f] << 8);

    //-----------------------------------------------------------------------------
    // アルファの有無を取得します。
    if (fileBuf[0x01] != 0) // color palette flag
    {
        // CI
        m_HasAlpha = (fileBuf[0x07] == 32); // palette color bit size
    }
    else
    {
        // RGB or GRAY
        const int attrBitSize = fileBuf[0x11] & 0x0f;
        m_HasAlpha = (attrBitSize != 0);
        if ((imageType == 3 || imageType == 11) && // GRAY
            bitPerPixel == 8)
        {
            m_HasAlpha = false;
        }
    }

    //-----------------------------------------------------------------------------
    // ビットマップをデコードします。
    const int fullBitmapSize = m_ImageW * m_ImageH * R_RGBA_BYTES;
    m_pImageData = new uint8_t[fullBitmapSize];
    DecodeTgaBitmap(reinterpret_cast<uint8_t*>(m_pImageData), fileBuf, m_ImageW, m_ImageH);

    R_UNUSED_VARIABLE(fileSize);

    return true;
}

//-----------------------------------------------------------------------------
//! @brief ファイルから画像データをリードします。
//-----------------------------------------------------------------------------
bool RImage::ReadFile(const std::string& filePath) // RImage_ReadFile
{
    //-----------------------------------------------------------------------------
    // free memory
    FreeMemory();

    //-----------------------------------------------------------------------------
    // check type
    m_FilePath = filePath;
    const std::string ext = RGetExtensionFromFilePath(m_FilePath);
    if (ext != "psd" && ext != "tga")
    {
        m_ErrorString = "Image type is not supported: " + m_FilePath; // RShowError
        return false;
    }

    //-----------------------------------------------------------------------------
    // read file
    RFileBuf fileBuf(m_FilePath);
    if (!fileBuf)
    {
        m_ErrorString = "Cannot open the file: " + m_FilePath; // RShowError
        return false;
    }

    //-----------------------------------------------------------------------------
    // decode file
    bool success = false;
    if (ext == "psd")
    {
        success = DecodePsd(fileBuf.GetBuf(), fileBuf.GetSize());
    }
    else if (ext == "tga")
    {
        success = DecodeTga(fileBuf.GetBuf(), fileBuf.GetSize());
    }
    return success;
}

//-----------------------------------------------------------------------------
//! @brief ビットマップを水平垂直に反転します。
//-----------------------------------------------------------------------------
void RFlipHVBitmap(uint8_t* dstBuf, const int dstW, const int dstH)
{
    // uint32_t 型のポインタを使用して 4 bytes 単位で処理
    const int pixSize = dstW * dstH;
    uint32_t* fp = reinterpret_cast<uint32_t*>(dstBuf);
    uint32_t* bp = fp + pixSize - 1;
    const int pixSizeHalf = pixSize / 2;
    for (int iPix = 0; iPix < pixSizeHalf; ++iPix)
    {
        uint32_t fval = *fp;
        *fp++ = *bp;
        *bp-- = fval;
    }
}

//-----------------------------------------------------------------------------
//! @brief ビットマップのすべてのピクセルで R = G = B なら true を返します。
//-----------------------------------------------------------------------------
bool RIsAllGrayBitmap(
    const void* pData,
    const int width,
    const int height,
    const bool isFloat
)
{
    const int pixCount = width * height;
    if (!isFloat)
    {
        const uint8_t* pSrc = reinterpret_cast<const uint8_t*>(pData);
        for (int iPix = 0; iPix < pixCount; ++iPix)
        {
            if (pSrc[0] != pSrc[1] || pSrc[0] != pSrc[2])
            {
                return false;
            }
            pSrc += R_RGBA_COUNT;
        }
    }
    else
    {
        const float* pSrcF32 = reinterpret_cast<const float*>(pData);
        for (int iPix = 0; iPix < pixCount; ++iPix)
        {
            if (pSrcF32[0] != pSrcF32[1] || pSrcF32[0] != pSrcF32[2])
            {
                return false;
            }
            pSrcF32 += R_RGBA_COUNT;
        }
    }
    return true;
}

//-----------------------------------------------------------------------------
//! @brief ビットマップの透明モードを返します。
//-----------------------------------------------------------------------------
RImage::TransparencyMode RGetBitmapTransparencyMode(
    const void* pData,
    const int width,
    const int height,
    const bool isFloat
)
{
    RImage::TransparencyMode xpaMode = RImage::OPA;
    const int pixCount = width * height;
    if (!isFloat)
    {
        const uint8_t* pSrc = reinterpret_cast<const uint8_t*>(pData);
        for (int iPix = 0; iPix < pixCount; ++iPix)
        {
            const uint8_t alpha = pSrc[3];
            if (alpha == 0x00)
            {
                xpaMode = RImage::MASK;
                // 残りの部分に中間的なアルファが存在する可能性があるので return しません。
            }
            else if (alpha < 0xff)
            {
                return RImage::XLU;
            }
            pSrc += R_RGBA_COUNT;
        }
    }
    else
    {
        const float* pSrcF32 = reinterpret_cast<const float*>(pData);
        for (int iPix = 0; iPix < pixCount; ++iPix)
        {
            const float alpha = pSrcF32[3];
            if (alpha == 0.0f)
            {
                xpaMode = RImage::MASK;
                // 残りの部分に中間的なアルファが存在する可能性があるので return しません。
            }
            else if (alpha < 1.0f)
            {
                return RImage::XLU;
            }
            pSrcF32 += R_RGBA_COUNT;
        }
    }
    return xpaMode;
}

//-----------------------------------------------------------------------------
//! @brief ピクセルデータの全成分を指定した値でフィルします。
//-----------------------------------------------------------------------------
void FillPixelsValue(
    uint8_t* pPixels,
    const size_t pixelDataSize,
    const uint8_t value,
    const float floatValue,
    const bool isFloat
)
{
    if (!isFloat)
    {
        memset(pPixels, value, pixelDataSize);
    }
    else
    {
        float* pDstF32 = reinterpret_cast<float*>(pPixels);
        const size_t valueCount = pixelDataSize / sizeof(float);
        for (size_t valueIdx = 0; valueIdx < valueCount; ++valueIdx)
        {
            *pDstF32++ = floatValue;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ピクセルデータの RGBA 成分をスワップします。
//-----------------------------------------------------------------------------
bool SwapPixelsRgbaComponent(
    uint8_t* pPixels,
    const size_t pixelDataSize,
    const bool isFloat,
    const int compSel
)
{
    const int compR = ((compSel >> FtxCompSelShiftR) & 0xff) % FtxComponent_Count;
    const int compG = ((compSel >> FtxCompSelShiftG) & 0xff) % FtxComponent_Count;
    const int compB = ((compSel >> FtxCompSelShiftB) & 0xff) % FtxComponent_Count;
    const int compA = ((compSel >> FtxCompSelShiftA) & 0xff) % FtxComponent_Count;
    const bool hasAlpha = (compA != FtxComponent_One);

    const size_t pixelBytes = (isFloat) ? R_RGBA_FLOAT_BYTES : R_RGBA_BYTES;
    const size_t pixelCount = pixelDataSize / pixelBytes;
    if (!isFloat)
    {
        uint8_t srcCompsU8[FtxComponent_Count] = { 0 };
        srcCompsU8[FtxComponent_Zero] = 0x00;
        srcCompsU8[FtxComponent_One ] = 0xff;
        uint8_t* pDstU8 = pPixels;
        for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
        {
            srcCompsU8[FtxComponent_Red  ] = pDstU8[0];
            srcCompsU8[FtxComponent_Green] = pDstU8[1];
            srcCompsU8[FtxComponent_Blue ] = pDstU8[2];
            srcCompsU8[FtxComponent_Alpha] = pDstU8[3];
            pDstU8[0] = srcCompsU8[compR];
            pDstU8[1] = srcCompsU8[compG];
            pDstU8[2] = srcCompsU8[compB];
            pDstU8[3] = srcCompsU8[compA];
            pDstU8 += R_RGBA_COUNT;
        }
    }
    else
    {
        float srcCompsF32[FtxComponent_Count] = { 0.0f };
        srcCompsF32[FtxComponent_Zero] = 0.0f;
        srcCompsF32[FtxComponent_One ] = 1.0f;
        float* pDstF32 = reinterpret_cast<float*>(pPixels);
        for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
        {
            srcCompsF32[FtxComponent_Red  ] = pDstF32[0];
            srcCompsF32[FtxComponent_Green] = pDstF32[1];
            srcCompsF32[FtxComponent_Blue ] = pDstF32[2];
            srcCompsF32[FtxComponent_Alpha] = pDstF32[3];
            pDstF32[0] = srcCompsF32[compR];
            pDstF32[1] = srcCompsF32[compG];
            pDstF32[2] = srcCompsF32[compB];
            pDstF32[3] = srcCompsF32[compA];
            pDstF32 += R_RGBA_COUNT;
        }
    }
    return hasAlpha;
}

//-----------------------------------------------------------------------------
//! @brief Mac の UI コントロールハンドルを返します。
//-----------------------------------------------------------------------------
#ifdef __PIMac__
ControlHandle GetControlHandleMac(const DialogPtr pDlg, const short id)
{
    short type;
    Handle handle;
    Rect rect;
    GetDialogItem(pDlg, id, &type, &handle, &rect);
    return (ControlHandle)handle;
}
#endif

//-----------------------------------------------------------------------------
//! @brief Mac の UI コントロールのアクティブ状態を設定します。
//-----------------------------------------------------------------------------
#ifdef __PIMac__
void ActivateControlMac(const DialogPtr pDlg, const short id,
    const bool activeFlag)
{
    ControlHandle hCtrl = GetControlHandleMac(pDlg, id);
    if (activeFlag)
    {
        ActivateControl(hCtrl);
    }
    else
    {
        DeactivateControl(hCtrl);
    }
}
#endif

//-----------------------------------------------------------------------------
//! @brief Mac の UI コントロール用の文字列を返します。
//!        '\n' を '\r' に変換します。
//-----------------------------------------------------------------------------
#ifdef __PIMac__
void GetStringForMacUI(Str255& dst, const char* src)
{
    int size = strlen(src);
    if (size > 255)
    {
        size = 255;
    }
    dst[0] = static_cast<uint8_t>(size);

    for (int ic = 0; ic < size; ++ic)
    {
        char c = src[ic];
        if (c == '\n')
        {
            c = '\r';
        }
        dst[1 + ic] = c;
    }
}
#endif

//-----------------------------------------------------------------------------
//! @brief Mac の UI コントロールにテキストを設定します。
//-----------------------------------------------------------------------------
#ifdef __PIMac__
void SetTextMacUI(DialogPtr pDlg, const short id, const char* text)
{
    Str255 str;
    GetStringForMacUI(str, text);
    StuffText(pDlg, id, str);
}
#endif

//-----------------------------------------------------------------------------
//! @brief Mac の UI コントロール用 ParamText を設定します。
//-----------------------------------------------------------------------------
#ifdef __PIMac__
void ParamTextForMacUI(
    const char* text0,
    const char* text1,
    const char* text2,
    const char* text3
)
{
    Str255 str0 = "\p";
    Str255 str1 = "\p";
    Str255 str2 = "\p";
    Str255 str3 = "\p";

    if (text0 != NULL)
    {
        GetStringForMacUI(str0, text0);
    }
    if (text1 != NULL)
    {
        GetStringForMacUI(str0, text1);
    }
    if (text2 != NULL)
    {
        GetStringForMacUI(str0, text2);
    }
    if (text3 != NULL)
    {
        GetStringForMacUI(str0, text3);
    }

    ParamText(str0, str1, str2, str3);
}
#endif

//-----------------------------------------------------------------------------
//! @brief プラグインのエラーを表示します。
//-----------------------------------------------------------------------------
void RShowError(GPtr globals, const char* format, ...)
{
    char buf[1024];
    va_list args;
    va_start(args, format);
    const int size = vsnprintf_s(buf, _TRUNCATE, format, args);
    va_end(args);
    #ifdef __PIWin__
    #ifdef gStuff
        PlatformData* platform = reinterpret_cast<PlatformData*>(gStuff->platformData);
        HWND hWnd = reinterpret_cast<HWND>(platform->hwnd);
    #else
        HWND hWnd = reinterpret_cast<HWND>(globals->mHwnd);
    #endif
    MessageBox(hWnd, buf, "Error", MB_OK | MB_ICONERROR);
    #else
    ParamTextForMacUI(buf);
    StopAlert(AlertID, NULL);
    #endif
    R_UNUSED_VARIABLE(size);
}

//-----------------------------------------------------------------------------
//! @brief プラグインの警告を表示します。
//-----------------------------------------------------------------------------
void RShowWarning(GPtr globals, const char* format, ...)
{
    char buf[1024];
    va_list args;
    va_start(args, format);
    const int size = vsnprintf_s(buf, _TRUNCATE, format, args);
    va_end(args);
    #ifdef __PIWin__
    #ifdef gStuff
        PlatformData* platform = reinterpret_cast<PlatformData*>(gStuff->platformData);
        HWND hWnd = reinterpret_cast<HWND>(platform->hwnd);
    #else
        HWND hWnd = reinterpret_cast<HWND>(globals->mHwnd);
    #endif
    MessageBox(hWnd, buf, "Warning", MB_OK | MB_ICONWARNING);
    #else
    ParamTextForMacUI(buf);
    CautionAlert(AlertID, NULL);
    #endif
    R_UNUSED_VARIABLE(size);
}

//-----------------------------------------------------------------------------
//! @brief printf の書式でメッセージボックスを表示します（デバッグ用）。
//-----------------------------------------------------------------------------
void RMsgBox(const char* format, ...)
{
    char buf[1024];
    va_list args;
    va_start(args, format);
    const int size = vsnprintf_s(buf, _TRUNCATE, format, args);
    va_end(args);
    #ifdef __PIWin__
    MessageBox(NULL, buf, "", MB_OK);
    #else
    ParamTextForMacUI(buf);
    Alert(AlertID, NULL);
    #endif
    R_UNUSED_VARIABLE(size);
}

//-----------------------------------------------------------------------------
//! @brief printf の書式でデバッグウィンドウに出力します（デバッグ用）。
//-----------------------------------------------------------------------------
void RTrace(const char* format, ...)
{
    #ifdef _DEBUG
    char buf[1024];
    va_list args;
    va_start(args, format);
    int size = vsnprintf_s(buf, _TRUNCATE, format, args);
    va_end(args);
    #ifdef __PIWin__
    OutputDebugString(buf);
    #else
    Str255 str;
    GetStringForMacUI(str, buf);
    DebugStr(str);
    #endif
    R_UNUSED_VARIABLE(size);
    #else
    R_UNUSED_VARIABLE(format);
    #endif
}

//-----------------------------------------------------------------------------
//! @brief デバッグウィンドウにメモリーダンプを出力します（デバッグ用）。
//-----------------------------------------------------------------------------
void RDumpMemory(const void* mem, const int size, const char* title)
{
    char strBuf[16];

    #ifdef __PIWin__
    if (title != NULL)
    {
        OutputDebugString(title);
        OutputDebugString("\n");
    }
    const int COLUMN_SIZE = 16;
    int ic = 0;
    for (int iByte = 0; iByte < size; ++iByte)
    {
        sprintf_s(strBuf, "%02x ", static_cast<int>(
            *(reinterpret_cast<const uint8_t*>(mem) + iByte)));
        OutputDebugString(strBuf);
        if (++ic == COLUMN_SIZE && iByte < size - 1)
        {
            OutputDebugString("\n");
            ic = 0;
        }
    }
    OutputDebugString("\n");
    #endif
}

//-----------------------------------------------------------------------------
//! @brief メッセージボックスにメモリーダンプを出力します（デバッグ用）。
//-----------------------------------------------------------------------------
void RDumpMemoryBox(const void* mem, const int size, const char* title)
{
    char buf[1024];
    char* dst = buf;
    if (title != NULL)
    {
        strcpy_s(dst, buf + sizeof(buf) - dst, title);
        dst += strlen(title);
        *dst++ = '\n';
    }
    const int COLUMN_SIZE = 16;
    int ic = 0;
    const uint8_t* pMem = reinterpret_cast<const uint8_t*>(mem);
    for (int iByte = 0; iByte < size; ++iByte)
    {
        sprintf_s(dst, buf + sizeof(buf) - dst, "%02x ", pMem[iByte]);
        dst += 3;
        if (++ic == COLUMN_SIZE && iByte < size - 1)
        {
            *dst++ = '\n';
            ic = 0;
        }
    }
    *dst = '\0';
    #ifdef __PIWin__
    MessageBox(NULL, buf, "", MB_OK);
    #else
    ParamTextForMacUI(buf);
    Alert(AlertID, NULL);
    #endif
}

//-----------------------------------------------------------------------------
//! @brief 開いているメモ帳のウィンドウにメッセージを出力します（デバッグ用）。
//-----------------------------------------------------------------------------
#ifdef __PIWin__
void RNoteTrace(const char* format, ...)
{
    HWND hWnd = FindWindow("Notepad", NULL);
    if (!hWnd)
    {
        return;
    }

    static const POINT pt = { 0, 0 };
    hWnd = ChildWindowFromPoint(hWnd, pt);
    if (!hWnd)
    {
        return;
    }

    char buf[1024];
    va_list args;
    va_start(args, format);
    const int size = vsnprintf_s(buf, _TRUNCATE, format, args);
    va_end(args);

    #if 1
    // 下の行ほど最新
    SendMessage(hWnd, EM_REPLACESEL, 0, (LPARAM)buf);
    SendMessage(hWnd, EM_REPLACESEL, 0, (LPARAM)"\r\n");
    #else
    // 上の行ほど最新
    SendMessage(hWnd, EM_SETSEL, 0, 0);
    SendMessage(hWnd, EM_REPLACESEL, 0, (LPARAM)"\r\n");
    SendMessage(hWnd, EM_SETSEL, 0, 0);
    SendMessage(hWnd, EM_REPLACESEL, 0, (LPARAM)buf);
    #endif
    R_UNUSED_VARIABLE(size);
}
#endif

//-----------------------------------------------------------------------------
//! @brief 開いているメモ帳のウィンドウにメモリーダンプを出力します（デバッグ用）。
//-----------------------------------------------------------------------------
#ifdef __PIWin__
void RNoteDumpMemory(const void* mem, const int size, const char* title)
{
    char strBuf[256];

    if (title != NULL)
    {
        RNoteTrace(title);
    }

    const int COLUMN_SIZE = 16;
    const uint8_t* pMem = reinterpret_cast<const uint8_t*>(mem);
    for (int iByte = 0; iByte < size; iByte += COLUMN_SIZE)
    {
        const int colSize = (iByte + COLUMN_SIZE <= size) ?
            COLUMN_SIZE : size - iByte;
        char* dst = strBuf;
        for (int ic = 0; ic < colSize; ++ic)
        {
            sprintf_s(dst, strBuf + sizeof(strBuf) - dst, "%02x ", pMem[iByte + ic]);
            dst += 3;
        }
        RNoteTrace(strBuf);
    }
}
#endif

//-----------------------------------------------------------------------------
//! @brief メモリーの状態をデバッグウィンドウに出力します（デバッグ用）。
//-----------------------------------------------------------------------------
void RCheckMemoryStatus(const char* title)
{
    #ifdef __PIWin__
    MEMORYSTATUS ms;
    ms.dwLength = sizeof(MEMORYSTATUS);
    GlobalMemoryStatus(&ms);
    char str[256];
    sprintf_s(str, "%17s: %lld / %lld", title, ms.dwAvailPhys, ms.dwTotalPhys);
    RNoteTrace(str);
    #endif
}

//-----------------------------------------------------------------------------
//! @brief Mac のアバウトダイアログ表示の共通処理です。
//-----------------------------------------------------------------------------
#ifdef __PIMac__
void ShowAboutCommonMac(const char* message, const char* title)
{
    const OSType hostSig = kPhotoshopSignature;
    const short dialogID = AboutID;

    //-----------------------------------------------------------------------------
    // get resource
    DialogTHndl tDlg = (DialogTHndl)GetResource('DLOG', dialogID);
    HNoPurge((Handle)tDlg);
    CenterDialog(tDlg);

    //-----------------------------------------------------------------------------
    // create dialog
    DialogPtr pDlg = GetNewDialog(dialogID, NULL, (WindowPtr)(-1));

    // resize
    //AutoSizeDialog(pDlg);
    const int NEW_WD = 550, NEW_HT = 130;
    SizeWindow(GetDialogWindow(pDlg), NEW_WD, NEW_HT, TRUE);

    const short ID_CANCEL = 2;
    const short ID_TEXT = 3;
    for (short id = ID_CANCEL; id <= ID_TEXT; ++id)
    {
        short itemType;
        Handle itemHandle;
        Rect itemRect;
        GetDialogItem(pDlg, id, &itemType, &itemHandle, &itemRect);
        //RMsgBox("item: %d %d %d %d", itemRect.left, itemRect.top, itemRect.right, itemRect.bottom);
        itemRect.right = itemRect.left + NEW_WD;
        SetDialogItem(pDlg, id, itemType, itemHandle, &itemRect);
    }

    // set text
    ParamTextForMacUI(message);
    char titleBuf[128];
    strcpy(titleBuf, title);
    #if !TARGET_CARBON
    strcat(titleBuf, " (Classic)");
    #endif
    Str255 strTitle;
    GetStringForMacUI(strTitle, titleBuf);
    SetWTitle(GetDialogWindow(pDlg), strTitle);

    // set default item
    (void)SetDialogDefaultItem(pDlg, ok); // hidden
    (void)SetDialogCancelItem(pDlg, ok); // either

    // set cursor
    SetArrowCursor(); // Requires QuickDraw globals

    //-----------------------------------------------------------------------------
    // show dialog
    short idRet;
    MoveableModalDialog(pDlg, NULL, NULL, &idRet);

    //-----------------------------------------------------------------------------
    // delete dialog
    DisposeDialog(pDlg);
    HPurge((Handle)tDlg);
    ReleaseResource((Handle)tDlg);

    //RMsgBox("test: %d", 3*4);
}
#endif

//-----------------------------------------------------------------------------
//! @brief Photoshop のバージョンが指定したバージョン以上かチェックします。
//-----------------------------------------------------------------------------
bool RIsPhotoshopVersionGreaterEqual(const int major, const int minor)
{
    int32 curMajor;
    int32 curMinor;
    int32 curFix;
    PIGetHostVersion(curMajor, curMinor, curFix);
    return (curMajor == major) ? (curMinor >= minor) : (curMajor > major);
}

//-----------------------------------------------------------------------------
//! @brief Nintendo Photoshop プラグイン用リソースデータのバージョンを返します。
//!        バージョン情報が存在しなければ 0 を返します。
//-----------------------------------------------------------------------------
#ifdef gStuff
int RGetResourceDataVersion(GPtr globals)
{
    const int resCount = CountPIResources(RD_TYPE_NN);
    for (int ires = 1; ires <= resCount; ++ires)
    {
        Handle hdl = GetPIResource(RD_TYPE_NN, static_cast<int16>(ires));
        const uint8_t* dataPtr = reinterpret_cast<uint8_t*>(*hdl);
        if (strncmp(reinterpret_cast<const char*>(dataPtr), RD_TAG_RD_VER, AI_TAG_SIZE) == 0)
        {
            uint32_t dataBlockSize;
            RGetMemLittle(dataBlockSize, dataPtr + AI_TAG_SIZE);
            if (dataBlockSize == 0)
            {
                return 0;
            }
            const uint8_t* dataBuf = dataPtr + AI_HEADER_SIZE;
            const uint32_t dataSize = dataBlockSize - AI_HEADER_SIZE;
            if (dataSize == 0)
            {
                return 0;
            }
            char* strBuf = new char[dataSize + 1];
            memcpy(strBuf, dataBuf, dataSize);
            strBuf[dataSize] = '\0';
            int version = atol(strBuf);
            delete[] strBuf;
            return version;
        }
    }
    return 0;
}
#endif

//-----------------------------------------------------------------------------
//! @brief Nintendo Photoshop プラグイン用リソースデータからミップマップのレベル数を取得します。
//!        ミップマップのレベル数情報が存在しなければ 1 を返します。
//!        現在使用されていません。
//-----------------------------------------------------------------------------
#ifdef gStuff
int RGetResourceDataMipLevel(GPtr globals)
{
    const int resCount = CountPIResources(RD_TYPE_NN);
    for (int ires = 1; ires <= resCount; ++ires)
    {
        Handle hdl = GetPIResource(RD_TYPE_NN, static_cast<int16>(ires));
        const uint8_t* dataPtr = reinterpret_cast<uint8_t*>(*hdl);
        if (strncmp(reinterpret_cast<const char*>(dataPtr), RD_TAG_MIP_LEVEL, AI_TAG_SIZE) == 0)
        {
            uint32_t dataBlockSize;
            RGetMemLittle(dataBlockSize, dataPtr + AI_TAG_SIZE);
            if (dataBlockSize == 0)
            {
                return 1;
            }
            const uint8_t* dataBuf = dataPtr + AI_HEADER_SIZE;
            const uint32_t dataSize = dataBlockSize - AI_HEADER_SIZE;
            if (dataSize == 0)
            {
                return 1;
            }
            char* strBuf = new char[dataSize + 1];
            memcpy(strBuf, dataBuf, dataSize);
            strBuf[dataSize] = '\0';
            int mipLevel = atol(strBuf);
            delete[] strBuf;
            if (mipLevel <= 0 || mipLevel > RImage::MipCountMax)
            {
                return 1;
            }
            return mipLevel;
        }
    }
    return 1;
}
#endif

//-----------------------------------------------------------------------------
//! @brief 指定した Nintendo Photoshop プラグイン用リソースデータが存在すれば true を返します。
//-----------------------------------------------------------------------------
#ifdef gStuff
bool RDoesResourceDataExist(GPtr globals, const char* tagName)
{
    const int resCount = CountPIResources(RD_TYPE_NN);
    for (int ires = 1; ires <= resCount; ++ires)
    {
        Handle hdl = GetPIResource(RD_TYPE_NN, static_cast<int16>(ires));
        const uint8_t* dataPtr = reinterpret_cast<uint8_t*>(*hdl);
        if (strncmp(reinterpret_cast<const char*>(dataPtr), tagName, AI_TAG_SIZE) == 0)
        {
            return TRUE;
        }
    }
    return FALSE;
}
#endif

//-----------------------------------------------------------------------------
//! @brief 指定した Nintendo Photoshop プラグイン用リソースデータを削除します。
//-----------------------------------------------------------------------------
#ifdef gStuff
bool RDeleteResourceData(GPtr globals, const char* tagName)
{
    const int resCount = CountPIResources(RD_TYPE_NN);
    for (int ires = 1; ires <= resCount; ++ires)
    {
        Handle hdl = GetPIResource(RD_TYPE_NN, static_cast<int16>(ires));
        const uint8_t* dataPtr = reinterpret_cast<uint8_t*>(*hdl);
        if (strncmp(reinterpret_cast<const char*>(dataPtr), tagName, AI_TAG_SIZE) == 0)
        {
            DeletePIResource(RD_TYPE_NN, static_cast<int16>(ires));
            return TRUE;
        }
    }
    return FALSE;
}
#endif

//-----------------------------------------------------------------------------
//! @brief Nintendo Photoshop プラグイン用リソースデータをすべて削除します。
//-----------------------------------------------------------------------------
#ifdef gStuff
void RDeleteResourceDataAll(GPtr globals)
{
    int resCount = CountPIResources(RD_TYPE_NN);
    //RMsgBox("del res: %d", resCount);
    while (resCount > 0)
    {
        DeletePIResource(RD_TYPE_NN, 1);
        // delete するたびにインデックスが変わるので注意
        resCount = CountPIResources(RD_TYPE_NN);
    }
}
#endif

//-----------------------------------------------------------------------------
//! @brief Nintendo Photoshop プラグイン用リソースデータブロックを追加します。
//-----------------------------------------------------------------------------
#ifdef gStuff
void RAddRDBlock(
    GPtr globals,
    const char* tagName,
    const void* pData,
    const uint32_t dataSize
)
{
    Handle hdl = PINewHandle(AI_HEADER_SIZE + dataSize);
    if (hdl == NULL)
    {
        return;
    }

    uint8_t* dataPtr = reinterpret_cast<uint8_t*>(*hdl);
    memcpy(dataPtr, tagName, AI_TAG_SIZE);
    RSetMemLittle(dataPtr + AI_TAG_SIZE, AI_HEADER_SIZE + dataSize);
    if (pData != NULL && dataSize > 0)
    {
        memcpy(dataPtr + AI_HEADER_SIZE, pData, dataSize);
    }

    PILockHandle(hdl, FALSE);
    AddPIResource(RD_TYPE_NN, hdl);
    PIUnlockHandle(hdl);
    PIDisposeHandle(hdl);
}

void RAddRDBlock(GPtr globals, const char* tagName, const std::string& dataStr)
{
    RAddRDBlock(globals, tagName, dataStr.c_str(), static_cast<int>(dataStr.size()));
}
#endif

//-----------------------------------------------------------------------------
//! @brief リソースデータ解析テーブルに従って Nintendo Photoshop プラグイン用リソースデータを解析します。
//-----------------------------------------------------------------------------
#ifdef gStuff
void RDoRDParseTable(
    GPtr globals,
    const RDParse* parseTable,
    const uint32_t parseSize
)
{
    const int resCount = CountPIResources(RD_TYPE_NN);
    for (int ires = 1; ires <= resCount; ++ires)
    {
        Handle hdl = GetPIResource(RD_TYPE_NN, static_cast<int16>(ires));
        const uint8_t* dataPtr = reinterpret_cast<uint8_t*>(*hdl);
        uint32_t dataBlockSize;
        RGetMemLittle(dataBlockSize, dataPtr + AI_TAG_SIZE);
        if (dataBlockSize == 0)
        {
            continue;
        }
        uint32_t itag;
        for (itag = 0; itag < parseSize; ++itag)
        {
            const RDParse& parse = parseTable[itag];
            if (strncmp(reinterpret_cast<const char*>(dataPtr), parse.m_Tag, AI_TAG_SIZE) == 0)
            {
                if (parse.m_Proc != NULL)
                {
                    //RMsgBox("get res: %s", parse.tag);
                    parse.m_Proc(globals, dataPtr + AI_HEADER_SIZE,
                        dataBlockSize - AI_HEADER_SIZE);
                }
                break;
            }
        }
    }
}
#endif

//-----------------------------------------------------------------------------
//! @brief プロパティに対応したアクション記述子を取得します。
//-----------------------------------------------------------------------------
OSErr RGetActionDescriptorForProperty(
    PIActionDescriptor* pDesc,
    const int index,
    const DescriptorClassID desiredClass,
    const DescriptorKeyID desiredKey
)
{
    Auto_Ref classRef;
    OSErr error = sPSActionReference->PutProperty(classRef.get(), classProperty, desiredKey);
    if (error == noErr)
    {
        if (index == -1)
        {
            error = sPSActionReference->PutEnumerated(classRef.get(), desiredClass, typeOrdinal, enumTarget);
        }
        else
        {
            error = sPSActionReference->PutIndex(classRef.get(), desiredClass, index);
        }
        if (error == noErr)
        {
            error = sPSBasicActionControl->Get(pDesc, classRef.get());
        }
    }
    return error;
}

//-----------------------------------------------------------------------------
//! @brief アクション記述子から整数型プロパティを取得します。
//-----------------------------------------------------------------------------
int RGetIntegerProperty(
    OSErr* pError,
    const int index,
    const DescriptorClassID desiredClass,
    const DescriptorKeyID desiredKey
)
{
    int result = 0;
    Auto_Desc resultDesc;
    OSErr error = RGetActionDescriptorForProperty(&resultDesc, index, desiredClass, desiredKey);
    if (error == noErr)
    {
        error = sPSActionDescriptor->GetInteger(resultDesc.get(),
            desiredKey, reinterpret_cast<int32*>(&result));
    }
    if (pError != nullptr)
    {
        *pError = error;
    }
    return result;
}

//-----------------------------------------------------------------------------
//! @brief アクション記述子から実数型プロパティを取得します。
//-----------------------------------------------------------------------------
double RGetFloatProperty(
    OSErr* pError,
    const int index,
    const DescriptorClassID desiredClass,
    const DescriptorKeyID desiredKey
)
{
    double result = 0.0;
    Auto_Desc resultDesc;
    OSErr error = RGetActionDescriptorForProperty(&resultDesc, index, desiredClass, desiredKey);
    if (error == noErr)
    {
        error = sPSActionDescriptor->GetFloat(resultDesc.get(), desiredKey, &result);
    }
    if (pError != nullptr)
    {
        *pError = error;
    }
    return result;
}

//-----------------------------------------------------------------------------
//! @brief アクション記述子から unit float 型プロパティを取得します。
//-----------------------------------------------------------------------------
double RGetUnitFloatProperty(
    OSErr* pError,
    DescriptorUnitID* pUnitId,
    const int index,
    const DescriptorClassID desiredClass,
    const DescriptorKeyID desiredKey
)
{
    double result = 0.0;
    DescriptorUnitID unitId = 0;
    Auto_Desc resultDesc;
    OSErr error = RGetActionDescriptorForProperty(&resultDesc, index, desiredClass, desiredKey);
    if (error == noErr)
    {
        error = sPSActionDescriptor->GetUnitFloat(resultDesc.get(),
            desiredKey, &unitId, &result);
    }
    if (pError != nullptr)
    {
        *pError = error;
    }
    if (pUnitId != nullptr)
    {
        *pUnitId = unitId;
    }
    return result;
}

//-----------------------------------------------------------------------------
//! @brief アクション記述子から文字列型プロパティを取得します。
//-----------------------------------------------------------------------------
std::string RGetStringProperty(
    OSErr* pError,
    const int index,
    const DescriptorClassID desiredClass,
    const DescriptorKeyID desiredKey
)
{
    std::string result;
    Auto_Desc resultDesc;
    OSErr error = RGetActionDescriptorForProperty(&resultDesc, index, desiredClass, desiredKey);
    if (error == noErr)
    {
        uint32 length = 0;
        error = sPSActionDescriptor->GetStringLength(resultDesc.get(), desiredKey, &length);
        if (error == noErr && length > 0)
        {
            char* strBuf = new char[length + 1];
            error = sPSActionDescriptor->GetString(resultDesc.get(), desiredKey, strBuf, length + 1);
            if (error == noErr)
            {
                result = std::string(strBuf, length);
            }
            delete[] strBuf;
        }
    }
    if (pError != nullptr)
    {
        *pError = error;
    }
    return result;
}

//-----------------------------------------------------------------------------
//! @brief アクション記述子から bool 型プロパティを取得します。
//-----------------------------------------------------------------------------
bool RGetBooleanProperty(
    OSErr* pError,
    const int index,
    const DescriptorClassID desiredClass,
    const DescriptorKeyID desiredKey
)
{
    bool result = false;
    Auto_Desc resultDesc;
    OSErr error = RGetActionDescriptorForProperty(&resultDesc, index, desiredClass, desiredKey);
    if (error == noErr)
    {
        Boolean value;
        error = sPSActionDescriptor->GetBoolean(resultDesc.get(), desiredKey, &value);
        if (error == noErr)
        {
            result = (value == TRUE);
        }
    }
    if (pError != nullptr)
    {
        *pError = error;
    }
    return result;
}

//-----------------------------------------------------------------------------
//! @brief アクション記述子から列挙型プロパティを取得します。
//-----------------------------------------------------------------------------
DescriptorEnumID RGetEnumeratedProperty(
    OSErr* pError,
    const int index,
    const DescriptorClassID desiredClass,
    const DescriptorKeyID desiredKey
)
{
    DescriptorEnumID result = 0;
    DescriptorEnumTypeID enumTypeId = 0;
    Auto_Desc resultDesc;
    OSErr error = RGetActionDescriptorForProperty(&resultDesc, index, desiredClass, desiredKey);
    if (error == noErr)
    {
        error = sPSActionDescriptor->GetEnumerated(resultDesc.get(),
            desiredKey, &enumTypeId, &result);
    }
    if (pError != nullptr)
    {
        *pError = error;
    }
    return result;
}

//-----------------------------------------------------------------------------
//! @brief ドキュメントのファイルパスを返します。
//-----------------------------------------------------------------------------
#ifdef gStuff
std::string RGetDocumentFilePath(GPtr globals)
{
    OSErr error;
    std::string docPath;
    int32 major, miner, fix;
    PIGetHostVersion(major, miner, fix);
    if (major >= PhotoshopVersionMajorCs6)
    {
        AutoSuite<PSBasicActionControlProcs> basicActionControl(
            kPSBasicActionControlSuite, kPSBasicActionControlSuiteVersion);
        Auto_Ref docRef;
        error = sPSActionReference->PutProperty(docRef.get(), classProperty, keyFileReference); // 高速化のため取得するプロパティを指定
        error = sPSActionReference->PutEnumerated(docRef.get(), classDocument, typeOrdinal, enumTarget);
        Auto_Desc docDesc;
        error = basicActionControl->Get(&docDesc, docRef.get());
        Handle hFileRef = NULL;
        error = sPSActionDescriptor->GetAlias(docDesc.get(), keyFileReference, &hFileRef);
             // 1 度も保存されていなければ -1715 を返します。
        //RNoteTrace("file ref: %d: %p", error, hFileRef);
        if (error == noErr && hFileRef != NULL)
        {
            // PIHandle2CString だと末尾に余計な文字が含まれるので修正版の RHandle2CString を使用します。
            char strBuf[512];
            RHandle2CString(gStuff->handleProcs, hFileRef, strBuf, sizeof(strBuf));
            PIDisposeHandle(hFileRef);
            docPath = strBuf;
        }
    }
    else
    {
        docPath = reinterpret_cast<SPPlatformFileSpecification*>(
            gStuff->documentInfo->fileSpec)->path; // 1 度も保存されていなければ空文字
    }
    return (!docPath.empty()) ? docPath : "untitled";
}
#endif

//-----------------------------------------------------------------------------
//! @brief 指定したチャンネルがリード可能なら true を返します。
//-----------------------------------------------------------------------------
bool RGetChannelReadFlag(const ReadChannelDesc* pChan)
{
    Boolean readFlag = FALSE; // FALSE で初期化する必要あり
    if (pChan != nullptr && pChan->port != nullptr)
    {
        if (sPSChannelProcs->CanRead(pChan->port, &readFlag) != noErr)
        {
            readFlag = FALSE;
        }
    }
    return (readFlag == TRUE);
}

//-----------------------------------------------------------------------------
//! @brief 指定したチャンネルがライト可能なら true を返します。
//-----------------------------------------------------------------------------
bool RGetChannelWriteFlag(const ReadChannelDesc* pChan)
{
    Boolean writeFlag = FALSE; // FALSE で初期化する必要あり
    if (pChan != nullptr && pChan->writePort != nullptr)
    {
        if (sPSChannelProcs->CanWrite(pChan->writePort, &writeFlag) != noErr)
        {
            writeFlag = FALSE;
        }
    }
    return (writeFlag == TRUE);
}

//-----------------------------------------------------------------------------
//! @brief チャンネル数を取得します。
//-----------------------------------------------------------------------------
int RGetChannelCount(const ReadChannelDesc* pChanTop)
{
    int chanCount = 0;
    const ReadChannelDesc* pChan = pChanTop;
    while (pChan != nullptr)
    {
        ++chanCount;
        pChan = pChan->next;
    }
    return chanCount;
}

//-----------------------------------------------------------------------------
//! @brief 選択されたチャンネル数を取得します。
//-----------------------------------------------------------------------------
int RGetSelectedChannelCount(const ReadChannelDesc* pChanTop)
{
    int chanCount = 0;
    const ReadChannelDesc* pChan = pChanTop;
    while (pChan != nullptr)
    {
        if (RGetChannelWriteFlag(pChan))
        {
            ++chanCount;
        }
        pChan = pChan->next;
    }
    return chanCount;
}

//-----------------------------------------------------------------------------
//! @brief 1 チャンネルのデータをパディングしながらピクセルデータに設定します。
//!        出力先ピクセルデータが大きい場合は端のピクセルの値を延長します。
//-----------------------------------------------------------------------------
void RSetOneChannelToPixelsPadding(
    uint8_t* dst,
    const uint8_t* src,
    const int dstW,
    const int dstH,
    const int srcW,
    const int srcH
)
{
    int iy;
    for (iy = 0; iy < srcH; ++iy)
    {
        int ix;
        for (ix = 0; ix < srcW; ++ix)
        {
            *dst = *src++;
            dst += R_RGBA_BYTES;
        }
        const uint8_t fill = *(src - 1);
        for ( ; ix < dstW; ++ix)
        {
            *dst = fill;
            dst += R_RGBA_BYTES;
        }
    }

    const uint8_t* bottomTop = dst - dstW * R_RGBA_BYTES;
    for ( ; iy < dstH; ++iy)
    {
        const uint8_t* bottom = bottomTop;
        for (int ix = 0; ix < dstW; ++ix)
        {
            *dst = *bottom;
            dst += R_RGBA_BYTES;
            bottom += R_RGBA_BYTES;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ピクセルデータから 1 チャンネルのデータを取得します。
//-----------------------------------------------------------------------------
void RGetOneChannelFromPixels(
    uint8_t* dst,
    const uint8_t* src,
    const int dstW,
    const int dstH,
    const int srcW
)
{
    for (int iy = 0; iy < dstH; ++iy)
    {
        for (int ix = 0; ix < dstW; ++ix)
        {
            *dst++ = *src;
            src += R_RGBA_BYTES;
        }
        src += (srcW - dstW) * R_RGBA_BYTES;
    }
}

//-----------------------------------------------------------------------------
//! @brief ピクセルデータの共通領域をコピーします。
//-----------------------------------------------------------------------------
void RCopyPixelsWithBounds(
    uint8_t* pDstPixels,
    const VRect& dstBounds,
    const size_t dstPixelBytes,
    const uint8_t* pSrcPixels,
    const VRect& srcBounds,
    const size_t srcPixelBytes,
    const bool multiplies
)
{
    //-----------------------------------------------------------------------------
    // コピー先とコピー元の共通領域がなければ何もしません。
    if (dstBounds.right  <= srcBounds.left ||
        srcBounds.right  <= dstBounds.left ||
        dstBounds.bottom <= srcBounds.top  ||
        srcBounds.bottom <= dstBounds.top)
    {
        return;
    }

    //-----------------------------------------------------------------------------
    // 共通領域を求めます。
    // パターン 1
    //   dddd          dddd
    //     ssss        ssssss

    // パターン 2
    //   dddd          dddd
    //    ss           ss

    // パターン 3
    //   dddd          dddd
    // ssss          ssssss

    // パターン 4
    //    dd
    //   ssss

    const int dstOfsX = (srcBounds.left > dstBounds.left) ?
        srcBounds.left - dstBounds.left : 0;
    const int dstOfsY = (srcBounds.top  > dstBounds.top ) ?
        srcBounds.top  - dstBounds.top  : 0;
    const int srcOfsX = (dstBounds.left > srcBounds.left) ?
        dstBounds.left - srcBounds.left : 0;
    const int srcOfsY = (dstBounds.top  > srcBounds.top ) ?
        dstBounds.top  - srcBounds.top  : 0;

    const int dstW = dstBounds.right  - dstBounds.left;
    const int dstH = dstBounds.bottom - dstBounds.top;
    const int srcW = srcBounds.right  - srcBounds.left;
    const int srcH = srcBounds.bottom - srcBounds.top;
    const int commonW = (dstBounds.left <= srcBounds.left) ?
        ((dstBounds.right  <= srcBounds.right ) ? dstBounds.right  - srcBounds.left : srcW) : // 1 : 2
        ((srcBounds.right  <= dstBounds.right ) ? srcBounds.right  - dstBounds.left : dstW);  // 3 : 4
    const int commonH = (dstBounds.top <= srcBounds.top) ?
        ((dstBounds.bottom <= srcBounds.bottom) ? dstBounds.bottom - srcBounds.top  : srcH) : // 1 : 2
        ((srcBounds.bottom <= dstBounds.bottom) ? srcBounds.bottom - dstBounds.top  : dstH);  // 3 : 4
    //RNoteTrace("common bounds: %d x %d: %dx%d (%d,%d) <- %dx%d (%d,%d): %d", commonW, commonH,
    //    dstW, dstH, dstOfsX, dstOfsY, srcW, srcH, srcOfsX, srcOfsY, multiplies);

    //-----------------------------------------------------------------------------
    // コピーします。
    const size_t dstLineBytes = dstPixelBytes * dstW;
    const size_t srcLineBytes = srcPixelBytes * srcW;
    uint8_t*       pDst = pDstPixels + dstLineBytes * dstOfsY + dstPixelBytes * dstOfsX;
    const uint8_t* pSrc = pSrcPixels + srcLineBytes * srcOfsY + srcPixelBytes * srcOfsX;
    for (int y = 0; y < commonH; ++y)
    {
        if (!multiplies)
        {
            for (int x = 0; x < commonW; ++x)
            {
                *pDst = *pSrc;
                pDst += dstPixelBytes;
                pSrc += srcPixelBytes;
            }
        }
        else
        {
            for (int x = 0; x < commonW; ++x)
            {
                if (*pSrc != 0xff)
                {
                    *pDst = static_cast<uint8_t>(*pDst * (*pSrc) / 0xff);
                }
                pDst += dstPixelBytes;
                pSrc += srcPixelBytes;
            }
        }
        pDst += dstPixelBytes * (dstW - commonW);
        pSrc += srcPixelBytes * (srcW - commonW);
    }
}

//-----------------------------------------------------------------------------
//! @brief チャンネルをリードしてピクセルデータに反映します。
//-----------------------------------------------------------------------------
void RReadChannelToPixels(
    uint8_t* pPixels,
    const VRect& dstBounds,
    const size_t dstPixelBytes,
    const ReadChannelDesc* pChan,
    const bool multiplies
)
{
    //-----------------------------------------------------------------------------
    // チャンネルをリードします。
    const VRect& srcBounds = pChan->bounds;
    const int srcW = srcBounds.right  - srcBounds.left;
    const int srcH = srcBounds.bottom - srcBounds.top;
    const size_t srcPixelCount = static_cast<size_t>(srcW) * srcH;
    const size_t srcPixelBytes = pChan->depth / 8;
    const size_t srcPixelDataSize = srcPixelBytes * srcPixelCount;
    uint8_t* chanBuf = new uint8_t[srcPixelDataSize];

    VRect bounds = srcBounds;
    PixelMemoryDesc memDesc;
    memDesc.data      = chanBuf;
    memDesc.depth     = pChan->depth;
    memDesc.colBits   = pChan->depth;
    memDesc.rowBits   = memDesc.colBits * srcW;
    memDesc.bitOffset = 0;
    sPSChannelProcs->ReadPixelsFromLevel(pChan->port, 0, &bounds, &memDesc);

    //-----------------------------------------------------------------------------
    // ピクセルデータに反映します。
    RCopyPixelsWithBounds(pPixels, dstBounds, dstPixelBytes,
        chanBuf, srcBounds, srcPixelBytes, multiplies);
    delete[] chanBuf;
}

//-----------------------------------------------------------------------------
//! @brief 複数チャンネルをリードしてピクセルデータに反映します。
//-----------------------------------------------------------------------------
void RReadMultiChannelsToPixels(
    uint8_t* pPixels,
    const VRect& dstBounds,
    const size_t dstPixelBytes,
    const ReadChannelDesc* pColorChan,
    const ReadChannelDesc* pTransparencyChan,
    const ReadChannelDesc* pMaskChan
)
{
    //-----------------------------------------------------------------------------
    // カラーチャンネルをリードします。
    int rgbIdx = 0;
    const ReadChannelDesc* pChan = pColorChan;
    while (pChan != nullptr && rgbIdx < R_RGB_COUNT)
    {
        RReadChannelToPixels(pPixels + rgbIdx, dstBounds, dstPixelBytes, pChan, false);
        ++rgbIdx;
        pChan = pChan->next;
    }

    //-----------------------------------------------------------------------------
    // ピクセルの不透明度チャンネルをリードします。
    if (pTransparencyChan != nullptr)
    {
        RReadChannelToPixels(pPixels + 3, dstBounds, dstPixelBytes, pTransparencyChan, false);
    }
    else
    {
        const int dstW = dstBounds.right  - dstBounds.left;
        const int dstH = dstBounds.bottom - dstBounds.top;
        const size_t pixelCount = static_cast<size_t>(dstW) * dstH;
        uint8_t* pA = pPixels + 3;
        for (size_t pixelIdx = 0; pixelIdx < pixelCount; ++pixelIdx)
        {
            *pA = 0xff;
            pA += dstPixelBytes;
        }
    }

    //-----------------------------------------------------------------------------
    // レイヤーマスクをリードして A 成分に乗算します。
    // レイヤーマスクが「使用しない」になっている場合はリードしません。
    if (pMaskChan != nullptr && pMaskChan->isEnabled == TRUE)
    {
        RReadChannelToPixels(pPixels + 3, dstBounds, dstPixelBytes, pMaskChan, true);
    }
}

//-----------------------------------------------------------------------------
//! @brief リード記述子から文字列パラメータを取得します。
//-----------------------------------------------------------------------------
#ifdef gStuff
bool RGetStringParam(
    GPtr globals,
    std::string* pString,
    const PIReadDescriptor& desc,
    const bool utf8ToSjis
)
{
    Handle hStr = nullptr;
    OSErr error = PIGetText(desc, &hStr);
    if (hStr != nullptr)
    {
        error = HandleToString(hStr, *pString);
        PIDisposeHandle(hStr);
        if (error == noErr && utf8ToSjis)
        {
            *pString = RGetShiftJisFromUtf8(*pString);
        }
    }
    return (error == noErr);
}
#endif

//-----------------------------------------------------------------------------
//! @brief リード記述子から文字列パラメータを取得します。
//-----------------------------------------------------------------------------
#ifdef gStuff
bool RGetStringParam(
    GPtr globals,
    char* stringBuf,
    const size_t maxStringSize,
    const PIReadDescriptor& desc,
    const bool utf8ToSjis
)
{
    std::string str;
    if (RGetStringParam(globals, &str, desc, utf8ToSjis))
    {
        memset(stringBuf, 0x00, maxStringSize);
        memcpy(stringBuf, str.c_str(), RMin(str.size(), maxStringSize - 1));
        return true;
    }
    return false;
}
#endif

//-----------------------------------------------------------------------------
//! @brief ライト記述子に文字列パラメータを設定します。
//-----------------------------------------------------------------------------
#ifdef gStuff
bool RPutStringParam(
    GPtr globals,
    const PIWriteDescriptor& token,
    const DescriptorKeyID key,
    const std::string& value
)
{
    Handle hStr = PICString2Handle(value.c_str());
    OSErr error = PIPutText(token, key, hStr);
    PIDisposeHandle(hStr);
    hStr = NULL;
    return (error == noErr);
}
#endif

//-----------------------------------------------------------------------------
//! @brief Windows のプロセスを作成します。
//!
//! @param[out] pi プロセスの情報を格納します。
//! @param[in] cmd コマンド文字列です。
//! @param[in] showWindow ウィンドウの表示指定です。
//!                       SW_SHOW, SW_HIDE, SW_SHOWNORMAL, SW_MAXIMIZE, SW_MINIMIZE
//!
//! @return 成功なら 0 以外を返します。
//-----------------------------------------------------------------------------
#ifdef __PIWin__
static int CreateProcessExec(
    PROCESS_INFORMATION& pi,
    const char* cmd,
    int showWindow = SW_HIDE)
{
    //-----------------------------------------------------------------------------
    // init process information struct
    ZeroMemory(&pi, sizeof(PROCESS_INFORMATION));

    //-----------------------------------------------------------------------------
    // init startup info struct
    STARTUPINFO si;
    ZeroMemory(&si, sizeof(STARTUPINFO));
    si.cb          = sizeof(STARTUPINFO);
    si.dwFlags     = STARTF_USESHOWWINDOW;
    si.wShowWindow = (WORD)showWindow;

    //-----------------------------------------------------------------------------
    // create process
    int retCode = CreateProcess(
        NULL,   // lpApplicationName
        const_cast<char*>(cmd), // lpCommandLine
        NULL,   // lpProcessAttributes
        NULL,   // lpThreadAttributes
        FALSE,  // bInheritHandles
        0,      // dwCreationFlags
        NULL,   // lpEnvironment
        NULL,   // lpCurrentDirectory
        &si,    // lpStartupInfo
        &pi);   // lpProcessInformation

    return retCode;
}
#endif

//-----------------------------------------------------------------------------
//! @brief Windows のプロセスを実行します。
//-----------------------------------------------------------------------------
#ifdef __PIWin__
int RExecProcess(
    int* pStatus,
    const char* cmd,
    const bool waitsExit,
    int showWindow
)
{
    //-----------------------------------------------------------------------------
    // set status to success
    if (pStatus != NULL)
    {
        *pStatus = 1;
    }

    //-----------------------------------------------------------------------------
    // create process
    PROCESS_INFORMATION pi;
    if (CreateProcessExec(pi, cmd, showWindow) == 0)
    {
        LPVOID lpMsgBuf;
        FormatMessage(
            FORMAT_MESSAGE_ALLOCATE_BUFFER |
            FORMAT_MESSAGE_FROM_SYSTEM |
            FORMAT_MESSAGE_IGNORE_INSERTS,
            NULL,
            GetLastError(),
            MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // デフォルト言語
            (LPTSTR)&lpMsgBuf,
            0,
            NULL);
        RMsgBox("Cannot create process: %s", lpMsgBuf);
        LocalFree(lpMsgBuf);
        if (pStatus != NULL)
        {
            *pStatus = 0;
        }
        return 1;
    }

    //-----------------------------------------------------------------------------
    // 終了を待たない場合
    if (!waitsExit)
    {
        CloseHandle(pi.hThread);
        CloseHandle(pi.hProcess);
        return 0;
    }

    //-----------------------------------------------------------------------------
    // wait until boot finish
    WaitForInputIdle(pi.hProcess, INFINITE);

    //-----------------------------------------------------------------------------
    // wait until command finish
    WaitForSingleObject(pi.hProcess, INFINITE);

    //-----------------------------------------------------------------------------
    // get return code
    DWORD exitCode;
    if (GetExitCodeProcess(pi.hProcess, &exitCode) == 0)
    {
        RMsgBox("Cannot get exit code");
        if (pStatus != NULL)
        {
            *pStatus = 0;
        }
        return 1;
    }

    //-----------------------------------------------------------------------------
    // close process handle
    CloseHandle(pi.hThread);
    CloseHandle(pi.hProcess);

    return exitCode;
}
#endif

//-----------------------------------------------------------------------------
//! @brief ヘルプの html ファイル名を取得します。
//!
//! @param[in] oldFile 古いヘルプの html ファイル名です。
//!
//! @param[in] ヘルプの html ファイル名を返します。
//-----------------------------------------------------------------------------
static std::string GetHelpFileName(const std::string& oldFile)
{
    const char* const helpConversions[] =
    {
        "PhotoshopPlugin.html"          , "Page_83956012.html",
        "html/NW4F_DistanceField.html"  , "Page_83956014.html",
        "html/NW4F_ftx.html"            , "Page_83955693.html",
        "html/NW4F_NormalMapFilter.html", "Page_83956005.html",
        NULL, NULL
    };

    std::string file;
    for (int iConv = 0; ; iConv += 2)
    {
        const char* const old = helpConversions[iConv];
        if (old == NULL)
        {
            break;
        }
        else if (oldFile == old)
        {
            file = helpConversions[iConv + 1];
            break;
        }
    }
    return (!file.empty()) ? file : "Page_83956012.html";
}

//-----------------------------------------------------------------------------
//! @brief Nintendo Photoshop プラグインのヘルプを表示します。
//-----------------------------------------------------------------------------
#ifdef __PIWin__
void RShowPluginHelp(const char* url)
{
    //-----------------------------------------------------------------------------
    // ヘルプファイルのパスを取得します。
    std::string helpPath = RGetNintendoPhotoshopRootPath() +
        "..\\..\\..\\Documents\\Package\\contents\\Pages\\";
    if (RFolderExists(helpPath))
    {
        helpPath = RGetFullFilePath(helpPath, false) + GetHelpFileName(url);
    }
    else
    {
        helpPath = GetOldNintendoWareRootPath();
        if (helpPath.empty())
        {
            RMsgBox("Help file cannot be found");
            return;
        }
        helpPath = helpPath + "\\Document\\Graphics\\DccPlugin\\Photoshop\\" + url;
    }

    //-----------------------------------------------------------------------------
    // 既定のブラウザでヘルプを開きます。
    ShellExecute(NULL, "open", helpPath.c_str(), NULL, NULL, SW_SHOWNORMAL);
}
#endif

//-----------------------------------------------------------------------------
//! @brief Windows の RECT を VRect に変換して設定します。
//-----------------------------------------------------------------------------
#ifdef __PIWin__
void RSetPSRectFromWinRect(VRect& dst, const RECT& src)
{
    dst.left   = src.left;
    dst.top    = src.top;
    dst.right  = src.right;
    dst.bottom = src.bottom;
}
#endif

//-----------------------------------------------------------------------------
//! @brief VRect を Windows の RECT に変換して設定します。
//-----------------------------------------------------------------------------
#ifdef __PIWin__
void RSetWinRectFromPSRect(RECT& dst, const VRect& src)
{
    dst.left   = src.left;
    dst.top    = src.top;
    dst.right  = src.right;
    dst.bottom = src.bottom;
}
#endif

//-----------------------------------------------------------------------------
//! @brief アバウトダイアログのダイアログプロシージャです。
//-----------------------------------------------------------------------------
#ifdef __PIWin__
BOOL WINAPI RAboutDialogProc(
    HWND hDlg,
    UINT uMsg,
    WPARAM wParam, LPARAM lParam
)
{
    static RAboutParam* paramPtr = NULL;

    switch (uMsg)
    {
        case WM_INITDIALOG:
            paramPtr = (RAboutParam*)lParam;
            SetWindowText(hDlg, paramPtr->m_TitleText);
            SetWindowText(GetDlgItem(hDlg, paramPtr->m_AboutTextId),
                paramPtr->m_MessageText);
            CenterDialog(hDlg);
            break;
        case WM_COMMAND:
        {
            short id = LOWORD(wParam);
            switch (id)
            {
                case IDOK:
                case IDCANCEL:
                    EndDialog(hDlg, id);
                    break;
                default:
                    if (id == paramPtr->m_HelpIndexId)
                    {
                        RShowPluginHelp(paramPtr->m_HelpIndexUrl);
                    }
                    break;
            }
        }
        default:
            return FALSE;
    }
    return TRUE;
}
#endif

//-----------------------------------------------------------------------------
//! @brief 日本語 UI を表示するなら true を返します。
//-----------------------------------------------------------------------------
bool RIsJapaneseUI()
{
    return (GetUserDefaultLCID() == 0x411);
    //return false;
}

//-----------------------------------------------------------------------------
//! @brief ダイアログのコントロールにツールチップを追加します。
//-----------------------------------------------------------------------------
void RAddToolTip(HWND hDlg, const int id, const char* pText)
{
    HWND hToolTip = CreateWindowEx(
        0,  // 拡張ウィンドウスタイル
        TOOLTIPS_CLASS, // クラスネーム
        NULL, // ウィンドウネーム
        TTS_ALWAYSTIP, // ウィンドウスタイル
        CW_USEDEFAULT, // X 座標
        CW_USEDEFAULT, // Y 座標
        CW_USEDEFAULT, // 幅
        CW_USEDEFAULT, // 高さ
        hDlg, // 親ウィンドウのハンドル
        NULL, // メニューハンドル
        GetDLLInstance(), // インスタンスハンドル
        NULL); // WM_CREATE データ

    TOOLINFO ti;
    ZeroMemory(&ti, sizeof(ti));
    ti.cbSize = sizeof(TOOLINFO);
    ti.uFlags = TTF_SUBCLASS;
    ti.hinst = NULL; // NULL でなければ lpszText で文字列リソースの識別子を指定
    ti.uId = 1; // コントロール内の複数の矩形に ti を追加する場合の識別子。TTM_NEWTOOLRECT で使用
    ti.hwnd = GetDlgItem(hDlg, id);
    GetClientRect(ti.hwnd, &ti.rect);
    ti.lpszText = const_cast<char*>(pText);
    LRESULT result = SendMessage(hToolTip, TTM_ADDTOOL, 0, reinterpret_cast<LPARAM>(&ti));
    //RNoteTrace("add tool: %d: %d", id, result);
    //if (std::string(pText).find('\n') != std::string::npos)
    {
        SendMessage(hToolTip, TTM_SETMAXTIPWIDTH, 0, 500); // 改行するには最大幅の設定が必要
    }
    //ti.rect.right /= 2;
    //SendMessage(hToolTip, TTM_NEWTOOLRECT, 0, reinterpret_cast<LPARAM>(&ti));
    R_UNUSED_VARIABLE(result);
}

//=============================================================================
// nps ネームスペースを終了します。
//=============================================================================
} // namespace nps
} // namespace tool
} // namespace gfx
} // namespace nn

