﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// GetEnvObjFullAnimValue AnalyzeEnvObjFullAnim GetEnvObjKeyAnim
// GetCameraFullAnimValue AnalyzeCameraFullAnim GetCameraKeyAnim
// GetLightFullAnimValue AnalyzeLightFullAnim GetLightKeyAnim
// GetFogFullAnimValue AnalyzeFogFullAnim GetFogKeyAnim
// SetUniqueEnvObjNames SortEnvObjByName
// OutputFsnFile OutCameraAnim OutLightAnim OutFogAnim

// YCameraT YLightT YFogT

//=============================================================================
// include
//=============================================================================

//-----------------------------------------------------------------------------
// my header
#include "DccOutput.h"
#include "Animation.h"
#include "Scene.h"

//-----------------------------------------------------------------------------
// maya header
#include <maya/MFnSpotLight.h>
#include <maya/M3dView.h>

//-----------------------------------------------------------------------------
// using namespace
using namespace std;
using namespace nn::gfx::tool::dcc;

//=============================================================================
// 処理選択用マクロです。
//=============================================================================
#define ADJUST_CAM_ROT_ZXY_SW // カメラの回転の角度が連続的になるように調整するなら定義します。
#define OPTIMIZE_VECTOR_BY_TOL_R // 回転の許容誤差を使用してベクトルのアニメーションを最適化するなら定義します。

//-----------------------------------------------------------------------------
//! @brief カメラのシェイプノードの DAG パスを返します。
//-----------------------------------------------------------------------------
MDagPath GetCameraShapePath(const MDagPath& xformPath, MStatus* pStatus)
{
    if (pStatus != NULL)
    {
        *pStatus = MS::kFailure;
    }
    const int childCount = xformPath.childCount();
    for (int ichild = 0; ichild < childCount; ++ichild)
    {
        MObject childObj = xformPath.child(ichild);
        if (childObj.hasFn(MFn::kCamera))
        {
            if (pStatus != NULL)
            {
                *pStatus = MS::kSuccess;
            }
            // インスタンス化されている可能性を考慮して xformPath の直接の
            // 子となるカメラシェイプの DAG パスを取得して返します。
            return GetDirectChildDagPath(xformPath, childObj);
        }
    }
    return xformPath;
}

//-----------------------------------------------------------------------------
//! @brief 立体表示用のステレオカメラの子の transform ノードなら true を返します。
//-----------------------------------------------------------------------------
bool IsStereoCameraChild(const MDagPath& xformPath)
{
    MDagPath parentPath = xformPath;
    if (parentPath.pop())
    {
        //cerr << "iscc: " << parentPath.partialPathName() << ": " << MFnDagNode(parentPath).typeName() << R_ENDL;
        return (MFnDagNode(parentPath).typeName() == "stereoRigTransform");
    }
    return false;
}

//-----------------------------------------------------------------------------
//! @brief カメラの回転を取得します。
//!
//! @param[in] xformPath transform ノードの DAG パスです。
//! @param[in] snapToZero 0 に近い値を 0 にするなら true を指定します。
//!
//! @return カメラの回転を返します。
//-----------------------------------------------------------------------------
static RVec3 GetCameraRotate(const MDagPath& xformPath, const bool snapToZero)
{
    MFnTransform xformFn(xformPath);
    MEulerRotation eulerRotate;
    if (xformPath.length() >= 2)
    {
        // 親ノードあり
        MQuaternion quat;
        xformFn.getRotation(quat, MSpace::kWorld);
        eulerRotate = quat.asEulerRotation();
    }
    else
    {
        // 親ノードなし
        xformFn.getRotation(eulerRotate);
    }

    // オーダーが zxy でなければ変換
    if (eulerRotate.order != MEulerRotation::kZXY)
    {
        eulerRotate.reorderIt(MEulerRotation::kZXY);
    }

    RVec3 rotate(
        static_cast<float>(eulerRotate.x * R_M_RAD_TO_DEG),
        static_cast<float>(eulerRotate.y * R_M_RAD_TO_DEG),
        static_cast<float>(eulerRotate.z * R_M_RAD_TO_DEG));
    if (snapToZero)
    {
        rotate.SnapToZero();
    }

    return rotate;
}

//-----------------------------------------------------------------------------
//! @brief カメラの注視点の位置を取得します。
//!        単位が cm でない場合の MFnCamera::centerOfInterestPoint の不具合を回避するため、
//!        注視点の transform ノードが存在すれば、その位置を取得します。
//!
//! @param[in] xformPath transform ノードの DAG パスです。
//! @param[in] snapToZero 0 に近い値を 0 にするなら true を指定します。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return カメラの注視点の位置を返します。
//-----------------------------------------------------------------------------
//static RVec3 GetCameraAim(
//  const MDagPath& xformPath,
//  const bool snapToZero,
//  const YExpOpt& yopt
//)
//{
//  //-----------------------------------------------------------------------------
//  // get from aim xform
//  MFnTransform xformFn(xformPath);
//  MPlugArray plugArray;
//  xformFn.findPlug("rx").connectedTo(plugArray, true, false);
//  if (plugArray.length() > 0 &&
//      plugArray[0].node().apiType() == MFn::kLookAt)
//  {
//      MDagPath lookAtPath;
//      MDagPath::getAPathTo(plugArray[0].node(), lookAtPath);
//      MFnDagNode lookAtFn(lookAtPath);
//      MPlug targetPlug = lookAtFn.findPlug("target");
//      if (targetPlug.numElements() == 1)
//      {
//          MPlug ttxPlug = targetPlug.elementByLogicalIndex(0).child(0).child(0);
//          ttxPlug.connectedTo(plugArray, true, false);
//          for (int iSrc = 0; iSrc < static_cast<int>(plugArray.length()); ++iSrc)
//          {
//              MObject obj = plugArray[iSrc].node();
//              if (obj.apiType() == MFn::kTransform)
//              {
//                  MDagPath aimPath;
//                  MDagPath::getAPathTo(obj, aimPath);
//                  //cerr << "get aim: " << aimPath.partialPathName() << R_ENDL;
//                  return GetRVec3(
//                      MFnTransform(aimPath).getTranslation(MSpace::kWorld) * yopt.m_InternalMagnify, snapToZero);
//              }
//          }
//      }
//  }
//
//  //-----------------------------------------------------------------------------
//  // get from attr
//  MStatus status;
//  MDagPath shapePath = GetCameraShapePath(xformPath, &status);
//  if (status)
//  {
//      //cerr << "get aim: " << shapePath.partialPathName() << R_ENDL;
//      MFnCamera camFn(shapePath);
//      return GetRVec3(
//          camFn.centerOfInterestPoint(MSpace::kWorld) * yopt.m_InternalMagnify, snapToZero);
//  }
//  else
//  {
//      return RVec3::kZero;
//  }
//}

//-----------------------------------------------------------------------------
//! @brief カメラの捻りを取得します。
//!
//! @param[in] xformPath transform ノードの DAG パスです。
//! @param[in] snapToZero 0 に近い値を 0 にするなら true を指定します。
//!
//! @return カメラの捻りを返します。
//-----------------------------------------------------------------------------
static float GetCameraTwist(const MDagPath& xformPath, const bool snapToZero)
{
    MFnTransform xformFn(xformPath);
    MEulerRotation eulerRotate;

    if (xformPath.length() >= 2)
    {
        // 親ノードあり
        MQuaternion quat;
        xformFn.getRotation(quat, MSpace::kWorld);
        eulerRotate = quat.asEulerRotation();
    }
    else
    {
        // 親ノードなし
        xformFn.getRotation(eulerRotate);
    }

    if (eulerRotate.order != MEulerRotation::kZXY)
    {
        // オーダーが zxy でなければ変換
        eulerRotate.reorderIt(MEulerRotation::kZXY);
    }

    const float twist = static_cast<float>(eulerRotate.z * R_M_RAD_TO_DEG);
    return (snapToZero) ? RSnapToZero(twist) : twist;
}

//-----------------------------------------------------------------------------
//! @brief カメラの正射影の高さを取得します。
//!
//! @param[in] orthoW 正射影の幅です。
//! @param[in] aspect アスペクト比（幅 / 高さ）です。
//! @param[in] filmFit レゾリューションゲート適合です。
//! @param[in] snapToZero 0 に近い値を 0 にするなら true を指定します。
//!
//! @return カメラの正射影の高さを返します。
//-----------------------------------------------------------------------------
static float GetCameraOrthoHeight(
    const float orthoW,
    const float aspect,
    const MFnCamera::FilmFit filmFit,
    const bool snapToZero
)
{
    float orthoH = orthoW;
    if (filmFit != MFnCamera::kVerticalFilmFit && aspect != 0.0f)
    {
        orthoH /= aspect;
    }
    return (snapToZero) ? RSnapToZero(orthoH) : orthoH;
}

//-----------------------------------------------------------------------------
//! @brief Maya 用のカメラのパラメータに対応する Maya のアトリビュート名を返します。
//-----------------------------------------------------------------------------
const char* YCamera::GetAttrName(const int paramIdx)
{
    static const char* const attrStrs[PARAM_COUNT] =
    {
        "tx", "ty", "tz",   // camera transform
        "tx", "ty", "tz",   // aim transform
        "twist",            // lookAt (camera group)
        "rx", "ry", "rz",   // camera transform
        "",                 // aspect（camera shape の他のアトリビュートから計算）
        "nearClipPlane",    // camera shape
        "farClipPlane",     // camera shape
        "",                 // ortho_height（camera shape の他のアトリビュートから計算）
        "",                 // persp_fovy（camera shape の他のアトリビュートから計算）
    };

    return attrStrs[paramIdx];
}

//-----------------------------------------------------------------------------
//! @brief Maya 用のカメラのコンストラクタです。
//-----------------------------------------------------------------------------
YCamera::YCamera( // YCameraT
    RScene& rscene,
    const MDagPath& xformPath,
    const YExpOpt& yopt,
    MStatus* pStatus
)
: m_XformPath(xformPath)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // 処理結果を失敗で初期化します。
    if (pStatus != NULL)
    {
        *pStatus = MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // エクスポートオプションからパラメータをコピーします。
    m_FrameCount = yopt.m_OutFrameCount;
    m_LoopAnim   = yopt.m_LoopAnim;

    //-----------------------------------------------------------------------------
    // 名前を取得します。
    MFnTransform xformFn(m_XformPath);
    m_OrgName = xformFn.name().asChar();
    m_Name = GetOutElementName(m_OrgName, yopt.m_RemoveNamespace);

    //-----------------------------------------------------------------------------
    // シェイプノードを取得します。
    m_ShapePath = GetCameraShapePath(m_XformPath, &status);
    if (!status)
    {
        YShowWarning(&rscene, // No camera shape: %s // 通常は発生しない
            "カメラのシェイプノードが存在しません: {0}",
            "A shape node of the camera cannot be found: {0}",
            m_OrgName);
        return;
    }

    //-----------------------------------------------------------------------------
    // Maya 上のカメラの種類、lookAt ノード、エイムロケータ、アップロケータを取得します。
    m_Kind = KIND_BASIC;
    MPlugArray plugArray;
    xformFn.findPlug("rx").connectedTo(plugArray, true, false);
    if (plugArray.length() > 0 &&
        plugArray[0].node().apiType() == MFn::kLookAt)
    {
        MDagPath::getAPathTo(plugArray[0].node(), m_LookAtPath);
        MFnDagNode lookAtFn(m_LookAtPath);
        MPlug targetPlug = lookAtFn.findPlug("target");
        if (targetPlug.numElements() == 1)
        {
            MPlug ttxPlug = targetPlug.elementByLogicalIndex(0).child(0).child(0);
            ttxPlug.connectedTo(plugArray, true, false);
            for (int iSrc = 0; iSrc < static_cast<int>(plugArray.length()); ++iSrc)
            {
                MObject obj = plugArray[iSrc].node();
                if (obj.apiType() == MFn::kTransform)
                {
                    MDagPath::getAPathTo(obj, m_AimPath);
                    m_Kind = KIND_AIM;
                    break;
                }
            }
        }
        if (m_Kind == KIND_AIM)
        {
            MPlug wumPlug = lookAtFn.findPlug("worldUpMatrix");
            wumPlug.connectedTo(plugArray, true, false);
            if (plugArray.length() > 0 &&
                plugArray[0].node().apiType() == MFn::kTransform)
            {
                MDagPath::getAPathTo(plugArray[0].node(), m_UpPath);
                m_Kind = KIND_AIM_UP;
            }
        }
    }

    //-----------------------------------------------------------------------------
    // アニメーションのパラメータに対応する Maya のプラグを取得します。
    MFnCamera camFn(m_ShapePath);
    for (int iParam = 0; iParam < PARAM_COUNT; ++iParam)
    {
        const char* attrName = GetAttrName(iParam);
        if (strlen(attrName) == 0)
        {
            continue;
        }
        else if (
            (POSITION_X <= iParam && iParam <= POSITION_Z) ||
            (ROTATE_X   <= iParam && iParam <= ROTATE_Z  ))
        {
            m_AnimPlugs[iParam] = xformFn.findPlug(attrName);
        }
        else if (AIM_X <= iParam && iParam <= AIM_Z)
        {
            if (m_Kind != KIND_BASIC)
            {
                m_AnimPlugs[iParam] = MFnDagNode(m_AimPath).findPlug(attrName);
            }
        }
        else if (iParam == TWIST)
        {
            if (m_Kind == KIND_AIM)
            {
                m_AnimPlugs[iParam] = MFnDagNode(m_LookAtPath).findPlug(attrName);
            }
        }
        else if (iParam == NEAR_CLIP || iParam == FAR_CLIP)
        {
            m_AnimPlugs[iParam] = camFn.findPlug(attrName);
        }
        //cerr << "cam plug: " << GetParamName(iParam) << ": " << m_AnimPlugs[iParam].info() << " " << m_AnimPlugs[iParam].isNull() << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // 属性の値を取得します。
    m_RotateMode = (m_Kind == KIND_BASIC) ? ROTATE_EULER_ZXY : ROTATE_AIM;
    m_ProjectionMode = (camFn.isOrtho()) ? ORTHO : PERSP;

    m_RotateOrder = xformFn.rotationOrder();
    m_Translate = GetRVec3(camFn.eyePoint(MSpace::kWorld) * yopt.m_InternalMagnify, true);
    m_Aim = RVec3::kZero;
    m_Twist = 0.0f;
    if (m_Kind == KIND_BASIC)
    {
        m_Rotate = GetCameraRotate(m_XformPath, true);
    }
    else
    {
        // MFnCamera::centerOfInterestPoint は不具合があり、単位が cm でない場合に正しい値を返さないので、
        // 注視点の transform ノードから位置を取得します。
        m_Aim = GetRVec3(
            MFnTransform(m_AimPath).getTranslation(MSpace::kWorld) * yopt.m_InternalMagnify, true);
        if (m_Kind == KIND_AIM)
        {
            m_AnimPlugs[TWIST].getValue(m_Twist);
            m_Twist = RSnapToZero(m_Twist * static_cast<float>(R_M_RAD_TO_DEG));
        }
        else
        {
            m_Twist = GetCameraTwist(m_XformPath, true);
        }
    }

    m_Aspect      = RSnapToZero(static_cast<float>(camFn.aspectRatio()));
    m_NearClip    = RSnapToZero(static_cast<float>(camFn.nearClippingPlane() * yopt.m_InternalMagnify));
    m_FarClip     = RSnapToZero(static_cast<float>(camFn.farClippingPlane()  * yopt.m_InternalMagnify));
    m_FilmFit     = camFn.filmFit();
    const float orthoW = static_cast<float>(camFn.orthoWidth() * yopt.m_InternalMagnify);
    m_OrthoHeight = GetCameraOrthoHeight(orthoW, m_Aspect, m_FilmFit, true);
    m_PerspFovy   = RSnapToZero(static_cast<float>(camFn.verticalFieldOfView() * R_M_RAD_TO_DEG));

    //-----------------------------------------------------------------------------
    // 処理結果を成功にして終了します。
    if (pStatus != NULL)
    {
        *pStatus = MS::kSuccess;
    }

    //-----------------------------------------------------------------------------
    // debug
    #ifdef DEBUG_PRINT_SW
    //cerr << "cam: " << m_OrgName << ": " << m_ShapePath.partialPathName() << ": " << m_PerspFovy << R_ENDL;
    #endif
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief 全サブフレームにおけるカメラのアニメーション値を取得します。
//!
//! @param[in,out] ycameras カメラ配列です。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetCameraFullAnimValue(YCameraArray& ycameras, const YExpOpt& yopt)
{
    const int camCount = static_cast<int>(ycameras.size());
    for (int iCam = 0; iCam < camCount; ++iCam)
    {
        YCamera& cam = ycameras[iCam];
        MFnTransform xformFn(cam.m_XformPath);
        MFnCamera camFn(cam.m_ShapePath);

        RVec3 position = GetRVec3(camFn.eyePoint(MSpace::kWorld) * yopt.m_InternalMagnify, true);
        RVec3 aim = RVec3::kZero;
        RVec3 rotate = RVec3::kZero;
        float twist = 0.0f;
        if (cam.m_Kind == YCamera::KIND_BASIC)
        {
            rotate = GetCameraRotate(cam.m_XformPath, true);
        }
        else
        {
            aim = GetRVec3(
                MFnTransform(cam.m_AimPath).getTranslation(MSpace::kWorld) * yopt.m_InternalMagnify, true);
            if (cam.m_Kind == YCamera::KIND_AIM)
            {
                cam.m_AnimPlugs[YCamera::TWIST].getValue(twist);
                twist = RSnapToZero(twist * static_cast<float>(R_M_RAD_TO_DEG));
            }
            else
            {
                twist = GetCameraTwist(cam.m_XformPath, true);
            }
        }
        const float nearClip = RSnapToZero(static_cast<float>(camFn.nearClippingPlane() * yopt.m_InternalMagnify));
        const float farClip  = RSnapToZero(static_cast<float>(camFn.farClippingPlane()  * yopt.m_InternalMagnify));
        const float aspect   = RSnapToZero(static_cast<float>(camFn.aspectRatio()));
        const float fovy     = RSnapToZero(static_cast<float>(camFn.verticalFieldOfView() * R_M_RAD_TO_DEG));
        const float orthoW = static_cast<float>(camFn.orthoWidth() * yopt.m_InternalMagnify);
        const float orthoH = GetCameraOrthoHeight(orthoW, aspect, cam.m_FilmFit, true);

        cam.m_Anims[YCamera::POSITION_X  ].m_FullValues.push_back(position.x);
        cam.m_Anims[YCamera::POSITION_Y  ].m_FullValues.push_back(position.y);
        cam.m_Anims[YCamera::POSITION_Z  ].m_FullValues.push_back(position.z);
        cam.m_Anims[YCamera::AIM_X       ].m_FullValues.push_back(aim.x);
        cam.m_Anims[YCamera::AIM_Y       ].m_FullValues.push_back(aim.y);
        cam.m_Anims[YCamera::AIM_Z       ].m_FullValues.push_back(aim.z);
        cam.m_Anims[YCamera::TWIST       ].m_FullValues.push_back(twist);
        cam.m_Anims[YCamera::ROTATE_X    ].m_FullValues.push_back(rotate.x);
        cam.m_Anims[YCamera::ROTATE_Y    ].m_FullValues.push_back(rotate.y);
        cam.m_Anims[YCamera::ROTATE_Z    ].m_FullValues.push_back(rotate.z);
        cam.m_Anims[YCamera::ASPECT      ].m_FullValues.push_back(aspect);
        cam.m_Anims[YCamera::NEAR_CLIP   ].m_FullValues.push_back(nearClip);
        cam.m_Anims[YCamera::FAR_CLIP    ].m_FullValues.push_back(farClip);
        cam.m_Anims[YCamera::PERSP_FOVY  ].m_FullValues.push_back(fovy);
        cam.m_Anims[YCamera::ORTHO_HEIGHT].m_FullValues.push_back(orthoH);
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief カメラのキーアニメーション値に適用するスケールを取得します。
//!
//! @param[in] paramIdx パラメーターインデックスです。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return カメラのキーアニメーション値に適用するスケールを返します。
//-----------------------------------------------------------------------------
static float GetCameraKeyScale(const int paramIdx, const YExpOpt& yopt)
{
    return (YCamera::IsRotate(paramIdx)   ) ? static_cast<float>(R_M_RAD_TO_DEG)         :
           (YCamera::IsTranslate(paramIdx)) ? static_cast<float>(yopt.m_InternalMagnify) :
           1.0f;
}

//-----------------------------------------------------------------------------
//! @brief カメラについて Maya 上のキーを取得可能か判定します。
//!
//! @param[in] cam カメラです。
//! @param[in] paramIdx パラメーターインデックスです。
//! @param[in] chan チャンネル入力情報です。
//! @param[in] fullValues ベイクしたデータです。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return Maya 上のキーを取得可能なら true を返します。
//-----------------------------------------------------------------------------
static bool GetsCameraMayaKey(
    const YCamera& cam,
    const int paramIdx,
    const ChannelInput& chan,
    const RFloatArray& fullValues,
    const YExpOpt& yopt
)
{
    if (yopt.m_BakeAllAnim   ||
        !chan.m_CurveFlag    ||
        chan.m_CurveWeighted)
    {
        return false;
    }

    if (YCamera::ROTATE_X <= paramIdx && paramIdx <= YCamera::ROTATE_Z)
    {
        if (cam.m_RotateOrder != MTransformationMatrix::kZXY ||
            IsQuaternionAnimCurve(chan.m_CurveObj))
        {
            return false;
        }
    }

    const float valueScale = GetCameraKeyScale(paramIdx, yopt);
    const float valueOfs = 0.0f;
    return IsValidAnimCurveValue(chan.m_CurveObj, fullValues, valueScale, valueOfs, yopt);
}

//-----------------------------------------------------------------------------
//! @brief 全サブフレームにおけるカメラのアニメーション値を分析します。
//!        回転の角度が連続的になるように調整します。
//!        アニメーション値が一定か判定します。
//!
//! @param[in,out] ycameras カメラ配列です。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static void AnalyzeCameraFullAnim(YCameraArray& ycameras, const YExpOpt& yopt)
{
    const int camCount = static_cast<int>(ycameras.size());
    for (int camIdx = 0; camIdx < camCount; ++camIdx)
    {
        //-----------------------------------------------------------------------------
        // カメラの各パラメーターについてアニメーション値が一定か判定し、
        // Maya 上のキーを取得可能か判定します。
        YCamera& cam = ycameras[camIdx];
        YAnimCurve* curves = cam.m_Anims;
        for (int paramIdx = 0; paramIdx < YCamera::PARAM_COUNT; ++paramIdx)
        {
            YAnimCurve& curve = curves[paramIdx];
            const bool isRotate = YCamera::IsRotate(paramIdx);
            curve.m_Name = cam.m_Name + "." + YCamera::GetParamName(paramIdx);
            curve.m_LoopFlag = yopt.m_LoopAnim;
            curve.m_AngleFlag = isRotate;
            curve.m_Tolerance =
                (isRotate                      ) ? yopt.m_TolR                                      :
                (YCamera::IsTranslate(paramIdx)) ? static_cast<float>(yopt.m_TolT * yopt.m_Magnify) :
                R_MAKE_KEY_TOL_CAMERA_OTHER; // ASPECT
                // ↑ m_TolT は Magnify を掛ける前の値に対する許容値なので Magnify を掛けます。
            curve.UpdateConstantFlag();

            const MPlug& plug = cam.m_AnimPlugs[paramIdx];
            ChannelInput& chan = cam.m_AnimChans[paramIdx];
            if (!plug.isNull())
            {
                chan.GetInput(plug);
            }
            curve.m_Baked = (curve.m_ConstantFlag ||
                !GetsCameraMayaKey(cam, paramIdx, chan, curve.m_FullValues, yopt));
        }

        //-----------------------------------------------------------------------------
        // 回転 XYZ のいずれも Maya 上のキーを取得しない場合、
        // ベイクしたデータの回転の角度が連続的になるように調整します。
        #ifdef ADJUST_CAM_ROT_ZXY_SW
        const bool hasRotateAnim =
            !curves[YCamera::ROTATE_X].m_ConstantFlag ||
            !curves[YCamera::ROTATE_Y].m_ConstantFlag ||
            !curves[YCamera::ROTATE_Z].m_ConstantFlag;
        if (hasRotateAnim &&
            curves[YCamera::ROTATE_X].m_Baked &&
            curves[YCamera::ROTATE_Y].m_Baked &&
            curves[YCamera::ROTATE_Z].m_Baked)
        {
            if (RMakeContinuousZxyAngleArray( // ZXY であることに注意！
                curves[YCamera::ROTATE_X].m_FullValues,
                curves[YCamera::ROTATE_Y].m_FullValues,
                curves[YCamera::ROTATE_Z].m_FullValues))
            {
                curves[YCamera::ROTATE_X].UpdateConstantFlag();
                curves[YCamera::ROTATE_Y].UpdateConstantFlag();
                curves[YCamera::ROTATE_Z].UpdateConstantFlag();
                //cerr << "rotate anim adjusted: " << cam.m_OrgName << R_ENDL;
            }
        }
        #endif
    }
}

//-----------------------------------------------------------------------------
//! @brief カメラのキーアニメーションを取得します。
//!
//! @param[in,out] ycameras カメラ配列です。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetCameraKeyAnim(YCameraArray& ycameras, const YExpOpt& yopt)
{
    const int camCount = static_cast<int>(ycameras.size());
    for (int camIdx = 0; camIdx < camCount; ++camIdx)
    {
        YCamera& cam = ycameras[camIdx];
        for (int paramIdx = 0; paramIdx < YCamera::PARAM_COUNT; ++paramIdx)
        {
            //-----------------------------------------------------------------------------
            // アニメーションカーブデータを取得または作成します。
            YAnimCurve& curve = cam.m_Anims[paramIdx];
            if (!curve.m_ConstantFlag)
            {
                if (!curve.m_Baked)
                {
                    const ChannelInput& chan = cam.m_AnimChans[paramIdx];
                    const float valueScale = GetCameraKeyScale(paramIdx, yopt);
                    const float valueOfs = 0.0f;
                    curve.GetKeys(chan.m_CurveObj, valueScale, valueOfs, yopt);
                }
                else
                {
                    #ifdef ADJUST_CAM_ROT_ZXY_SW
                    // ROTATE_X/Y/Z は 3 軸同時に調整しているので単一軸の調整は不要
                    const bool continuousAngleFlag = (
                        paramIdx == YCamera::TWIST      ||
                        paramIdx == YCamera::PERSP_FOVY);
                    #else
                    const bool continuousAngleFlag = true;
                    #endif
                    curve.MakeKeys(GetFloatFrameFromSubFrame, &yopt, continuousAngleFlag);
                }
            }
        }
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief <camera_anim> 要素を出力します。
//-----------------------------------------------------------------------------
void YCamera::OutAnim( // OutCameraAnim
    std::ostream& os,
    RDataStreamArray& dataStreams,
    const int tabCount,
    const int index
) const
{
    static const char* const rotateModeStrs[] = { "aim", "euler_zxy" };
    static const char* const projectionModeStrs[] = { "ortho", "persp" };

    const int& tc = tabCount;

    //-----------------------------------------------------------------------------
    // begin
    os << RTab(tc) << "<camera_anim"
       << " index=\"" << index
       << "\" camera_name=\"" << RGetUtf8FromShiftJis(m_Name) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "frame_count=\"" << m_FrameCount
       << "\" loop=\"" << RBoolStr(m_LoopAnim) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "rotate_mode=\"" << rotateModeStrs[m_RotateMode] << "\"" << R_ENDL;
    os << RTab(tc + 1) << "projection_mode=\"" << projectionModeStrs[m_ProjectionMode] << "\"" << R_ENDL;
    os << RTab(tc) << ">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // camera anim targets
    for (int iParam = 0; iParam < PARAM_COUNT; ++iParam)
    {
        if (m_RotateMode == ROTATE_AIM)
        {
            if (ROTATE_X <= iParam && iParam <= ROTATE_Z) continue;
        }
        else // ROTATE_EULER_ZXY
        {
            if (AIM_X <= iParam && iParam <= AIM_Z) continue;
            if (iParam == TWIST) continue;
        }

        if (m_ProjectionMode == ORTHO)
        {
            if (iParam == PERSP_FOVY) continue;
        }
        else // PERSP
        {
            if (iParam == ORTHO_HEIGHT) continue;
        }

        const YAnimCurve& curve = m_Anims[iParam];
        curve.Out(os, dataStreams, tc + 1, "camera_anim_target",
            GetParamName(iParam), false);
    }

    //-----------------------------------------------------------------------------
    // カメラのユーザーデータ配列を出力します。
    ROutArrayElement(os, tc + 1, m_UserDatas, "user_data_array");

    //-----------------------------------------------------------------------------
    // end
    os << RTab(tc) << "</camera_anim>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief ライトの方向を返します。
//!
//! @param[in] shapePath シェイプノードの DAG パスです。
//! @param[in] snapToZero 0 に近い値を 0 にするなら true を指定します。
//!
//! @return ライトの方向を返します。
//-----------------------------------------------------------------------------
static RVec3 GetLightDirection(const MDagPath& shapePath, const bool snapToZero)
{
    const MMatrix wm = shapePath.inclusiveMatrix();
    MVector dir(-wm[2][0], -wm[2][1], -wm[2][2]); // -Z
    dir.normalize();
    return GetRVec3(dir, snapToZero);
}

//-----------------------------------------------------------------------------
//! @brief Maya 用のライトの方向を返します。
//!        ただし、ディレクショナルライト以外は -Z 方向を返します。
//-----------------------------------------------------------------------------
RVec3 YLight::GetDirection(const bool snapToZero) const
{
    return (m_Type == DIRECTIONAL) ?
        GetLightDirection(m_ShapePath, snapToZero) : RVec3::kZNegAxis;
}

//-----------------------------------------------------------------------------
//! @brief ライトの目標の位置を返します。
//!
//! @param[in] shapePath シェイプノードの DAG パスです。
//! @param[in] position ライトの位置です。
//! @param[in] magnify 移動値に掛ける倍率です。
//! @param[in] snapToZero 0 に近い値を 0 にするなら true を指定します。
//!
//! @return ライトの目標の位置を返します。
//-----------------------------------------------------------------------------
static RVec3 GetLightAim(
    const MDagPath& shapePath,
    const RVec3& position,
    const double magnify,
    const bool snapToZero
)
{
    const double distance = MFnLight(shapePath).centerOfIllumination() * magnify;
    const RVec3 direction = GetLightDirection(shapePath, false);
    RVec3 aim = position + static_cast<float>(distance) * direction;
    if (snapToZero)
    {
        aim.SnapToZero();
    }
    return aim;
}

//-----------------------------------------------------------------------------
//! @brief Maya 用のライトの目標の位置を返します。
//!        ただし、スポットライト以外は原点を返します。
//-----------------------------------------------------------------------------
RVec3 YLight::GetAim(
    const RVec3& position,
    const double magnify,
    const bool snapToZero
) const
{
    return (m_Type == SPOT) ?
        GetLightAim(m_ShapePath, position, magnify, snapToZero) : RVec3::kZero;
}

//-----------------------------------------------------------------------------
//! @brief Maya 用のライトの距離減衰を取得します。
//-----------------------------------------------------------------------------
void YLight::GetDistAttn(
    float& startDist,
    float& endDist,
    const double magnify,
    const bool snapToZero
) const
{
    startDist =  0.0f;
    endDist   = 10.0f;
    if (!m_AnimPlugs[DIST_ATTN_START].isNull())
    {
        m_AnimPlugs[DIST_ATTN_START].getValue(startDist);
    }
    if (!m_AnimPlugs[DIST_ATTN_END].isNull())
    {
        m_AnimPlugs[DIST_ATTN_END].getValue(endDist);
    }

    const float magnifyF = static_cast<float>(magnify);
    startDist *= magnifyF;
    endDist   *= magnifyF;
    if (snapToZero)
    {
        startDist = RSnapToZero(startDist);
        endDist   = RSnapToZero(endDist  );
    }
}

//-----------------------------------------------------------------------------
//! @brief Maya 用のライトの角度減衰を取得します。
//!        ただし、スポットライト以外は (0, 0) を返します。
//-----------------------------------------------------------------------------
void YLight::GetAngleAttn(float& startAngle, float& endAngle, const bool snapToZero) const
{
    if (m_Type == SPOT)
    {
        MFnSpotLight spotFn(m_ShapePath);
        const float coneAngle     = static_cast<float>(spotFn.coneAngle()     * R_M_RAD_TO_DEG);
        const float penumbraAngle = static_cast<float>(spotFn.penumbraAngle() * R_M_RAD_TO_DEG);
        if (penumbraAngle >= 0.0f)
        {
            startAngle = coneAngle * 0.5f;
            endAngle   = startAngle + penumbraAngle;
        }
        else
        {
            endAngle   = coneAngle * 0.5f;
            startAngle = endAngle + penumbraAngle;
        }
        // Maya 上で coneAngle * 0.5 は 90 度までですが、
        // coneAngle * 0.5 + penumbraAngle は 90 度以上可能なので、
        // [0, 180] でクランプします。
        startAngle = RClampValue(0.0f, 180.0f, startAngle);
        endAngle   = RClampValue(0.0f, 180.0f, endAngle  );
        if (snapToZero)
        {
            startAngle = RSnapToZero(startAngle);
            endAngle   = RSnapToZero(endAngle);
        }
    }
    else
    {
        startAngle = endAngle = 0.0f;
    }
}

//-----------------------------------------------------------------------------
//! @brief Maya 用のライトのパラメータに対応する Maya のアトリビュート名を返します。
//-----------------------------------------------------------------------------
const char* YLight::GetAttrName(const int iParam)
{
    static const char* const attrStrs[PARAM_COUNT] =
    {
        "visibility",           // light transform
        "tx", "ty", "tz",       // light transform
        "", "", "",             // direction（transform の他のアトリビュートから計算）
        "", "", "",             // aim（transform と shape の他のアトリビュートから計算）
        "NW4F_AttnStartDist", "NW4F_AttnEndDist",   // light shape（カスタムアトリビュート）
        "", "",                 // angle_attn_start, angle_attn_end（shape の他のアトリビュートから計算）
        "cr", "cg", "cb",       // light shape
        "", "", "",             // color1（0, 0, 0 で固定）
    };

    return attrStrs[iParam];
}

//-----------------------------------------------------------------------------
//! @brief Maya 用のライトのコンストラクタです。
//-----------------------------------------------------------------------------
YLight::YLight( // YLightT
    RScene& rscene,
    const MDagPath& xformPath,
    const YExpOpt& yopt,
    MStatus* pStatus
)
: m_XformPath(xformPath)
{
    MStatus status;

    //-----------------------------------------------------------------------------
    // 処理結果を失敗で初期化します。
    if (pStatus != NULL)
    {
        *pStatus = MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // エクスポートオプションからパラメータをコピーします。
    m_FrameCount = yopt.m_OutFrameCount;
    m_LoopAnim   = yopt.m_LoopAnim;

    //-----------------------------------------------------------------------------
    // 名前を取得します。
    MFnTransform xformFn(m_XformPath);
    m_OrgName = xformFn.name().asChar();
    m_Name = GetOutElementName(m_OrgName, yopt.m_RemoveNamespace);

    //-----------------------------------------------------------------------------
    // シェイプノードを取得します。
    m_ShapePath = m_XformPath;
    if (!m_ShapePath.extendToShape())
    {
        YShowWarning(&rscene, // No light shape: %s // 通常は発生しない
            "ライトのシェイプノードが存在しません: {0}",
            "A shape node of the light cannot be found: {0}",
            m_OrgName);
        return;
    }

    //-----------------------------------------------------------------------------
    // タイプを取得します。
    MFnLight litFn(m_ShapePath, &status);
    const MFn::Type apiType = m_ShapePath.node().apiType();
    switch (apiType)
    {
    case MFn::kAmbientLight:
        m_Type = AMBIENT;
        break;
    case MFn::kDirectionalLight:
        m_Type = DIRECTIONAL;
        break;
    case MFn::kPointLight:
        m_Type = POINT;
        break;
    case MFn::kSpotLight:
        m_Type = SPOT;
        break;
    default:
        m_Type = POINT; // 通常はここには来ません。
        break;
    }

    //-----------------------------------------------------------------------------
    // アニメーションのパラメータに対応する Maya のプラグを取得します。
    for (int iParam = 0; iParam < PARAM_COUNT; ++iParam)
    {
        const char* attrName = GetAttrName(iParam);
        if (strlen(attrName) == 0)
        {
            continue;
        }
        else if (iParam == DIST_ATTN_START ||
                 iParam == DIST_ATTN_END)
        {
            if (m_Type == POINT || m_Type == SPOT)
            {
                m_AnimPlugs[iParam] = FindPlugQuiet(litFn, attrName, &status);
            }
        }
        else if (iParam == ENABLE || IsTranslate(iParam))
        {
            m_AnimPlugs[iParam] = xformFn.findPlug(attrName);
        }
        else if (IsColor(iParam))
        {
            m_AnimPlugs[iParam] = litFn.findPlug(attrName);
        }
        //cerr << "lgt: " << GetParamName(iParam) << ": " << m_AnimPlugs[iParam].info() << " " << m_AnimPlugs[iParam].isNull() << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // 属性の値を取得します。
    m_Enable = IsVisibleNodeIncludeParent(m_ShapePath);
    m_Translate = GetRVec3(xformFn.getTranslation(MSpace::kWorld) * yopt.m_InternalMagnify, true);
    m_Direction = GetDirection(true);
    m_Aim = GetAim(m_Translate, yopt.m_InternalMagnify, true);

    const int decayRate = (m_Type == POINT || m_Type == SPOT) ?
        MFnNonAmbientLight(m_ShapePath).decayRate() : 0; // 0=No Decay
    m_UsesDistAttn = decayRate != 0 &&
        !m_AnimPlugs[DIST_ATTN_START].isNull() &&
        !m_AnimPlugs[DIST_ATTN_END  ].isNull();
    GetDistAttn(m_DistAttnStart, m_DistAttnEnd, yopt.m_InternalMagnify, true);

    GetAngleAttn(m_AngleAttnStart, m_AngleAttnEnd, true);

    m_EmitsDiffuse  = litFn.lightDiffuse();
    m_EmitsSpecular = litFn.lightSpecular();

    //const float intensity = RSnapToZero(litFn.intensity());
    m_Color0 = GetRVec3Color(litFn.color(), true, Y_CLAMPS_COLOR);
    m_Color1 = RVec3::kZero;

    //-----------------------------------------------------------------------------
    // 処理結果を成功にして終了します。
    if (pStatus != NULL)
    {
        *pStatus = MS::kSuccess;
    }

    //-----------------------------------------------------------------------------
    // debug
    #ifdef DEBUG_PRINT_SW
    cerr << "lgt: " << m_OrgName << ": " << m_Type << ": " << m_ShapePath.partialPathName() << R_ENDL;
    #endif
}

//-----------------------------------------------------------------------------
//! @brief 全サブフレームにおけるライトのアニメーション値を取得します。
//!
//! @param[in,out] ylights ライト配列です。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetLightFullAnimValue(YLightArray& ylights, const YExpOpt& yopt)
{
    const int lgtCount = static_cast<int>(ylights.size());
    for (int iLgt = 0; iLgt < lgtCount; ++iLgt)
    {
        //-----------------------------------------------------------------------------
        // get
        YLight& lgt = ylights[iLgt];
        const MFnTransform xformFn(lgt.m_XformPath);
        const MFnLight litFn(lgt.m_ShapePath);

        const bool enable = IsVisibleNodeIncludeParent(lgt.m_ShapePath);
        const RVec3 position = GetRVec3(xformFn.getTranslation(MSpace::kWorld) * yopt.m_InternalMagnify, true);
        const RVec3 direction = lgt.GetDirection(true);
        const RVec3 aim = lgt.GetAim(position, yopt.m_InternalMagnify, true);
        float distAttnStart;
        float distAttnEnd;
        lgt.GetDistAttn(distAttnStart, distAttnEnd, yopt.m_InternalMagnify, true);
        float angleAttnStart;
        float angleAttnEnd;
        lgt.GetAngleAttn(angleAttnStart, angleAttnEnd, true);

        const RVec3 color0 = GetRVec3Color(litFn.color(), true, Y_CLAMPS_COLOR);
        const RVec3 color1 = RVec3::kZero;

        //-----------------------------------------------------------------------------
        // append
        lgt.m_Anims[YLight::ENABLE          ].m_FullValues.push_back(static_cast<float>(enable));
        lgt.m_Anims[YLight::POSITION_X      ].m_FullValues.push_back(position.x);
        lgt.m_Anims[YLight::POSITION_Y      ].m_FullValues.push_back(position.y);
        lgt.m_Anims[YLight::POSITION_Z      ].m_FullValues.push_back(position.z);
        lgt.m_Anims[YLight::DIRECTION_X     ].m_FullValues.push_back(direction.x);
        lgt.m_Anims[YLight::DIRECTION_Y     ].m_FullValues.push_back(direction.y);
        lgt.m_Anims[YLight::DIRECTION_Z     ].m_FullValues.push_back(direction.z);
        lgt.m_Anims[YLight::AIM_X           ].m_FullValues.push_back(aim.x);
        lgt.m_Anims[YLight::AIM_Y           ].m_FullValues.push_back(aim.y);
        lgt.m_Anims[YLight::AIM_Z           ].m_FullValues.push_back(aim.z);
        lgt.m_Anims[YLight::DIST_ATTN_START ].m_FullValues.push_back(distAttnStart);
        lgt.m_Anims[YLight::DIST_ATTN_END   ].m_FullValues.push_back(distAttnEnd);
        lgt.m_Anims[YLight::ANGLE_ATTN_START].m_FullValues.push_back(angleAttnStart);
        lgt.m_Anims[YLight::ANGLE_ATTN_END  ].m_FullValues.push_back(angleAttnEnd);
        lgt.m_Anims[YLight::COLOR0_R        ].m_FullValues.push_back(color0.x);
        lgt.m_Anims[YLight::COLOR0_G        ].m_FullValues.push_back(color0.y);
        lgt.m_Anims[YLight::COLOR0_B        ].m_FullValues.push_back(color0.z);
        lgt.m_Anims[YLight::COLOR1_R        ].m_FullValues.push_back(color1.x);
        lgt.m_Anims[YLight::COLOR1_G        ].m_FullValues.push_back(color1.y);
        lgt.m_Anims[YLight::COLOR1_B        ].m_FullValues.push_back(color1.z);
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief ライトのキーアニメーション値に適用するスケールを取得します。
//!
//! @param[in] paramIdx パラメーターインデックスです。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return ライトのキーアニメーション値に適用するスケールを返します。
//-----------------------------------------------------------------------------
static float GetLightKeyScale(const int paramIdx, const YExpOpt& yopt)
{
    return (YLight::IsRotate(paramIdx)   ) ? static_cast<float>(R_M_RAD_TO_DEG)         :
           (YLight::IsTranslate(paramIdx)) ? static_cast<float>(yopt.m_InternalMagnify) :
           1.0f;
}

//-----------------------------------------------------------------------------
//! @brief ライトについて Maya 上のキーを取得可能か判定します。
//!
//! @param[in] paramIdx パラメーターインデックスです。
//! @param[in] chan チャンネル入力情報です。
//! @param[in] fullValues ベイクしたデータです。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return Maya 上のキーを取得可能なら true を返します。
//-----------------------------------------------------------------------------
static bool GetsLightMayaKey(
    const int paramIdx,
    const ChannelInput& chan,
    const RFloatArray& fullValues,
    const YExpOpt& yopt
)
{
    if (yopt.m_BakeAllAnim   ||
        !chan.m_CurveFlag    ||
        chan.m_CurveWeighted)
    {
        return false;
    }

    const float valueScale = GetLightKeyScale(paramIdx, yopt);
    const float valueOfs = 0.0f;
    return IsValidAnimCurveValue(chan.m_CurveObj, fullValues, valueScale, valueOfs, yopt);
}

//-----------------------------------------------------------------------------
//! @brief 全サブフレームにおけるライトのアニメーション値を分析します。
//!        アニメーション値が一定か判定します。
//!
//! @param[in,out] ylights ライト配列です。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static void AnalyzeLightFullAnim(YLightArray& ylights, const YExpOpt& yopt)
{
    const int lgtCount = static_cast<int>(ylights.size());
    for (int lgtIdx = 0; lgtIdx < lgtCount; ++lgtIdx)
    {
        YLight& lgt = ylights[lgtIdx];
        YAnimCurve* curves = lgt.m_Anims;
        for (int paramIdx = 0; paramIdx < YLight::PARAM_COUNT; ++paramIdx)
        {
            //-----------------------------------------------------------------------------
            // ライトの各パラメーターについてアニメーション値が一定か判定し、
            // Maya 上のキーを取得可能か判定します。
            YAnimCurve& curve = curves[paramIdx];
            const bool isRotate = YLight::IsRotate(paramIdx);
            curve.m_Name = lgt.m_Name + "." + YLight::GetParamName(paramIdx);
            curve.m_LoopFlag = yopt.m_LoopAnim;
            curve.m_AngleFlag = isRotate;

            #ifdef OPTIMIZE_VECTOR_BY_TOL_R
            if (YLight::IsVector(paramIdx))
            {
                // ベクトルは開始フレームのベクトルとの角度差で一定か判定します（3 軸同時に判定）。
                if (paramIdx == YLight::DIRECTION_X)
                {
                    RUpdateVectorAnimConstantFlag(
                        curve, curves[paramIdx + 1], curves[paramIdx + 2], yopt.m_TolR);
                }
                curve.m_Baked = true; // ベクトルは常にベイクしたデータからアニメーションカーブデータを作成します。
                continue;
            }
            #endif
            curve.m_Tolerance =
                (isRotate                     ) ? yopt.m_TolR                                      :
                (YLight::IsTranslate(paramIdx)) ? static_cast<float>(yopt.m_TolT * yopt.m_Magnify) :
                (YLight::IsColor(paramIdx)    ) ? yopt.m_TolC                                      :
                (YLight::IsVector(paramIdx)   ) ? R_SAME_TOLERANCE_F                               :
                R_MAKE_KEY_TOL_LIGHT_OTHER; // ENABLE
                // ↑ m_TolT は Magnify を掛ける前の値に対する許容値なので Magnify を掛けます。
                // ↑ 現在、ベクトルは別の関数で 3 軸同時に一定か判定しています。
            curve.UpdateConstantFlag();

            const MPlug& plug = lgt.m_AnimPlugs[paramIdx];
            ChannelInput& chan = lgt.m_AnimChans[paramIdx];
            if (!plug.isNull())
            {
                chan.GetInput(plug);
            }
            curve.m_Baked = (curve.m_ConstantFlag ||
                !GetsLightMayaKey(paramIdx, chan, curve.m_FullValues, yopt));
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief ライトのキーアニメーションを取得します。
//!
//! @param[in,out] ylights ライト配列です。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetLightKeyAnim(YLightArray& ylights, const YExpOpt& yopt)
{
    const int lgtCount = static_cast<int>(ylights.size());
    for (int lgtIdx = 0; lgtIdx < lgtCount; ++lgtIdx)
    {
        YLight& lgt = ylights[lgtIdx];
        YAnimCurve* curves = lgt.m_Anims;
        for (int paramIdx = 0; paramIdx < YLight::PARAM_COUNT; ++paramIdx)
        {
            //-----------------------------------------------------------------------------
            // ベクトルは 3 軸同時にアニメーションカーブデータを作成します。
            YAnimCurve& curve = curves[paramIdx];
            #ifdef OPTIMIZE_VECTOR_BY_TOL_R
            if (YLight::IsVector(paramIdx))
            {
                if (paramIdx == YLight::DIRECTION_X)
                {
                    RMakeVectorAnimKeys(
                        curve, curves[paramIdx + 1], curves[paramIdx + 2],
                        GetFloatFrameFromSubFrame, &yopt);
                }
                continue;
            }
            #endif

            //-----------------------------------------------------------------------------
            // アニメーションカーブデータを取得または作成します。
            if (!curve.m_ConstantFlag)
            {
                if (!curve.m_Baked)
                {
                    const ChannelInput& chan = lgt.m_AnimChans[paramIdx];
                    if (paramIdx == YLight::ENABLE)
                    {
                        curve.GetStepKeys(chan.m_CurveObj, yopt);
                    }
                    else
                    {
                        const float valueScale = GetLightKeyScale(paramIdx, yopt);
                        const float valueOfs = 0.0f;
                        curve.GetKeys(chan.m_CurveObj, valueScale, valueOfs, yopt);
                    }
                }
                else
                {
                    if (paramIdx == YLight::ENABLE)
                    {
                        curve.MakeStepKeys(GetFloatFrameFromSubFrame, &yopt);
                    }
                    else
                    {
                        curve.MakeKeys(GetFloatFrameFromSubFrame, &yopt, true);
                    }
                }
            }
        }
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief <light_anim> 要素を出力します。
//-----------------------------------------------------------------------------
void YLight::OutAnim( // OutLightAnim
    std::ostream& os,
    RDataStreamArray& dataStreams,
    const int tabCount,
    const int index
) const
{
    static const char* const typeStrs[] = { "ambient", "directional", "point", "spot" };

    const int& tc = tabCount;

    //-----------------------------------------------------------------------------
    // begin
    os << RTab(tc) << "<light_anim"
       << " index=\"" << index
       << "\" light_name=\"" << RGetUtf8FromShiftJis(m_Name) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "frame_count=\"" << m_FrameCount
       << "\" loop=\"" << RBoolStr(m_LoopAnim) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "type=\"" << typeStrs[m_Type] << "\"" << R_ENDL;
    os << RTab(tc + 1) << "dist_attn_func =\"" << m_DistAttnFunc << "\"" << R_ENDL;
    os << RTab(tc + 1) << "angle_attn_func =\"" << m_AngleAttnFunc << "\"" << R_ENDL;
    os << RTab(tc) << ">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // light anim targets
    for (int iParam = 0; iParam < PARAM_COUNT; ++iParam)
    {
        if (POSITION_X <= iParam && iParam <= POSITION_Z)
        {
            if (m_Type != POINT && m_Type != SPOT) continue;
        }
        else if (DIRECTION_X <= iParam && iParam <= DIRECTION_Z)
        {
            if (m_Type != DIRECTIONAL) continue;
        }
        else if (AIM_X <= iParam && iParam <= AIM_Z)
        {
            if (m_Type != SPOT) continue;
        }
        else if (iParam == DIST_ATTN_START || iParam == DIST_ATTN_END)
        {
            if (!m_UsesDistAttn) continue;
        }
        else if (iParam == ANGLE_ATTN_START || iParam == ANGLE_ATTN_END)
        {
            if (m_Type != SPOT) continue;
        }
        else if (COLOR1_R <= iParam && iParam <= COLOR1_B)
        {
            continue; // Maya からは color1_* を出力しません。
        }

        const YAnimCurve& curve = m_Anims[iParam];
        curve.Out(os, dataStreams, tc + 1, "light_anim_target",
            GetParamName(iParam), false);
    }

    //-----------------------------------------------------------------------------
    // ライトのユーザーデータ配列を出力します。
    ROutArrayElement(os, tc + 1, m_UserDatas, "user_data_array");

    //-----------------------------------------------------------------------------
    // end
    os << RTab(tc) << "</light_anim>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief Maya 用のフォグの距離減衰を取得します。
//-----------------------------------------------------------------------------
void YFog::GetDistAttn(
    float& startDist,
    float& endDist,
    const double magnify,
    const bool snapToZero
) const
{
    m_AnimPlugs[DIST_ATTN_START].getValue(startDist);
    m_AnimPlugs[DIST_ATTN_END  ].getValue(endDist  );
    const float magnifyF = static_cast<float>(magnify);
    startDist *= magnifyF;
    endDist   *= magnifyF;
    if (snapToZero)
    {
        startDist = RSnapToZero(startDist);
        endDist   = RSnapToZero(endDist  );
    }
}

//-----------------------------------------------------------------------------
//! @brief Maya 用のフォグのカラーを返します。
//-----------------------------------------------------------------------------
RVec3 YFog::GetColor(const bool snapToZero) const
{
    RVec3 color;
    m_AnimPlugs[COLOR_R].getValue(color.x);
    m_AnimPlugs[COLOR_G].getValue(color.y);
    m_AnimPlugs[COLOR_B].getValue(color.z);
    if (snapToZero)
    {
        color.SnapToZero();
    }
    return color;
}

//-----------------------------------------------------------------------------
//! @brief Maya 用のフォグのパラメータに対応する Maya のアトリビュート名を返します。
//-----------------------------------------------------------------------------
const char* YFog::GetAttrName(const int iParam)
{
    static const char* const attrStrs[PARAM_COUNT] =
    {
        "fogNearDistance", "fogFarDistance",    // fog material
        "cr", "cg", "cb",                       // fog material
    };

    return attrStrs[iParam];
}

//-----------------------------------------------------------------------------
//! @brief Maya 用のフォグのコンストラクタです。
//-----------------------------------------------------------------------------
YFog::YFog( // YFogT
    RScene& rscene,
    const MDagPath& xformPath,
    const YExpOpt& yopt,
    MStatus* pStatus
)
: m_XformPath(xformPath)
{
    //-----------------------------------------------------------------------------
    // 処理結果を失敗で初期化します。
    if (pStatus != NULL)
    {
        *pStatus = MS::kFailure;
    }

    //-----------------------------------------------------------------------------
    // エクスポートオプションからパラメータをコピーします。
    m_FrameCount = yopt.m_OutFrameCount;
    m_LoopAnim   = yopt.m_LoopAnim;

    //-----------------------------------------------------------------------------
    // 名前を取得します。
    MFnTransform xformFn(m_XformPath);
    m_OrgName = xformFn.name().asChar();
    m_Name = GetOutElementName(m_OrgName, yopt.m_RemoveNamespace);

    //-----------------------------------------------------------------------------
    // シェイプノードを取得します。
    bool shapeExists = false;
    for (int iChild = 0; iChild < static_cast<int>(xformFn.childCount()); ++iChild)
    {
        const MObject childObj = xformFn.child(iChild);
        if (childObj.apiType() == MFn::kEnvFogShape)
        {
            if (MDagPath::getAPathTo(childObj, m_ShapePath))
            {
                shapeExists = true;
                break;
            }
        }
    }
    if (!shapeExists)
    {
        YShowWarning(&rscene, // No fog shape: %s // 通常は発生しない
            "環境フォグのシェイプノードが存在しません: {0}",
            "A shape node of the environment fog cannot be found: {0}",
            m_OrgName);
        return;
    }

    //-----------------------------------------------------------------------------
    // shadingEngine ノードを取得します。
    MFnDependencyNode shapeFn(m_ShapePath.node());
    MPlug instObjPlug = shapeFn.findPlug("instObjGroups");
    MPlugArray plugArray;
    instObjPlug[0].connectedTo(plugArray, false, true);
    if (plugArray.length() == 0)
    {
        YShowWarning(&rscene, // No fog shading group: %s // 通常は発生しない
            "環境フォグのシェーディンググループが存在しません: {0}",
            "A shading group of the environment fog cannot be found: {0}",
            m_OrgName);
        return;
    }
    m_SgObj = plugArray[0].node();

    //-----------------------------------------------------------------------------
    // 環境フォグ（Env Fog）マテリアル（envFog ノード）を取得します。
    MFnDependencyNode(m_SgObj).findPlug("volumeShader").connectedTo(plugArray, true, false);
    if (plugArray.length() == 0 ||
        plugArray[0].node().apiType() != MFn::kEnvFogMaterial)
    {
        YShowWarning(&rscene, // No fog material: %s // 通常は発生しない
            "環境フォグのマテリアル（envFog ノード）が存在しません: {0}",
            "A material (envFog node) of the environment fog cannot be found: {0}",
            m_OrgName);
        return;
    }
    m_MatObj = plugArray[0].node();

    //-----------------------------------------------------------------------------
    // アニメーションのパラメータに対応する Maya のプラグを取得します。
    MFnDependencyNode matFn(m_MatObj);
    for (int iParam = 0; iParam < PARAM_COUNT; ++iParam)
    {
        const char* attrName = GetAttrName(iParam);
        if (strlen(attrName) == 0)
        {
            continue;
        }
        else
        {
            m_AnimPlugs[iParam] = matFn.findPlug(attrName);
        }
        //cerr << "fog: " << GetParamName(iParam) << ": " << m_AnimPlugs[iParam].info() << " " << m_AnimPlugs[iParam].isNull() << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // get attr
    GetDistAttn(m_DistAttnStart, m_DistAttnEnd, yopt.m_InternalMagnify, true);
    m_Color = GetColor(true);

    //-----------------------------------------------------------------------------
    // 処理結果を成功にして終了します。
    if (pStatus != NULL)
    {
        *pStatus = MS::kSuccess;
    }

    //-----------------------------------------------------------------------------
    // debug
    #ifdef DEBUG_PRINT_SW
    cerr << "fog: " << m_OrgName << ": " << m_ShapePath.partialPathName() << ": " << matFn.name() << R_ENDL;
    #endif
}

//-----------------------------------------------------------------------------
//! @brief 全サブフレームにおけるフォグのアニメーション値を取得します。
//!
//! @param[in,out] yfogs フォグ配列です。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetFogFullAnimValue(YFogArray& yfogs, const YExpOpt& yopt)
{
    const int fogCount = static_cast<int>(yfogs.size());
    for (int iFog = 0; iFog < fogCount; ++iFog)
    {
        //-----------------------------------------------------------------------------
        // get
        YFog& fog = yfogs[iFog];

        float distAttnStart, distAttnEnd;
        fog.GetDistAttn(distAttnStart, distAttnEnd, yopt.m_InternalMagnify, true);
        const RVec3 color = fog.GetColor(true);

        //-----------------------------------------------------------------------------
        // append
        fog.m_Anims[YFog::DIST_ATTN_START].m_FullValues.push_back(distAttnStart);
        fog.m_Anims[YFog::DIST_ATTN_END  ].m_FullValues.push_back(distAttnEnd);
        fog.m_Anims[YFog::COLOR_R        ].m_FullValues.push_back(color.x);
        fog.m_Anims[YFog::COLOR_G        ].m_FullValues.push_back(color.y);
        fog.m_Anims[YFog::COLOR_B        ].m_FullValues.push_back(color.z);
    }

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief フォグのキーアニメーション値に適用するスケールを取得します。
//!
//! @param[in] paramIdx パラメーターインデックスです。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return フォグのキーアニメーション値に適用するスケールを返します。
//-----------------------------------------------------------------------------
static float GetFogKeyScale(const int paramIdx, const YExpOpt& yopt)
{
    return (YFog::IsTranslate(paramIdx)) ? static_cast<float>(yopt.m_InternalMagnify) :
           1.0f;
}

//-----------------------------------------------------------------------------
//! @brief フォグについて Maya 上のキーを取得可能か判定します。
//!
//! @param[in] paramIdx パラメーターインデックスです。
//! @param[in] chan チャンネル入力情報です。
//! @param[in] fullValues ベイクしたデータです。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return Maya 上のキーを取得可能なら true を返します。
//-----------------------------------------------------------------------------
static bool GetsFogMayaKey(
    const int paramIdx,
    const ChannelInput& chan,
    const RFloatArray& fullValues,
    const YExpOpt& yopt
)
{
    if (yopt.m_BakeAllAnim   ||
        !chan.m_CurveFlag    ||
        chan.m_CurveWeighted)
    {
        return false;
    }

    const float valueScale = GetFogKeyScale(paramIdx, yopt);
    const float valueOfs = 0.0f;
    return IsValidAnimCurveValue(chan.m_CurveObj, fullValues, valueScale, valueOfs, yopt);
}

//-----------------------------------------------------------------------------
//! @brief 全サブフレームにおけるフォグのアニメーション値を分析します。
//!        アニメーション値が一定か判定します。
//!
//! @param[in,out] yfogs フォグ配列です。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static void AnalyzeFogFullAnim(YFogArray& yfogs, const YExpOpt& yopt)
{
    const int fogCount = static_cast<int>(yfogs.size());
    for (int fogIdx = 0; fogIdx < fogCount; ++fogIdx)
    {
        YFog& fog = yfogs[fogIdx];
        for (int paramIdx = 0; paramIdx < YFog::PARAM_COUNT; ++paramIdx)
        {
            //-----------------------------------------------------------------------------
            // フォグの各パラメーターについてアニメーション値が一定か判定し、
            // Maya 上のキーを取得可能か判定します。
            YAnimCurve& curve = fog.m_Anims[paramIdx];
            curve.m_Name = fog.m_Name + "." + YFog::GetParamName(paramIdx);
            curve.m_LoopFlag = yopt.m_LoopAnim;
            curve.m_Tolerance =
                (YFog::IsTranslate(paramIdx)) ? static_cast<float>(yopt.m_TolT * yopt.m_Magnify) :
                yopt.m_TolC;
                // ↑ m_TolT は Magnify を掛ける前の値に対する許容値なので Magnify を掛けます。
            curve.UpdateConstantFlag();

            const MPlug& plug = fog.m_AnimPlugs[paramIdx];
            ChannelInput& chan = fog.m_AnimChans[paramIdx];
            if (!plug.isNull())
            {
                chan.GetInput(plug);
            }
            curve.m_Baked = (curve.m_ConstantFlag ||
                !GetsFogMayaKey(paramIdx, chan, curve.m_FullValues, yopt));
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief フォグのキーアニメーションを取得します。
//!
//! @param[in,out] yfogs フォグ配列です。
//! @param[in] yopt エクスポートオプションです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
static MStatus GetFogKeyAnim(YFogArray& yfogs, const YExpOpt& yopt)
{
    const int fogCount = static_cast<int>(yfogs.size());
    for (int fogIdx = 0; fogIdx < fogCount; ++fogIdx)
    {
        YFog& fog = yfogs[fogIdx];
        for (int paramIdx = 0; paramIdx < YFog::PARAM_COUNT; ++paramIdx)
        {
            //-----------------------------------------------------------------------------
            // アニメーションカーブデータを取得または作成します。
            YAnimCurve& curve = fog.m_Anims[paramIdx];
            if (!curve.m_ConstantFlag)
            {
                if (!curve.m_Baked)
                {
                    const ChannelInput& chan = fog.m_AnimChans[paramIdx];
                    const float valueScale = GetFogKeyScale(paramIdx, yopt);
                    const float valueOfs = 0.0f;
                    curve.GetKeys(chan.m_CurveObj, valueScale, valueOfs, yopt);
                }
                else
                {
                    curve.MakeKeys(GetFloatFrameFromSubFrame, &yopt, true);
                }
            }
        }
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief <fog_anim> 要素を出力します。
//-----------------------------------------------------------------------------
void YFog::OutAnim( // OutFogAnim
    std::ostream& os,
    RDataStreamArray& dataStreams,
    const int tabCount,
    const int index
) const
{
    const int& tc = tabCount;

    //-----------------------------------------------------------------------------
    // begin
    os << RTab(tc) << "<fog_anim"
       << " index=\"" << index
       << "\" fog_name=\"" << RGetUtf8FromShiftJis(m_Name) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "frame_count=\"" << m_FrameCount
       << "\" loop=\"" << RBoolStr(m_LoopAnim) << "\"" << R_ENDL;
    os << RTab(tc + 1) << "dist_attn_func =\"" << m_DistAttnFunc << "\"" << R_ENDL;
    os << RTab(tc) << ">" << R_ENDL;

    //-----------------------------------------------------------------------------
    // light anim targets
    for (int iParam = 0; iParam < PARAM_COUNT; ++iParam)
    {
        const YAnimCurve& curve = m_Anims[iParam];
        curve.Out(os, dataStreams, tc + 1, "fog_anim_target",
            GetParamName(iParam), false);
    }

    //-----------------------------------------------------------------------------
    // フォグのユーザーデータ配列を出力します。
    ROutArrayElement(os, tc + 1, m_UserDatas, "user_data_array");

    //-----------------------------------------------------------------------------
    // end
    os << RTab(tc) << "</fog_anim>" << R_ENDL;
}

//-----------------------------------------------------------------------------
//! @brief 任意の型のオブジェクトが配列内でユニークな名前を持つように名前を変更します。
//!
//! @param[in,out] rscene シーンです。
//! @param[in,out] array オブジェクト配列です。
//! @param[in] typeStr オブジェクトの型を表す文字列です（警告表示用）。
//! @param[in] yopt エクスポートオプションです。
//-----------------------------------------------------------------------------
template <typename T>
void SetUniqueObjNames(
    RScene& rscene,
    std::vector<T>& array,
    const char* typeStr,
    const YExpOpt& yopt
)
{
    const std::string bar = "_";

    const int elemCount = static_cast<int>(array.size());
    if (elemCount <= 1)
    {
        return;
    }
    for (int iElem = 1; iElem < elemCount; ++iElem)
    {
        T& elem = array[iElem];
        const std::string baseName = elem.m_Name;
        std::string name = baseName;
        int id = 1;
        for (;;)
        {
            bool found = false;
            for (int iOther = 0; iOther < iElem; ++iOther)
            {
                const T& other = array[iOther];
                if (other.m_Name == name)
                {
                    found = true;
                    break;
                }
            }
            if (found)
            {
                name = baseName + bar + RGetNumberString(id);
                ++id;
            }
            else
            {
                break;
            }
        }
        if (id > 1)
        {
            if (yopt.m_WarnsNodeNameChanged && !yopt.m_CheckElementFlag)
            {
                const std::string typeStrJp =
                    (strcmp(typeStr, "Camera") == 0) ? "カメラ" :
                    (strcmp(typeStr, "Light" ) == 0) ? "ライト" :
                    "フォグ";
                const std::string typeStrEn =
                    (strcmp(typeStr, "Camera") == 0) ? "cameras" :
                    (strcmp(typeStr, "Light" ) == 0) ? "lights"  :
                    "fogs";
                YShowWarning(&rscene,
                    "同じ名前の" + typeStrJp + "が複数存在するため、中間ファイルに出力される" +
                    typeStrJp + "名が変更されました: {0} -> {1}",
                    "Multiple " + typeStrEn + " with the same name exist, so the names of the " +
                    typeStrEn + " exported to the intermediate file were changed: {0} -> {1}",
                    elem.m_OrgName, name);
            }
            elem.m_Name = name;
        }
    }
}

//-----------------------------------------------------------------------------
//! @brief 環境オブジェクトが各カテゴリ内（カメラ、ライト、フォグ）で
//!        ユニークな名前を持つように名前を変更します。
//-----------------------------------------------------------------------------
void SetUniqueEnvObjNames(YEnvObjs& yenvObjs, RScene& rscene, const YExpOpt& yopt)
{
    SetUniqueObjNames(rscene, yenvObjs.m_YCameras, "Camera", yopt);
    SetUniqueObjNames(rscene, yenvObjs.m_YLights , "Light" , yopt);
    SetUniqueObjNames(rscene, yenvObjs.m_YFogs   , "Fog"   , yopt);
}

//-----------------------------------------------------------------------------
//! @brief カメラをアルファベット順にソートするための比較関数です。
//-----------------------------------------------------------------------------
static bool CameraPtrNameLess(YCamera*& r1, YCamera*& r2)
{
    return (strcmp(r1->m_Name.c_str(), r2->m_Name.c_str()) < 0);
}

//-----------------------------------------------------------------------------
//! @brief ライトをアルファベット順にソートするための比較関数です。
//-----------------------------------------------------------------------------
static bool LightPtrNameLess(YLight*& r1, YLight*& r2)
{
    return (strcmp(r1->m_Name.c_str(), r2->m_Name.c_str()) < 0);
}

//-----------------------------------------------------------------------------
//! @brief フォグをアルファベット順にソートするための比較関数です。
//-----------------------------------------------------------------------------
static bool FogPtrNameLess(YFog*& r1, YFog*& r2)
{
    return (strcmp(r1->m_Name.c_str(), r2->m_Name.c_str()) < 0);
}

//-----------------------------------------------------------------------------
//! @brief 任意の型のオブジェクト配列からポインタ配列を取得します。
//!
//! @param[out] ptrArray ポインタ配列を格納します。
//! @param[in] array オブジェクト配列です。
//-----------------------------------------------------------------------------
template <typename T>
void SetObjPtrArray(std::vector<T*>& ptrArray, std::vector<T>& array)
{
    const int objCount = static_cast<int>(array.size());
    if (objCount > 0)
    {
        ptrArray.resize(objCount);
        for (int iObj = 0; iObj < objCount; ++iObj)
        {
            ptrArray[iObj] = &array[iObj];
        }
    }
    else
    {
        ptrArray.clear();
    }
}

//-----------------------------------------------------------------------------
//! @brief 環境オブジェクトを中間ファイルに出力する順（名前順）にソートします。
//-----------------------------------------------------------------------------
void SortEnvObjByName(YEnvObjs& yenvObjs)
{
    //-----------------------------------------------------------------------------
    // camera
    SetObjPtrArray(yenvObjs.m_pOutYCameras, yenvObjs.m_YCameras);
    if (!yenvObjs.m_pOutYCameras.empty())
    {
        std::sort(yenvObjs.m_pOutYCameras.begin(), yenvObjs.m_pOutYCameras.end(),
            CameraPtrNameLess);
    }

    //-----------------------------------------------------------------------------
    // light
    SetObjPtrArray(yenvObjs.m_pOutYLights, yenvObjs.m_YLights);
    if (!yenvObjs.m_pOutYLights.empty())
    {
        std::sort(yenvObjs.m_pOutYLights.begin(), yenvObjs.m_pOutYLights.end(),
            LightPtrNameLess);
    }

    //-----------------------------------------------------------------------------
    // fog
    SetObjPtrArray(yenvObjs.m_pOutYFogs, yenvObjs.m_YFogs);
    if (!yenvObjs.m_pOutYFogs.empty())
    {
        std::sort(yenvObjs.m_pOutYFogs.begin(), yenvObjs.m_pOutYFogs.end(),
            FogPtrNameLess);
    }
}

//-----------------------------------------------------------------------------
//! @brief 全サブフレームにおける環境オブジェクトのアニメーション値を取得します。
//!        現在のフレームを変化させるたびに呼ばれます。
//-----------------------------------------------------------------------------
void GetEnvObjFullAnimValue(YEnvObjs& yenvObjs, const YExpOpt& yopt)
{
    GetCameraFullAnimValue(yenvObjs.m_YCameras, yopt);
    GetLightFullAnimValue(yenvObjs.m_YLights, yopt);
    GetFogFullAnimValue(yenvObjs.m_YFogs, yopt);
}

//-----------------------------------------------------------------------------
//! @brief 全サブフレームにおける環境オブジェクトのアニメーション値を分析します。
//!        回転の角度が連続的になるように調整します。
//!        アニメーション値が一定か判定します。
//-----------------------------------------------------------------------------
void AnalyzeEnvObjFullAnim(YEnvObjs& yenvObjs, const YExpOpt& yopt)
{
    AnalyzeCameraFullAnim(yenvObjs.m_YCameras, yopt);
    AnalyzeLightFullAnim(yenvObjs.m_YLights, yopt);
    AnalyzeFogFullAnim(yenvObjs.m_YFogs, yopt);
}

//-----------------------------------------------------------------------------
//! @brief 環境オブジェクトのキーアニメーションを取得します。
//-----------------------------------------------------------------------------
MStatus GetEnvObjKeyAnim(YEnvObjs& yenvObjs, const YExpOpt& yopt)
{
    GetCameraKeyAnim(yenvObjs.m_YCameras, yopt);
    GetLightKeyAnim(yenvObjs.m_YLights, yopt);
    GetFogKeyAnim(yenvObjs.m_YFogs, yopt);

    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief 環境オブジェクト群のユーザーデータを取得します。
//-----------------------------------------------------------------------------
void GetEnvObjUserData(YEnvObjs* pYEnvObjs, RScene* pScene)
{
    //-----------------------------------------------------------------------------
    // カメラ
    YCameraArray& ycameras = pYEnvObjs->m_YCameras;
    const int camCount = static_cast<int>(ycameras.size());
    for (int camIdx = 0; camIdx < camCount; ++camIdx)
    {
        ycameras[camIdx].GetUserData(pScene);
    }

    //-----------------------------------------------------------------------------
    // ライト
    YLightArray& ylights = pYEnvObjs->m_YLights;
    const int lgtCount = static_cast<int>(ylights.size());
    for (int lgtIdx = 0; lgtIdx < lgtCount; ++lgtIdx)
    {
        ylights[lgtIdx].GetUserData(pScene);
    }

    //-----------------------------------------------------------------------------
    // フォグ
    YFogArray& yfogs = pYEnvObjs->m_YFogs;
    const int fogCount = static_cast<int>(yfogs.size());
    for (int fogIdx = 0; fogIdx < fogCount; ++fogIdx)
    {
        yfogs[fogIdx].GetUserData(pScene);
    }
}

//-----------------------------------------------------------------------------
//! @brief 環境オブジェクトの数が制限を超えていないか判定します。
//-----------------------------------------------------------------------------
MStatus CheckEnvObjCount(const YEnvObjs& yenvObjs, const YExpOpt& yopt)
{
    R_UNUSED_VARIABLE(yenvObjs);

    if (!yopt.m_CheckElementFlag)
    {
        //if (static_cast<int>(yenvObjs.m_YCameras.size()) > R_NW_CAMERA_SIZE)
        //{
        //  YShowWarning(&rscene, "", "The number of cameras is over 32"); // 現在は発生しない
        //}
        //if (static_cast<int>(yenvObjs.m_YLights.size()) > R_NW_LIGHT_SIZE)
        //{
        //  YShowWarning(&rscene, "", "The number of lights is over 128"); // 現在は発生しない
        //}
        //if (static_cast<int>(yenvObjs.m_YFogs.size()) > R_NW_FOG_SIZE)
        //{
        //  YShowWarning(&rscene, "", "The number of fogs is over 32"); // 現在は発生しない
        //}
    }
    return MS::kSuccess;
}

//-----------------------------------------------------------------------------
//! @brief fsn ファイル（シーンアニメーション）を出力します。
//-----------------------------------------------------------------------------
MStatus OutputFsnFile(
    std::ostream& os,
    RDataStreamArray& dataStreams,
    const YEnvObjs& yenvObjs
)
{
    const int tc = 0;

    //-----------------------------------------------------------------------------
    // camera
    const int camCount = static_cast<int>(yenvObjs.m_pOutYCameras.size());
    if (camCount > 0)
    {
        os << RTab(tc) << "<camera_anim_array length=\"" << camCount << "\">" << R_ENDL;
        for (int iCam = 0; iCam < camCount; ++iCam)
        {
            yenvObjs.m_pOutYCameras[iCam]->OutAnim(os, dataStreams, tc + 1, iCam);
        }
        os << RTab(tc) << "</camera_anim_array>" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // light
    const int lgtCount = static_cast<int>(yenvObjs.m_pOutYLights.size());
    if (lgtCount > 0)
    {
        os << RTab(tc) << "<light_anim_array length=\"" << lgtCount << "\">" << R_ENDL;
        for (int iLgt = 0; iLgt < lgtCount; ++iLgt)
        {
            yenvObjs.m_pOutYLights[iLgt]->OutAnim(os, dataStreams, tc + 1, iLgt);
        }
        os << RTab(tc) << "</light_anim_array>" << R_ENDL;
    }

    //-----------------------------------------------------------------------------
    // fog
    const int fogCount = static_cast<int>(yenvObjs.m_pOutYFogs.size());
    if (fogCount > 0)
    {
        os << RTab(tc) << "<fog_anim_array length=\"" << fogCount << "\">" << R_ENDL;
        for (int iFog = 0; iFog < fogCount; ++iFog)
        {
            yenvObjs.m_pOutYFogs[iFog]->OutAnim(os, dataStreams, tc + 1, iFog);
        }
        os << RTab(tc) << "</fog_anim_array>" << R_ENDL;
    }

    return MS::kSuccess;
}

