﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// CheckQuantizeType
// CheckUVQuantizeTypeCompatibility
// CheckColorQuantizeTypeCompatibility

//=============================================================================
// include
//=============================================================================
#include "LodQuantize.h"

#include "LodCommon.h"
#include "LodModel.h"

#include <cfloat>

#include <maya/MIntArray.h>
#include <maya/MFloatArray.h>
#include <maya/MFloatVectorArray.h>
#include <maya/MColorArray.h>
#include <maya/MFnMesh.h>
#include <maya/MItMeshPolygon.h>

//=============================================================================
// constants
//=============================================================================
namespace {

//! 量子化タイプ
enum QuantizeType
{
    none,
    unorm_8,
    uint_8,
    snorm_8,
    sint_8,
    uint_to_float_8,
    sint_to_float_8,
    unorm_4_4,
    unorm_16,
    uint_16,
    snorm_16,
    sint_16,
    float_16,
    uint_to_float_16,
    sint_to_float_16,
    unorm_8_8,
    uint_8_8,
    snorm_8_8,
    sint_8_8,
    uint_to_float_8_8,
    sint_to_float_8_8,
    uint_32,
    sint_32,
    float_32,
    unorm_16_16,
    uint_16_16,
    snorm_16_16,
    sint_16_16,
    float_16_16,
    uint_to_float_16_16,
    sint_to_float_16_16,
    float_10_11_11,
    unorm_8_8_8_8,
    uint_8_8_8_8,
    snorm_8_8_8_8,
    sint_8_8_8_8,
    uint_to_float_8_8_8_8,
    sint_to_float_8_8_8_8,
    unorm_10_10_10_2,
    uint_10_10_10_2,
    snorm_10_10_10_2,
    sint_10_10_10_2,
    uint_32_32,
    sint_32_32,
    float_32_32,
    unorm_16_16_16_16,
    uint_16_16_16_16,
    snorm_16_16_16_16,
    sint_16_16_16_16,
    float_16_16_16_16,
    uint_to_float_16_16_16_16,
    sint_to_float_16_16_16_16,
    uint_32_32_32,
    sint_32_32_32,
    float_32_32_32,
    uint_32_32_32_32,
    sint_32_32_32_32,
    float_32_32_32_32,
};

QuantizeType s_S8Types[] = { snorm_8, snorm_8_8, snorm_8_8_8_8, snorm_8_8_8_8 };
QuantizeType s_U8Types[] = { unorm_8, unorm_8_8, unorm_8_8_8_8, unorm_8_8_8_8 };
QuantizeType s_F32Types[] = { float_32, float_32_32, float_32_32_32, float_32_32_32_32 };

} // namespace unnamed

//=============================================================================
// lod ネームスペースを開始します。
//=============================================================================
namespace lod {

//! 法線の量子化タイプを取得します。
//QuantizeType GetNormalQuantizeType(const Shape *pShape)
//{
//  MItMeshPolygon polyIt(pShape->dag, const_cast<Shape*>(pShape)->faceComp);
//
//  for ( ; !polyIt.isDone(); polyIt.next())
//  {
//      MVectorArray normals;
//      polyIt.getNormals(normals);
//
//      for (unsigned i = 0 ; i < normals.length() ; ++i)
//      {
//          const MVector& vec = normals[i];
//          if ( fabs(vec.x) > 1.0 || fabs(vec.y) > 1.0 || fabs(vec.z) > 1.0)
//          {
//              return float_32_32_32;
//          }
//      }
//
//  }
//
//  return snorm_10_10_10_2;
//}

//! 接線/従法線の量子化タイプを取得します。
//QuantizeType GetTanBinQuantizeType(const Shape *pShape,bool isTangent)
//{
//  MFloatVectorArray tanOrBins;
//  MFnMesh meshFn(pShape->dag);
//  if (isTangent) meshFn.getTangents(tanOrBins);
//  else meshFn.getBinormals(tanOrBins);
//
//  MItMeshPolygon polyIt(pShape->dag, const_cast<Shape*>(pShape)->faceComp);
//  for ( ; !polyIt.isDone(); polyIt.next())
//  {
//      MIntArray vtxIds;
//      polyIt.getVertices(vtxIds);
//
//      for (unsigned i = 0 ; i < vtxIds.length(); ++i)
//      {
//          unsigned tanId = polyIt.tangentIndex(i);
//          const MFloatVector& vec = tanOrBins[tanId];
//
//          if ( fabs(vec.x) > 1.0f || fabs(vec.y) > 1.0f || fabs(vec.z) > 1.0f)
//          {
//              return float_32_32_32_32;
//          }
//      }
//
//  }
//
//  return snorm_8_8_8_8;
//}

//! 指定シェイプの法線が量子化タイプに適合するかチェックします。
//bool CheckNormalQuantizeTypeCompatibility(MSelectionList& errorSelection,QuantizeType type,const Shape *pShape)
//{
//  if (type == snorm_10_10_10_2)
//  {
//      MIntArray errorVtxIds,errorFaceIds;
//      bool hasError = false;
//
//      MItMeshPolygon polyIt(pShape->dag, const_cast<Shape*>(pShape)->faceComp);
//      for ( ; !polyIt.isDone(); polyIt.next())
//      {
//          MVectorArray normals;
//          MIntArray vtxIds;
//          polyIt.getNormals(normals);
//          polyIt.getVertices(vtxIds);
//          for (unsigned i = 0 ; i < normals.length() ; ++i)
//          {
//              const MVector& n = normals[i];
//              if ( fabs(n.x) > 1.0 || fabs(n.y) > 1.0 || fabs(n.z) > 1.0)
//              {
//                  errorVtxIds.append(vtxIds[i]);
//                  errorFaceIds.append(polyIt.index());
//                  hasError = true;
//              }
//          }
//      }
//
//      if (hasError)
//      {
//          PrintF("Invalid normal quantize type: %s",pShape->dag.partialPathName().asChar());
//          for (unsigned i = 0 ; i < errorVtxIds.length() ; ++i)
//          {
//              if (i >= LOD_VTX_PRINT_COUNT_MAX)
//              {
//                  PrintF("      ...");
//                  break;
//              }
//              PrintF("      %s.vtxFace[%d][%d]",pShape->dag.partialPathName().asChar(),errorVtxIds[i],errorFaceIds[i]);
//          }
//
//          // エラー選択リストに追加
//          MObject vtxFaceComp;
//          GetMeshVertexFaceComponentFromIndices(vtxFaceComp,errorVtxIds,errorFaceIds);
//          errorSelection.add(pShape->dag,vtxFaceComp);
//
//          return false;
//      }
//      else
//      {
//          return true;
//      }
//  }
//  else
//  {
//      return true;
//  }
//
//}

//! 指定シェイプの接線/従法線列が量子化タイプに適合するかチェックします。
//bool CheckTanBinQuantizeTypeCompatibility(MSelectionList& errorSelection,QuantizeType type,const Shape *pShape,bool isTangent)
//{
//  if (type == snorm_8_8_8_8)
//  {
//      MFnMesh meshFn(pShape->dag);
//
//      MFloatVectorArray tanOrBins;
//      if (isTangent) meshFn.getTangents(tanOrBins);
//      else meshFn.getBinormals(tanOrBins);
//
//      MIntArray errorVtxIds,errorFaceIds;
//      bool hasError = false;
//
//      MItMeshPolygon polyIt(pShape->dag, const_cast<Shape*>(pShape)->faceComp);
//      for ( ; !polyIt.isDone(); polyIt.next())
//      {
//          MIntArray vtxIds;
//          polyIt.getVertices(vtxIds);
//          for (unsigned v = 0 ; v < vtxIds.length() ; ++v)
//          {
//              unsigned tanId = polyIt.tangentIndex(v);
//
//              const MFloatVector& tb = tanOrBins[tanId];
//              if ( fabs(tb.x) > 1.0f || fabs(tb.y) > 1.0f || fabs(tb.z) > 1.0f)
//              {
//                  errorVtxIds.append(vtxIds[v]);
//                  errorFaceIds.append(polyIt.index());
//                  hasError = true;
//              }
//          }
//      }
//
//      if (hasError)
//      {
//          PrintF("Invalid %s quantize type: %s",isTangent ? "tangent" : "binormal", pShape->dag.partialPathName().asChar());
//          for (unsigned i = 0 ; i < errorVtxIds.length() ; ++i)
//          {
//              if (i >= LOD_VTX_PRINT_COUNT_MAX)
//              {
//                  PrintF("      ...");
//                  break;
//              }
//              PrintF("      %s.vtxFace[%d][%d]",pShape->dag.partialPathName().asChar(),errorVtxIds[i],errorFaceIds[i]);
//          }
//
//          // エラー選択リストに追加
//          MObject vtxFaceComp;
//          GetMeshVertexFaceComponentFromIndices(vtxFaceComp,errorVtxIds,errorFaceIds);
//          errorSelection.add(pShape->dag,vtxFaceComp);
//
//          return false;
//      }
//      else
//      {
//          return true;
//      }
//
//
//  }
//  else
//  {
//      return true;
//  }
//
//}

//! UVの量子化タイプを取得します。
//! ただし、本チェックでは値域系のみしか行わないため、現状は正負、絶対値が 1.0 を超えるかの条件で
//! snorm_16_16, unorm_16_16, float_32_32 しか返しません。
QuantizeType GetUVQuantizeType(const Shape* pShape, int index)
{
    bool negative = false;
    bool overOne = false;

    const MString& uvSetName = pShape->uvSets[index];

    MObject faceComp = pShape->faceComp;
    MItMeshPolygon polyIt(pShape->dag, faceComp);
    for ( ; !polyIt.isDone() ; polyIt.next())
    {
        MFloatArray uArray;
        MFloatArray vArray;
        polyIt.getUVs(uArray, vArray, &uvSetName);

        for (unsigned i = 0 ; i < uArray.length() ; ++i)
        {
            float u = uArray[i];
            float v = vArray[i];

            if (u < 0.0f || v < 0.0f) negative = true;
            if (fabs(u) > 1.0f || fabs(v) > 1.0f) overOne = true;
        }

        // 両方フラグが立っている場合はもうチェック不要です。
        if (negative && overOne) break;
    }

    if (overOne)
    {
        return float_32_32;
    }
    else
    {
        return negative ? snorm_16_16 : unorm_16_16;
    }
}

//! 指定シェイプの UV が量子化タイプに適合するかチェックします。
bool CheckUVQuantizeTypeCompatibility(MSelectionList& errorSelection,QuantizeType type,const Shape *pShape,int index)
{
    const bool isJp = IsLodJapaneseUi();

    // float_32_32の場合は無条件でＯＫです。
    if (type == float_32_32) return true;

    const MString& uvSetName = pShape->uvSets[index];

    MObject faceComp = pShape->faceComp;
    MItMeshPolygon polyIt(pShape->dag,faceComp);

    MIntArray errorUVIndex;

    if (type == snorm_16_16)
    {
        // 絶対値 1.0 を超えるものがあればエラーです
        for ( ; !polyIt.isDone() ; polyIt.next())
        {
            MFloatArray uArray,vArray;
            polyIt.getUVs(uArray,vArray,&uvSetName);

            for (unsigned i = 0 ; i < uArray.length() ; ++i)
            {
                float u = uArray[i];
                float v = vArray[i];

                if (fabs(u) > 1.0f || fabs(v) > 1.0f)
                {
                    int uvId;
                    polyIt.getUVIndex(i,uvId,&uvSetName);
                    errorUVIndex.append(uvId);
                }
            }
        }
    }
    else if (type == unorm_16_16)
    {
        // 負値、絶対値 1.0 を超えるものがあればエラーです
        for ( ; !polyIt.isDone() ; polyIt.next())
        {
            MFloatArray uArray,vArray;
            polyIt.getUVs(uArray,vArray,&uvSetName);

            for (unsigned i = 0 ; i < uArray.length() ; ++i)
            {
                float u = uArray[i];
                float v = vArray[i];

                if (u < 0.0f || v < 0.0f || fabs(u) > 1.0f || fabs(v) > 1.0f)
                {
                    int uvId;
                    polyIt.getUVIndex(i,uvId,&uvSetName);
                    errorUVIndex.append(uvId);
                }
            }
        }
    }

    if (errorUVIndex.length() > 0)
    {
        MString condition;
        if (isJp)
        {
            condition = (type == unorm_16_16) ?
                "値を [0,1] の範囲に収めることを推奨します" :
                "値を [-1,1] の範囲に収めることを推奨します";
        }
        else
        {
            condition = (type == unorm_16_16) ?
                "value shoud be in range [0,1]" :
                "value shoud be in range [-1,1]";
        }
        MString shapeName = pShape->dag.partialPathName();

        MGlobal::displayWarning(MStringF((isJp ? "UV の量子化タイプが一致していません (%s): %s [%s]" :
            "UV quantize type does not match (%s): %s [%s]"),
            condition.asChar(), shapeName.asChar(), uvSetName.asChar()));
        for (unsigned i = 0 ; i < errorUVIndex.length() ; ++i)
        {
            if (i >= LOD_VTX_PRINT_COUNT_MAX)
            {
                PrintF("      ...");
                break;
            }
            PrintF("      %s.map[%d]",shapeName.asChar(),errorUVIndex[i]);
        }

        // エラー選択リストに追加
        MObject uvComp;
        GetMeshComponentFromIndices(uvComp, errorUVIndex, MFn::kMeshMapComponent);
        errorSelection.add(pShape->dag,uvComp);

        return false;
    }
    else
    {
        return true;
    }
}

//! 頂点カラーの量子化タイプを取得します。
//! ただし、本チェックでは値域系のみしか行わないため、現状は正負、絶対値が 1.0 を超えるかの条件で
//! snorm8系,unorm8系,float32系しか返しません。
QuantizeType GetColorQuantizeType(const Shape *pShape,int index)
{
    const Shape::ColorSet& colorSet = pShape->colorSets[index];
    const MString& colorSetName = colorSet.name;

    bool negative = false;
    bool overOne = false;

    MObject faceComp = pShape->faceComp;
    MItMeshPolygon polyIt(pShape->dag,faceComp);
    for ( ; !polyIt.isDone() ; polyIt.next())
    {
        MColorArray colors;
        polyIt.getColors(colors,&colorSetName);

        for (unsigned i = 0 ; i < colors.length() ; ++i)
        {
            if (colorSet.repr == MFnMesh::kAlpha)
            {
                float a = colors[i].a;
                if (a < 0.0f) negative = true;
                else if ( fabs(a) > 1.0f) overOne = true;
            }
            else
            {
                for (int c = 0; c < colorSet.compCount; ++c)
                {
                    const float v = (colorSet.isRa && c == 1) ? colors[i][3] : colors[i][c];

                    if (v < 0.0f) negative = true;
                    else if ( fabs(v) > 1.0f) overOne = true;
                }

            }

        }

        // 両方フラグが立っている場合はもうチェック不要です。
        if (negative && overOne) break;
    }

    if (overOne)
    {
        return s_F32Types[colorSet.compCount - 1];
    }
    else
    {
        if (negative) return s_S8Types[colorSet.compCount - 1];
        else return s_U8Types[colorSet.compCount - 1];
    }

}

//! 指定シェイプの頂点カラーが量子化タイプに適合するかチェックします。
bool CheckColorQuantizeTypeCompatibility(MSelectionList& errorSelection,QuantizeType type,const Shape *pShape,int index)
{
    const bool isJp = IsLodJapaneseUi();

    // float32系はサイズチェックは事前に済ませてあるので無条件ＯＫ
    switch (type)
    {
    case float_32:
    case float_32_32:
    case float_32_32_32:
    case float_32_32_32_32:
        return true;
    default:
        break;
    }

    const Shape::ColorSet& colorSet = pShape->colorSets[index];
    const MString& colorSetName = colorSet.name;

    MFnMesh meshFn(pShape->dag);

    MIntArray errorColorIds;

    MObject faceComp = pShape->faceComp;
    MItMeshPolygon polyIt(pShape->dag,faceComp);


    for ( ; !polyIt.isDone() ; polyIt.next())
    {
        MColorArray colors;
        polyIt.getColors(colors,&colorSetName);

        for (unsigned i = 0 ; i < colors.length() ; ++i)
        {
            std::vector<float> checkValues;
            bool errorColor = false;

            // カラーセットタイプが A の場合はαのみチェック
            if (colorSet.repr == MFnMesh::kAlpha)
            {
                checkValues.push_back(colors[i].a);
            }
            else
            {
                for (int c = 0; c < colorSet.compCount; ++c)
                {
                    checkValues.push_back(
                        (colorSet.isRa && c == 1) ? colors[i][3] : colors[i][c]);
                }
            }

            for (size_t j = 0 ; j < checkValues.size() ; ++j)
            {
                float v = checkValues[j];

                switch(type)
                {
                case snorm_8:
                case snorm_8_8:
                case snorm_8_8_8_8:
                    if (fabs(v) > 1.0f) errorColor = true;
                    break;

                case unorm_8:
                case unorm_8_8:
                case unorm_8_8_8_8:
                    if (v < 0.0f || fabs(v) > 1.0f) errorColor = true;
                    break;

                default:
                    break;
                }
            }

            if (errorColor)
            {
                int colorId;
                polyIt.getColorIndex(i,colorId,&colorSetName);
                errorColorIds.append(colorId);
            }


        }
    }

    if (errorColorIds.length() > 0)
    {
        MString condition;
        switch (type)
        {
        case snorm_8:
        case snorm_8_8:
        case snorm_8_8_8_8:
            condition = (isJp ? "値を [-1,1] の範囲に収めることを推奨します" :
                "value shoud be in range [-1,1]");
            break;

        case unorm_8:
        case unorm_8_8:
        case unorm_8_8_8_8:
            condition = (isJp ? "値を [0,1] の範囲に収めることを推奨します" :
                "value shoud be in range [0,1]");
            break;

        default:
            break;
        }

        MString shapeName = pShape->dag.partialPathName();

        MGlobal::displayWarning(MStringF((isJp ? "頂点カラーの量子化タイプが一致していません (%s): %s [%s]" :
            "Color quantize type does not match (%s): %s [%s]"),
            condition.asChar(), shapeName.asChar(), colorSetName.asChar()));

        // エラー原因の FV を特定
        MIntArray errorFaceIds,errorVtxIds;

        const int numPolygons = meshFn.numPolygons();
        for (int f = 0 ; f < numPolygons ; ++f)
        {
            MIntArray vtxIdsInFace;
            meshFn.getPolygonVertices(f,vtxIdsInFace);
            for (int v = 0; v < static_cast<int>(vtxIdsInFace.length()); ++v)
            {
                int colorIndex;
                meshFn.getColorIndex(f,v,colorIndex,&colorSetName);

                if ( FindInMArray(errorColorIds,colorIndex) )
                {
                    errorFaceIds.append(f);
                    errorVtxIds.append(vtxIdsInFace[v]);

                    if (errorFaceIds.length() <= LOD_VTX_PRINT_COUNT_MAX)
                    {
                        PrintF("      %s.vtxFace[%d][%d]", shapeName.asChar(), v, f);
                    }
                    else if (errorFaceIds.length() == LOD_VTX_PRINT_COUNT_MAX + 1)
                    {
                        PrintF("      ...");
                    }
                }
            }
        }

        // エラー選択
        MObject vtxFaceComp;
        GetMeshVertexFaceComponentFromIndices(vtxFaceComp,errorVtxIds,errorFaceIds);
        errorSelection.add(pShape->dag,vtxFaceComp);

        return false;
    }
    else
    {
        return true;
    }
} // NOLINT(impl/function_size)

//! モデル間の量子化タイプチェック
bool CheckQuantizeType(MSelectionList& errorCause, const Shape* pBaseShape, const Shape* pLodShape)
{
    // memo)
    // 中間ファイルオプティマイザーの LOD 結合時に量子化タイプが再設定されるため、致命的なエラーデータが
    // 発生する可能性はありません。再設定の際には値域条件は全 LOD 頂点が考慮されますが、
    // ビット幅、誤差条件はベースモデル頂点のみが考慮されます。つまり、LOD モデルによって
    // ビット幅、誤差条件が変わり負荷が増える事はありません。
    // 以上の前提条件から、LOD モデルによって値域条件が変わるケースのみをチェックし、
    // 警告を行う方針とします。

    bool isOK = true;

    // ・座標 : 行うチェックはありません。

    // ・法線
    //   1.0 を超えるか
    //QuantizeType baseNormalQuantizeType = GetNormalQuantizeType(pBaseShape);
    //if (!CheckNormalQuantizeTypeCompatibility(errorCause,baseNormalQuantizeType,pLodShape))
    //{
    //  isOK = false;
    //}

    // ・接線、従法線
    //   1.0 を超えるか
    //QuantizeType baseTanQuantizeType = GetTanBinQuantizeType(pBaseShape,true);
    //if (!CheckTanBinQuantizeTypeCompatibility(errorCause,baseTanQuantizeType,pLodShape,true))
    //{
    //  isOK = false;
    //}

    //QuantizeType baseBinQuantizeType = GetTanBinQuantizeType(pBaseShape,false);
    //if (!CheckTanBinQuantizeTypeCompatibility(errorCause,baseBinQuantizeType,pLodShape,false))
    //{
    //  isOK = false;
    //}

    // ・テクスチャ座標 (UV)
    //   以下のチェックを行います
    //    - ベースが 正の数のみなら LOD も
    //    - ベースが 1.0 を超えない場合は LOD も
    // 事前に有効な UV セット構成が同じなのは確認済みなのでここではチェックを行いません。
    for (unsigned i = 0; i < pBaseShape->uvSets.length(); ++i)
    {
        QuantizeType baseUVQuantizeType = GetUVQuantizeType(pBaseShape, i);
        if (!CheckUVQuantizeTypeCompatibility(errorCause, baseUVQuantizeType, pLodShape, i))
        {
            isOK = false;
        }
    }

    // ・カラー
    //   ベースが正の数のみなら LOD も正のみかチェックします。
    // 事前に有効な カラーセット構成が同じなのは確認済みなのでここではチェックを行いません。
    for (unsigned i = 0; i < pBaseShape->colorSets.size(); ++i)
    {
        QuantizeType baseColorQuantizeType = GetColorQuantizeType(pBaseShape, i);
        if (!CheckColorQuantizeTypeCompatibility(errorCause, baseColorQuantizeType, pLodShape, i))
        {
            isOK = false;
        }
    }

    // ・ウェイトインデックス : 行うチェックはありません。
    // ・ウェイト : 行うチェックはありません。

    return isOK;
}

//=============================================================================
// lod ネームスペースを終了します。
//=============================================================================
} // namespace lod

