﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <iostream>
#include <cstdarg>

#include <maya/MStatus.h>
#include <maya/MObject.h>
#include <maya/MPlug.h>
#include <maya/MDagPath.h>
#include <maya/MIntArray.h>
#include <maya/MObjectArray.h>
#include <maya/MFnDependencyNode.h>

//=============================================================================
// lod ネームスペースを開始します。
//=============================================================================
namespace lod {

const int LOD_VTX_PRINT_COUNT_MAX = 10; //!< エラー・警告表示する頂点の最大数です。

MString LodGetEnvVar(const char* name);
bool IsLodJapaneseUi();
void PrintF(const char* format,...);
MString MStringF(const char* format,...);

#ifdef _DEBUG
#define TRACE(msg,...) PrintF(msg, __VA_ARGS__)
#else
#define TRACE(msg,...)
#endif

bool CheckStatusOK(const MStatus& stat,const char* format,...);

MDagPath GetDagPathByName(const MString& name);
MString GetNodeName(const MDagPath& dag);
MString GetNodeName(const MObject& obj);

MObject GetMaterialFromShadingEngine(const MObject& oSG);
MObject SearchUpstreamFileNode(const MPlug& plug);
void SearchUpstreamFileNodes(MObjectArray& oFiles,MPlug plug);
MObject GetRelativeSkinCluster(const MObject& oMesh);

MStatus GetIndicesFromMeshComponent(MIntArray& indices,const MObject& comp);
MStatus GetMeshComponentFromIndices(MObject& comp,const MIntArray& indices,MFn::Type meshCompType);

MStatus GetMeshVertexFaceComponentFromIndices(MObject& comp,const MIntArray& vtxIds,const MIntArray& faceIds);

//! 配列を値の小さい順にソートします。
void SortArrayAscending(MIntArray& array);

template<typename MArray,typename Type>
inline bool FindInMArray(const MArray& arr,const Type& item)
{
    const unsigned length = arr.length();
    for(unsigned int i = 0; i < length; ++i)
    {
        if( arr[i] == item) return true;
    }

    return false;
}

//! @brief Nintendo Maya プラグイン用の UV セット名なら true を返します。
//!
inline bool IsUvSetNameForNintendo(const MString& uvSetName)
{
    return (uvSetName.indexW("nw4f_uv" ) == 0 ||
            uvSetName.indexW("nw4f_fix") == 0);
}

//-----------------------------------------------------------------------------
//!
class MayaNode
{
private:
    MFnDependencyNode m_nodeFn;

public:
    explicit MayaNode(MObject obj)
    : m_nodeFn(obj)
    {
    }
    virtual ~MayaNode(){}

    MString GetName() const{ return m_nodeFn.name(); }

    //! アトリビュート値取得
    template <typename T>
    bool GetAttr(const char* attrName,T& result) const;

    //! bool 値取得
    bool GetBoolAttr(const char* attrName) const;

    //! int 値取得
    int GetIntAttr(const char* attrName) const;

};

template <typename T>
bool MayaNode::GetAttr(const char* attrName,T& result) const
{
    MStatus stat;
    MPlug plug = m_nodeFn.findPlug(attrName,&stat);
    if (!CheckStatusOK(stat,"MFnDependencyNode::findPlug")) return false;

    stat = plug.getValue(result);
    if (!CheckStatusOK(stat,"MPlug::getValue")) return false;

    return true;
}

inline bool MayaNode::GetBoolAttr(const char* attrName) const
{
    bool ret;
    if( this->GetAttr(attrName,ret) )
    {
        return ret;
    }
    else
    {
        return false;
    }
}

inline int MayaNode::GetIntAttr(const char* attrName) const
{
    int ret;
    if( this->GetAttr(attrName,ret) )
    {
        return ret;
    }
    else
    {
        return false;
    }
}

//=============================================================================
// lod ネームスペースを終了します。
//=============================================================================
} // namespace lod

