﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "LodCommon.h"

#include <maya/MGlobal.h>
#include <maya/MObject.h>
#include <maya/MDagPathArray.h>
#include <maya/MDoubleArray.h>
#include <maya/MSelectionList.h>
#include <maya/MMatrix.h>
#include <maya/MFnSingleIndexedComponent.h>
#include <maya/MFnDoubleIndexedComponent.h>
#include <maya/MFnDagNode.h>
#include <maya/MFnMesh.h>
#include <maya/MFnSkinCluster.h>
#include <maya/MItDependencyGraph.h>
#include <maya/MItDag.h>
#include <maya/MItMeshPolygon.h>

//=============================================================================
// lod ネームスペースを開始します。
//=============================================================================
namespace lod {

MString LodGetEnvVar(const char* name)
{
    MString ret;
    char* pValue;
    size_t length;
    if (_dupenv_s(&pValue, &length, name) == 0)
    {
        // 環境変数が見つからない場合は、pValue が NULL に
        // length が 0、戻り値が 0（成功）となります。
        if (pValue != NULL)
        {
            ret = pValue;
            free(pValue);
        }
    }
    return ret;
}

//! 配列を値の小さい順にソートします。
void SortArrayAscending(MIntArray& array)
{
    const int count = array.length();
    for (int iValue = 0; iValue < count - 1; ++iValue)
    {
        for (int iOther = iValue + 1; iOther < count; ++iOther)
        {
            if (array[iValue] > array[iOther])
            {
                const int value = array[iValue];
                array[iValue] = array[iOther];
                array[iOther] = value;
            }
        }
    }
}

MStatus GetIndicesFromMeshComponent(MIntArray& indices,const MObject& comp)
{
    MStatus stat;
    MFnSingleIndexedComponent compFn(comp,&stat);
    if( !CheckStatusOK(stat,"MFnSingleIndexedComponent::MFnSingleIndexedComponent()") ) return stat;

    compFn.getElements(indices);

    return MS::kSuccess;
}

MStatus GetMeshComponentFromIndices(MObject& comp,const MIntArray& indices,MFn::Type meshCompType)
{
    MStatus stat;

    MFnSingleIndexedComponent compFn;
    compFn.create(meshCompType,&stat);
    if( !CheckStatusOK(stat,"MFnSingleIndexedComponent::create()") ) return stat;

    compFn.addElements( const_cast<MIntArray&>(indices) );
    comp = compFn.object(&stat);
    if( !CheckStatusOK(stat,"MFnSingleIndexedComponent::object()") ) return stat;

    return MS::kSuccess;
}

MStatus GetMeshVertexFaceComponentFromIndices(MObject& comp,const MIntArray& vtxIds,const MIntArray& faceIds)
{
    MStatus stat;

    MFnDoubleIndexedComponent compFn;
    compFn.create(MFn::kMeshVtxFaceComponent,&stat);
    if( !CheckStatusOK(stat,"MFnDoubleIndexedComponent::create()") ) return stat;

    compFn.addElements(vtxIds,faceIds);
    comp = compFn.object(&stat);
    if( !CheckStatusOK(stat,"MFnDoubleIndexedComponent::object()") ) return stat;

    return MS::kSuccess;
}



MObject GetMaterialFromShadingEngine(const MObject& oSG)
{
    MFnDependencyNode nodeFn(oSG);
    MPlug shaderPlug = nodeFn.findPlug("surfaceShader");

    MItDependencyGraph dgIter(shaderPlug,
                                MFn::kInvalid,
                                MItDependencyGraph::kUpstream,
                                MItDependencyGraph::kDepthFirst,
                                MItDependencyGraph::kNodeLevel);

    // 最初は SG 自身なのでスキップ
    dgIter.next();
    return dgIter.currentItem();
}

MObject GetRelativeSkinCluster(const MObject& oMesh)
{
    if (!oMesh.hasFn(MFn::kMesh)) return MObject::kNullObj;

    MStatus stat;
    MFnDependencyNode nodeFn(oMesh);
    MPlug inMeshPlug = nodeFn.findPlug("inMesh",&stat);
    if (!CheckStatusOK(stat,"findPlug")) return MObject::kNullObj;

    MItDependencyGraph dgIt(inMeshPlug,MFn::kSkinClusterFilter,
                            MItDependencyGraph::kUpstream,
                            MItDependencyGraph::kDepthFirst,
                            MItDependencyGraph::kNodeLevel,&stat);
    // 無い場合
    if (stat.error())
    {
        return MObject::kNullObj;
    }
    else
    {
        return dgIt.currentItem();
    }
}


MDagPath GetDagPathByName(const MString& name)
{
    MSelectionList selList;
    MGlobal::getSelectionListByName(name,selList);

    if (selList.length() > 0)
    {
        MDagPath dag;
        selList.getDagPath(0,dag);
        return dag;
    }
    else
    {
        return MDagPath();
    }
}

MString GetNodeName(const MDagPath& dag)
{
    MStringArray arr;
    dag.partialPathName().split('|',arr);

    return arr[arr.length() - 1];
}

MString GetNodeName(const MObject& obj)
{
    MFnDependencyNode nodeFn(obj);
    return nodeFn.name();
}



#define _FORMAT(msg,fmt) \
{ \
    va_list argptr; \
    va_start(argptr,fmt); \
    vsprintf_s(msg, sizeof(msg), fmt, argptr); \
    va_end(argptr); \
}

bool CheckStatusOK(const MStatus& stat, const char* format, ...)
{
    if(stat.error())
    {
        char msg[4096];
        _FORMAT(msg,format);

        stat.perror(msg);
        return false;
    }
    else{
        return true;
    }
}

bool IsLodJapaneseUi()
{
    if (LodGetEnvVar("NINTENDO_MAYA_FORCE_JAPANESE_UI") == "1")
    {
        return true;
    }

    MString lang;
    MGlobal::executeCommand("about -uil", lang);
    return (lang == "ja_JP");
}

void PrintF(const char* format, ...)
{
    char msg[4096];
    _FORMAT(msg, format);

    //std::cout << msg;
    //std::cout.flush();
    //std::cerr << msg << std::endl;
    MGlobal::displayInfo(msg);
}

MString MStringF(const char* format, ...)
{
    char msg[4096];
    _FORMAT(msg, format);

    return MString(msg);
}

//=============================================================================
// lod ネームスペースを終了します。
//=============================================================================
} // namespace lod

