﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//=============================================================================
// include
//=============================================================================
#include "CheckLod.h"

#include <cfloat>
#include <cassert>
#include <cstdarg>
#include <vector>

#define MNoVersionString
#define MNoPluginEntry
#include <maya/MFnPlugin.h>

#include <maya/MGlobal.h>
#include <maya/MObject.h>
#include <maya/MDagPathArray.h>
#include <maya/MDoubleArray.h>
#include <maya/MSelectionList.h>
#include <maya/MMatrix.h>
#include <maya/MFnSingleIndexedComponent.h>
#include <maya/MFnDagNode.h>
#include <maya/MFnMesh.h>
#include <maya/MFnSkinCluster.h>
#include <maya/MItDependencyGraph.h>
#include <maya/MItDag.h>
#include <maya/MItMeshPolygon.h>

#include "LodCommon.h"
#include "LodModel.h"
#include "LodQuantize.h"

#include "LodCommon.cpp"
#include "LodModel.cpp"
#include "LodQuantize.cpp"

using namespace lod;

#undef min
#undef max

//-----------------------------------------------------------------------------
//! @brief CheckLod コマンドを実行します。
//!
//! @param[in] args 引数リストです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
MStatus NintendoCheckLodCmd::doIt(const MArgList& args)
{
    MStatus status;

    bool isValid = true;

    //-----------------------------------------------------------------------------
    // チェック対象の階層ルートを引数から取得します。
    int index(0);
    const MString& baseRoot = args.asString(index++);
    const MString& lodRoot = args.asString(index++);
    const MString& options = args.asString(index++);

    this->ParseOptions(options);

    MDagPath dagBaseRoot = GetDagPathByName(baseRoot);
    MDagPath dagLodRoot = GetDagPathByName(lodRoot);

    if (!dagBaseRoot.isValid())
    {
        MGlobal::displayError("Invalid base root: " + baseRoot); // 通常は発生しない
        return MS::kFailure;
    }

    if (!dagLodRoot.isValid())
    {
        MGlobal::displayError("Invalid lod root: " + lodRoot); // 通常は発生しない
        return MS::kFailure;
    }

    //PrintF("Checking %s ...",lodRoot.asChar());

    m_pBaseModel = new lod::Model();
    m_pLodModel = new lod::Model();

    m_pBaseModel->SetRootNode(dagBaseRoot);
    m_pLodModel->SetRootNode(dagLodRoot);

    //-----------------------------------------------------------------------------
    if (!this->CheckBones() ) isValid = false;
    if (!this->CheckMaterials() ) isValid = false;
    if (!this->CheckShapes() ) isValid = false;

    delete m_pBaseModel;
    delete m_pLodModel;

    //-----------------------------------------------------------------------------
    // 返り値を設定します。
    if (isValid)
    {
        //PrintF("[OK] : %s", lodRoot.asChar());
        this->setResult((m_WarningCount == 0) ? 1 : 2);
    }
    else
    {
        PrintF("[NG] : %s", lodRoot.asChar());
        this->setResult(0);
        if (!m_errorSelection.isEmpty())
        {
            MGlobal::setActiveSelectionList(m_errorSelection, MGlobal::kAddToList);
        }
    }

    return MS::kSuccess;
}

//! 「key1=value1;key2=value2;...」形式のオプション文字列を解析します。
bool NintendoCheckLodCmd::ParseOptions(const MString& options)
{
    MStringArray params;
    options.split(';',params);

    for (unsigned i = 0 ; i < params.length() ; ++i)
    {
        MStringArray keyValue;
        params[i].split('=',keyValue);

        if (keyValue.length() == 2)
        {
            const MString& key = keyValue[0];
            const MString& value = keyValue[1];

            if (key == "check_quantize_type")
            {
                m_treatQuantizeTypeMismatchAsError = (value != "false");
            }


        }

    }

    return true;
}



//! ボーンのチェック
bool NintendoCheckLodCmd::CheckBones()
{
    const bool isJp = IsLodJapaneseUi();

    // ボーン数が不一致の場合は不正です。
    const int numBaseBones = m_pBaseModel->GetBoneCount();
    const int numLodBones = m_pLodModel->GetBoneCount();

    if ( numBaseBones != numLodBones)
    {
        PrintF((isJp ? "ボーン数が一致していません: %s(%d) %s(%d)" :
            "Bone count does not match: %s(%d) %s(%d)"),
                m_pBaseModel->GetName().asChar(), numBaseBones,
                m_pLodModel->GetName().asChar(), numLodBones);

        // ベースに使用されていて LOD で使用されていないものを調査
        for (int i = 0 ; i < numBaseBones ; ++i)
        {
            const lod::Bone *pBaseBone = m_pBaseModel->GetBoneAt(i);
            const lod::Bone *pLodBone = m_pLodModel->GetBoneByDag(pBaseBone->dag);

            if (!pLodBone)
            {
                PrintF((isJp ? "    '%s' はベースモデルでのみ使用されています。" :
                    "    '%s' is used in only base model."),
                    pBaseBone->GetName().asChar());
            }
        }

        // LOD に使用されていてベースで使用されていないものを調査
        for (int i = 0 ; i < numLodBones ; ++i)
        {
            const lod::Bone *pLodBone = m_pLodModel->GetBoneAt(i);
            const lod::Bone *pBaseBone = m_pBaseModel->GetBoneByDag(pLodBone->dag);

            if (!pBaseBone)
            {
                PrintF((isJp ? "    '%s' は LOD モデルでのみ使用されています。" :
                    "    '%s' is used in only lod model."),
                    pLodBone->GetName().asChar());
            }
        }

        return false;
    }

    // ボーンリストの各インデックス要素の DAG パスが異なる場合は不正です。
    // 中間ファイルでは名前の一致のみですが、Mayaでは同一の骨を使う前提のため
    // DAG パスの比較を行います。
    bool isValid = true;

    for (int i = 0 ; i < numBaseBones ; ++i)
    {
        const lod::Bone *pBaseBone = m_pBaseModel->GetBoneAt(i);
        const lod::Bone *pLodBone = m_pLodModel->GetBoneAt(i);
        if (!pLodBone)
        {
            isValid = false;
            continue;
        }

        if ( !(pBaseBone->dag == pLodBone->dag) )
        {
            PrintF((isJp ? "ボーンの DAG パスが異なります [%d]: %s <-> %s" :
                "Different bone [%d]: %s <-> %s"),
                i, pBaseBone->GetName().asChar(), pLodBone->GetName().asChar());
            isValid = false;
        }

        // ベースの骨がスムースでない場合、LODでスムースを使うのは不正です。
        // 逆に、ベースの骨がスムースで LOD がリジッドなのは問題ありません。
        if (!pBaseBone->useSmoothSkin && pLodBone->useSmoothSkin)
        {
            PrintF((isJp ? "ボーンが LOD モデルではスムーススキニングに使用されていますが、ベースモデルでは使用されていません: %s" :
                "Lod bone is used as smooth, but base one isn't: %s"),
                pLodBone->GetName().asChar());

            // 原因となるシェイプを特定します。
            // ボーンのスムース状態は１つでも関連シェイプにスムースがあると引きずられてしまうため、
            // 単一シェイプ同士での比較では不十分で、ここまとめてシェイプをチェックする必要があります。
            const int numLodShapes = m_pLodModel->GetShapeCount();
            for (int s = 0 ; s < numLodShapes ; ++s)
            {
                const lod::Shape *pShape = m_pLodModel->GetShapeAt(s);

            //  TRACE("bone(%s) : shape(%s) isUsed(%d) isSmoothSkin(%d)\n",
            //              pLodBone->GetName().asChar(),
            //              pShape->dag.partialPathName().asChar(),
            //              pShape->IsUsedBone(pLodBone),pShape->isSmoothSkin
            //              );

                if (pShape->IsUsedBone(pLodBone) && pShape->skinMode == lod::Shape::cSmoothSkin)
                {
                    PrintF("    mesh(%s) material(%s)", pShape->dag.partialPathName().asChar(), MayaNode(pShape->material).GetName().asChar());
                    for (unsigned v = 0 ; v < pShape->smoothVtxIds.length() ; ++v)
                    {
                        if (v >= LOD_VTX_PRINT_COUNT_MAX)
                        {
                            PrintF("      ...");
                            break;
                        }
                        PrintF("      %s.vtx[%d]", pShape->dag.partialPathName().asChar(), pShape->smoothVtxIds[v]);
                    }

                    MObject vtxComp;
                    GetMeshComponentFromIndices(vtxComp,pShape->smoothVtxIds,MFn::kMeshVertComponent);
                    m_errorSelection.add(pShape->dag,vtxComp,true);
                }

            }

            isValid = false;
        }
    }

    return isValid;
}

bool NintendoCheckLodCmd::CheckMaterials()
{
    const bool isJp = IsLodJapaneseUi();
    bool isValid = true;

    MObjectArray baseMaterials,lodMaterials;
    int numBaseMaterials = m_pBaseModel->GetMaterials(baseMaterials);
    int numLodMaterials = m_pLodModel->GetMaterials(lodMaterials);

    // ベース側マテリアルが LOD で使われているか？
    for (int i = 0 ; i < numBaseMaterials ; ++i)
    {
        if (!FindInMArray(lodMaterials,baseMaterials[i]))
        {
            PrintF((isJp ? "マテリアル '%s' はベースモデルでのみ使用されています。" :
                "Material '%s' is used in only base model."),
                MayaNode(baseMaterials[i]).GetName().asChar());
            isValid = false;
        }
    }

    // LOD 側マテリアルがベースで使われているか？
    for (int i = 0 ; i < numLodMaterials ; ++i)
    {
        if (!FindInMArray(baseMaterials,lodMaterials[i]))
        {
            PrintF((isJp ? "マテリアル '%s' は LOD モデルでのみ使用されています。" :
                "Material '%s' is used in only lod model"),
                MayaNode(lodMaterials[i]).GetName().asChar());
            isValid = false;
        }
    }

    return isValid;
}

namespace {

//-----------------------------------------------------------------------------
//! @brief モデルのプリント用のシェイプ名配列を取得します。
//!
//! @param[out] shapeNames プリント用のシェイプ名配列を格納します。
//! @param[in] pModel モデルへのポインタです。
//-----------------------------------------------------------------------------
void GetShapeNamesForPrint(MStringArray& shapeNames, const lod::Model* pModel)
{
    shapeNames.clear();
    const int shapeCount = pModel->GetShapeCount();
    for (int iShape = 0; iShape < shapeCount; ++iShape)
    {
        const lod::Shape* pShape = pModel->GetShapeAt(iShape);
        const MString xformName = GetNodeName(pShape->dag);
        const MString matName = MFnDependencyNode(pShape->material).name();
        const MString shapeName = xformName + "(" + matName + ")";
        shapeNames.append(shapeName);
    }
}

//-----------------------------------------------------------------------------
//! @brief 対象のシェイプ名配列にのみ存在するシェイプ名を表示します。
//!
//! @param[in] shapeNames 対象のシェイプ名配列です。
//! @param[in] otherShapeNames 比較するシェイプ名配列です。
//! @param[in] modelName モデル名です。
//-----------------------------------------------------------------------------
void PrintLackedShapeNames(
    const MStringArray& shapeNames,
    const MStringArray& otherShapeNames,
    const MString& modelName
)
{
    MString onlyStr;
    for (int iShape = 0; iShape < static_cast<int>(shapeNames.length()); ++iShape)
    {
        const MString& shapeName = shapeNames[iShape];
        if (!FindInMArray(otherShapeNames, shapeName))
        {
            if (onlyStr != "")
            {
                onlyStr += ", ";
            }
            onlyStr += shapeName;
        }
    }
    if (onlyStr != "")
    {
        MGlobal::displayInfo("  " + modelName + " only: " + onlyStr);
    }
}

//-----------------------------------------------------------------------------
//! @brief ベースモデルと LOD モデルで、一方にのみ存在するシェイプの情報を表示します。
//!
//! @param[in] pBaseModel ベースモデルへのポインタです。
//! @param[in] pLodModel LOD モデルへのポインタです。
//-----------------------------------------------------------------------------
void PrintLackedShapes(const lod::Model* pBaseModel, const lod::Model* pLodModel)
{
    MStringArray baseShapeNames;
    GetShapeNamesForPrint(baseShapeNames, pBaseModel);
    MStringArray lodShapeNames;
    GetShapeNamesForPrint(lodShapeNames, pLodModel);

    PrintLackedShapeNames(baseShapeNames, lodShapeNames , pBaseModel->GetName());
    PrintLackedShapeNames(lodShapeNames , baseShapeNames, pLodModel->GetName());
}

} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief シェイプ情報群の一致をチェックします。
//-----------------------------------------------------------------------------
bool NintendoCheckLodCmd::CheckShapes()
{
    const bool isJp = IsLodJapaneseUi();

    // 同じマテリアルが割り当てられたフェース群単位でのチェックです。
    // シェイプ圧縮を利用すると中間ファイル上の <shape> がまとめられる可能性があり、
    // 厳密には圧縮後と比べチェック条件が厳しくなりますが、
    // 運用ルールシンプル化のためにこの単位での一致チェックとします。
    // なお、ベースと LOD で同一マテリアル、ボーンを使用する前提のため、
    // 中間ファイルオプティマイザーのシェイプ圧縮状況は両者で変わらず、
    // このチェックを通る場合は中間ファイルの条件を満たす事が保証されます。

    //-----------------------------------------------------------------------------
    // シェイプの数が一致しない場合は不正です。
    const int baseShapeCount = m_pBaseModel->GetShapeCount();
    const int lodShapeCount = m_pLodModel->GetShapeCount();
    if (baseShapeCount != lodShapeCount)
    {
        PrintF((isJp ? "シェイプ数が一致していません: %s(%d) %s(%d)" :
            "Shape count does not match: %s(%d) %s(%d)"),
                    m_pBaseModel->GetName().asChar(), baseShapeCount,
                    m_pLodModel->GetName().asChar(), lodShapeCount);
        PrintLackedShapes(m_pBaseModel, m_pLodModel);
        return false;
    }

    //-----------------------------------------------------------------------------
    // スキニングのウェイト値を分配可能な最大ノード数をコンフィグファイルから取得します。
    int maxVertexSkinningCount = 8;
    MString maxVertexSkinningCountString;
    const MString configCmd = "NintendoGetConfigFileKeyword \"max_vertex_skinning_count\"";
    if (MGlobal::executeCommand(configCmd, maxVertexSkinningCountString) &&
        maxVertexSkinningCountString.length() != 0)
    {
        maxVertexSkinningCount = atoi(maxVertexSkinningCountString.asChar());
    }

    //-----------------------------------------------------------------------------
    // 各シェイプ情報を比較します。
    bool isValid = true;
    for (int iShape = 0; iShape < baseShapeCount; ++iShape)
    {
        const lod::Shape* pBaseShape = m_pBaseModel->GetShapeAt(iShape);
        const lod::Shape* pLodShape = m_pLodModel->GetShapeAt(iShape);
        if (!CheckShape(m_pLodModel->GetName(), pBaseShape, pLodShape, iShape, maxVertexSkinningCount))
        {
            isValid = false;
        }
    }
    return isValid;
}

namespace {

//-----------------------------------------------------------------------------
//! @brief UV セットのファミリー名を返します。
//-----------------------------------------------------------------------------
MString GetUvSetFamilyName(const MString& uvSetName)
{
    const int iParenthesis = uvSetName.index('(');
    return (iParenthesis > 0) ?
        uvSetName.substring(0, iParenthesis - 1) : uvSetName;
}

//-----------------------------------------------------------------------------
//! @brief スキニングの最大インフルエンス数が 5 以上かどうか表示します。
//-----------------------------------------------------------------------------
void PrintIsInfluenceOver4(
    const MString& modelName,
    const MString& xformName,
    const bool isOver4,
    const bool isJp
)
{
    const char* msg = (isJp) ?
        ((isOver4) ? "5 以上"     : "4 以下"    ) :
        ((isOver4) ? "Over 4    " : "Not over 4");
    PrintF("    %s: %s(%s)", msg, modelName.asChar(), xformName.asChar());
}

} // unnamed namespace

//-----------------------------------------------------------------------------
//! @brief 2 つのシェイプ情報の一致をチェックします。
//-----------------------------------------------------------------------------
bool NintendoCheckLodCmd::CheckShape(
    const MString& lodModelName,
    const lod::Shape* pBaseShape,
    const lod::Shape* pLodShape,
    const int shapeIndex,
    const int maxVertexSkinningCount
)
{
    const bool isJp = IsLodJapaneseUi();
    if (!pBaseShape || !pLodShape)
    {
        PrintF((isJp ? "不明なシェイプのエラーです" : "Unknown shape error")); // 通常は発生しない
        return false;
    }

    //-----------------------------------------------------------------------------
    // 名前の一致チェック。DAG パスでなくノード名のみの比較です。
    const MString baseXformName = GetNodeName(pBaseShape->dag);
    const MString lodXformName = GetNodeName(pLodShape->dag);
    if (baseXformName != lodXformName)
    {
        PrintF((isJp ? "ノード名が一致していません: [%d] Base(%s) <-> %s(%s)" :
            "Node name does not match: [%d] Base(%s) <-> %s(%s)"),
            shapeIndex,
            baseXformName.asChar(), lodModelName.asChar(), lodXformName.asChar());
        return false;
    }

    //-----------------------------------------------------------------------------
    // マテリアルの一致チェック。
    if (pBaseShape->material != pLodShape->material)
    {
        PrintF((isJp ? "マテリアルノードが一致していません: [%d] %s(%s) <-> %s(%s)" :
            "Material node does not match: [%d] %s(%s) <-> %s(%s)"),
            shapeIndex,
            baseXformName.asChar(), GetNodeName(pBaseShape->material).asChar(),
            lodXformName.asChar(), GetNodeName(pLodShape->material).asChar());
        return false;
    }

    //-----------------------------------------------------------------------------
    // UV セット構成の一致チェック
    if (pBaseShape->uvSets.length() == pLodShape->uvSets.length())
    {
        for (unsigned i = 0; i < pBaseShape->uvSets.length(); ++i)
        {
            const MString baseUvSetName = GetUvSetFamilyName(pBaseShape->uvSets[i]);
            const MString lodUvSetName  = GetUvSetFamilyName(pLodShape->uvSets[i]);
            if (IsUvSetNameForNintendo(baseUvSetName) ||
                IsUvSetNameForNintendo(lodUvSetName))
            {
                if (baseUvSetName != lodUvSetName)
                {
                    PrintF((isJp ? "UV セットが一致していません: %s(%s) <-> %s(%s)" :
                        "UV set does not match: %s(%s) <-> %s(%s)"),
                        pBaseShape->GetMeshName().asChar(), pBaseShape->uvSets[i].asChar(),
                        pLodShape->GetMeshName().asChar(), pLodShape->uvSets[i].asChar());
                    return false;
                }
            }
        }
    }
    else
    {
        PrintF((isJp ? "UV セット数が一致していません: %s <-> %s" :
            "The number of uv set does not match: %s <-> %s"),
            pBaseShape->GetMeshName().asChar(),
            pLodShape->GetMeshName().asChar());
        return false;
    }

    //-----------------------------------------------------------------------------
    // カラーセット構成の一致チェック
    bool colorSetError = false;
    if ( pBaseShape->colorSets.size() == pLodShape->colorSets.size() )
    {
        for (size_t i = 0 ; i < pBaseShape->colorSets.size() ; ++i)
        {
            const lod::Shape::ColorSet& baseColorSet = pBaseShape->colorSets[i];
            const lod::Shape::ColorSet& lodColorSet = pLodShape->colorSets[i];

            if (baseColorSet.compCount != lodColorSet.compCount)
            {
                colorSetError = true;
                break;
            }

            if (baseColorSet.type != lodColorSet.type)
            {
                colorSetError = true;
                break;
            }

        }
    }
    else
    {
        colorSetError = true;
    }

    if (colorSetError)
    {
        PrintF((isJp ? "カラーセットが一致していません: %s <-> %s" :
            "Color set does not match: %s <-> %s"),
            pBaseShape->GetMeshName().asChar(), pLodShape->GetMeshName().asChar());
        return false;
    }

    //-----------------------------------------------------------------------------
    // ベースシェイプで使用していないボーンは LOD で使用してはいけません。
    bool boneError = false;
    for (size_t i = 0 ; i < pLodShape->bones.size() ; ++i)
    {
        const lod::Bone *pBone = pLodShape->bones[i];
        if (!pBaseShape->IsUsedBone(pBone))
        {
            PrintF((isJp ? "ボーン '%s' は LOD モデルでのみ使用されています。" :
                "Bone '%s' is used in only lod model."),
                pBone->GetName().asChar());

            // エラー原因の頂点を特定
            std::map<const lod::Bone*,MIntArray>::const_iterator it = pLodShape->boneToVtxIds.find(pBone);
            if (it != pLodShape->boneToVtxIds.end())
            {
                const MIntArray& vtxIds = it->second;
                for (unsigned v = 0 ; v < vtxIds.length() ; ++v)
                {
                    if (v >= LOD_VTX_PRINT_COUNT_MAX)
                    {
                        PrintF("    ...");
                        break;
                    }
                    PrintF("    %s.vtx[%d] (material=%s)",
                                        pLodShape->GetMeshName().asChar(),vtxIds[v],
                                        pLodShape->GetMaterialName().asChar());
                }

                // エラー選択リストに頂点を登録
                MObject vtxComp;
                GetMeshComponentFromIndices(vtxComp,it->second,MFn::kMeshVertComponent);
                m_errorSelection.add(pLodShape->dag,vtxComp,true);
            }

            boneError = true;
        }

    }

    if (boneError) return false;


//  TRACE("smooth base(%d) lod(%d)\n",pBaseShape->isSmoothSkin,pLodShape->isSmoothSkin);

    //-----------------------------------------------------------------------------
    // スキンタイプのチェック。
    if (pBaseShape->skinMode != lod::Shape::cSmoothSkin)
    {
        // ベースがスムースでない場合、LOD はスムースではいけません。
        if (pLodShape->skinMode == lod::Shape::cSmoothSkin)
        {
            const MString meshName = pLodShape->dag.partialPathName();

            PrintF((isJp ? "LOD モデルのスキニングタイプがベースモデルと異なります: mesh(%s) material(%s)" :
                "LOD model uses different skinning type than base model: mesh(%s) material(%s)"),
                meshName.asChar(), MayaNode(pLodShape->material).GetName().asChar());
            for (unsigned v = 0; v < pLodShape->smoothVtxIds.length(); ++v)
            {
                if (v >= LOD_VTX_PRINT_COUNT_MAX)
                {
                    PrintF("      ...");
                    break;
                }
                PrintF("      %s.vtx[%d]", meshName.asChar(), pLodShape->smoothVtxIds[v]);
            }

            MObject vtxComp;
            GetMeshComponentFromIndices(vtxComp, pLodShape->smoothVtxIds, MFn::kMeshVertComponent);
            m_errorSelection.add(pLodShape->dag,vtxComp,true);
        }
    }

    //-----------------------------------------------------------------------------
    // スキニングの最大インフルエンス数が 4 以下のレベルと、5 以上のレベルが
    // 混在するかどうかチェックします。
    const bool isBaseInfluenceOver4 =
        (std::min(pBaseShape->m_VertexSkinningCount, maxVertexSkinningCount) > 4);
    const bool isLodInfluenceOver4 =
        (std::min(pLodShape->m_VertexSkinningCount , maxVertexSkinningCount) > 4);
    if (isBaseInfluenceOver4 != isLodInfluenceOver4)
    {
        PrintF((isJp ? "スキニングの最大インフルエンス数が一致していません: [%d] Base(%s) <-> %s(%s)" :
            "The maximum number of weighted influences does not match: [%d] Base(%s) <-> %s(%s)"),
            shapeIndex,
            baseXformName.asChar(), lodModelName.asChar(), lodXformName.asChar());
        PrintIsInfluenceOver4("Base"      , baseXformName, isBaseInfluenceOver4, isJp);
        PrintIsInfluenceOver4(lodModelName, lodXformName , isLodInfluenceOver4 , isJp);
        return false;
    }

    //-----------------------------------------------------------------------------
    // 量子化タイプをチェックします。
    MSelectionList quantizeTypeErrorSelection;
    if (!CheckQuantizeType(quantizeTypeErrorSelection, pBaseShape,pLodShape))
    {
        // 量子化タイプの相違をエラーとみなす場合はエラー選択に追加して false を返します。
        if (m_treatQuantizeTypeMismatchAsError)
        {
            m_errorSelection.merge(quantizeTypeErrorSelection);
            return false;
        }
        else
        {
            ++m_WarningCount;
            return true;
        }
    }

    return true;
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
//! @brief コマンドを登録します。
//!
//! @param[in,out] plugin プラグインオブジェクトです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
MStatus NintendoCheckLodCmd::Register(MFnPlugin& plugin)
{
    MStatus status = plugin.registerCommand("NintendoCheckLodCmd", NintendoCheckLodCmd::creator);
    if (status.error())
    {
        status.perror("Register NintendoCheckLodCmd");
    }
    return status;
}

//-----------------------------------------------------------------------------
//! @brief コマンドを登録解除します。
//!
//! @param[in,out] plugin プラグインオブジェクトです。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
MStatus NintendoCheckLodCmd::Deregister(MFnPlugin& plugin)
{
    MStatus status = plugin.deregisterCommand("NintendoCheckLodCmd");
    if (status.error())
    {
        status.perror("Deregister NintendoCheckLodCmd");
    }
    return status;
}

