﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DccUtilitySceneMaterials.h"
#include "DccUtilityLogger.h"

namespace Dcc = nn::gfx::tool::dcc;

/******************************************************************************
    begin name space utility
******************************************************************************/
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {
namespace utility {

//----------------------------------------------------------------------------
// コンストラクタ
RSceneMaterials::RSceneMaterials(void)
: doesExportTexture(false), mDoesExportAnimation(false)
{
    init();
}

//----------------------------------------------------------------------------
// デストラクタ
RSceneMaterials::~RSceneMaterials(void)
{
    pointerMatinfoMap::iterator it = materialMap.begin();
    while(it != materialMap.end())
    {
        if(it->second.material)
        {
            delete it->second.material;
        }
        it->second.material = nullptr;
        ++it;
    }
    init();
}

//----------------------------------------------------------------------------
// 初期化
void RSceneMaterials::init(void)
{
    materialMap.clear();
    texLibrary.init();
    mTexPatAnims.clear();
    mTexSrtAnims.clear();
}


//----------------------------------------------------------------------------
// マテリアルアニメーションの出力準備をします。
void RSceneMaterials::PrepareAnimations(const Dcc::RExpOpt& rOpt)
{
    if(!mDoesExportAnimation) return;

    // ベイクしたカラーアニメーションをカーブに変換する
    for(int i = 0; i < getNumFMaterials(); i++)
    {
        FMaterial* mat = getFMaterialByIndex(i);
        if(mat)
        {
            mat->PrepareAnimations(rOpt);
        }
    }


    // ベイクされたテクスチャSRTアニメーションをカーブに変換する
    //const TexSrtAnim& texSrtAnim = mTexSrtAnims[texSrtAnimIdx];

    for(UINT iAnim = 0; iAnim < mTexSrtAnims.size(); iAnim++)
    {
        TexSrtAnim& texSrtAnim = mTexSrtAnims[iAnim];
        for(int iSRT = 0; iSRT < Dcc::ROriginalTexsrt::PARAM_COUNT; iSRT++)
        {
            Dcc::RAnimCurve& curve = texSrtAnim.m_Anims[iSRT];

            //	最終的に出力されるキーが空の場合のみ更新する。
            //	キーフレーム出力の場合はすでにこの段階でキーのリストが作成済みの状態となる。
            if( curve.m_Keys.size() == 0 )
            {
                // 値が設定されていたら常にアニメーションを出力する
                if (curve.m_FullValues.size() > 0)
                {
                    curve.m_LoopFlag = rOpt.m_LoopAnim;
                    switch(iSRT)
                    {
                    case Dcc::ROriginalTexsrt::SCALE_X:
                    case Dcc::ROriginalTexsrt::SCALE_Y:
                        curve.m_Tolerance = rOpt.m_TolTexS;
                        break;
                    case Dcc::ROriginalTexsrt::TRANSLATE_X:
                    case Dcc::ROriginalTexsrt::TRANSLATE_Y:
                        curve.m_Tolerance = rOpt.m_TolTexT;
                        break;
                    case Dcc::ROriginalTexsrt::ROTATE:
                        curve.m_Tolerance = rOpt.m_TolTexR;
                        curve.m_AngleFlag = true;
                        break;
                    }
                    curve.m_UseFlag = true;
                    curve.UpdateConstantFlag();
                    if(!curve.m_ConstantFlag)
                    {
                        curve.MakeKeys(GetFloatFrameFromSubFrame4f, nullptr, (iSRT == Dcc::ROriginalTexsrt::ROTATE));
                    }
                }
                else
                {
                    curve.m_UseFlag = false;
                    curve.m_ConstantFlag = true;
                }
            }
        }
    }

}

//----------------------------------------------------------------------------
// ポインタに対応するマテリアル情報を返す。
// 無ければ作成して返す。
RSceneMaterials::materialInfo* RSceneMaterials::getMaterialInfo(const void* addr)
{
    pointerMatinfoMap::iterator it = materialMap.find(addr);
    if(it == materialMap.end())
    {
        materialInfo matinfo;
        matinfo.index = static_cast<int>(materialMap.size());
        matinfo.material = new FMaterial();
        materialMap.insert(pointerMatinfoMap::value_type(addr, matinfo));
        it = materialMap.find(addr);
    }
    return &(it->second);
}

//----------------------------------------------------------------------------
// インデックスに対応するFMaterialを返す。
FMaterial* RSceneMaterials::getFMaterialByIndex(int index) const
{
    FMaterial* fmat = nullptr;
    pointerMatinfoMap::const_iterator it = materialMap.begin();
    while(it != materialMap.end())
    {
        const materialInfo& matinfo = it->second;
        if(matinfo.index == index)
        {
            fmat = matinfo.material;
            break;
        }
        ++it;
    }

    return fmat;
}

//----------------------------------------------------------------------------
// ポインタに対応する FMaterial が存在するか？
const bool RSceneMaterials::isExistFMaterial(const void* addr) const
{
    pointerMatinfoMap::const_iterator it = materialMap.find(addr);
    return 	(it != materialMap.end());
}

//----------------------------------------------------------------------------
// マテリアル名の重複を解消する。
bool RSceneMaterials::SetNamesUnique(void)
{
    pointerMatinfoMap::iterator it = materialMap.begin();
    ++it;
    while(it != materialMap.end())
    {
        int count = 1;
        pointerMatinfoMap::iterator sit = materialMap.begin();
        const char* iName = it->second.material->m_Name.c_str();
        char orgName[256], newName[256];
        sprintf_s(orgName, "%s", iName);
        sprintf_s(newName, "%s", iName);
        while(sit != it && sit != materialMap.end())
        {
            const char* sName = sit->second.material->m_Name.c_str();
            int cmp = _stricmp(newName, sName);
            // 同じ名前
            if(cmp == 0)
            {
                sprintf_s(newName, "%s_%02d", orgName, count);
                it->second.material->m_Name = newName;
                sit = materialMap.begin();
                count++;
                RLogger::LogMessagebyID(RLogger::kLogWRN_MaterialNameIsChanged, orgName);
            }
            else
            {
                ++sit;
            }
        }
        ++it;
    }
    return true;
}

//----------------------------------------------------------------------------
// マテリアル名の重複を解消する。
bool RSceneMaterials::isSamplerNameHintValid( void ) const
{
    const int matCount = getNumFMaterials();
    if (matCount > 0)
    {
        for (int iMat = 0; iMat < matCount; ++iMat)
        {
            FMaterial* mat = getFMaterialByIndex( iMat );
            const int samplerCount = static_cast<int>(mat->m_Samplers.size());
            for (int iSampler = 0; iSampler < samplerCount; ++iSampler)
            {
                const Dcc::RSampler& sampler = mat->m_Samplers[iSampler];
                const std::string name = sampler.GetName();
                const std::string hint = sampler.GetHintString();

                if (!Dcc::RIsValidElementNameString(name))
                {
                    string str = mat->m_Name + " : "  + name;
                    RLogger::LogMessagebyID(RLogger::kLogERR_SamplerNameIsWrong, str);
                    return false;
                }
                if (!Dcc::RIsValidElementNameString(hint))
                {
                    string str = mat->m_Name + " : "  + hint;
                    RLogger::LogMessagebyID(RLogger::kLogERR_SamplerHintIsWrong, str);
                    return false;
                }

                if(iSampler == 0) continue;

                for (int iOther = 0; iOther < iSampler; ++iOther)
                {
                    const Dcc::RSampler& other = mat->m_Samplers[iOther];
                    if (other.GetName() == name)
                    {
                        string str = mat->m_Name + " : "  + name;
                        RLogger::LogMessagebyID(RLogger::kLogERR_SamplerNameIsDuplicate, str);
                        return false;
                    }
                    if (other.GetHintString() == hint && !hint.empty())
                    {
                        string str = mat->m_Name + " : "  + hint;
                        RLogger::LogMessagebyID(RLogger::kLogERR_SamplerHintIsDuplicate, str);
                        return false;
                    }
                }
            }
        }
    }

    return true;
}


//-----------------------------------------------------------------------------
//! @brief テクスチャイメージの情報を取得して m_TexImgs に追加します。
//!
//! @param[in,out] ymodel モデルです。
//! @param[out] retIndex 追加したテクスチャイメージの m_TexImgs 内のインデックスを格納します。
//! @param[in] filePaths 画像ファイルのパス配列です。
//! @param[in] hint テクスチャの関連付けに利用するヒント情報です。
//!
//! @return 処理結果を返します。
//-----------------------------------------------------------------------------
bool RTextureLibrary::addTextureFile( int& retIndex, const Dcc::RStringArray& filePaths, const Dcc::RSampler& sampler, const int curSwizzle, const Dcc::RExpOpt& rOpt )
{
    //-----------------------------------------------------------------------------
    // 次元を決定します。
    bool checksCubeWH = false;
    int imageW = 1;
    int imageH = 1;
#if	0
    if (hint == RSampler::Reflection && filePaths.size() == 1)
    {
        // 反射マップの場合、画像の幅と高さを調べてキューブマップか判定します。
        checksCubeWH = true;
        GetImageInfoForMaya(imageW, imageH,
            yscene, filePaths[0], yopt.m_CheckElementFlag);
    }
    const RImage::Dimension dimension = RImage::GetDimension(
        static_cast<int>(filePaths.size()), checksCubeWH, imageW, imageH);
    const std::string mainFilePath = (dimension == RImage::DimCubeSeparate) ?
        filePaths[RImage::CubeFacePz] : filePaths[0];
#else
    const Dcc::RImage::Dimension dimension = Dcc::RImage::GetDimension(
        static_cast<int>(filePaths.size()), checksCubeWH, imageW, imageH);
    const std::string mainFilePath = filePaths[0];

#endif
    //cerr << "image dim: " << mainFilePath << ": " << dimension << endl;

    //-----------------------------------------------------------------------------
    // 用途を決定します。
    /*
    Dcc::RImage::Usage usage = Dcc::RImage::ColorMap;
    if      (hint == Dcc::RSampler::OPACITY) usage = Dcc::RImage::AlphaMap;
    else if (hint == Dcc::RSampler::NORMAL ) usage = Dcc::RImage::NormalTangent;
    */
    const string imgHint = sampler.GetImageHintString();
    //int curSwizzle = 0; //texImg.GetInitialSwizzle();

    //-----------------------------------------------------------------------------
    // 既存のテクスチャイメージから同じ画像ファイルに対するデータを探します。
    const int texImgCount = static_cast<int>(m_TexImgs.size());
    for (int iTexImg = 0; iTexImg < texImgCount; ++iTexImg)
    {
        const Dcc::RImage& other = m_TexImgs[iTexImg];
        if (Dcc::RIsSameStringNoCase(other.GetMainFilePath(), mainFilePath))
        {
            if (other.GetDimension()        != dimension        ||
                other.GetFilePaths().size() != filePaths.size())
            {
                if (other.GetDimension() == Dcc::RImage::DIM_2D || dimension == Dcc::RImage::DIM_2D)
                {
                    //YShowError(yscene, "Texture is used for 2D and cube map: %s", mainFilePath.c_str());
                }
                else
                {
                    //YShowError(yscene, "Texture is used for different type cube map: %s", mainFilePath.c_str());
                }
                return false;
            }
            if (dimension == Dcc::RImage::DIM_CUBE_SEPARATE)
            {
                for (int iFile = 0; iFile < static_cast<int>(filePaths.size()); ++iFile)
                {
                    if (!Dcc::RIsSameStringNoCase(other.GetFilePaths()[iFile], filePaths[iFile]))
                    {
                        //YShowError(yscene, "Cube map image is not same: %s %s (%s)",
                        //	other.GetFilePaths()[iFile].c_str(), filePaths[iFile].c_str(),
                        //	GetEnvCubeFaceAttrName(iFile).c_str());
                        return false;
                    }
                }
            }
            // NW4F では同じテクスチャファイルを使う複数のサンプラでもOK
#if 0
            if (other.GetHint() != imgHint)
            {
                //YShowError(yscene, "Texture is used for different usage: %s", mainFilePath.c_str());
                return false;
            }
#endif
            // 既存のテクスチャイメージの初期スウィズル値が小さければ更新します。
            if (other.GetInitialSwizzle() < curSwizzle)
            {
                m_TexImgs[iTexImg].SetInitialSwizzle(curSwizzle);
            }

            retIndex = iTexImg;
            return true;
        }
    }

    //-----------------------------------------------------------------------------
    // add tex img
    Dcc::RImage texImg(texImgCount);

    texImg.SetHint(imgHint);
    texImg.SetLinearFlag(rOpt.GetTexLinearFlag(imgHint));
    texImg.SetDimension(dimension);
    texImg.SetInitialSwizzle(curSwizzle);

    //texImg.SetUsage(usage);

    texImg.SetFilePaths(filePaths); // パスからテクスチャ名も設定されます。
    if (texImg.GetName() != texImg.GetOrgName())
    {
        RLogger::LogMessagebyID( RLogger::kLogWRN_TexNameIsChanged, (texImg.GetOrgName() + " -> " + texImg.GetName()).c_str() );
    }

    m_TexImgs.push_back(texImg);

    retIndex = texImgCount;

    return true;
}

//----------------------------------------------------------------------------
// テクスチャライブラリを初期化
void RTextureLibrary::init()
{
    m_TexImgs.clear();
}


/******************************************************************************
    end name space utility
******************************************************************************/
}}}}} // namespace utility
