﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/******************************************************************************
    include
******************************************************************************/

#include "DccCommon.h"
#include "DccShape.h"
#include "DccAnimation.h"

#include "DccUtilityModel.h"
#include "DccUtilityShape.h"
#include "DccUtilitySceneMaterials.h"

/******************************************************************************
    begin name space utility
******************************************************************************/
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {
namespace utility {

class RModel;
class FShape;
//class RSceneMaterials;

/**
座標系を指定する列挙子
\par 解説：
- LocalSpace ローカル座標系、もしくは親階層の座標系
- WorldSpace ワールド座標系
*/
enum ECoordSystem
{
    LocalSpace,
    WorldSpace,
};

/**
モデルやボーンの配置を表すノードを扱うためのクラス
\par 解説：
　モデルやボーンの配置を表すノードを扱うためのクラス。
RNode を使ってノードの階層構造を構築した後、
ノードの階層構造を処理する RNodeProcessor クラスによって出力可能に処理する。
*/
class RNode : public nn::gfx::tool::dcc::RBone
{
public:
    /**
    コンストラクタ
    */
    RNode(void);

    /**
    デストラクタ
    \par 解説：
    　 setParent() により接続された子階層のノードを再帰的に削除する。
    */
    ~RNode(void);

    /**
    ノード名を設定
    \param name ノード名
    \par 解説：
    　ノード名を設定する。
    ノード名は<Bone>タグの"Name"属性に出力される。
    */
    void setName(const std::string &name)
    { m_Name = name; }

    /**
    ノード名を取得
    \return ノード名
    \par 解説：
    　ノード名を取得する。
    */
    const std::string getName(void) const
    { return m_Name; }

    /**
    階層構造の親を指定
    \param parent 階層構造の親ノード
    \par 解説：
    　親ノードを指定することで階層構造を作成する。
    この関数は一回しか呼び出すことはできない。
    2回目以降の呼び出しはアサートされる。
    これは階層構造の変更ができないことを意味するが、
    エキスポータでは問題にならない。
    */
    void setParent(RNode* parent);
    /**
    階層構造の親を取得
    \return 階層構造の親ノード
    \par 解説：
    　階層構造の親を取得する。
    */
    RNode* getParent(void) const
    { return mParent; }

    /**
    階層構造の子の先頭を取得
    \return 階層構造の子の先頭ノード
    \par 解説：
    　階層構造の子の先頭を取得する。
    */
    RNode* getChild(void) const
    { return mChild; }

    /**
    このノードのモデルを設定
    \param model モデルを表す RModel オブジェクト
    \par 解説：
    　このノードがモデルを持つ場合、
    それを表す RModel オブジェクトを設定する。
    <br />
    　一つの RModel オブジェクトを複数の RNode から参照する
    インスタンスの設定は可能である。
    しかし実際にインスタンスが行えるかは中間フォーマットの仕様に依存し、
    現在のバージョンではインスタンスは表現できない。
    インスタンスの設定が行われた場合には RNodeProcessor 処理内で複製が作成される。
    \par
    　シーンを RNodeProcessor で削除した場合、
    ノードに登録したモデル (RModel) は RNodeProcessor のデストラクタで削除される。
    この関数を繰り返し呼び出した場合、
    直前に設定されていたモデルはメモリリークとなるので注意すること。
    */
    void setModel(RModel* model);

    /**
    ノードのモデルを取得
    \return モデルを表す RModel オブジェクト
    \par 解説：
    　 setModel() で設定したモデルを取得する。
    */
    RModel* getModel(void) const
    { return mModel; }

    /**
    ノードの表示/非表示を設定
    \param visible trueで表示
    \par 解説：
    　ノードの表示/非表示を設定する。
    初期状態では表示(true)である。
    非表示のノードが参照するモデルも出力されるが、<Mesh>の IsVisible 属性がfalseになる。
    \sa
    isVisible()
    */
    void setVisible(bool visible)
    { m_Visibility = visible; }

    /**
    ノードの表示/非表示を取得
    \retval true 表示
    \retval false 非表示
    \sa
    setVisible()
    */
    bool isVisible(void) const
    { return m_Visibility; }

    /**
    Visibility アニメーションの状態を設定
    \param visibility	表示状態（0.0f ならば非表示、それ以外なら表示）
    \par 解説：
    　このノードの表示状態を設定する。
    */
    void setVisibilityAnim(float visibility);

    /**
    Shape アニメーションを設定
    \param shapeID シェイプID （0.0f - 1.0f）
    \param weight 重み（0.0f - 1.0f）
    \par 解説：
    　キーシェープの重みのアニメーションを追加する。
    */
    void setShapeAnim(UINT shapeID,  float weight);

    /**
    ローカル変換行列を設定
    \param frame フレーム番号
    \param mtx ローカル変換行列
    \par 解説：
    　親ノードの座標系で表された、このノードの変換行列を設定する。
    フレーム番号を指定して時系列の変化を設定できる。
    ただし時間軸に沿った順番で指定をする必要がある。
    最初に指定したフレームより前のフレームを指定するとアサートされる。
    */
    void setLocalMatrix(int frame, const nn::gfx::tool::dcc::RMtx44 &mtx);

    /**
    ローカル変換行列を取得
    \param frame フレーム番号
    \return ローカル変換行列
    \par 解説：
    　親ノードの座標系で表された、このノードの変換行列を取得する。
    フレーム番号を指定して時系列の変化を取得できる。
    最後のフレームより後のフレームを参照した場合には、その最後のフレームの値を返す。
    最初のフレームより前のフレームを参照した場合には、最初のフレームの値を返す。
    */
    const nn::gfx::tool::dcc::RMtx44 &getLocalMatrix(int frame) const;

    /**
    ワールド変換行列を取得
    \param frame フレーム番号
    \return ワールド変換行列
    \par 解説：
    　このノードをワールド座標系まで変換する変換行列を取得する。
    フレーム番号を指定して時系列の変化を取得できる。
    行列が設定されていないフレームを参照した場合には、その直前に指定されているフレームの値を返す。
    最初のフレームより前のフレームを参照した場合には、最初のフレームの値を返す。
    */
    const nn::gfx::tool::dcc::RMtx44 getWorldMatrix(int frame) const;

    /**
    スキニング用のバインド行列を設定
    \param mtx バインド行列(ワールド座標系)
    \par 解説：
    　スキニングのボーンとして使われているノードの、バインド姿勢の行列を設定する。
    行列はワールド座標系で表され、逆行列にする必要は無い。
    リグの階層内もしくは親階層にあるがボーンとして使われていないノードには、
    バインド行列の設定を行わなくてもかまわない。
    */
    void setBindMatrix(const nn::gfx::tool::dcc::RMtx44 &mtx)
    {
        mBindMatrix = mtx;
        mHasBindMatrix = true;
    }

    /**
    スキニング用のバインド行列を取得
    \param coordSystem 行列を取得する座標系
    \return バインド行列
    \par 解説：
    　スキニングのボーンとして使われているノードの、バインド姿勢の行列を取得する。
    行列は親階層の座標系かワールド座標系で表され、逆行列にはなっていない。
    setBindMatrix() でバインド行列が設定されていないノードでは単位行列が戻る。
    */
    const nn::gfx::tool::dcc::RMtx44 getBindMatrix(ECoordSystem coordSystem) const;

    /**
    ノードインデックスを取得
    \return ノードインデックス
    \par 解説：
    　<SkeletalModel><Skeleton><Bones> 配列の何番目に出力されるかを取得する。
    この値は RNode の階層構造を RNodeProcessor で処理した後に有効になる。
    それより前のタイミングで呼び出すとアサートされる。
    */
    int getNodeArrayIndex(void) const
    { assert(0 <= mNodeArrayIndex); return mNodeArrayIndex; }

    /**
    スケーリングルールを設定する
    \param rule スケーリングルール
    \par 解説：
    　ノードのスケーリング方式を設定する。
    この設定は階層構造で全て同じ設定になるようにしなければいけない。
    設定が異なるノードがあると RNodeProcessor::processNodeTree() でエラーとなる。
    */
    void setScalingRule( nn::gfx::tool::dcc::RSkeleton::ScaleMode rule )
    { mScalingRule = rule; }

    /**
    スケーリングルールを取得する
    \return スケーリングルール
    \par 解説：
    　ノードのスケーリング方式を取得する。
    */
    nn::gfx::tool::dcc::RSkeleton::ScaleMode getScalingRule() const
    { return mScalingRule; }

    /**
    ノードを無効と判断しないように設定
    \param CompressEnable true:ノードが圧縮可能ならば圧縮する。
    \par 解説：
    　最適化コンバーターによる最適化処理でこのノード圧縮可能であることをフラグで設定する。
    */
    void setCompressEnableFlag(bool CompressEnable)
    { m_CompressEnable = CompressEnable; }

    /**
    このノードのビルボードモードを設定します。
    \param mode	ビルボードモード
    \par 解説：
    　中間ファイルへ出力されるビルボードモードを設定します。
    初期値は R_BILLBOARD_OFF である。
    */
    void setBillboardMode(RBone::Billboard mode)
    { m_Billboard = mode; }

    /**
    ノードの有効/無効を設定
    \param active ノードが有効かどうか
    \param recursive 子階層へ再帰するか
    \par 解説：
    　ノードの有効/無効を設定する。
    初期状態では有効である。
    この設定は以下の機能に影響する。
    - enumNodeToArray()
    - RNodeProcessor::buildModelInstanceList()
    */
    void setActive(bool active, bool recursive);

    /**
    ノードの有効/無効を取得
    \retval true ノードは有効
    \retval false ノードは無効
    */
    bool isActive(void) const
    { return mActive; }

    /**
    シーンを構成するノードを配列に並べる(再帰)(内部使用)
    \param nodeList シーンを構成するノードを並べる配列
    \par 解説：
    　ノードの階層構造を深さ優先探索でたどり、配列に並べる。
    この関数は RNodeProcessor::processNodeTree() から呼び出される。
    クラスユーザーはこの関数を使う必要は無い。
    */
    void enumNodeToArray(std::vector<RNode*> &nodeList);

    /**
    設定されたローカル変換行列アニメーションから移動、回転、スケールの
    アニメーションに変換する。
    \par 解説：
    　setLocalMatrix() で設定された行列アニメーションから移動、回転、スケールの
    値のアニメーションに変換する。
    に変換する。
    */
    void convertMatrixAnimToSRTAnim();

    /**
    ビジビリティアニメーションデータを取得
    \par 解説：
    　ビジビリティアニメーションデータを取得します。
    */
    nn::gfx::tool::dcc::RAnimCurve& getVisibilityAnimationData( void ){ return mVisAnim; }

    /**
    アニメーションのサブサンプリングの数を指定
    \param[in] sub サブサンプリング数
    \par 解説：
    　1フレーム内で複数のサンプリングを行う場合の数を指定する。
      1,2,5,10が指定できる
    */
    static void setAnimSubSample(int sub)
    {
        assert(sub == 1 || sub == 2 || sub == 5 || sub == 10);
        mAnimSubSample = sub;
    }

    /**
    アニメーションのサブサンプリングの数を取得
    \return サブサンプリング数
    \par 解説：
    　1フレーム内で複数のサンプリングを行う場合の数を取得する。
    */
    static int getAnimSubSample(void)
    {
        return mAnimSubSample;
    }

    /**
    ユーザーデータを追加
    \param[in] rData 追加されるユーザーデータ
    \par 解説：
    　このノードにユーザーデータを追加します。
    */
    void addUserData( nn::gfx::tool::dcc::RUserData&	rData ){ m_UserDatas.push_back( rData ); }

    /**
    アニメーションキーを強制的に出力するフラグを設定
    */
    void setForceExportKeyFlag(bool scale, bool rotate, bool translate, bool visibility)
    {
        mForceExportScaleKey = scale;
        mForceExportRotateKey = rotate;
        mForceExportTransKey= translate;
        mForceExportVisibility = visibility;
    }

    /**
    アニメーション圧縮の許容値を設定
    */
    static void setTolerances(float toleranceTrans, float toleranceRotate, float toleranceScale)
    {
        mScaleTolerance = toleranceScale;
        mRotateTolerance = toleranceRotate;
        mTransTolerance = toleranceTrans;
    }

    /**
    ノードのSRT各要素がアニメーションしているかどうかを表すフラグを設定する。
    */
    void setAnimationFlag();

    /**
    ノードがアニメーションしているかどうかを表すフラグ
    */
    bool hasAnimation()
    {
        return (mHasScaleAnim || mHasRotateAnim || mHasTransAnim);
    }

    bool hasVisibilityAnimation()
    {
        return !mVisAnim.m_ConstantFlag || mForceExportVisibility || mVisAnim.m_FullValues.size() > 0;
    }

    void ClearShapeAnim()
    {
        mShapeAnims.clear();
    }

protected:
    /**
    階層構造の親ノード
    */
    RNode* mParent;

    /**
    階層構造の子ノード
    */
    RNode* mChild;

    /**
    階層構造の同階層の次のノード
    */
    RNode* mNext;

    /**
    モデル
    */
    RModel* mModel;

    /**
    ノードの有効/無効
    \par 解説：
    　ノードの出力を制御する。
    無効だと出力から除かれる。
    */
    bool mActive;

    /**
    変換行列配列の最初のフレーム数
    */
    int mMatrixArrayFirstFrame;

public:
    /**
    ローカル変換行列の配列
    */
    std::vector<nn::gfx::tool::dcc::RMtx44> mLocalMatrix;

    /**
    スキニング用のバインド行列
    \par 解説：
    　ワールド座標系で設定される。
    */
    nn::gfx::tool::dcc::RMtx44 mBindMatrix;

    /**
    バインド行列が設定されているかどうか。
    */
    bool mHasBindMatrix;

    /**
    ノードインデックス
    \par 解説：
    　<SkeletalModel><Skeleton><Bones> 配列の何番目に出力されるかを取得する。
    RNodeProcessor::processNodeTree() によって決定される。
    */
    int mNodeArrayIndex;

    /**
    スケーリングルール
    */
    nn::gfx::tool::dcc::RSkeleton::ScaleMode mScalingRule;

    /**
    アニメーションがない場合の固定出力値
    */
    nn::gfx::tool::dcc::RVec3		mNoAnimConstScale;
    /**
    アニメーションがない場合の固定出力値
    */
    nn::gfx::tool::dcc::RVec3		mNoAnimConstRotate;
    /**
    アニメーションがない場合の固定出力値
    */
    nn::gfx::tool::dcc::RVec3		mNoAnimConstTrans;

    /**
    スケーリングのアニメーション
    */
    nn::gfx::tool::dcc::RAnimCurve		mScaleCurve[3];

    /**
    回転のアニメーション
    */
    nn::gfx::tool::dcc::RAnimCurve		mRotateCurve[3];

    /**
    移動のアニメーション
    */
    nn::gfx::tool::dcc::RAnimCurve		mTransCurve[3];

    /**
    Visibility アニメーションデータ
    */
    nn::gfx::tool::dcc::RAnimCurve		mVisAnim;

    /**
    Shape アニメーションデータ
    */
    std::vector<nn::gfx::tool::dcc::RAnimCurve> mShapeAnims;

    /**
    アニメーションを持っているかどうか
    */
    bool mHasScaleAnim;
    bool mHasRotateAnim;
    bool mHasTransAnim;

    /**
    アニメーションキーを強制的に出力する
    */
    bool mForceExportScaleKey;
    bool mForceExportRotateKey;
    bool mForceExportTransKey;
    bool mForceExportVisibility;

    int m_TotalTriangleCount; //!< 出力した三角形の総数です。
    int m_TotalIndexCount; //!< 出力した頂点インデックスの総数です。
    int m_TotalVertexCount; //!< 出力した頂点の総数です。
    int m_TotalProcessVertexCount; //!< 頂点シェーダが処理する頂点の総数です。

    ////////////////////////
    // static
    /**
    アニメーション範囲
    */
    static int mAnimSubSample;

    /**
    アニメーション圧縮の許容値
    */
    static float mScaleTolerance;
    static float mRotateTolerance;
    static float mTransTolerance;

};

//----------------------------------------------------------------------------
/**
ノードの階層構造を処理するクラス
\par 解説：
　ノードの階層構造を受け取り処理するクラス。
RNodeの階層構造を受け取り、<SkeletalModel>以下の下に列挙したデータを出力する。
それぞれのデータを出力する関数が用意される。
- <Shapes>タグ内の<SeparateDataShapeCtr>配列 -> outShapesData()
- <Skeleton>タグ内の<Bones>配列 -> outSkeletonData()

\par
　ノードの階層構造からスキンモデルのボーンとして使われている部分を探し出してボーン配列を作成する。
この処理により <SkeletalModel><Skeleton><Bones> 配列の内容と、
各ボーン毎の(頂点データに記録する)ボーンインデックスが決まる。
複数のスキンモデルとスタティックモデルを一括して処理することができる。
\par
　スキンモデルではないスタティックなモデルは、
リジッドスキンの手法でスキンモデルとして出力することができる。
*/
class RNodeProcessor
{
public:
    /**
    コンストラクタ
    */
    RNodeProcessor(void);

    /**
    デストラクタ
    */
    ~RNodeProcessor(void);

    /**
    Init
    */
    void Init(void)
    {
        this->doesUnifyQuantizeScale = false;
        this->m_magnify = 1.0f;
        this->m_RigidSkinningMtxCount = 0;
        this->m_SmoothSkinningMtxCount = 0;
        this->mModelInstList.clear();
        this->mNodeList.clear();
    }

    /**
    RNode 関連データを中間ファイルとして出力できるように処理する
    \param NodeList	出力するノードのリスト
    \par 解説：
    　RNode 関連データを中間ファイルとして出力できるように処理する
    */
    bool processNodeList(std::vector<RNode*>&	NodeList, const nn::gfx::tool::dcc::RVtxMtxArray& vtxMtxs);

    /**
    シェイプの数を取得
    \return シェイプの数
    \par 解説：
    何個のシェイプがあったかを返す。
    事前に processNodeTree() を呼び出す必要がある。
    */
    unsigned int getNumShapes(void) const
    {
        unsigned int numShapes = 0;
        std::vector<RModelInstance>::const_iterator it = mModelInstList.begin();
        while(it != mModelInstList.end())
        {
            numShapes += it->model->getNumShapes();
            ++it;
        }
        return numShapes;
    }

    /**
    ボーンの数を取得
    \return ボーンの数
    \par 解説：
    何個のボーンがあったかを返す。
    事前に processNodeTree() を呼び出す必要がある。
    */
    std::vector<RNode*>::size_type getNumBones(void) const
    {
        return mNodeList.size();
    }

    /**
    スムーススキニングのシェイプの数を取得
    \return シェイプの数
    \par 解説：
    事前に processNodeTree() を呼び出す必要がある。
    */
    unsigned int getNumSmoothSkinningShapes(void) const
    {
        unsigned int numShapes = 0;
        std::vector<RModelInstance>::const_iterator it = mModelInstList.begin();
        while(it != mModelInstList.end())
        {
            numShapes += it->model->m_SmoothSkinningShapeCount;
            ++it;
        }
        return numShapes;
    }

    /**
    リジッドスキニングのシェイプの数を取得
    \return シェイプの数
    \par 解説：
    事前に processNodeTree() を呼び出す必要がある。
    */
    unsigned int getNumRigidSkinningShapes(void) const
    {
        unsigned int numShapes = 0;
        std::vector<RModelInstance>::const_iterator it = mModelInstList.begin();
        while(it != mModelInstList.end())
        {
            numShapes += it->model->m_RigidSkinningShapeCount;
            ++it;
        }
        return numShapes;
    }

    /**
    スムーススキニングのマトリクスの数を取得
    \return マトリクスの数
    \par 解説：
    事前に processNodeTree() を呼び出す必要がある。
    */
    unsigned int getNumSmoothSkinningMtxs(void) const { return m_SmoothSkinningMtxCount; }

    /**
    リジッドスキニングのマトリクスの数を取得
    \return マトリクスの数
    \par 解説：
    事前に processNodeTree() を呼び出す必要がある。
    */
    unsigned int getNumRigidSkinningMtxs(void) const { return m_RigidSkinningMtxCount; }

    /**
    出力した三角形の総数を取得
    \par 解説：
    事前に outShapeData() を呼び出す必要がある。
    */
    unsigned int getTotalTriangleCount(void) const
    {
        unsigned int numTri = 0;
        std::vector<RModelInstance>::const_iterator it = mModelInstList.begin();
        while(it != mModelInstList.end())
        {
            numTri += it->node->m_TotalTriangleCount;
            ++it;
        }
        return numTri;
    }

    /**
    出力した頂点インデックスの総数を取得
    \par 解説：
    事前に outShapeData() を呼び出す必要がある。
    */
    unsigned int getTotalIndexCount(void) const
    {
        unsigned int numIdx = 0;
        std::vector<RModelInstance>::const_iterator it = mModelInstList.begin();
        while(it != mModelInstList.end())
        {
            numIdx += it->node->m_TotalIndexCount;
            ++it;
        }
        return numIdx;
    }

    /**
    出力した頂点の総数を取得
    \par 解説：
    事前に outShapeData() を呼び出す必要がある。
    */
    unsigned int getTotalVertexCount(void) const
    {
        unsigned int numVtx = 0;
        std::vector<RModelInstance>::const_iterator it = mModelInstList.begin();
        while(it != mModelInstList.end())
        {
            numVtx += it->node->m_TotalVertexCount;
            ++it;
        }
        return numVtx;
    }

    /**
    頂点シェーダが処理する頂点の総数を取得
    \par 解説：
    事前に outShapeData() を呼び出す必要がある。
    */
    unsigned int getTotalProcessVertexCount(void) const
    {
        unsigned int numVtx = 0;
        std::vector<RModelInstance>::const_iterator it = mModelInstList.begin();
        while(it != mModelInstList.end())
        {
            numVtx += it->node->m_TotalProcessVertexCount;
            ++it;
        }
        return numVtx;
    }

    /**
    全ての FShape を集めます。
    */
    void collectShapeData( std::vector<FShape*>& ShapeDataList );

    /**
    <SkeletalModel><Shapes>タグを出力
    \param[in] os 出力ストリーム
    \param[in] tabSize <SkeletalModel>タグのインデントに必要なタブの数
    \param[in] magnify シーン全体のスケール値
    \return true は成功
    \par 解説：
    　<SkeletalModel><Shapes>タグ内の<SeparateDataShapeCtr>配列を出力する。
    これはシーン内のモデルを構成する全シェイプを列挙する処理となる。
    <Shapes>タグも出力する。
    事前に processNodeTree() を呼び出す必要がある。
    */
    bool outShapesData(std::ostream& os, nn::gfx::tool::dcc::RDataStreamArray& dataStreams, int tc, nn::gfx::tool::dcc::RVtxMtxArray vtxMtxs, RSceneMaterials& sceneMaterials);

    /**
    <SkeletalModel><Skeleton>タグを出力
    \param[in] os 出力ストリーム
    \param[in] tc <skeleton>タグのインデントに必要なタブの数
    \par 解説：
    　<skeleton></skeleton>タグ内の<Bones>配列を出力する。
    事前に processNodeTree() を呼び出す必要がある。
    */
    void outSkeletonData(std::ostream& os, int tc);

    //************************************
    // Method:    outBoneAnim
    // FullName:  utility::RNodeProcessor::outBoneAnim
    // Access:    public
    // Returns:   void
    // Qualifier:
    // Parameter: ostream & os
    // Parameter: nn::gfx::tool::dcc::RDataStreamArray dataStreams
    //************************************
    void outBoneAnim( ostream & os, nn::gfx::tool::dcc::RDataStreamArray& dataStreams);

    /**
    \brief ボーンビジビリティアニメーションを 1 つ出力します。

    \param[in,out] os 出力ストリームです。
    \param[in,out] dataStreams データ列配列です。
    \param[in] tc <bone_vis_bone_anim> 要素のインデントに必要なタブの数です。
    \param[in] animIdx ボーンビジビリティアニメーションのインデックスです。
    \param[in] node ノードです。
    */
    void outBoneVisAnim(
        std::ostream& os,
        nn::gfx::tool::dcc::RDataStreamArray& dataStreams,
        const int tc,
        const int animIdx,
        const RNode& node );


    /**
    BoneVisibility アニメーションの出力
    \param os 出力ストリーム
    \param dataStreams データ出力ストリーム
    \par 解説：
    　ボーンビジビリティアニメーションのデータタグを出力します。
    <bone_vis_bone_anim_array/> 以下が出力されます。
    */
    void outVisibilityAnimationData(std::ostream &os, nn::gfx::tool::dcc::RDataStreamArray& dataStreams, const nn::gfx::tool::dcc::RExpOpt& rOpt );

    /**
    シェイプアニメーションの出力
    \param os 出力ストリーム
    \param dataStreams データ出力ストリーム
    \par 解説：
    　シェイプアニメーションのデータタグを出力します。
    <vertex_shape_anim_array /> 以下が出力されます。
    */
    void outShapeAnimationData(std::ostream &os, nn::gfx::tool::dcc::RDataStreamArray& dataStreams, const nn::gfx::tool::dcc::RExpOpt& rOpt );

    /**
    シーン全体の拡大縮小を行う値を設定
    \param[in] val スケール
    \par 解説：
    　シーン全体の拡大・縮小を行う値を設定する。
    */
    void setMaginify(float scale)
    {
        m_magnify = scale;
    }

    /**
    アニメーション持つノードがあるか
    */
    bool hasAnimationNode();

    /**
    ビジビリティアニメーション持つノードがあるか
    */
    bool hasVisibilityAnimationNode();

    /**
    量子化スケールの統一を行うかどうか設定する
    */
    void setUnifyQuantizeScale(bool flag)
    {
        doesUnifyQuantizeScale = flag;
    }


protected:
    /**
    シーン内のモデルに対して、スキニングに関するモードおよび座標系の変換処理を行う
    \par 解説：
    */
    void convertSkinningData(const nn::gfx::tool::dcc::RVtxMtxArray& vtxMtxs);

    /**
    シーンに登場するモデルの一覧を作成
    \par 解説：
    　シーンに登場するモデルを列挙する mModelInstList 配列を作成する。
    この時点でモデル内の各シェイプに対して、<Shapes>タグ内での順番が確定する。
    */
    void buildModelInstanceList(void);

    /**
    @brief スキニングに使用するボーンの行列パレットインデックスを設定します。
           行列パレットはスムーススキン用配列の後にリジッドスキン用配列が並ぶようにします。
    */
    void setSkinningMtxIdx( void );

    /**
    モデルデータの整理を行う
    \par 解説：
    　同じ値を持つ頂点属性データのマージなど。
    */
    bool optimizeModels(const nn::gfx::tool::dcc::RVtxMtxArray& vtxMtxs, const nn::gfx::tool::dcc::RIntArray& smoothMtxIdxs, const nn::gfx::tool::dcc::RIntArray& rigidMtxIdxs, const RSceneMaterials& sceneMaterials);

    /**
    ノードのアニメーションフラグを設定
    \par 解説：
      ノードのSRT各要素がアニメーションしているかどうかを表すフラグを設定する。
      アニメーションのキー及び強制出力フラグを反映する。
    */
    void setAnimationFlag(void);

protected:
    /**
    ノードとモデルのペア(モデルのインスタンス)
    */
    struct RModelInstance
    {
        /**
        ノード
        */
        RNode* node;

        /**
        ノードから参照されるモデル
        \par 解説：
        　モデルがインスタンスされている場合、
        インスタンス毎に RModel の複製を作成する。
        よってこの変数が指すモデルは他との重複はない。
        */
        RModel* model;

        /**
        このインスタンスが<Shapes>配列内の何番目から始まるか
        */
        int shapeIndexBase;
    };

    /**
    ノードとモデルのペア(モデルのインスタンス)の配列
    \par 解説：
    　 processNodeTree() で作成される。
    */
    std::vector<RModelInstance> mModelInstList;

    /**
    シーンを構成するノードの配列
    \par 解説：
    　中間フォーマットの仕様に従い、ノードを深さ優先探索で列挙した配列。
    同階層のノードは名前を辞書順にソートした順番で並んでいる。
    <Skeleton><Bones>タグ内での順番となる。
    processNodeTree() で作成される。
    */
    std::vector<RNode*> mNodeList;

    /**
    出力するデータの拡大・縮小を決める値
    \par 解説：
    　 setMagnify() を参照。
    */
    float m_magnify;

    /**
    量子化スケールの統一を行うかどうかのフラグ
    */
    bool doesUnifyQuantizeScale;

    unsigned int m_SmoothSkinningMtxCount; //!< スムーススキニングのマトリクス数です。
    unsigned int m_RigidSkinningMtxCount; //!< リジッドスキニングのマトリクス数です。

};


extern const nn::gfx::tool::dcc::RMtx44 cancelMatrix( const nn::gfx::tool::dcc::RMtx44& cmtx, const nn::gfx::tool::dcc::RMtx44& pmtx );
extern float GetFloatFrameFromSubFrame(const int subFrame);
extern float GetFloatFrameFromSubFrame4f(const int subFrame, const void* pParam);

/******************************************************************************
    end name space utility
******************************************************************************/
}}}}} // namespace utility
