﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/******************************************************************************
    include
******************************************************************************/
#include "DccUtilityCommon.h"

#include "DccMaterial.h"
#include "DccShape.h"
#include "DccAnimation.h"

/******************************************************************************
    begin name space utility
******************************************************************************/
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {
namespace utility {

/******************************************************************************
    extern
******************************************************************************/
extern float GetFloatFrameFromSubFrame4f(const int subFrame, const void* pParam);

/**
nn::gfx::tool::dcc::RMaterial に存在しないパラメータを一時的を保持するためのクラス。
\par 解説：
　
utility の RMaterial にあわせるために、一部 nn::gfx::tool::dcc::RMaterial へのインターフェイスメソッドを定義しています。
*/
class FMaterial : public nn::gfx::tool::dcc::RMaterial
{
public:
    //! @brief コンストラクタです。
    FMaterial( void ) :
        nn::gfx::tool::dcc::RMaterial(),
        m_OrgIndex(0),
        m_Index(0)
    {
        for(int i = 0; i < nn::gfx::tool::dcc::RPrimVtx::VTX_TEX_MAX; i++) { m_UseCafeUV[i] = false; }
        m_SamplerReferUVs.clear();
    }

    //! @brief デストラクタです。
    virtual ~FMaterial( void ){}

    //! @brief Diffuse カラーの値を設定します。
    void SetDiffuse(nn::gfx::tool::dcc::RVec4 col)
    {
        m_Diffuse.x = col.x;
        m_Diffuse.y = col.y;
        m_Diffuse.z = col.z;
    }

    //! @brief Opacity の値を設定します。
    void SetOpacity(float opacity)
    {
        m_Opacity.x = opacity;
        m_Opacity.y = opacity;
        m_Opacity.z = opacity;
    }

    //! @brief Specular カラーの値を設定します。
    void SetSpecular(nn::gfx::tool::dcc::RVec4 col)
    {
        m_Specular.x = col.x;
        m_Specular.y = col.y;
        m_Specular.z = col.z;
    }

    //! @brief Emission カラーの値を設定します。
    void SetEmission(nn::gfx::tool::dcc::RVec4 col)
    {
        m_Emission.x = col.x;
        m_Emission.y = col.y;
        m_Emission.z = col.z;
    }

    //! @brief Ambient カラーの値を設定します。
    void SetAmbient(nn::gfx::tool::dcc::RVec4 col)
    {
        m_Ambient.x = col.x;
        m_Ambient.y = col.y;
        m_Ambient.z = col.z;
    }

    /////////////////////////////
    // カラーアニメーション
    //	DIFFUSE_R , DIFFUSE_G , DIFFUSE_B ,
    //	OPACITY_R , OPACITY_G , OPACITY_B ,
    //	AMBIENT_R , AMBIENT_G , AMBIENT_B ,
    //	EMISSION_R, EMISSION_G, EMISSION_B,
    //	SPECULAR_R, SPECULAR_G, SPECULAR_B,

    /**
    ディフューズカラー(RGB)アニメーションを設定する
    */
    void SetDiffuseAnim(nn::gfx::tool::dcc::RVec3 col)
    {
        m_Anims[DIFFUSE_R].m_FullValues.push_back(col.x);
        m_Anims[DIFFUSE_G].m_FullValues.push_back(col.y);
        m_Anims[DIFFUSE_B].m_FullValues.push_back(col.z);
    }

    /**
    ディフューズのアルファアニメーションを設定する
    */
    void SetOpacityAnim(float fAlpha)
    {
        m_Anims[OPACITY_R].m_FullValues.push_back(fAlpha);
        m_Anims[OPACITY_G].m_FullValues.push_back(fAlpha);
        m_Anims[OPACITY_B].m_FullValues.push_back(fAlpha);
    }

    /**
    スペキュラーカラーアニメーションを設定する
    */
    void SetSpecularAnim(nn::gfx::tool::dcc::RVec4 col)
    {
        m_Anims[SPECULAR_R].m_FullValues.push_back(col.x);
        m_Anims[SPECULAR_G].m_FullValues.push_back(col.y);
        m_Anims[SPECULAR_B].m_FullValues.push_back(col.z);
    }

    /**
    エミッションカラーアニメーションを設定する
    */
    void SetEmissionAnim(nn::gfx::tool::dcc::RVec4 col)
    {
        m_Anims[EMISSION_R].m_FullValues.push_back(col.x);
        m_Anims[EMISSION_G].m_FullValues.push_back(col.y);
        m_Anims[EMISSION_B].m_FullValues.push_back(col.z);
    }

    /**
    アンビエントカラーアニメーションを設定する
    */
    void SetAmbientAnim(nn::gfx::tool::dcc::RVec4 col)
    {
        m_Anims[AMBIENT_R].m_FullValues.push_back(col.x);
        m_Anims[AMBIENT_G].m_FullValues.push_back(col.y);
        m_Anims[AMBIENT_B].m_FullValues.push_back(col.z);
    }


    //! @brief 異方性反射を使用するかどうかのフラグを設定します。
    void EnableAnisotropic( bool flag )
    {
        m_IsAnisotropic = flag;
    }

    //! @brief スペキュラを使用するかどうかのフラグを設定します。
    void EnableSpecular( bool flag )
    {
        m_IsSpecularEnable = flag;
    }

    //! @brief スペキュラの強さを設定します。
    void SetShiness( float fShininess )
    {
        m_Shininess = fShininess;
    }

    //! @brief マテリアルがテクスチャパターンアニメーションを持っているかチェックします。
    bool IsContainingTexPatAnimData( void ) const;

    //! @brief マテリアルがテクスチャSRTアニメーションを持っているかチェックします。
    bool IsContainingTexSRTAnimData( void ) const;

    //! @brief マテリアルがカラーアニメーションを持っているかチェックします。
    bool IsContainingColorAnimData( void ) const;

    //! @brief Bakeしたアニメーションをカーブに変換します。
    void PrepareAnimations( const nn::gfx::tool::dcc::RExpOpt& rOpt );

    //! @brief カラーアニメーションを 1 つ出力します。
    void OutputColorAnims(std::basic_ostream<char>& os, std::vector<RDataStream>& dataStreams, const int tc) const;

    /**
    ユーザーデータを追加
    \param[in] rData 追加されるユーザーデータ
    \par 解説：
    　このノードにユーザーデータを追加します。
    */
    void addUserData( nn::gfx::tool::dcc::RUserData&	rData ){ m_UserDatas.push_back( rData ); }

public:
    int m_OrgIndex; //!< オリジナルのインデックスです。sceneMaterials 内のインデックスです。
    int m_Index; //!< 出力用のインデックスです。nw4fmax::outMaterials 内のインデックスです。

    //このマテリアルで使用するCafeのUV座標を表すフラグ
    bool	m_UseCafeUV[nn::gfx::tool::dcc::RPrimVtx::VTX_TEX_MAX];
    /*
    Sampler が参照するUV座標を指定する。0から(nn::gfx::tool::dcc::RPrimVtx::VTX_TEX_MAX-)で指定
    この値を元に、FShape::deleteUnusedUV 内で  RSampler.m_OriginalTexsrt.m_HintIndex が
    設定される。
    */
    nn::gfx::tool::dcc::RIntArray m_SamplerReferUVs;
    //-----------------------------------------------------------------------------
    // カラーアニメーション

    //! @brief カラーアニメーションが設定されていれば true です。
    bool m_ColAnimFlag;

    //! @brief カラーアニメーションのパラメータに対応するアニメーションカーブ配列です。
    nn::gfx::tool::dcc::RAnimCurve m_Anims[nn::gfx::tool::dcc::RMaterial::COLOR_PARAM_COUNT];
};



/******************************************************************************
    end name space utility
******************************************************************************/
}}}}} // namespace utility
