﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/******************************************************************************
    include
******************************************************************************/
#include "DccCommon.h"
#include "DccAnimation.h"

/******************************************************************************
    begin name space utility
******************************************************************************/
namespace nn {
namespace gfx {
namespace tool {
namespace dcc {
namespace utility {


/*!--------------------------------------------------------------------------*
    @brief カメラ情報を出力するためのクラスです。
    ライト情報の中間ファイルへの出力には RSceneCameras を使用します。
 *---------------------------------------------------------------------------*/
class RCamera
{
public:
    // シーンの拡大率
    static float mMagnify;
    //! シーン全体の拡大・縮小率を設定する
    static void SetMagnify(float magnify)
    {
        mMagnify = magnify;
    }

    //! コンストラクタです。
    RCamera();

    //!	@brief カメラアニメーションの出力準備をします。
    //!	アニメーションカーブデータを確認し、出力する必要のあるデータを判断します。
    //!	OutAnim を呼ぶ前に一度呼び出す必要があります。
    //!	@param[in] loop ループ再生フラグ
    //!	@param[in] toleranceTrans 移動アニメーションの許容誤差
    //!	@param[in] toleranceRotate 回転アニメーションの許容誤差
    //!	@param[in] toleranceScale スケールアニメーションの許容誤差
    void PrepareAnimations( const nn::gfx::tool::dcc::RExpOpt& rOpt );

    //! @brief <camera_anim> 要素を出力します。
    //!
    //! @param[in,out] os 出力ストリームです。
    //! @param[in,out] dataStreams データ列配列です。
    //! @param[in] tabCount <camera_anim> 要素のインデントに必要なタブの数です。
    //! @param[in] index <camera_anim_array> 内でのインデックスです。
    void OutAnim(
        std::ostream& os,
        nn::gfx::tool::dcc::RDataStreamArray& dataStreams,
        const int tabCount,
        const int index,
        const nn::gfx::tool::dcc::RExpOpt& rOpt) const;

    //! 名前を指定します。
    void SetName(const char* name) { m_Name = name; }
    //! 名前を取得します。
    const char* GetName() const { return m_Name.c_str(); }

    //! カメラタイプを指定します。
    void SetRotateMode(nn::gfx::tool::dcc::RCamera::RotateMode mode) { m_CameraRotMode= mode; }
    //! カメラタイプを取得します。
    nn::gfx::tool::dcc::RCamera::RotateMode GetRotateMode() const { return m_CameraRotMode; }
    //! プロジェクション方法を指定します。
    void SetProjectionMode(nn::gfx::tool::dcc::RCamera::ProjectionMode mode) { m_CameraProjMode= mode; }
    //! プロジェクション方法を取得します。
    nn::gfx::tool::dcc::RCamera::ProjectionMode GetProjectionMode() const { return m_CameraProjMode; }

    //! @brief カメラの可視性を指定します。
    //! カメラの可視性を指定します。
    //! カメラに親のノードが設定されている場合は親の可視性を引き継いだ値を指定します。
    //! @param[in] visible	カメラの可視状態です。
    void SetBranchVisible(bool visible) { mIsBranchVisible = visible; }

    void SetRotateAnim(nn::gfx::tool::dcc::RVec3 rot)
    {
        m_Anims[nn::gfx::tool::dcc::RCamera::ROTATE_X].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(rot.x));
        m_Anims[nn::gfx::tool::dcc::RCamera::ROTATE_Y].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(rot.y));
        m_Anims[nn::gfx::tool::dcc::RCamera::ROTATE_Z].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(rot.z));
    }
    void SetTransAnim(nn::gfx::tool::dcc::RVec3 pos)
    {
        pos *= mMagnify;
        m_Anims[nn::gfx::tool::dcc::RCamera::POSITION_X].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(pos.x));
        m_Anims[nn::gfx::tool::dcc::RCamera::POSITION_Y].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(pos.y));
        m_Anims[nn::gfx::tool::dcc::RCamera::POSITION_Z].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(pos.z));
    }
    void SetTargetAnim(nn::gfx::tool::dcc::RVec3 pos)
    {
        pos *= mMagnify;
        m_Anims[nn::gfx::tool::dcc::RCamera::AIM_X].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(pos.x));
        m_Anims[nn::gfx::tool::dcc::RCamera::AIM_Y].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(pos.y));
        m_Anims[nn::gfx::tool::dcc::RCamera::AIM_Z].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(pos.z));
    }

    void SetTwistAnim(float v)
    { m_Anims[nn::gfx::tool::dcc::RCamera::TWIST].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(v));}
    void SetNearAnim(float v)
    { m_Anims[nn::gfx::tool::dcc::RCamera::NEAR_CLIP].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(v * mMagnify));}
    void SetFarAnim(float v)
    { m_Anims[nn::gfx::tool::dcc::RCamera::FAR_CLIP].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(v * mMagnify));}
    void SetAspectAnim(float v)
    { m_Anims[nn::gfx::tool::dcc::RCamera::ASPECT].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(v));}
    void SetFovYAnim(float v)
    { m_Anims[nn::gfx::tool::dcc::RCamera::PERSP_FOVY].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(v));}
    void SetOrthoHAnim(float v)
    { m_Anims[nn::gfx::tool::dcc::RCamera::ORTHO_HEIGHT].m_FullValues.push_back(nn::gfx::tool::dcc::RSnapToZero(v * mMagnify));}

public:
    std::string m_Name;
    bool mIsBranchVisible;

    nn::gfx::tool::dcc::RCamera::RotateMode m_CameraRotMode;
    nn::gfx::tool::dcc::RCamera::ProjectionMode m_CameraProjMode;

    //	anim
    nn::gfx::tool::dcc::RAnimCurve m_Anims[nn::gfx::tool::dcc::RCamera::PARAM_COUNT];
};


/******************************************************************************
    end name space utility
******************************************************************************/
}}}}} // namespace utility
