﻿/*--------------------------------------------------------------------------------*
Copyright (C)Nintendo All rights reserved.

These coded instructions, statements, and computer programs contain proprietary
information of Nintendo and/or its licensed developers and are protected by
national and international copyright laws. They may not be disclosed to third
parties or copied or duplicated in any form, in whole or in part, without the
prior written consent of Nintendo.

The content herein is highly confidential and should be handled accordingly.
*--------------------------------------------------------------------------------*/


//------------------------------------------------------------------------------
//  SDK ヘッダの include
//------------------------------------------------------------------------------
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/vi.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/gfx/gfx_Variation-api.gl.h>

#if defined( NN_BUILD_CONFIG_OS_WIN )
#include <nn/nn_Windows.h>
#include <GL/glew.h>
#include <shellapi.h>
#endif

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
#include <nv/nv_MemoryManagement.h>
#endif

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
#include <nvnTool/nvnTool_GlslcInterface.h>
#endif
#endif

//------------------------------------------------------------------------------
//  デモ用ヘッダの #include
//------------------------------------------------------------------------------
#include <nns/nac/nac_Pad.h>
#include <nns/nac/nac_Mouse.h>
#include <nns/nac/nac_File.h>

//------------------------------------------------------------------------------
//  固有ヘッダの #include
//------------------------------------------------------------------------------
#include <Descriptor.h>
#include <Grid.h>
#include <Camera.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_FrameBuffer.h>
#include <nns/gfx/gfx_ColorBuffer.h>
#include <nns/gfx/gfx_DepthStencilBuffer.h>

#if defined( NN_BUILD_CONFIG_OS_WIN )
#include <detail/util_PngIO.h>
#endif

#include <CombModelPreview.h>

//------------------------------------------------------------------------------
//  定数・マクロ定義
//------------------------------------------------------------------------------
#ifndef _MAX_PATH
#define _MAX_PATH 260  //NOLINT(preprocessor/const)
#endif


//------------------------------------------------------------------------------
//  グローバル変数定義
//------------------------------------------------------------------------------
nn::gfx::DescriptorPool                     g_SamplerDescriptorPool;
nn::gfx::DescriptorPool                     g_TextureDescriptorPool;
nn::gfx::TextureView                        g_ColorTextureView;
nn::gfx::DescriptorSlot                     g_ColorTextureDescSlot;
nn::gfx::TextureView                        g_DepthStencilTextureView;
nn::gfx::DescriptorSlot                     g_DepthStencilTextureDescSlot;
nns::gfx::FrameBuffer                       g_NodePreviewRenderTarget;
nns::gfx::FrameBuffer                       g_CopyRenderTarget;
nn::gfx::DescriptorSlot                     g_ColorCopyTextureDescSlot;
nn::gfx::DescriptorSlot                     g_DepthStencilCopyTextureDescSlot;
nn::gfx::ViewportScissorState               g_NodePreviewViewportScissor;
nn::gfx::ViewportScissorState               g_2DViewportScissor;
nn::gfx::RasterizerState                    m_ModelRasterizerState;
Camera                                      g_DemoCamera;
nn::util::Vector3fType                      g_CameraPosition;               // カメラ位置
nn::util::Vector3fType                      g_CameraLookAt;                 // カメラ視点
nn::util::Matrix4x4fType                    g_ProjectionMatrix;             // プロジェクション
int                                         g_Width;                        // 画面の横幅です。
int                                         g_Height;                       // 画面の縦幅です。
int                                         g_DeviceWidth;                  // ウィンドウ画面の横幅です。
int                                         g_DeviceHeight;                 // ウィンドウ画面の縦幅です。
int                                         g_ThumsWidth;                   // サムネイルの横幅です。
int                                         g_ThumsHeight;                  // サムネイルの縦幅です。
nns::gfx::GraphicsFramework::FrameworkMode  g_DrawFrameworkMode;
detail::TextureDescriptorIndexAllocator     g_TextureDescPoolAllocator;
detail::SamplerDescriptorIndexAllocator     g_SamplerDescPoolAllocator;

nn::gfx::Buffer                             g_CaptureBuffer;
int                                         g_CaptureIndex;
int                                         g_CapturedFrameNumber;

nns::gfx::PrimitiveRenderer::BlendType      g_BlendType = nns::gfx::PrimitiveRenderer::BlendType_Opacity;
float                                       g_RotateY   = 0.001f;
int                                         g_PreviewModelIndex = 0;
bool										g_Initialized = false;

//------------------------------------------------------------------------------
//  フォント用定義
//------------------------------------------------------------------------------
nns::gfx::GraphicsFramework::DebugFontTextWriter g_Writer;

//------------------------------------------------------------------------------
//  プラットフォーム固有の変数定義
//------------------------------------------------------------------------------
#if defined( NN_BUILD_CONFIG_OS_WIN )
HWND                                    g_HWnd;
WNDPROC                                 g_CustomWndProc;
nns::nac::Mouse                         g_Mouse;
char                                    g_WorkingDir[ _MAX_PATH ];
TCHAR                                   g_BaseDirectory[ _MAX_PATH ];
nn::lmem::HeapHandle                    g_HeapHandle;
#endif

nns::nac::Pad                           g_Pad;                          // デバッグ用パッド管理クラスのインスタンスへのポインタです。
bool                                    g_IsSetFocus;                   // ウィンドウのフォーカス監視

char                                    g_ScreenCapturedFilePath[ _MAX_PATH ];
bool                                    g_ReservationScreenCapture = false;



//------------------------------------------------------------------------------
//  GraphicsFramework
//------------------------------------------------------------------------------
nns::gfx::GraphicsFramework::FrameworkInfo g_GraphicsFrameworkInfo;
nns::gfx::GraphicsFramework                g_GraphicsFramework;


#if NN_GFX_IS_TARGET_NVN && defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
//------------------------------------------------------------------------------
// GLSLC 用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* GlslcAllocateFunction( size_t size, size_t alignment, void* userPtr )
{
    NN_UNUSED( userPtr );
    return aligned_alloc( alignment, size );
}

static void GlslcFreeFunction( void* addr, void* userPtr )
{
    NN_UNUSED( userPtr );
    free( addr );
}

static void* GlslcReallocateFunction( void* addr, size_t newSize, void* userPtr )
{
    NN_UNUSED( userPtr );
    return realloc( addr, newSize );
}
#endif

//!--------------------------------------------------------------------------------------
//! @brief GpuBuffer 用メモリ確保関数
//!--------------------------------------------------------------------------------------
void* GpuBufferAllocateFunction( size_t size, size_t alignment, void* pUserData )
{
    NN_SDK_ASSERT_NOT_NULL( pUserData );

    // ユーザーデータポインタでアロケータを渡してもらう.
    nns::nac::MemoryAllocator*    pAllocator = static_cast<nns::nac::MemoryAllocator*>( pUserData );

    return pAllocator->Alloc( size, alignment );
}

//!--------------------------------------------------------------------------------------
//! @brief GpuBuffer 用メモリ開放関数
//!--------------------------------------------------------------------------------------
void GpuBufferDeallocateFunction( void* ptr, void* pUserData )
{
    NN_SDK_ASSERT_NOT_NULL( pUserData );

    // ユーザーデータポインタでアロケータを渡してもらう.
    nns::nac::MemoryAllocator*    pAllocator = static_cast<nns::nac::MemoryAllocator*>( pUserData );

    pAllocator->Free( ptr );
}

//------------------------------------------------------------------------------
//  入力関連 初期化処理
//------------------------------------------------------------------------------
void InitializeInputInterface()
{
#if defined( NN_BUILD_CONFIG_OS_WIN )
    g_Pad.Initialize( g_HWnd );
    g_Pad.SetUseKeyboardForcus( true );
#else
    g_Pad.Initialize();
#endif

#if defined( NN_BUILD_CONFIG_OS_WIN )
    g_Mouse.Initialize( g_HWnd );
    g_Mouse.SetUseKeyboardForcus( true );
#endif
}

//------------------------------------------------------------------------------
//  入力関連 更新処理
//------------------------------------------------------------------------------
void UpdateInputInterface()
{
#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
    g_Pad.SetPadState( true );
#endif
    g_Pad.Update( g_IsSetFocus );

#if defined( NN_BUILD_CONFIG_OS_WIN )
    g_Mouse.Update( g_IsSetFocus );
#endif
}

//------------------------------------------------------------------------------
//  入力関連 終了処理
//------------------------------------------------------------------------------
void FinalizeInputInterface()
{
    g_Pad.Finalize();
#if defined( NN_BUILD_CONFIG_OS_WIN )
    g_Mouse.Finalize();
#endif
}

//---------------------------------------------------------------------------
//  Pad 管理クラスのインスタンスへのポインタを取得します。
//---------------------------------------------------------------------------
nns::nac::Pad* GetPad() { return &g_Pad; }

//---------------------------------------------------------------------------
//  マウス管理クラスのインスタンスへのポインタを取得します。
//---------------------------------------------------------------------------
#if defined( NN_BUILD_CONFIG_OS_WIN )
nns::nac::Mouse* GetMouse() { return &g_Mouse; }
#else
nns::nac::Mouse* GetMouse() { return NULL; }
#endif

//------------------------------------------------------------------------------
//  メモリ関数の定義
//------------------------------------------------------------------------------
template< typename T >
T* AlignedAlloc( void** ppMemory, size_t size, size_t alignment = 1 )
{
    *ppMemory = reinterpret_cast< void* >( (
        reinterpret_cast< uintptr_t >( *ppMemory ) + alignment - 1 ) & ~( alignment - 1 ) );
    T* ret = static_cast< T* >( *ppMemory );
    *ppMemory = static_cast< uint8_t* >( *ppMemory ) + size;
    return ret;
}

#if defined( NN_BUILD_CONFIG_OS_WIN )

enum
{
    MESSAGE_TO_COMBEDITOR_VIEWER_INITIALIZE = 1,
    MESSAGE_TO_COMBEDITOR_PREVIEW_PIC_PATH = 2,
    MESSAGE_TO_COMBEDITOR_TEXTURE_RELOAD = 3,
    MESSAGE_TO_COMBEDITOR_PREVIEW_SHADER_RELOAD = 4,
    MESSAGE_TO_COMBEDITOR_OUTPUT_SHADER_RELOAD = 5,
    MESSAGE_TO_COMBEDITOR_TEXTURE_RESET = 6,
};

enum
{
    MESSAGE_FROM_COMBEDITOR_CONNECT = 0,
    MESSAGE_FROM_COMBEDITOR_OUTPUTSHADER = 1,
    MESSAGE_FROM_COMBEDITOR_PREVIEW = 2,
    MESSAGE_FROM_COMBEDITOR_PARAM = 3,
};

HWND g_HwndTarget = NULL;

void SendMessageToCombinerEditor( int message, void* buffer, DWORD bufferSize )
{
    if ( g_HwndTarget != NULL )
    {
        COPYDATASTRUCT data;
        data.dwData = message;
        data.cbData = bufferSize;
        data.lpData = buffer;
        SendMessage( g_HwndTarget, WM_COPYDATA, ( WPARAM )g_HWnd, ( LPARAM )&data );
    }
}


//------------------------------------------------------------------------------
//  WM_COPYDATA受信処理
//------------------------------------------------------------------------------
LRESULT CopyDataProc( HWND hWnd, WPARAM wp, LPARAM lp )
{
    NN_UNUSED( hWnd );
    NN_UNUSED( wp );

    COPYDATASTRUCT* pCopyData = reinterpret_cast< COPYDATASTRUCT* >( lp );

    char* receiveData = reinterpret_cast< char* >( pCopyData->lpData );

    int type;
    memcpy( &type, receiveData, sizeof( uint32_t ) );
    receiveData += sizeof( uint32_t );

    if ( type == MESSAGE_FROM_COMBEDITOR_CONNECT )
    {
        if ( g_Initialized )
        {
            memcpy( &g_HwndTarget, receiveData, sizeof( uint32_t ) );
            SendMessageToCombinerEditor( MESSAGE_TO_COMBEDITOR_VIEWER_INITIALIZE, nullptr, 0 );
        }
    }
    else if ( type == MESSAGE_FROM_COMBEDITOR_OUTPUTSHADER )
    {
        detail::CombinerPreviewer::GetInstance()->SetReloadOutputShaderFilePath( receiveData );

        if ( detail::CombinerPreviewer::GetInstance()->IsOutputShaderReload() )
        {
            SendMessageToCombinerEditor( MESSAGE_TO_COMBEDITOR_OUTPUT_SHADER_RELOAD, nullptr, 0 );
        }
    }
    else if ( type == MESSAGE_FROM_COMBEDITOR_PREVIEW )
    {
        detail::CombinerPreviewer::GetInstance()->SetReloadPreviewShaderFilePath( receiveData );
    }
    else if ( type == MESSAGE_FROM_COMBEDITOR_PARAM )
    {
        detail::CombinerPreviewer::GetInstance()->ReceiveUpdateParameter( receiveData );
    }

    return 0;
}


//------------------------------------------------------------------------------
//  ウィンドウサイズの変更(WM_SIZE)
//------------------------------------------------------------------------------
LRESULT ResizeWindowProc( HWND hWnd, WPARAM wp, LPARAM lp )
{
    NN_UNUSED( wp );
    NN_UNUSED( lp );

    RECT rw, rc;

    // ウィンドウのクライアント領域と非クライアント領域の取得
    GetWindowRect( hWnd, &rw );
    GetClientRect( hWnd, &rc );

    g_DeviceWidth  = rc.right;
    g_DeviceHeight = rc.bottom;

    return 0;
}


LRESULT CALLBACK CustomWindowProc( HWND hWnd, UINT msg, WPARAM wp, LPARAM lp )
{
    switch ( msg )
    {
    case WM_SIZE:
        return ResizeWindowProc( hWnd, wp, lp );
    case WM_COPYDATA:
        return CopyDataProc( hWnd, wp, lp );
    case WM_KILLFOCUS:
        g_IsSetFocus = false;
        break;
    case WM_SETFOCUS:
        g_IsSetFocus = true;
        break;
    case WM_DROPFILES:
    {
        char szFileName[ _MAX_PATH ];
        HDROP  hDrop = ( HDROP )wp;
        UINT uFileNo = DragQueryFile( ( HDROP )wp, 0xFFFFFFFF, NULL, 0 );
        for ( int i = 0; i < ( int )uFileNo; i++ )
        {
            DragQueryFileA( hDrop, i, szFileName, sizeof( szFileName ) );
            if ( strncmp( &szFileName[ strlen( szFileName ) - 4 ], "fmdb", 4 ) == 0 )
            {
                char outputDir[ _MAX_PATH ];
                sprintf_s( outputDir, "%s\\ScreenCap\\", g_WorkingDir );
                detail::CombinerPreviewer::GetInstance()->LoadG3dModelBinary( szFileName, outputDir );
            }
        }
        break;
    }

    default:
        break;
    }
    return CallWindowProc( g_CustomWndProc, hWnd, msg, wp, lp );
}

#endif

//------------------------------------------------------------------------------
//  Gfx初期処理
//------------------------------------------------------------------------------
void InitializeGfx( nn::gfx::Device* pDevice, nns::nac::MemoryAllocator* pAllocator, size_t poolSize, int width, int height )
{
    NN_SDK_ASSERT_NOT_NULL( pAllocator );
    NN_SDK_ASSERT_GREATER( width, 0 );
    NN_SDK_ASSERT_GREATER( width, 0 );
    NN_SDK_ASSERT_GREATER( height, 0 );
    NN_UNUSED( poolSize );
    NN_UNUSED( width );
    NN_UNUSED( height );


#if defined( NN_BUILD_CONFIG_OS_WIN )
    nn::vi::GetNativeWindow( ( nn::vi::NativeWindowHandle* )&g_HWnd, g_GraphicsFramework.GetLayer() );
    RECT rw, rc;
    ::GetWindowRect( g_HWnd, &rw );
    ::GetClientRect( g_HWnd, &rc );

    // クライアント領域と非クライアント領域の差分
    const int offsetW = ( rw.right - rw.left ) - ( rc.right - rc.left );
    const int offsetH = ( rw.bottom - rw.top ) - ( rc.bottom - rc.top );

    // 2015/4/1現在、gfxではウインドウサイズが設定出来ないようなので
    // Win32APIを呼び出して強制的にウインドウサイズを設定
    // 将来的には削除予定
    SetWindowPos( static_cast<HWND>( g_HWnd ), NULL, 0, 0, g_DeviceWidth + offsetW, g_DeviceHeight + offsetH, SWP_NOMOVE | SWP_NOZORDER );

    // ウィンドウのサブクラス化
    HWND hWnd = static_cast<HWND>( g_HWnd );
    g_CustomWndProc = reinterpret_cast<WNDPROC>(SetWindowLongPtr(hWnd, GWLP_WNDPROC, reinterpret_cast<LONG_PTR>(CustomWindowProc)));

    LONG lStyle = GetWindowLong( g_HWnd, GWL_STYLE );
    lStyle |= WS_SIZEBOX;
    lStyle |= WS_THICKFRAME;
    lStyle = SetWindowLong( g_HWnd, GWL_STYLE, lStyle );

    // ウィンドウ名を設定
    SetWindowText( hWnd, L"CombinerViewer " );
#endif

    // テクスチャサンプラプールを初期化
    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_Sampler );
        info.SetSlotCount( 1024 );

        ptrdiff_t offset = g_GraphicsFramework.AllocatePoolMemory(
            nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
            nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( pDevice, info ),
            nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( pDevice, info ) );

        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( pDevice, info );

        g_SamplerDescriptorPool.Initialize( pDevice, info, g_GraphicsFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer ), offset, size );

        // サンプラ・ディスクリプタスロット・アロケータの初期化
        int baseIndex = 256;
        int staticSlotCount = 64;
        int transientSlotCount = 256;
        g_SamplerDescPoolAllocator.Initialize( &g_SamplerDescriptorPool, baseIndex, staticSlotCount, transientSlotCount );
    }

    // テクスチャデスクリプタプールを初期化
    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_TextureView );
        info.SetSlotCount( 1024 * 32 );

        ptrdiff_t offset = g_GraphicsFramework.AllocatePoolMemory(
            nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
            nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( pDevice, info ),
            nn::gfx::DescriptorPool::GetDescriptorPoolAlignment( pDevice, info ) );

        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( pDevice, info );

        g_TextureDescriptorPool.Initialize( pDevice, info, g_GraphicsFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer ), offset, size );

        // テクスチャ・ディスクリプタスロット・アロケータの初期化
        int baseIndex = 256;
        int staticSlotCount = 64;
        int transientSlotCount = 1024;
        g_TextureDescPoolAllocator.Initialize( &g_TextureDescriptorPool, baseIndex, staticSlotCount, transientSlotCount );
    }

    // カラーバッファのテクスチャビューを初期化
    {
        nn::gfx::TextureView::InfoType viewInfo;
        viewInfo.SetDefault();
        viewInfo.SetImageDimension( nn::gfx::ImageDimension_2d );
        viewInfo.SetImageFormat( nn::gfx::ImageFormat_R16_G16_B16_A16_Float );
        viewInfo.SetTexturePtr( g_GraphicsFramework.GetColorBuffer() );
        g_ColorTextureView.Initialize( pDevice, viewInfo );

        // デスクリプタスロット設定
        g_TextureDescPoolAllocator.AllocateStaticTextureDescriptorSlot( &g_ColorTextureDescSlot, g_ColorTextureView, NULL );
    }

    {
        nns::gfx::FrameBuffer::InfoType info( g_Width, g_Height );

        info.SetDefault();
        info.SetColorBufferFormat( nn::gfx::ImageFormat_R16_G16_B16_A16_Float );
        info.SetDepthBufferEnabled();
        info.SetDepthBufferFormat( nn::gfx::ImageFormat_D32_Float );
        info.SetWidth( g_Width );
        info.SetHeight( g_Height );

        ptrdiff_t offset = g_GraphicsFramework.AllocatePoolMemory(
            nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,
            nns::gfx::FrameBuffer::GetRequiredMemoryPoolSize( pDevice, info ),
            nns::gfx::FrameBuffer::GetMemoryPoolAlignment( pDevice, info ) );

        g_CopyRenderTarget.Initialize( pDevice, &info,
            g_GraphicsFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget ),
            offset, GpuBufferAllocateFunction, GpuBufferDeallocateFunction, pAllocator );

        // デスクリプタスロット設定
        g_TextureDescPoolAllocator.AllocateStaticTextureDescriptorSlot( &g_ColorCopyTextureDescSlot, *g_CopyRenderTarget.GetColorBuffer()->GetTextureView(), NULL );
        g_TextureDescPoolAllocator.AllocateStaticTextureDescriptorSlot( &g_DepthStencilCopyTextureDescSlot, *g_CopyRenderTarget.GetDepthBuffer()->GetTextureView(), NULL );
    }

    {
        nns::gfx::FrameBuffer::InfoType info( g_ThumsWidth, g_ThumsHeight );

        info.SetDefault();
        info.SetColorBufferFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb );
        info.SetDepthBufferEnabled();
        info.SetDepthBufferFormat( nn::gfx::ImageFormat_D16_Unorm );
        info.SetWidth( g_ThumsWidth );
        info.SetHeight( g_ThumsHeight );

        ptrdiff_t offset = g_GraphicsFramework.AllocatePoolMemory(
            nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,
            nns::gfx::FrameBuffer::GetRequiredMemoryPoolSize( pDevice, info ),
            nns::gfx::FrameBuffer::GetMemoryPoolAlignment( pDevice, info ) );

        g_NodePreviewRenderTarget.Initialize( pDevice, &info,
            g_GraphicsFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget ),
            offset, GpuBufferAllocateFunction, GpuBufferDeallocateFunction, pAllocator );
    }

    // 深度ステンシルテクスチャビューを初期化
    {
        nn::gfx::TextureView::InfoType texViewInfo;
        texViewInfo.SetDefault();
        texViewInfo.SetImageDimension( nn::gfx::ImageDimension_2d );
        texViewInfo.SetImageFormat( nn::gfx::ImageFormat_D32_Float );
        texViewInfo.SetTexturePtr( g_GraphicsFramework.GetDepthStencilBuffer() );
        g_DepthStencilTextureView.Initialize( pDevice, texViewInfo );

        // デスクリプタスロット設定
        g_TextureDescPoolAllocator.AllocateStaticTextureDescriptorSlot( &g_DepthStencilTextureDescSlot, g_DepthStencilTextureView, NULL );
    }

    // モデル表示用ラスタライザステートを初期化
    {
        nn::gfx::RasterizerState::InfoType info;
        info.SetDefault();
        info.SetCullMode( nn::gfx::CullMode_Back );
        info.SetPrimitiveTopologyType( nn::gfx::PrimitiveTopologyType_Triangle );
        info.SetScissorEnabled( true );
        info.SetDepthClipEnabled( false );
        m_ModelRasterizerState.Initialize( pDevice, info );
    }

    // ビューポートシザーを初期化
    {
        nn::gfx::ViewportScissorState::InfoType info;
        info.SetDefault();
        info.SetScissorEnabled( true );
        nn::gfx::ViewportStateInfo viewportInfo;
        {
            viewportInfo.SetDefault();
            viewportInfo.SetWidth( static_cast< float >( g_ThumsWidth ) );
            viewportInfo.SetHeight( static_cast< float >( g_ThumsHeight ) );

        }
        nn::gfx::ScissorStateInfo scissorInfo;
        {
            scissorInfo.SetDefault();
            scissorInfo.SetWidth( g_ThumsWidth );
            scissorInfo.SetHeight( g_ThumsHeight );
        }
        info.SetViewportStateInfoArray( &viewportInfo, 1 );
        info.SetScissorStateInfoArray( &scissorInfo, 1 );
        g_NodePreviewViewportScissor.Initialize( pDevice, info );
    }

    {
        nn::gfx::ViewportScissorState::InfoType info;
        info.SetDefault();
        info.SetScissorEnabled( true );
        nn::gfx::ViewportStateInfo viewportInfo;
        {
            viewportInfo.SetDefault();
            viewportInfo.SetWidth( static_cast< float >( g_DeviceWidth ) );
            viewportInfo.SetHeight( static_cast< float >( g_DeviceHeight ) );

        }
        nn::gfx::ScissorStateInfo scissorInfo;
        {
            scissorInfo.SetDefault();
            scissorInfo.SetWidth( g_DeviceWidth );
            scissorInfo.SetHeight( g_DeviceHeight );
        }
        info.SetViewportStateInfoArray( &viewportInfo, 1 );
        info.SetScissorStateInfoArray( &scissorInfo, 1 );
        g_2DViewportScissor.Initialize( pDevice, info );
    }



#if defined( NN_BUILD_CONFIG_OS_WIN )
    // キャプチャフレーム用のバッファ初期化
    {
        g_CaptureIndex = 0;
        g_CapturedFrameNumber = 0;

        size_t imageBufferSize = g_ThumsWidth * g_ThumsHeight * 16;
        nn::gfx::Buffer::InfoType info;
        info.SetDefault();
        info.SetSize( imageBufferSize );
        info.SetGpuAccessFlags( nn::gfx::GpuAccess_ColorBuffer );

        ptrdiff_t offset = g_GraphicsFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer,
            info.GetSize(),
            nn::gfx::Buffer::GetBufferAlignment( pDevice, info ) );

        g_CaptureBuffer.Initialize(
            pDevice,
            info,
            g_GraphicsFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer ),
            offset,
            g_GraphicsFramework.GetMemoryPoolAllocator( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer )->GetSize() );
    }
#endif
}

//------------------------------------------------------------------------------
//  Gfx終了処理
//------------------------------------------------------------------------------
void FinalizeGfx( nn::gfx::Device* pDevice, nns::nac::MemoryAllocator* pAllocator )
{
    NN_SDK_ASSERT_NOT_NULL( pAllocator );

    m_ModelRasterizerState.Finalize( pDevice );
    g_DepthStencilTextureView.Finalize( pDevice );
    g_ColorTextureView.Finalize( pDevice );
    g_TextureDescriptorPool.Finalize( pDevice );
    g_SamplerDescriptorPool.Finalize( pDevice );

    g_CopyRenderTarget.Finalize( pDevice, GpuBufferAllocateFunction, GpuBufferDeallocateFunction, pAllocator );
    g_NodePreviewRenderTarget.Finalize( pDevice, GpuBufferAllocateFunction, GpuBufferDeallocateFunction, pAllocator );

#if defined( NN_BUILD_CONFIG_OS_WIN )
    g_CaptureBuffer.Finalize( pDevice );
#endif
}


//------------------------------------------------------------------------------
//  PrimitiveRenderer 初期化処理
//------------------------------------------------------------------------------
nns::gfx::PrimitiveRenderer::Renderer* g_pPrimitiveRenderer;
void InitializePrimitiveRenderer( nn::gfx::Device* pDevice, nns::nac::MemoryAllocator* pAllocator )
{
    NN_SDK_ASSERT_NOT_NULL( pDevice );
    NN_SDK_ASSERT_NOT_NULL( pAllocator );

    nns::gfx::PrimitiveRenderer::RendererInfo info;
    info.SetDefault();
    info.SetAllocator( GpuBufferAllocateFunction, pAllocator );
    info.SetAdditionalBufferSize( 1024 * 1024 );

    // ダブルバッファで運用する場合は、SetMultiBufferQuantity で 2 を指定し、
    // プリミティブレンダラ内部のバッファをダブルにしたうえで、
    // g_pPrimitiveRenderer->Update(); で利用するバッファを選択( 0 -> 1 -> 0 -> 1 )する
    info.SetMultiBufferQuantity( 2 );

    // PrimitiveRendererのインスタンス
    g_pPrimitiveRenderer = nns::gfx::PrimitiveRenderer::CreateRenderer( pDevice, info );
    g_pPrimitiveRenderer->SetScreenWidth( g_Width );
    g_pPrimitiveRenderer->SetScreenHeight( g_Height );
}

//------------------------------------------------------------------------------
//  PrimitiveRenderer 破棄処理
//------------------------------------------------------------------------------
void FinalizePrimitiveRenderer( nn::gfx::Device* pDevice, nns::nac::MemoryAllocator* pAllocator )
{
    NN_SDK_ASSERT_NOT_NULL( pDevice );

    nns::gfx::PrimitiveRenderer::DestroyRenderer( g_pPrimitiveRenderer, pDevice, GpuBufferDeallocateFunction, pAllocator );
}


#if defined( NN_BUILD_CONFIG_OS_WIN )
//------------------------------------------------------------------------------
//      スクリーンキャプチャ処理
//------------------------------------------------------------------------------
void SaveCurrentBufferAsPngFile( const char* fileName, unsigned int width, unsigned int height, nn::lmem::HeapHandle heapHandle, bool linear = 0 )
{
    typedef struct
    {
        unsigned char r;
        unsigned char g;
        unsigned char b;
        unsigned char a;
    }Rgb;

    NN_UNUSED( linear );

    // キャプチャ画像のファイル名を設定
    char filePath[ _MAX_PATH ];
    sprintf_s( filePath, "%s\\ScreenCap\\%s", g_WorkingDir, fileName );

    // キャプチャ画像を取得
    GLubyte* pBuffer = static_cast<GLubyte*>( g_CaptureBuffer.Map() );
    g_CaptureBuffer.InvalidateMappedRange( 0, 4 * g_Width * g_Height );

    //  RGB情報の書き込み
    int      real_width = width * 4 + width % 4;
    uint8_t* bmp_data = new uint8_t[ width * height * 4 ];
    Rgb*     imagePtr = reinterpret_cast<Rgb*>( pBuffer );

    for ( int i = 0; i < ( int )height; i++ )
    {
        for ( int j = 0; j < ( int )width; j++ )
        {
            /* キャプチャ画像の色が正しくない為、コメントアウトします。
            if ( linear )
            {
            double dr = imagePtr[i * width + j].r/255.0;
            double dg = imagePtr[i * width + j].g/255.0;
            double db = imagePtr[i * width + j].b/255.0;
            double da = imagePtr[i * width + j].a/255.0;
            bmp_data[(i * (width*3)) + (j*3)]      = static_cast<uint8_t>( pow( dr, 1.0/2.2 ) * 255.f );
            bmp_data[(i * (width*3)) + (j*3) + 1]  = static_cast<uint8_t>( pow( dg, 1.0/2.2 ) * 255.f );
            bmp_data[(i * (width*3)) + (j*3) + 2]  = static_cast<uint8_t>( pow( db, 1.0/2.2 ) * 255.f );
            bmp_data[(i * (width*3)) + (j*3) + 3]  = static_cast<uint8_t>( pow( da, 1.0/2.2 ) * 255.f );
            }
            else
            */
            {
                bmp_data[ ( i * ( width * 3 ) ) + ( j * 3 ) ] = imagePtr[ i * width + j ].r;
                bmp_data[ ( i * ( width * 3 ) ) + ( j * 3 ) + 1 ] = imagePtr[ i * width + j ].g;
                bmp_data[ ( i * ( width * 3 ) ) + ( j * 3 ) + 2 ] = imagePtr[ i * width + j ].b;
                bmp_data[ ( i * ( width * 3 ) ) + ( j * 3 ) + 3 ] = imagePtr[ i * width + j ].a;
            }
        }

        for ( int j = width * 4; j < real_width; j++ )
        {
            bmp_data[ ( i * width ) + j ] = 0;
        }
    }

    // キャプチャ画像を保存
    detail::VfxViewerUtilPngIO::WritePng( filePath, ( uint8_t* )bmp_data, heapHandle, width, height, 8, detail::VfxViewerUtilPngIO::PNG_COLOR_TYPE_RGB, 0, 0, 0 );

    g_CaptureBuffer.Unmap();
    delete bmp_data;
}

void PushCaptureCommand( nn::gfx::CommandBuffer* pCommandBuffer, const TCHAR* baseDirectory )
{
    // Rボタン(キー)でスクリーンキャプチャを行う
    bool holdR = g_Pad.IsHold( nns::nac::Pad::MASK_R );

    if ( !holdR && !g_ReservationScreenCapture )
    {
        return;
    }

    int interval = 4;

    // Shift + Rボタン(キー)で連続キャプチャを行う
    if ( g_Pad.IsHold( nns::nac::Pad::MASK_LS_SHIFT ) || g_Pad.IsHold( nns::nac::Pad::MASK_RS_SHIFT ) )
    {
        interval = 1;
    }

    NN_SDK_LOG( "Capture Frame: \n" );

    WIN32_FIND_DATA fd = { 0 };

    SetCurrentDirectory( baseDirectory );
    if ( FindFirstFile( L"ScreenCap", &fd ) == INVALID_HANDLE_VALUE )
    {
        CreateDirectory( L"ScreenCap", NULL );
    }
    SetCurrentDirectory( L"ScreenCap" );

    if ( g_CaptureIndex == 0 )
    {
        // *.pngファイルを削除する
        {
            wchar_t searchBuf[ _MAX_PATH ];
            HANDLE hFind = INVALID_HANDLE_VALUE;

            wsprintf( searchBuf, L"%s", L"*.png" );
            hFind = ( FindFirstFile( searchBuf, &fd ) );
            if ( hFind != INVALID_HANDLE_VALUE )
            {
                do {
                    if ( !( fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY ) )
                    {
                        if ( wcsstr( fd.cFileName, L".png" ) != NULL )
                        {
                            DeleteFile( fd.cFileName );
                        }
                    }
                } while ( FindNextFile( hFind, &fd ) );
            }
            FindClose( hFind );
        }
    }

    // baseDirectoryに戻します。
    SetCurrentDirectory( baseDirectory );

    // ここではキャプチャをしないのでg_CaptureIndexで一つ前のタイミングでコマンドを積みます。
    nn::gfx::BufferTextureCopyRegion region;
    region.SetDefault();
    region.SetBufferImageWidth( g_ThumsWidth );
    region.SetBufferImageHeight( g_ThumsHeight );
    region.EditTextureCopyRegion().SetWidth( g_ThumsWidth );
    region.EditTextureCopyRegion().SetHeight( g_ThumsHeight );

    pCommandBuffer->CopyImageToBuffer( &g_CaptureBuffer, g_NodePreviewRenderTarget.GetColorBuffer()->GetTexture(), region );

    if ( g_ReservationScreenCapture )
    {
        g_CapturedFrameNumber = g_CaptureIndex - 1;
    }
    g_CaptureIndex++;

    // 撮影し保存する枚数を制限
    g_CaptureIndex = g_CaptureIndex % 64;
}

void CaptureFrame( nn::lmem::HeapHandle heapHandle )
{
    if ( g_CapturedFrameNumber > 0 )
    {
        sprintf_s( g_ScreenCapturedFilePath, _MAX_PATH, "ScreenCap_%05d.png", g_CapturedFrameNumber );
        SaveCurrentBufferAsPngFile( g_ScreenCapturedFilePath, g_ThumsWidth, g_ThumsHeight, heapHandle, true );
        g_CapturedFrameNumber = 0;

        // プレビューノード用サムネイル撮影後に撮影した画像ファイルパスを送信
        char filePath[ _MAX_PATH ];
        sprintf_s( filePath, "%s\\ScreenCap\\%s", g_WorkingDir, g_ScreenCapturedFilePath );
        SendMessageToCombinerEditor( MESSAGE_TO_COMBEDITOR_PREVIEW_PIC_PATH, filePath, sizeof( filePath ) );

        g_ReservationScreenCapture = false;
    }
}

#endif



void MakeCommandCallback( nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData )
{
    NN_UNUSED( pGraphicsFramework );
    NN_UNUSED( pUserData );

    nn::gfx::CommandBuffer*    pCommandBuffer = g_GraphicsFramework.GetRootCommandBuffer( bufferIndex );
    nn::gfx::ColorTargetView*  pColorTargetView = g_GraphicsFramework.GetColorTargetView();
    nn::gfx::DepthStencilView* pDepthStecilView = g_GraphicsFramework.GetDepthStencilView();

    nn::gfx::Texture* pColorBuffer = g_GraphicsFramework.GetColorBuffer();
    nn::gfx::Texture* pDepthStencilBuffer = g_GraphicsFramework.GetDepthStencilBuffer();

    nn::util::Matrix4x3fType identityViewMatrix;
    {
        nn::util::Vector3fType pos;
        nn::util::Vector3fType at;
        nn::util::Vector3fType up;
        nn::util::VectorSet( &pos, 0.0f, 0.0f, 1.0f );
        nn::util::VectorSet( &at, 0.0f, 0.0f, 0.0f );
        nn::util::VectorSet( &up, 0.0f, 1.0f, 0.0f );
        nn::util::MatrixLookAtRightHanded( &identityViewMatrix, pos, at, up );
    }


    g_GraphicsFramework.BeginFrame( bufferIndex );
    {


        pCommandBuffer->InvalidateMemory( nn::gfx::GpuAccess_Descriptor | nn::gfx::GpuAccess_ShaderCode );
        pCommandBuffer->SetDescriptorPool( &g_TextureDescriptorPool );
        pCommandBuffer->SetDescriptorPool( &g_SamplerDescriptorPool );
        pCommandBuffer->SetViewportScissorState( g_GraphicsFramework.GetViewportScissorState() );

        g_pPrimitiveRenderer->Update( bufferIndex );
        g_pPrimitiveRenderer->SetViewMatrix( &g_DemoCamera.GetMatrix() );
        g_pPrimitiveRenderer->SetProjectionMatrix( &g_ProjectionMatrix );

        // ノード経過プレビュー描画
        nn::gfx::ColorTargetView*   pNodePreviewColorTargetView  = g_NodePreviewRenderTarget.GetColorBuffer()->GetColorTargetView();
        nn::gfx::DepthStencilView*  pNodePreviewDepthStencilView = g_NodePreviewRenderTarget.GetDepthBuffer()->GetDepthStencilView();

        {
            const float fovy = nn::util::FloatPi / 3.0f;
            const float aspect = static_cast< float >( g_ThumsWidth ) / static_cast< float >( g_ThumsHeight );
            nn::util::MatrixPerspectiveFieldOfViewRightHanded( &g_ProjectionMatrix, fovy, aspect, 0.1f, 1000.f );
        }


        pCommandBuffer->ClearColor( pNodePreviewColorTargetView, 0.1f, 0.1f, 0.1f, 0.1f, NULL );
        pCommandBuffer->ClearDepthStencil( pNodePreviewDepthStencilView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );
        pCommandBuffer->SetRenderTargets( 1, &pNodePreviewColorTargetView, pNodePreviewDepthStencilView );
        pCommandBuffer->SetViewportScissorState( &g_NodePreviewViewportScissor );

        {
            pCommandBuffer->SetRasterizerState( &m_ModelRasterizerState );
            g_pPrimitiveRenderer->SetDepthStencilState( pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType_DepthWriteTest );
            g_pPrimitiveRenderer->SetBlendState( pCommandBuffer, nns::gfx::PrimitiveRenderer::BlendType_Opacity );
            g_pPrimitiveRenderer->SetBlendState( pCommandBuffer, nns::gfx::PrimitiveRenderer::BlendType_Normal );


            g_TextureDescPoolAllocator.BeginAllocateTransientDescriptorSlot();
            g_SamplerDescPoolAllocator.BeginAllocateTransientDescriptorSlot();

            detail::CombinerPreviewer::GetInstance()->UpdateDescriptorSlot( &g_TextureDescPoolAllocator, &g_SamplerDescPoolAllocator );

            g_TextureDescPoolAllocator.EndAllocateTransientDescriptorSlot();
            g_SamplerDescPoolAllocator.EndAllocateTransientDescriptorSlot();

            // コンバイナプレビュー描画処理
            detail::CombinerPreviewer::GetInstance()->Draw( 1, pCommandBuffer,
                g_pPrimitiveRenderer->GetGpuBuffer(), &g_ProjectionMatrix, &identityViewMatrix, &g_CameraPosition,
                g_ColorCopyTextureDescSlot, g_DepthStencilCopyTextureDescSlot );
        }

        {
            const float fovy = nn::util::FloatPi / 3.0f;
            const float aspect = static_cast< float >( g_DeviceWidth ) / static_cast< float >( g_DeviceHeight );
            nn::util::MatrixPerspectiveFieldOfViewRightHanded( &g_ProjectionMatrix, fovy, aspect, 0.1f, 1000.f );
        }

        pCommandBuffer->ClearColor( pColorTargetView, 0.1f, 0.1f, 0.1f, 0.0f, NULL );
        pCommandBuffer->ClearDepthStencil( pDepthStecilView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL );
        pCommandBuffer->SetRenderTargets( 1, &pColorTargetView, pDepthStecilView );
        pCommandBuffer->SetViewportScissorState( g_GraphicsFramework.GetViewportScissorState() );

        {
            // ビューア描画処理
            g_pPrimitiveRenderer->SetDepthStencilState( pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType_DepthWriteTest );
            g_pPrimitiveRenderer->SetBlendState( pCommandBuffer, nns::gfx::PrimitiveRenderer::BlendType_Opacity );
            pCommandBuffer->SetRasterizerState( &m_ModelRasterizerState );

            // グリッド処理
            detail::DefaultGrid defaultGrid;
            defaultGrid.ProcDraw( pCommandBuffer, g_pPrimitiveRenderer );

            // 背景モデル・画像 を描画

            // カラーバッファをコピー
            {
                nn::gfx::TextureSubresource dstSubResource;
                dstSubResource.SetDefault();
                dstSubResource.SetMipLevel( 0 );
                dstSubResource.SetArrayIndex( 0 );
                nn::gfx::TextureCopyRegion srcCopyRegion;
                srcCopyRegion.SetDefault();
                srcCopyRegion.SetWidth( g_Width );
                srcCopyRegion.SetHeight( g_Height );
                srcCopyRegion.EditSubresource().SetDefault();
                srcCopyRegion.EditSubresource().SetMipLevel( 0 );
                srcCopyRegion.EditSubresource().SetArrayIndex( 0 );
                pCommandBuffer->CopyImage( g_CopyRenderTarget.GetColorBuffer()->GetTexture(), dstSubResource, 0, 0, 0, pColorBuffer, srcCopyRegion );
            }

            // 深度ステンシルバッファをコピー
            {
                nn::gfx::TextureSubresource dstSubResource;
                dstSubResource.SetDefault();
                dstSubResource.SetMipLevel( 0 );
                dstSubResource.SetArrayIndex( 0 );
                nn::gfx::TextureCopyRegion srcCopyRegion;
                srcCopyRegion.SetDefault();
                srcCopyRegion.SetWidth( g_Width );
                srcCopyRegion.SetHeight( g_Height );
                srcCopyRegion.EditSubresource().SetDefault();
                srcCopyRegion.EditSubresource().SetMipLevel( 0 );
                srcCopyRegion.EditSubresource().SetArrayIndex( 0 );
                pCommandBuffer->CopyImage( g_CopyRenderTarget.GetDepthBuffer()->GetTexture(), dstSubResource, 0, 0, 0, pDepthStencilBuffer, srcCopyRegion );
            }

            pCommandBuffer->FlushMemory( nn::gfx::GpuAccess_ColorBuffer );
            pCommandBuffer->FlushMemory( nn::gfx::GpuAccess_DepthStencil );

            // レンダーターゲットを戻す
            pCommandBuffer->SetRenderTargets( 1, &pColorTargetView, pDepthStecilView );

            g_pPrimitiveRenderer->SetDepthStencilState( pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType_DepthWriteTest );
            g_pPrimitiveRenderer->SetBlendState( pCommandBuffer, g_BlendType );

            pCommandBuffer->SetRasterizerState( &m_ModelRasterizerState );

            // コンバイナプレビュー描画処理
            detail::CombinerPreviewer::GetInstance()->Draw( g_PreviewModelIndex, pCommandBuffer,
                g_pPrimitiveRenderer->GetGpuBuffer(), &g_ProjectionMatrix, &g_DemoCamera.GetMatrix(), &g_CameraPosition,
                g_ColorCopyTextureDescSlot, g_DepthStencilCopyTextureDescSlot );
        }

        // フォント描画処理
        g_Writer.object.Draw( pCommandBuffer );

#if defined( NN_BUILD_CONFIG_OS_WIN )
        if ( g_HwndTarget != NULL )
        {
            g_2DViewportScissor.Finalize( g_GraphicsFramework.GetDevice() );

            float w = static_cast<float>( g_Width  ) / g_DeviceWidth  * g_Width;
            float h = static_cast<float>( g_Height ) / g_DeviceHeight * g_Height;

            {
                static float gh = 0;

                nn::gfx::ViewportScissorState::InfoType info;
                info.SetDefault();
                info.SetScissorEnabled( true );
                nn::gfx::ViewportStateInfo viewportInfo;
                {
                    viewportInfo.SetDefault();
                    viewportInfo.SetOriginX( 0 );
                    viewportInfo.SetOriginY( g_Height - h );
                    viewportInfo.SetWidth( w );
                    viewportInfo.SetHeight( h );

                }
                nn::gfx::ScissorStateInfo scissorInfo;
                {
                    scissorInfo.SetDefault();
                    scissorInfo.SetOriginX( 0 );
                    scissorInfo.SetOriginY( 0 );
                    scissorInfo.SetWidth( static_cast<int>( w ) );
                    scissorInfo.SetHeight( static_cast<int>( h ) );
                }
                info.SetViewportStateInfoArray( &viewportInfo, 1 );
                info.SetScissorStateInfoArray( &scissorInfo, 1 );
                g_2DViewportScissor.Initialize( g_GraphicsFramework.GetDevice(), info );
            }

            pCommandBuffer->SetViewportScissorState( &g_2DViewportScissor );

            nn::util::Uint8x4 baseColor = { { 64, 128, 64, 255 } };
            nn::util::Uint8x4 lineColor = { { 128, 255, 128, 255 } };
            nn::util::Matrix4x3fType identityMatrix34;
            nn::util::Matrix4x4fType identityMatrix44;
            nn::util::MatrixIdentity( &identityMatrix34 );
            nn::util::MatrixIdentity( &identityMatrix44 );

            float scale = 1.f;

            g_pPrimitiveRenderer->SetViewMatrix( &identityMatrix34 );
            g_pPrimitiveRenderer->SetModelMatrix( &identityMatrix34 );
            g_pPrimitiveRenderer->SetProjectionMatrix( &identityMatrix44 );
            g_pPrimitiveRenderer->SetColor( baseColor );
            g_pPrimitiveRenderer->Draw2DRect( pCommandBuffer, 2, 2, 16 * scale, 16 * scale );

            g_pPrimitiveRenderer->SetColor( lineColor );
            g_pPrimitiveRenderer->Draw2DRect( pCommandBuffer, 5, 12, 2 * scale, 4 * scale );
            g_pPrimitiveRenderer->Draw2DRect( pCommandBuffer, 9, 8, 2 * scale, 8 * scale );
            g_pPrimitiveRenderer->Draw2DRect( pCommandBuffer, 13, 4, 2 * scale, 12 * scale );
        }
#endif

        // カラーバッファ、クエリバッファをフラッシュ
        pCommandBuffer->FlushMemory( nn::gfx::GpuAccess_ColorBuffer || nn::gfx::GpuAccess_QueryBuffer );

#if defined( NN_BUILD_CONFIG_OS_WIN )
        //------------------------------------------
        // Windows のみ：画面のキャプチャコマンドを追加
        //------------------------------------------
        PushCaptureCommand( pCommandBuffer, g_BaseDirectory );
#endif
    }
    g_GraphicsFramework.EndFrame( bufferIndex );

    return;
}


//------------------------------------------------------------------------------
//  nn::fs用メモリ取得用関数
//------------------------------------------------------------------------------
void* Allocate( size_t size )
{
    return malloc( size );
}

//------------------------------------------------------------------------------
//  nn::fs用メモリ解放用関数
//------------------------------------------------------------------------------
void Deallocate( void* p, size_t size )
{
    NN_UNUSED( size );
    free( p );
}

//------------------------------------------------------------------------------
//  メイン関数
//------------------------------------------------------------------------------
extern "C" void nnMain()
{
    //------------------------------------------
    // ファイルシステムのアロケータ設定
    //------------------------------------------
    nn::fs::SetAllocator( Allocate, Deallocate );
    nn::fs::MountHostRoot();

#if defined( NN_BUILD_CONFIG_OS_WIN ) && defined( NN_SDK_BUILD_DEBUG )
    _CrtSetDbgFlag( _CRTDBG_LEAK_CHECK_DF | _CRTDBG_ALLOC_MEM_DF );
#endif

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    static const size_t   GraphicsSystemMemorySize = 8 * 1024 * 1024;
    g_GraphicsFramework.InitializeGraphicsSystem( GraphicsSystemMemorySize );
#endif

#if NN_GFX_IS_TARGET_NVN && defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
    // GLSLC のためのメモリアロケータを設定します。
    glslcSetAllocator( GlslcAllocateFunction, GlslcFreeFunction, GlslcReallocateFunction, NULL );
#endif

    //------------------------------------------
    // アロケータ初期化
    //------------------------------------------

    static const uint32_t MemSizeForCmb = 128 * 1024 * 1024;     // Vfx 用メモリサイズ
    static const uint32_t MemSizeForGfx = 128 * 1024 * 1024;     // Gfx 用メモリサイズ

    nns::nac::MemoryAllocator allocatorForGfx;
    uint8_t* const pMemForGfx = new uint8_t[ MemSizeForGfx ];
    allocatorForGfx.Initialize( pMemForGfx, MemSizeForGfx );

    nns::nac::MemoryAllocator allocatorForCombinerPreview;
    uint8_t* const pMemForCmb = new uint8_t[ MemSizeForCmb ];
    allocatorForCombinerPreview.Initialize( pMemForCmb, MemSizeForCmb );

    // コンバイナプレビューモードの場合は、小窓サイズ
    g_Width  = 1280;
    g_Height = 720;

    g_DeviceWidth  = 256;
    g_DeviceHeight = 256;

    g_ThumsWidth   = 256;
    g_ThumsHeight  = 256;

    //------------------------------------------
    // 描画フレームワーク設定
    //------------------------------------------
    g_DrawFrameworkMode = nns::gfx::GraphicsFramework::FrameworkMode_Immediate;

    //------------------------------------------
    // GraphicsFrameWorkの初期化
    //------------------------------------------
    g_GraphicsFrameworkInfo.SetDefault();
    g_GraphicsFrameworkInfo.SetDisplayWidth( g_Width );
    g_GraphicsFrameworkInfo.SetDisplayHeight( g_Height );
    if ( g_DrawFrameworkMode == nns::gfx::GraphicsFramework::FrameworkMode_DeferredExecution )
    {
        g_GraphicsFrameworkInfo.SetBufferCount( 2 );
        g_GraphicsFrameworkInfo.SetSwapChainBufferCount( 2 );
    }
    else
    {
        g_GraphicsFrameworkInfo.SetBufferCount( 1 );
        g_GraphicsFrameworkInfo.SetSwapChainBufferCount( 1 );
    }
    g_GraphicsFrameworkInfo.SetColorBufferFormat( nn::gfx::ImageFormat_R16_G16_B16_A16_Float );
    g_GraphicsFrameworkInfo.SetDepthStencilBufferFormat( nn::gfx::ImageFormat_D32_Float );
    g_GraphicsFrameworkInfo.SetMemoryPoolSize( nns::gfx::GraphicsFramework::MemoryPoolType_CommandBuffer, 1024 * 1024 * 64 );
    g_GraphicsFrameworkInfo.SetMemoryPoolSize( nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, 1024 * 1024 * 64 );
    g_GraphicsFrameworkInfo.SetMemoryPoolSize( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, 1024 * 1024 * 64 );

    //if ( 1 )//config.m_IsLinearEditMode )
    {
        g_GraphicsFrameworkInfo.SetSwapChainFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb );
    }
    //else
    //{
    //    g_GraphicsFrameworkInfo.SetSwapChainFormat( nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm );
    //}
    g_GraphicsFrameworkInfo.SetDebugMode( nn::gfx::DebugMode_Full );

    g_GraphicsFramework.Initialize( g_GraphicsFrameworkInfo );

    g_GraphicsFramework.SetMakeCommandCallback( MakeCommandCallback, nullptr );
    g_GraphicsFramework.SetFrameworkMode( g_DrawFrameworkMode );

    //------------------------------------------
    // nn::gfx の初期化
    //------------------------------------------
    InitializeGfx( g_GraphicsFramework.GetDevice(), &allocatorForGfx, MemSizeForGfx, g_Width, g_Height );

    // debug フォントの初期化
    nn::util::Unorm8x4 color0 = { { 255, 255, 255, 255 } };
    const float ScaleW = 0.8f;
    const float ScaleH = 1.0f;
    g_GraphicsFramework.InitializeDebugFontTextWriter( &g_Writer, 1024 );
    g_Writer.object.SetScale( ScaleW, ScaleH );
    g_Writer.object.SetTextColor( color0 );
    g_Writer.object.SetFixedWidthEnabled( true );
    g_Writer.object.SetTextureDescriptor( &g_TextureDescriptorPool, g_TextureDescPoolAllocator.GetStaticDescriptorPoolIndex( 1 ) );
    g_Writer.object.SetSamplerDescriptor( &g_SamplerDescriptorPool, g_SamplerDescPoolAllocator.GetStaticDescriptorPoolIndex( 1 ) );

    //------------------------------------------
    // PrimitiveRenderer の初期化
    //------------------------------------------
    InitializePrimitiveRenderer( g_GraphicsFramework.GetDevice(), &allocatorForGfx );

    //------------------------------------------
    // 入力インターフェースの初期化
    //------------------------------------------
    InitializeInputInterface();

    //------------------------------------------
    // ビュー系初期化
    //------------------------------------------
    // カメラを初期化
    nn::util::VectorSet( &g_CameraPosition, 0.0f, 2.0f, 3.0f );
    nn::util::VectorSet( &g_CameraLookAt, 0.0f, 0.0f, 0.0f );
    g_DemoCamera.SetPos( g_CameraPosition );
    g_DemoCamera.SetLookAtPos( g_CameraLookAt );
    g_DemoCamera.Preset();

    // プロジェクションを初期化
    const float fovy = nn::util::FloatPi / 3.0f;
    const float aspect = static_cast< float >( g_Width ) / static_cast< float >( g_Height );
    nn::util::MatrixPerspectiveFieldOfViewRightHanded( &g_ProjectionMatrix, fovy, aspect, 0.1f, 1000.f );

    //------------------------------------------
    // スクリーンキャプチャの初期化
    //------------------------------------------
#if defined( NN_BUILD_CONFIG_OS_WIN )
    // スクリーンキャプチャ用のフォルダを削除
    RemoveDirectory( L"ScreenCap" );

    // 起動フォルダパスを取得
    GetCurrentDirectory( _MAX_PATH, g_BaseDirectory );
    WideCharToMultiByte( CP_ACP, 0, g_BaseDirectory, -1, g_WorkingDir, sizeof( g_WorkingDir ), NULL, NULL );

    const uint32_t memSizeCapture = 16 * 1024 * 1024;
    uint8_t* pMemForScreenCapture = new uint8_t[ memSizeCapture ];
    g_HeapHandle = nn::lmem::CreateExpHeap( reinterpret_cast<void*>( pMemForScreenCapture ), memSizeCapture, nn::lmem::CreationOption_NoOption );
#endif

    //------------------------------------------
    // コンバイナプレビューモデル初期化
    //------------------------------------------
    detail::CombinerPreviewer::GetInstance()->InitializeCombinerPreviewer( g_GraphicsFramework.GetDevice(), &allocatorForCombinerPreview );

#if defined( NN_BUILD_CONFIG_OS_WIN )
    // Drop
    DragAcceptFiles( g_HWnd, true );
#endif

    //------------------------------------------
    // ＜初期化ここまで＞
    //------------------------------------------

    g_Initialized = true;

#if defined( NN_BUILD_CONFIG_OS_WIN )
    MSG  msg;
#endif

    //------------------------------------------
    // 毎フレームのレンダリング
    //------------------------------------------
    for ( ; ; )
    {
#if defined( NN_BUILD_CONFIG_OS_WIN )
        //------------------------------------------
        // Windows メッセージ処理
        //------------------------------------------
        if ( PeekMessage( &msg, NULL, 0, 0, PM_REMOVE ) )
        {
            TranslateMessage( &msg );
            if ( msg.message == WM_QUIT )
            {
                break;
            }
            DispatchMessage( &msg );
        }
#endif

        //------------------------------------------
        // システムの状態更新
        //------------------------------------------
        {
            // フレームレート設定
            g_GraphicsFramework.SetPresentInterval( 1 );

            // システム デモパッドの更新
            UpdateInputInterface();

            // カメラの更新
            g_DemoCamera.UpdateCamera( GetPad(), GetMouse(), GetMouse()->IsPointerOn(), true );
            g_DemoCamera.GetPos( &g_CameraPosition );
            g_DemoCamera.GetLookAtPos( &g_CameraLookAt );

            // ブレンドモード切替
            if ( GetMouse()->IsTrig( nns::nac::Mouse::MASK_MBUTTON ) )
            {
                if ( g_BlendType == nns::gfx::PrimitiveRenderer::BlendType_Opacity )
                {
                    g_BlendType = nns::gfx::PrimitiveRenderer::BlendType_Normal;
                }
                else
                {
                    g_BlendType = nns::gfx::PrimitiveRenderer::BlendType_Opacity;
                }
            }

            // 回転 ON/OFF
            if ( GetMouse()->IsTrig( nns::nac::Mouse::MASK_RBUTTON ) )
            {
                if ( g_RotateY == 0.0f )
                {
                    g_RotateY = 0.001f;
                }
                else
                {
                    g_RotateY = 0.0f;
                }
            }


            // モデル切り替え
            if ( GetMouse()->IsHold( nns::nac::Mouse::MASK_MBUTTON ) && GetMouse()->IsTrig( nns::nac::Mouse::MASK_RBUTTON ) )
            {
                g_PreviewModelIndex = 1 - g_PreviewModelIndex;
            }
        }

        //------------------------------------------
        // コンバイナプレビューの状態更新
        //------------------------------------------
        {
            // コンバイナプレビュー計算処理
            detail::CombinerPreviewer::GetInstance()->CalcBlock( &g_DemoCamera.GetMatrix(), g_RotateY );

            if ( detail::CombinerPreviewer::GetInstance()->IsTextureReload() )
            {
                g_TextureDescPoolAllocator.BeginAllocateTransientDescriptorSlot();
                g_SamplerDescPoolAllocator.BeginAllocateTransientDescriptorSlot();

                detail::CombinerPreviewer::GetInstance()->UpdateDescriptorSlot( &g_TextureDescPoolAllocator, &g_SamplerDescPoolAllocator );

                g_TextureDescPoolAllocator.EndAllocateTransientDescriptorSlot();
                g_SamplerDescPoolAllocator.EndAllocateTransientDescriptorSlot();

                // テクスチャリロード
#if defined( NN_BUILD_CONFIG_OS_WIN )
                SendMessageToCombinerEditor( MESSAGE_TO_COMBEDITOR_TEXTURE_RELOAD, nullptr, 0 );
#endif
            }

            if ( detail::CombinerPreviewer::GetInstance()->IsPreviewShaderReload() )
            {
                g_ReservationScreenCapture = true;

                // プレビューシェーダリロード済み
#if defined( NN_BUILD_CONFIG_OS_WIN )
                SendMessageToCombinerEditor( MESSAGE_TO_COMBEDITOR_PREVIEW_SHADER_RELOAD, nullptr, 0 );
#endif
            }

#if defined( NN_BUILD_CONFIG_OS_WIN )
            if ( detail::CombinerPreviewer::GetInstance()->IsOutputShaderReload() )
            {
                // 出力シェーダリロード済み
                SendMessageToCombinerEditor( MESSAGE_TO_COMBEDITOR_OUTPUT_SHADER_RELOAD, nullptr, 0 );
            }
#endif

#if defined( NN_BUILD_CONFIG_OS_WIN )
            if ( detail::CombinerPreviewer::GetInstance()->IsTextureReset() )
            {
                // テクスチャリセット済み
                SendMessageToCombinerEditor( MESSAGE_TO_COMBEDITOR_TEXTURE_RESET, nullptr, 0 );
            }
#endif
        }

        //------------------------------------------
        // グラフィックスフレームワーク定期処理
        // 描画コマンド発行はこの中で行われる
        //------------------------------------------
        g_GraphicsFramework.ProcessFrame();

        //------------------------------------------
        // Windows のみ：画面のキャプチャ
        //------------------------------------------
#if defined( NN_BUILD_CONFIG_OS_WIN )
        CaptureFrame( g_HeapHandle );
#endif
    }

    //------------------------------------------
    // 各オブジェクトを破棄
    //------------------------------------------
    detail::CombinerPreviewer::GetInstance()->FinalizeCombinerPreviewer( g_GraphicsFramework.GetDevice(), &allocatorForCombinerPreview );
    FinalizePrimitiveRenderer( g_GraphicsFramework.GetDevice(), &allocatorForGfx );
    FinalizeGfx( g_GraphicsFramework.GetDevice(), &allocatorForGfx );
    g_GraphicsFramework.FinalizeDebugFontTextWriter( &g_Writer );
    g_GraphicsFramework.Finalize();

    FinalizeInputInterface();
    nn::fs::UnmountHostRoot();

#if defined( NN_BUILD_CONFIG_OS_WIN )
    delete[] pMemForScreenCapture;
#endif
    delete[] pMemForCmb;
    delete[] pMemForGfx;

    return;
}

