﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <g3dif/SkeletalAnim.h>
#include <util/UtilMath.h>

namespace nw { namespace g3d { namespace tool {
namespace g3dif {

G3DIF_DEFINE_ENUM_TABLE(
    bone_anim_target, target,
    S11N_DEFINE_ENUM_ID(scale_x),
    S11N_DEFINE_ENUM_ID(scale_y),
    S11N_DEFINE_ENUM_ID(scale_z),
    S11N_DEFINE_ENUM_ID(rotate_x),
    S11N_DEFINE_ENUM_ID(rotate_y),
    S11N_DEFINE_ENUM_ID(rotate_z),
    S11N_DEFINE_ENUM_ID(rotate_w),
    S11N_DEFINE_ENUM_ID(translate_x),
    S11N_DEFINE_ENUM_ID(translate_y),
    S11N_DEFINE_ENUM_ID(translate_z)
);

void elem_bone_anim_target::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        target << pElem;
        base_value << pElem;

        curve.Validate();
        if (!curve->Read(pElem))
        {
            curve.Invalidate();
        }
    }
    CATCH_THROW_XML_ERROR()
}

void elem_bone_anim::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        //bind_index << pElem;
        bone_name << pElem;
        parent_name << pElem;
        scale_compensate << pElem;
        binarize_scale << pElem;
        binarize_rotate << pElem;
        binarize_translate << pElem;

        bone_anim_target_array.reserve(elem_bone_anim_target::num_target);
        for (const util::XMLElement* pTarget = pElem->Child(elem_bone_anim_target::Id());
            pTarget; pTarget = pTarget->NextSibling(elem_bone_anim_target::Id()))
        {
            bone_anim_target_array.push_back(elem_bone_anim_target());
            bone_anim_target_array.back() << pTarget;
        }
    }
    CATCH_THROW_XML_ERROR()
}

void elem_skeletal_anim_info::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        frame_count << pElem;
        scale_mode << pElem;
        rotate_mode << pElem;
        loop << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_skeletal_anim::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        version << pElem;
        skeletal_anim_info << pElem->Child(elem_skeletal_anim_info::Id());
        bone_anim_array << pElem->Child(elem_bone_anim::IdArray());
        stream_array << pElem->Child(elem_stream::IdArray());
        user_data_array << pElem->Child(elem_user_data::IdArray());
    }
    CATCH_THROW_XML_ERROR()
}


void elem_skeletal_anim::PostProcess()
{
    // 参照の解決。
    for (auto iter = bone_anim_array.begin(); iter != bone_anim_array.end(); ++iter)
    {
        for (auto iterTarget = iter->bone_anim_target_array.begin(); iterTarget != iter->bone_anim_target_array.end(); ++ iterTarget)
        {
            if (iterTarget->curve)
            {
                elem_anim_curve& curve = iterTarget->curve.Get();
                try
                {
                    int stream_array_size = static_cast<int>(stream_array.size());

                    if (stream_array_size <= curve.stream_index.value)
                    {
                        THROW_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                            "Out of range. stream_array: %d skeletal_anim.stream_index: %d",
                            stream_array_size, curve.stream_index.value);
                    }
                    // テキストデータを解析します。
                    elem_stream& xmlStream = stream_array[curve.stream_index.value];
                    curve.stream.rawdata = AnalizeAndCopyData(xmlStream.textData, xmlStream.count.value, xmlStream.GetStreamType());
                    curve.stream.count = xmlStream.count.value;
                    curve.stream.type = static_cast<StreamType>(xmlStream.type.value);
                }
                CATCH_THROW_XML_ERROR()
            }
        }
    }

    // 0スケールを取り除く
    AdjustZeroScale();

    // ユーザーデータの解析
    for (auto iter = user_data_array.begin(); iter != user_data_array.end(); ++iter)
    {
        iter->PostProcess(stream_array);
    }
}

void elem_skeletal_anim::PostBinaryProcess(void* data)
{
    try
    {
        StreamArray streamArray;
        AnalizeBinaryData(streamArray, data);

        int stream_array_size = static_cast<int>(streamArray.streamChunk.size());

        for (auto iter = bone_anim_array.begin(); iter != bone_anim_array.end(); ++iter)
        {
            for (auto iterTarget = iter->bone_anim_target_array.begin(); iterTarget != iter->bone_anim_target_array.end(); ++ iterTarget)
            {
                if (iterTarget->curve)
                {
                    elem_anim_curve& curve = iterTarget->curve.Get();

                    if (stream_array_size <= curve.stream_index.value)
                    {
                        THROW_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                            "Out of range. stream_array: %d skeletal_anim.stream_index: %d",
                            stream_array_size, curve.stream_index.value);
                    }
                    void* rawdata = CopyRawData(streamArray.streamChunk[curve.stream_index.value]);
                    curve.stream.rawdata.reset(rawdata, free);
                    curve.stream.count = streamArray.streamChunk[curve.stream_index.value].count;
                    curve.stream.type = streamArray.streamChunk[curve.stream_index.value].type;
                }
            }
        }

        // 0スケールを取り除く
        AdjustZeroScale();

        for (auto iter = user_data_array.begin(); iter != user_data_array.end(); ++iter)
        {
            iter->PostBinaryProcess(streamArray);
        }
    }
    CATCH_THROW_XML_ERROR()
}

void elem_skeletal_anim::AdjustZeroScale()
{
    for (auto itAnim = bone_anim_array.begin(); itAnim != bone_anim_array.end(); ++itAnim)
    {
        for (auto itTarget = itAnim->bone_anim_target_array.begin();
            itTarget != itAnim->bone_anim_target_array.end(); ++itTarget)
        {
            if (itTarget->target.value <= g3dif::elem_bone_anim_target::scale_z)
            {
                itTarget->base_value.value = ScaleAdjuster::ZeroToMin(itTarget->base_value.value);
                if (itTarget->curve)
                {
                    g3dif::elem_anim_curve& curve = itTarget->curve.Get();
                    static const int tblStride[] = { 4, 2, 2 };
                    int stride = tblStride[curve.curve_type];
                    float* pKey = static_cast<float*>(curve.stream.rawdata.get()) + 1;
                    for (int idx = 0; idx < curve.count.value; ++idx, pKey += stride)
                    {
                        if (curve.key_type.value <= g3dif::elem_anim_curve::key32)
                        {
                            *pKey = ScaleAdjuster::ZeroToMin(*pKey);
                        }
                        else if (curve.key_type.value == g3dif::elem_anim_curve::key16)
                        {
                            *pKey = ScaleAdjuster::ZeroToMin<s16>(*pKey, curve.scale.value, curve.offset.value);
                        }
                        else if (curve.key_type.value == g3dif::elem_anim_curve::key8)
                        {
                            *pKey = ScaleAdjuster::ZeroToMin<s8>(*pKey, curve.scale.value, curve.offset.value);
                        }
                        else
                        {
                            THROW_ERROR(ERRCODE_ANIM_INVALID_QUANTIZE_TYPE,
                                "Bone Anim supports only key32, key16, key8 and key_none");
                        }
                    }
                }
            }
        }
    }
}

void
elem_skeletal_anim::CheckData(int /*flag*/)
{
    for (auto iter = bone_anim_array.cbegin(); iter != bone_anim_array.cend(); ++iter)
    {
        // bone_anim_target は 9 か 10 でなければならない
        if (iter->bone_anim_target_array.size() < 9 || iter->bone_anim_target_array.size() > 10)
        {
            THROW_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                "Out of range. bone_anim_target count: %d",
                iter->bone_anim_target_array.size());
        }
    }
}

} // namespace g3dif

} // namespace tool
} // namespace g3d
} // namespace nw
