﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <g3dif/Shape.h>

namespace nw { namespace g3d { namespace tool {
namespace g3dif {

G3DIF_DEFINE_ENUM_TABLE(
    mesh, mode,
    S11N_DEFINE_ENUM_ID(triangles),
    S11N_DEFINE_ENUM_ID(triangle_fan),
    S11N_DEFINE_ENUM_ID(triangle_strip)
);

G3DIF_DEFINE_ENUM_TABLE(
    mesh, quantize_type,
    S11N_DEFINE_ENUM_ID(none),
    S11N_DEFINE_ENUM_ID(uint_16),
    S11N_DEFINE_ENUM_ID(uint_32)
);

//--------------------------------------------------------------------------------------------------

void elem_submesh::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        offset << pElem;
        count << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_mesh::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        mode << pElem;
        quantize_type << pElem;
        count << pElem;
        stream_index << pElem;

        submesh_array << pElem->Child(elem_submesh::IdArray());
    }
    CATCH_THROW_XML_ERROR()
}

//--------------------------------------------------------------------------------------------------
void elem_target_attrib::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        attrib_name << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

void elem_key_shape::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        name << pElem;

        target_attrib_array << pElem->Child(elem_target_attrib::IdArray());
    }
    CATCH_THROW_XML_ERROR()
}

//--------------------------------------------------------------------------------------------------

void elem_shape_info::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        mat_name << pElem;
        bone_name << pElem;
        vertex_index << pElem;
        vertex_skinning_count << pElem;
    }
    CATCH_THROW_XML_ERROR()
}

//--------------------------------------------------------------------------------------------------

void elem_shape::operator<<(const util::XMLElement* pElem)
{
    VerifyElement(pElem, Id());

    try
    {
        name << pElem;

        shape_info << pElem->Child(elem_shape_info::Id());
        mesh_array << pElem->Child(elem_mesh::IdArray());
        key_shape_array << pElem->Child(elem_key_shape::IdArray());
        user_data_array << pElem->Child(elem_user_data::IdArray());
    }
    CATCH_THROW_XML_ERROR()
}

void elem_shape::PostProcess(std::vector<elem_stream>& stream_array)
{
    int stream_array_size = static_cast<int>(stream_array.size());

    int meshIndex = 0;
    for (auto iter = mesh_array.begin(); iter != mesh_array.end(); ++iter, ++meshIndex)
    {
        if (stream_array_size <= iter->stream_index.value)
        {
            THROW_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                "Out of range. stream_array: %d mesh.stream_index: %d",
                stream_array_size, iter->stream_index.value);
        }
        iter->index = meshIndex;
        // テキストデータを解析します。
        elem_stream& xmlStream = stream_array[iter->stream_index.value];
        iter->stream.rawdata = AnalizeAndCopyData(xmlStream.textData, xmlStream.count.value, xmlStream.GetStreamType());
        iter->stream.count = xmlStream.count.value;
        iter->stream.type = static_cast<StreamType>(xmlStream.type.value);
    }
}

void elem_shape::PostBinaryProcess(StreamArray& streamArray)
{
    int stream_array_size = static_cast<int>(streamArray.streamChunk.size());

    int meshIndex = 0;
    for (auto iter = mesh_array.begin(); iter != mesh_array.end(); ++iter, ++meshIndex)
    {
        if (stream_array_size <= iter->stream_index.value)
        {
            THROW_ERROR(ERRCODE_XML_OUT_OF_RANGE,
                "Out of range. stream_array: %d mesh.stream_index: %d",
                stream_array_size, iter->stream_index.value);
        }
        iter->index = meshIndex;
        void* rawdata = CopyRawData(streamArray.streamChunk[iter->stream_index.value]);
        iter->stream.rawdata.reset(rawdata, free);
        iter->stream.count = streamArray.streamChunk[iter->stream_index.value].count;
        iter->stream.type = streamArray.streamChunk[iter->stream_index.value].type;
    }
}

} // namespace g3dif

} // namespace tool
} // namespace g3d
} // namespace nw
