﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <g3dif/Serialization.h>
#include <g3dif/Vertex.h>
#include <g3dif/UserData.h>

namespace nw { namespace g3d { namespace tool {
namespace g3dif {

//--------------------------------------------------------------------------------------------------

class elem_shader_param
{
public:
    G3DIF_DEFINE_ENUM(type,
        type_bool,
        type_bool2,
        type_bool3,
        type_bool4,
        type_int,
        type_int2,
        type_int3,
        type_int4,
        type_uint,
        type_uint2,
        type_uint3,
        type_uint4,
        type_float,
        type_float2,
        type_float3,
        type_float4,
        type_float2x2,
        type_float2x3,
        type_float2x4,
        type_float3x2,
        type_float3x3,
        type_float3x4,
        type_float4x2,
        type_float4x3,
        type_float4x4,
        type_srt2d,
        type_srt3d,
        type_texsrt,
        type_texsrt_ex
        );

    elem_shader_param() : id(), type(type_float4), depend(), textData(), rawdata(), dataSize(0) {}

    G3DIF_DEFINE_ELEM_ARRAY(shader_param);

    G3DIF_DEFINE_ATTRIB(std::string, id);
    G3DIF_DEFINE_ATTRIB(enum_type, type);
    G3DIF_DEFINE_ATTRIB(std::string, depend);

    typedef struct ParamFormat
    {
        int row;
        int col;
    } ParamFormat;

    const char* textData;
    std::shared_ptr<void> rawdata;
    size_t dataSize;

    // type からフォーマットを取得します。
    static const ParamFormat& GetShaderParamFormat( elem_shader_param::enum_type type );

    // 文字列から型を取得します。
    static enum_type GetType(std::string str);

    // 型から文字列を取得します。
    static std::string GetType(enum_type type);

    // 型から SrcParam のデータサイズを取得します。
    static size_t GetSrcParamSize(enum_type type);

    static std::pair<void*, size_t> TextToValue(const char* text, enum_type type);
    static void ValueToSrcParam(void* pDst, enum_type type);

    static const int MODE_COMPONENT_INDEX          = 0;
    static const int ROTATE_2D_COMPONENT_INDEX     = 2;
    static const int ROTATE_2D_COMPONENT_INDEX_SRT = 3;
    static const int ROTATE_3D_COMPONENT_INDEX_X   = 3;
    static const int ROTATE_3D_COMPONENT_INDEX_Y   = ROTATE_3D_COMPONENT_INDEX_X + 1;
    static const int ROTATE_3D_COMPONENT_INDEX_Z   = ROTATE_3D_COMPONENT_INDEX_Y + 1;
    static const int MULT_MATRIX_COMPONENT_INDEX   = 6;
};

class elem_render_info
{
public:
    G3DIF_DEFINE_ENUM(type,
        type_int,
        type_float,
        type_string
        );

    elem_render_info() : name(), count(0), textData(NULL) {}

    G3DIF_DEFINE_ELEM_ARRAY(render_info);

    G3DIF_DEFINE_ATTRIB(std::string, name);
    G3DIF_DEFINE_ATTRIB(enum_type, type);
    G3DIF_DEFINE_ATTRIB(int, count);

    const char* textData;

    // int, float の場合
    std::shared_ptr<void> rawdata;
    size_t dataSize;

    // string の場合
    std::vector<std::string> stringArray;

    static std::pair<void*, size_t> TextToValue(const char* text, int count, enum_type type);
};

//--------------------------------------------------------------------------------------------------

class elem_sampler_assign
{
public:
    elem_sampler_assign()
        : id()
        , sampler_name()
        , sampler_index(-1) {}

    G3DIF_DEFINE_ELEM_ARRAY(sampler_assign);

    G3DIF_DEFINE_ATTRIB(std::string, id);
    G3DIF_DEFINE_ATTRIB(std::string, sampler_name);

    int sampler_index;
};

//--------------------------------------------------------------------------------------------------

class elem_shader_option
{
public:
    elem_shader_option() : id(), value() {}

    G3DIF_DEFINE_ELEM_ARRAY(shader_option);

    G3DIF_DEFINE_ATTRIB(std::string, id);
    G3DIF_DEFINE_ATTRIB(std::string, value);
};

//--------------------------------------------------------------------------------------------------

class elem_attrib_assign
{
public:
    elem_attrib_assign()
        : id()
        , attrib_name() {}

    G3DIF_DEFINE_ELEM_ARRAY(attrib_assign);

    G3DIF_DEFINE_ATTRIB(std::string, id);
    G3DIF_DEFINE_ATTRIB(std::string, attrib_name);
};

//--------------------------------------------------------------------------------------------------

class elem_shader_assign
{
public:
    elem_shader_assign()
        : shader_archive()
        , shading_model()
        , revision()
        , render_info_array()
        , shader_option_array()
        , sampler_assign_array()
        , shader_param_array()
        , attrib_assign_array() {}

    G3DIF_DEFINE_ELEM(shader_assign);

    G3DIF_DEFINE_ATTRIB(std::string, shader_archive);
    G3DIF_DEFINE_ATTRIB(std::string, shading_model);
    G3DIF_DEFINE_ATTRIB(int, revision);

    std::vector<elem_render_info> render_info_array;
    std::vector<elem_shader_option> shader_option_array;
    std::vector<elem_sampler_assign> sampler_assign_array;
    std::vector<elem_shader_param> shader_param_array;
    std::vector<elem_attrib_assign> attrib_assign_array;
};

//--------------------------------------------------------------------------------------------------

class elem_depth_test
{
public:
    G3DIF_DEFINE_ENUM(func,
        never,
        equal,
        less,
        lequal,
        greater,
        gequal,
        nequal,
        always
    );

    elem_depth_test() : enable(false), write(true), func(less) {}

    G3DIF_DEFINE_ELEM(depth_test);

    G3DIF_DEFINE_ATTRIB(bool, enable);
    G3DIF_DEFINE_ATTRIB(bool, write);
    G3DIF_DEFINE_ATTRIB(enum_func, func);
};

class elem_alpha_test
{
public:
    G3DIF_DEFINE_ENUM(func,
        never,
        equal,
        less,
        lequal,
        greater,
        gequal,
        nequal,
        always
    );

    elem_alpha_test() : enable(false), func(less), value(0.0f) {}

    G3DIF_DEFINE_ELEM(alpha_test);

    G3DIF_DEFINE_ATTRIB(bool, enable);
    G3DIF_DEFINE_ATTRIB(enum_func, func);
    G3DIF_DEFINE_ATTRIB(float, value);
};

class elem_color_blend
{
public:
    G3DIF_DEFINE_ENUM(func,
        func_zero,
        func_one,
        func_src_color,
        func_one_minus_src_color,
        func_src_alpha,
        func_one_minus_src_alpha,
        func_dst_color,
        func_one_minus_dst_color,
        func_dst_alpha,
        func_one_minus_dst_alpha,
        func_const_color,
        func_one_minus_const_color,
        func_const_alpha,
        func_one_minus_const_alpha,
        func_src1_color,
        func_one_minus_src1_color,
        func_src1_alpha,
        func_one_minus_src1_alpha,
        func_src_alpha_saturate
    );

    G3DIF_DEFINE_ENUM(op,
        op_add,
        op_src_minus_dst,
        op_dst_minus_src,
        op_min,
        op_max
    );

    elem_color_blend()
        : rgb_src_func(func_src_alpha)
        , rgb_dst_func(func_one_minus_src_alpha)
        , rgb_op(op_add)
        , alpha_src_func(func_one)
        , alpha_dst_func(func_zero)
        , alpha_op(op_add)
        , const_color()
    {
        const_color->x = const_color->y = const_color->z = const_color->w = 0.0f;
    }

    G3DIF_DEFINE_ELEM(color_blend);

    G3DIF_DEFINE_ATTRIB(enum_func, rgb_src_func);
    G3DIF_DEFINE_ATTRIB(enum_func, rgb_dst_func);
    G3DIF_DEFINE_ATTRIB(enum_op, rgb_op);
    G3DIF_DEFINE_ATTRIB(enum_func, alpha_src_func);
    G3DIF_DEFINE_ATTRIB(enum_func, alpha_dst_func);
    G3DIF_DEFINE_ATTRIB(enum_op, alpha_op);
    G3DIF_DEFINE_ATTRIB(util::Vec4_t, const_color);
};

class elem_logical_blend
{
public:
    G3DIF_DEFINE_ENUM(op,
        op_clear,
        op_set,
        op_copy,
        op_inv_copy,
        op_no_op,
        op_inv,
        op_and,
        op_nand,
        op_or,
        op_nor,
        op_xor,
        op_equiv,
        op_rev_and,
        op_inv_and,
        op_rev_or,
        op_inv_or
    );

    elem_logical_blend() : op(op_copy) {}

    G3DIF_DEFINE_ELEM(logical_blend);

    G3DIF_DEFINE_ATTRIB(enum_op, op);
};

class elem_render_state
{
public:
    G3DIF_DEFINE_ENUM(mode,
        custom,
        opaque,
        mask,
        translucent
        );

    G3DIF_DEFINE_ENUM(display_face,
        both,
        front,
        back,
        face_none
        );

    G3DIF_DEFINE_ENUM(blend_mode,
        blend_none,
        color,
        logic
        );

    elem_render_state() : mode(custom), display_face(both), blend_mode(blend_none), depth_test(), alpha_test(), color_blend(), logical_blend() {}

    G3DIF_DEFINE_ELEM(render_state);

    G3DIF_DEFINE_ATTRIB(enum_mode, mode);
    G3DIF_DEFINE_ATTRIB(enum_display_face, display_face);
    G3DIF_DEFINE_ATTRIB(enum_blend_mode, blend_mode);
    util::Optional<elem_depth_test> depth_test;
    util::Optional<elem_alpha_test> alpha_test;
    util::Optional<elem_color_blend> color_blend;
    util::Optional<elem_logical_blend> logical_blend;
};

//--------------------------------------------------------------------------------------------------

class elem_wrap
{
public:
    G3DIF_DEFINE_ENUM(mode,
        repeat,
        mirror,
        clamp,
        mirror_once
    );

    elem_wrap() : u(repeat), v(repeat), w(repeat) {}

    G3DIF_DEFINE_ELEM(wrap);

    G3DIF_DEFINE_ATTRIB(enum_mode, u);
    G3DIF_DEFINE_ATTRIB(enum_mode, v);
    G3DIF_DEFINE_ATTRIB(enum_mode, w);
};

class elem_filter
{
public:
    G3DIF_DEFINE_ENUM(mode,
        none,
        point,
        linear
    );

    G3DIF_DEFINE_ENUM(max_aniso,
        aniso_1,
        aniso_2,
        aniso_4,
        aniso_8,
        aniso_16
    );

    elem_filter() : mag(point), min(point), mip(point) {}

    G3DIF_DEFINE_ELEM(filter);

    G3DIF_DEFINE_ATTRIB(enum_mode, mag);
    G3DIF_DEFINE_ATTRIB(enum_mode, min);
    G3DIF_DEFINE_ATTRIB(enum_mode, mip);
    G3DIF_DEFINE_ATTRIB(enum_max_aniso, max_aniso);
};

class elem_lod
{
public:
    elem_lod() : min(0.0f), max(0.0f), bias(0.0f) {}

    G3DIF_DEFINE_ELEM(lod);

    G3DIF_DEFINE_ATTRIB(float, min);
    G3DIF_DEFINE_ATTRIB(float, max);
    G3DIF_DEFINE_ATTRIB(float, bias);
};

class elem_sampler
{
public:
    elem_sampler()
        : name()
        , tex_name()
        , wrap()
        , filter()
        , lod()
    {
    }

    G3DIF_DEFINE_ELEM_ARRAY(sampler);

    G3DIF_DEFINE_ATTRIB(std::string, name);
    G3DIF_DEFINE_ATTRIB(std::string, tex_name);

    elem_wrap wrap;
    elem_filter filter;
    elem_lod lod;
};

class elem_material_info
{
public:
    elem_material_info()
        : visibility(true)
        , mesh_adjacency()
    {}

    G3DIF_DEFINE_ELEM(material_info);

    G3DIF_DEFINE_ATTRIB(bool, visibility);
    G3DIF_DEFINE_ATTRIB(bool, mesh_adjacency);
};


//--------------------------------------------------------------------------------------------------

class elem_material
{
public:
    elem_material()
        : material_info()
        , render_state()
        , sampler_array()
        , shader_assign()
        , user_data_array()
    {
    }

    G3DIF_DEFINE_ELEM_ARRAY(material);

    G3DIF_DEFINE_ATTRIB(std::string, name);

    elem_material_info material_info;
    elem_render_state render_state;
    std::vector<elem_sampler> sampler_array;
    util::Optional<elem_shader_assign> shader_assign;
    std::vector<elem_user_data> user_data_array;

    // textData を展開する。
    void Expand();
};

} // namespace g3dif

} // namespace tool
} // namespace g3d
} // namespace nw
