﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.Foundation.IO;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Linq.Expressions;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading.Tasks;

namespace nw.g3d.toollib.Ext
{
    public static class CommandLineUtility
    {
        // コマンドライン引数を定義したクラスのプロパティから、オプション名を取得する。
        public static string GetOptionName<T, U>(this T owner, Expression<Func<U>> property)
        {
            Nintendo.Foundation.Contracts.Assertion.Argument.True(owner != null);
            var body = property.Body as MemberExpression;

            var propName = body.Member.Name;

            var attributes = typeof(T).GetProperty(propName).GetCustomAttributes(typeof(CommandLineOptionAttribute), true);

            if (attributes != null && attributes.Length == 1)
            {
                var attribute = (CommandLineOptionAttribute)attributes[0];
                if (!string.IsNullOrEmpty(attribute.LongName))
                {
                    return "--" + attribute.LongName;
                }

                return "-" + attribute.ShortName.ToString();
            }

            return null;
        }

        /// <summary>
        /// コマンドラインの引数部分を分ける
        /// </summary>
        public static string[] SplitArguments(string commandLine)
        {
            if (commandLine == null)
            {
                return new string[0];
            }

            int argc;

            // コマンド部分が抜けているのでダミーのコマンド c をつけておく
            var argv = CommandLineToArgvW("c " + commandLine, out argc);

            var arguments = new string[argc - 1];
            try
            {
                // ダミーのコマンドは飛ばす
                for (int i = 1; i < argc; i++)
                {
                    // 文字列へのポインタを取得
                    var arg = Marshal.ReadIntPtr((IntPtr)argv, i * IntPtr.Size);

                    // string へ変換
                    arguments[i - 1] = Marshal.PtrToStringUni(arg);
                }
            }
            finally
            {
                // CommandLineToArgvW で確保されたメモリは呼び出し元が解放する
                Marshal.FreeHGlobal(argv);
            }

            return arguments;
        }

        [DllImport("shell32.dll", SetLastError = true)]
        private static extern IntPtr CommandLineToArgvW([MarshalAs(UnmanagedType.LPWStr)]string lpCmdLine, out int pNumArgs);

        public static T SubOptionConverter<T>(string commandLine) where T : class, new()
        {
            var args = CommandLineUtility.SplitArguments(commandLine.Replace('\'', '"'));
            if (args.Length == 0)
            {
                return null;
            }

            var setting = new CommandLineParserSettings()
            {
                // 出力しない
                HelpWriter = x => { },
                // 例外処理に任せる
                ErrorWriter = x => { },
            };

            T t;
            if (!(new CommandLineParser().ParseArgs(args, out t)))
            {
                return null;
            }

            return t;
        }
    }
}
