﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <functional>
#include <iterator>
#include <nn/util/util_BitArray.h>
#include <ShdrFile.h>
#include <ShaderCompilerManager.h>
#include <pugixml.hpp>

namespace nn { namespace g3dTool {
namespace {


//! @brief エンディアンを判定するための BOM です。
const u16 BYTE_ORDER_MARK = 0xFEFF;
const nw::g3d::tool::util::StringRef str_empty      = "";
const nw::g3d::tool::util::StringRef str_wild_card  = "*";

struct ShaderSrc
{
    std::string fullpath;
    std::string include;
    std::string filename;
};

struct IncludePath
{
    size_t length;
    std::string originalPath;
    std::string fullPath;
};

ShaderSrc FindFile(std::vector<IncludePath>& include_paths, std::string filename)
{
    ShaderSrc src;
    for (auto include = include_paths.cbegin(); include != include_paths.cend(); ++include)
    {
        std::replace(filename.begin(), filename.end(), '/', '\\');
        std::string fullpath = nw::g3d::tool::util::Path::Combine(include->fullPath, filename);
        if (nw::g3d::tool::util::Path::FileExists(fullpath))
        {
            src.fullpath = fullpath;
            src.include = include->fullPath;
            src.filename = filename;
            break;
        }
    }

    return src;
}

struct CheckDuplication
{
public:
    CheckDuplication(StringRef str)
        : m_Str(str)
    {}

    bool operator()(IncludePath include_path)
    {
        if (m_Str.Len() < include_path.length)
        {
            return (include_path.fullPath + "\\").find(m_Str.Str() + "\\") != std::string::npos;
        }
        else
        {
            return (m_Str.Str() + "\\").find(include_path.fullPath + "\\") != std::string::npos;
        }
    }

private:
    StringRef m_Str;
};

class VariationOptionFinder
{
public:
    VariationOptionFinder(StringRef id)
        : m_Id(id)
    {}

    bool operator()(const nw::g3d::tool::g3dif::elem_option& option)
    {
        return option.id.value == m_Id;
    }

private:
    StringRef m_Id;
};

struct KeyCmp
{
    KeyCmp(size_t length) : length(length) {}

    bool operator()(const ShaderProgram* lhs, const ShaderProgram* rhs) const
    {
        const uint32_t* pLhs = static_cast<const uint32_t*>(lhs->m_Key.get());
        const uint32_t* pRhs = static_cast<const uint32_t*>(rhs->m_Key.get());
        for (size_t word = 0; word < length; ++word, ++pLhs, ++pRhs)
        {
            if (*pLhs != *pRhs)
            {
                return *pLhs < *pRhs;
            }
        }
        return false;
    }

    size_t length;
};

struct Writer
{
    FILE* fp;
    Writer(const char* filename) : fp(nullptr)
    {
        fopen_s(&fp, filename, "wt");
        if (fp == nullptr)
        {
            PRINT_ERROR_LOG("Open File Failed : %hs\n", filename);
        }
    }
    ~Writer()
    {
        if (fp)
        {
            fclose(fp);
            fp = nullptr;
        }
    }
};

void CreateVariation(
    std::vector<ShaderVariationBuilder>& variations,
    const std::vector<int>& choiceIndexArray,
    const nw::g3d::tool::g3dif::elem_shading_model& el_shading_model,
    bool forceVariation)
{
    // choice を展開しながらバリエーションを作成します。
    variations.emplace_back( static_cast<int>( el_shading_model.keyLength ) );
    auto& variation = variations.back();
    el_shading_model.CreateKey(static_cast<uint32_t*>(variation.m_Key.get()), choiceIndexArray);

    const std::vector<nw::g3d::tool::g3dif::elem_option_var>& el_option_vars = el_shading_model.option_var_array;
    // choice とその index を保存しておきます。
    variation.m_ChoiceArray.resize(choiceIndexArray.size());
    for (int optionIndex2 = 0; optionIndex2 < static_cast<int>(choiceIndexArray.size()); ++optionIndex2)
    {
        auto& option = el_option_vars[optionIndex2];
        for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
        {
            if (option.branch.value && !forceVariation)
            {
                if (option.type.value == nw::g3d::tool::g3dif::elem_option_var::type_static)
                {
                    variation.m_ChoiceArray[optionIndex2].m_Symbol[stage] = option.altUniformSymbol[stage];
                }
                else	// type=dynamic
                {
                    variation.m_ChoiceArray[optionIndex2].m_Symbol[stage] = option.choiceArray[option.defaultValueIndex].first;
                }
            }
            else
            {
                variation.m_ChoiceArray[optionIndex2].m_Symbol[stage] = option.choiceArray[choiceIndexArray[optionIndex2]].first;
            }
        }
    }

    variation.m_ChoiceIndexArray = choiceIndexArray;
}

// ShadingModel 中の各 variation が持つ ub をマージします。
void MergeUniformBlockMap(UniformBlockMap* dst, const UniformBlockMap* src)
{
    for (auto& uniformBlock : *src)
    {
        auto found = dst->find(uniformBlock.first);
        if (found == dst->end())
        {
            dst->insert(uniformBlock);
            continue;
        }

        // ub のサイズを持つ時にサイズ違いの ub があった時にエラーとする。
        if( ( ( found->second != 0 ) && ( uniformBlock.second != 0 ) ) && ( found->second != uniformBlock.second  ) )
        {
            THROW_ERROR( ERRCODE_SHADER_CONVERTER_INVALID_UNIFORM_BLOCK,
                    "Invalid uniform block(%hs). Inconsistent uniform block size between shader programs", uniformBlock.first.c_str() );
        }

        // ub のサイズが 0 の場合はサイズを持つものに更新する。
        // c.f. uniform offset が歯抜けでも ub のサイズは変わらないが ub に所属する
        //		offset が全部 invalid の場合は ub のサイズが 0 になる。
        if( ( found->second == 0 ) && ( uniformBlock.second != 0 ) )
        {
            found->second = uniformBlock.second;
        }
    }
}

void MergeUniformMap(UniformMap* dst, const UniformMap* src)
{
    for (auto& uniform : *src)
    {
        auto found = dst->find(uniform.first);

        if (found == dst->end())
        {
            dst->insert(uniform);
            continue;
        }

        // 各 variation の uniform の整合性をチェックする。
        if( ( uniform.second.arrayCount != found->second.arrayCount ) || ( uniform.second.type != found->second.type ) ||
            ( ( found->second.offset != -1 ) && ( uniform.second.offset != -1 ) && ( uniform.second.offset != found->second.offset ) )
          )
        {
            THROW_ERROR( ERRCODE_SHADER_CONVERTER_INVALID_UNIFORM, "Invalid uniform(%hs).", uniform.first.c_str() );
        }

        // 整合性を取るために、offset -1 のメンバは offset を持つ variation の offset 値に更新する。
        if( ( found->second.offset == -1 ) && ( uniform.second.offset != -1 ) )
        {
            found->second.offset = uniform.second.offset;
        }
    }
}

bool IsValidKey(
    const std::vector<int>& choiceIndexArray,
    const std::vector<std::vector<bool>>& choiceEnableArray,
    const std::vector<ShaderBuilder>& builders,
    const nw::g3d::tool::g3dif::elem_shading_model& el_shading_model)
{
    const std::vector<nw::g3d::tool::g3dif::elem_option_var>& el_option_vars = el_shading_model.option_var_array;
    int optionCount = static_cast<int>(el_option_vars.size());
    for (int optionIndex = 0; optionIndex < optionCount; ++optionIndex)
    {
        int choiceIndex = choiceIndexArray[optionIndex];

        // 生成フラグを確認してバリエーションの生成を制御する。
        if (!choiceEnableArray[optionIndex][choiceIndex])
        {
            return false;
        }
    }

    std::unique_ptr<uint32_t> key(new uint32_t[el_shading_model.keyLength]);
    el_shading_model.CreateKey(key.get(), choiceIndexArray);
    for (auto builder = builders.cbegin(); builder != builders.cend(); ++builder)
    {
        for (auto variation = builder->m_ShaderVariationBuilderArray.cbegin();
            variation != builder->m_ShaderVariationBuilderArray.cend(); ++variation)
        {
            // 既に作成済みのキーのバリエーションは追加しない
            if (memcmp(variation->m_Key.get(), key.get(),
                el_shading_model.keyLength * sizeof(uint32_t)) == 0)
            {
                return false;
            }
        }
    }

    return true;
}

bool GetNext(
    std::vector<int>& choiceIndexArray,
    const std::vector<bool>& choiceLockArray,
    const std::vector<std::vector<bool>>& choiceEnableArray)
{
    int optionCount = static_cast<int>(choiceIndexArray.size());
    for (int optionIndex = optionCount - 1; optionIndex >= 0; --optionIndex)
    {
        if (choiceLockArray[optionIndex])
        {
            continue;
        }

        int choiceCount = static_cast<int>(choiceEnableArray[optionIndex].size());
        if (choiceIndexArray[optionIndex] < choiceCount - 1)
        {
            // choice が末尾ではない場合はインクリメントする。
            ++choiceIndexArray[optionIndex];
            for (int fillOptionIndex = optionIndex + 1; fillOptionIndex < optionCount; ++fillOptionIndex)
            {
                if (!choiceLockArray[fillOptionIndex])
                {
                    choiceIndexArray[fillOptionIndex] = 0;
                }
            }

            return true;
        }
    }

    return false;
}

void CreateVariations(
    std::vector<ShaderVariationBuilder>& variations,
    std::vector<int>& choiceIndexArray,
    const std::vector<std::vector<bool>>& choiceEnableArray,
    const std::vector<bool>& choiceLockArray,
    const std::vector<ShaderBuilder>& builders,
    const nw::g3d::tool::g3dif::elem_shading_model& el_shading_model,
    bool forceVariation)
{
    do
    {
        if (IsValidKey(choiceIndexArray, choiceEnableArray, builders, el_shading_model))
        {
            CreateVariation(variations, choiceIndexArray, el_shading_model, forceVariation);
        }
    } while (GetNext(choiceIndexArray, choiceLockArray, choiceEnableArray));
}

void InsertForceInclude( ShaderProgram& program,
                   const ShadingModelBuilder& builder,
                   const std::vector<std::string>& forceIncludes)
{
    FileDB fileDB;

    for (auto source = builder.m_ShaderSourceArray.cbegin(); source != builder.m_ShaderSourceArray.cend(); ++source)
    {
        const StringRef& relativePath = source->m_Path;
        const StringRef& stream = static_cast<char*>(source->m_Stream.get());
        std::string path(relativePath.Data(), relativePath.Len());
        std::replace(path.begin(), path.end(), '/', '\\');
        if (fileDB.find(path) == fileDB.end())
        {
            auto insert = fileDB.insert(std::make_pair(path,
                std::make_pair(static_cast<int>(fileDB.size()), std::string())));
            if (!insert.second)
            {
                THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
            }
            insert.first->second.second.assign(stream.Data(), stream.Len());
        }
    }

    for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
    {
        if (!builder.m_ShaderPath[stage].empty())
        {
            const StringRef& fullpath = builder.m_ShaderPath[stage];

            std::string path(fullpath.Data(), fullpath.Len());
            std::replace(path.begin(), path.end(), '/', '\\');

            StringRef src = fileDB.find(path)->second.second;

            std::stringstream ss;

            // fsc/fsd force_include になっている文字列を追加
            for (auto& forceIncludeFullPath : forceIncludes)
            {
                PRINT_LOG("force:  %hs", forceIncludeFullPath);
                ss << "#include<" << forceIncludeFullPath << ">\n";
            }

            // 最後にソースの文字列を挿入
            ss.write(src.Data(), src.Len());

            program.m_ShaderSource[stage].assign( std::istreambuf_iterator<char>(ss.rdbuf()), std::istreambuf_iterator<char>());
        }
    }
}

void DumpKey(const char* pDumpFolder, const nw::g3d::tool::g3dif::elem_shading_model& el_shading_model,
             const std::vector<ShaderProgram>& programs)
{
    char path[_MAX_PATH];
    const char* pShadingModel = el_shading_model.name.value.c_str();
    sprintf_s(path, "%hs%hs_key.csv", pDumpFolder, pShadingModel);
    Writer fpKey(path);
    if (fpKey.fp)
    {
        fprintf(fpKey.fp, "id,branch,type,key_block,key_pos,key_width,key_mask,default,choice\n");
        for (auto el_option = el_shading_model.option_var_array.cbegin();
            el_option != el_shading_model.option_var_array.cend(); ++el_option)
        {
            if (el_option->id.value != "system_id")
            {
                int block = static_cast<int>(el_option->keyOffset) >> 5;
                uint32_t shift = ( 32 - ( static_cast<int>(el_option->keyOffset) & 0x1F )
                    - static_cast<int>(el_option->bitWidth) );
                fprintf(fpKey.fp, "%hs,%hs,%hs,%d,%zd,%zd,_%08X_ ,%hs",
                    el_option->id.value.c_str(),
                    el_option->branch.value ? "true" : "false",
                    el_option->type.value == nw::g3d::tool::g3dif::elem_option_var::type_dynamic ? "dynamic" : "static",
                    block,
                    el_option->keyOffset & 0x1F,
                    el_option->bitWidth,
                    ( ( 1 << el_option->bitWidth ) - 1 ) << shift,
                    el_option->default_value.value.c_str());
                // choice
                for (auto choice = el_option->choiceArray.cbegin();
                    choice != el_option->choiceArray.cend(); ++choice)
                {
                    fprintf(fpKey.fp, "%s%s", choice == el_option->choiceArray.cbegin() ?
                        ",\"" : ",", choice->first.c_str());
                }
                fprintf(fpKey.fp, "\"\n");
            }
        }
    }

    sprintf_s(path, "%hs%hs_shader_key.csv", pDumpFolder, pShadingModel);
    Writer fpShaderKey(path);
    if(fpShaderKey.fp)
    {
        fprintf(fpShaderKey.fp, "number,key\n");
        for(int idxProgram = 0, numProgram = static_cast< int >( programs.size() );
            idxProgram < numProgram; ++idxProgram)
        {
            fprintf(fpShaderKey.fp, "%05d,%hs\n", idxProgram,
                ShaderKeyToString(programs[idxProgram].m_Key.get(), programs[idxProgram].m_KeyLength).c_str());
        }
    }
}

void DumpShaderSource(
    const char* pDumpFolder,
    const char* pArchieName,
    const ShaderCompilerManager* pShaderCompilerManager,
    const ShaderProgram* pProgram,
    const ShaderCompilerManager::PipelineInfo* pShader,
    const nw::g3d::tool::g3dif::elem_shading_model* el_shading_model,
    //u64 optflags,
    const ShaderProgram* pSharedProgram,
    int programIndex)
{
    NN_UNUSED( pSharedProgram );

    char path[_MAX_PATH];
    const char* pShadingModel = el_shading_model->name.value.c_str();

    sprintf_s(path, "%hs%hs_%05d_key.fsva", pDumpFolder, pShadingModel, programIndex);
    Writer fpKey(path);
    if (fpKey.fp)
    {
        static const u8 BOM[] = { 0xEF, 0xBB, 0xBF };
        fwrite(BOM, sizeof(u8), 3, fpKey.fp);
        fprintf(fpKey.fp, "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n");
        fprintf(fpKey.fp, "<nw4f_3dif version=\"%d.%d.%d\">\n",
            NN_G3D_VERSION_MAJOR, NN_G3D_VERSION_MINOR, NN_G3D_VERSION_MICRO);
        fprintf(fpKey.fp, "  <shader_variation>\n");
        fprintf(fpKey.fp, "    <shader_variation_info shader_archive=\"%s\" />\n", pArchieName);
        fprintf(fpKey.fp, "    <target_shader_array length=\"1\">\n");
        fprintf(fpKey.fp, "      <target_shader index=\"0\" shading_model_name=\"%hs\">\n", el_shading_model->name.value.c_str());
        fprintf(fpKey.fp, "        <shader_program_array length=\"1\">\n");
        fprintf(fpKey.fp, "          <shader_program index=\"0\">\n");
        if (el_shading_model->option_var_array.size() > 0)
        {
            fprintf(fpKey.fp, "            <option_array length=\"%zd\">\n", el_shading_model->option_var_array.size() - 1);
            for (auto el_option = el_shading_model->option_var_array.cbegin();
                el_option != el_shading_model->option_var_array.cend(); ++el_option)
            {
                if (el_option->id.value != "system_id")
                {
                    int block = static_cast<int>(el_option->keyOffset) >> 5;
                    u32 blockKey = *(static_cast<u32*>(pProgram->m_Key.get()) + block);
                    u32 shift = ( 32 - ( static_cast<int>(el_option->keyOffset) & 0x1F ) - static_cast<int>(el_option->bitWidth) );
                    u32 mask = ( ( 1 << el_option->bitWidth ) - 1 ) << shift;
                    int choiceIndex = ( blockKey & mask ) >> shift;
                    fprintf(fpKey.fp, "              <option index=\"%td\" id=\"%hs\" choice=\"%hs\" />\n",
                        std::distance(el_shading_model->option_var_array.cbegin(), el_option),
                        el_option->id.value.c_str(), el_option->choiceArray[choiceIndex].first.c_str());
                }
            }
            fprintf(fpKey.fp, "            </option_array>\n");
        }
        fprintf(fpKey.fp, "          </shader_program>\n");
        fprintf(fpKey.fp, "        </shader_program_array>\n");
        fprintf(fpKey.fp, "      </target_shader>\n");
        fprintf(fpKey.fp, "    </target_shader_array>\n");
        fprintf(fpKey.fp, "  </shader_variation>\n");
        fprintf(fpKey.fp, "</nw4f_3dif>");
    }

    auto dumpProgram = [&](const ShaderCompilerManager::ShaderInfo* pShader, const char* pStageName)
    {
        const char* pDump = pShader->pSource;
        if (pDump && *pDump)
        {
            char dumpPath[_MAX_PATH];
            sprintf_s(dumpPath, "%hs%hs_%05d_%hs.txt", pDumpFolder, pShadingModel, programIndex, pStageName);
            Writer fpDump(dumpPath);
            fprintf(fpDump.fp, "[ShaderKey:%hs]\n\n",
                ShaderKeyToString(pProgram->m_Key.get(), pProgram->m_KeyLength).c_str());

            fprintf(fpDump.fp, pDump);

            std::stringstream ss;
            pShaderCompilerManager->DumpStatistics( &ss, pShader);

            fprintf(fpDump.fp, ss.str().data());
        }
    };
    dumpProgram(&pShader->vertexShaderInfo, "vs");
    dumpProgram(&pShader->geometryShaderInfo, "gs");
    dumpProgram(&pShader->pixelShaderInfo, "fs");
    dumpProgram(&pShader->computeShaderInfo, "cs");
}

void DumpXmlStatistics(
    const char* pDumpFolder,
    const ShaderCompilerManager* pShaderCompilerManager,
    const ShaderProgram* pProgram,
    const ShaderCompilerManager::PipelineInfo* pShader,
    const nw::g3d::tool::g3dif::elem_shading_model* el_shading_model,
    int programIndex)
{
    const char* pShadingModel = el_shading_model->name.value.c_str();
    auto dumpXmlStatistics = [&](const ShaderCompilerManager::ShaderInfo* pShader, const char* pStageName)
    {
        const char* pDump = pShader->pSource;
        if (pDump && *pDump)
        {
            char xmlDumpPath[_MAX_PATH];
            sprintf_s(xmlDumpPath, "%hs%hs_%05d_%hs.xml", pDumpFolder, pShadingModel, programIndex, pStageName);
            pugi::xml_document xmlDump;
            pugi::xml_node root = xmlDump.append_child("shader_debug_dump_information");
            root.append_child("shader_key").append_child(pugi::node_pcdata).set_value(ShaderKeyToString(pProgram->m_Key.get(), pProgram->m_KeyLength).c_str());
            pShaderCompilerManager->DumpXmlStatistics(&root, pShader);
            xmlDump.save_file(xmlDumpPath);
        }
    };
    dumpXmlStatistics(&pShader->vertexShaderInfo, "vs");
    dumpXmlStatistics(&pShader->geometryShaderInfo, "gs");
    dumpXmlStatistics(&pShader->pixelShaderInfo, "fs");
    dumpXmlStatistics(&pShader->computeShaderInfo, "cs");
}

std::string EscapeDotCharactor(const std::string& source)
{
    return nw::g3d::tool::util::ReplaceAll<std::string>(source, std::string("."), std::string("_dot_escape_"));
}

const nw::g3d::tool::g3dif::elem_block_var* GetElemBlockVarForOption( const std::vector<nw::g3d::tool::g3dif::elem_block_var>& elemBlockVarArray )
{
    // elemBlockVarArray から type=Option の物を探す。
    for( auto& elemBlockVar : elemBlockVarArray )
    {
        if( elemBlockVar.type.value == nw::g3d::tool::g3dif::elem_block_var::option  )
        {
            return &elemBlockVar;
        }
    }
    return nullptr;
}

const nw::g3d::tool::g3dif::elem_uniform_var* GetElemUniformVar( const std::vector<nw::g3d::tool::g3dif::elem_block_var>& elemBlockVarArray,
                                                                nw::g3d::tool::g3dif::elem_block_var::enum_type blockType, const std::string& id )
{
    // elemBlockVarArray から type=Option の物を探す。
    for( auto& elemBlockVar : elemBlockVarArray )
    {
        if( !( elemBlockVar.type.value == blockType ) )
        {
            continue;
        }

        // id が合致する elem_uniform_var を検索して返す。
        auto uniformVarIter = std::find_if( elemBlockVar.uniform_var_array.begin(), elemBlockVar.uniform_var_array.end(),
                                            [&]( const nw::g3d::tool::g3dif::elem_uniform_var& elemUniformVar )
                                            {
                                                return elemUniformVar.id.value == id;
                                            } );

        if( uniformVarIter == elemBlockVar.uniform_var_array.end() )
        {
            return nullptr;
        }

        return &( *uniformVarIter );
    }

    return nullptr;
}

bool IsBatchCompileApplicable( bool isForceVariation,
                               bool hasUniform,
                               nw::g3d::tool::g3dif::elem_option_var& elemOptionVar )
{
    // c.f. type=dynamic は branch=true かつ uniform が存在でバッチコンパイル可能だが、
    //		- チョイスのデフォルト値でのみバリエーション化されるのでバッチコンパイルする意味がない
    //		- 過去との互換を保つ
    //		の 2 点からバッチコンパイル不可として扱う。
    return hasUniform && ( !elemOptionVar.branch.value || isForceVariation );
}

} // anonymous namespace

void ShdrFile::Make(bool unifiedAnnotation, bool autoExtract)
{
    std::shared_ptr<nw::g3d::tool::g3dif::elem_shader_definition> elem =
        std::shared_ptr<nw::g3d::tool::g3dif::elem_shader_definition>(new nw::g3d::tool::g3dif::elem_shader_definition());
    std::vector<IncludePath> include_paths;

    m_ExtractedFiles.clear();
    FileDB tmp;
    m_ExtractedFiles.swap(tmp);
    m_Extractor.ClearSettings(); // ClearMacro を含む
    m_Extractor.ClearResults();

    // 複数の fsc の統合には対応していません。
    if (m_FscArray.size() >= 2)
    {
        THROW_ERROR(ERRCODE_UNSUPPORTED, "More than two fsc files are unsupported.");
    }

    auto fsc = m_FscArray.cbegin();

    // シェーディングモデル抽出の準備
    {
        m_Extractor.SetCodePage((*fsc)->shader_config_info.code_page.value);

        // インクルードパス：fsc ファイルからの相対パスをフルパスに展開したもの、もしくは環境変数相対（絶対パス）
        //                 ：fsd の shader_src へ格納するときは入力された状態のままにする。
        // ソースファイル　：インクルードパスからの相対パスをフルパスに展開したもの
        std::string drive;
        std::string dir;
        nw::g3d::tool::util::Path::Split((*fsc)->path.c_str(), &drive, &dir, nullptr, nullptr);
        std::stringstream include_stream;
        std::string base_path;
        include_stream << drive.c_str() << dir.c_str();
        base_path = include_stream.str();
        const std::vector<nw::g3d::tool::g3dif::elem_include_path>& include_path_array = (*fsc)->include_path_array;
        if(include_path_array.empty())
        {
            THROW_ERROR(ERRCODE_XML_ELEMENT_NOT_FOUND, "<include_path> is not found. Please add <include_path> to the fsc file.");
        }
        for (auto include = include_path_array.cbegin(); include != include_path_array.cend(); ++include)
        {
            // include_path の環境変数を展開する
            std::string include_path = nw::g3d::tool::util::Path::ExpandEnvironment(include->path.value);

            // 相対パスの場合は fsc からの相対パスなので、パスの連結を行う。
            if (nw::g3d::tool::util::Path::IsRelative(include_path))
            {
                // フルパス解決を行う
                include_path = nw::g3d::tool::util::Path::ToFull(base_path + include->path.value);
            }

            // パスの正規化の前に / を \ に変換する必要がある。
            std::replace(include_path.begin(), include_path.end(), '/', '\\');

            // パスの最後尾に \ が有れば削除しておく
            if (include_path.back() == '\\')
            {
                include_path.pop_back();
            }

            // パスの正規化
            include_path = nw::g3d::tool::util::Path::Canonicalize(include_path);

            // インクルードパスが重複していないか確認する。
            auto found = std::find_if(include_paths.cbegin(), include_paths.cend(), CheckDuplication(include_path));
            if (include_paths.cend() != found)
            {
                THROW_ERROR_INTERNAL(ERRCODE_INVALID_DIRECTORY_NAME, "Include paths are duplicated. `%hs`and`%hs`. Don't specify parent-child relation directories.", found->fullPath.c_str(), include_path.c_str());
            }

            IncludePath includePath;
            includePath.originalPath = include->path.value;
            includePath.length = include_path.length();
            includePath.fullPath.assign(include_path);

            include_paths.push_back(includePath);

            m_Extractor.AddIncludePath(include_paths.back().fullPath);
        }

        m_Extractor.AddGlslVersion(m_GlslVersion);

        const std::vector<nw::g3d::tool::g3dif::elem_force_include_file>& force_include_file_array = (*fsc)->force_include_file_array;
        for (auto include = force_include_file_array.cbegin(); include != force_include_file_array.cend(); ++include)
        {
            m_Extractor.AddForceInclude(include->path.value);
        }
    }

    // シェーディングモデルごとにアノテーションを抽出
    {
        const std::vector<nw::g3d::tool::g3dif::elem_shader>& shader_array = (*fsc)->shader_array;
        for (auto shader = shader_array.cbegin(); shader != shader_array.cend(); ++shader)
        {
            PRINT_LOG("Shader name: %hs", shader->name.value.c_str());

            m_Extractor.ClearResults();
            m_Extractor.ClearMacro();

            nw::g3d::tool::g3dif::elem_shading_model el_shading_model;
            el_shading_model.name = (*shader).name.value;
            el_shading_model.material_shader = shader->material_shader;

            std::string sources[ShaderStage_StageCount];

            const std::vector<nw::g3d::tool::g3dif::elem_macro>& macro_array = shader->macro_array;
            for (auto macro = macro_array.cbegin(); macro != macro_array.cend(); ++macro)
            {
                m_Extractor.AddFixedMacro(macro->name.value, macro->value.value);
            }
            for (auto& macro : m_Macros)
            {
                m_Extractor.AddFixedMacro(macro);
            }

            // 各シェーダーステージのソース読み込みとアノテーション抽出
            for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
            {
                if (shader->shader_path[stage])
                {
                    ShaderSrc shader_src = FindFile(include_paths, shader->shader_path[stage]->path.value);
                    if (shader_src.fullpath.empty())
                    {
                        THROW_ERROR_INTERNAL(ERRCODE_INVALID_FILENAME, "Shader '%hs' is not found.", shader->shader_path[stage]->path.value.c_str());
                    }
                    m_Extractor.Extract(shader_src.fullpath, static_cast<ShaderStage>(stage), &m_ExtractedFiles);
                }
            }

            // ステージ間のアノテーション設定をまとめる
            m_Extractor.Merge(unifiedAnnotation);

            // 抽出したシェーダ定義を書き出す。
            m_Extractor.WriteDefinition(el_shading_model, &m_ExtractedFiles);

            // ui_group 循環参照チェック
            {
                nn::util::BitArray::size_type groupArraySize = static_cast<nn::util::BitArray::size_type>( el_shading_model.group_array.size() );
                size_t bitArraySize = nn::util::BitArray::CalculateWorkMemorySize( groupArraySize );
                std::unique_ptr<char[]> pBitArrayWorkMem( new char[ bitArraySize * 2 ] );
                nn::util::BitArray	referedGroupFlagArray( pBitArrayWorkMem.get(), bitArraySize, groupArraySize );
                nn::util::BytePtr pRecursiveCheckSkipFlagArrayWorkMem( pBitArrayWorkMem.get() );
                pRecursiveCheckSkipFlagArrayWorkMem.Advance( bitArraySize );
                nn::util::BitArray	recursiveCheckSkipFlagArray( pRecursiveCheckSkipFlagArrayWorkMem.Get(), bitArraySize, groupArraySize );
                referedGroupFlagArray.reset();			// 任意の探索開始位置から親を辿った時の循環を調べるための flag、探索開始毎にリセットする。探索済みで true
                recursiveCheckSkipFlagArray.reset();	// 探索済みの index を格納する bit 列。探索済みで true

                auto checkRecursivity = [&](std::vector<nw::g3d::tool::g3dif::elem_group>::const_iterator iterElemGroupInitial)
                {
                    auto iterElemGroup = iterElemGroupInitial;
                    for(;;)
                    {
                        int idxGroup = static_cast<int>( std::distance(el_shading_model.group_array.cbegin(), iterElemGroup) );
                        if( recursiveCheckSkipFlagArray.test( idxGroup ) == true )
                        {
                            return;
                        }
                        referedGroupFlagArray.set(idxGroup, true);

                        const char* groupName = (*iterElemGroup).name->c_str();
                        const char* uiGroupName = nullptr;
                        if( !(*iterElemGroup).ui_group )
                        {
                            return;
                        }

                        uiGroupName = (*iterElemGroup).ui_group->group_name.value.c_str();

                        auto iterParentElemGroup =
                            std::find_if(el_shading_model.group_array.cbegin(), el_shading_model.group_array.cend(),
                            [uiGroupName](const nw::g3d::tool::g3dif::elem_group& elemGroup)
                            {
                                return elemGroup.name.value == std::string(uiGroupName);
                            });

                        const nw::g3d::tool::g3dif::elem_group& parentElemGroup = *iterParentElemGroup;

                        int idxParentElemGroup = static_cast<int>( std::distance(el_shading_model.group_array.cbegin(), iterParentElemGroup) );
                        if( referedGroupFlagArray.test(idxParentElemGroup) )
                        {
                            THROW_ERROR(ERRCODE_SHADER_ANNOTATION_RECURSIVE_UI_GROUP,
                                "Recursive ui group in shader annotation. group=\"%hs\" and group=\"%hs\" are recursive.",
                                groupName, parentElemGroup.name.value.c_str() );
                            return;
                        }

                        iterElemGroup = iterParentElemGroup;
                    }
                };

                for( auto iterElemGroup = el_shading_model.group_array.begin();  iterElemGroup != el_shading_model.group_array.end(); ++iterElemGroup )
                {
                    checkRecursivity(iterElemGroup);
                    recursiveCheckSkipFlagArray.Or( &recursiveCheckSkipFlagArray, recursiveCheckSkipFlagArray, referedGroupFlagArray );
                    referedGroupFlagArray.reset();
                }
            }

            // choice を与えられた filter (name, value, default, branch) の範囲に狭める。
            nw::g3d::tool::g3dif::filter_map copy_filter_map(shader->option_filter);
            for (auto el_option = el_shading_model.option_var_array.begin(); el_option != el_shading_model.option_var_array.end(); ++el_option)
            {
                auto comp = copy_filter_map.find(el_option->id.value);

                if (comp == copy_filter_map.end())
                {
                    // option uniform に option id と同じ id で定義された uniform が存在するかどうか
                    const nw::g3d::tool::g3dif::elem_uniform_var* option_uniform = el_shading_model.GetOptionUniform(*el_option);
                    // variation が指定されなかった場合にアノテーション側の情報のみから fsd を生成するモード
                    if (autoExtract && option_uniform != nullptr)
                    {
                        // option uniform にアノテーションが付けられていた場合は自動的に branch を true にする。
                        el_option->Filter(str_wild_card, str_empty);
                        el_option->branch.value = true;
                    }
                    else
                    {
                        // 自動で variation を増やすことはせずデフォルト値を採用する
                        el_option->Filter(str_empty, str_empty);
                    }
                }
                else
                {
                    el_option->Filter(comp->second.choice, comp->second.default_value);

                    // branch の設定をコピーする。
                    el_option->branch = comp->second.branch;
                    copy_filter_map.erase(comp);
                }
            }

            // fsc に書かれていて、アノテーションに存在しない option がある。
            if (!copy_filter_map.empty())
            {
                for (auto iter = copy_filter_map.cbegin(); iter != copy_filter_map.cend(); ++iter)
                {
                    PRINT_ERROR_LOG("There are no shader_option:%hs in the shader code.", iter->first.c_str());
                }
                THROW_ERROR(ERRCODE_SHADER_CONVERTER_XML_VARIATION_ID, " Invalid variation id in fsc.");
            }

            // システム定義のオプションを追加します。
            el_shading_model.AddSystemOption();

            // choice を展開する。
            el_shading_model.Expand();

            elem->shading_model_array.push_back(el_shading_model);
        }
    }

    // fileDB から .modify が付いたソースを削除する。
    auto iterFile = m_ExtractedFiles.begin();
    // インデックスを振りなおす。
    int fileIndex = 0;
    while(iterFile != m_ExtractedFiles.end())
    {
        iterFile->second.first = fileIndex;
        if (iterFile->first.find(NW_MODIFY_FILE) != std::string::npos)
        {
            m_ExtractedFiles.erase(iterFile);
            iterFile = m_ExtractedFiles.begin();
            fileIndex = 0;
            continue;
        }
        ++iterFile;
        ++fileIndex;
    }

    // シェーディングモデル以外の書き出し
    {
        // shader_definition_info の書き出し
        elem->shader_definition_info.code_page = (*fsc)->shader_config_info.code_page.value;
        elem->shader_definition_info.config_path = (*fsc)->relative_path;

        // FileDB の書き出し
        std::vector<const std::pair<const std::string, std::pair<int, std::string>>*> fileVec;
        fileVec.reserve(m_ExtractedFiles.size());
        for (auto iter = m_ExtractedFiles.cbegin(); iter != m_ExtractedFiles.cend(); ++iter)
        {
            fileVec.push_back(&*iter);
        }
        std::sort(fileVec.begin(), fileVec.end(), [](
            const std::pair<const std::string, std::pair<int, std::string>>* lhs,
            const std::pair<const std::string, std::pair<int, std::string>>* rhs) -> bool
        {
            return lhs->second.first < rhs->second.first;
        });

        std::vector<std::string> force_include_array;
        const std::vector<nw::g3d::tool::g3dif::elem_force_include_file>& force_include_file_array = (*fsc)->force_include_file_array;
        force_include_array.reserve(force_include_file_array.size());
        for (auto include = force_include_file_array.cbegin(); include != force_include_file_array.cend(); ++include)
        {
            force_include_array.emplace_back(include->path.value);
        }

        for (auto file = fileVec.cbegin(); file != fileVec.cend(); ++file)
        {
            nw::g3d::tool::g3dif::elem_shader_src el_shader_src;

            // インクルードパスを分離する。
            // ソースファイルのフルパスには必ずインクルードパスが含まれるので find は失敗しないはず。
            std::vector<IncludePath>::const_iterator include_iter = include_paths.cend();
            for (auto include = include_paths.cbegin(); include != include_paths.cend(); ++include)
            {
                if ((*file)->first.find(include->fullPath) != std::string::npos)
                {
                    include_iter = include;
                    break;
                }
            }

            if (include_iter == include_paths.cend())
            {
                THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
            }

            el_shader_src.include_path = include_iter->originalPath;
            std::replace(el_shader_src.include_path->begin(), el_shader_src.include_path->end(), '\\', '/');

            char relativePath[_MAX_PATH];
            PathRelativePathToA(relativePath, include_iter->fullPath.c_str(), FILE_ATTRIBUTE_DIRECTORY, (*file)->first.c_str(), FILE_ATTRIBUTE_NORMAL);
            el_shader_src.path.value.assign(relativePath);
            std::replace(el_shader_src.path->begin(), el_shader_src.path->end(), '\\', '/');
            // .\ を取り除く
            if (el_shader_src.path.value.find("./") == 0)
            {
                el_shader_src.path.value = el_shader_src.path.value.substr(2);
            }
            el_shader_src.stream_index = (*file)->second.first;

            el_shader_src.stream.type = nw::g3d::tool::g3dif::StreamTypeString;
            el_shader_src.stream.count = 1;

            // シェーダソースを UTF-8 で保持する。
            void* rawdata = malloc((*file)->second.second.size() + 1);
            memcpy(rawdata, (*file)->second.second.data(), (*file)->second.second.size());
            // null 文字を追加
            static_cast<char*>(rawdata)[(*file)->second.second.size()] = '\0';

            el_shader_src.stream.rawdata.reset(rawdata, free);

            elem->shader_src_array.push_back(el_shader_src);

            // force_include を探索
            auto found = std::find(force_include_array.begin(), force_include_array.end(), el_shader_src.path.value);
            if (found != force_include_array.end())
            {
                nw::g3d::tool::g3dif::elem_force_include force_include;
                force_include.src_index.value = static_cast<int>(elem->shader_src_array.size()) - 1;
                force_include.path = *found;
                elem->force_include_array.push_back(force_include);
                force_include_array.erase(found);
            }
        }

        if (force_include_array.size() != 0)
        {
            THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
        }

        // stream_index を指すように src_index を埋める
        for (auto el_shading_model = elem->shading_model_array.begin(); el_shading_model != elem->shading_model_array.end(); ++el_shading_model)
        {
            for (int stageIndex = 0; stageIndex < ShaderStage_StageCount; ++stageIndex)
            {
                if (el_shading_model->shader_stage[stageIndex])
                {
                    auto found = m_ExtractedFiles.find(el_shading_model->shader_stage[stageIndex]->path);
                    if (found == m_ExtractedFiles.end())
                    {
                        THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
                    }
                    el_shading_model->shader_stage[stageIndex]->src_index = found->second.first;
                }
            }
        }
    }

    Add(elem);
    m_DefinitionFromConfig = elem;
}

void ShdrFile::ConstructFsdFromBfsha()
{
    // bfsha を elem_shader_definition に変換する。elem_shader_definition は完全に復元できないが、以降の処理に必要な要素は復元できているので問題ない。
    std::shared_ptr<nw::g3d::tool::g3dif::elem_shader_definition> elemShaderDefinition(new nw::g3d::tool::g3dif::elem_shader_definition());

    elemShaderDefinition->name = m_BfshaArray.front().get()->ToData().pShaderArchive.Get()->GetName();
    elemShaderDefinition->path = m_BfshaArray.front().get()->ToData().pShaderArchive.Get()->GetPath();

    // 全 bfsha のフラグ内容と一貫性をチェックする
    nn::Bit16 frontShaderArchiveFlag = m_BfshaArray.front().get()->ToData().pShaderArchive.Get()->ToData().flag;
    if (frontShaderArchiveFlag & nn::g3d::ResShaderArchive::Flag_Souce)
    {
        SetShaderCodeTypeSource(true);
    }
    if (frontShaderArchiveFlag & nn::g3d::ResShaderArchive::Flag_Binary)
    {
        SetShaderCodeTypeBinary(true);
    }
    if (frontShaderArchiveFlag & nn::g3d::ResShaderArchive::Flag_Ir)
    {
        SetShaderCodeTypeIr(true);
    }
    if (frontShaderArchiveFlag & nn::g3d::ResShaderArchive::Flag_BinaryAvailable)
    {
        SetBinaryAvailable(true);
    }
    bool isForceVariation = true;
    for (auto bfsha = m_BfshaArray.cbegin(); bfsha != m_BfshaArray.cend(); ++bfsha)
    {
        nn::Bit16 shaderArchiveFlag = bfsha->get()->ToData().pShaderArchive.Get()->ToData().flag;
        // １つでも forceVariation がない bfsha があればフラグを落とす
        if (!(shaderArchiveFlag & nn::g3d::ResShaderArchive::Flag_ForceVariation))
        {
            isForceVariation = false;
        }
        // ForceVariation 以外のフラグが完全一致していることを確認する
        shaderArchiveFlag &= !nn::g3d::ResShaderArchive::Flag_ForceVariation;
        nn::Bit16 refefenceShaderArchiveFlag = frontShaderArchiveFlag && !nn::g3d::ResShaderArchive::Flag_ForceVariation;
        if (shaderArchiveFlag != refefenceShaderArchiveFlag)
        {
            THROW_ERROR(ERRCODE_SHADER_CONVERTER_NO_COHERENCE, "ResShaderArchive::flag of input .bfsha files does not have consistency.");
        }
        // ShaderArchive 名が一致していることを確認する
        {
            if (strcmp(elemShaderDefinition->name.c_str(), bfsha->get()->ToData().pShaderArchive.Get()->GetName()))
            {
                THROW_ERROR(ERRCODE_SHADER_CONVERTER_NO_COHERENCE, "ShaderArchive name of input .bfsha files does not have consistency. (%hs vs %hs)"
                , elemShaderDefinition->name.c_str(), bfsha->get()->ToData().pShaderArchive.Get()->GetName());
            }
        }
    }
    SetForceVariation(isForceVariation);

    for (auto bfsha = m_BfshaArray.cbegin(); bfsha != m_BfshaArray.cend(); ++bfsha)
    {
        const nn::g3d::ResShaderArchive* pResShaderArchive = bfsha->get()->ToData().pShaderArchive.Get();
        for (int shadingModelIndex = 0; shadingModelIndex < pResShaderArchive->GetShadingModelCount(); ++shadingModelIndex)
        {
            // 入力の bfsha から 全シェーディングモデルを抽出して elem_shading_model に逆変換する
            const nn::g3d::ResShadingModel* pResShadingModel = pResShaderArchive->GetShadingModel(shadingModelIndex);
            nw::g3d::tool::g3dif::elem_shading_model newElemShadingModel;
            newElemShadingModel.name.value = pResShadingModel->GetName();

            // Key
            int keyLength = pResShadingModel->GetKeyLength();
            newElemShadingModel.keyLength = keyLength;
            if (pResShadingModel->GetDefaultProgramIndex() != -1)
            {
                newElemShadingModel.defaultKey.reset(malloc(keyLength * sizeof(u32)));
                memcpy(newElemShadingModel.defaultKey.get(), pResShadingModel->GetKey(pResShadingModel->GetDefaultProgramIndex()), sizeof(u32) * keyLength);
            }

            // Option
            int optionCount = pResShadingModel->GetStaticOptionCount() + pResShadingModel->GetDynamicOptionCount();
            int keyLengthBit = 0;
            for (int optionIndex = 0; optionIndex < optionCount; ++optionIndex)
            {
                nw::g3d::tool::g3dif::elem_option_var elemOptionVar;
                const nn::g3d::ResShaderOption* pResShaderOption;
                if (optionIndex < pResShadingModel->GetStaticOptionCount())
                {
                    pResShaderOption = pResShadingModel->GetStaticOption(optionIndex);
                    elemOptionVar.type.value = nw::g3d::tool::g3dif::elem_option_var::type_static;
                    if (pResShaderOption->IsBranch())
                    {
                        // static かつ branch の場合はバリエーション用のユニフォーム変数を特定する
                        const nn::g3d::ResUniformBlockVar* pResOptionUniformBlockVar = NULL;
                        for (int uniformBlockIndex = 0; uniformBlockIndex < pResShadingModel->GetUniformBlockCount(); ++uniformBlockIndex)
                        {
                            if (pResShadingModel->GetUniformBlock(uniformBlockIndex)->GetType() == nn::g3d::ResUniformBlockVar::Type_Option)
                            {
                                pResOptionUniformBlockVar = pResShadingModel->GetUniformBlock(uniformBlockIndex);
                            }
                        }
                        if(pResOptionUniformBlockVar)
                        {
                            for (int uniformIndex = 0; uniformIndex < pResOptionUniformBlockVar->GetUniformCount(); ++uniformIndex)
                            {
                                const nn::g3d::ResUniformVar* pResUniformVar = pResOptionUniformBlockVar->GetUniform(uniformIndex);
                                if (pResUniformVar->GetOffset() == pResShaderOption->GetBranchOffset())
                                {
                                    // OptionUniform の offset の特定にのみ使用するため、内容は全ステージ共通でよい
                                    for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
                                    {
                                        elemOptionVar.altUniformSymbol[stage] = pResOptionUniformBlockVar->ToData().pUniformDic.Get()->GetKey(uniformIndex).data();
                                    }
                                }
                            }
                        }
                    }
                }
                else
                {
                    pResShaderOption = pResShadingModel->GetDynamicOption(optionIndex - pResShadingModel->GetStaticOptionCount());
                    elemOptionVar.type.value = nw::g3d::tool::g3dif::elem_option_var::type_dynamic;
                }
                elemOptionVar.id.value = pResShaderOption->GetName();
                elemOptionVar.defaultValueIndex = pResShaderOption->GetDefaultIndex();
                elemOptionVar.branch.value = pResShaderOption->IsBranch();
                for (int choiceIndex = 0; choiceIndex < pResShaderOption->GetChoiceCount(); ++choiceIndex)
                {
                    elemOptionVar.choiceArray.emplace_back(pResShaderOption->ToData().pChoiceDic.Get()->GetKey(choiceIndex).data(), "");
                    if (elemOptionVar.branch.value && elemOptionVar.type.value == nw::g3d::tool::g3dif::elem_option_var::type_static)
                    {
                        elemOptionVar.choiceUintArray.push_back(pResShaderOption->ToData().pChoiceValues.Get()[choiceIndex]);
                    }
                }
                int bitWidth = 1;
                for (int i = 2; i < static_cast<int>(pResShaderOption->GetChoiceCount()); i <<= 1, ++bitWidth)
                {
                    // 何もしない
                }
                elemOptionVar.bitWidth = bitWidth;
                elemOptionVar.keyOffset = 32 - bitWidth - pResShaderOption->ToData().bit32Shift + (pResShaderOption->ToData().bit32Index * 32);
                keyLengthBit += bitWidth;
                if (keyLengthBit > 32 || optionIndex == pResShadingModel->GetStaticOptionCount())
                {
                    keyLengthBit = bitWidth;
                }
                newElemShadingModel.option_var_array.push_back(elemOptionVar);
            }

            // Attribute
            for (int attributeIndex = 0; attributeIndex < pResShadingModel->GetAttrCount(); ++attributeIndex)
            {
                nw::g3d::tool::g3dif::elem_attrib_var elemAttributeVar;
                // id と symbol は同じものを入れておく
                elemAttributeVar.id.value = pResShadingModel->ToData().pAttrDic.Get()->GetKey(attributeIndex).data();
                elemAttributeVar.vertex_symbol.name.value = elemAttributeVar.id.value;
                newElemShadingModel.attrib_var_array.push_back(elemAttributeVar);
            }

            // Sampler
            const nn::util::BinPtrToString* pSamplerTable = pResShadingModel->GetShaderInfo()->pSamplerTable.Get();
            for (int samplerIndex = 0; samplerIndex < pResShadingModel->GetSamplerCount(); ++samplerIndex)
            {
                const nn::g3d::ResSamplerVar* pResSampler = pResShadingModel->GetSampler(samplerIndex);
                nw::g3d::tool::g3dif::elem_sampler_var elemSamplerVar;
                elemSamplerVar.id.value = pResShadingModel->GetSamplerName(samplerIndex);
                elemSamplerVar.alt.value = pResSampler->GetAlt();
                for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
                {
                    const nn::util::BinString* pName = pSamplerTable->Get();
                    if (pName && pName->GetLength())
                    {
                        elemSamplerVar.shader_symbol[stage].Validate();
                        elemSamplerVar.shader_symbol[stage]->name.value = pName->GetData();
                    }
                    ++pSamplerTable;
                }
                newElemShadingModel.sampler_var_array.push_back(elemSamplerVar);
            }

            // UniformBlock
            const nn::util::BinPtrToString* pUniformBlockTable = pResShadingModel->GetShaderInfo()->pUniformBlockTable.Get();
            for (int uniformBlockIndex = 0; uniformBlockIndex < pResShadingModel->GetUniformBlockCount(); ++uniformBlockIndex)
            {
                const nn::g3d::ResUniformBlockVar* pResUniformBlockVar = pResShadingModel->GetUniformBlock(uniformBlockIndex);
                nw::g3d::tool::g3dif::elem_block_var elemUniformBlockVar;
                elemUniformBlockVar.id.value = pResShadingModel->GetUniformBlockName(uniformBlockIndex);
                elemUniformBlockVar.type.value = static_cast<nw::g3d::tool::g3dif::elem_block_var::enum_type>(pResUniformBlockVar->GetType());
                for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
                {
                    const nn::util::BinString* pName = pUniformBlockTable->Get();
                    if (pName && pName->GetLength())
                    {
                        elemUniformBlockVar.shader_symbol[stage].Validate();
                        elemUniformBlockVar.shader_symbol[stage]->name.value = pName->GetData();
                    }
                    ++pUniformBlockTable;
                }
                if (elemUniformBlockVar.type.value == nw::g3d::tool::g3dif::elem_block_var::enum_type::material)
                {
                    elemUniformBlockVar.pConvertedDefaultValue.reset(malloc(pResUniformBlockVar->GetSize()));
                    memcpy(elemUniformBlockVar.pConvertedDefaultValue.get(), pResUniformBlockVar->GetDefault(), pResUniformBlockVar->GetSize());
                }
                for (int uniformIndex = 0; uniformIndex < pResUniformBlockVar->GetUniformCount(); ++uniformIndex)
                {
                    const nn::g3d::ResUniformVar* pResUniformVar = pResUniformBlockVar->GetUniform(uniformIndex);
                    nw::g3d::tool::g3dif::elem_uniform_var elemUniformVar;
                    elemUniformVar.id.value = pResUniformBlockVar->GetUniformName(uniformIndex);
                    elemUniformVar.converter.value = pResUniformVar->GetConverter();
                    for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
                    {
                        // symbol には id を入れておく。offset の特定に使用するで全ステージ共通でよい
                        elemUniformVar.shader_symbol[stage].Validate();
                        elemUniformVar.shader_symbol[stage]->name.value = elemUniformVar.id.value;
                    }
                    elemUniformBlockVar.uniform_var_array.push_back(elemUniformVar);
                }
                newElemShadingModel.block_var_array.push_back(elemUniformBlockVar);
            }

            // ShaderStorageBlock
            const nn::util::BinPtrToString* pShaderStorageBlockTable = pResShadingModel->GetShaderInfo()->pShaderStorageBlockTable.Get();
            for (int shaderStorageBlockIndex = 0; shaderStorageBlockIndex < pResShadingModel->GetShaderStorageBlockCount(); ++shaderStorageBlockIndex)
            {
                const nn::g3d::ResShaderStorageBlockVar* pResShaderStorageBlockVar = pResShadingModel->GetShaderStorageBlock(shaderStorageBlockIndex);
                nw::g3d::tool::g3dif::elem_ssbo_block_var elemShaderStorageBlockVar;
                elemShaderStorageBlockVar.id.value = pResShadingModel->GetShaderStorageBlockName(shaderStorageBlockIndex);
                elemShaderStorageBlockVar.type.value = static_cast<nw::g3d::tool::g3dif::elem_ssbo_block_var::enum_type>(pResShaderStorageBlockVar->GetType());
                for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
                {
                    const nn::util::BinString* pName = pShaderStorageBlockTable->Get();
                    if (pName && pName->GetLength())
                    {
                        elemShaderStorageBlockVar.shader_symbol[stage].Validate();
                        elemShaderStorageBlockVar.shader_symbol[stage]->name.value = pName->GetData();
                    }
                    ++pShaderStorageBlockTable;
                }
                for (int uniformIndex = 0; uniformIndex < pResShaderStorageBlockVar->GetShaderStorageCount(); ++uniformIndex)
                {
                    const nn::g3d::ResShaderStorageVar* pResShaderStorageVar = pResShaderStorageBlockVar->GetShaderStorage(uniformIndex);
                    nw::g3d::tool::g3dif::elem_ssbo_uniform_var elemSsboUniformVar;
                    elemSsboUniformVar.id.value = pResShaderStorageBlockVar->GetShaderStorageName(uniformIndex);
                    elemSsboUniformVar.converter.value = pResShaderStorageVar->GetConverter();
                    for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
                    {
                        elemSsboUniformVar.shader_symbol[stage].Validate();
                        elemSsboUniformVar.shader_symbol[stage]->name.value = elemSsboUniformVar.id.value;
                    }
                    elemShaderStorageBlockVar.uniform_var_array.push_back(elemSsboUniformVar);
                }
                newElemShadingModel.ssbo_block_var_array.push_back(elemShaderStorageBlockVar);
            }

            auto pReferenceElemShadingModel = std::find_if(elemShaderDefinition->shading_model_array.begin(), elemShaderDefinition->shading_model_array.end(),
                [&](const nw::g3d::tool::g3dif::elem_shading_model &shadingModel) { return newElemShadingModel.name.value == shadingModel.name.value; });
            if (pReferenceElemShadingModel == elemShaderDefinition->shading_model_array.end())
            {
                // シェーディングモデルが追加されていない場合は追加する
                elemShaderDefinition->shading_model_array.push_back(newElemShadingModel);
            }
            else
            {
                // 既に同じシェーディングモデルが追加してある場合、整合性があるかをチェックしつつ既存のオブジェクトに不完全な要素を補間していく。
                // デフォルトキー
                if (newElemShadingModel.defaultKey)
                {
                    pReferenceElemShadingModel->defaultKey.reset(malloc(keyLength * sizeof(u32)));
                    memcpy(pReferenceElemShadingModel->defaultKey.get(), newElemShadingModel.defaultKey.get(), sizeof(u32) * keyLength);
                }

                // オプションは fsd で決定されるため、個数は完全一致しているべき。
                if (pReferenceElemShadingModel->option_var_array.size() != newElemShadingModel.option_var_array.size())
                {
                    THROW_ERROR(ERRCODE_SHADER_CONVERTER_NO_COHERENCE, "ResOptionVar count of input .bfsha files does not have consistency.");
                }
                for (int optionIndex = 0; optionIndex < newElemShadingModel.option_var_array.size(); ++optionIndex)
                {
                    const nw::g3d::tool::g3dif::elem_option_var& referenceOption = pReferenceElemShadingModel->option_var_array[optionIndex];
                    const nw::g3d::tool::g3dif::elem_option_var& newOption = newElemShadingModel.option_var_array[optionIndex];
                    if (referenceOption.branch.value != newOption.branch.value || referenceOption.id.value != newOption.id.value ||
                        referenceOption.type.value != newOption.type.value)
                    {
                        THROW_ERROR(ERRCODE_SHADER_CONVERTER_NO_COHERENCE, "ResOptionVar of input .bfsha files does not have consistency.");
                    }
                }

                // attribute
                for (int attributeIndex = 0; attributeIndex < newElemShadingModel.attrib_var_array.size(); ++attributeIndex)
                {
                    nw::g3d::tool::g3dif::elem_attrib_var& newAttribute = newElemShadingModel.attrib_var_array[attributeIndex];
                    auto found = std::find_if(pReferenceElemShadingModel->attrib_var_array.cbegin(), pReferenceElemShadingModel->attrib_var_array.cend(),
                        [&](const nw::g3d::tool::g3dif::elem_attrib_var &attribute) { return newAttribute.id.value == attribute.id.value; });
                    if (found == pReferenceElemShadingModel->attrib_var_array.cend())
                    {
                        // 既存のシェーディングモデルに存在しない頂点属性がある場合は追加する
                        // 一括コンバートのバイナリーと並び順で差分が出るが仕方がない
                        pReferenceElemShadingModel->attrib_var_array.push_back(newAttribute);
                    }
                }

                // Sampler
                for (int samplerIndex = 0; samplerIndex < newElemShadingModel.sampler_var_array.size(); ++samplerIndex)
                {
                    nw::g3d::tool::g3dif::elem_sampler_var& newSampler = newElemShadingModel.sampler_var_array[samplerIndex];
                    auto found = std::find_if(pReferenceElemShadingModel->sampler_var_array.begin(), pReferenceElemShadingModel->sampler_var_array.end(),
                        [&](const nw::g3d::tool::g3dif::elem_sampler_var &sampler) { return newSampler.id.value == sampler.id.value; });
                    if (found == pReferenceElemShadingModel->sampler_var_array.end())
                    {
                        // 既存のシェーディングモデルに存在しないサンプラーがある場合は追加する
                        // 一括コンバートのバイナリーと並び順で差分が出るが仕方がない
                        pReferenceElemShadingModel->sampler_var_array.push_back(newSampler);
                    }
                    else
                    {
                        for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
                        {
                            // 既存の symbol が空 & 新 symbol が有効な場合は symbol を有効にする
                            if (!found->shader_symbol[stage] && newSampler.shader_symbol[stage])
                            {
                                found->shader_symbol[stage].Validate();
                                found->shader_symbol[stage]->name.value = newSampler.shader_symbol[stage]->name.value;
                            }
                        }
                    }
                }

                // UniformBlock
                for (int uniformBlockIndex = 0; uniformBlockIndex < pResShadingModel->GetUniformBlockCount(); ++uniformBlockIndex)
                {
                    nw::g3d::tool::g3dif::elem_block_var& newBlock = newElemShadingModel.block_var_array[uniformBlockIndex];
                    auto found = std::find_if(pReferenceElemShadingModel->block_var_array.begin(), pReferenceElemShadingModel->block_var_array.end(),
                        [&](const nw::g3d::tool::g3dif::elem_block_var &block) { return newBlock.id.value == block.id.value; });
                    if (found == pReferenceElemShadingModel->block_var_array.end())
                    {
                        // 既存のシェーディングモデルに存在しない UBO がある場合は追加する
                        // 一括コンバートのバイナリーと並び順で差分が出るが仕方がない
                        pReferenceElemShadingModel->block_var_array.push_back(newBlock);
                    }
                    else
                    {
                        for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
                        {
                            // 既存の symbol が空 & 新 symbol が有効な場合は symbol を有効にする
                            if (!found->shader_symbol[stage] && newBlock.shader_symbol[stage])
                            {
                                found->shader_symbol[stage].Validate();
                                found->shader_symbol[stage]->name.value = newBlock.shader_symbol[stage]->name.value;
                            }
                        }
                    }
                }

                // ShaderStorageBlock
                for (int shaderStorageBlockIndex = 0; shaderStorageBlockIndex < pResShadingModel->GetShaderStorageBlockCount(); ++shaderStorageBlockIndex)
                {
                    nw::g3d::tool::g3dif::elem_ssbo_block_var& newBlock = newElemShadingModel.ssbo_block_var_array[shaderStorageBlockIndex];
                    auto found = std::find_if(pReferenceElemShadingModel->ssbo_block_var_array.begin(), pReferenceElemShadingModel->ssbo_block_var_array.end(),
                        [&](const nw::g3d::tool::g3dif::elem_ssbo_block_var &block) { return newBlock.id.value == block.id.value; });
                    if (found == pReferenceElemShadingModel->ssbo_block_var_array.end())
                    {
                        // 既存のシェーディングモデルに存在しない SSBO がある場合は追加する
                        // 一括コンバートのバイナリーと並び順で差分が出るが仕方がない
                        pReferenceElemShadingModel->ssbo_block_var_array.push_back(newBlock);
                    }
                    else
                    {
                        for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
                        {
                            // 既存の symbol が空 & 新 symbol が有効な場合は symbol を有効にする
                            if (!found->shader_symbol[stage] && newBlock.shader_symbol[stage])
                            {
                                found->shader_symbol[stage].Validate();
                                found->shader_symbol[stage]->name.value = newBlock.shader_symbol[stage]->name.value;
                            }
                        }
                    }
                }
            }
        }
    }
    Add(elemShaderDefinition);
}

void ShdrFile::MakeVariation()
{
    // 複数の fsv には未対応
    if (m_FsvArray.size() >= 2)
    {
        THROW_ERROR(ERRCODE_UNSUPPORTED, "More than two fsv files are unsupported.");
    }

    // 複数の fsd には未対応
    if (m_FsdArray.size() >= 2)
    {
        THROW_ERROR(ERRCODE_UNSUPPORTED, "More than two fsd files are unsupported.");
    }

    const nw::g3d::tool::g3dif::elem_shader_definition* el_shader_definition = m_FsdArray.front().get();
    const std::vector<nw::g3d::tool::g3dif::elem_shading_model>& definition_array = el_shader_definition->shading_model_array;

    m_ArchiveBuilder.m_ShadingModelBuilderArray.clear();
    m_ArchiveBuilder.m_ForceIncludePathArray.clear();
    m_ArchiveBuilder.m_ShadingModelBuilderArray.resize(definition_array.size());
    m_ArchiveBuilder.m_ForceIncludePathArray.reserve(el_shader_definition->force_include_array.size());
    for (auto el_force_include = el_shader_definition->force_include_array.cbegin(); el_force_include != el_shader_definition->force_include_array.cend(); ++el_force_include)
    {
        m_ArchiveBuilder.m_ForceIncludePathArray.push_back(el_force_include->path);
    }

    int idxShadingModel = 0;
    for (auto el_shading_model = definition_array.cbegin(); el_shading_model != definition_array.cend(); ++el_shading_model, ++idxShadingModel)
    {
        // オプションが１つも存在しない場合は例外。fsd 読み込み時に g3d のシステムオプションが追加されるのでオプションは必ず存在する。
        if (el_shading_model->option_var_array.size() == 0)
        {
            THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
        }

        ShadingModelBuilder& shadingModelBuilder = m_ArchiveBuilder.m_ShadingModelBuilderArray[idxShadingModel];
        shadingModelBuilder.m_Name = el_shading_model->name.value;
        shadingModelBuilder.m_KeyLength = static_cast<int>(el_shading_model->keyLength);

        for (int stageIndex = 0; stageIndex < ShaderStage_StageCount; ++stageIndex)
        {
            if (el_shading_model->shader_stage[stageIndex])
            {
                shadingModelBuilder.m_ShaderPath[stageIndex] = el_shader_definition->shader_src_array[el_shading_model->shader_stage[stageIndex]->src_index.value].path.value;
            }
        }

        // streamout
        if (el_shading_model->streamout)
        {
            nw::g3d::tool::util::Split(shadingModelBuilder.m_StreamoutArray, el_shading_model->streamout->varying.value, " \t");
        }

        // symbol
        int idxAttrib = 0;
        shadingModelBuilder.m_AttribArray.resize(el_shading_model->attrib_var_array.size());
        for (auto attrib = el_shading_model->attrib_var_array.cbegin();
            attrib != el_shading_model->attrib_var_array.cend();
            ++attrib, ++idxAttrib)
        {
            shadingModelBuilder.m_AttribArray[idxAttrib].m_Symbol[ShaderStage_Vertex] = attrib->vertex_symbol.name.value;
        }
        int idxSampler = 0;
        shadingModelBuilder.m_SamplerArray.resize(el_shading_model->sampler_var_array.size());
        for (auto sampler = el_shading_model->sampler_var_array.cbegin();
            sampler != el_shading_model->sampler_var_array.cend();
            ++sampler, ++idxSampler)
        {
            for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
            {
                if (sampler->shader_symbol[stage])
                {
                    shadingModelBuilder.m_SamplerArray[idxSampler].m_Symbol[stage] = sampler->shader_symbol[stage]->name.value;
                }
            }
        }
        int idxBlock = 0;
        shadingModelBuilder.m_BlockArray.resize(el_shading_model->block_var_array.size());
        for (auto block = el_shading_model->block_var_array.cbegin(); block != el_shading_model->block_var_array.cend(); ++block, ++idxBlock)
        {
            for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
            {
                if (block->shader_symbol[stage])
                {
                    shadingModelBuilder.m_BlockArray[idxBlock].m_Symbol[stage] = block->shader_symbol[stage]->name.value;
                }
            }
        }
        idxBlock = 0;
        shadingModelBuilder.m_SsboBlockArray.resize(el_shading_model->ssbo_block_var_array.size());
        for (auto block = el_shading_model->ssbo_block_var_array.cbegin(); block != el_shading_model->ssbo_block_var_array.cend(); ++block, ++idxBlock)
        {
            for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
            {
                if (block->shader_symbol[stage])
                {
                    shadingModelBuilder.m_SsboBlockArray[idxBlock].m_Symbol[stage] = block->shader_symbol[stage]->name.value;
                }
            }
        }

        int idxOption = 0;
        shadingModelBuilder.m_OptionArray.resize(el_shading_model->option_var_array.size());
        shadingModelBuilder.m_OptionDefaultIndex.resize(el_shading_model->option_var_array.size());
        for (auto option = el_shading_model->option_var_array.cbegin();
            option != el_shading_model->option_var_array.cend();
            ++option, ++idxOption)
        {
            for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
            {
                if (option->shader_symbol[stage])
                {
                    shadingModelBuilder.m_OptionArray[idxOption].m_Symbol[stage] = option->shader_symbol[stage]->name.value;
                }
            }

            shadingModelBuilder.m_OptionDefaultIndex[idxOption] = option->defaultValueIndex;
        }

        // マージの場合はバリエーションの設定を行う必要が無いためここまで
        if (m_ConvertMode == ConvertMode_MergeBfsha)
        {
            continue;
        }

        // macro
        int idxMacro = 0;
        shadingModelBuilder.m_MacroArray.resize(el_shading_model->macro_array.size());
        for (auto macro = el_shading_model->macro_array.cbegin();
            macro != el_shading_model->macro_array.cend();
            ++macro, ++idxMacro)
        {
            shadingModelBuilder.m_MacroArray[idxMacro].m_Name = macro->name.value;
            shadingModelBuilder.m_MacroArray[idxMacro].m_Value = macro->value.value;
        }

        // 参照シェーダソース
        int idxSource = 0;
        shadingModelBuilder.m_ShaderSourceArray.resize(el_shader_definition->shader_src_array.size());
        for (auto source = el_shader_definition->shader_src_array.cbegin();
            source != el_shader_definition->shader_src_array.cend();
            ++source, ++idxSource)
        {
            shadingModelBuilder.m_ShaderSourceArray[idxSource].m_Path = source->path.value;
            shadingModelBuilder.m_ShaderSourceArray[idxSource].m_Stream = source->stream.rawdata;
        }

        std::vector<int> choiceIndexArray;
        choiceIndexArray.resize(el_shading_model->option_var_array.size(), 0);
        std::vector<bool> choiceLockArray;
        choiceLockArray.resize(el_shading_model->option_var_array.size(), false);

        // オプション変数に対応する uniform のシンボル名を取得する。
        // uniform の情報を保持させます。
        shadingModelBuilder.m_OptionUniformInfoArray.resize( el_shading_model->option_var_array.size() );
        auto pUniformBlockNameForOption = GetElemBlockVarForOption( el_shading_model->block_var_array );
        for( int optionVarIdx = 0; optionVarIdx < el_shading_model->option_var_array.size(); ++optionVarIdx )
        {
            auto& optionUniformInfo = shadingModelBuilder.m_OptionUniformInfoArray[optionVarIdx];
            auto pElemUniformVar = GetElemUniformVar( el_shading_model->block_var_array,
                                                      nw::g3d::tool::g3dif::elem_block_var::option,
                                                      el_shading_model->option_var_array[optionVarIdx].id.value );

            if( pElemUniformVar == nullptr )
            {
                continue;
            }

            if( nw::g3d::tool::g3dif::elem_shader_param::type_srt2d <= pElemUniformVar->type.value )
            {
                THROW_ERROR( ERRCODE_UNSUPPORTED, "Uniform constant type doesn't support sampler type.\n" );
            }
            for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
            {
                if (pUniformBlockNameForOption->shader_symbol[stage])
                {
                    optionUniformInfo.m_pOptionUniformBlockName[stage] = &pUniformBlockNameForOption->shader_symbol[stage]->name.value;
                }
                if (pElemUniformVar->shader_symbol[stage])
                {
                    optionUniformInfo.m_pOptionUniformName[stage] = &pElemUniformVar->shader_symbol[stage].Get().name.value;
                }
            }

            optionUniformInfo.m_pElemUniformVar = pElemUniformVar;
            optionUniformInfo.m_pOptionUniformId = &el_shading_model->option_var_array[optionVarIdx].id.value;

            auto& paramFormat = nw::g3d::tool::g3dif::elem_shader_param::GetShaderParamFormat( pElemUniformVar->type.value );
            if( nw::g3d::tool::g3dif::elem_shader_param::type_float2x2 <= pElemUniformVar->type.value )
            {
                optionUniformInfo.m_Type = nngfxToolShaderCompilerVariableType_Float32;
                optionUniformInfo.matrix.column = static_cast<uint8_t>( paramFormat.col );
                optionUniformInfo.matrix.row = static_cast<uint8_t>( paramFormat.row );
            }
            else
            {
                optionUniformInfo.m_Type = static_cast<nngfxToolShaderCompilerVariableType>( pElemUniformVar->type.value / 4 );
                optionUniformInfo.m_VectorComponents = static_cast<uint8_t>( paramFormat.col );
            }
        }

        // fsv が存在している場合
        if (m_FsvArray.size() > 0)
        {
            const nw::g3d::tool::g3dif::elem_shader_variation* el_shader_variation = m_FsvArray.front().get();
            const std::vector<nw::g3d::tool::g3dif::elem_target_shader>& target_shader_array = el_shader_variation->target_shader_array;
            for (auto el_target_shader = target_shader_array.cbegin(); el_target_shader != target_shader_array.cend(); ++el_target_shader)
            {
                if (el_target_shader->shading_model_name.value == el_shading_model->name.value)
                {
                    // fsv で指定されたバリエーションを生成
                    for (auto program = el_target_shader->shader_program_array.cbegin();
                        program != el_target_shader->shader_program_array.cend();
                        ++program)
                    {
                        // オプションの重複チェック
                        if (program->option_array.size() > 1)
                        {
                            for (auto option = program->option_array.cbegin(); option != program->option_array.cend() - 1; ++option)
                            {
                                auto found = std::find_if(option + 1, program->option_array.cend(),
                                    [&](const nw::g3d::tool::g3dif::elem_option &op) { return option->id.value == op.id.value; });
                                if (found != program->option_array.cend())
                                {
                                    THROW_ERROR(ERRCODE_SHADER_CONVERTER_OVERLAP_OPTION,
                                        "option(%hs) is duplicated.\n  target shader:%hs, program:%d",
                                        option->id.value.c_str(), el_target_shader->shading_model_name.value.c_str(),
                                        std::distance(el_target_shader->shader_program_array.cbegin(), program));
                                }
                            }
                        }

                        shadingModelBuilder.m_ShaderBuilderArray.resize(shadingModelBuilder.m_ShaderBuilderArray.size() + 1);
                        ShaderBuilder& shaderBuilder = shadingModelBuilder.m_ShaderBuilderArray[shadingModelBuilder.m_ShaderBuilderArray.size() - 1];

                        // 全オプションの全 choice に対応する bool 要素を持つ変数
                        std::vector<std::vector<bool>> choiceEnableArray;
                        choiceEnableArray.resize(el_shading_model->option_var_array.size());
                        int optionCount = static_cast<int>(el_shading_model->option_var_array.size());
                        for (int optionIndex = 0; optionIndex < optionCount; ++optionIndex)
                        {
                            // デフォルト挙動は false なのでバリエーションは作られない。
                            choiceEnableArray[optionIndex].resize(
                                el_shading_model->option_var_array[optionIndex].choiceArray.size(), false);
                        }

                        int optionSetIndex = 0;
                        for (auto el_option = el_shading_model->option_var_array.cbegin(); el_option != el_shading_model->option_var_array.cend(); ++el_option, ++optionSetIndex)
                        {
                            auto& optionEnableArray = choiceEnableArray[optionSetIndex];

                            auto option =
                                std::find_if(program->option_array.cbegin(), program->option_array.cend(), VariationOptionFinder(el_option->id.value));
                            if (option != program->option_array.cend())
                            {
                                for (auto choice = option->choiceArray.cbegin(); choice != option->choiceArray.cend(); ++choice)
                                {
                                    const nw::g3d::tool::g3dif::elem_option_var& option_var = el_shading_model->option_var_array[optionSetIndex];
                                    if (choice->first == str_wild_card || (option_var.branch.value && !m_IsForceVariation))
                                    {
                                        // * が指定された場合と branch=true の場合は全て enable にする。
                                        for (int choiceIndex = 0; choiceIndex < static_cast<int>(optionEnableArray.size()); ++choiceIndex)
                                        {
                                            optionEnableArray[choiceIndex] = true;
                                        }
                                        break;
                                    }
                                    else
                                    {
                                        // 指定された choice のみを有効にする。
                                        int valueIndex = nw::g3d::tool::g3dif::IncludeValue(el_option->choiceArray, choice->first);
                                        if (valueIndex == -1)
                                        {
                                            THROW_ERROR(ERRCODE_SHADER_CONVERTER_SHADER_VARIATION_VALUE_NOT_FOUND,
                                                "The option value(%hs: %hs) was not found in shading model(%hs).",
                                                option->id.value.c_str(), choice->first.c_str(), el_target_shader->shading_model_name.value.c_str());
                                        }

                                        optionEnableArray[valueIndex] = true;
                                    }
                                }
                            }
                            else
                            {
                                // option が指定されなかった場合はデフォルトインデクスになる
                                optionEnableArray[el_option->defaultValueIndex] = true;
                            }
                        }

                        // １つの choice のみが有効の場合は choiceIndexArray と choiceLockArray で指定する。
                        std::fill(choiceIndexArray.begin(), choiceIndexArray.end(), 0);
                        std::fill(choiceLockArray.begin(), choiceLockArray.end(), false);
                        // 最適化のために branch の場合はデフォルトインデクスで埋める。
                        // choice が 1 つに絞られている場合はその choice で埋める
                        for (int optionIndex = 0; optionIndex < optionCount; ++optionIndex)
                        {
                            const nw::g3d::tool::g3dif::elem_option_var& option_var = el_shading_model->option_var_array[optionIndex];
                            if (option_var.branch.value && !m_IsForceVariation)
                            {
                                choiceLockArray[optionIndex] = true;
                                int defaultIndex = shadingModelBuilder.m_OptionDefaultIndex[optionIndex];
                                choiceIndexArray[optionIndex] = defaultIndex;
                                if (!choiceEnableArray[optionIndex][defaultIndex])
                                {
                                    THROW_ERROR(ERRCODE_INTERNAL,
                                        "Internal Error. The option(%hs) uses branch. But the default value(%hs) was not found.",
                                        option_var.id.value.c_str(), option_var.default_value.value.c_str());
                                }
                            }
                            else
                            {
                                int uniqueChoiceIndex = 0;
                                int enableChoiceCount = 0;
                                for (int choiceIndex = 0; choiceIndex < static_cast<int>(choiceEnableArray[optionIndex].size()); ++choiceIndex)
                                {
                                    if (choiceEnableArray[optionIndex][choiceIndex])
                                    {
                                        ++enableChoiceCount;
                                        uniqueChoiceIndex = choiceIndex;
                                    }
                                }
                                if (enableChoiceCount <= 1)
                                {
                                    choiceLockArray[optionIndex] = true;
                                    choiceIndexArray[optionIndex] = uniqueChoiceIndex;
                                }
                            }
                        }

                        // シェーダバリエーション用の choice 生成
                        CreateVariations(
                            shaderBuilder.m_ShaderVariationBuilderArray,
                            choiceIndexArray,
                            choiceEnableArray,
                            choiceLockArray,
                            shadingModelBuilder.m_ShaderBuilderArray,
                            *el_shading_model,
                            m_IsForceVariation);
                    }
                }
            }
        }
        else
        {
            // fsv で制限しない場合は全バリエーションの組み合わせを生成する。
            shadingModelBuilder.m_ShaderBuilderArray.resize(1);
            ShaderBuilder& shaderBuilder = shadingModelBuilder.m_ShaderBuilderArray[0];
            std::vector<std::vector<bool>> choiceEnableArray;
            choiceEnableArray.resize(el_shading_model->option_var_array.size());
            int optionCount = static_cast<int>(el_shading_model->option_var_array.size());
            for (int optionIndex = 0; optionIndex < optionCount; ++optionIndex)
            {
                choiceEnableArray[optionIndex].resize(
                    el_shading_model->option_var_array[optionIndex].choiceArray.size(), true);
            }

            // 最適化のために branch の場合はデフォルトインデクスで埋める。
            for (int optionIndex = 0; optionIndex < optionCount; ++optionIndex)
            {
                const nw::g3d::tool::g3dif::elem_option_var& option_var = el_shading_model->option_var_array[optionIndex];
                if (option_var.branch.value && !m_IsForceVariation)
                {
                    choiceLockArray[optionIndex] = true;
                    choiceIndexArray[optionIndex] = shadingModelBuilder.m_OptionDefaultIndex[optionIndex];
                }
            }

            // シェーダバリエーション用の choice 生成
            CreateVariations(
                shaderBuilder.m_ShaderVariationBuilderArray,
                choiceIndexArray,
                choiceEnableArray,
                choiceLockArray,
                shadingModelBuilder.m_ShaderBuilderArray,
                *el_shading_model,
                m_IsForceVariation);
        }

        // 部分バリエーション化
        if( m_VariationBeginRatio != 0.0f || m_VariationEndRatio != 1.0f )
        {
            if( m_VariationBeginRatio < 0.0f || m_VariationBeginRatio > 1.0f ||
                m_VariationEndRatio < 0.0f || m_VariationEndRatio > 1.0f ||
                m_VariationBeginRatio > m_VariationEndRatio )
            {
                THROW_ERROR( ERRCODE_INVALID_OPTION_SET, "Variation ratio [%f,%f] is invalid.",
                    m_VariationBeginRatio, m_VariationEndRatio );
            }
            int allVariationCount = 0;
            for( auto&& shaderBuilder : shadingModelBuilder.m_ShaderBuilderArray )
            {
                allVariationCount += nw::g3d::tool::NumericCast<int>( shaderBuilder.m_ShaderVariationBuilderArray.size() );
            }
            int variationBegin = static_cast< int >( std::ceilf( static_cast< float >( allVariationCount - 1 ) * m_VariationBeginRatio ) );
            int variationEnd = static_cast< int >( static_cast<float>( allVariationCount - 1 ) * m_VariationEndRatio );
            decltype( shadingModelBuilder.m_ShaderBuilderArray ) shaderBuilderArray;
            shaderBuilderArray.reserve( shadingModelBuilder.m_ShaderBuilderArray.size() );
            int idxVariation = 0;
            for( int idxShaderBuilder = 0, shaderBuilderCount = static_cast<int>( shadingModelBuilder.m_ShaderBuilderArray.size() );
                idxShaderBuilder < shaderBuilderCount && idxVariation <= variationEnd; ++idxShaderBuilder )
            {
                auto& srcShaderBuilder = shadingModelBuilder.m_ShaderBuilderArray[ idxShaderBuilder ];
                auto& srcShaderVariationBuilderArray = srcShaderBuilder.m_ShaderVariationBuilderArray;
                int variationCount = static_cast<int>( srcShaderVariationBuilderArray.size() );
                auto idxBegin = std::max NN_PREVENT_MACRO_FUNC( variationBegin - idxVariation, 0 );
                auto idxEnd = std::min NN_PREVENT_MACRO_FUNC( variationEnd - idxVariation, variationCount - 1 );
                if( idxBegin <= idxEnd )
                {
                    shaderBuilderArray.resize( shaderBuilderArray.size() + 1 );
                    auto& dstShaderBuilder = shaderBuilderArray.back();
                    auto& dstShaderVariationBuilderArray = dstShaderBuilder.m_ShaderVariationBuilderArray;
                    if( idxBegin == 0 && idxEnd == variationCount - 1 )
                    {
                        srcShaderVariationBuilderArray.swap( dstShaderVariationBuilderArray );
                    }
                    else
                    {
                        dstShaderVariationBuilderArray.reserve( idxEnd - idxBegin + 1 );
                        std::copy( srcShaderVariationBuilderArray.cbegin() + idxBegin,
                            srcShaderVariationBuilderArray.cbegin() + idxEnd + 1,
                            std::back_inserter( dstShaderVariationBuilderArray ) );
                    }
                }
                idxVariation += variationCount;
            }
            shadingModelBuilder.m_ShaderBuilderArray.swap( shaderBuilderArray );
        }
    }
}

namespace {
bool BuildFixedMacroMap(std::unordered_map<std::string, std::string>& fixedMacroMap, const StringRef& name, const StringRef& value)
{
    auto comp = fixedMacroMap.find(name.Str());
    if (comp == fixedMacroMap.cend())
    {
        fixedMacroMap.insert(std::make_pair(nw::g3d::tool::util::EraseNull(name.Str()), value.Str()));
        return true;
    }
    comp->second = value.Str();
    return false;
}

bool BuildFixedMacroMap(std::unordered_map<std::string, std::string>& fixedMacroMap, const StringRef& macro)
{
    auto posEq = macro.Str().find_first_of('=');
    if (posEq == std::string::npos)
    {
        return BuildFixedMacroMap(fixedMacroMap, macro.Str(), "1");
    }
    return BuildFixedMacroMap(fixedMacroMap, macro.Str().substr(0, posEq), macro.Str().data() + posEq + 1);
}
}

static const std::string g_SystemStageMacroName[ShaderStage_StageCount] = {
    "NN_G3D_VERTEX_SHADER",
    "NN_G3D_GEOMETRY_SHADER",
    "NN_G3D_FRAGMENT_SHADER",
    "NN_G3D_COMPUTE_SHADER"
};
static const std::string g_SystemStageMacroValue = "1";

void ShdrFile::CalculateShaderFileSize(std::shared_ptr<Context> pCtx)
{
    // 複数の fsd には未対応
    if (m_FsdArray.size() >= 2)
    {
        THROW_ERROR(ERRCODE_UNSUPPORTED, "More than two fsd files are unsupported.");
    }

    Context& shdr_ctx = *pCtx;

    // bfres のマージの場合
    if (m_ConvertMode == ConvertMode_MergeBfsha)
    {
        // ubo,ssbo のサイズと id の紐づけ、uniform の offset と id 紐づけ。
        // sampler,ubo,ssbo,attribute の location table を復元
        // shaderConverter.dll によるマージ
        // ShaderProgram をシェーダーキーでソート
        auto& shadingModelBuilderArray = m_ArchiveBuilder.m_ShadingModelBuilderArray;
        for (auto shadingModelBuilder = shadingModelBuilderArray.begin(); shadingModelBuilder != shadingModelBuilderArray.end(); ++shadingModelBuilder)
        {
            PRINT_LOG("Shading model: %hs", shadingModelBuilder->m_Name.c_str());
            shdr_ctx.m_ShaderContexts.resize(shdr_ctx.m_ShaderContexts.size() + 1);
            Context::ShaderContext& shaderContext = shdr_ctx.m_ShaderContexts.back();

            // コンパイル結果の UniformBlock,Uniform,SsboBlock,Ssbo のテーブルを復元する
            for (auto bfsha = m_BfshaArray.cbegin(); bfsha != m_BfshaArray.cend(); ++bfsha)
            {
                const nn::g3d::ResShaderArchive* pResShaderArchive = bfsha->get()->ToData().pShaderArchive.Get();
                for (int shadingModelIndex = 0; shadingModelIndex < pResShaderArchive->GetShadingModelCount(); ++shadingModelIndex)
                {
                    const nn::g3d::ResShadingModel* pResShadingModel = pResShaderArchive->GetShadingModel(shadingModelIndex);
                    if (strcmp(shadingModelBuilder->m_Name.c_str(), pResShadingModel->GetName()))
                    {
                        continue;
                    }

                    // UniformBlock
                    const nn::util::BinPtrToString* pUniformBlockTable = pResShadingModel->GetShaderInfo()->pUniformBlockTable.Get();
                    for (int uniformBlockIndex = 0; uniformBlockIndex < pResShadingModel->GetUniformBlockCount(); ++uniformBlockIndex)
                    {
                        const nn::g3d::ResUniformBlockVar* pResUniformBlockVar = pResShadingModel->GetUniformBlock(uniformBlockIndex);
                        for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
                        {
                            const nn::util::BinString* pName = pUniformBlockTable->Get();
                            if (pName)
                            {
                                UniformBlockMap& uniformBlockMap = shaderContext.m_UniformBlockMap;
                                auto comp = uniformBlockMap.find(pName->GetData());
                                if (comp == uniformBlockMap.end())
                                {
                                    uniformBlockMap.insert(std::make_pair(pName->GetData(), static_cast<size_t>(pResUniformBlockVar->GetSize())));
                                }
                            }
                            ++pUniformBlockTable;
                        }
                        for (int uniformIndex = 0; uniformIndex < pResUniformBlockVar->GetUniformCount(); ++uniformIndex)
                        {
                            const nn::g3d::ResUniformVar* pResUniformVar = pResUniformBlockVar->GetUniform(uniformIndex);
                            // offset の特定にのみ使用するので、offset だけを設定しておく
                            ShaderCompilerManager::UniformInfo info;
                            info.offset = static_cast<int>(pResUniformVar->GetOffset());
                            shaderContext.m_UniformMap.insert(std::make_pair(pResUniformBlockVar->ToData().pUniformDic.Get()->GetKey(uniformIndex).data(), info));
                        }
                    }

                    // ShaderStorageBlock
                    const nn::util::BinPtrToString* pShaderStorageBlockTable = pResShadingModel->GetShaderInfo()->pShaderStorageBlockTable.Get();
                    for (int shaderStorageBlockIndex = 0; shaderStorageBlockIndex < pResShadingModel->GetShaderStorageBlockCount(); ++shaderStorageBlockIndex)
                    {
                        const nn::g3d::ResShaderStorageBlockVar* pResShaderStorageBlockVar = pResShadingModel->GetShaderStorageBlock(shaderStorageBlockIndex);
                        for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
                        {
                            const nn::util::BinString* pName = pShaderStorageBlockTable->Get();
                            if (pName)
                            {
                                SsboBlockMap& ssboBlockMap = shaderContext.m_SsboBlockMap;
                                auto comp = ssboBlockMap.find(pName->GetData());
                                if (comp == ssboBlockMap.end())
                                {
                                    ssboBlockMap.insert(std::make_pair(pName->GetData(), static_cast<size_t>(pResShaderStorageBlockVar->GetSize())));
                                }
                            }
                            ++pShaderStorageBlockTable;
                        }
                        for (int uniformIndex = 0; uniformIndex < pResShaderStorageBlockVar->GetShaderStorageCount(); ++uniformIndex)
                        {
                            const nn::g3d::ResShaderStorageVar* pResShaderStorageVar = pResShaderStorageBlockVar->GetShaderStorage(uniformIndex);
                            ShaderCompilerManager::SsboInfo info;
                            info.offset = static_cast<int>(pResShaderStorageVar->GetOffset());
                            shaderContext.m_SsboMap.insert(std::make_pair(pResShaderStorageBlockVar->ToData().pShaderStorageDic.Get()->GetKey(uniformIndex).data(), info));
                        }
                    }
                }
            }

            auto& programs = shaderContext.m_Programs;
            int gfxShaderVariationTotalCount = 0;
            int g3dShaderProgramTotalCount = 0;
            for (auto bfsha = m_BfshaArray.cbegin(); bfsha != m_BfshaArray.cend(); ++bfsha)
            {
                const nn::g3d::ResShaderArchive* pResShaderArchive = bfsha->get()->ToData().pShaderArchive.Get();
                for (int shadingModelIndex = 0; shadingModelIndex < pResShaderArchive->GetShadingModelCount(); ++shadingModelIndex)
                {
                    const nn::g3d::ResShadingModel* pResShadingModel = pResShaderArchive->GetShadingModel(shadingModelIndex);
                    if (strcmp(shadingModelBuilder->m_Name.c_str(), pResShadingModel->GetName()))
                    {
                        continue;
                    }
                    const nn::gfx::ResShaderFile* pResGfxShaderFile = pResShadingModel->GetGfxResShaderFile();
                    const nn::gfx::ResShaderContainer* pResShaderContainer = pResGfxShaderFile->GetShaderContainer();
                    int gfxShaderVariationCount = pResShaderContainer->GetShaderVariationCount();
                    int g3dShaderProgramCount = pResShadingModel->GetShaderProgramCount();
                    if (g3dShaderProgramCount == 0)
                    {
                        continue;
                    }

                    // ShaderCompilerManager に渡すソースリストを作成
                    shadingModelBuilder->m_ShaderCompilerManager.AddMergeResShaderFile(pResGfxShaderFile);

                    for (int programIndex = 0; programIndex < g3dShaderProgramCount; ++programIndex)
                    {
                        const nn::g3d::ResShaderProgram* pResShaderProgram = pResShadingModel->GetShaderProgram(programIndex);
                        for (int variationIndex = 0; variationIndex < gfxShaderVariationCount; ++variationIndex)
                        {
                            const nn::gfx::ResShaderVariationData* pResShaderVariationData =
                                pResShaderContainer->ToData().pShaderVariationArray.ToPtr(const_cast<nn::gfx::ResShaderFileData*>(&pResGfxShaderFile->ToData())) + variationIndex;
                            if (pResShaderVariationData != &pResShaderProgram->ToData().pShader.Get()->ToData())
                            {
                                continue;
                            }

                            // 既に同じシェーダーキーが含まれている場合は ShaderProgram を追加しない
                            const nn::Bit32* pKey = pResShadingModel->GetKey(programIndex);
                            size_t keySize = shadingModelBuilder->m_KeyLength * sizeof(nn::Bit32);
                            auto found = std::find_if(programs.cbegin(), programs.cend(), [&](const ShaderProgram &shaderProgram)
                            {
                                return !memcmp(pKey, shaderProgram.m_Key.get(), keySize);
                            });
                            if (found != programs.cend())
                            {
                                break;
                            }

                            // ShaderProgram を追加する
                            std::shared_ptr<void> newKey(malloc(keySize));
                            memcpy(newKey.get(), pKey, keySize);
                            ShaderProgram newShaderProgram(newKey, shadingModelBuilder->m_KeyLength, NULL);

                            // gfx ShaderVariation と g3d ShaderProgram の index を紐づける。g3d ShaderProgram 側に ShaderVariation の index を持たせる。
                            // gfx の ShaderVariation は入力した順番に並ぶので、その前提で処理する。
                            newShaderProgram.m_ResShaderVariationIdx = gfxShaderVariationTotalCount + variationIndex;

                            // index テーブルの作成。ResShaderProgram の sampler,ubo,ssbo の要素は ShadingModelBuilder より少ない場合がある。
                            newShaderProgram.m_SamplerTable.resize(shadingModelBuilder->m_SamplerArray.size());
                            for (int samplerIndex = 0; samplerIndex < shadingModelBuilder->m_SamplerArray.size(); ++samplerIndex)
                            {
                                ShaderProgram::ShaderSet set;
                                const nn::util::BinPtrToString* pSamplerTable = pResShadingModel->GetShaderInfo()->pSamplerTable.Get();
                                for (int originalSamplerIndex = 0; originalSamplerIndex < pResShadingModel->GetSamplerCount(); ++originalSamplerIndex)
                                {
                                    for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
                                    {
                                        const nn::util::BinString* pName = pSamplerTable->Get();
                                        if (pName && strcmp(shadingModelBuilder->m_SamplerArray[samplerIndex].m_Symbol[stage].c_str(), pName->GetData()) == 0)
                                        {
                                            set.index[stage] = static_cast<int8_t>(pResShaderProgram->GetSamplerLocation(samplerIndex, static_cast<nn::g3d::Stage>(stage)));
                                        }
                                        pSamplerTable++;
                                    }
                                }
                                newShaderProgram.m_SamplerTable[samplerIndex] = set;
                            }
                            newShaderProgram.m_UniformBlockTable.resize(shadingModelBuilder->m_BlockArray.size());
                            for (int uniformBlockIndex = 0; uniformBlockIndex < shadingModelBuilder->m_BlockArray.size(); ++uniformBlockIndex)
                            {
                                ShaderProgram::ShaderSet set;
                                const nn::util::BinPtrToString* pUniformBlockTable = pResShadingModel->GetShaderInfo()->pUniformBlockTable.Get();
                                for (int originalUniformBlockIndex = 0; originalUniformBlockIndex < pResShadingModel->GetUniformBlockCount(); ++originalUniformBlockIndex)
                                {
                                    for (int stage = 0; stage < nn::g3d::Stage_End; ++stage)
                                    {
                                        const nn::util::BinString* pName = pUniformBlockTable->Get();
                                        if (pName && strcmp(shadingModelBuilder->m_BlockArray[uniformBlockIndex].m_Symbol[stage].c_str(), pName->GetData()) == 0)
                                        {
                                            set.index[stage] = static_cast<int8_t>(pResShaderProgram->GetUniformBlockLocation(uniformBlockIndex, static_cast<nn::g3d::Stage>(stage)));
                                        }
                                        pUniformBlockTable++;
                                    }
                                }
                                newShaderProgram.m_UniformBlockTable[uniformBlockIndex] = set;
                            }
                            newShaderProgram.m_ShaderStorageBlockTable.resize(shadingModelBuilder->m_SsboBlockArray.size());
                            for (int shaderStorageBlockIndex = 0; shaderStorageBlockIndex < shadingModelBuilder->m_SsboBlockArray.size(); ++shaderStorageBlockIndex)
                            {
                                ShaderProgram::ShaderSet set;
                                const nn::util::BinPtrToString* pShaderStorageBlockTable = pResShadingModel->GetShaderInfo()->pShaderStorageBlockTable.Get();
                                for (int originalShaderStorageBlockIndex = 0; originalShaderStorageBlockIndex < pResShadingModel->GetShaderStorageBlockCount(); ++originalShaderStorageBlockIndex)
                                {
                                    for (int stage = 0; stage < nn::g3d::Stage_End; ++stage)
                                    {
                                        const nn::util::BinString* pName = pShaderStorageBlockTable->Get();
                                        if (pName && strcmp(shadingModelBuilder->m_SsboBlockArray[shaderStorageBlockIndex].m_Symbol[stage].c_str(), pName->GetData()) == 0)
                                        {
                                            set.index[stage] = static_cast<int8_t>(pResShaderProgram->GetShaderStorageBlockLocation(shaderStorageBlockIndex, static_cast<nn::g3d::Stage>(stage)));
                                        }
                                        pShaderStorageBlockTable++;
                                    }
                                }
                                newShaderProgram.m_ShaderStorageBlockTable[shaderStorageBlockIndex] = set;
                            }

                            // attribute は全ステージで共通なので ResShadingModel から引っ張ってくる
                            newShaderProgram.m_AttribTable.resize(shadingModelBuilder->m_AttribArray.size());
                            for (int attributeIndex = 0; attributeIndex < shadingModelBuilder->m_AttribArray.size(); ++attributeIndex)
                            {
                                ShaderProgram::ShaderSet set;
                                int offset = 0;
                                for (int originalAttributeIndex = 0; originalAttributeIndex < pResShadingModel->GetAttrCount(); ++originalAttributeIndex, ++offset)
                                {
                                    if (strcmp(shadingModelBuilder->m_AttribArray[attributeIndex].m_Symbol[ShaderStage_Vertex].c_str(), pResShadingModel->GetAttrName(originalAttributeIndex)) == 0)
                                    {
                                        if (pResShaderProgram->ToData().attribActiveFlag & (1 << offset))
                                        {
                                            set.index[ShaderStage_Vertex] = static_cast<int8_t>(pResShadingModel->GetAttr(originalAttributeIndex)->GetLocation());
                                        }
                                    }
                                }
                                newShaderProgram.m_AttribTable[attributeIndex] = set;
                            }

                            programs.push_back(newShaderProgram);
                            break;
                        }
                    }
                    gfxShaderVariationTotalCount += gfxShaderVariationCount;
                    g3dShaderProgramTotalCount += g3dShaderProgramCount;
                }
            }

            // gfx 以下のデータのコンバート(マージ)
            PRINT_LOG("Merging program count: %d -> %d", g3dShaderProgramTotalCount, programs.size());
            {
                shadingModelBuilder->m_ShaderCompilerManager.InitializeGfxToolConverterOptions(GetGfxToolConverterOptions());
                if (shadingModelBuilder->m_ShaderCompilerManager.InitializeConvertArg() != nngfxToolResultCode_Succeeded)
                {
                    THROW_ERROR(ERRCODE_INTERNAL, "Failed to initialize GfxConverter convert parameters.");
                }

                // ShaderConverter.dll によるコンバート
                bool isSucceeded = shadingModelBuilder->m_ShaderCompilerManager.Convert(&shaderContext.m_ResShaderFileSize, &shaderContext.m_ResShaderFileAlignment);
                if (!isSucceeded)
                {
                    THROW_ERROR(ERRCODE_INTERNAL, "Failed to calculate GfxConverter shader size.");
                }
                shadingModelBuilder->m_ShaderCompilerManager.FinalizeConvertArg();
            }

            // ShaderProgram をシェーダーキーでソート
            auto& pPrograms = shaderContext.m_pPrograms;
            pPrograms.resize(programs.size());
            {
                std::vector<ShaderProgram*> pProgramsTmp;
                pProgramsTmp.resize(programs.size());
                int programIndex = 0;
                for (auto program = programs.begin(); program != programs.end(); ++program, ++programIndex)
                {
                    pProgramsTmp[programIndex] = &(*program);
                }
                std::sort(pProgramsTmp.begin(), pProgramsTmp.end(), KeyCmp(shadingModelBuilder->m_KeyLength));
                for (int idxProgram = 0, numProgram = static_cast<int>(pProgramsTmp.size());
                    idxProgram < numProgram; ++idxProgram)
                {
                    pPrograms[idxProgram] = pProgramsTmp[idxProgram];
                }
            }
        }
        return;
    }

    if (m_GlslVersion.empty() && m_ConvertMode != ConvertMode_MergeBfsha)
    {
        PRINT_WARNING(ERRCODE_SHADER_CONVERTER_GLSL_SYNTAX_ERROR,
            "GLSL version is not specified. See --glsl-version option.");
    }

    int idxShadingModel = 0;
    auto& shadingModelBuilderArray =  m_ArchiveBuilder.m_ShadingModelBuilderArray;
    for (auto shadingModelBuilder = shadingModelBuilderArray.begin();
        shadingModelBuilder != shadingModelBuilderArray.end();
        ++shadingModelBuilder, ++idxShadingModel)
    {
        PRINT_LOG("Shading model: %hs", shadingModelBuilder->m_Name.c_str());

        auto el_shading_model = m_FsdArray[0]->shading_model_array[idxShadingModel];

        shdr_ctx.m_ShaderContexts.resize(shdr_ctx.m_ShaderContexts.size() + 1);
        Context::ShaderContext& shaderContext = shdr_ctx.m_ShaderContexts.back();
        auto& programs = shaderContext.m_Programs;
        auto& pPrograms = shaderContext.m_pPrograms;

        auto& shaderBuilderArray = shadingModelBuilder->m_ShaderBuilderArray;
        int programCount = 0;
        for (auto& shaderBuilder : shaderBuilderArray)
        {
            programCount += nw::g3d::tool::NumericCast<int>(shaderBuilder.m_ShaderVariationBuilderArray.size());
        }

        programs.reserve(programCount);
        for(auto& shaderBuilder : shaderBuilderArray)
        {
            for(auto& variation : shaderBuilder.m_ShaderVariationBuilderArray)
            {
                programs.emplace_back( ShaderProgram( variation.m_Key, shadingModelBuilder->m_KeyLength, &variation.m_ChoiceArray ) );
            }
        }

        PRINT_LOG("Compiling program count: %d", programCount);

        // variation 用の uniform の値を格納するバッファを確保します。
        for( int idxAltUniform = 0; idxAltUniform < shadingModelBuilder->m_OptionUniformInfoArray.size(); ++idxAltUniform )
        {
            shadingModelBuilder->m_OptionUniformInfoArray[idxAltUniform].m_Uint32Array.resize( programCount );
            for( int idxVariation = 0; idxVariation < programCount; ++idxVariation )
            {
                shadingModelBuilder->m_OptionUniformInfoArray[idxAltUniform].m_Uint32Array[idxVariation].resize( shadingModelBuilder->m_OptionUniformInfoArray[idxAltUniform].matrix.column );
            }
        }

        if (!m_DumpFolder.empty())
        {
            DumpKey(m_DumpFolder.c_str(), el_shading_model, programs);
        }

        if (programCount > 0)
        {
            shaderContext.m_SharedProgram.reset(new ShaderProgram( shaderBuilderArray[0].m_ShaderVariationBuilderArray[0].m_Key,
                                                                   shadingModelBuilder->m_KeyLength, &shaderBuilderArray[0].m_ShaderVariationBuilderArray[0].m_ChoiceArray ) );
        }
        else
        {
            continue;
        }

        ShaderProgram& sharedProgram = *shaderContext.m_SharedProgram.get();

        // Variation 生成のための元になる glsl を生成します。
        {
            // force_include の機能が gfxTool にはないのでここでコード成型します。
            InsertForceInclude( sharedProgram, *shadingModelBuilder, m_ArchiveBuilder.m_ForceIncludePathArray);

            // gfxTool で include ファイル展開に必要な CB に渡すコールバックパラメータを構築します。
            for( auto& shaderSource : shadingModelBuilder->m_ShaderSourceArray )
            {
                ShaderCompilerManager::ShaderFileInfo shaderFileInfo;
                shaderFileInfo.strLength				= static_cast<uint32_t>(strlen(static_cast<const char*>(shaderSource.m_Stream.get())));	//!< string しか入力されないはずなので文字数を数える。
                shaderFileInfo.pFileData				= shaderSource.m_Stream.get();
                shaderFileInfo.pRelativeFilePathName	= &shaderSource.m_Path;
                shadingModelBuilder->m_ShaderCompilerManager.AddCallbackParam( shaderFileInfo );
            }
        }

        // Compile Gfx: Gfx 版シェーダバイナリの作成
        {
            std::string codePageOption = "--code-page=" +
                std::to_string( m_FsdArray.begin()->get()->shader_definition_info.code_page.value );
            AddGfxToolConverterOptions( codePageOption );
            shadingModelBuilder->m_ShaderCompilerManager.InitializeGfxToolConverterOptions( GetGfxToolConverterOptions() );

            // stage 共通の fixed macro
            std::unordered_map<std::string, std::string> fixedMacros;
            std::for_each(m_Macros.cbegin(), m_Macros.cend(), [&](const std::string& macro)
            {
                BuildFixedMacroMap(fixedMacros, macro);
            });

            for( auto fsdMacroIter = shadingModelBuilder->m_MacroArray.begin();
                fsdMacroIter != shadingModelBuilder->m_MacroArray.end();
                ++fsdMacroIter )
            {
                auto fixedMacroIter = fixedMacros.find( fsdMacroIter->m_Name );
                if( fixedMacroIter != fixedMacros.end() )							// fixedMacro に fsd 内 macro の定義が見つかった場合は fsd macro の値を上書きする。
                {
                    fsdMacroIter->m_Value = fixedMacroIter->second;
                    fixedMacros.erase( fixedMacroIter );
                }
            }

            // fsdMacroArray に fixedMacro でのみ定義されたものを追加する。
            for( auto fixedMacroIter = fixedMacros.begin(); fixedMacroIter != fixedMacros.end(); ++fixedMacroIter )
            {
                ShaderMacro shaderMacro;
                shaderMacro.m_Name = fixedMacroIter->first;
                shaderMacro.m_Value = fixedMacroIter->second;
                shadingModelBuilder->m_MacroArray.push_back( shaderMacro );
            }

            // variation definition の構築
            VariationDefinition	variationDefinition;
            {
                for( int stageIdx= 0; stageIdx < ShaderStage_StageCount; ++stageIdx )
                {
                    if( !sharedProgram.m_ShaderSource[ stageIdx ].empty() )
                    {
                        // マクロを preprocessor variation definition に追加します。
                        std::vector<const std::string*> symbolValueNameArary;

                        // g3d のシェーダーステージマクロ
                        symbolValueNameArary.push_back( &g_SystemStageMacroName[stageIdx] );

                        // fsd macro、--define-macro で追加されるマクロ
                        for( int preprocessorIdx = 0; preprocessorIdx < shadingModelBuilder->m_MacroArray.size(); ++preprocessorIdx  )
                        {
                            const std::string& symbolValueName = shadingModelBuilder->m_MacroArray[ preprocessorIdx ].m_Name;
                            if( !symbolValueName.empty() )
                            {
                                symbolValueNameArary.push_back( &symbolValueName );
                            }
                        }

                        // オプション変数によるマクロ。バッチコンパイル使用されるものは値を uniform 名にする必要があるため追加する。
                        for( int optionIdx = 0; optionIdx < shadingModelBuilder->m_OptionArray.size(); ++optionIdx )
                        {
                            const std::string& symbolValueName = shadingModelBuilder->m_OptionArray[ optionIdx ].m_Symbol[ stageIdx ];
                            if( !symbolValueName.empty() )
                            {
                                symbolValueNameArary.push_back( &symbolValueName );
                            }
                        }

                        // uniform constant variation definition を構築します。
                        // IsBatchCompileApplicable() なオプション変数が追加されます。
                        std::vector<VariationDefinition::UniformVariationDefinitionInfo> uniformVariationDefinitionInfoArray;
                        if( !this->IsUsePreprocessorVariation() )
                        {
                            uniformVariationDefinitionInfoArray.reserve( shadingModelBuilder->m_OptionArray.size() );
                            for( int optionIdx = 0; optionIdx < shadingModelBuilder->m_OptionArray.size(); ++optionIdx )
                            {
                                auto& elemOptionVar = el_shading_model.option_var_array[optionIdx];
                                auto& optionUniformInfo = shadingModelBuilder->m_OptionUniformInfoArray[optionIdx];
                                const std::string* pUniformName = optionUniformInfo.m_pOptionUniformName[stageIdx];
                                bool hasUniform = ( pUniformName != nullptr );

                                if( IsBatchCompileApplicable( m_IsForceVariation, hasUniform, elemOptionVar ) )
                                {
                                    VariationDefinition::UniformVariationDefinitionInfo uniformVariationDefinitionInfo;
                                    uniformVariationDefinitionInfo.pUniformBufferName = optionUniformInfo.m_pOptionUniformBlockName[stageIdx];
                                    uniformVariationDefinitionInfo.pName = pUniformName;
                                    uniformVariationDefinitionInfo.type = optionUniformInfo.m_Type;
                                    uniformVariationDefinitionInfo.arrayLength = static_cast<uint32_t>( optionUniformInfo.matrix.column );
                                    uniformVariationDefinitionInfo.matrix.row = optionUniformInfo.matrix.row;
                                    uniformVariationDefinitionInfo.matrix.column = optionUniformInfo.matrix.column;
                                    uniformVariationDefinitionInfoArray.emplace_back( uniformVariationDefinitionInfo );
                                }
                            }
                        }

                        variationDefinition.CreateVariationDefinitionArg( symbolValueNameArary, uniformVariationDefinitionInfoArray, static_cast<ShaderCompilerManager::ShaderType>( stageIdx ) );
                    }
                }
                shadingModelBuilder->m_ShaderCompilerManager.SetVariationDefinition( variationDefinition.GetVariationDefinitionArg() );
            }

            // variation value の構築
            std::vector<VariationValue>		variationValueArray;
            variationValueArray.resize( programs.size() );
            for (int variationIdx = 0; variationIdx < programs.size(); ++variationIdx )
            {
                const std::vector<SymbolName>*	pOptionMacroValueArray = programs[ variationIdx ].m_ChoiceArray;
                const std::vector<ShaderMacro>* pMacroValueArray = &shadingModelBuilder->m_MacroArray;

                for( int stageIdx= 0; stageIdx < ShaderStage_StageCount; ++stageIdx )
                {
                    if( !sharedProgram.m_ShaderSource[ stageIdx ].empty() )
                    {
                        std::vector<const std::string*> symbolValueNameArary;
                        symbolValueNameArary.reserve( 1 + pMacroValueArray->size() + pOptionMacroValueArray->size() );

                        // g3d のシェーダーステージマクロ
                        symbolValueNameArary.push_back(&g_SystemStageMacroValue);

                        // fsd macro、--define-macro で追加されるマクロ
                        for( int preprocessorIdx = 0; preprocessorIdx < pMacroValueArray->size(); ++preprocessorIdx  )
                        {
                            if( !(*pMacroValueArray)[ preprocessorIdx ].m_Name.empty() )
                            {
                                symbolValueNameArary.push_back( &(*pMacroValueArray)[ preprocessorIdx ].m_Value );
                            }
                        }

                        // オプション変数によるマクロ
                        for( int optionIdx = 0; optionIdx < pOptionMacroValueArray->size(); ++optionIdx )
                        {
                            if( !shadingModelBuilder->m_OptionArray[ optionIdx ].m_Symbol[ stageIdx ].empty() )	// マクロ名が存在しない場合はその値を追加しません。
                            {
                                if( this->IsUsePreprocessorVariation() )
                                {
                                    symbolValueNameArary.push_back( &(*pOptionMacroValueArray)[ optionIdx ].m_Symbol[ stageIdx ] );
                                }
                                else
                                {
                                    const std::string* pUniformName = shadingModelBuilder->m_OptionUniformInfoArray[optionIdx].m_pOptionUniformName[stageIdx];
                                    bool hasUniform = ( pUniformName != nullptr );
                                    auto& elemOptionVar = el_shading_model.option_var_array[optionIdx];
                                    if( IsBatchCompileApplicable( m_IsForceVariation, hasUniform, elemOptionVar ) )
                                    {
                                        symbolValueNameArary.push_back( pUniformName );
                                    }
                                    else
                                    {
                                        symbolValueNameArary.push_back( &(*pOptionMacroValueArray)[ optionIdx ].m_Symbol[ stageIdx ] );
                                    }
                                }
                            }
                        }

                        // uniform constant value の variation リストを追加
                        std::vector<VariationValue::VariationConstantValueInfo> variationConstantValueInfoArray;
                        if( !this->IsUsePreprocessorVariation() )
                        {
                            variationConstantValueInfoArray.reserve(pOptionMacroValueArray->size() );

                            for( int optionIdx = 0; optionIdx < pOptionMacroValueArray->size(); ++optionIdx )
                            {
                                auto& altUniform = shadingModelBuilder->m_OptionUniformInfoArray[optionIdx];
                                const std::string* pUniformName = altUniform.m_pOptionUniformName[stageIdx];
                                bool hasUniform = ( pUniformName != nullptr );
                                auto& elemOptionVar = el_shading_model.option_var_array[optionIdx];
                                if( IsBatchCompileApplicable( m_IsForceVariation, hasUniform, elemOptionVar ) )
                                {
                                    auto uniformValueStrStream = (*pOptionMacroValueArray)[optionIdx].m_Symbol[stageIdx].c_str();
                                    uint32_t* pBuff = &shadingModelBuilder->m_OptionUniformInfoArray[optionIdx].m_Uint32Array[variationIdx][0];
                                    nw::g3d::tool::g3dif::ReadArray<uint32_t>( uniformValueStrStream,
                                                                            static_cast<int>( shadingModelBuilder->m_OptionUniformInfoArray[optionIdx].m_Uint32Array[variationIdx].size() ),
                                                                            pBuff );
                                    VariationValue::VariationConstantValueInfo variationConstantValueInfo;
                                    variationConstantValueInfo.pData = pBuff;
                                    variationConstantValueInfo.sizeIn32Bit = static_cast<uint32_t>( shadingModelBuilder->m_OptionUniformInfoArray[optionIdx].m_Uint32Array[variationIdx].size() );
                                    variationConstantValueInfoArray.push_back( variationConstantValueInfo );
                                }
                            }
                        }

                        variationValueArray[ variationIdx ].CreateVariationValueArg( symbolValueNameArary, variationConstantValueInfoArray, static_cast<ShaderCompilerManager::ShaderType>( stageIdx ) );
                    }
                }
                shadingModelBuilder->m_ShaderCompilerManager.AddVariationValueArg( variationValueArray[ variationIdx ].GetVariationValueArg() );
            }

            shadingModelBuilder->m_ShaderCompilerManager.SetShaderCacheOptionInfo(m_ShaderCacheOptionInfo);
            if( shadingModelBuilder->m_ShaderCompilerManager.InitializeConvertArg() != nngfxToolResultCode_Succeeded )
            {
                THROW_ERROR(ERRCODE_INTERNAL, "Failed to initialize GfxConverter convert parameters.");
            }

            // シェーダソースのセット
            for( int stageIdx= 0; stageIdx < ShaderStage_StageCount; ++stageIdx )
            {
                if( !sharedProgram.m_ShaderSource[ stageIdx ].empty() )
                {
                    shadingModelBuilder->m_ShaderCompilerManager.SetShaderSource(
                        sharedProgram.m_ShaderSource[ stageIdx ], static_cast<ShaderCompilerManager::ShaderType>( stageIdx ) );
                }
            }

            size_t shaderSize = 0;
            size_t shaderAlignment = 0;
            bool isSucceeded = shadingModelBuilder->m_ShaderCompilerManager.Convert(&shaderSize, &shaderAlignment);
            if(!isSucceeded)
            {
                THROW_ERROR(ERRCODE_INTERNAL, "Failed to calculate GfxConverter shader size.");
            }
            if( m_SkipConvert )
            {
                continue;
            }

            shaderContext.m_ResShaderFileSize = shaderSize;
            shaderContext.m_ResShaderFileAlignment = shaderAlignment;

            // ShaderPrgoram の ResShaderfile の Container の index を更新する。
            for (int variationIdx = 0; variationIdx < programs.size(); ++variationIdx )
            {
                programs[ variationIdx ].m_ResShaderVariationIdx = variationIdx;
            }

            // リフレクションデータを見ながら attribute, uniform offset 等を更新する。
            for (int variationIdx = 0; variationIdx < programs.size(); ++variationIdx )
            {
                ShaderProgram&	program = programs[ variationIdx ];
                ShadingModelBuilder& builder = *shadingModelBuilder;
                ShaderCompilerManager& shaderCompilerManager = builder.m_ShaderCompilerManager;

                // attrib
                int attribIndex = 0;
                program.m_AttribTable.resize(builder.m_AttribArray.size());
                for (auto attrib = builder.m_AttribArray.cbegin(); attrib != builder.m_AttribArray.cend(); ++attrib, ++attribIndex)
                {
                    ShaderProgram::ShaderSet set;
                    const ShaderCompilerManager::AttribInfo* pAttribInfo =
                        shaderCompilerManager.GetAttribInfo(
                        attrib->m_Symbol[ShaderStage_Vertex], variationIdx, m_ShaderReflectionCodeType );

                    if ( pAttribInfo != nullptr )
                    {
                        set.index[ShaderStage_Vertex] = static_cast<int8_t>( pAttribInfo->slot );
                        //set.arrayCount = 1;
                        set.arrayCount = pAttribInfo->arrayCount;
                    }

                    program.m_AttribTable[attribIndex] = set;
                }

                // block
                static const nngfxToolShaderCompilerReflectionStageReference s_StageReferences[] =
                {
                    nngfxToolShaderCompilerReflectionStageReference_Vertex,
                    nngfxToolShaderCompilerReflectionStageReference_Geometry,
                    nngfxToolShaderCompilerReflectionStageReference_Pixel,
                    nngfxToolShaderCompilerReflectionStageReference_Compute
                };
                auto buildBlock = [&]( auto& uniformBlockMap, auto& uniformBlockTable, const auto& blockArray, const auto& uniformBlockInfoSet, const auto& getBlockSizeFunc )
                {
                    int uniformBlockIndex = 0;
                    for (auto block = blockArray.cbegin(); block != blockArray.cend(); ++block, ++uniformBlockIndex)
                    {
                        ShaderProgram::ShaderSet set;
                        for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
                        {
                            if (!block->m_Symbol[stage].empty())
                            {
                                int size = getBlockSizeFunc(block->m_Symbol[stage], variationIdx, m_ShaderReflectionCodeType );

                                if (size < 0)
                                {
                                    continue;
                                }
                                auto comp = uniformBlockMap.find(block->m_Symbol[stage]);
                                if (comp == uniformBlockMap.end())
                                {
                                    uniformBlockMap.insert(std::make_pair(block->m_Symbol[stage], static_cast<size_t>(size)));
                                }
                                else if (comp->second != static_cast<size_t>(size))		// uniform block の size はステージ間で一致している必要がある。
                                {
                                    THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Uniform block size is inconsistent between stages.");
                                }

                                auto iter = uniformBlockInfoSet.find(block->m_Symbol[stage]);
                                const ShaderCompilerManager::UniformBlockInfo& uniformBlockInfo = iter->second;

                                if( uniformBlockInfo.stageBit & s_StageReferences[ stage ] )
                                {
                                    if(s_StageReferences[ stage ] == nngfxToolShaderCompilerReflectionStageReference_Vertex)
                                    {
                                        set.index[stage] = static_cast<s8>(uniformBlockInfo.shaderSlot.vertexShaderSlot);
                                    }
                                    else if(s_StageReferences[ stage ] == nngfxToolShaderCompilerReflectionStageReference_Geometry)
                                    {
                                        set.index[stage] = static_cast<s8>(uniformBlockInfo.shaderSlot.geometryShaderSlot);
                                    }
                                    else if(s_StageReferences[ stage ] == nngfxToolShaderCompilerReflectionStageReference_Pixel)
                                    {
                                        set.index[stage] = static_cast<s8>(uniformBlockInfo.shaderSlot.pixelShaderSlot);
                                    }
                                    else if(s_StageReferences[ stage ] == nngfxToolShaderCompilerReflectionStageReference_Compute)
                                    {
                                        set.index[stage] = static_cast<s8>(uniformBlockInfo.shaderSlot.computeShaderSlot);
                                    }
                                }
                            }
                        }
                        uniformBlockTable[uniformBlockIndex] = set;
                    }
                };
                auto getUniformBlocksize = [&]( std::string name, int variation, int shaderCodeType ) ->int
                {
                    return shaderCompilerManager.GetUniformBlockSize( name, variation ,shaderCodeType );
                };
                auto getSsboBlocksize = [&]( std::string name, int variation, int shaderCodeType ) ->int
                {
                    return shaderCompilerManager.GetSsboBlockSize( name, variation ,shaderCodeType );
                };
                program.m_UniformBlockTable.resize( builder.m_BlockArray.size() );
                auto& uniformBlockInfoSet = shaderCompilerManager.GetUniformBlockInfoSet( variationIdx, m_ShaderReflectionCodeType );
                buildBlock( program.m_UniformBlockMap, program.m_UniformBlockTable, builder.m_BlockArray, uniformBlockInfoSet, getUniformBlocksize );
                program.m_ShaderStorageBlockTable.resize( builder.m_SsboBlockArray.size() );
                auto& ssboBlockInfoSet = shaderCompilerManager.GetSsboBlockInfoSet( variationIdx, m_ShaderReflectionCodeType );
                buildBlock( program.m_SsboBlockMap, program.m_ShaderStorageBlockTable, builder.m_SsboBlockArray, ssboBlockInfoSet, getSsboBlocksize );

                // uniform
                auto buildUniform = [&]( auto& uniformMap, const auto& inforSet )
                {
                    for( auto iter = inforSet.begin(); iter != inforSet.end(); ++iter )
                    {
                        std::string name( iter->first );

                        const auto& uniform = iter->second;
                        {
                            auto comp = uniformMap.find(name);
                            if (comp == uniformMap.end())
                            {
                                uniformMap.insert(std::make_pair(name, uniform));
                            }
                            else if (comp->second.offset != uniform.offset)
                            {
                                THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
                            }
                        }
                    }
                };
                auto& uniformInfoSet = shaderCompilerManager.GetUniformInfoSet( variationIdx, m_ShaderReflectionCodeType );
                buildUniform( program.m_UniformMap, uniformInfoSet );
                auto& ssboInfoSet = shaderCompilerManager.GetSsboInfoSet( variationIdx, m_ShaderReflectionCodeType );
                buildUniform( program.m_SsboMap, ssboInfoSet );

                // sampler
                const auto& samplerInfoSet = shaderCompilerManager.GetSamplerInfoSet( variationIdx, m_ShaderReflectionCodeType );
                int samplerIndex = 0;
                program.m_SamplerTable.resize(builder.m_SamplerArray.size());
                //program.m_samplertable.resize(samplerinfoset.size());	//!< TODO: テーブル作成方法はこちらが正しいはず。

                for (auto sampler = builder.m_SamplerArray.cbegin(); sampler != builder.m_SamplerArray.cend(); ++sampler, ++samplerIndex)
                {
                    ShaderProgram::ShaderSet set;
                    for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
                    {
                        if( !sampler->m_Symbol[ stage ].empty() &&
                            samplerInfoSet.find( sampler->m_Symbol[ stage ] ) != samplerInfoSet.end() )
                        {
                            auto iter = samplerInfoSet.find( sampler->m_Symbol[ stage ] );
                            const ShaderCompilerManager::SamplerInfo& samplerInfo = iter->second;
                            if( samplerInfo.stageBit & s_StageReferences[ stage ] )
                            {
                                if(s_StageReferences[ stage ] == nngfxToolShaderCompilerReflectionStageReference_Vertex)
                                {
                                    set.index[stage] = static_cast<s8>(samplerInfo.shaderSlot.vertexShaderSlot);
                                }
                                else if(s_StageReferences[ stage ] == nngfxToolShaderCompilerReflectionStageReference_Geometry)
                                {
                                    set.index[stage] = static_cast<s8>(samplerInfo.shaderSlot.geometryShaderSlot);
                                }
                                else if(s_StageReferences[ stage ] == nngfxToolShaderCompilerReflectionStageReference_Pixel)
                                {
                                    set.index[stage] = static_cast<s8>(samplerInfo.shaderSlot.pixelShaderSlot);
                                }
                                else if(s_StageReferences[ stage ] == nngfxToolShaderCompilerReflectionStageReference_Compute)
                                {
                                    set.index[stage] = static_cast<s8>(samplerInfo.shaderSlot.computeShaderSlot);
                                }
                            }
                        }
                    }
                    program.m_SamplerTable[samplerIndex] = set;
                }

                // ユニフォームマップのマージ
                MergeUniformBlockMap(&shaderContext.m_UniformBlockMap, &programs[variationIdx].m_UniformBlockMap);
                MergeUniformMap(&shaderContext.m_UniformMap, &programs[variationIdx].m_UniformMap);
                UniformBlockMap().swap(programs[variationIdx].m_UniformBlockMap);
                UniformMap().swap(programs[variationIdx].m_UniformMap);

                // Ssboのマージ
                // とりあえず従来のuboのマージアルゴリズムをそのまま使用する。任意長配列はそのgpuで使用可能な最大長になる。
                MergeUniformBlockMap(&shaderContext.m_SsboBlockMap, &programs[variationIdx].m_SsboBlockMap);
                MergeUniformMap(&shaderContext.m_SsboMap, &programs[variationIdx].m_SsboMap);
                SsboBlockMap().swap(programs[variationIdx].m_SsboBlockMap);
                SsboMap().swap(programs[variationIdx].m_SsboMap);

                // シェーダコードのダンプ
                if( !m_DumpFolder.empty() )
                {
                    // ソースは Source/Intermidiate/Binary どこから取得しても大丈夫なはず。
                    // Source/IL/BIN どれを含むかの判定する仕組みが欲しい。
                    ShaderCompilerManager::ShaderCodeType shaderCodeType = ShaderCompilerManager::ShaderCodeType_Source;
                    // TODO: シェーダコードタイプを判定する仕組み
                    if( m_IsShaderCodeTypeBinary )
                    {
                        shaderCodeType = ShaderCompilerManager::ShaderCodeType_Binary;
                    }
                    else if( m_IsShaderCodeTypeIr )
                    {
                        shaderCodeType = ShaderCompilerManager::ShaderCodeType_Intermediate;
                    }

                    ShaderCompilerManager::PipelineInfo pipelineInfo =
                        shaderCompilerManager.GetPipelineInfo( variationIdx, shaderCodeType );
                    if (m_IsDumpAll)
                    {
                        DumpShaderSource(m_DumpFolder.c_str(),
                            m_Name.c_str(),
                            &shaderCompilerManager,
                            &program,
                            &pipelineInfo,
                            &el_shading_model,
                            &sharedProgram,
                            variationIdx);
                    }
                    DumpXmlStatistics(m_DumpFolder.c_str(),
                        &shaderCompilerManager,
                        &program,
                        &pipelineInfo,
                        &el_shading_model,
                        variationIdx);
                }
            }
            shadingModelBuilder->m_ShaderCompilerManager.FinalizeConvertArg();
        }

        for (int variationIdx = 0; variationIdx < programs.size(); ++variationIdx )
        {
            ShaderProgram&	program = programs[ variationIdx ];
            ShadingModelBuilder& builder = *shadingModelBuilder;
            program.m_SamplerTable.resize(builder.m_SamplerArray.size());
            program.m_UniformBlockTable.resize( builder.m_BlockArray.size() );
        }

        pPrograms.resize( programs.size() );

        // マージ用に非 const でソート
        std::vector<ShaderProgram*> pProgramsTmp;
        pProgramsTmp.resize(programs.size());
        int programIndex = 0;
        // program のポインタを管理
        for (auto program = programs.begin(); program != programs.end(); ++program, ++programIndex)
        {
            pProgramsTmp[programIndex] = &(*program);
        }

        // シェーダプログラムのソート
        std::sort(pProgramsTmp.begin(), pProgramsTmp.end(), KeyCmp(shadingModelBuilder->m_KeyLength));

        for (int idxProgram = 0, numProgram = static_cast<int>(pProgramsTmp.size());
            idxProgram < numProgram; ++idxProgram)
        {
            pPrograms[idxProgram] = pProgramsTmp[idxProgram];
        }
    }
}

void ShdrFile::Compile(std::shared_ptr<Context> pCtx)	// 実質ここは Compile() じゃない、gfx converter で Serialize() しているだけ。
{
    Context& shdr_ctx	= *pCtx;

    int idxShadingModel = 0;
    auto shaderContextIter = shdr_ctx.m_ShaderContexts.begin();
    auto& shadingModelBuilderArray =  m_ArchiveBuilder.m_ShadingModelBuilderArray;
    for (auto shadingModelBuilder = shadingModelBuilderArray.begin();
        shadingModelBuilder != shadingModelBuilderArray.end();
        ++shadingModelBuilder, ++idxShadingModel, ++shaderContextIter)
    {
        Context::ShaderContext& shaderContext = *shaderContextIter;

        if (m_ConvertMode == ConvertMode_FsdToBfsha)
        {
            auto& programs = shaderContext.m_Programs;

            auto& shaderBuilderArray = shadingModelBuilder->m_ShaderBuilderArray;
            int programCount = 0;
            for (auto& shaderBuilder : shaderBuilderArray)
            {
                programCount += nw::g3d::tool::NumericCast<int>(shaderBuilder.m_ShaderVariationBuilderArray.size());
            }

            if (!m_DumpFolder.empty())
            {
                auto el_shading_model = m_FsdArray[0]->shading_model_array[idxShadingModel];
                DumpKey(m_DumpFolder.c_str(), el_shading_model, programs);
            }

            if (programCount == 0)
            {
                continue;
            }
        }

        // Serialize Gfx: Gfx 版シェーダバイナリの作成
        shadingModelBuilder->m_ShaderCompilerManager.Serialize(
            shaderContext.m_pBinShader->GetResShaderFileDataPtr( pCtx ),
            shaderContext.m_ResShaderFileSize );
    }
}

void ShdrFile::Build(std::shared_ptr<Context> pCtx)
{
    Context& shdr_ctx = *pCtx;

    shdr_ctx.blocks.push_back(this);

    // 出力ファイル名を文字列プールに追加する。
    pCtx->SetStr( this->m_Name.c_str() );

    // バイナリ化対象のファイルを収集する。
    const nw::g3d::tool::g3dif::elem_shader_definition& fsd = *m_FsdArray[0];

    //ctx.pool.AddPath(fsd.path);
    pCtx->SetStr( fsd.path.c_str() );

    if (fsd.shading_model_array.size() != shdr_ctx.m_ShaderContexts.size())
    {
        THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
    }

    // 全プログラムが 0 だとエラー
    size_t alllProgramCount = 0;
    for (auto context = shdr_ctx.m_ShaderContexts.cbegin(); context != shdr_ctx.m_ShaderContexts.cend(); ++context)
    {
        alllProgramCount += context->m_pPrograms.size();
    }
    if (alllProgramCount == 0)
    {
        THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
    }

    std::vector<const nw::g3d::tool::g3dif::elem_shading_model*> definitionArray;
    std::vector<const Context::ShaderContext*> shaderContextArray;
    auto numShader = m_FsdArray.size();
    definitionArray.reserve(numShader);
    shaderContextArray.reserve(numShader);

    auto iterShaderContext = shdr_ctx.m_ShaderContexts.cbegin();
    for (auto definition = fsd.shading_model_array.cbegin(); definition != fsd.shading_model_array.cend();
        ++definition, ++iterShaderContext)
    {
        // program が 0 のシェーディングモデルはスキップする。
        if (iterShaderContext->m_Programs.size() == 0)
        {
            continue;
        }

        definitionArray.push_back(&(*definition));
        shaderContextArray.push_back(&(*iterShaderContext));
    }

    // 可変長要素の辞書をコンバート対象に追加する。
    m_DicShader.Build(pCtx, definitionArray.size());

    // 中間ファイルにはない要素を追加する必要があるので、ここで設定する。
    auto size = definitionArray.size();
    m_ShaderArray.resize(size);

    auto iterContext = shaderContextArray.begin();
    auto iterDst = m_ShaderArray.begin();
    while (iterDst != m_ShaderArray.end())
    {
        iterDst->SetShaderContext( const_cast<Context::ShaderContext*>( *iterContext ) );
        iterDst->SetShaderArchive(this);
        iterDst->SetForceVariation(m_IsForceVariation);

        ++iterDst;
        ++iterContext;
    }

    BuildArray(pCtx, m_ShaderArray, definitionArray);

    // 文字列の登録。
    int shaderIndex = 0;
    for (auto iter = definitionArray.cbegin(); iter != definitionArray.cend(); ++iter, ++shaderIndex)
    {
        m_DicShader.SetName(shaderIndex, (*iter)->name.value);
    }
}

void ShdrFile::CalculateSize()
{
    m_Chunk[ ChunkType_ShaderFileData ].size = sizeof( nn::g3d::ResShaderFileData );
    m_Chunk[ ChunkType_ShaderArchiveData ].size = sizeof( nn::g3d::ResShaderArchiveData );
    m_Chunk[ ChunkType_ShadingModelDataArray ].size = sizeof( nn::g3d::ResShadingModelData ) * m_ShaderArray.size();

    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_Count, 8));
}

void ShdrFile::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);

    // 子にヒープへのオフセットを与えます。
    ptrdiff_t offset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ ChunkType_ShadingModelDataArray ].offset;
    for (auto iter = m_ShaderArray.begin(); iter != m_ShaderArray.end(); ++iter)
    {
        iter->SetStructOffset( offset );
        offset += sizeof( nn::g3d::ResShadingModelData );
    }
}

void ShdrFile::Convert( std::shared_ptr<Context> pCtx )
{
    // ResShaderFileData
    nn::g3d::ResShaderFileData& shaderFileData =
        *GetPtr<nn::g3d::ResShaderFileData>( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_ShaderFileData ].offset );
    nn::util::BinaryFileHeader& header = shaderFileData.fileHeader;
    header.signature.SetPacked( nn::g3d::ResShaderFile::Signature );
    header.version.SetPacked( NN_UTIL_CREATE_BINVERSION( NN_G3D_SHADER_BINARY_VERSION_MAJOR, NN_G3D_SHADER_BINARY_VERSION_MINOR, NN_G3D_SHADER_BINARY_VERSION_MICRO ) );
    header.SetByteOrderMark( nn::util::ByteOrderMark_Normal );	// 0xFEFF
    shaderFileData.sizeStringPool = static_cast<uint32_t>( pCtx->GetStrPoolMemBlock()->GetSize() );
    pCtx->LinkPtr( &shaderFileData.pStringPool, pCtx->GetStrPoolMemBlock()->Get( pCtx->GetBasePtr() ) );
    pCtx->LinkPtr( &shaderFileData.pShaderArchive,
        GetPtr<nn::g3d::ResShaderArchive>( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_ShaderArchiveData ].offset ) );

    // ResShaderArchiveData
    nn::g3d::ResShaderArchiveData& rShaderArchiveData =
        *GetPtr<nn::g3d::ResShaderArchiveData>( pCtx, Context::MemBlockType_Main, m_Chunk[ ChunkType_ShaderArchiveData ].offset );

    pCtx->LinkStr( &rShaderArchiveData.pName, this->m_Name.c_str() );

    // バイナリ化対象のファイルを収集する。
    const nw::g3d::tool::g3dif::elem_shader_definition& fsd = *m_FsdArray[0];
    pCtx->LinkStr( &rShaderArchiveData.pPath, fsd.path.c_str() );


    // ShadingModel
    pCtx->LinkPtr( &rShaderArchiveData.pShadingModelArray,
        GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_ShadingModelDataArray].offset));

    rShaderArchiveData.shadingModelCount = static_cast<u16>(m_ShaderArray.size());
    rShaderArchiveData.flag = 0;
    if (m_IsShaderCodeTypeSource)
    {
        rShaderArchiveData.flag |= nn::g3d::ResShaderArchive::Flag_Souce;
    }
    if (m_IsShaderCodeTypeBinary)
    {
        rShaderArchiveData.flag |= nn::g3d::ResShaderArchive::Flag_Binary;
    }
    if( m_IsShaderCodeTypeIr )
    {
        rShaderArchiveData.flag |= nn::g3d::ResShaderArchive::Flag_Ir;
    }
    if (m_IsForceVariation)
    {
        rShaderArchiveData.flag |= nn::g3d::ResShaderArchive::Flag_ForceVariation;
    }
    if (m_IsBinaryAvailable)
    {
        rShaderArchiveData.flag |= nn::g3d::ResShaderArchive::Flag_BinaryAvailable;
    }

    rShaderArchiveData.pUserPtr.Set(nullptr);
    rShaderArchiveData.pCallback.Set(nullptr);
    rShaderArchiveData.pWorkMemoryPtr.Set(nullptr);

    m_DicShader.ConvertData(pCtx, rShaderArchiveData.pShadingModelDic, m_ShaderArray);
}

void ShdrFile::Adjust( std::shared_ptr<Context> pCtx )
{
}

} // namespace g3dTool
} // namespace nn
