﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <BinShader.h>
#include <ShdrFile.h>

#if defined( near ) && defined( far )	// windows.h で両方 define される
#undef near
#undef far
#include <nn/g3d/g3d_ResMaterial.h>	// nn:util:math で near, far を使用している
#define near
#define far
#else
#include <nn/g3d/g3d_ResMaterial.h>
#endif

#include <util/UtilBit.h>

#define BIN_DEFINE_ENUM_TABLE(type, name, table, ...)                                                \
namespace {                                                                                          \
const u32 tbl_##type##_##name##_##table[] = {                                                        \
    __VA_ARGS__                                                                                      \
};                                                                                                   \
}                                                                                                    \
u32 ToEnum_##table(nw::g3d::tool::g3dif::elem_##type::enum_##name value)                                            \
{                                                                                                    \
    if (!(0 <= value || value < nw::g3d::tool::g3dif::elem_##type::num_##name))                                     \
    {                                                                                                \
        THROW_ERROR(ERRCODE_UNKNOWN_ENUM, "Unknown enum value. %d", value);                          \
    }                                                                                                \
    return tbl_##type##_##name##_##table[value];                                                     \
}                                                                                                    \
static_assert(sizeof(tbl_##type##_##name##_##table) / sizeof(u32) == nw::g3d::tool::g3dif::elem_##type::num_##name, \
    "Invalid table size.")

namespace nn { namespace g3dTool {

BIN_DEFINE_ENUM_TABLE(
    shader_param, type, binary,
    nn::g3d::ResShaderParam::Type_Bool, nn::g3d::ResShaderParam::Type_Bool2, nn::g3d::ResShaderParam::Type_Bool3, nn::g3d::ResShaderParam::Type_Bool4,
    nn::g3d::ResShaderParam::Type_Int, nn::g3d::ResShaderParam::Type_Int2, nn::g3d::ResShaderParam::Type_Int3, nn::g3d::ResShaderParam::Type_Int4,
    nn::g3d::ResShaderParam::Type_Uint, nn::g3d::ResShaderParam::Type_Uint2, nn::g3d::ResShaderParam::Type_Uint3, nn::g3d::ResShaderParam::Type_Uint4,
    nn::g3d::ResShaderParam::Type_Float, nn::g3d::ResShaderParam::Type_Float2, nn::g3d::ResShaderParam::Type_Float3, nn::g3d::ResShaderParam::Type_Float4,
    nn::g3d::ResShaderParam::Type_Float2x2, nn::g3d::ResShaderParam::Type_Float2x3, nn::g3d::ResShaderParam::Type_Float2x4,
    nn::g3d::ResShaderParam::Type_Float3x2, nn::g3d::ResShaderParam::Type_Float3x3, nn::g3d::ResShaderParam::Type_Float3x4,
    nn::g3d::ResShaderParam::Type_Float4x2, nn::g3d::ResShaderParam::Type_Float4x3, nn::g3d::ResShaderParam::Type_Float4x4,
    nn::g3d::ResShaderParam::Type_Srt2d, nn::g3d::ResShaderParam::Type_Srt3d,
    nn::g3d::ResShaderParam::Type_Texsrt, nn::g3d::ResShaderParam::Type_Texsrt
    );

namespace {
    struct UniformEntry
    {
        u16 arrayLength;
        s32 offset;
        int type;
    };

    UniformEntry GetUniformEntry(const std::string* pNames[ShaderStage_StageCount], const UniformMap& uniformMap)
    {
        int arrayLength = -1;
        int offset = -1;
        int type = -1;

        for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
        {
            if (pNames[stage] != nullptr)
            {
                // シェーダープログラムから Uniform を検索する。
                auto comp = uniformMap.find(*pNames[stage]);
                if (comp != uniformMap.end())
                {
#if 1
                    // arrayLength, offset, type は全プログラムで一致している必要がある。
                    if ((arrayLength != -1 || offset != -1 || type != -1) &&
                        (arrayLength != static_cast<int>(comp->second.arrayCount) || offset != static_cast<int>(comp->second.offset) || type != comp->second.type))
                    {
                        THROW_ERROR(ERRCODE_SHADER_CONVERTER_INVALID_UNIFORM, "Invalid uniform(%hs).", pNames[stage]->c_str());
                    }
                    else
#endif
                    {
                        arrayLength = comp->second.arrayCount;
                        offset = comp->second.offset;
                        type = comp->second.type;
                    }
                }
                else
                {
                    // 最下層でない配列の場合はインデクスつきで問い合わせる必要がある
                    // もっとも小さいオフセット、もっとも大きい配列長を採用する
                    auto nameLength = pNames[stage]->size();
                    uint32_t minOffset = std::numeric_limits<uint32_t>::max();
                    uint32_t maxArrayCount = std::numeric_limits<uint32_t>::min();
                    for (auto uniform : uniformMap)
                    {
                        if (uniform.first.size() > nameLength && uniform.first[nameLength] == '['
                            && uniform.first.compare(0, nameLength, *pNames[stage]) == 0)
                        {
                            if ( minOffset > uniform.second.offset)
                            {
                                minOffset = uniform.second.offset;
                            }
                            auto right = uniform.first.find(']', nameLength + 1);
                            if (right != std::string::npos)
                            {
                                uint32_t arrayCount;
                                std::istringstream iss(uniform.first.substr(nameLength + 1, right - nameLength - 1));
                                iss >> arrayCount;
                                if (maxArrayCount < ++arrayCount)
                                {
                                    maxArrayCount = arrayCount;
                                }
                            }
                        }
                    }
                    if (minOffset != (std::numeric_limits<uint32_t>::max)())
                    {
                        if ((arrayLength != -1 || offset != -1) &&
                            (arrayLength != nw::g3d::tool::NumericCast<int>(maxArrayCount) || offset != nw::g3d::tool::NumericCast<int>(minOffset)))
                        {
                            THROW_ERROR(ERRCODE_SHADER_CONVERTER_INVALID_UNIFORM, "Invalid uniform(%hs).", pNames[stage]->c_str());
                        }
                        else
                        {
                            arrayLength = nw::g3d::tool::NumericCast<int>(maxArrayCount);
                            offset = nw::g3d::tool::NumericCast<int>(minOffset);
                        }
                    }
                }
            }
        }

        UniformEntry entry;
        entry.arrayLength = (arrayLength != -1) ? static_cast<u16>(arrayLength) : 0;
        entry.offset = offset;
        entry.type = type;
        return entry;
    }

    // byte で渡される。
    bool CheckAndFillBit(void* bit, size_t byteSize, size_t offset, size_t width)
    {
        // Uniform Block の全体領域を超えている。
        if (offset + width > byteSize)
        {
            return false;
        }

        size_t startByte = offset / 8;
        size_t startBit  = offset % 8;

        void* checkBit = nw::g3d::tool::util::AddOffset(bit, startByte);
        for (int i = 0; i < static_cast<int>(width); ++i)
        {
            int offsetByte = (i + static_cast<int>(startBit)) / 8;
            int offsetBit  = (i + static_cast<int>(startBit)) % 8;
            u8* targetByte = nw::g3d::tool::util::AddOffset<u8>(checkBit, offsetByte);
            nw::g3d::tool::util::BitUtil<u8> bitUtil(*targetByte);
            u8 bitMask = nw::g3d::tool::util::BitUtil<u8>::MakeHiMask(offsetBit);
            if (bitUtil.IsMaskOn(bitMask))
            {
                return false;
            }

            bitUtil.SetMaskOn(bitMask);
            *targetByte = bitUtil;
        }

        return true;
    }
}	// namespace

void FindAttribLocation(std::vector<int>& locationTable, const std::vector<ShaderProgram::ShaderSet>& attribTable)
{
    int attribIndex = 0;
    for (auto attrib = attribTable.cbegin(); attrib != attribTable.cend(); ++attrib, ++attribIndex)
    {
        int8_t location = attrib->index[ShaderStage_Vertex];
        if (locationTable[attribIndex] == -1)
        {
            locationTable[attribIndex] = location;
        }
        else if ((location != -1) && (locationTable[attribIndex] != location))
        {
            THROW_ERROR(ERRCODE_SHADER_CONVERTER_ATTRIBUTE, "Attribute locations are indetermination. Please use layout(location=N) to attribute declaration.")
        }
    }
}

void BinShader::Build(std::shared_ptr<Context> pCtx, const nw::g3d::tool::g3dif::elem_shading_model& elem)
{
    Context& shdr_ctx = *pCtx;

    shdr_ctx.blocks.push_back(this);
    m_pElem = &elem;

    std::vector<const nw::g3d::tool::g3dif::elem_option_var*> optionArray;
    auto numOption = elem.option_var_array.size();
    optionArray.reserve(numOption);

    m_StaticOptionArray.reserve(numOption);
    m_DynamicOptionArray.reserve(numOption);

    std::vector<const nw::g3d::tool::g3dif::elem_attrib_var*> attribArray;
    auto numAttrib = elem.attrib_var_array.size();
    attribArray.reserve(numAttrib);

    auto numSampler = elem.sampler_var_array.size();
    m_SamplerArray.reserve(numSampler);

    auto numBlock = elem.block_var_array.size();
    m_BlockArray.reserve(numBlock);

    m_SsboBlockArray.reserve( elem.ssbo_block_var_array.size() );

    // attribute, sampelr, uniform_block が全てのプログラムで
    // オミットされていないかを調べる。

    // attribute
    m_AttribLocationTable.resize(elem.attrib_var_array.size(), -1);
    for (auto program = m_ShaderContext->m_pPrograms.cbegin(); program != m_ShaderContext->m_pPrograms.cend(); ++program)
    {
        // 全ての program から共通の location を収集する。
        // location は固定されていないといけない。
        FindAttribLocation(m_AttribLocationTable, (*program)->m_AttribTable);
    }

    int attribIndex = 0;
    for (auto el_attrib = elem.attrib_var_array.cbegin(); el_attrib != elem.attrib_var_array.cend(); ++el_attrib, ++attribIndex)
    {
        // 有効な Attribute のみを収集する。
        if (m_AttribLocationTable[attribIndex] != -1)
        {
            ++m_ActiveAttribCount;
        }
    }

    // sampler
    for (auto el_sampler = elem.sampler_var_array.cbegin(); el_sampler != elem.sampler_var_array.cend(); ++el_sampler)
    {
        m_SamplerArray.push_back(&*el_sampler);
    }

    // uniform_block
    for (auto el_block = elem.block_var_array.cbegin(); el_block != elem.block_var_array.cend(); ++el_block)
    {
        if (el_block->type.value == nw::g3d::tool::g3dif::elem_block_var::material)
        {
            const std::string* name[ShaderStage_StageCount] = { nullptr, nullptr, nullptr, nullptr };

            for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
            {
                if (el_block->shader_symbol[stage])
                {
                    name[stage] = &el_block->shader_symbol[stage]->name.value;
                }
            }
            for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
            {
                if (name[stage])
                {
                    auto comp = m_ShaderContext->m_UniformBlockMap.find(*name[stage]);
                    if (comp != m_ShaderContext->m_UniformBlockMap.end())
                    {
                        m_DefaultMaterialSize = comp->second;
                        break;
                    }
                }
            }
        }

        m_BlockArray.push_back(&*el_block);
    }

    // ssbo_block
    for (auto el_block = elem.ssbo_block_var_array.cbegin(); el_block != elem.ssbo_block_var_array.cend(); ++el_block)
    {
        m_SsboBlockArray.push_back(&*el_block);
    }

    for (auto iter = elem.option_var_array.cbegin(); iter != elem.option_var_array.cend(); ++iter)
    {
        optionArray.push_back(&*iter);
        if (iter->type.value == nw::g3d::tool::g3dif::elem_option_var::type_static)
        {
            m_StaticOptionArray.emplace_back(&*iter);
        }
        else if (iter->type.value == nw::g3d::tool::g3dif::elem_option_var::type_dynamic)
        {
            m_DynamicOptionArray.emplace_back(&*iter);
        }
    }

    for (auto iter = elem.attrib_var_array.cbegin(); iter != elem.attrib_var_array.cend(); ++iter)
    {
        attribArray.push_back(&*iter);
    }

    // 可変長要素の辞書をコンバート対象に追加する。
    m_DicStaticOption.Build(pCtx, m_StaticOptionArray.size());
    m_DicDynamicOption.Build(pCtx, m_DynamicOptionArray.size());
    m_DicAttrib.Build(pCtx, m_ActiveAttribCount);
    m_DicSampler.Build(pCtx, m_SamplerArray.size());
    m_DicBlock.Build(pCtx, m_BlockArray.size());
    m_DicSsboBlock.Build(pCtx, m_SsboBlockArray.size());

    m_DicUniforms.resize(m_BlockArray.size());
    int blockIndex = 0;
    for (auto block = m_BlockArray.cbegin(); block != m_BlockArray.cend(); ++block, ++blockIndex)
    {
        m_DicUniforms[blockIndex].Build(pCtx, (*block)->uniform_var_array.size());
    }

    m_DicSsboUniforms.resize(m_SsboBlockArray.size());
    blockIndex = 0;
    for (auto block = m_SsboBlockArray.cbegin(); block != m_SsboBlockArray.cend(); ++block, ++blockIndex)
    {
        m_DicSsboUniforms[blockIndex].Build(pCtx, (*block)->uniform_var_array.size());
    }

    m_DicChoices.resize(elem.option_var_array.size());
    int optionIndex = 0;
    for (auto option = elem.option_var_array.cbegin(); option != elem.option_var_array.cend(); ++option, ++optionIndex)
    {
        m_DicChoices[optionIndex].Build(pCtx, option->choiceArray.size());
    }

    // BinProgram の追加
    m_ProgramArray.resize(m_ShaderContext->m_pPrograms.size());

    int programIndex = 0;
    for (auto program = m_ProgramArray.begin(); program != m_ProgramArray.end(); ++program, ++programIndex)
    {
        program->Build(pCtx, m_ShaderContext->m_pPrograms[programIndex]);
        program->SetShadingModel(this);
    }

    // 文字列の登録。
    pCtx->SetStr( elem.name.value.c_str() );

    for (auto option = optionArray.cbegin(); option != optionArray.cend(); ++option)
    {
        pCtx->SetStr( (*option)->id.value.c_str() );
        for (auto choice = (*option)->choiceArray.cbegin(); choice != (*option)->choiceArray.cend(); ++choice)
        {
            pCtx->SetStr( choice->first.c_str() );
        }
    }

    attribIndex = 0;
    for (auto attrib = attribArray.cbegin(); attrib != attribArray.cend(); ++attrib, ++attribIndex)
    {
        if (m_AttribLocationTable[attribIndex] != -1)
        {
            pCtx->SetStr( (*attrib)->id.value.c_str() );
        }
    }

    for (auto sampler = m_SamplerArray.cbegin(); sampler != m_SamplerArray.cend(); ++sampler)
    {
        pCtx->SetStr( (*sampler)->id.value.c_str() );
        if (!(*sampler)->alt.value.empty())
        {
            pCtx->SetStr( (*sampler)->alt.value.c_str() );
        }
    }

    for (auto block = m_BlockArray.cbegin(); block != m_BlockArray.cend(); ++block)
    {
        pCtx->SetStr( (*block)->id.value.c_str() );
        for (auto uniform = (*block)->uniform_var_array.cbegin(); uniform != (*block)->uniform_var_array.cend(); ++uniform)
        {
            pCtx->SetStr( uniform->id.value.c_str() );
            if (!uniform->converter.value.empty())
            {
                pCtx->SetStr( uniform->converter.value.c_str() );
            }
        }
    }

    for (auto block = m_SsboBlockArray.cbegin(); block != m_SsboBlockArray.cend(); ++block)
    {
        pCtx->SetStr( (*block)->id.value.c_str() );
        for (auto uniform = (*block)->uniform_var_array.cbegin(); uniform != (*block)->uniform_var_array.cend(); ++uniform)
        {
            pCtx->SetStr( uniform->id.value.c_str() );
        }
    }

    // シェーダ内記載の sampler 名
    for (auto el_sampler = m_pElem->sampler_var_array.cbegin();
        el_sampler != m_pElem->sampler_var_array.cend(); ++el_sampler)
    {
        for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
        {
            if (el_sampler->shader_symbol[stage])
            {
                pCtx->SetStr(el_sampler->shader_symbol[stage]->name.value.c_str());
            }
        }
    }

    // シェーダ内記載の ubo/ssbo 名
    auto setBlockName = [&]( const auto& blockVarArray )
    {
        for (auto el_block = blockVarArray.cbegin(); el_block != blockVarArray.cend(); ++el_block)
        {
            for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
            {
                if (el_block->shader_symbol[stage])
                {
                    pCtx->SetStr(el_block->shader_symbol[stage]->name.value.c_str());
                }
            }
        }
    };
    setBlockName( m_pElem->block_var_array );
    setBlockName( m_pElem->ssbo_block_var_array );


    // 辞書のインデクスと名前を関連付け
    optionIndex = 0;
    for (auto iter = optionArray.cbegin(); iter != optionArray.cend(); ++iter, ++optionIndex)
    {
        int choiceIndex = 0;
        for (auto choice = (*iter)->choiceArray.cbegin(); choice != (*iter)->choiceArray.cend(); ++choice, ++choiceIndex)
        {
            m_DicChoices[optionIndex].SetName(choiceIndex, choice->first);
        }
    }

    int staticOptionIndex = 0;
    for (auto iter = m_StaticOptionArray.cbegin(); iter != m_StaticOptionArray.cend(); ++iter, ++staticOptionIndex)
    {
        m_DicStaticOption.SetName(staticOptionIndex, (*iter)->id.value);
    }

    int dynamicOptionIndex = 0;
    for (auto iter = m_DynamicOptionArray.cbegin(); iter != m_DynamicOptionArray.cend(); ++iter, ++dynamicOptionIndex)
    {
        m_DicDynamicOption.SetName(dynamicOptionIndex, (*iter)->id.value);
    }

    int activeAttribIndex = 0;
    attribIndex = 0;
    for (auto iter = attribArray.cbegin(); iter != attribArray.cend(); ++iter, ++attribIndex)
    {
        if (m_AttribLocationTable[attribIndex] != -1)
        {
            m_DicAttrib.SetName(activeAttribIndex, (*iter)->id.value);
            ++activeAttribIndex;
        }
    }

    int samplerIndex = 0;
    for (auto iter = m_SamplerArray.cbegin(); iter != m_SamplerArray.cend(); ++iter, ++samplerIndex)
    {
        m_DicSampler.SetName(samplerIndex, (*iter)->id.value);
    }

    blockIndex = 0;
    for (auto iter = m_BlockArray.cbegin(); iter != m_BlockArray.cend(); ++iter, ++blockIndex)
    {
        m_DicBlock.SetName(blockIndex, (*iter)->id.value);
        int uniformIndex = 0;
        for (auto uniform = (*iter)->uniform_var_array.cbegin(); uniform != (*iter)->uniform_var_array.cend(); ++uniform, ++uniformIndex)
        {
            m_DicUniforms[blockIndex].SetName(uniformIndex, uniform->id.value);
        }
    }

    blockIndex = 0;
    for (auto iter = m_SsboBlockArray.cbegin(); iter != m_SsboBlockArray.cend(); ++iter, ++blockIndex)
    {
        m_DicSsboBlock.SetName(blockIndex, (*iter)->id.value);
        int uniformIndex = 0;
        for (auto uniform = (*iter)->uniform_var_array.cbegin(); uniform != (*iter)->uniform_var_array.cend(); ++uniform, ++uniformIndex)
        {
            m_DicSsboUniforms[blockIndex].SetName(uniformIndex, uniform->id.value);
        }
    }

}

void BinShader::CalculateSize()
{
    m_Chunk[ChunkType_Option].size = sizeof(nn::g3d::ResShaderOptionData) * m_pElem->option_var_array.size();
    m_Chunk[ChunkType_Attribute].size = sizeof(nn::g3d::ResAttrVarData) * m_ActiveAttribCount;
    m_Chunk[ChunkType_Sampler].size = sizeof(nn::g3d::ResSamplerVarData) * m_SamplerArray.size();
    m_Chunk[ChunkType_Block].size = sizeof(nn::g3d::ResUniformBlockVarData) * m_BlockArray.size();
    m_Chunk[ChunkType_SsboBlock].size = sizeof(nn::g3d::ResShaderStorageBlockVarData) * m_SsboBlockArray.size();
    auto countUniform = []( int& numUniform, const auto& blockArray )
    {
        for (auto block = blockArray.cbegin(); block != blockArray.cend(); ++block)
        {
            numUniform += static_cast<int>( (*block)->uniform_var_array.size() );
        }
    };
    int numUniform = 0;
    countUniform( numUniform, m_BlockArray );
    m_Chunk[ChunkType_Uniform].size = sizeof(nn::g3d::ResUniformVarData) * numUniform;
    numUniform = 0;
    countUniform( numUniform, m_SsboBlockArray );
    m_Chunk[ChunkType_Ssbo].size = sizeof(nn::g3d::ResShaderStorageVarData) * numUniform;

    m_Chunk[ChunkType_Program].size = sizeof(nn::g3d::ResShaderProgramData) * m_ProgramArray.size();
    m_Chunk[ChunkType_Key].size = sizeof(uint32_t) * m_pElem->keyLength * m_ProgramArray.size();
    m_Chunk[ChunkType_DefaultValue].size = m_DefaultMaterialSize;

    size_t choiceSize = 0;
    if (!m_IsForceVariation)
    {
        for (auto option = m_pElem->option_var_array.cbegin(); option != m_pElem->option_var_array.cend(); ++option)
        {
            choiceSize += option->choiceUintArray.size() * sizeof(s32);
        }
    }
    m_Chunk[ChunkType_ChoiceValue].size = choiceSize;

    // ResShaderInfo 周辺のメモリ構造は下記の様になっている。
    // sizeof( ResShaderInfo )
    // sizeof( sampler name ptr array )
    // sizeof( uniform block name ptr array )
    // sizeof( streamout name ptr array )
    size_t shaderInfoSize = 0;
    {
        // ResShaderInfo
        shaderInfoSize += sizeof(nn::g3d::ResShaderInfo);

        // sampler var array
        shaderInfoSize += m_pElem->sampler_var_array.size() * ShaderStage_StageCount * sizeof( nn::util::BinTPtr<nn::util::BinPtrToString> );

        // uniform block var array
        shaderInfoSize += m_pElem->block_var_array.size() * ShaderStage_StageCount * sizeof(nn::util::BinTPtr<nn::util::BinPtrToString>);

        // shader storage block var array
        shaderInfoSize += m_pElem->ssbo_block_var_array.size() * ShaderStage_StageCount * sizeof(nn::util::BinTPtr<nn::util::BinPtrToString>);

        // streamout
        if (m_pElem->streamout)
        {
            std::vector<std::string> streamouts;
            nw::g3d::tool::util::Split(streamouts, m_pElem->streamout->varying.value, " \t");
            shaderInfoSize += streamouts.size() * sizeof(nn::util::BinTPtr<nn::util::BinPtrToString>);
        }
    }

    m_Chunk[ChunkType_ShaderInfo].size = nw::g3d::tool::util::Align(shaderInfoSize);

    SetBlockSize(Context::MemBlockType_Main, CalcChunk(m_Chunk, ChunkType_ChunkCount, 8));

    // ResShaderFileData のアライメントを整えるためのパディングサイズを計算する。
    {
        m_ShaderFileChunk[ ShaderFileChunk_ShaderFileData ].size = m_ShaderContext->m_ResShaderFileSize;

        if( !nn::util::is_aligned( m_ShaderContext->m_ResShaderFileSize, m_ShaderContext->m_ResShaderFileAlignment) )
        {
            m_ShaderFileChunk[ ShaderFileChunk_Padding ].size =
                nn::util::align_up( m_ShaderContext->m_ResShaderFileSize, m_ShaderContext->m_ResShaderFileAlignment) - m_ShaderContext->m_ResShaderFileSize;
        }

        SetBlockSize(Context::MemBlockType_ShaderFile, CalcChunk(m_ShaderFileChunk, ShaderFileChunk_ChunkCount), m_ShaderContext->m_ResShaderFileAlignment);
    }
}

void BinShader::CalculateOffset( std::shared_ptr<Context> pCtx )
{
    BinaryBlock::CalculateOffset(pCtx);
    ptrdiff_t programOffset = GetBlockOffset(Context::MemBlockType_Main) + m_Chunk[ChunkType_Program].offset;
    for (auto iter = m_ProgramArray.begin(); iter != m_ProgramArray.end(); ++iter)
    {
        iter->SetStructOffset(programOffset);
        programOffset += sizeof(nn::g3d::ResShaderProgramData);
    }
}

void BinShader::Convert( std::shared_ptr<Context> pCtx )
{
    // NOTE: fsv で全てのバリエーションが削除された場合、正しくコンバートできません。
    //       いまのところは全てのバリエーションが削除された場合は ShadingModel ごと削除します。

    nn::g3d::ResShadingModelData& rShadingModelData =
        *BinaryBlock::GetPtr<nn::g3d::ResShadingModelData>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) );

    pCtx->LinkStr( &rShadingModelData.pName, m_pElem->name.value.c_str() );
    rShadingModelData.pCallbackParam.Set(nullptr);

    // ResShaderArchive
    pCtx->LinkPtr( &rShadingModelData.pShaderArchive,
        m_pFile->GetPtr( static_cast<char*>( pCtx->GetMemBlockPtr( Context::MemBlockType_Main ) )
        + sizeof( nn::g3d::ResShaderFile ) ) );	//!< 直接 ResShaderArchive へのアドレスを取得できないので sizeof( ResShaderFile ) をオフセットとして加えている。

    // ResShaderFile.
    void* pShaderFile = GetPtr(pCtx, Context::MemBlockType_ShaderFile, m_ShaderFileChunk[ShaderFileChunk_ShaderFileData].offset);
    if( !nn::util::is_aligned( reinterpret_cast<uintptr_t>( pShaderFile ), m_ShaderContext->m_ResShaderFileAlignment) )
    {
        THROW_ERROR( ERRCODE_INTERNAL, "ShaderFile alignment should be %d\n", m_ShaderContext->m_ResShaderFileAlignment);
    }
    pCtx->LinkPtr( &rShadingModelData.pShaderFile, pShaderFile );

    // Program
    pCtx->LinkPtr( &rShadingModelData.pShaderProgramArray, GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Program].offset));

    // keytable
    uint32_t* pKeyTable = GetPtr<uint32_t>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Key].offset);
    pCtx->LinkPtr( &rShadingModelData.pKeyTable, pKeyTable);
    uint32_t* dstKey = pKeyTable;

    int defaultProgramIndex = nn::g3d::ShaderLocationProgramNone;

    int programIndex = 0;
    for (auto program = m_ProgramArray.cbegin(); program != m_ProgramArray.cend(); ++program, ++programIndex)
    {
        const ShaderProgram* shaderProgram = program->GetShaderProgram();
        if (m_pElem->defaultKey && memcmp(shaderProgram->m_Key.get(), m_pElem->defaultKey.get(), m_pElem->keyLength * sizeof(uint32_t)) == 0)
        {
            defaultProgramIndex = programIndex;
        }
        memcpy(dstKey, shaderProgram->m_Key.get(), m_pElem->keyLength * sizeof(uint32_t));
        dstKey += m_pElem->keyLength;
    }

    rShadingModelData.defaultProgramIndex = static_cast<s32>(defaultProgramIndex);
    rShadingModelData.shaderProgramCount = static_cast<u16>(m_ProgramArray.size());
    rShadingModelData.pUserPtr.Set(nullptr);
    rShadingModelData.pMutexTypePtr.Set(nullptr);

    // ShaderInfo
    if( m_Chunk[ChunkType_ShaderInfo].size > 0 )
    {
        // SymbolTable
        nn::g3d::ResShaderInfo* pShaderInfo =
            GetPtr<nn::g3d::ResShaderInfo>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_ShaderInfo].offset);
        memset(pShaderInfo, 0, m_Chunk[ChunkType_ShaderInfo].size);
        char* ptr = reinterpret_cast<char*>(pShaderInfo);
        pCtx->LinkPtr( &rShadingModelData.pShaderInfo, ptr );

        ptr += sizeof(nn::g3d::ResShaderInfo);


        pCtx->LinkPtr( &pShaderInfo->pSamplerTable, ptr );	// sampler name ptr array へリンクする
        nn::util::BinPtrToString* pName = reinterpret_cast<nn::util::BinPtrToString*>( ptr );
        for (auto el_sampler = m_pElem->sampler_var_array.cbegin();
            el_sampler != m_pElem->sampler_var_array.cend(); ++el_sampler)
        {
            for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
            {
                if (el_sampler->shader_symbol[stage])
                {
                    pCtx->LinkStr(pName, el_sampler->shader_symbol[stage]->name.value.c_str());
                }
                else
                {
                    pCtx->LinkStr(pName, "");
                }
                ++pName;
            }
        }
        if (m_pElem->sampler_var_array.size() <= 0)
        {
            pShaderInfo->pSamplerTable.Set(nullptr);
        }

        // ここから uniform/ssbo block name table.
        auto buildBlockTable = [&]( auto& pName, auto&pBlockTable, const auto& blockVarArray )
        {
            pCtx->LinkPtr( &pBlockTable, pName );
            for (auto el_block = blockVarArray.cbegin();
                el_block != blockVarArray.cend(); ++el_block)
            {
                for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
                {
                    if (el_block->shader_symbol[stage])
                    {
                        pCtx->LinkStr(pName, el_block->shader_symbol[stage]->name.value.c_str());
                    }
                    else
                    {
                        pCtx->LinkStr(pName, "");
                    }
                    ++pName;
                }
            }
            if (blockVarArray.size() <= 0)
            {
                pBlockTable.Set(nullptr);
            }
        };
        buildBlockTable( pName, pShaderInfo->pUniformBlockTable, m_pElem->block_var_array );
        buildBlockTable( pName, pShaderInfo->pShaderStorageBlockTable, m_pElem->ssbo_block_var_array );

        // StreamOut
        if (m_pElem->streamout)
        {
            pCtx->LinkPtr( &pShaderInfo->pStreamout, pName );
            std::vector<std::string> streamouts;
            nw::g3d::tool::util::Split(streamouts, m_pElem->streamout->varying.value, " \t");
            pShaderInfo->streamoutCount = static_cast<uint32_t>(streamouts.size());
            for (auto streamout = streamouts.cbegin(); streamout != streamouts.cend(); ++streamout)
            {
                pCtx->LinkStr( pName, streamout->c_str() );
                ++pName;
            }
        }
        else
        {
            pShaderInfo->streamoutCount = 0;
            pShaderInfo->pStreamout.Set(nullptr);
        }
    }

    // Option
    ConvertOption(rShadingModelData, pCtx);

    // Attrib
    ConvertAttrib(rShadingModelData, pCtx);

    // Sampler
    ConvertSampler(rShadingModelData, pCtx);

    // SSBO/UBO Block
    ConvertBlock(rShadingModelData, pCtx);

    // Uniform
    ConvertUniform(rShadingModelData, pCtx);
}

void BinShader::ConvertOption(nn::g3d::ResShadingModelData &shadingModel, std::shared_ptr<Context> pCtx)
{
    int staticKeyLength = static_cast<int>(m_pElem->keyLength);
    for (auto el_option_var = m_pElem->option_var_array.cbegin(); el_option_var != m_pElem->option_var_array.cend(); ++el_option_var)
    {
        if (el_option_var->type.value == nw::g3d::tool::g3dif::elem_option_var::type_dynamic)
        {
            staticKeyLength = static_cast<int>(el_option_var->keyOffset) / 32;
            break;
        }
    }

    shadingModel.staticKeyLength = static_cast<u8>(staticKeyLength);
    shadingModel.dynamicKeyLength = static_cast<u8>(m_pElem->keyLength - staticKeyLength);
    shadingModel.staticOptionCount = static_cast<u16>(m_StaticOptionArray.size());
    shadingModel.dynamicOptionCount = static_cast<u16>(m_DynamicOptionArray.size());
    nn::g3d::ResShaderOptionData* pOptionVar =
        GetPtr<nn::g3d::ResShaderOptionData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Option].offset);

    if (m_StaticOptionArray.size() != 0)
    {
        pCtx->LinkPtr( &shadingModel.pStaticOptionArray, pOptionVar );
    }
    else
    {
        shadingModel.pStaticOptionArray.Set(nullptr);
    }

    if (m_DynamicOptionArray.size() != 0)
    {
        pCtx->LinkPtr( &shadingModel.pDynamicOptionArray, pOptionVar + m_StaticOptionArray.size());
    }
    else
    {
        shadingModel.pDynamicOptionArray.Set(nullptr);
    }

    m_DicStaticOption.ConvertData( pCtx, shadingModel.pStaticOptionDic );
    m_DicDynamicOption.ConvertData( pCtx, shadingModel.pDynamicOptionDic );

    s32* pChoiceValues =
        GetPtr<s32>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_ChoiceValue].offset);

    int index = 0;
    for (auto iter = m_pElem->option_var_array.cbegin();
        iter != m_pElem->option_var_array.cend(); ++iter, ++pOptionVar, ++index)
    {
        // branch かつ static オプション
        if (iter->branch.value  && !m_IsForceVariation && (iter->type.value == nw::g3d::tool::g3dif::elem_option_var::type_static))
        {
            const std::string* pNames[ShaderStage_StageCount] = { nullptr, nullptr, nullptr, nullptr };
            for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
            {
                if (!iter->altUniformSymbol[stage].empty())
                {
                    pNames[stage] = &(iter->altUniformSymbol[stage]);
                }
            }

            UniformEntry entry = GetUniformEntry(pNames, m_ShaderContext->m_UniformMap);
            // シェーダコンパイラがデッドストリップを行う場合があるので
            // オフセットが取得できなくてもエラーにしません。

            if (entry.offset != nn::g3d::ResShaderOption::InvalidOffset)
            {
                pOptionVar->branchOffset = static_cast<u16>(entry.offset + 1);
            }
            else
            {
                pOptionVar->branchOffset = static_cast<u16>(nn::g3d::ResShaderOption::InvalidOffset + 1);
            }
        }
        else
        {
            pOptionVar->branchOffset = static_cast<u16>(nn::g3d::ResShaderOption::InvalidOffset + 1);
        }
        pOptionVar->choiceCount = static_cast<u8>(iter->choiceArray.size());
        pOptionVar->defaultIndex = static_cast<u8>(iter->defaultValueIndex);
        uint32_t flag = 0;
        flag |= ( iter->branch.value && !m_IsForceVariation ) ? nn::g3d::ResShaderOption::Flag_Branch : flag;
        flag |= (iter->type.value == nw::g3d::tool::g3dif::elem_option_var::type_static) ? nn::g3d::ResShaderOption::Flag_Static : flag;
        pOptionVar->flag = static_cast<u8>(flag);

        pOptionVar->bit32Index = static_cast<u8>(iter->keyOffset / 32);

        if (iter->type.value == nw::g3d::tool::g3dif::elem_option_var::type_static)
        {
            pOptionVar->keyOffset = 0;
        }
        else
        {
            pOptionVar->keyOffset = shadingModel.staticKeyLength;
        }

        int lhs = iter->keyOffset % 32;
        int rhs = lhs + static_cast<int>(iter->bitWidth) - 1;
        pOptionVar->bit32Shift = static_cast<u8>(32 - rhs - 1);

        nw::g3d::tool::util::BitUtil<uint32_t> bit;
        bit.SetAllZero();
        bit.SetHiBitOn(lhs, rhs, 0xFFFFFFFF);

        pOptionVar->bit32Mask = static_cast<uint32_t>(bit.GetDirect());
        pCtx->LinkStr( &pOptionVar->pName, iter->id.value.c_str() );

        m_DicChoices[index].ConvertData( pCtx, pOptionVar->pChoiceDic );

        if (iter->choiceUintArray.size() != 0 && !m_IsForceVariation)
        {
            pCtx->LinkPtr( &pOptionVar->pChoiceValues, pChoiceValues);
            for (auto choice = iter->choiceUintArray.cbegin(); choice != iter->choiceUintArray.cend(); ++choice, ++pChoiceValues)
            {
                *pChoiceValues = *choice; // choice を int に変換した値を代入する。
            }
        }
        else
        {
            pOptionVar->pChoiceValues.Set(nullptr);
        }
    }
}

void BinShader::ConvertAttrib(nn::g3d::ResShadingModelData &shadingModel, std::shared_ptr<Context> pCtx)
{
    shadingModel.attrCount = static_cast<u8>(m_ActiveAttribCount);
    nn::g3d::ResAttrVarData* pAttribVar =
        GetPtr<nn::g3d::ResAttrVarData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Attribute].offset);
    pCtx->LinkPtr( &shadingModel.pAttrArray, pAttribVar );

    m_DicAttrib.ConvertData( pCtx, shadingModel.pAttrDic );

    int index = 0;
    int activeIndex = 0;
    for (auto iter = m_pElem->attrib_var_array.cbegin();
        iter != m_pElem->attrib_var_array.cend(); ++iter, ++index)
    {
        if (m_AttribLocationTable[index] != -1)
        {
            pAttribVar->index = static_cast<u8>(activeIndex);

            // NOTE: array は意味がないので対応しません。
            //pAttribVar->arrayLength = static_cast<u8>(1);
            pAttribVar->location = static_cast<int8_t>(m_AttribLocationTable[index]);

            ++pAttribVar;
            ++activeIndex;
        }
    }
}

void BinShader::ConvertSampler(nn::g3d::ResShadingModelData &shadingModel, std::shared_ptr<Context> pCtx)
{
    shadingModel.samplerCount = static_cast<u8>(m_SamplerArray.size());
    nn::g3d::ResSamplerVarData* pSamplerVar =
        GetPtr<nn::g3d::ResSamplerVarData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Sampler].offset);
    pCtx->LinkPtr( &shadingModel.pSamplerArray, pSamplerVar);

    m_DicSampler.ConvertData( pCtx, shadingModel.pSamplerDic );

    int samplerIndex = 0;
    for (auto iter = m_SamplerArray.cbegin();
        iter != m_SamplerArray.cend(); ++iter, ++samplerIndex)
    {
        pSamplerVar->index = static_cast<u8>(samplerIndex);

        pCtx->LinkStr( &pSamplerVar->pAlt, (*iter)->alt.value.c_str() );
        ++pSamplerVar;
    }
}

void BinShader::ConvertBlock(nn::g3d::ResShadingModelData &shadingModel, std::shared_ptr<Context> pCtx)
{
    shadingModel.uniformBlockCount = static_cast<u8>(m_BlockArray.size());
    shadingModel.shaderStorageBlockCount = static_cast<u8>(m_SsboBlockArray.size());

    void* pDefaultValue = GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_DefaultValue].offset);
    memset(pDefaultValue, 0, m_Chunk[ChunkType_DefaultValue].size);

    m_DicBlock.ConvertData( pCtx, shadingModel.pUniformBlockDic );
    m_DicSsboBlock.ConvertData( pCtx, shadingModel.pShaderStorageBlockDic );

    nn::g3d::ResUniformBlockVarData* pBlockVar = GetPtr<nn::g3d::ResUniformBlockVarData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Block].offset);
    pCtx->LinkPtr( &shadingModel.pUniformBlockArray, pBlockVar );
    int blockIndex = 0;
    int resUniformVarBaseIdx = 0;
    auto pUniformVarArray = GetPtr<nn::g3d::ResUniformVarData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Uniform].offset);
    for (auto block = m_BlockArray.cbegin(); block != m_BlockArray.cend(); ++block, ++blockIndex)
    {
        pBlockVar->index = static_cast<u8>(blockIndex);
        pBlockVar->type = static_cast<u8>((*block)->type.value);
        pBlockVar->size = 0;

        const std::string* names[ShaderStage_StageCount] = { nullptr, nullptr, nullptr, nullptr };

        for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
        {
            if ((*block)->shader_symbol[stage])
            {
                names[stage] = &(*block)->shader_symbol[stage]->name.value;
            }
        }

        // シェーダープログラムから UniformBlock を検索する。
        for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
        {
            if (names[stage])
            {
                auto comp = m_ShaderContext->m_UniformBlockMap.find(*names[stage]);
                if (comp != m_ShaderContext->m_UniformBlockMap.end() && pBlockVar->size < static_cast<u16>(comp->second))
                {
                    if (comp->second >= 0xFFFF)
                    {
                        THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
                    }
                    pBlockVar->size = static_cast<u16>(comp->second);
                }
            }
        }

        // 全 static シェーダーオプションがプリプロセッサ分岐の場合、バリエーション切り替えにユニフォームブロックを使用しないので OptionUniformBlock の size を 0 にする。
        if (pBlockVar->type == nn::g3d::ResUniformBlockVar::Type_Option)
        {
            bool useOptionUniformBlock = false;
            if (!m_IsForceVariation)
            {
                for (auto iter = m_pElem->option_var_array.cbegin(); iter != m_pElem->option_var_array.cend(); ++iter)
                {
                    if (iter->type.value == nw::g3d::tool::g3dif::elem_option_var::type_static && iter->branch.value)
                    {
                        useOptionUniformBlock = true;
                        break;
                    }
                }
            }
            if (!useOptionUniformBlock)
            {
                pBlockVar->size = 0;
            }
        }

        pBlockVar->uniformCount = static_cast<u16>((*block)->uniform_var_array.size());
        m_DicUniforms[blockIndex].ConvertData( pCtx, pBlockVar->pUniformDic );
        pCtx->LinkPtr( &pBlockVar->pUniformArray, &pUniformVarArray[resUniformVarBaseIdx] );
        resUniformVarBaseIdx += pBlockVar->uniformCount;

        // material の場合だけ Default が入る。
        if( ( (*block)->type.value == nw::g3d::tool::g3dif::elem_block_var::material ) && ( m_DefaultMaterialSize != 0 ) )
        {
            // マージの場合は外からデフォルト値をもらう
            if (m_pFile->GetConvertMode() == ConvertMode_MergeBfsha)
            {
                memcpy(pDefaultValue, (*block)->pConvertedDefaultValue.get(), m_Chunk[ChunkType_DefaultValue].size);
            }

            pCtx->LinkPtr( &pBlockVar->pDefault, pDefaultValue );
        }
        else
        {
            pBlockVar->pDefault.Set(nullptr);
        }

        ++pBlockVar;
    }

    nn::g3d::ResShaderStorageBlockVarData* pSsboBlockVar = GetPtr<nn::g3d::ResShaderStorageBlockVarData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_SsboBlock].offset);
    pCtx->LinkPtr( &shadingModel.pShaderStorageBlockArray, pSsboBlockVar );
    blockIndex = 0;
    resUniformVarBaseIdx = 0;
    auto pShaderStorageVarArray = GetPtr<nn::g3d::ResUniformVarData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Ssbo].offset);
    for (auto block = m_SsboBlockArray.cbegin(); block != m_SsboBlockArray.cend(); ++block, ++blockIndex)
    {
        pSsboBlockVar->index = static_cast<u8>(blockIndex);
        pSsboBlockVar->type = static_cast<u8>((*block)->type.value);
        pSsboBlockVar->size = 0;

        const std::string* names[ShaderStage_StageCount] = { nullptr, nullptr, nullptr, nullptr };
        for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
        {
            if ((*block)->shader_symbol[stage])
            {
                names[stage] = &(*block)->shader_symbol[stage]->name.value;
            }
        }
        for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
        {
            if (names[stage])
            {
                auto comp = m_ShaderContext->m_SsboBlockMap.find(*names[stage]);
                if ( ( comp != m_ShaderContext->m_SsboBlockMap.end() ) && ( pSsboBlockVar->size < static_cast<u16>(comp->second) ) )
                {
                    if (comp->second >= 0xFFFF)
                    {
                        THROW_ERROR_INTERNAL(ERRCODE_INTERNAL, "Internal error.");
                    }
                    pSsboBlockVar->size = static_cast<u16>(comp->second);
                }
            }
        }

        pSsboBlockVar->shaderStorageCount = static_cast<u16>((*block)->uniform_var_array.size());
        m_DicSsboUniforms[blockIndex].ConvertData( pCtx, pSsboBlockVar->pShaderStorageDic );
        pCtx->LinkPtr( &pSsboBlockVar->pShaderStorageArray, &pShaderStorageVarArray[resUniformVarBaseIdx] );
        resUniformVarBaseIdx += pSsboBlockVar->shaderStorageCount;

        // materialに対してssboは使用できないので常にnullptrをセットする
        pSsboBlockVar->pDefault.Set(nullptr);

        ++pSsboBlockVar;
    }

    // system block indicesの更新
    auto buildSystemBlockIndices = [&]( int8_t* pIndices, int blockTypeCount, const auto& blockArray )
    {
        for( int idxBlockType = 0; idxBlockType < blockTypeCount; ++idxBlockType )
        {
            pIndices[idxBlockType] = -1;
        }
        for( int idxBlock = 0; idxBlock < blockArray.size(); ++idxBlock )
        {
            auto& block = blockArray[idxBlock];
            if( block->type.value != nw::g3d::tool::g3dif::elem_block_var::none )
            {
                int type = static_cast<int>( block->type.value - 1 );
                pIndices[type] = static_cast<int8_t>(idxBlock);
            }
        }
    };

    buildSystemBlockIndices( shadingModel.systemBlockIndices, nn::g3d::ResUniformBlockVar::Type_Num - 1, m_BlockArray );
    buildSystemBlockIndices( shadingModel.systemShaderStorageBlockIndices, nn::g3d::ResShaderStorageBlockVar::Type_Num - 1, m_SsboBlockArray );
}

void BinShader::ConvertUniform(nn::g3d::ResShadingModelData &shadingModel, std::shared_ptr<Context> pCtx)
{
    auto countUniform = []( int& numUniform, const auto& blockArray )
    {
        for (auto block = blockArray.cbegin(); block != blockArray.cend(); ++block)
        {
            numUniform += static_cast<int>( (*block)->uniform_var_array.size() );
        }
    };
    countUniform( shadingModel.uniformCount, m_BlockArray );
    countUniform( shadingModel.shaderStorageCount, m_SsboBlockArray );

    auto buildUniform = [&]( auto pUniformVar, const auto& blockArray, const auto& uniformMap )
    {
        void* pMaterialDefaultValue = GetPtr(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_DefaultValue].offset);
        size_t checkBitCount = nw::g3d::tool::util::Align(m_DefaultMaterialSize, 8);
        std::unique_ptr<char> checkBit;
        checkBit.reset(new char[checkBitCount / 8]);
        memset(checkBit.get(), 0, checkBitCount / 8);

        int blockIndex = 0;
        int uniformIndex = 0;
        for (auto block = blockArray.cbegin(); block != blockArray.cend(); ++block, ++blockIndex)
        {
            //!< TODO: (*block)->uniform_var_arrayがssboだとssbo_uniform_var_arrayになる。
            for (auto uniform = (*block)->uniform_var_array.cbegin(); uniform != (*block)->uniform_var_array.cend(); ++uniform, ++uniformIndex, ++pUniformVar)
            {
                if (uniform->type.value == nw::g3d::tool::g3dif::elem_shader_param::type_texsrt_ex)
                {
                    THROW_ERROR(ERRCODE_UNSUPPORTED,
                        "texsrt_ex type is abolished (%hs). Please use texsrt type and depend annotation.",
                        uniform->id.value.c_str());
                }

                pUniformVar->index = uniformIndex;
                pUniformVar->blockIndex = static_cast<u8>(blockIndex);
                pCtx->LinkStr(&pUniformVar->pConverterName, uniform->converter.value.c_str());

                // シェーダープログラムから Uniform を検索する。
                const std::string* pNames[ShaderStage_StageCount] = { nullptr, nullptr, nullptr, nullptr };

                for (int stage = 0; stage < ShaderStage_StageCount; ++stage)
                {
                    if (uniform->shader_symbol[stage])
                    {
                        pNames[stage] = &(uniform->shader_symbol[stage]->name.value);
                    }
                }

                UniformEntry entry = GetUniformEntry(pNames, uniformMap );
                pUniformVar->offset = static_cast<u16>(entry.offset) + 1;	// gfx tool のコンバータの出力は invalid で -1 g3d では invalid で 0 を返す仕様になっている。g3d 内部ではバイトオフセットを求めるために -1 をしている。

                // マージの場合は外からデフォルト値をもらう
                if (m_pFile->GetConvertMode() == ConvertMode_MergeBfsha)
                {
                    continue;
                }

                // ssbo の type は none, skeleton のみなので以下の処理が行なわれることはない
                if ( ( (*block)->type.value == nw::g3d::tool::g3dif::elem_block_var::material ) && ( pUniformVar->offset != 0 ) )
                {
                    int byteOffset = pUniformVar->offset - 1;
                    void* dst = nw::g3d::tool::util::AddOffset(pMaterialDefaultValue, byteOffset);
                    // 一度 ResShaderParam に入れて RawParam に変換する。
                    nn::g3d::ResShaderParamData param;
                    param.pId.Set(nullptr);
                    param.srcOffset = 0;
                    param.offset = 0;
                    param.type = static_cast<u8>(ToEnum_binary(uniform->type.value));

                    nn::g3d::ResShaderParam* rShaderParam = static_cast<nn::g3d::ResShaderParam*>( &param );

                    // 各シェーダパラメータの rawParam が
                    // 他のシェーダパラメータの領域を侵食していないかチェックする。
                    if (m_DefaultMaterialSize > 0)
                    {
                        size_t rawSize = rShaderParam->GetSize();
                        if (!CheckAndFillBit(checkBit.get(), m_DefaultMaterialSize, byteOffset, rawSize))
                        {
                            THROW_ERROR(
                                ERRCODE_SHADER_CONVERTER_INVALID_UNIFORM_BLOCK_DEFINITION,
                                "The uniform(%hs) intrudes other uniform area in the uniform block(%hs).",
                                uniform->id.value.c_str(), (*block)->id.value.c_str());
                        }
                    }

                    // 中間ファイルのデフォルト値から SrcParam を作成し、
                    // さらに、RawParam に変換する。
                    std::shared_ptr<void> srcParam;
                    size_t srcParamSize = nw::g3d::tool::g3dif::elem_shader_param::GetSrcParamSize(uniform->type.value);
                    srcParam.reset(malloc(srcParamSize), free);
                    memcpy(srcParam.get(), uniform->rawdata.get(), uniform->dataSize);
                    nw::g3d::tool::g3dif::elem_shader_param::ValueToSrcParam(srcParam.get(), uniform->type.value);

                    rShaderParam->SetConvertShaderParamCallback(nullptr);
                    if (param.type >= nn::g3d::ResShaderParam::Type_Srt2d)
                    {
                        static const nn::g3d::ResShaderParam::ConvertShaderParamCallback converter_table[] = {
                            &nn::g3d::ResShaderParam::ConvertSrt2dCallback,
                            &nn::g3d::ResShaderParam::ConvertSrt3dCallback,
                            &nn::g3d::ResShaderParam::ConvertTexSrtCallback };

                        // スワップ <false>
                        converter_table[param.type - nn::g3d::ResShaderParam::Type_Srt2d](dst, srcParam.get(), rShaderParam, nullptr);
                    }
                    else
                    {
                        rShaderParam->Convert<false>(dst, srcParam.get(), nullptr);
                    }
                }
            }
        }
    };

    nn::g3d::ResUniformVarData* pUniformVar = GetPtr<nn::g3d::ResUniformVarData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Uniform].offset);
    pCtx->LinkPtr( &shadingModel.pUniformArray, pUniformVar);
    buildUniform( pUniformVar, m_BlockArray, m_ShaderContext->m_UniformMap );

    nn::g3d::ResShaderStorageVarData* pSsboVar = GetPtr<nn::g3d::ResShaderStorageVarData>(pCtx, Context::MemBlockType_Main, m_Chunk[ChunkType_Ssbo].offset);
    pCtx->LinkPtr( &shadingModel.pShaderStorageArray, pSsboVar);
    buildUniform( pSsboVar, m_SsboBlockArray, m_ShaderContext->m_SsboMap );
}

} // namespace g3dTool
} // namespace nn
